//
//  Copyright (c) Microsoft Corporation. All rights reserved.
//  Licensed under the MIT License.
//

import FluentUI
import UIKit

class SideTabBarDemoController: DemoController {
    override func viewDidLoad() {
        super.viewDidLoad()

        presentSideTabBar()
    }

    private struct Constants {
        static let initialBadgeNumbers: [UInt] = [5, 50, 250, 4, 65, 135]
        static let initialHigherBadgeNumbers: [UInt] = [1250, 25505, 3050528, 50890, 2304, 28745]
        static let optionsSpacing: CGFloat = 5.0
    }

    private let sideTabBar: SideTabBar = {
        return SideTabBar(frame: .zero)
    }()

    private var contentViewController: UIViewController?

    private var badgeNumbers: [UInt] = Constants.initialBadgeNumbers
    private var higherBadgeNumbers: [UInt] = Constants.initialHigherBadgeNumbers

    private var showBadgeNumbers: Bool = false {
        didSet {
            updateBadgeNumbers()
            updateBadgeButtons()
        }
    }

    private var useHigherBadgeNumbers: Bool = false {
        didSet {
            updateBadgeNumbers()
        }
    }

    private lazy var incrementBadgeButton: Button = {
        let button = Button()
        button.image = UIImage(named: "ic_fluent_add_20_regular")
        button.accessibilityLabel = "Increment badge numbers"
        button.addTarget(self, action: #selector(incrementBadgeNumbers), for: .touchUpInside)
        return button
    }()

    private lazy var decrementBadgeButton: Button = {
        let button = Button()
        button.image = UIImage(named: "ic_fluent_subtract_20_regular")
        button.accessibilityLabel = "Decrement badge numbers"
        button.addTarget(self, action: #selector(decrementBadgeNumbers), for: .touchUpInside)
        return button
    }()

    private lazy var homeItem: TabBarItem = {
        return TabBarItem(title: "Home", image: UIImage(named: "Home_28")!, selectedImage: UIImage(named: "Home_Selected_28")!)
    }()

    private func presentSideTabBar() {
        let contentViewController = UIViewController(nibName: nil, bundle: nil)
        self.contentViewController = contentViewController

        contentViewController.modalPresentationStyle = .fullScreen
        contentViewController.view.backgroundColor = view.backgroundColor
        contentViewController.view.addSubview(sideTabBar)

        sideTabBar.delegate = self

        sideTabBar.topItems = [
            homeItem,
            TabBarItem(title: "New", image: UIImage(named: "New_28")!, selectedImage: UIImage(named: "New_Selected_28")!),
            TabBarItem(title: "Open", image: UIImage(named: "Open_28")!, selectedImage: UIImage(named: "Open_Selected_28")!)
        ]

        // Set the Open item to be unread
        sideTabBar.topItems[2].isUnreadDotVisible = true

        var premiumImage = UIImage(named: "ic_fluent_premium_24_regular")!
        let primaryColor: UIColor = view.fluentTheme.color(.brandForegroundTint)
        premiumImage = premiumImage.withTintColor(primaryColor, renderingMode: .alwaysOriginal)

        sideTabBar.bottomItems = [
            TabBarItem(title: "Go Premium", image: premiumImage),
            TabBarItem(title: "Help", image: UIImage(named: "Help_24")!),
            TabBarItem(title: "Settings", image: UIImage(named: "ic_fluent_settings_24_regular")!)
        ]

        let contentView = UIView(frame: .zero)
        contentView.translatesAutoresizingMaskIntoConstraints = false
        contentViewController.view.insertSubview(contentView, belowSubview: sideTabBar)

        let optionTableView = UITableView(frame: .zero, style: .plain)
        optionTableView.translatesAutoresizingMaskIntoConstraints = false
        optionTableView.register(TableViewCell.self, forCellReuseIdentifier: TableViewCell.identifier)
        optionTableView.register(BooleanCell.self, forCellReuseIdentifier: BooleanCell.identifier)
        optionTableView.register(ActionsCell.self, forCellReuseIdentifier: ActionsCell.identifier)
        optionTableView.dataSource = self
        optionTableView.delegate = self
        optionTableView.separatorStyle = .none
        contentView.addSubview(optionTableView)

        showAvatarView(true)

        NSLayoutConstraint.activate([
            sideTabBar.leadingAnchor.constraint(equalTo: contentViewController.view.leadingAnchor),
            sideTabBar.topAnchor.constraint(equalTo: contentViewController.view.topAnchor),
            sideTabBar.bottomAnchor.constraint(equalTo: contentViewController.view.bottomAnchor),
            contentView.leadingAnchor.constraint(equalTo: sideTabBar.trailingAnchor),
            contentView.trailingAnchor.constraint(equalTo: contentViewController.view.trailingAnchor),
            contentView.topAnchor.constraint(equalTo: contentViewController.view.topAnchor),
            contentView.bottomAnchor.constraint(equalTo: contentViewController.view.bottomAnchor),
            optionTableView.topAnchor.constraint(equalTo: contentView.topAnchor),
            optionTableView.leadingAnchor.constraint(equalTo: contentView.leadingAnchor),
            optionTableView.trailingAnchor.constraint(equalTo: contentView.trailingAnchor),
            optionTableView.bottomAnchor.constraint(equalTo: contentView.bottomAnchor)
        ])

        present(contentViewController, animated: false)

        updateBadgeNumbers()
        updateBadgeButtons()
    }

    @objc private func showTooltipForHomeButton() {
        guard let view = sideTabBar.itemView(with: homeItem) else {
            return
        }

        Tooltip.shared.show(with: "Tap anywhere to dismiss this tooltip",
                            for: view,
                            preferredArrowDirection: .left,
                            offset: .init(x: 9, y: 0),
                            dismissOn: .tapAnywhere)
    }

    @objc private func dismissSideTabBar() {
        dismiss(animated: false) {
            self.navigationController?.popViewController(animated: true)
        }
    }

    @objc private func toggleAvatarView(_ cell: BooleanCell) {
        showAvatarView(cell.isOn)
    }

    @objc private func toggleShowBadgeNumbers(_ cell: BooleanCell) {
        showBadgeNumbers = cell.isOn
    }

    @objc private func toggleUseHigherBadgeNumbers(_ cell: BooleanCell) {
        useHigherBadgeNumbers = cell.isOn
    }

    @objc private func toggleShowTopItemTitles(_ cell: BooleanCell) {
        sideTabBar.showTopItemTitles = cell.isOn
    }

    @objc private func toggleShowBottomItemTitles(_ cell: BooleanCell) {
        sideTabBar.showBottomItemTitles = cell.isOn
    }

    private func showAvatarView(_ show: Bool) {
        var avatar: MSFAvatar?
        if let image = UIImage(named: "avatar_kat_larsson"), show {
            avatar = MSFAvatar(style: .accent,
                               size: .size32)
            if let avatarState = avatar?.state {
                avatarState.primaryText = "Kat Larson"
                avatarState.image = image
                avatarState.hasPointerInteraction = true
            }
        }

        sideTabBar.avatar = avatar
    }

    private func updateBadgeNumbers() {
        var numbers = useHigherBadgeNumbers ? higherBadgeNumbers : badgeNumbers
        if !showBadgeNumbers {
            numbers = [0, 0, 0, 0, 0, 0]
        }

        sideTabBar.topItems[0].setBadgeNumber(numbers[0])
        sideTabBar.topItems[1].setBadgeNumber(numbers[1])
        sideTabBar.topItems[2].setBadgeNumber(numbers[2])
        sideTabBar.bottomItems[0].setBadgeNumber(numbers[3])
        sideTabBar.bottomItems[1].setBadgeNumber(numbers[4])
        sideTabBar.bottomItems[2].setBadgeNumber(numbers[5])
    }

    private func updateBadgeButtons() {
        incrementBadgeButton.isEnabled = showBadgeNumbers
        decrementBadgeButton.isEnabled = showBadgeNumbers
    }

    private func modifyBadgeNumbers(increment: Int) {
        var numbers = useHigherBadgeNumbers ? higherBadgeNumbers : badgeNumbers
        for (index, value) in numbers.enumerated() {
            let newValue = Int(value) + increment
            if newValue > 0 {
                numbers[index] = UInt(newValue)
            } else {
                numbers[index] = 0
            }
        }

        if useHigherBadgeNumbers {
            higherBadgeNumbers = numbers
        } else {
            badgeNumbers = numbers
        }

        updateBadgeNumbers()
    }

    @objc private func incrementBadgeNumbers() {
        modifyBadgeNumbers(increment: 1)
    }

    @objc private func decrementBadgeNumbers() {
        modifyBadgeNumbers(increment: -1)
    }

    private let optionsCellItems: [CellItem] = {
        return [CellItem(title: "Show Avatar View", type: .boolean, action: #selector(toggleAvatarView(_:)), isOn: true),
                CellItem(title: "Show top item titles", type: .boolean, action: #selector(toggleShowTopItemTitles(_:))),
                CellItem(title: "Show bottom item titles", type: .boolean, action: #selector(toggleShowBottomItemTitles(_:))),
                CellItem(title: "Show badge numbers", type: .boolean, action: #selector(toggleShowBadgeNumbers(_:))),
                CellItem(title: "Use higher badge numbers", type: .boolean, action: #selector(toggleUseHigherBadgeNumbers(_:))),
                CellItem(title: "Modify badge numbers", type: .stepper, action: nil),
                CellItem(title: "Show Appearance Menu", type: .appearanceMenu, action: nil),
                CellItem(title: "Show tooltip for Home button", type: .action, action: #selector(showTooltipForHomeButton)),
                CellItem(title: "Dismiss", type: .action, action: #selector(dismissSideTabBar))
        ]
    }()
}

// MARK: - SideTabBarDemoController: SideTabBarDelegate

extension SideTabBarDemoController: SideTabBarDelegate {
    func sideTabBar(_ sideTabBar: SideTabBar, didSelect item: TabBarItem, fromTop: Bool) {
        let alert = UIAlertController(title: "\(item.title) was selected", message: nil, preferredStyle: .alert)
        let action = UIAlertAction(title: "OK", style: .default)
        alert.addAction(action)
        contentViewController?.present(alert, animated: true)
    }

    func sideTabBar(_ sideTabBar: SideTabBar, didActivate avatarView: MSFAvatar) {
        let alert = UIAlertController(title: "Avatar view was tapped", message: nil, preferredStyle: .alert)
        let action = UIAlertAction(title: "OK", style: .default)
        alert.addAction(action)
        contentViewController?.present(alert, animated: true)
    }
}

extension SideTabBarDemoController: UITableViewDataSource {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }

    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return optionsCellItems.count
    }

    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let item = optionsCellItems[indexPath.row]

        switch item.type {
        case .boolean:
            guard let cell = tableView.dequeueReusableCell(withIdentifier: BooleanCell.identifier) as? BooleanCell else {
                return UITableViewCell()
            }
            cell.setup(title: item.title, isOn: item.isOn)
            cell.titleNumberOfLines = 0
            cell.onValueChanged = { [weak self, weak cell] in
                self?.perform(item.action, with: cell)
            }
            return cell
        case .action:
            guard let cell = tableView.dequeueReusableCell(withIdentifier: ActionsCell.identifier) as? ActionsCell else {
                return UITableViewCell()
            }
            cell.setup(action1Title: item.title)
            if let action = item.action {
                cell.action1Button.addTarget(self, action: action, for: .touchUpInside)
            }
            cell.bottomSeparatorType = .full
            return cell
        case .stepper:
            guard let cell = tableView.dequeueReusableCell(withIdentifier: TableViewCell.identifier) as? TableViewCell else {
                return UITableViewCell()
            }

            let stackView = UIStackView(frame: CGRect(x: 0, y: 0, width: 100, height: 40))
            stackView.addArrangedSubview(decrementBadgeButton)
            stackView.addArrangedSubview(incrementBadgeButton)
            stackView.distribution = .fillEqually
            stackView.alignment = .center
            stackView.spacing = 4

            cell.setup(title: item.title, customAccessoryView: stackView)
            cell.titleNumberOfLines = 0
            return cell
        case .appearanceMenu:
            guard let cell = tableView.dequeueReusableCell(withIdentifier: TableViewCell.identifier) as? TableViewCell else {
                return UITableViewCell()
            }

            appearanceControlView.tintColor = view.fluentTheme.color(.foreground1)

            cell.setup(title: item.title, customAccessoryView: appearanceControlView)
            cell.titleNumberOfLines = 0
            return cell
        }
    }
}

// MARK: - SideTabBarDemoController: DemoAppearanceDelegate
extension SideTabBarDemoController: DemoAppearanceDelegate {
    func themeWideOverrideDidChange(isOverrideEnabled: Bool) {
        guard let fluentTheme = contentViewController?.view.window?.fluentTheme else {
            return
        }

        fluentTheme.register(tokenSetType: SideTabBarTokenSet.self,
                             tokenSet: isOverrideEnabled ? themeWideOverrideSideTabBarTokens : nil)
    }

    func perControlOverrideDidChange(isOverrideEnabled: Bool) {
        let tokens = (isOverrideEnabled ? perControlOverrideSideTabBarItemTokens : nil)
        sideTabBar.tokenSet.replaceAllOverrides(with: tokens)
    }

    func isThemeWideOverrideApplied() -> Bool {
        return contentViewController?.view.window?.fluentTheme.tokens(for: SideTabBarTokenSet.self) != nil
    }

    // MARK: - Custom tokens
    private var themeWideOverrideSideTabBarTokens: [SideTabBarTokenSet.Tokens: ControlTokenValue] {
        return [
            .tabBarItemSelectedColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.burgundy, .tint10),
                               lightHighContrast: GlobalTokens.sharedColor(.pumpkin, .tint10),
                               dark: GlobalTokens.sharedColor(.darkTeal, .tint40),
                               darkHighContrast: GlobalTokens.sharedColor(.teal, .tint40))
            },
            .tabBarItemUnselectedColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.darkTeal, .tint20),
                               lightHighContrast: GlobalTokens.sharedColor(.teal, .tint40),
                               dark: GlobalTokens.sharedColor(.pumpkin, .tint40),
                               darkHighContrast: GlobalTokens.sharedColor(.burgundy, .tint40))
            }
        ]
    }

    private var perControlOverrideSideTabBarItemTokens: [SideTabBarTokenSet.Tokens: ControlTokenValue] {
        return [
            .backgroundColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.grape, .tint10),
                               dark: GlobalTokens.sharedColor(.grape, .tint40))
            },
            .tabBarItemTitleLabelFontPortrait: .uiFont {
                return UIFont(descriptor: .init(name: "Papyrus", size: 20.0), size: 20.0)
            },
            .tabBarItemTitleLabelFontLandscape: .uiFont {
                return UIFont(descriptor: .init(name: "Papyrus", size: 20.0), size: 20.0)
            },
            .tabBarItemSelectedColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.burgundy, .tint10),
                               lightHighContrast: GlobalTokens.sharedColor(.pumpkin, .tint10),
                               dark: GlobalTokens.sharedColor(.darkTeal, .tint40),
                               darkHighContrast: GlobalTokens.sharedColor(.teal, .tint40))
            },
            .tabBarItemUnselectedColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.darkTeal, .tint20),
                               lightHighContrast: GlobalTokens.sharedColor(.teal, .tint40),
                               dark: GlobalTokens.sharedColor(.pumpkin, .tint40),
                               darkHighContrast: GlobalTokens.sharedColor(.burgundy, .tint40))
            },
            .separatorColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.hotPink, .tint10),
                               dark: GlobalTokens.sharedColor(.hotPink, .tint40))
            }
        ]
    }
}

extension SideTabBarDemoController: UITableViewDelegate {
    func tableView(_ tableView: UITableView, shouldHighlightRowAt indexPath: IndexPath) -> Bool {
       return false
    }
}

enum CellType {
    case action
    case boolean
    case stepper
    case appearanceMenu
}

struct CellItem {
    let title: String
    let type: CellType
    let action: Selector?
    var isOn: Bool = false
}
