//
//  Copyright (c) Microsoft Corporation. All rights reserved.
//  Licensed under the MIT License.
//

import FluentUI
import UIKit

// MARK: PeoplePickerSampleData

class PeoplePickerSampleData {
    struct Variant {
        let description: String
        let numberOfLines: Int
        let pickedPersonas: [Persona]
        let allowsPickedPersonasToAppearAsSuggested: Bool
        let showsSearchDirectoryButton: Bool
        let hidePersonaListViewWhenNoSuggestedPersonas: Bool
        let showsAvatar: Bool

        init(description: String, numberOfLines: Int = 0, pickedPersonas: [Persona] = [], allowsPickedPersonasToAppearAsSuggested: Bool = true, showsSearchDirectoryButton: Bool = true, hidePersonaListViewWhenNoSuggestedPersonas: Bool = false, showsAvatar: Bool = false) {
            self.description = description
            self.numberOfLines = numberOfLines
            self.pickedPersonas = pickedPersonas
            self.allowsPickedPersonasToAppearAsSuggested = allowsPickedPersonasToAppearAsSuggested
            self.showsSearchDirectoryButton = showsSearchDirectoryButton
            self.hidePersonaListViewWhenNoSuggestedPersonas = hidePersonaListViewWhenNoSuggestedPersonas
            self.showsAvatar = showsAvatar
        }
    }

    static let variants: [Variant] = [
        Variant(description: "Standard implementation with one line of picked personas", numberOfLines: 1, pickedPersonas: [samplePersonas[0], samplePersonas[4], samplePersonas[11], samplePersonas[14]]),
        Variant(description: "Doesn't allow picked personas to appear as suggested", pickedPersonas: [samplePersonas[0], samplePersonas[8]], allowsPickedPersonasToAppearAsSuggested: false),
        Variant(description: "Showcases persona's avatar", pickedPersonas: [samplePersonas[0], samplePersonas[1]], allowsPickedPersonasToAppearAsSuggested: false, showsAvatar: true),
        Variant(description: "Hides search directory button", pickedPersonas: [samplePersonas[13]], showsSearchDirectoryButton: false, hidePersonaListViewWhenNoSuggestedPersonas: true),
        Variant(description: "Includes callback when picking a suggested persona")
    ]
}

final class AsyncImageDemoPersona: PersonaData {
    static let samples: [AsyncImageDemoPersona] = samplePersonas.map { persona in
        AsyncImageDemoPersona(name: persona.name,
                              email: persona.email,
                              subtitle: persona.subtitle)
    }

    public func fetchImage(completion: @escaping (UIImage?) -> Void) {
        // for demo purposes, the "fetched" image is not being cached. The image will be "re-fetched" every time the cell appears on the screen.
        DispatchQueue.main.asyncAfter(deadline: .now() + 1) {
            let avatarImageName = "avatar_\(self.name.lowercased().replacingOccurrences(of: " ", with: "_"))"
            let image = UIImage(named: avatarImageName)
            completion(image)
        }
    }
}

// MARK: - PeoplePickerDemoController

class PeoplePickerDemoController: DemoController {
    var peoplePickers: [PeoplePicker] = []

    private let asyncImageSwitch = BrandedSwitch()

    override func viewDidLoad() {
        super.viewDidLoad()

        asyncImageSwitch.addTarget(self, action: #selector(onAsyncImageSwitchValueChanged), for: .valueChanged)
        setupView()
    }

    private func setupView() {
        container.addArrangedSubview(createAsyncImageToggle())
        container.addArrangedSubview(Separator())

        for (index, variant) in PeoplePickerSampleData.variants.enumerated() {
            addDescription(text: variant.description)
            addPeoplePicker(for: variant)
            if index != PeoplePickerSampleData.variants.count - 1 {
                container.addArrangedSubview(Separator())
            }
        }
    }

    private func addPeoplePicker(for variant: PeoplePickerSampleData.Variant) {
        let peoplePicker = PeoplePicker()
        peoplePicker.label = "Send to:"
        peoplePicker.availablePersonas = samplePersonas
        peoplePicker.pickedPersonas = variant.pickedPersonas
        peoplePicker.showsSearchDirectoryButton = variant.showsSearchDirectoryButton
        peoplePicker.numberOfLines = variant.numberOfLines
        peoplePicker.allowsPickedPersonasToAppearAsSuggested = variant.allowsPickedPersonasToAppearAsSuggested
        peoplePicker.hidePersonaListViewWhenNoSuggestedPersonas = variant.hidePersonaListViewWhenNoSuggestedPersonas
        peoplePicker.showsAvatar = variant.showsAvatar
        peoplePicker.delegate = self
        peoplePicker.isActive = true
        peoplePickers.append(peoplePicker)
        container.addArrangedSubview(peoplePicker)
    }

    private func createAsyncImageToggle() -> UIStackView {
        let asyncImageRow = UIStackView()
        asyncImageRow.axis = .horizontal
        asyncImageRow.alignment = .center
        asyncImageRow.distribution = .equalSpacing

        let asyncImageLabel = Label()
        asyncImageLabel.text = "Load persona images asynchronously"

        asyncImageRow.addArrangedSubview(asyncImageLabel)
        asyncImageRow.addArrangedSubview(asyncImageSwitch)
        return asyncImageRow
    }

    @objc private func onAsyncImageSwitchValueChanged() {
        for peoplePicker in peoplePickers {
            peoplePicker.availablePersonas = asyncImageSwitch.isOn ? AsyncImageDemoPersona.samples : samplePersonas
        }
    }
}

// MARK: - PeoplePickerDemoController: PeoplePickerDelegate

extension PeoplePickerDemoController: PeoplePickerDelegate {
    func peoplePicker(_ peoplePicker: PeoplePicker, personaFromText text: String) -> Persona {
        return samplePersonas.first { return $0.name.lowercased() == text.lowercased() } ?? PersonaData(name: text)
    }

    func peoplePicker(_ peoplePicker: PeoplePicker, personaIsValid persona: Persona) -> Bool {
        let availablePersonas = samplePersonas + searchDirectoryPersonas
        return availablePersonas.contains { $0.name == persona.name }
    }

    func peoplePicker(_ peoplePicker: PeoplePicker, didPickPersona persona: Persona) {
        if peoplePicker == peoplePickers.last {
            showMessage("\(persona.name) was picked")
        }
    }

    func peoplePicker(_ peoplePicker: PeoplePicker, didTapSelectedPersona persona: Persona) {
        peoplePicker.badge(for: persona)?.isSelected = false
        showMessage("\(persona.name) was tapped")
    }

    func peoplePicker(_ peoplePicker: PeoplePicker, searchDirectoryWithCompletion completion: @escaping ([Persona], Bool) -> Void) {
        // Delay added for 2 seconds to demo activity indicator
        DispatchQueue.main.asyncAfter(deadline: .now() + 2) {
            let text = peoplePicker.textFieldContent.lowercased()
            let personas = searchDirectoryPersonas.filter { $0.name.lowercased().contains(text) }
            completion(personas, true)
        }
    }

    func peoplePickerDidHidePersonaSuggestions(_ peoplePicker: PeoplePicker) {
        UIAccessibility.post(notification: .layoutChanged, argument: peoplePicker.badges.last)
    }
}

extension PeoplePickerDemoController: DemoAppearanceDelegate {
    func themeWideOverrideDidChange(isOverrideEnabled: Bool) {
        guard let fluentTheme = self.view.window?.fluentTheme else {
            return
        }

        fluentTheme.register(tokenSetType: PeoplePickerTokenSet.self, tokenSet: isOverrideEnabled ? themeWideOverridePeoplePickerTokens : nil)
    }

    func perControlOverrideDidChange(isOverrideEnabled: Bool) {
        for peoplePicker in peoplePickers {
            peoplePicker.tokenSet.replaceAllOverrides(with: isOverrideEnabled ? perControlOverridePeoplePickerTokens : nil)
        }
    }

    func isThemeWideOverrideApplied() -> Bool {
        return self.view.window?.fluentTheme.tokens(for: PeoplePickerTokenSet.self)?.isEmpty == false
    }

    // MARK: - Custom tokens
    private var themeWideOverridePeoplePickerTokens: [PeoplePickerTokenSet.Tokens: ControlTokenValue] {
        return [
            .backgroundColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.peach, .tint40),
                               dark: GlobalTokens.sharedColor(.peach, .shade30))
            }
        ]
    }

    private var perControlOverridePeoplePickerTokens: [PeoplePickerTokenSet.Tokens: ControlTokenValue] {
        return [
            .backgroundColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.hotPink, .tint40),
                               dark: GlobalTokens.sharedColor(.hotPink, .shade30))
            },
            .labelColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.navy, .tint40),
                               dark: GlobalTokens.sharedColor(.navy, .shade30))
            },
            .placeholderColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.navy, .tint40),
                               dark: GlobalTokens.sharedColor(.navy, .shade30))
            },
            .textFieldColor: .uiColor {
                return UIColor(light: GlobalTokens.sharedColor(.navy, .tint40),
                               dark: GlobalTokens.sharedColor(.navy, .shade30))
            },
            .labelFont: .uiFont {
                return UIFont(descriptor: .init(name: "Times", size: 18.0), size: 18.0)
            },
            .placeholderFont: .uiFont {
                return UIFont(descriptor: .init(name: "Times", size: 18.0), size: 18.0)
            },
            .textFieldFont: .uiFont {
                return UIFont(descriptor: .init(name: "Times", size: 18.0), size: 18.0)
            }
        ]
    }
}
