﻿//------------------------------------------------------------------------------
// <copyright file="LogMessage.cs" company="Microsoft">
// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
// </copyright>
//------------------------------------------------------------------------------
using System;

namespace Microsoft.Azure.Kinect.Sensor
{
    /// <summary>
    /// A debug log message that is part of an exception.
    /// </summary>
    public class LogMessage
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="LogMessage"/> class.
        /// </summary>
        /// <param name="time">The time at which the log message was received.</param>
        /// <param name="logLevel">The level of the log message that was created.</param>
        /// <param name="fileName">The file name of the source file that generated the message.</param>
        /// <param name="line">The line number of the source file that generated the message.</param>
        /// <param name="message">The messaged generated by the Azure Kinect SDK.</param>
        internal LogMessage(DateTime time, LogLevel logLevel, string fileName, int line, string message)
        {
            this.Time = time;
            this.LogLevel = logLevel;
            this.FileName = fileName ?? throw new ArgumentNullException(nameof(fileName));
            this.Line = line;
            this.Message = message ?? throw new ArgumentNullException(nameof(message));
        }

        /// <summary>
        /// Gets the time at which the log message was received.
        /// </summary>
        public DateTime Time { get; }

        /// <summary>
        /// Gets the level of the message that was created.
        /// </summary>
        public LogLevel LogLevel { get; }

        /// <summary>
        /// Gets the file name of the source file that generated the message.
        /// </summary>
        public string FileName { get; }

        /// <summary>
        /// Gets the line number of the source file that generated the message.
        /// </summary>
        public int Line { get; }

        /// <summary>
        /// Gets the messaged generated by the Azure Kinect SDK.
        /// </summary>
        public string Message { get; }

        /// <summary>
        /// Gets the full formated exception log message with all of the details.
        /// </summary>
        public string FormatedMessage
        {
            get
            {
                return $"{this.Time} [{this.LogLevel}] {this.FileName}@{this.Line}: {this.Message}";
            }
        }

        /// <inheritdoc/>
        public override string ToString()
        {
            return this.Message;
        }
    }
}
