/****************************************************************************
 * sched/signal/sig_pause.c
 *
 *   Copyright (C) 2012, 2016 Gregory Nutt. All rights reserved.
 *   Author: Gregory Nutt <gnutt@nuttx.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name NuttX nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <unistd.h>
#include <signal.h>
#include <errno.h>

#include <nuttx/signal.h>
#include <nuttx/cancelpt.h>

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: pause
 *
 * Description:
 *   The pause() function will suspend the calling thread until delivery of a
 *   non-blocked signal.
 *
 * Input Parameters:
 *   None
 *
 * Returned Value:
 *   Since pause() suspends thread execution indefinitely unless interrupted
 *   a signal, there is no successful completion return value. A value of -1
 *   will always be returned and errno set to indicate the error (EINTR).
 *
 * POSIX compatibility:
 *   In the POSIX description of this function is the pause() function will
 *   suspend the calling thread until delivery of a signal whose action is
 *   either to execute a signal-catching function or to terminate the
 *   process.  This implementation only waits for any non-blocked signal
 *   to be received.
 *
 ****************************************************************************/

int pause(void)
{
  sigset_t set;
  int ret;

  /* pause() is a cancellation point */

  (void)enter_cancellation_point();

  /* Set up for the sleep.  Using the empty set means that we are not
   * waiting for any particular signal.  However, any unmasked signal
   * can still awaken sigtimedwait().
   */

  (void)sigemptyset(&set);

  /* sigtwaitinfo() cannot succeed.  It should always return error EINTR
   * meaning that some unblocked signal was caught.
   */

  ret = nxsig_waitinfo(&set, NULL);
  if (ret < 0)
    {
      set_errno(-ret);
      ret = ERROR;
    }

  leave_cancellation_point();
  return ret;
}
