#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
chemdataextractor.text.latex
~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Tools for converting LaTeX to unicode.

"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from __future__ import unicode_literals
import re
import string

import six

from . import NAME_SMALL, SMALL


def latex_to_unicode(text, capitalize=False):
    """Replace LaTeX entities with the equivalent unicode and optionally capitalize.

    :param text: The LaTeX string to be converted
    :param capitalize: Can be 'sentence', 'name', 'title', 'upper', 'lower'
    """
    if capitalize:
        res = []
        brac_count = 0
        for i, c in enumerate(text):
            if c == '{':
                brac_count += 1
            if c == '}':
                brac_count -= 1
            if brac_count > 0:
                res.append(c)
            elif capitalize == 'upper' or (i == 0 and not capitalize == 'lower'):
                res.append(c.upper())
            elif capitalize == 'sentence' and (i > 2 and text[i - 1] == ' ' and text[i - 2] == '.'):
                res.append(c.upper())
            elif (capitalize == 'name' and text[i - 1] in [' ', '-']) or (capitalize == 'title' and text[i - 1] == ' '):
                nextword = text[i:].split(' ', 1)[0].rstrip(string.punctuation)
                nextword = nextword[:1].lower() + nextword[1:] if text else ''
                if capitalize == 'name' and nextword in NAME_SMALL:
                    res.append(c.lower())
                elif capitalize == 'title' and nextword in SMALL:
                    res.append(c.lower())
                else:
                    res.append(c.upper())
            elif capitalize == 'name' and c == c.upper():
                n1 = text[i - 1] if i > 0 else None
                n2 = text[i - 2] if i > 1 else None
                n3 = text[i - 3] if i > 2 else None
                n4 = text[i - 4] if i > 3 else None
                if n2 == 'M' and n1 == 'c' and (not n3 or n3 == ' '):
                    res.append(c)  # McCartney
                elif n2 == 'O' and n1 == '\'' and (not n3 or n3 == ' '):
                    res.append(c)  # O'Boyle
                elif n3 == 'M' and n2 == 'a' and n1 == 'c' and (not n4 or n4 == ' '):
                    res.append(c)  # MacGarry
                else:
                    res.append(c.lower())
            else:
                res.append(c.lower())
        text = ''.join(res)
    if any(i in text for i in ['\\', '{', '}', '$', '&', '%', '#', '_']):
        for k, v in six.iteritems(LATEX_MAPPINGS):
            text = text.replace(k, v)
        for k, v in six.iteritems(LATEX_SUB_MAPPINGS):
            text = text.replace(k, v)
        for mod in ['mathbb', 'mathbf', 'mathbit', 'mathfrak', 'mathrm', 'mathscr', 'mathsf', 'mathsfbf', 'mathsfbfsl', 
                    'mathsfsl', 'mathsl', 'mathslbb', 'mathtt']:
            text = re.sub(r'\\%s\{([\\\w]+)\}' % mod, r'\1', text)
        for k, v in six.iteritems(LATEX_SUB_SUB_MAPPINGS):
            text = text.replace(k, v)
        for k, v in six.iteritems(LATEX_COMBINING_CHARS):
            text = re.sub(r'%s\{?(\w)\}?' % k, r'\1%s' % v, text)
        text = re.sub(r'\\noopsort\{.*?\}', r'', text)
        text = re.sub(r'\\path\|(.*?)\|', r'\1', text)
        text = re.sub(r'(?<!\\)[{}$]', r'', text)
        text = re.sub(r'\\([{}$&_])', r'\1', text)
    return text

# The LaTeX characters apply to the subsequent character, while the unicode applies to the previous character
# The LaTeX keys are regular expressions
LATEX_COMBINING_CHARS = {
    r"\\=": "\u0304",
    r"\\'": "\u0301",
    r"\\\"": "\u0308",
    r"\\\.": "\u0307",
    r"\\\^": "\u0302",
    r"\\~": "\u0303",
    r"\\`": "\u0300",
    r"\\cyrchar\\C": "\u030F",
    r"\\c": "\u0327",
    r"\\H": "\u030B",
    r"\\r": "\u030A",
    r"\\u": "\u0306",
    r"\\v": "\u030C",
    r"\\k": "\u0328",
}

# These mapping are substring of other mappings, so must ensure they are done last
LATEX_SUB_MAPPINGS = {
    "{{/}\\!\\!{/}}": "\u2AFD",
    "\\-": "\u00AD",  # Soft hyphen (hyphenation "hint") - removed by normalize function
    "\\approx ": "\u2248",
    "\\cal": "",
    "\\cong ": "\u2245",
    "\\ddot{\\iota}": "\u03CA",
    "\\ddot{\\upsilon}": "\u03CB",
    "\\em ": "",
    "\\emph": "",
    "\\equiv ": "\u2261",
    "\\geq ": "\u2265",
    "\\hspace{-.9ex}": "",
    "\\hspace{.2ex}": "",
    "\\in ": "\u2208",
    "\\int\\!\\int ": "\u222C",
    "\\it ": "",
    "\\leq ": "\u2264",
    "\\mbox": "",
    "\\mu": "\u03BC",
    "\\ni ": "\u220B",
    "\\prec ": "\u227A",
    "\\preceq ": "\u2AAF",
    "\\raisebox{.5ex}": "",
    "\\raisebox{1.3ex}": "",
    "\\raisebox{2ex}": "",
    "\\raisebox{-.3ex}": "",
    "\\rotatebox[]{180}": "",
    "\\scriptsize ": "",
    "\\sim ": "\u223C",
    "\\simeq ": "\u2243",
    "\\sqsubseteq ": "\u2291",
    "\\sqsupseteq ": "\u2292",
    "\\subset ": "\u2282",
    "\\subseteq ": "\u2286",
    "\\succ ": "\u227B",
    "\\succeq ": "\u2AB0",
    "\\supset ": "\u2283",
    "\\supseteq ": "\u2287",
    "\\times ": "\u00D7 ",
    "\\tt ": "",
}

# Further substrings and greek letters that may appear after stripping math fonts
LATEX_SUB_SUB_MAPPINGS = {
    "\\i": "\u0131",
    "\\l": "\u0142",
    "\\L": "\u0141",
    "\\O": "\u00D8",
    "\\o": "\u00F8",
    "\\int ": "\u222B",
    "\\Alpha ": "\u0391",
    "\\alpha ": "\u03B1",
    "\\Beta ": "\u0392",
    "\\beta ": "\u03B2",
    "\\Chi ": "\u03A7",
    "\\chi ": "\u03C7",
    "\\Delta ": "\u0394",
    "\\delta ": "\u03B4",
    "\\Epsilon ": "\u0395",
    "\\epsilon ": "\u03B5",
    "\\Eta ": "\u0397",
    "\\eta ": "\u03B7",
    "\\Gamma ": "\u0393",
    "\\gamma ": "\u03B3",
    "\\Iota ": "\u0399",
    "\\iota ": "\u03B9",
    "\\Kappa ": "\u039A",
    "\\kappa ": "\u03BA",
    "\\Lambda ": "\u039B",
    "\\lambda ": "\u03BB",
    "\\nabla ": "\u2207",
    "\\Omega ": "\u03A9",
    "\\omega ": "\u03C9",
    "\\Phi ": "\u03A6",
    "\\phi ": "\u03D5",
    "\\Pi ": "\u03A0",
    "\\pi ": "\u03C0",
    "\\Psi ": "\u03A8",
    "\\psi ": "\u03C8",
    "\\Rho ": "\u03A1",
    "\\rho ": "\u03C1",
    "\\Sigma ": "\u03A3",
    "\\sigma ": "\u03C3",
    "\\Tau ": "\u03A4",
    "\\tau ": "\u03C4",
    "\\theta ": "\u03B8",
    "\\Theta ": "\u0398",
    "\\Upsilon ": "\u03A5",
    "\\upsilon ": "\u03C5",
    "\\varepsilon ": "\u025B",
    "\\varkappa ": "\u03F0",
    "\\varphi ": "\u03C6",
    "\\varpi ": "\u03D6",
    "\\varrho ": "\u03F1",
    "\\varsigma ": "\u03C2",
    "\\vartheta ": "\u03D1",
    "\\Xi ": "\u039E",
    "\\xi ": "\u03BE",
    "\\Zeta ": "\u0396",
    "\\zeta ": "\u03B6",
}


# Unicode to LaTeX character mappings, derived from https://gist.github.com/piquadrat/798549
LATEX_MAPPINGS = {
    ",,": "\u201E",
    ":=": "\u2254",
    "''''": "\u2057",
    "{_\\ast}": "\u2217",
    "{'''}": "\u2034",
    "{''}": "\u2033",
    "{'}": "\u2032",
    "{\\LaTeX}": "LaTeX",
    "{\\TeX}": "TeX",
    "{\\fontencoding{LECO}\\selectfont\\char177}": "\u0311",
    "{\\fontencoding{LECO}\\selectfont\\char184}": "\u0318",
    "{\\fontencoding{LECO}\\selectfont\\char185}": "\u0319",
    "{\\fontencoding{LECO}\\selectfont\\char203}": "\u032B",
    "{\\fontencoding{LECO}\\selectfont\\char207}": "\u032F",
    "{\\fontencoding{LECO}\\selectfont\\char215}": "\u0337",
    "{\\fontencoding{LECO}\\selectfont\\char216}": "\u0338",
    "{\\fontencoding{LECO}\\selectfont\\char218}": "\u033A",
    "{\\fontencoding{LECO}\\selectfont\\char219}": "\u033B",
    "{\\fontencoding{LECO}\\selectfont\\char220}": "\u033C",
    "{\\fontencoding{LECO}\\selectfont\\char221}": "\u033D",
    "{\\fontencoding{LECO}\\selectfont\\char225}": "\u0361",
    "{\\fontencoding{LEIP}\\selectfont\\char202}": "\u027F",
    "{\\fontencoding{LEIP}\\selectfont\\char61}": "\u0258",
    "{\\fontencoding{LELA}\\selectfont\\char195}": "\u01BA",
    "{\\fontencoding{LELA}\\selectfont\\char201}": "\u013F",
    "{\\fontencoding{LELA}\\selectfont\\char202}": "\u0140",
    "{\\fontencoding{LELA}\\selectfont\\char40}": "\u0126",
    "{\\fontencoding{LELA}\\selectfont\\char47}": "\u0166",
    "{\\fontencoding{LELA}\\selectfont\\char63}": "\u0167",
    "{\\fontencoding{LELA}\\selectfont\\char91}": "\u0138",
    "{\\user}": "\u007E",
    "{\\rlap{\\textbackslash}{{/}\\!\\!{/}}}": "\u2AFD\u20E5",
    "{^1}": "\u00B9",
    "{^2}": "\u00B2",
    "{^3}": "\u00B3",
    "\\\\": "\n",
    "\\_": "\u005F",
    "\\;": " ",   # \u2009?
    "\\.{C}": "\u010A",
    "\\.{c}": "\u010B",
    "\\.{E}": "\u0116",
    "\\.{e}": "\u0117",
    "\\.{G}": "\u0120",
    "\\.{g}": "\u0121",
    "\\.{I}": "\u0130",
    "\\.{Z}": "\u017B",
    "\\.{z}": "\u017C",
    "\\'{}{I}": "\u038A",
    "\\'{}O": "\u038C",
    "\\'{\\i}": "\u00ED",
    "\\'{$\\alpha$}": "\u03AC",
    "\\'{A}": "\u00C1",
    "\\'{a}": "\u00E1",
    "\\'{C}": "\u0106",
    "\\'{c}": "\u0107",
    "\\'{E}": "\u00C9",
    "\\'{e}": "\u00E9",
    "\\'{g}": "\u01F5",
    "\\'{H}": "\u0389",
    "\\'{I}": "\u00CD",
    "\\'{L}": "\u0139",
    "\\'{l}": "\u013A",
    "\\'{N}": "\u0143",
    "\\'{n}": "\u0144",
    "\\'{O}": "\u00D3",
    "\\'{o}": "\u00F3",
    "\\'{R}": "\u0154",
    "\\'{r}": "\u0155",
    "\\'{S}": "\u015A",
    "\\'{s}": "\u015B",
    "\\'{U}": "\u00DA",
    "\\'{u}": "\u00FA",
    "\\'{Y}": "\u00DD",
    "\\'{y}": "\u00FD",
    "\\'{Z}": "\u0179",
    "\\'{z}": "\u017A",
    "\\\"{\\i}": "\u00EF",
    "\\\"{A}": "\u00C4",
    "\\\"{a}": "\u00E4",
    "\\\"{E}": "\u00CB",
    "\\\"{e}": "\u00EB",
    "\\\"{I}": "\u00CF",
    "\\\"{O}": "\u00D6",
    "\\\"{o}": "\u00F6",
    "\\\"{U}": "\u00DC",
    "\\\"{u}": "\u00FC",
    "\\\"{y}": "\u00FF",
    "\\\"{Y}": "\u0178",
    "\\&amp;": "\u0026",
    "\\#": "\u0023",
    "\\%": "\u0025",
    "\\`{\\i}": "\u00EC",
    "\\`{A}": "\u00C0",
    "\\`{a}": "\u00E0",
    "\\`{E}": "\u00C8",
    "\\`{e}": "\u00E8",
    "\\`{I}": "\u00CC",
    "\\`{O}": "\u00D2",
    "\\`{o}": "\u00F2",
    "\\`{U}": "\u00D9",
    "\\`{u}": "\u00F9",
    "\\^{}": "\u005E",
    "\\^{\\i}": "\u00EE",
    "\\^{\\j}": "\u0135",
    "\\^{A}": "\u00C2",
    "\\^{a}": "\u00E2",
    "\\^{C}": "\u0108",
    "\\^{c}": "\u0109",
    "\\^{E}": "\u00CA",
    "\\^{e}": "\u00EA",
    "\\^{G}": "\u011C",
    "\\^{g}": "\u011D",
    "\\^{H}": "\u0124",
    "\\^{h}": "\u0125",
    "\\^{I}": "\u00CE",
    "\\^{J}": "\u0134",
    "\\^{O}": "\u00D4",
    "\\^{o}": "\u00F4",
    "\\^{S}": "\u015C",
    "\\^{s}": "\u015D",
    "\\^{U}": "\u00DB",
    "\\^{u}": "\u00FB",
    "\\^{W}": "\u0174",
    "\\^{w}": "\u0175",
    "\\^{Y}": "\u0176",
    "\\^{y}": "\u0177",
    "\\={\\i}": "\u012B",
    "\\={A}": "\u0100",
    "\\={a}": "\u0101",
    "\\={E}": "\u0112",
    "\\={e}": "\u0113",
    "\\={I}": "\u012A",
    "\\={O}": "\u014C",
    "\\={o}": "\u014D",
    "\\={U}": "\u016A",
    "\\={u}": "\u016B",
    "\\~{\\i}": "\u0129",
    "\\~{A}": "\u00C3",
    "\\~{a}": "\u00E3",
    "\\~{I}": "\u0128",
    "\\~{N}": "\u00D1",
    "\\~{n}": "\u00F1",
    "\\~{O}": "\u00D5",
    "\\~{o}": "\u00F5",
    "\\~{U}": "\u0168",
    "\\~{u}": "\u0169",
    "\\aa": "\u00E5",
    "\\AA": "\u212B",
    "\\acute{\\ddot{\\iota}}": "\u0390",
    "\\acute{\\ddot{\\upsilon}}": "\u03B0",
    "\\acute{\\epsilon}": "\u03AD",
    "\\acute{\\eta}": "\u03AE",
    "\\acute{\\iota}": "\u03AF",
    "\\acute{\\omega}": "\u03CE",
    "\\acute{\\upsilon}": "\u03CD",
    "\\AE": "\u00C6",
    "\\ae": "\u00E6",
    "\\aleph ": "\u2135",
    "\\allequal ": "\u224C",
    "\\Alpha ": "\u0391",
    "\\alpha ": "\u03B1",
    "\\amalg ": "\u2A3F",
    "\\angle ": "\u2220",
    "\\Angle ": "\u299C",
    "\\approxeq ": "\u224A",
    "\\approxnotequal ": "\u2246",
    "\\aquarius ": "\u2652",
    "\\aries ": "\u2648",
    "\\arrowwaveright ": "\u219D",
    "\\ast ": "\u002A",
    "\\asymp ": "\u224D",
    "\\backepsilon ": "\u03F6",
    "\\backprime ": "\u2035",
    "\\backsim ": "\u223D",
    "\\backsimeq ": "\u22CD",
    "\\barwedge ": "\u2305",
    "\\because ": "\u2235",
    "\\Beta ": "\u0392",
    "\\beta ": "\u03B2",
    "\\beth ": "\u2136",
    "\\between ": "\u226C",
    "\\bigcap ": "\u22C2",
    "\\bigcirc ": "\u25EF",
    "\\bigcup ": "\u22C3",
    "\\bigodot ": "\u2299",
    "\\bigoplus ": "\u2295",
    "\\bigotimes ": "\u2297",
    "\\bigsqcup ": "\u2294",
    "\\bigstar ": "\u2605",
    "\\bigtriangledown ": "\u25BD",
    "\\bigtriangleup ": "\u25B3",
    "\\biguplus ": "\u228E",
    "\\bigvee ": "\u22C1",
    "\\bigwedge ": "\u22C0",
    "\\blacklozenge ": "\u29EB",
    "\\blacksquare ": "\u25AA",
    "\\blacktriangle ": "\u25B4",
    "\\blacktriangledown ": "\u25BE",
    "\\blacktriangleleft ": "\u25C2",
    "\\blacktriangleright ": "\u25B8",
    "\\bot ": "\u22A5",
    "\\bowtie ": "\u22C8",
    "\\box ": "\u25A1",
    "\\boxdot ": "\u22A1",
    "\\boxminus ": "\u229F",
    "\\boxplus ": "\u229E",
    "\\boxtimes ": "\u22A0",
    "\\bullet ": "\u2219",
    "\\Bumpeq ": "\u224E",
    "\\bumpeq ": "\u224F",
    "\\c{}": "\u00B8",
    "\\c{C}": "\u00C7",
    "\\c{c}": "\u00E7",
    "\\c{G}": "\u0122",
    "\\c{g}": "\u0123",
    "\\c{K}": "\u0136",
    "\\c{k}": "\u0137",
    "\\c{L}": "\u013B",
    "\\c{l}": "\u013C",
    "\\c{N}": "\u0145",
    "\\c{n}": "\u0146",
    "\\c{R}": "\u0156",
    "\\c{r}": "\u0157",
    "\\c{S}": "\u015E",
    "\\c{s}": "\u015F",
    "\\c{T}": "\u0162",
    "\\c{t}": "\u0163",
    "\\cancer ": "\u264B",
    "\\cap ": "\u2229",
    "\\Cap ": "\u22D2",
    "\\capricornus ": "\u2651",
    "\\cdot ": "\u00B7",
    "\\cdots ": "\u22EF",
    "\\centerdot ": "\u22C5",
    "\\cents ": "\u00A2",
    "\\Chi ": "\u03A7",
    "\\chi ": "\u03C7",
    "\\chi_t": "\u03C7t",
    "\\circ ": "\u2218",
    "\\circeq ": "\u2257",
    "\\circlearrowleft ": "\u21BA",
    "\\circlearrowright ": "\u21BB",
    "\\circledast ": "\u229B",
    "\\circledcirc ": "\u229A",
    "\\circleddash ": "\u229D",
    "\\circledS ": "\u24C8",
    "\\clockoint ": "\u2A0F",
    "\\clubsuit ": "\u2663",
    "\\clwintegral ": "\u2231",
    "\\Colon ": "\u2237",
    "\\complement ": "\u2201",
    "\\coprod ": "\u2210",
    "\\copyright ": "\u00A9",
    "\\cup ": "\u222A",
    "\\Cup ": "\u22D3",
    "\\curlyeqprec ": "\u22DE",
    "\\curlyeqsucc ": "\u22DF",
    "\\curlyvee ": "\u22CE",
    "\\curlywedge ": "\u22CF",
    "\\curvearrowleft ": "\u21B6",
    "\\curvearrowright ": "\u21B7",
    "\\cyrchar{\\'\\CYRG}": "\u0403",
    "\\cyrchar{\\'\\cyrg}": "\u0453",
    "\\cyrchar{\\'\\CYRK}": "\u040C",
    "\\cyrchar{\\'\\cyrk}": "\u045C",
    "\\cyrchar\\CYRA ": "\u0410",
    "\\cyrchar\\cyra ": "\u0430",
    "\\cyrchar\\CYRABHCH ": "\u04BC",
    "\\cyrchar\\cyrabhch ": "\u04BD",
    "\\cyrchar\\CYRABHCHDSC ": "\u04BE",
    "\\cyrchar\\cyrabhchdsc ": "\u04BF",
    "\\cyrchar\\CYRABHDZE ": "\u04E0",
    "\\cyrchar\\cyrabhdze ": "\u04E1",
    "\\cyrchar\\CYRABHHA ": "\u04A8",
    "\\cyrchar\\cyrabhha ": "\u04A9",
    "\\cyrchar\\CYRAE ": "\u04D4",
    "\\cyrchar\\cyrae ": "\u04D5",
    "\\cyrchar\\CYRB ": "\u0411",
    "\\cyrchar\\cyrb ": "\u0431",
    "\\cyrchar\\CYRBYUS ": "\u046A",
    "\\cyrchar\\CYRC ": "\u0426",
    "\\cyrchar\\cyrc ": "\u0446",
    "\\cyrchar\\CYRCH ": "\u0427",
    "\\cyrchar\\cyrch ": "\u0447",
    "\\cyrchar\\CYRCHLDSC ": "\u04CB",
    "\\cyrchar\\cyrchldsc ": "\u04CC",
    "\\cyrchar\\CYRCHRDSC ": "\u04B6",
    "\\cyrchar\\cyrchrdsc ": "\u04B7",
    "\\cyrchar\\CYRCHVCRS ": "\u04B8",
    "\\cyrchar\\cyrchvcrs ": "\u04B9",
    "\\cyrchar\\CYRD ": "\u0414",
    "\\cyrchar\\cyrd ": "\u0434",
    "\\cyrchar\\CYRDJE ": "\u0402",
    "\\cyrchar\\cyrdje ": "\u0452",
    "\\cyrchar\\CYRDZE ": "\u0405",
    "\\cyrchar\\cyrdze ": "\u0455",
    "\\cyrchar\\CYRDZHE ": "\u040F",
    "\\cyrchar\\cyrdzhe ": "\u045F",
    "\\cyrchar\\CYRE ": "\u0415",
    "\\cyrchar\\cyre ": "\u0435",
    "\\cyrchar\\CYREREV ": "\u042D",
    "\\cyrchar\\cyrerev ": "\u044D",
    "\\cyrchar\\CYRERY ": "\u042B",
    "\\cyrchar\\cyrery ": "\u044B",
    "\\cyrchar\\CYRF ": "\u0424",
    "\\cyrchar\\cyrf ": "\u0444",
    "\\cyrchar\\CYRFITA ": "\u0472",
    "\\cyrchar\\CYRG ": "\u0413",
    "\\cyrchar\\cyrg ": "\u0433",
    "\\cyrchar\\CYRGHCRS ": "\u0492",
    "\\cyrchar\\cyrghcrs ": "\u0493",
    "\\cyrchar\\CYRGHK ": "\u0494",
    "\\cyrchar\\cyrghk ": "\u0495",
    "\\cyrchar\\CYRGUP ": "\u0490",
    "\\cyrchar\\cyrgup ": "\u0491",
    "\\cyrchar\\CYRH ": "\u0425",
    "\\cyrchar\\cyrh ": "\u0445",
    "\\cyrchar\\CYRHDSC ": "\u04B2",
    "\\cyrchar\\cyrhdsc ": "\u04B3",
    "\\cyrchar\\CYRHRDSN ": "\u042A",
    "\\cyrchar\\cyrhrdsn ": "\u044A",
    "\\cyrchar\\cyrhundredthousands ": "\u0488",
    "\\cyrchar\\CYRI ": "\u0418",
    "\\cyrchar\\cyri ": "\u0438",
    "\\cyrchar\\CYRIE ": "\u0404",
    "\\cyrchar\\cyrie ": "\u0454",
    "\\cyrchar\\CYRII ": "\u0406",
    "\\cyrchar\\cyrii ": "\u0456",
    "\\cyrchar\\CYRIOTBYUS ": "\u046C",
    "\\cyrchar\\cyriotbyus ": "\u046D",
    "\\cyrchar\\CYRIOTE ": "\u0464",
    "\\cyrchar\\cyriote ": "\u0465",
    "\\cyrchar\\CYRIOTLYUS ": "\u0468",
    "\\cyrchar\\cyriotlyus ": "\u0469",
    "\\cyrchar\\CYRISHRT ": "\u0419",
    "\\cyrchar\\cyrishrt ": "\u0439",
    "\\cyrchar\\CYRIZH ": "\u0474",
    "\\cyrchar\\CYRJE ": "\u0408",
    "\\cyrchar\\cyrje ": "\u0458",
    "\\cyrchar\\CYRK ": "\u041A",
    "\\cyrchar\\cyrk ": "\u043A",
    "\\cyrchar\\CYRKBEAK ": "\u04A0",
    "\\cyrchar\\cyrkbeak ": "\u04A1",
    "\\cyrchar\\CYRKDSC ": "\u049A",
    "\\cyrchar\\cyrkdsc ": "\u049B",
    "\\cyrchar\\CYRKHCRS ": "\u049E",
    "\\cyrchar\\cyrkhcrs ": "\u049F",
    "\\cyrchar\\CYRKHK ": "\u04C3",
    "\\cyrchar\\cyrkhk ": "\u04C4",
    "\\cyrchar\\CYRKOPPA ": "\u0480",
    "\\cyrchar\\cyrkoppa ": "\u0481",
    "\\cyrchar\\CYRKSI ": "\u046E",
    "\\cyrchar\\cyrksi ": "\u046F",
    "\\cyrchar\\CYRKVCRS ": "\u049C",
    "\\cyrchar\\cyrkvcrs ": "\u049D",
    "\\cyrchar\\CYRL ": "\u041B",
    "\\cyrchar\\cyrl ": "\u043B",
    "\\cyrchar\\CYRLJE ": "\u0409",
    "\\cyrchar\\cyrlje ": "\u0459",
    "\\cyrchar\\CYRLYUS ": "\u0466",
    "\\cyrchar\\cyrlyus ": "\u0467",
    "\\cyrchar\\CYRM ": "\u041C",
    "\\cyrchar\\cyrm ": "\u043C",
    "\\cyrchar\\cyrmillions ": "\u0489",
    "\\cyrchar\\CYRN ": "\u041D",
    "\\cyrchar\\cyrn ": "\u043D",
    "\\cyrchar\\CYRNDSC ": "\u04A2",
    "\\cyrchar\\cyrndsc ": "\u04A3",
    "\\cyrchar\\CYRNG ": "\u04A4",
    "\\cyrchar\\cyrng ": "\u04A5",
    "\\cyrchar\\CYRNHK ": "\u04C7",
    "\\cyrchar\\cyrnhk ": "\u04C8",
    "\\cyrchar\\CYRNJE ": "\u040A",
    "\\cyrchar\\cyrnje ": "\u045A",
    "\\cyrchar\\CYRO ": "\u041E",
    "\\cyrchar\\cyro ": "\u043E",
    "\\cyrchar\\CYROMEGA ": "\u0460",
    "\\cyrchar\\cyromega ": "\u0461",
    "\\cyrchar\\CYROMEGARND ": "\u047A",
    "\\cyrchar\\cyromegarnd ": "\u047B",
    "\\cyrchar\\CYROMEGATITLO ": "\u047C",
    "\\cyrchar\\cyromegatitlo ": "\u047D",
    "\\cyrchar\\CYROT ": "\u047E",
    "\\cyrchar\\cyrot ": "\u047F",
    "\\cyrchar\\CYROTLD ": "\u04E8",
    "\\cyrchar\\cyrotld ": "\u04E9",
    "\\cyrchar\\CYRP ": "\u041F",
    "\\cyrchar\\cyrp ": "\u043F",
    "\\cyrchar\\CYRpalochka ": "\u04C0",
    "\\cyrchar\\CYRPHK ": "\u04A6",
    "\\cyrchar\\cyrphk ": "\u04A7",
    "\\cyrchar\\CYRPSI ": "\u0470",
    "\\cyrchar\\cyrpsi ": "\u0471",
    "\\cyrchar\\CYRR ": "\u0420",
    "\\cyrchar\\cyrr ": "\u0440",
    "\\cyrchar\\CYRRTICK ": "\u048E",
    "\\cyrchar\\cyrrtick ": "\u048F",
    "\\cyrchar\\CYRS ": "\u0421",
    "\\cyrchar\\cyrs ": "\u0441",
    "\\cyrchar\\CYRSCHWA ": "\u04D8",
    "\\cyrchar\\cyrschwa ": "\u04D9",
    "\\cyrchar\\CYRSDSC ": "\u04AA",
    "\\cyrchar\\cyrsdsc ": "\u04AB",
    "\\cyrchar\\CYRSEMISFTSN ": "\u048C",
    "\\cyrchar\\cyrsemisftsn ": "\u048D",
    "\\cyrchar\\CYRSFTSN ": "\u042C",
    "\\cyrchar\\cyrsftsn ": "\u044C",
    "\\cyrchar\\CYRSH ": "\u0428",
    "\\cyrchar\\cyrsh ": "\u0448",
    "\\cyrchar\\CYRSHCH ": "\u0429",
    "\\cyrchar\\cyrshch ": "\u0449",
    "\\cyrchar\\CYRSHHA ": "\u04BA",
    "\\cyrchar\\cyrshha ": "\u04BB",
    "\\cyrchar\\CYRT ": "\u0422",
    "\\cyrchar\\cyrt ": "\u0442",
    "\\cyrchar\\CYRTDSC ": "\u04AC",
    "\\cyrchar\\cyrtdsc ": "\u04AD",
    "\\cyrchar\\CYRTETSE ": "\u04B4",
    "\\cyrchar\\cyrtetse ": "\u04B5",
    "\\cyrchar\\cyrthousands ": "\u0482",
    "\\cyrchar\\CYRTSHE ": "\u040B",
    "\\cyrchar\\cyrtshe ": "\u045B",
    "\\cyrchar\\CYRU ": "\u0423",
    "\\cyrchar\\cyru ": "\u0443",
    "\\cyrchar\\CYRUK ": "\u0478",
    "\\cyrchar\\cyruk ": "\u0479",
    "\\cyrchar\\CYRUSHRT ": "\u040E",
    "\\cyrchar\\cyrushrt ": "\u045E",
    "\\cyrchar\\CYRV ": "\u0412",
    "\\cyrchar\\cyrv ": "\u0432",
    "\\cyrchar\\CYRY ": "\u04AE",
    "\\cyrchar\\cyry ": "\u04AF",
    "\\cyrchar\\CYRYA ": "\u042F",
    "\\cyrchar\\cyrya ": "\u044F",
    "\\cyrchar\\CYRYAT ": "\u0462",
    "\\cyrchar\\CYRYHCRS ": "\u04B0",
    "\\cyrchar\\cyryhcrs ": "\u04B1",
    "\\cyrchar\\CYRYI ": "\u0407",
    "\\cyrchar\\cyryi ": "\u0457",
    "\\cyrchar\\CYRYO ": "\u0401",
    "\\cyrchar\\cyryo ": "\u0451",
    "\\cyrchar\\CYRYU ": "\u042E",
    "\\cyrchar\\cyryu ": "\u044E",
    "\\cyrchar\\CYRZ ": "\u0417",
    "\\cyrchar\\cyrz ": "\u0437",
    "\\cyrchar\\CYRZDSC ": "\u0498",
    "\\cyrchar\\cyrzdsc ": "\u0499",
    "\\cyrchar\\CYRZH ": "\u0416",
    "\\cyrchar\\cyrzh ": "\u0436",
    "\\cyrchar\\CYRZHDSC ": "\u0496",
    "\\cyrchar\\cyrzhdsc ": "\u0497",
    "\\cyrchar\\textnumero ": "\u2116",
    "\\dagger ": "\u2020",
    "\\dashleftarrow ": "\u21E0",
    "\\dashrightarrow ": "\u21E2",
    "\\ddagger ": "\u2021",
    "\\daleth ": "\u2138",
    "\\dashv ": "\u22A3",
    "\\dblarrowupdown ": "\u21C5",
    "\\ddddot ": "\u20DC",
    "\\dddot ": "\u20DB",
    "\\ddots ": "\u22F1",
    "\\degree ": "\u00B0",
    "\\Delta ": "\u0394",
    "\\delta ": "\u03B4",
    "\\DH ": "\u00D0",
    "\\dh ": "\u00F0",
    "\\diagup ": "\u2571",
    "\\diamond ": "\u22C4",
    "\\diamondsuit ": "\u2662",
    "\\Digamma ": "\u03DC",
    "\\digamma ": "\u03DD",
    "\\ding{100}": "\u2744",
    "\\ding{101}": "\u2745",
    "\\ding{102}": "\u2746",
    "\\ding{103}": "\u2747",
    "\\ding{104}": "\u2748",
    "\\ding{105}": "\u2749",
    "\\ding{106}": "\u274A",
    "\\ding{107}": "\u274B",
    "\\ding{108}": "\u25CF",
    "\\ding{109}": "\u274D",
    "\\ding{110}": "\u25A0",
    "\\ding{111}": "\u274F",
    "\\ding{112}": "\u2750",
    "\\ding{113}": "\u2751",
    "\\ding{114}": "\u2752",
    "\\ding{115}": "\u25B2",
    "\\ding{116}": "\u25BC",
    "\\ding{117}": "\u25C6",
    "\\ding{118}": "\u2756",
    "\\ding{119}": "\u25D7",
    "\\ding{120}": "\u2758",
    "\\ding{121}": "\u2759",
    "\\ding{122}": "\u275A",
    "\\ding{123}": "\u275B",
    "\\ding{124}": "\u275C",
    "\\ding{125}": "\u275D",
    "\\ding{126}": "\u275E",
    "\\ding{161}": "\u2761",
    "\\ding{162}": "\u2762",
    "\\ding{163}": "\u2763",
    "\\ding{164}": "\u2764",
    "\\ding{165}": "\u2765",
    "\\ding{166}": "\u2766",
    "\\ding{167}": "\u2767",
    "\\ding{168}": "\u2663",
    "\\ding{169}": "\u2666",
    "\\ding{170}": "\u2665",
    "\\ding{171}": "\u2660",
    "\\ding{172}": "\u2460",
    "\\ding{173}": "\u2461",
    "\\ding{174}": "\u2462",
    "\\ding{175}": "\u2463",
    "\\ding{176}": "\u2464",
    "\\ding{177}": "\u2465",
    "\\ding{178}": "\u2466",
    "\\ding{179}": "\u2467",
    "\\ding{180}": "\u2468",
    "\\ding{181}": "\u2469",
    "\\ding{182}": "\u2776",
    "\\ding{183}": "\u2777",
    "\\ding{184}": "\u2778",
    "\\ding{185}": "\u2779",
    "\\ding{186}": "\u277A",
    "\\ding{187}": "\u277B",
    "\\ding{188}": "\u277C",
    "\\ding{189}": "\u277D",
    "\\ding{190}": "\u277E",
    "\\ding{191}": "\u277F",
    "\\ding{192}": "\u2780",
    "\\ding{193}": "\u2781",
    "\\ding{194}": "\u2782",
    "\\ding{195}": "\u2783",
    "\\ding{196}": "\u2784",
    "\\ding{197}": "\u2785",
    "\\ding{198}": "\u2786",
    "\\ding{199}": "\u2787",
    "\\ding{200}": "\u2788",
    "\\ding{201}": "\u2789",
    "\\ding{202}": "\u278A",
    "\\ding{203}": "\u278B",
    "\\ding{204}": "\u278C",
    "\\ding{205}": "\u278D",
    "\\ding{206}": "\u278E",
    "\\ding{207}": "\u278F",
    "\\ding{208}": "\u2790",
    "\\ding{209}": "\u2791",
    "\\ding{210}": "\u2792",
    "\\ding{211}": "\u2793",
    "\\ding{212}": "\u2794",
    "\\ding{216}": "\u2798",
    "\\ding{217}": "\u2799",
    "\\ding{218}": "\u279A",
    "\\ding{219}": "\u279B",
    "\\ding{220}": "\u279C",
    "\\ding{221}": "\u279D",
    "\\ding{222}": "\u279E",
    "\\ding{223}": "\u279F",
    "\\ding{224}": "\u27A0",
    "\\ding{225}": "\u27A1",
    "\\ding{226}": "\u27A2",
    "\\ding{227}": "\u27A3",
    "\\ding{228}": "\u27A4",
    "\\ding{229}": "\u27A5",
    "\\ding{230}": "\u27A6",
    "\\ding{231}": "\u27A7",
    "\\ding{232}": "\u27A8",
    "\\ding{233}": "\u27A9",
    "\\ding{234}": "\u27AA",
    "\\ding{235}": "\u27AB",
    "\\ding{236}": "\u27AC",
    "\\ding{237}": "\u27AD",
    "\\ding{238}": "\u27AE",
    "\\ding{239}": "\u27AF",
    "\\ding{241}": "\u27B1",
    "\\ding{242}": "\u27B2",
    "\\ding{243}": "\u27B3",
    "\\ding{244}": "\u27B4",
    "\\ding{245}": "\u27B5",
    "\\ding{246}": "\u27B6",
    "\\ding{247}": "\u27B7",
    "\\ding{248}": "\u27B8",
    "\\ding{249}": "\u27B9",
    "\\ding{250}": "\u27BA",
    "\\ding{251}": "\u27BB",
    "\\ding{252}": "\u27BC",
    "\\ding{253}": "\u27BD",
    "\\ding{254}": "\u27BE",
    "\\ding{33}": "\u2701",
    "\\ding{34}": "\u2702",
    "\\ding{35}": "\u2703",
    "\\ding{36}": "\u2704",
    "\\ding{37}": "\u260E",
    "\\ding{38}": "\u2706",
    "\\ding{39}": "\u2707",
    "\\ding{40}": "\u2708",
    "\\ding{41}": "\u2709",
    "\\ding{42}": "\u261B",
    "\\ding{43}": "\u261E",
    "\\ding{44}": "\u270C",
    "\\ding{45}": "\u270D",
    "\\ding{46}": "\u270E",
    "\\ding{47}": "\u270F",
    "\\ding{48}": "\u2710",
    "\\ding{49}": "\u2711",
    "\\ding{50}": "\u2712",
    "\\ding{51}": "\u2713",
    "\\ding{52}": "\u2714",
    "\\ding{53}": "\u2715",
    "\\ding{54}": "\u2716",
    "\\ding{55}": "\u2717",
    "\\ding{56}": "\u2718",
    "\\ding{57}": "\u2719",
    "\\ding{58}": "\u271A",
    "\\ding{59}": "\u271B",
    "\\ding{60}": "\u271C",
    "\\ding{61}": "\u271D",
    "\\ding{62}": "\u271E",
    "\\ding{63}": "\u271F",
    "\\ding{64}": "\u2720",
    "\\ding{65}": "\u2721",
    "\\ding{66}": "\u2722",
    "\\ding{67}": "\u2723",
    "\\ding{68}": "\u2724",
    "\\ding{69}": "\u2725",
    "\\ding{70}": "\u2726",
    "\\ding{71}": "\u2727",
    "\\ding{72}": "\u2605",
    "\\ding{73}": "\u2606",
    "\\ding{74}": "\u272A",
    "\\ding{75}": "\u272B",
    "\\ding{76}": "\u272C",
    "\\ding{77}": "\u272D",
    "\\ding{78}": "\u272E",
    "\\ding{79}": "\u272F",
    "\\ding{80}": "\u2730",
    "\\ding{81}": "\u2731",
    "\\ding{82}": "\u2732",
    "\\ding{83}": "\u2733",
    "\\ding{84}": "\u2734",
    "\\ding{85}": "\u2735",
    "\\ding{86}": "\u2736",
    "\\ding{87}": "\u2737",
    "\\ding{88}": "\u2738",
    "\\ding{89}": "\u2739",
    "\\ding{90}": "\u273A",
    "\\ding{91}": "\u273B",
    "\\ding{92}": "\u273C",
    "\\ding{93}": "\u273D",
    "\\ding{94}": "\u273E",
    "\\ding{95}": "\u273F",
    "\\ding{96}": "\u2740",
    "\\ding{97}": "\u2741",
    "\\ding{98}": "\u2742",
    "\\ding{99}": "\u2743",
    "\\div ": "\u00F7",
    "\\divideontimes ": "\u22C7",
    "\\DJ ": "\u0110",
    "\\dj ": "\u0111",
    "\\doteq ": "\u2250",
    "\\doteqdot ": "\u2251",
    "\\dotplus ": "\u2214",
    "\\dots ": "\u2026",
    "\\downarrow ": "\u2193",
    "\\Downarrow ": "\u21D3",
    "\\DownArrowBar ": "\u2913",
    "\\DownArrowUpArrow ": "\u21F5",
    "\\downdownarrows ": "\u21CA",
    "\\downharpoonleft ": "\u21C3",
    "\\downharpoonright ": "\u21C2",
    "\\DownLeftRightVector ": "\u2950",
    "\\DownLeftTeeVector ": "\u295E",
    "\\DownLeftVectorBar ": "\u2956",
    "\\DownRightTeeVector ": "\u295F",
    "\\DownRightVectorBar ": "\u2957",
    "\\downslopeellipsis ": "\u22F1",
    "\\eighthnote ": "\u266A",
    "\\ell ": "\u2113",
    "\\Elolarr ": "\u2940",
    "\\ElOr ": "\u2A56",
    "\\Elorarr ": "\u2941",
    "\\Elroang ": "\u2986",
    "\\ElsevierGlyph{2129}": "\u2129",
    "\\ElsevierGlyph{21B3}": "\u21B3",
    "\\ElsevierGlyph{2232}": "\u2232",
    "\\ElsevierGlyph{2233}": "\u2233",
    "\\ElsevierGlyph{2238}": "\u2238",
    "\\ElsevierGlyph{2242}": "\u2242",
    "\\ElsevierGlyph{225A}": "\u225A",
    "\\ElsevierGlyph{225F}": "\u225F",
    "\\ElsevierGlyph{2274}": "\u2274",
    "\\ElsevierGlyph{2275}": "\u2275",
    "\\ElsevierGlyph{22C0}": "\u22C0",
    "\\ElsevierGlyph{22C1}": "\u22C1",
    "\\ElsevierGlyph{300A}": "\u300A",
    "\\ElsevierGlyph{300B}": "\u300B",
    "\\ElsevierGlyph{3018}": "\u3018",
    "\\ElsevierGlyph{3019}": "\u3019",
    "\\ElsevierGlyph{E20A}": "\u2926",
    "\\ElsevierGlyph{E20B}": "\u2925",
    "\\ElsevierGlyph{E20C}": "\u2923",
    "\\ElsevierGlyph{E20D}": "\u2924",
    "\\ElsevierGlyph{E20E}": "\u2928",
    "\\ElsevierGlyph{E20F}": "\u2929",
    "\\ElsevierGlyph{E210}": "\u292A",
    "\\ElsevierGlyph{E211}": "\u2927",
    "\\ElsevierGlyph{E212}": "\u2905",
    "\\ElsevierGlyph{E214}": "\u297C",
    "\\ElsevierGlyph{E215}": "\u297D",
    "\\ElsevierGlyph{E219}": "\u2937",
    "\\ElsevierGlyph{E21A}": "\u2936",
    "\\ElsevierGlyph{E21C}": "\u2933",
    "\\ElsevierGlyph{E21D}": "\u2933\u0338",
    "\\ElsevierGlyph{E259}": "\u2A3C",
    "\\ElsevierGlyph{E25A}": "\u2A25",
    "\\ElsevierGlyph{E25B}": "\u2A2A",
    "\\ElsevierGlyph{E25C}": "\u2A2D",
    "\\ElsevierGlyph{E25D}": "\u2A2E",
    "\\ElsevierGlyph{E25E}": "\u2A35",
    "\\ElsevierGlyph{E260}": "\u29B5",
    "\\ElsevierGlyph{E291}": "\u2994",
    "\\ElsevierGlyph{E30D}": "\u2AEB",
    "\\ElsevierGlyph{E36E}": "\u2A55",
    "\\ElsevierGlyph{E372}": "\u29DC",
    "\\ElsevierGlyph{E381}": "\u25B1",
    "\\ElsevierGlyph{E395}": "\u2A10",
    "\\ElsevierGlyph{E61B}": "\u29B6",
    "\\ElsevierGlyph{E838}": "\u233D",
    "\\Elxsqcup ": "\u2A06",
    "\\Elxuplus ": "\u2A04",
    "\\ElzAnd ": "\u2A53",
    "\\Elzbar ": "\u0336",
    "\\Elzbtdl ": "\u026C",
    "\\ElzCint ": "\u2A0D",
    "\\Elzcirfb ": "\u25D2",
    "\\Elzcirfl ": "\u25D0",
    "\\Elzcirfr ": "\u25D1",
    "\\Elzclomeg ": "\u0277",
    "\\Elzddfnc ": "\u2999",
    "\\Elzdefas ": "\u29CB",
    "\\Elzdlcorn ": "\u23A3",
    "\\Elzdshfnc ": "\u2506",
    "\\Elzdyogh ": "\u02A4",
    "\\Elzesh ": "\u0283",
    "\\Elzfhr ": "\u027E",
    "\\Elzglst ": "\u0294",
    "\\Elzhlmrk ": "\u02D1",
    "\\ElzInf ": "\u2A07",
    "\\Elzinglst ": "\u0296",
    "\\Elzinvv ": "\u028C",
    "\\Elzinvw ": "\u028D",
    "\\ElzLap ": "\u29CA",
    "\\Elzlmrk ": "\u02D0",
    "\\Elzlow ": "\u02D5",
    "\\Elzlpargt ": "\u29A0",
    "\\Elzltlmr ": "\u0271",
    "\\Elzltln ": "\u0272",
    "\\Elzminhat ": "\u2A5F",
    "\\Elzopeno ": "\u0254",
    "\\ElzOr ": "\u2A54",
    "\\Elzpalh ": "\u0321",
    "\\Elzpbgam ": "\u0264",
    "\\Elzpgamma ": "\u0263",
    "\\Elzpscrv ": "\u028B",
    "\\Elzpupsil ": "\u028A",
    "\\Elzrais ": "\u02D4",
    "\\Elzrarrx ": "\u2947",
    "\\Elzreapos ": "\u201B",
    "\\Elzreglst ": "\u0295",
    "\\Elzrh ": "\u0322",
    "\\Elzrl ": "\u027C",
    "\\ElzRlarr ": "\u2942",
    "\\ElzrLarr ": "\u2944",
    "\\Elzrtld ": "\u0256",
    "\\Elzrtll ": "\u026D",
    "\\Elzrtln ": "\u0273",
    "\\Elzrtlr ": "\u027D",
    "\\Elzrtls ": "\u0282",
    "\\Elzrtlt ": "\u0288",
    "\\Elzrtlz ": "\u0290",
    "\\Elzrttrnr ": "\u027B",
    "\\Elzrvbull ": "\u25D8",
    "\\Elzsbbrg ": "\u032A",
    "\\Elzsblhr ": "\u02D3",
    "\\Elzsbrhr ": "\u02D2",
    "\\Elzschwa ": "\u0259",
    "\\Elzsqfl ": "\u25E7",
    "\\Elzsqfnw ": "\u2519",
    "\\Elzsqfr ": "\u25E8",
    "\\Elzsqfse ": "\u25EA",
    "\\Elzsqspne ": "\u22E5",
    "\\ElzSup ": "\u2A08",
    "\\Elztdcol ": "\u2AF6",
    "\\Elztesh ": "\u02A7",
    "\\Elztfnc ": "\u2980",
    "\\ElzThr ": "\u2A05",
    "\\ElzTimes ": "\u2A2F",
    "\\Elztrna ": "\u0250",
    "\\Elztrnh ": "\u0265",
    "\\Elztrnm ": "\u026F",
    "\\Elztrnmlr ": "\u0270",
    "\\Elztrnr ": "\u0279",
    "\\Elztrnrl ": "\u027A",
    "\\Elztrnsa ": "\u0252",
    "\\Elztrnt ": "\u0287",
    "\\Elztrny ": "\u028E",
    "\\Elzverti ": "\u02CC",
    "\\Elzverts ": "\u02C8",
    "\\Elzvrecto ": "\u25AF",
    "\\Elzxh ": "\u0127",
    "\\Elzxl ": "\u0335",
    "\\Elzxrat ": "\u211E",
    "\\Elzyogh ": "\u0292",
    "\\emdash ": "\u2014 ",
    "\\emptyset ": "\u2205",
    "\\ensuremath{\\Elzpes}": "\u20A7",
    "\\Epsilon ": "\u0395",
    "\\epsilon ": "\u03B5",
    "\\eqcirc ": "\u2256",
    "\\eqslantgtr ": "\u2A96",
    "\\eqslantless ": "\u2A95",
    "\\Equal ": "\u2A75",
    "\\estimates ": "\u2259",
    "\\Eta ": "\u0397",
    "\\eta ": "\u03B7",
    "\\eth ": "\u01AA",
    "\\euro ": "\u20AC",
    "\\exists ": "\u2203",
    "\\fallingdotseq ": "\u2252",
    "\\fbox{~~}": "\u25AD",
    "\\finv ": "\u2132",
    "\\flat ": "\u266D",
    "\\forall ": "\u2200",
    "\\forcesextra ": "\u22A8",
    "\\frown ": "\u2322",
    "\\Gamma ": "\u0393",
    "\\gamma ": "\u03B3",
    "\\gemini ": "\u264A",
    "\\geqq ": "\u2267",
    "\\geqslant ": "\u2A7E",
    "\\gg ": "\u226B",
    "\\ggg ": "\u22D9",
    "\\gimel ": "\u2137",
    "\\gnapprox ": "\u2A8A",
    "\\gneq ": "\u2A88",
    "\\gneqq ": "\u2269",
    "\\gnsim ": "\u22E7",
    "\\greaterequivlnt ": "\u2273",
    "\\gtrapprox ": "\u2A86",
    "\\gtrdot ": "\u22D7",
    "\\gtreqless ": "\u22DB",
    "\\gtreqqless ": "\u2A8C",
    "\\gtrless ": "\u2277",
    "\\gtrsim ": "\u2273",
    "\\guillemotleft ": "\u00AB",
    "\\guillemotright ": "\u00BB",
    "\\guilsinglleft ": "\u2039",
    "\\guilsinglright ": "\u203A",
    "\\gvertneqq ": "\u2269\uFE00",
    "\\H{}": "\u02DD",
    "\\H{O}": "\u0150",
    "\\H{o}": "\u0151",
    "\\H{U}": "\u0170",
    "\\H{u}": "\u0171",
    "\\hbar ": "\u210F",
    "\\heartsuit ": "\u2661",
    "\\hermitconjmatrix ": "\u22B9",
    "\\homothetic ": "\u223B",
    "\\hookleftarrow ": "\u21A9",
    "\\hookrightarrow ": "\u21AA",
    "\\hphantom{,}": "\u2008",
    "\\hphantom{0}": "\u2007",
    "\\hslash ": "\u210F",
    "\\hspace{0.166em}": "\u2006",
    "\\hspace{0.167em}": "\u2009",
    "\\hspace{0.25em}": "\u2005",
    "\\hspace{0.33em}": "\u2004",
    "\\hspace{0.6em}": "\u2002",
    "\\hspace{1em}": "\u2003",
    "\\im ": "\u2111",
    "\\image ": "\u22B7",
    "\\implies ": "\u21D2",
    "\\infty ": "\u221E",
    "\\int\\!\\int\\!\\int ": "\u222D",
    "\\intercal ": "\u22BA",
    "\\Iota ": "\u0399",
    "\\iota ": "\u03B9",
    "\\join ": "\u22C8",
    "\\jupiter ": "\u2643",
    "\\k{}": "\u02DB",
    "\\k{A}": "\u0104",
    "\\k{a}": "\u0105",
    "\\k{E}": "\u0118",
    "\\k{e}": "\u0119",
    "\\k{I}": "\u012E",
    "\\k{i}": "\u012F",
    "\\k{U}": "\u0172",
    "\\k{u}": "\u0173",
    "\\Kappa ": "\u039A",
    "\\kappa ": "\u03BA",
    "\\Koppa ": "\u03DE",
    "\\L ": "\u0141",
    "\\l ": "\u0142",
    "\\Lambda ": "\u039B",
    "\\lambda ": "\u03BB",
    "\\langle ": "\u2329",
    "\\lazysinv ": "\u223E",
    "\\lbrace ": "\u007B",
    "\\lceil ": "\u2308",
    "\\ldots ": "\u2026",
    "\\le ": "\u2264",
    "\\leadsto ": "\u219D",
    "\\leftarrow ": "\u2190",
    "\\Leftarrow ": "\u21D0",
    "\\leftarrowtail ": "\u21A2",
    "\\LeftDownTeeVector ": "\u2961",
    "\\LeftDownVectorBar ": "\u2959",
    "\\leftharpoondown ": "\u21BD",
    "\\leftharpoonup ": "\u21BC",
    "\\leftleftarrows ": "\u21C7",
    "\\leftrightarrow ": "\u2194",
    "\\Leftrightarrow ": "\u21D4",
    "\\leftrightarrows ": "\u21C6",
    "\\leftrightharpoons ": "\u21CB",
    "\\leftrightsquigarrow ": "\u21AD",
    "\\LeftRightVector ": "\u294E",
    "\\LeftTeeVector ": "\u295A",
    "\\leftthreetimes ": "\u22CB",
    "\\LeftTriangleBar ": "\u29CF",
    "\\LeftUpDownVector ": "\u2951",
    "\\LeftUpTeeVector ": "\u2960",
    "\\LeftUpVectorBar ": "\u2958",
    "\\LeftVectorBar ": "\u2952",
    "\\leo ": "\u264C",
    "\\leqq ": "\u2266",
    "\\leqslant ": "\u2A7D",
    "\\lessapprox ": "\u2A85",
    "\\lessdot ": "\u22D6",
    "\\lesseqgtr ": "\u22DA",
    "\\lesseqqgtr ": "\u2A8B",
    "\\lessequivlnt ": "\u2272",
    "\\lessgtr ": "\u2276",
    "\\lfloor ": "\u230A",
    "\\libra ": "\u264E",
    "\\ll ": "\u226A",
    "\\llcorner ": "\u231E",
    "\\Lleftarrow ": "\u21DA",
    "\\lmoustache ": "\u23B0",
    "\\lnapprox ": "\u2A89",
    "\\lneq ": "\u2A87",
    "\\lneqq ": "\u2268",
    "\\lnot ": "\u00AC",
    "\\lnsim ": "\u22E6",
    "\\longleftarrow ": "\u27F5",
    "\\Longleftarrow ": "\u27F8",
    "\\longleftrightarrow ": "\u27F7",
    "\\Longleftrightarrow ": "\u27FA",
    "\\longmapsto ": "\u27FC",
    "\\longrightarrow ": "\u27F6",
    "\\Longrightarrow ": "\u27F9",
    "\\looparrowleft ": "\u21AB",
    "\\looparrowright ": "\u21AC",
    "\\lozenge ": "\u25CA",
    "\\lrcorner ": "\u231F",
    "\\Lsh ": "\u21B0",
    "\\ltimes ": "\u22C9",
    "\\lvertneqq ": "\u2268\uFE00",
    "\\male ": "\u2642",
    "\\mapsto ": "\u21A6",
    "\\mathbb{C}": "\u2102",
    "\\mathbb{H}": "\u210D",
    "\\mathbb{N}": "\u2115",
    "\\mathbb{P}": "\u2119",
    "\\mathbb{Q}": "\u211A",
    "\\mathbb{R}": "\u211D",
    "\\mathbb{Z}": "\u2124",
    "\\mathbin{{:}\\!\\!{-}\\!\\!{:}}": "\u223A",
    "\\mathchar\"2208": "\u2316",
    "\\mathfrak{R}": "\u211C",
    "\\mathrm{'\\Omega}": "\u038F",
    "\\mathrm{'Y}": "\u038E",
    "\\mathrm{\\ddot{I}}": "\u03AA",
    "\\mathrm{\\ddot{Y}}": "\u03AB",
    "\\mathrm{\\mu}": "\u00B5",
    "\\mathscr{B}": "\u212C",
    "\\mathscr{e}": "\u212F",
    "\\mathscr{E}": "\u2130",
    "\\mathscr{F}": "\u2131",
    "\\mathscr{g}": "\u210A",
    "\\mathscr{H}": "\u210B",
    "\\mathscr{I}": "\u2110",
    "\\mathscr{L}": "\u2112",
    "\\mathscr{l}": "\u2113",
    "\\mathscr{M}": "\u2133",
    "\\mathscr{o}": "\u2134",
    "\\mathscr{R}": "\u211B",
    "\\mbox{\\texteuro} ": "\u20AC",
    "\\measuredangle ": "\u2221",
    "\\mercury ": "\u263F",
    "\\mho ": "\u2127",
    "\\mid ": "\u2223",
    "\\mkern1mu ": "\u200A",
    "\\mkern4mu ": "\u205F",
    "\\models ": "\u22A8",
    "\\mp ": "\u2213",
    "\\multimap ": "\u22B8",
    "\\nabla ": "\u2207",
    "\\natural ": "\u266E",
    "\\nearrow ": "\u2197",
    "\\neg ": "\u00AC",
    "\\neptune ": "\u2646",
    "\\neq ": "\u2260",
    "\\NestedGreaterGreater ": "\u2AA2",
    "\\NestedLessLess ": "\u2AA1",
    "\\nexists ": "\u2204",
    "\\NG ": "\u014A",
    "\\ng ": "\u014B",
    "\\ngeqslant ": "\u2A7E\u0338",
    "\\nleftarrow ": "\u219A",
    "\\nLeftarrow ": "\u21CD",
    "\\nleftrightarrow ": "\u21AE",
    "\\nLeftrightarrow ": "\u21CE",
    "\\nleqslant ": "\u2A7D\u0338",
    "\\nmid ": "\u2224",
    "\\nolinebreak ": "\u2060",
    "\\not =": "\u2260",
    "\\not\\apid ": "\u224B\u0338",
    "\\not\\approx ": "\u2249",
    "\\not\\cong ": "\u2247",
    "\\not\\doteq": "\u2250\u0338",
    "\\not\\equiv ": "\u2262",
    "\\not\\geq ": "\u2271",
    "\\not\\in ": "\u2209",
    "\\not\\kern-0.3em\\times ": "\u226D",
    "\\not\\leq ": "\u2270",
    "\\not\\ni ": "\u220C",
    "\\not\\prec ": "\u2280",
    "\\not\\preceq ": "\u2AAF\u0338",
    "\\not\\sim ": "\u2241",
    "\\not\\simeq ": "\u2244",
    "\\not\\sqsubseteq ": "\u22E2",
    "\\not\\sqsupseteq ": "\u22E3",
    "\\not\\subset ": "\u2284",
    "\\not\\subseteq ": "\u2288",
    "\\not\\succ ": "\u2281",
    "\\not\\succeq ": "\u2AB0\u0338",
    "\\not\\supset ": "\u2285",
    "\\not\\supseteq ": "\u2289",
    "\\not&gt;": "\u226F",
    "\\not&lt;": "\u226E",
    "\\NotEqualTilde ": "\u2242\u0338",
    "\\NotGreaterGreater ": "\u226B\u0338",
    "\\notgreaterless ": "\u2279",
    "\\NotHumpDownHump ": "\u224E\u0338",
    "\\NotHumpEqual ": "\u224F\u0338",
    "\\notin ": "\u2209",
    "\\NotLeftTriangleBar ": "\u29CF\u0338",
    "\\notlessgreater ": "\u2278",
    "\\NotLessLess ": "\u226A\u0338",
    "\\NotNestedGreaterGreater ": "\u2AA2\u0338",
    "\\NotNestedLessLess ": "\u2AA1\u0338",
    "\\NotPrecedesTilde ": "\u227E\u0338",
    "\\NotRightTriangleBar ": "\u29D0\u0338",
    "\\NotSquareSubset ": "\u228F\u0338",
    "\\NotSquareSuperset ": "\u2290\u0338",
    "\\NotSucceedsTilde ": "\u227F\u0338",
    "\\nparallel ": "\u2226",
    "\\nrightarrow ": "\u219B",
    "\\nRightarrow ": "\u21CF",
    "\\nsubseteqq ": "\u2AC5\u0338",
    "\\nsupseteqq": "\u2AC6\u0338",
    "\\ntriangleleft ": "\u22EA",
    "\\ntrianglelefteq ": "\u22EC",
    "\\ntriangleright ": "\u22EB",
    "\\ntrianglerighteq ": "\u22ED",
    "\\nu ": "\u03BD",
    "\\nvdash ": "\u22AC",
    "\\nvDash ": "\u22AD",
    "\\nVdash ": "\u22AE",
    "\\nVDash ": "\u22AF",
    "\\nwarrow ": "\u2196",
    "\\odot ": "\u2299",
    "\\OE ": "\u0152",
    "\\oe ": "\u0153",
    "\\oint ": "\u222E",
    "\\Omega ": "\u03A9",
    "\\omega ": "\u03C9",
    "\\ominus ": "\u2296",
    "\\openbracketleft ": "\u301A",
    "\\openbracketright ": "\u301B",
    "\\oplus ": "\u2295",
    "\\original ": "\u22B6",
    "\\oslash ": "\u2298",
    "\\otimes ": "\u2297",
    "\\parallel ": "\u2225",
    "\\partial ": "\u2202",
    "\\perp ": "\u22A5",
    "\\perspcorrespond ": "\u2A5E",
    "\\Phi ": "\u03A6",
    "\\phi ": "\u03D5",
    "\\Pi ": "\u03A0",
    "\\pi ": "\u03C0",
    "\\pilcrow ": "\u00B6",
    "\\pisces ": "\u2653",
    "\\Pisymbol{ppi020}{105}": "\u2A9E",
    "\\Pisymbol{ppi020}{117}": "\u2A9D",
    "\\Pisymbol{ppi022}{87}": "\u03D0",
    "\\pitchfork ": "\u22D4",
    "\\pluto ": "\u2647",
    "\\pm ": "\u00B1",
    "\\pound ": "\u00A3",
    "\\precapprox ": "\u2AB7",
    "\\preccurlyeq ": "\u227C",
    "\\precedesnotsimilar ": "\u22E8",
    "\\precnapprox ": "\u2AB9",
    "\\precneqq ": "\u2AB5",
    "\\precsim ": "\u227E",
    "\\prime ": "\u2032",
    "\\prod ": "\u220F",
    "\\propto ": "\u221D",
    "\\Psi ": "\u03A8",
    "\\psi ": "\u03C8",
    "\\quarternote ": "\u2669",
    "\\r{}": "\u02DA",
    "\\r{U}": "\u016E",
    "\\r{u}": "\u016F",
    "\\raisebox{.5ex}{\\scritpsize TM}": "\u2122",
    "\\rangle ": "\u232A",
    "\\rbrace ": "\u007D",
    "\\rceil ": "\u2309",
    "\\re ": "\u211C",
    "\\recorder ": "\u2315",
    "\\registered ": "\u00AE",
    "\\ReverseUpEquilibrium ": "\u296F",
    "\\rfloor ": "\u230B",
    "\\rhd ": "\u22b3",
    "\\Rho ": "\u03A1",
    "\\rho ": "\u03C1",
    "\\rightangle ": "\u221F",
    "\\rightanglearc ": "\u22BE",
    "\\rightarrow ": "\u2192",
    "\\Rightarrow ": "\u21D2",
    "\\rightarrowtail ": "\u21A3",
    "\\RightDownTeeVector ": "\u295D",
    "\\RightDownVectorBar ": "\u2955",
    "\\rightharpoondown ": "\u21C1",
    "\\rightharpoonup ": "\u21C0",
    "\\rightleftarrows ": "\u21C4",
    "\\rightleftharpoons ": "\u21CC",
    "\\rightmoon ": "\u263E",
    "\\rightrightarrows ": "\u21C9",
    "\\rightsquigarrow ": "\u21DD",
    "\\RightTeeVector ": "\u295B",
    "\\rightthreetimes ": "\u22CC",
    "\\RightTriangleBar ": "\u29D0",
    "\\RightUpDownVector ": "\u294F",
    "\\RightUpTeeVector ": "\u295C",
    "\\RightUpVectorBar ": "\u2954",
    "\\RightVectorBar ": "\u2953",
    "\\risingdotseq ": "\u2253",
    "\\rmoustache ": "\u23B1",
    "\\RoundImplies ": "\u2970",
    "\\Rrightarrow ": "\u21DB",
    "\\Rsh ": "\u21B1",
    "\\rtimes ": "\u22CA",
    "\\rule{1em}{1pt}": "\u2015",
    "\\RuleDelayed ": "\u29F4",
    "\\sagittarius ": "\u2650",
    "\\Sampi ": "\u03E0",
    "\\saturn ": "\u2644",
    "\\scorpio ": "\u264F",
    "\\searrow ": "\u2198",
    "\\section ": "\u00A7",
    "\\setminus ": "\u2216",
    "\\sharp ": "\u266F",
    "\\shortparallel ": "\u2225",
    "\\Sigma ": "\u03A3",
    "\\sigma ": "\u03C3",
    "\\sim\\joinrel\\leadsto": "\u27FF",
    "\\slash ": "\u002F",
    "\\smallfrown ": "\u2322",
    "\\smallsetminus ": "\u2216",
    "\\smallsmile ": "\u2323",
    "\\smile ": "\u2323",
    "\\space ": "\u0020",
    "\\spadesuit ": "\u2660",
    "\\sphericalangle ": "\u2222",
    "\\sqcap ": "\u2293",
    "\\sqcup ": "\u2294",
    "\\sqrint ": "\u2A16",
    "\\sqsubset ": "\u228F",
    "\\sqsupset ": "\u2290",
    "\\square ": "\u25A1",
    "\\ss ": "\u00DF",
    "\\stackrel{*}{=}": "\u2A6E",
    "\\star ": "\u22C6",
    "\\starequal ": "\u225B",
    "\\Stigma ": "\u03DA",
    "\\Subset ": "\u22D0",
    "\\subseteqq ": "\u2AC5",
    "\\subsetneq ": "\u228A",
    "\\subsetneqq ": "\u2ACB",
    "\\succapprox ": "\u2AB8",
    "\\succcurlyeq ": "\u227D",
    "\\succnapprox ": "\u2ABA",
    "\\succneqq ": "\u2AB6",
    "\\succnsim ": "\u22E9",
    "\\sum ": "\u2211",
    "\\Supset ": "\u22D1",
    "\\supseteqq ": "\u2AC6",
    "\\supsetneq ": "\u228B",
    "\\supsetneqq ": "\u2ACC",
    "\\surd ": "\u221A",
    "\\surfintegral ": "\u222F",
    "\\swarrow ": "\u2199",
    "\\Tau ": "\u03A4",
    "\\tau ": "\u03C4",
    "\\taurus ": "\u2649",
    "\\textasciiacute ": "\u00B4",
    "\\textasciibreve ": "\u02D8",
    "\\textasciicaron ": "\u02C7",
    "\\textasciidieresis ": "\u00A8",
    "\\textasciigrave ": "\u0060",
    "\\textasciimacron ": "\u00AF",
    "\\textasciitilde ": "\u007E",
    "\\textbackslash ": "\u005C",
    "\\textbrokenbar ": "\u00A6",
    "\\textbullet ": "\u2022",
    "\\textcent ": "\u00A2",
    "\\textcopyright ": "\u00A9",
    "\\textcurrency ": "\u00A4",
    "\\textdagger ": "\u2020",
    "\\textdaggerdbl ": "\u2021",
    "\\textdegree ": "\u00B0",
    "\\textdollar ": "\u0024",
    "\\textdoublepipe ": "\u01C2",
    "\\textemdash ": "\u2014 ",
    "\\textendash ": "\u2013 ",
    "\\textexclamdown ": "\u00A1",
    "\\textfrac{1}{3}": "\u2153",
    "\\textfrac{1}{5}": "\u2155",
    "\\textfrac{1}{6}": "\u2159",
    "\\textfrac{1}{8}": "\u215B",
    "\\textfrac{2}{3}": "\u2154",
    "\\textfrac{2}{5}": "\u2156",
    "\\textfrac{3}{5}": "\u2157",
    "\\textfrac{3}{8}": "\u215C",
    "\\textfrac{4}{5}": "\u2158",
    "\\textfrac{5}{6}": "\u215A",
    "\\textfrac{5}{8}": "\u215D",
    "\\textfrac{7}{8}": "\u215E",
    "\\texthvlig ": "\u0195",
    "\\textnrleg ": "\u019E",
    "\\textonehalf ": "\u00BD",
    "\\textonequarter ": "\u00BC",
    "\\textordfeminine ": "\u00AA",
    "\\textordmasculine ": "\u00BA",
    "\\textparagraph ": "\u00B6",
    "\\textperiodcentered ": "\u02D9",
    "\\textpertenthousand ": "\u2031",
    "\\textperthousand ": "\u2030",
    "\\textphi ": "\u0278",
    "\\textquestiondown ": "\u00BF",
    "\\textquotedblleft ": "\u201C",
    "\\textquotedblright ": "\u201D",
    "\\textquotesingle ": "\u0027",
    "\\textregistered ": "\u00AE",
    "\\textsection ": "\u00A7",
    "\\textsterling ": "\u00A3",
    "\\texttheta ": "\u03B8",
    "\\textTheta ": "\u03F4",
    "\\textthreequarters ": "\u00BE",
    "\\texttildelow ": "\u02DC",
    "\\texttimes ": "\u00D7",
    "\\texttrademark ": "\u2122",
    "\\textturnk ": "\u029E",
    "\\textvartheta ": "\u03D1",
    "\\textvisiblespace ": "\u2423",
    "\\textyen ": "\u00A5",
    "\\TH ": "\u00DE",
    "\\th ": "\u00FE",
    "\\therefore ": "\u2234",
    "\\theta ": "\u03B8",
    "\\Theta ": "\u0398",
    "\\thickapprox ": "\u2248",
    "\\thicksim ": "\u223C",
    "\\tildetrpl ": "\u224B",
    "\\tone{11}": "\u02E9",
    "\\tone{22}": "\u02E8",
    "\\tone{33}": "\u02E7",
    "\\tone{44}": "\u02E6",
    "\\tone{55}": "\u02E5",
    "\\top ": "\u22A4",
    "\\trademark ": "\u2122",
    "\\triangle ": "\u25B3",
    "\\triangledown ": "\u25BF",
    "\\triangleleft ": "\u25C3",
    "\\trianglelefteq ": "\u22B4",
    "\\triangleq ": "\u225C",
    "\\triangleright ": "\u25B9",
    "\\trianglerighteq ": "\u22B5",
    "\\truestate ": "\u22A7",
    "\\twoheadleftarrow ": "\u219E",
    "\\twoheadrightarrow ": "\u21A0",
    "\\u{\\i}": "\u012D",
    "\\u{A}": "\u0102",
    "\\u{a}": "\u0103",
    "\\u{E}": "\u0114",
    "\\u{e}": "\u0115",
    "\\u{G}": "\u011E",
    "\\u{g}": "\u011F",
    "\\u{I}": "\u012C",
    "\\u{O}": "\u014E",
    "\\u{o}": "\u014F",
    "\\u{U}": "\u016C",
    "\\u{u}": "\u016D",
    "\\ulcorner ": "\u231C",
    "\\unlhd ": "\u22B4",
    "\\unrhd ": "\u22B5",
    "\\uparrow ": "\u2191",
    "\\Uparrow ": "\u21D1",
    "\\UpArrowBar ": "\u2912",
    "\\updownarrow ": "\u2195",
    "\\Updownarrow ": "\u21D5",
    "\\UpEquilibrium ": "\u296E",
    "\\upharpoonleft ": "\u21BF",
    "\\upharpoonright ": "\u21BE",
    "\\uplus ": "\u228E",
    "\\Upsilon ": "\u03A5",
    "\\upsilon ": "\u03C5",
    "\\upslopeellipsis ": "\u22F0",
    "\\upuparrows ": "\u21C8",
    "\\uranus ": "\u2645",
    "\\urcorner ": "\u231D",
    "\\user{}": "\u007E",
    "\\v{C}": "\u010C",
    "\\v{c}": "\u010D",
    "\\v{D}": "\u010E",
    "\\v{d}": "\u010F",
    "\\v{E}": "\u011A",
    "\\v{e}": "\u011B",
    "\\v{L}": "\u013D",
    "\\v{l}": "\u013E",
    "\\v{N}": "\u0147",
    "\\v{n}": "\u0148",
    "\\v{R}": "\u0158",
    "\\v{r}": "\u0159",
    "\\v{S}": "\u0160",
    "\\v{s}": "\u0161",
    "\\v{T}": "\u0164",
    "\\v{t}": "\u0165",
    "\\v{Z}": "\u017D",
    "\\v{z}": "\u017E",
    "\\varepsilon ": "\u025B",
    "\\varkappa ": "\u03F0",
    "\\varnothing ": "\u2205",
    "\\varphi ": "\u03C6",
    "\\varpi ": "\u03D6",
    "\\varpropto ": "\u221D",
    "\\varrho ": "\u03F1",
    "\\varsigma ": "\u03C2",
    "\\varsubsetneqq ": "\u228A\uFE00",
    "\\varsupsetneq ": "\u228B\uFE00",
    "\\vartheta ": "\u03D1",
    "\\vartriangle ": "\u25B5",
    "\\vartriangleleft ": "\u22B2",
    "\\vartriangleright ": "\u22B3",
    "\\vdash ": "\u22A2",
    "\\Vdash ": "\u22A9",
    "\\VDash ": "\u22AB",
    "\\vdots ": "\u22EE",
    "\\vee ": "\u2228",
    "\\veebar ": "\u22BB",
    "\\venus ": "\u2640",
    "\\vert ": "\u007C",
    "\\Vert ": "\u2016",
    "\\verymuchgreater ": "\u22D9",
    "\\verymuchless ": "\u22D8",
    "\\virgo ": "\u264D",
    "\\volintegral ": "\u2230",
    "\\Vvdash ": "\u22AA",
    "\\wedge ": "\u2227",
    "\\wp ": "\u2118",
    "\\wr ": "\u2240",
    "\\Xi ": "\u039E",
    "\\xi ": "\u03BE",
    "\\Zeta ": "\u0396",
    "\\zeta ": "\u03B6",
    "&lt;\\kern-0.58em(": "\u2993",
    "=:": "\u2255",
    "\\log": "log",
    "\\sqrt": "\u221A",
}
