﻿using System;
using System.CommandLine;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text.RegularExpressions;
using Xcst;
using Xcst.Compiler;
using XcstWebExtension = Xcst.Web.Extension.ExtensionPackageV2;

namespace XcstCodeGen;

partial class Program {

   string?
   _language;

   public required Uri
   ProjectUri { get; init; }

   public required string
   RootNamespace { get; init; }

   public required string[]
   SourceFiles { get; init; }

   public string?
   Nullable { get; init; }

   public decimal
   TargetRuntime { get; set; }

   public bool
   PageEnable { get; set; }

   public string?
   PageBaseType { get; set; }

   public string[]
   Libraries { get; set; } = Array.Empty<string>();

   public string[]
   Extensions { get; set; } = Array.Empty<string>();

   private string
   Language => _language ??= ProjectLang(ProjectUri);

   private bool
   V1 => TargetRuntime != default
      && TargetRuntime < 2m;

   private bool
   SingleOutput => Language == "cs";

   static string
   ProjectLang(Uri projectUri) {
      var projExt = Path.GetExtension(projectUri.LocalPath).TrimStart('.');
      return projExt.Substring(0, projExt.Length - "proj".Length);
   }

   void
   AddExtensions(XcstCompiler compiler) {

      foreach (var ext in Extensions) {

         // URI example: clitype:Foo.FooPackage?from=C:\Foo\Foo.dll

         var extUri = new Uri(ext);
         var extType = extUri.AbsolutePath;
         var queryParams = extUri.Query.Substring(1)
            .Split(';', StringSplitOptions.RemoveEmptyEntries);

         var loc = Uri.UnescapeDataString(queryParams
            .First(p => p.StartsWith("from="))
            .Substring("from=".Length));

         var asm = Assembly.LoadFrom(loc)!;

         compiler.RegisterExtension(() =>
            (IXcstPackage)Activator.CreateInstance(asm.GetType(extType)!)!);
      }
   }

   string
   FileNamespace(Uri fileUri, Uri startUri) {

      var ns = RootNamespace;
      var relativePath = startUri.MakeRelativeUri(fileUri).OriginalString;

      if (relativePath.Contains('/')) {

         var relativeDir = startUri
            .MakeRelativeUri(new Uri(Path.GetDirectoryName(fileUri.LocalPath)!, UriKind.Absolute))
            .OriginalString;

         ns = String.Join(".", relativeDir
            .Split('/')
            .Select(n => CleanIdentifier(n))
            .Prepend(ns));
      }

      return ns;
   }

   // Transforms invalid identifier (class, namespace, variable) characters
   static string
   CleanIdentifier(string identifier) =>
      CleanIdentifierRegex().Replace(identifier, "_");

   [GeneratedRegex("[^a-z0-9_]", RegexOptions.IgnoreCase)]
   private static partial Regex
   CleanIdentifierRegex();

   // Show compilation errors on Visual Studio's Error List
   // Also makes the error on the Output window clickable
   static void
   VisualStudioErrorLog(RuntimeException ex) {

      var errorData = (dynamic?)ex.ErrorData;

      if (errorData != null) {

         var uriString = (string?)errorData.ModuleUri;
         var path = (Uri.TryCreate(uriString, UriKind.Absolute, out var uri) && uri.IsFile) ?
            uri.LocalPath
            : uriString;

         Console.WriteLine($"{path}({errorData.LineNumber}): XCST error {ex.ErrorCode}: {ex.Message}");
      }
   }

   void
   WriteAutogeneratedComment(TextWriter output) {

      var prefix = (Language == "vb") ? "'" : "//";

      output.WriteLine(prefix + "------------------------------------------------------------------------------");
      output.WriteLine(prefix + " <auto-generated>");
      output.WriteLine(prefix + $"     This code was generated by {typeof(XcstCompiler).Namespace}.");
      output.WriteLine(prefix + "");
      output.WriteLine(prefix + "     Changes to this file may cause incorrect behavior and will be lost if");
      output.WriteLine(prefix + "     the code is regenerated.");
      output.WriteLine(prefix + " </auto-generated>");
      output.WriteLine(prefix + "------------------------------------------------------------------------------");
   }

   static TextWriter
   CreateOutput(Uri outputUri) {

      var output = File.CreateText(outputUri.LocalPath);

      // Because XML parsers normalize CRLF to LF,
      // we want to be consistent with the additional content we create
      output.NewLine = "\n";

      return output;
   }

   void
   Run() {

      var startUri = new Uri(ProjectUri, ".");

      var compiler = new XcstCompiler {
         PackageFileDirectory = startUri.LocalPath
      };

      if (TargetRuntime != default) {
         compiler.TargetRuntime = TargetRuntime;
      }

      // Enable "application" extension
      compiler.RegisterExtension(() => {

         var appExtPkg = new XcstWebExtension {
            ApplicationUri = startUri,
            GenerateLinkTo = true,
            AnnotateVirtualPath = true
         };

         return appExtPkg;
      });

      AddExtensions(compiler);

      if (!String.IsNullOrEmpty(Nullable)) {
         compiler.NullableAnnotate = true;
         compiler.NullableContext = Nullable;
      }

      foreach (var lib in Libraries) {
         compiler.AddPackageLibrary(lib);
      }

      using var output = (SingleOutput) ?
         CreateOutput(new Uri(ProjectUri, $"xcst.generated.{Language}"))
         : null;

      if (output != null) {

         WriteAutogeneratedComment(output);

         if (PageEnable
            && V1) {

            output.WriteLine();
            output.WriteLine("[assembly: global::Xcst.Web.Precompilation.PrecompiledModule]");
         }

         compiler.CompilationUnitHandler = href => output;
      }

      foreach (var file in SourceFiles) {

         var fileUri = new Uri(file, UriKind.Absolute);
         var fileName = Path.GetFileName(file);
         var fileBaseName = Path.GetFileNameWithoutExtension(file);

         // Ignore files starting with underscore
         if (fileName[0] == '_') {
            continue;
         }

         // Treat files ending with 'Package' as library packages; other files as pages
         // An alternative would be to use different file extensions for library packages and pages
         var isPage = PageEnable
            && !fileBaseName.EndsWith("Package");

         compiler.TargetNamespace = FileNamespace(fileUri, startUri);

         if (isPage) {

            compiler.TargetClass = "_Page_" + CleanIdentifier(fileBaseName);

            if (PageBaseType != null) {
               compiler.TargetBaseTypes = new[] { PageBaseType };
            }

         } else {

            compiler.TargetClass = CleanIdentifier(fileBaseName);
            compiler.TargetBaseTypes = null;
         }

         XcstWebExtension.IsPage(compiler.SetTunnelParam, isPage);

         var pkgOutput = default(TextWriter);

         if (output is null) {

            if (Language == "vb") {

               int cuIndex = 0;

               compiler.CompilationUnitHandler = href => {

                  var outputUri = new Uri(fileUri, $"{fileName}.{cuIndex++}.generated.{Language}");
                  var output = CreateOutput(outputUri);

                  WriteAutogeneratedComment(output);

                  return output;
               };

            } else {

               var pkgOutputUri = new Uri(fileUri, $"{fileName}.generated.{Language}");
               pkgOutput = CreateOutput(pkgOutputUri);

               WriteAutogeneratedComment(pkgOutput);

               compiler.CompilationUnitHandler = href => pkgOutput;
            }
         }

         try {
            compiler.Compile(fileUri);

         } catch (RuntimeException ex) {
            VisualStudioErrorLog(ex);
            throw;

         } finally {
            pkgOutput?.Dispose();
         }
      }
   }

   public static int
   Main(string[] args) {

      const string optPrefix = "-";

      var projectPathOpt = new Option<string>(optPrefix + "ProjectPath") {
         IsRequired = true
      };

      var rootNamespaceOpt = new Option<string>(optPrefix + nameof(RootNamespace)) {
         IsRequired = true
      };

      var nullableOpt = new Option<string?>(optPrefix + nameof(Nullable));
      var targetRuntimeOpt = new Option<decimal>(optPrefix + nameof(TargetRuntime));
      var pageEnableOpt = new Option<bool>(optPrefix + nameof(PageEnable));
      var pageBaseTypeOpt = new Option<string?>(optPrefix + nameof(PageBaseType));
      var libraryOpt = new Option<string[]>(optPrefix + "Library");
      var extensionOpt = new Option<string[]>(optPrefix + "Extension");
      var sourceFilesArg = new Argument<string[]>(nameof(SourceFiles));

      var rootCmd = new RootCommand("xcst-codegen") {
         projectPathOpt,
         rootNamespaceOpt,
         nullableOpt,
         targetRuntimeOpt,
         pageEnableOpt,
         pageBaseTypeOpt,
         libraryOpt,
         extensionOpt,
         sourceFilesArg
      };

      rootCmd.SetHandler(ctx => {

         var result = ctx.ParseResult;

         var currentDir = Environment.CurrentDirectory;

         if (currentDir[^1] != Path.DirectorySeparatorChar) {
            currentDir += Path.DirectorySeparatorChar;
         }

         var callerBaseUri = new Uri(currentDir, UriKind.Absolute);
         var projectUri = new Uri(callerBaseUri, result.GetValueForOption(projectPathOpt));

         var program = new Program {
            ProjectUri = projectUri,
            RootNamespace = result.GetValueForOption(rootNamespaceOpt)!,
            Nullable = result.GetValueForOption(nullableOpt),
            TargetRuntime = result.GetValueForOption(targetRuntimeOpt),
            PageEnable = result.GetValueForOption(pageEnableOpt),
            PageBaseType = result.GetValueForOption(pageBaseTypeOpt),
            Libraries = result.GetValueForOption(libraryOpt)!,
            Extensions = result.GetValueForOption(extensionOpt)!,
            SourceFiles = result.GetValueForArgument(sourceFilesArg)
               .Select(p => new Uri(projectUri, p).LocalPath)
               .ToArray()
         };

         program.Run();
      });

      return rootCmd.Invoke(args);
   }
}
