<?php

declare(strict_types=1);

namespace IterTools\Tests\Single;

use IterTools\Single;
use IterTools\Tests\Fixture\ArrayIteratorFixture;
use IterTools\Tests\Fixture\GeneratorFixture;
use IterTools\Tests\Fixture\IteratorAggregateFixture;

class SliceTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @test slice example usage
     */
    public function testSliceExampleUsage(): void
    {
        // Given
        $olympics = [1992, 1994, 1996, 1998, 2000, 2002, 2004, 2006, 2008, 2010, 2012, 2014, 2016, 2018, 2020, 2022];

        // And
        $summerOlympics = [];
        $winterOlympics = [];

        // When
        foreach (Single::slice($olympics, 0, 8, 2) as $summerYear) {
            $summerOlympics[] = $summerYear;
        }
        foreach (Single::slice($olympics, 1, 8, 2) as $winterYear) {
            $winterOlympics[] = $winterYear;
        }

        // Then
        $this->assertEquals([1992, 1996, 2000, 2004, 2008, 2012, 2016, 2020], $summerOlympics);
        $this->assertEquals([1994, 1998, 2002, 2006, 2010, 2014, 2018, 2022], $winterOlympics);
    }

    /**
     * @dataProvider dataProviderForArray
     * @param array $iterable
     * @param array $config
     * @param array $expected
     * @return void
     */
    public function testArray(array $iterable, array $config, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::slice($iterable, ...$config) as $value) {
            $result[] = $value;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForArray(): array
    {
        return [
            [
                [],
                [],
                [],
            ],
            [
                [],
                [0],
                [],
            ],
            [
                [],
                [1],
                [],
            ],
            [
                [],
                [0, 0],
                [],
            ],
            [
                [],
                [1, 1],
                [],
            ],
            [
                [],
                [1, 2],
                [],
            ],
            [
                [],
                [1, 2, 1],
                [],
            ],
            [
                [],
                [1, 2, 2],
                [],
            ],
            [
                [],
                [1, 2, 3],
                [],
            ],
            [
                [1],
                [],
                [1],
            ],
            [
                [1],
                [0],
                [1],
            ],
            [
                [1],
                [1],
                [],
            ],
            [
                [1],
                [0, 0],
                [],
            ],
            [
                [1],
                [1, 1],
                [],
            ],
            [
                [1],
                [1, 2],
                [],
            ],
            [
                [1],
                [1, 2, 1],
                [],
            ],
            [
                [1],
                [1, 2, 2],
                [],
            ],
            [
                [1],
                [1, 2, 3],
                [],
            ],
            [
                [1, 2],
                [],
                [1, 2],
            ],
            [
                [1, 2],
                [0],
                [1, 2],
            ],
            [
                [1, 2],
                [1],
                [2],
            ],
            [
                [1, 2],
                [0, 0],
                [],
            ],
            [
                [1, 2],
                [1, 1],
                [2],
            ],
            [
                [1, 2],
                [1, 2],
                [2],
            ],
            [
                [1, 2],
                [1, 2, 1],
                [2],
            ],
            [
                [1, 2],
                [1, 2, 2],
                [2],
            ],
            [
                [1, 2],
                [1, 2, 3],
                [2],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [0],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1],
                [2, 3, 4, 5, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [0, 0],
                [],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 1],
                [2],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 2],
                [2, 3],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 4],
                [2, 3, 4, 5],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 2, 1],
                [2, 3],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 4, 1],
                [2, 3, 4, 5],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 2, 2],
                [2, 4],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 3, 2],
                [2, 4, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 4, 2],
                [2, 4, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 2, 3],
                [2, 5],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 3, 3],
                [2, 5],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, 3, 10],
                [2],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, null, 10],
                [2],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, null, 3],
                [2, 5],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [0, null, 1],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, null, 1],
                [2, 3, 4, 5, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [1, null, 2],
                [2, 4, 6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [5, null, 2],
                [6],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [6, null, 2],
                [],
            ],
            [
                [1, 2, 3, 4, 5, 6],
                [7, null, 2],
                [],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [0],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [0, 0],
                [],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 1],
                ['b'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 2],
                ['b', 'c'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 4],
                ['b', 'c', 'd', 'e'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 2, 1],
                ['b', 'c'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 4, 1],
                ['b', 'c', 'd', 'e'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 2, 2],
                ['b', 'd'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 3, 2],
                ['b', 'd', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 4, 2],
                ['b', 'd', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 2, 3],
                ['b', 'e'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 3, 3],
                ['b', 'e'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, 3, 10],
                ['b'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, null, 10],
                ['b'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, null, 3],
                ['b', 'e'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [0, null, 1],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, null, 1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [1, null, 2],
                ['b', 'd', 'f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [5, null, 2],
                ['f'],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [6, null, 2],
                [],
            ],
            [
                ['a', 'b', 'c', 'd', 'e', 'f'],
                [7, null, 2],
                [],
            ],
            [
                [[1], [2], [3], [4], [5], [6], [7], [8]],
                [1, 3, 2],
                [[2], [4], [6]],
            ],
            [
                [1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]],
                [0, 4, 2],
                [1, '3', false, 'abc'],
            ],
            [
                [1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]],
                [1, 4, 2],
                [2.2, true, null, [1, 2, 3]],
            ],
            [
                [1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]],
                [7, null, 1],
                [[1, 2, 3], (object)[4, 5, 6]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForGenerators
     * @param \Generator $iterable
     * @param array $config
     * @param array $expected
     * @return void
     */
    public function testGenerators(\Generator $iterable, array $config, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::slice($iterable, ...$config) as $value) {
            $result[] = $value;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForGenerators(): array
    {
        $gen = fn ($data) => GeneratorFixture::getGenerator($data);

        return [
            [
                $gen([]),
                [],
                [],
            ],
            [
                $gen([]),
                [0],
                [],
            ],
            [
                $gen([]),
                [1],
                [],
            ],
            [
                $gen([]),
                [0, 0],
                [],
            ],
            [
                $gen([]),
                [1, 1],
                [],
            ],
            [
                $gen([]),
                [1, 2],
                [],
            ],
            [
                $gen([]),
                [1, 2, 1],
                [],
            ],
            [
                $gen([]),
                [1, 2, 2],
                [],
            ],
            [
                $gen([]),
                [1, 2, 3],
                [],
            ],
            [
                $gen([1]),
                [],
                [1],
            ],
            [
                $gen([1]),
                [0],
                [1],
            ],
            [
                $gen([1]),
                [1],
                [],
            ],
            [
                $gen([1]),
                [0, 0],
                [],
            ],
            [
                $gen([1]),
                [1, 1],
                [],
            ],
            [
                $gen([1]),
                [1, 2],
                [],
            ],
            [
                $gen([1]),
                [1, 2, 1],
                [],
            ],
            [
                $gen([1]),
                [1, 2, 2],
                [],
            ],
            [
                $gen([1]),
                [1, 2, 3],
                [],
            ],
            [
                $gen([1, 2]),
                [],
                [1, 2],
            ],
            [
                $gen([1, 2]),
                [0],
                [1, 2],
            ],
            [
                $gen([1, 2]),
                [1],
                [2],
            ],
            [
                $gen([1, 2]),
                [0, 0],
                [],
            ],
            [
                $gen([1, 2]),
                [1, 1],
                [2],
            ],
            [
                $gen([1, 2]),
                [1, 2],
                [2],
            ],
            [
                $gen([1, 2]),
                [1, 2, 1],
                [2],
            ],
            [
                $gen([1, 2]),
                [1, 2, 2],
                [2],
            ],
            [
                $gen([1, 2]),
                [1, 2, 3],
                [2],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [0],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1],
                [2, 3, 4, 5, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [0, 0],
                [],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 1],
                [2],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 2],
                [2, 3],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 4],
                [2, 3, 4, 5],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 2, 1],
                [2, 3],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 4, 1],
                [2, 3, 4, 5],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 2, 2],
                [2, 4],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 3, 2],
                [2, 4, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 4, 2],
                [2, 4, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 2, 3],
                [2, 5],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 3, 3],
                [2, 5],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, 3, 10],
                [2],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, null, 10],
                [2],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, null, 3],
                [2, 5],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [0, null, 1],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, null, 1],
                [2, 3, 4, 5, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [1, null, 2],
                [2, 4, 6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [5, null, 2],
                [6],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [6, null, 2],
                [],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6]),
                [7, null, 2],
                [],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [0],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [0, 0],
                [],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 1],
                ['b'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2],
                ['b', 'c'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4],
                ['b', 'c', 'd', 'e'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 1],
                ['b', 'c'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4, 1],
                ['b', 'c', 'd', 'e'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 2],
                ['b', 'd'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 2],
                ['b', 'd', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4, 2],
                ['b', 'd', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 3],
                ['b', 'e'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 3],
                ['b', 'e'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 10],
                ['b'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 10],
                ['b'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 3],
                ['b', 'e'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [0, null, 1],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 2],
                ['b', 'd', 'f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [5, null, 2],
                ['f'],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [6, null, 2],
                [],
            ],
            [
                $gen(['a', 'b', 'c', 'd', 'e', 'f']),
                [7, null, 2],
                [],
            ],
            [
                $gen([[1], [2], [3], [4], [5], [6], [7], [8]]),
                [1, 3, 2],
                [[2], [4], [6]],
            ],
            [
                $gen([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [0, 4, 2],
                [1, '3', false, 'abc'],
            ],
            [
                $gen([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [1, 4, 2],
                [2.2, true, null, [1, 2, 3]],
            ],
            [
                $gen([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [7, null, 1],
                [[1, 2, 3], (object)[4, 5, 6]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForIterators
     * @param \Iterator $iterable
     * @param array $config
     * @param array $expected
     * @return void
     */
    public function testIterators(\Iterator $iterable, array $config, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::slice($iterable, ...$config) as $value) {
            $result[] = $value;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForIterators(): array
    {
        $iter = fn ($data) => new ArrayIteratorFixture($data);

        return [
            [
                $iter([]),
                [],
                [],
            ],
            [
                $iter([]),
                [0],
                [],
            ],
            [
                $iter([]),
                [1],
                [],
            ],
            [
                $iter([]),
                [0, 0],
                [],
            ],
            [
                $iter([]),
                [1, 1],
                [],
            ],
            [
                $iter([]),
                [1, 2],
                [],
            ],
            [
                $iter([]),
                [1, 2, 1],
                [],
            ],
            [
                $iter([]),
                [1, 2, 2],
                [],
            ],
            [
                $iter([]),
                [1, 2, 3],
                [],
            ],
            [
                $iter([1]),
                [],
                [1],
            ],
            [
                $iter([1]),
                [0],
                [1],
            ],
            [
                $iter([1]),
                [1],
                [],
            ],
            [
                $iter([1]),
                [0, 0],
                [],
            ],
            [
                $iter([1]),
                [1, 1],
                [],
            ],
            [
                $iter([1]),
                [1, 2],
                [],
            ],
            [
                $iter([1]),
                [1, 2, 1],
                [],
            ],
            [
                $iter([1]),
                [1, 2, 2],
                [],
            ],
            [
                $iter([1]),
                [1, 2, 3],
                [],
            ],
            [
                $iter([1, 2]),
                [],
                [1, 2],
            ],
            [
                $iter([1, 2]),
                [0],
                [1, 2],
            ],
            [
                $iter([1, 2]),
                [1],
                [2],
            ],
            [
                $iter([1, 2]),
                [0, 0],
                [],
            ],
            [
                $iter([1, 2]),
                [1, 1],
                [2],
            ],
            [
                $iter([1, 2]),
                [1, 2],
                [2],
            ],
            [
                $iter([1, 2]),
                [1, 2, 1],
                [2],
            ],
            [
                $iter([1, 2]),
                [1, 2, 2],
                [2],
            ],
            [
                $iter([1, 2]),
                [1, 2, 3],
                [2],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [0],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1],
                [2, 3, 4, 5, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [0, 0],
                [],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 1],
                [2],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 2],
                [2, 3],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 4],
                [2, 3, 4, 5],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 2, 1],
                [2, 3],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 4, 1],
                [2, 3, 4, 5],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 2, 2],
                [2, 4],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 3, 2],
                [2, 4, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 4, 2],
                [2, 4, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 2, 3],
                [2, 5],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 3, 3],
                [2, 5],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, 3, 10],
                [2],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, null, 10],
                [2],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, null, 3],
                [2, 5],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [0, null, 1],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, null, 1],
                [2, 3, 4, 5, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [1, null, 2],
                [2, 4, 6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [5, null, 2],
                [6],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [6, null, 2],
                [],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6]),
                [7, null, 2],
                [],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [0],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [0, 0],
                [],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 1],
                ['b'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2],
                ['b', 'c'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4],
                ['b', 'c', 'd', 'e'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 1],
                ['b', 'c'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4, 1],
                ['b', 'c', 'd', 'e'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 2],
                ['b', 'd'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 2],
                ['b', 'd', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4, 2],
                ['b', 'd', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 3],
                ['b', 'e'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 3],
                ['b', 'e'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 10],
                ['b'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 10],
                ['b'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 3],
                ['b', 'e'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [0, null, 1],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 2],
                ['b', 'd', 'f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [5, null, 2],
                ['f'],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [6, null, 2],
                [],
            ],
            [
                $iter(['a', 'b', 'c', 'd', 'e', 'f']),
                [7, null, 2],
                [],
            ],
            [
                $iter([[1], [2], [3], [4], [5], [6], [7], [8]]),
                [1, 3, 2],
                [[2], [4], [6]],
            ],
            [
                $iter([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [0, 4, 2],
                [1, '3', false, 'abc'],
            ],
            [
                $iter([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [1, 4, 2],
                [2.2, true, null, [1, 2, 3]],
            ],
            [
                $iter([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [7, null, 1],
                [[1, 2, 3], (object)[4, 5, 6]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForTraversables
     * @param \Traversable $iterable
     * @param array $config
     * @param array $expected
     * @return void
     */
    public function testTraversables(\Traversable $iterable, array $config, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::slice($iterable, ...$config) as $value) {
            $result[] = $value;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForTraversables(): array
    {
        $trav = fn ($data) => new IteratorAggregateFixture($data);

        return [
            [
                $trav([]),
                [],
                [],
            ],
            [
                $trav([]),
                [0],
                [],
            ],
            [
                $trav([]),
                [1],
                [],
            ],
            [
                $trav([]),
                [0, 0],
                [],
            ],
            [
                $trav([]),
                [1, 1],
                [],
            ],
            [
                $trav([]),
                [1, 2],
                [],
            ],
            [
                $trav([]),
                [1, 2, 1],
                [],
            ],
            [
                $trav([]),
                [1, 2, 2],
                [],
            ],
            [
                $trav([]),
                [1, 2, 3],
                [],
            ],
            [
                $trav([1]),
                [],
                [1],
            ],
            [
                $trav([1]),
                [0],
                [1],
            ],
            [
                $trav([1]),
                [1],
                [],
            ],
            [
                $trav([1]),
                [0, 0],
                [],
            ],
            [
                $trav([1]),
                [1, 1],
                [],
            ],
            [
                $trav([1]),
                [1, 2],
                [],
            ],
            [
                $trav([1]),
                [1, 2, 1],
                [],
            ],
            [
                $trav([1]),
                [1, 2, 2],
                [],
            ],
            [
                $trav([1]),
                [1, 2, 3],
                [],
            ],
            [
                $trav([1, 2]),
                [],
                [1, 2],
            ],
            [
                $trav([1, 2]),
                [0],
                [1, 2],
            ],
            [
                $trav([1, 2]),
                [1],
                [2],
            ],
            [
                $trav([1, 2]),
                [0, 0],
                [],
            ],
            [
                $trav([1, 2]),
                [1, 1],
                [2],
            ],
            [
                $trav([1, 2]),
                [1, 2],
                [2],
            ],
            [
                $trav([1, 2]),
                [1, 2, 1],
                [2],
            ],
            [
                $trav([1, 2]),
                [1, 2, 2],
                [2],
            ],
            [
                $trav([1, 2]),
                [1, 2, 3],
                [2],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [0],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1],
                [2, 3, 4, 5, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [0, 0],
                [],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 1],
                [2],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 2],
                [2, 3],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 4],
                [2, 3, 4, 5],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 2, 1],
                [2, 3],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 4, 1],
                [2, 3, 4, 5],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 2, 2],
                [2, 4],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 3, 2],
                [2, 4, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 4, 2],
                [2, 4, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 2, 3],
                [2, 5],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 3, 3],
                [2, 5],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, 3, 10],
                [2],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, null, 10],
                [2],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, null, 3],
                [2, 5],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [0, null, 1],
                [1, 2, 3, 4, 5, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, null, 1],
                [2, 3, 4, 5, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [1, null, 2],
                [2, 4, 6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [5, null, 2],
                [6],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [6, null, 2],
                [],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6]),
                [7, null, 2],
                [],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [0],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [0, 0],
                [],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 1],
                ['b'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2],
                ['b', 'c'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4],
                ['b', 'c', 'd', 'e'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 1],
                ['b', 'c'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4, 1],
                ['b', 'c', 'd', 'e'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 2],
                ['b', 'd'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 2],
                ['b', 'd', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 4, 2],
                ['b', 'd', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 2, 3],
                ['b', 'e'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 3],
                ['b', 'e'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, 3, 10],
                ['b'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 10],
                ['b'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 3],
                ['b', 'e'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [0, null, 1],
                ['a', 'b', 'c', 'd', 'e', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 1],
                ['b', 'c', 'd', 'e', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [1, null, 2],
                ['b', 'd', 'f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [5, null, 2],
                ['f'],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [6, null, 2],
                [],
            ],
            [
                $trav(['a', 'b', 'c', 'd', 'e', 'f']),
                [7, null, 2],
                [],
            ],
            [
                $trav([[1], [2], [3], [4], [5], [6], [7], [8]]),
                [1, 3, 2],
                [[2], [4], [6]],
            ],
            [
                $trav([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [0, 4, 2],
                [1, '3', false, 'abc'],
            ],
            [
                $trav([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [1, 4, 2],
                [2.2, true, null, [1, 2, 3]],
            ],
            [
                $trav([1, 2.2, '3', true, false, null, 'abc', [1, 2, 3], (object)[4, 5, 6]]),
                [7, null, 1],
                [[1, 2, 3], (object)[4, 5, 6]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForArray
     * @param array $iterable
     * @param array $config
     * @param array $expected
     * @return void
     */
    public function testIteratorToArray(array $iterable, array $config, array $expected): void
    {
        // Given
        $iterator = Single::slice($iterable, ...$config);

        // When
        $result = \iterator_to_array($iterator);

        // Then
        $this->assertEquals($expected, $result);
    }

    /**
     * @dataProvider dataProviderForInvalidArgumentStart
     * @param array $config
     * @return void
     */
    public function testInvalidArgumentStart(array $config): void
    {
        // Given
        $input = [1, 2, 3];

        // Then
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("Parameter 'start' cannot be negative");

        // When
        foreach (Single::slice($input, ...$config) as $_) {
            $this->fail();
        }
    }

    public function dataProviderForInvalidArgumentStart(): array
    {
        return [
            [[-1]],
            [[-2]],
            [[-1, null]],
            [[-2, null]],
            [[-1, 2]],
            [[-1, null, 0]],
            [[-1, 2, 0]],
            [[-1, null, 2]],
            [[-1, 2, 2]],
            [[-1, -2, 2]],
            [[-1, 2, -2]],
            [[-1, -2, -2]],
        ];
    }

    /**
     * @dataProvider dataProviderForInvalidArgumentCount
     * @param array $config
     * @return void
     */
    public function testInvalidArgumentCount(array $config): void
    {
        // Given
        $input = [1, 2, 3];

        // Then
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("Parameter 'count' cannot be negative");

        // When
        foreach (Single::slice($input, ...$config) as $_) {
            $this->fail();
        }
    }

    public function dataProviderForInvalidArgumentCount(): array
    {
        return [
            [[0, -1]],
            [[1, -1]],
            [[0, -2]],
            [[0, -1, 1]],
            [[0, -2, 2]],
            [[0, -1, -1]],
        ];
    }

    /**
     * @dataProvider dataProviderForInvalidArgumentStep
     * @param array $config
     * @return void
     */
    public function testInvalidArgumentStep(array $config): void
    {
        // Given
        $input = [1, 2, 3];

        // Then
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("Parameter 'step' must be positive");

        // When
        foreach (Single::slice($input, ...$config) as $_) {
            $this->fail();
        }
    }

    public function dataProviderForInvalidArgumentStep(): array
    {
        return [
            [[0, null, 0]],
            [[0, null, -1]],
            [[0, null, -2]],
            [[0, 0, 0]],
            [[0, 0, -1]],
            [[0, 0, -2]],
            [[0, 1, 0]],
            [[0, 1, -1]],
            [[0, 1, -2]],
        ];
    }
}
