<?php

declare(strict_types=1);

namespace IterTools\Tests\Single;

use IterTools\Single;
use IterTools\Tests\Fixture\ArrayIteratorFixture;
use IterTools\Tests\Fixture\GeneratorFixture;
use IterTools\Tests\Fixture\IteratorAggregateFixture;

class ChunkwiseOverlapTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @dataProvider dataProviderForArray
     * @param array $data
     * @param int $chunkSize
     * @param int $overlapSize
     * @param bool $excludeIncompleteTail
     * @param array $expected
     */
    public function testArray(array $data, int $chunkSize, int $overlapSize, bool $excludeIncompleteTail, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::chunkwiseOverlap($data, $chunkSize, $overlapSize, $excludeIncompleteTail) as $datum) {
            $result[] = $datum;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForArray(): array
    {
        return [
            [
                [],
                1,
                0,
                true,
                [],
            ],
            [
                [],
                1,
                0,
                false,
                [],
            ],
            [
                [],
                2,
                0,
                true,
                [],
            ],
            [
                [],
                2,
                0,
                false,
                [],
            ],
            [
                [],
                2,
                1,
                true,
                [],
            ],
            [
                [],
                2,
                1,
                false,
                [],
            ],
            [
                [1],
                1,
                0,
                true,
                [[1]],
            ],
            [
                [1],
                1,
                0,
                false,
                [[1]],
            ],
            [
                [1],
                1,
                0,
                true,
                [[1]],
            ],
            [
                [1],
                1,
                0,
                false,
                [[1]],
            ],
            [
                [1],
                2,
                0,
                true,
                [[1]],
            ],
            [
                [1],
                2,
                0,
                false,
                [],
            ],
            [
                [1],
                2,
                1,
                true,
                [[1]],
            ],
            [
                [1],
                2,
                1,
                false,
                [],
            ],
            [
                [1],
                3,
                0,
                true,
                [[1]],
            ],
            [
                [1],
                3,
                0,
                false,
                [],
            ],
            [
                [1],
                3,
                1,
                true,
                [[1]],
            ],
            [
                [1],
                3,
                1,
                false,
                [],
            ],
            [
                [1],
                3,
                2,
                true,
                [[1]],
            ],
            [
                [1],
                3,
                2,
                false,
                [],
            ],
            [
                [1, 2],
                1,
                0,
                true,
                [[1], [2]],
            ],
            [
                [1, 2],
                1,
                0,
                false,
                [[1], [2]],
            ],
            [
                [1, 2],
                2,
                0,
                true,
                [[1, 2]],
            ],
            [
                [1, 2],
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                [1, 2],
                2,
                1,
                true,
                [[1, 2]],
            ],
            [
                [1, 2],
                2,
                1,
                false,
                [[1, 2]],
            ],
            [
                [1, 2],
                3,
                0,
                true,
                [[1, 2]],
            ],
            [
                [1, 2],
                3,
                0,
                false,
                [],
            ],
            [
                [1, 2],
                3,
                1,
                true,
                [[1, 2]],
            ],
            [
                [1, 2],
                3,
                1,
                false,
                [],
            ],
            [
                [1, 2],
                3,
                2,
                true,
                [[1, 2]],
            ],
            [
                [1, 2],
                3,
                2,
                false,
                [],
            ],
            [
                [1, 2, 3],
                1,
                0,
                true,
                [[1], [2], [3]],
            ],
            [
                [1, 2, 3],
                1,
                0,
                false,
                [[1], [2], [3]],
            ],
            [
                [1, 2, 3],
                2,
                0,
                true,
                [[1, 2], [3]],
            ],
            [
                [1, 2, 3],
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                [1, 2, 3],
                2,
                1,
                true,
                [[1, 2], [2, 3]],
            ],
            [
                [1, 2, 3],
                2,
                1,
                false,
                [[1, 2], [2, 3]],
            ],
            [
                [1, 2, 3],
                3,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                3,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                3,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                3,
                2,
                false,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                4,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                4,
                0,
                false,
                [],
            ],
            [
                [1, 2, 3],
                4,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                4,
                1,
                false,
                [],
            ],
            [
                [1, 2, 3],
                4,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                4,
                2,
                false,
                [],
            ],
            [
                [1, 2, 3],
                4,
                3,
                true,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3],
                4,
                3,
                false,
                [],
            ],
            [
                [1, 2, 3, 4],
                2,
                0,
                true,
                [[1, 2], [3, 4]],
            ],
            [
                [1, 2, 3, 4],
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                [1, 2, 3, 4],
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                [1, 2, 3, 4],
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                [1, 2, 3, 4],
                3,
                0,
                true,
                [[1, 2, 3], [4]],
            ],
            [
                [1, 2, 3, 4],
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3, 4],
                3,
                1,
                true,
                [[1, 2, 3], [3, 4]],
            ],
            [
                [1, 2, 3, 4],
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3, 4],
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                [1, 2, 3, 4],
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                [1, 2, 3, 4, 5],
                2,
                0,
                true,
                [[1, 2], [3, 4], [5]],
            ],
            [
                [1, 2, 3, 4, 5],
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                [1, 2, 3, 4, 5],
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                [1, 2, 3, 4, 5],
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                [1, 2, 3, 4, 5],
                3,
                0,
                true,
                [[1, 2, 3], [4, 5]],
            ],
            [
                [1, 2, 3, 4, 5],
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                [1, 2, 3, 4, 5],
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                [1, 2, 3, 4, 5],
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                [1, 2, 3, 4, 5],
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                [1, 2, 3, 4, 5],
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                2,
                0,
                true,
                [[1.1, 2.2], [3.3, 4.4], [5.5]],
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                2,
                0,
                false,
                [[1.1, 2.2], [3.3, 4.4]],
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                2,
                1,
                true,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                2,
                1,
                false,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                ['1', '2', '3', '4', '5'],
                2,
                0,
                true,
                [['1', '2'], ['3', '4'], ['5']],
            ],
            [
                ['1', '2', '3', '4', '5'],
                2,
                0,
                false,
                [['1', '2'], ['3', '4']],
            ],
            [
                ['1', '2', '3', '4', '5'],
                2,
                1,
                true,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                ['1', '2', '3', '4', '5'],
                2,
                1,
                false,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                [[1], [2], [3], [4], [5]],
                2,
                0,
                true,
                [[[1], [2]], [[3], [4]], [[5]]],
            ],
            [
                [[1], [2], [3], [4], [5]],
                2,
                0,
                false,
                [[[1], [2]], [[3], [4]]],
            ],
            [
                [[1], [2], [3], [4], [5]],
                2,
                1,
                true,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                [[1], [2], [3], [4], [5]],
                2,
                1,
                false,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                [true, true, false, false],
                2,
                0,
                true,
                [[true, true], [false, false]],
            ],
            [
                [true, true, false, false],
                2,
                0,
                false,
                [[true, true], [false, false]],
            ],
            [
                [true, true, false, false],
                2,
                1,
                true,
                [[true, true], [true, false], [false, false]],
            ],
            [
                [true, true, false, false],
                2,
                1,
                false,
                [[true, true], [true, false], [false, false]],
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                3,
                0,
                true,
                [[1, 2.2, '3'], [[4], true, null], ['test data']],
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                3,
                0,
                false,
                [[1, 2.2, '3'], [[4], true, null]],
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                3,
                1,
                true,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                3,
                1,
                false,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                3,
                2,
                true,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                3,
                2,
                false,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                2,
                0,
                true,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                2,
                0,
                false,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                2,
                1,
                true,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                2,
                1,
                false,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                [1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9], [9, 10]],
            ],
            [
                [1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForGenerators
     * @param \Generator $data
     * @param int $chunkSize
     * @param int $overlapSize
     * @param bool $excludeIncompleteTail
     * @param array $expected
     */
    public function testGenerators(\Generator $data, int $chunkSize, int $overlapSize, bool $excludeIncompleteTail, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::chunkwiseOverlap($data, $chunkSize, $overlapSize, $excludeIncompleteTail) as $datum) {
            $result[] = $datum;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForGenerators(): array
    {
        $gen = static function (array $data) {
            return GeneratorFixture::getGenerator($data);
        };

        return [
            [
                $gen([]),
                1,
                0,
                true,
                [],
            ],
            [
                $gen([]),
                1,
                0,
                false,
                [],
            ],
            [
                $gen([]),
                2,
                0,
                true,
                [],
            ],
            [
                $gen([]),
                2,
                0,
                false,
                [],
            ],
            [
                $gen([]),
                2,
                1,
                true,
                [],
            ],
            [
                $gen([]),
                2,
                1,
                false,
                [],
            ],
            [
                $gen([1]),
                1,
                0,
                true,
                [[1]],
            ],
            [
                $gen([1]),
                1,
                0,
                false,
                [[1]],
            ],
            [
                $gen([1]),
                1,
                0,
                true,
                [[1]],
            ],
            [
                $gen([1]),
                1,
                0,
                false,
                [[1]],
            ],
            [
                $gen([1]),
                2,
                0,
                true,
                [[1]],
            ],
            [
                $gen([1]),
                2,
                0,
                false,
                [],
            ],
            [
                $gen([1]),
                2,
                1,
                true,
                [[1]],
            ],
            [
                $gen([1]),
                2,
                1,
                false,
                [],
            ],
            [
                $gen([1]),
                3,
                0,
                true,
                [[1]],
            ],
            [
                $gen([1]),
                3,
                0,
                false,
                [],
            ],
            [
                $gen([1]),
                3,
                1,
                true,
                [[1]],
            ],
            [
                $gen([1]),
                3,
                1,
                false,
                [],
            ],
            [
                $gen([1]),
                3,
                2,
                true,
                [[1]],
            ],
            [
                $gen([1]),
                3,
                2,
                false,
                [],
            ],
            [
                $gen([1, 2]),
                1,
                0,
                true,
                [[1], [2]],
            ],
            [
                $gen([1, 2]),
                1,
                0,
                false,
                [[1], [2]],
            ],
            [
                $gen([1, 2]),
                2,
                0,
                true,
                [[1, 2]],
            ],
            [
                $gen([1, 2]),
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                $gen([1, 2]),
                2,
                1,
                true,
                [[1, 2]],
            ],
            [
                $gen([1, 2]),
                2,
                1,
                false,
                [[1, 2]],
            ],
            [
                $gen([1, 2]),
                3,
                0,
                true,
                [[1, 2]],
            ],
            [
                $gen([1, 2]),
                3,
                0,
                false,
                [],
            ],
            [
                $gen([1, 2]),
                3,
                1,
                true,
                [[1, 2]],
            ],
            [
                $gen([1, 2]),
                3,
                1,
                false,
                [],
            ],
            [
                $gen([1, 2]),
                3,
                2,
                true,
                [[1, 2]],
            ],
            [
                $gen([1, 2]),
                3,
                2,
                false,
                [],
            ],
            [
                $gen([1, 2, 3]),
                1,
                0,
                true,
                [[1], [2], [3]],
            ],
            [
                $gen([1, 2, 3]),
                1,
                0,
                false,
                [[1], [2], [3]],
            ],
            [
                $gen([1, 2, 3]),
                2,
                0,
                true,
                [[1, 2], [3]],
            ],
            [
                $gen([1, 2, 3]),
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                $gen([1, 2, 3]),
                2,
                1,
                true,
                [[1, 2], [2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                2,
                1,
                false,
                [[1, 2], [2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                3,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                3,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                3,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                3,
                2,
                false,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                4,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                4,
                0,
                false,
                [],
            ],
            [
                $gen([1, 2, 3]),
                4,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                4,
                1,
                false,
                [],
            ],
            [
                $gen([1, 2, 3]),
                4,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                4,
                2,
                false,
                [],
            ],
            [
                $gen([1, 2, 3]),
                4,
                3,
                true,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3]),
                4,
                3,
                false,
                [],
            ],
            [
                $gen([1, 2, 3, 4]),
                2,
                0,
                true,
                [[1, 2], [3, 4]],
            ],
            [
                $gen([1, 2, 3, 4]),
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                $gen([1, 2, 3, 4]),
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                $gen([1, 2, 3, 4]),
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                $gen([1, 2, 3, 4]),
                3,
                0,
                true,
                [[1, 2, 3], [4]],
            ],
            [
                $gen([1, 2, 3, 4]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3, 4]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4]],
            ],
            [
                $gen([1, 2, 3, 4]),
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3, 4]),
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                $gen([1, 2, 3, 4]),
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                2,
                0,
                true,
                [[1, 2], [3, 4], [5]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                3,
                0,
                true,
                [[1, 2, 3], [4, 5]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                $gen([1, 2, 3, 4, 5]),
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                $gen([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                0,
                true,
                [[1.1, 2.2], [3.3, 4.4], [5.5]],
            ],
            [
                $gen([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                0,
                false,
                [[1.1, 2.2], [3.3, 4.4]],
            ],
            [
                $gen([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                1,
                true,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                $gen([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                1,
                false,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                $gen(['1', '2', '3', '4', '5']),
                2,
                0,
                true,
                [['1', '2'], ['3', '4'], ['5']],
            ],
            [
                $gen(['1', '2', '3', '4', '5']),
                2,
                0,
                false,
                [['1', '2'], ['3', '4']],
            ],
            [
                $gen(['1', '2', '3', '4', '5']),
                2,
                1,
                true,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                $gen(['1', '2', '3', '4', '5']),
                2,
                1,
                false,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                $gen([[1], [2], [3], [4], [5]]),
                2,
                0,
                true,
                [[[1], [2]], [[3], [4]], [[5]]],
            ],
            [
                $gen([[1], [2], [3], [4], [5]]),
                2,
                0,
                false,
                [[[1], [2]], [[3], [4]]],
            ],
            [
                $gen([[1], [2], [3], [4], [5]]),
                2,
                1,
                true,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                $gen([[1], [2], [3], [4], [5]]),
                2,
                1,
                false,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                $gen([true, true, false, false]),
                2,
                0,
                true,
                [[true, true], [false, false]],
            ],
            [
                $gen([true, true, false, false]),
                2,
                0,
                false,
                [[true, true], [false, false]],
            ],
            [
                $gen([true, true, false, false]),
                2,
                1,
                true,
                [[true, true], [true, false], [false, false]],
            ],
            [
                $gen([true, true, false, false]),
                2,
                1,
                false,
                [[true, true], [true, false], [false, false]],
            ],
            [
                $gen([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                0,
                true,
                [[1, 2.2, '3'], [[4], true, null], ['test data']],
            ],
            [
                $gen([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                0,
                false,
                [[1, 2.2, '3'], [[4], true, null]],
            ],
            [
                $gen([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                1,
                true,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                $gen([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                1,
                false,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                $gen([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                2,
                true,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                $gen([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                2,
                false,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                $gen(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                0,
                true,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                $gen(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                0,
                false,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                $gen(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                1,
                true,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                $gen(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                1,
                false,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9], [9, 10]],
            ],
            [
                $gen([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]),
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForIterators
     * @param \Iterator $data
     * @param int $chunkSize
     * @param int $overlapSize
     * @param bool $excludeIncompleteTail
     * @param array $expected
     */
    public function testIterators(\Iterator $data, int $chunkSize, int $overlapSize, bool $excludeIncompleteTail, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::chunkwiseOverlap($data, $chunkSize, $overlapSize, $excludeIncompleteTail) as $datum) {
            $result[] = $datum;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForIterators(): array
    {
        $iter = static function (array $data) {
            return new ArrayIteratorFixture($data);
        };

        return [
            [
                $iter([]),
                1,
                0,
                true,
                [],
            ],
            [
                $iter([]),
                1,
                0,
                false,
                [],
            ],
            [
                $iter([]),
                2,
                0,
                true,
                [],
            ],
            [
                $iter([]),
                2,
                0,
                false,
                [],
            ],
            [
                $iter([]),
                2,
                1,
                true,
                [],
            ],
            [
                $iter([]),
                2,
                1,
                false,
                [],
            ],
            [
                $iter([1]),
                1,
                0,
                true,
                [[1]],
            ],
            [
                $iter([1]),
                1,
                0,
                false,
                [[1]],
            ],
            [
                $iter([1]),
                1,
                0,
                true,
                [[1]],
            ],
            [
                $iter([1]),
                1,
                0,
                false,
                [[1]],
            ],
            [
                $iter([1]),
                2,
                0,
                true,
                [[1]],
            ],
            [
                $iter([1]),
                2,
                0,
                false,
                [],
            ],
            [
                $iter([1]),
                2,
                1,
                true,
                [[1]],
            ],
            [
                $iter([1]),
                2,
                1,
                false,
                [],
            ],
            [
                $iter([1]),
                3,
                0,
                true,
                [[1]],
            ],
            [
                $iter([1]),
                3,
                0,
                false,
                [],
            ],
            [
                $iter([1]),
                3,
                1,
                true,
                [[1]],
            ],
            [
                $iter([1]),
                3,
                1,
                false,
                [],
            ],
            [
                $iter([1]),
                3,
                2,
                true,
                [[1]],
            ],
            [
                $iter([1]),
                3,
                2,
                false,
                [],
            ],
            [
                $iter([1, 2]),
                1,
                0,
                true,
                [[1], [2]],
            ],
            [
                $iter([1, 2]),
                1,
                0,
                false,
                [[1], [2]],
            ],
            [
                $iter([1, 2]),
                2,
                0,
                true,
                [[1, 2]],
            ],
            [
                $iter([1, 2]),
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                $iter([1, 2]),
                2,
                1,
                true,
                [[1, 2]],
            ],
            [
                $iter([1, 2]),
                2,
                1,
                false,
                [[1, 2]],
            ],
            [
                $iter([1, 2]),
                3,
                0,
                true,
                [[1, 2]],
            ],
            [
                $iter([1, 2]),
                3,
                0,
                false,
                [],
            ],
            [
                $iter([1, 2]),
                3,
                1,
                true,
                [[1, 2]],
            ],
            [
                $iter([1, 2]),
                3,
                1,
                false,
                [],
            ],
            [
                $iter([1, 2]),
                3,
                2,
                true,
                [[1, 2]],
            ],
            [
                $iter([1, 2]),
                3,
                2,
                false,
                [],
            ],
            [
                $iter([1, 2, 3]),
                1,
                0,
                true,
                [[1], [2], [3]],
            ],
            [
                $iter([1, 2, 3]),
                1,
                0,
                false,
                [[1], [2], [3]],
            ],
            [
                $iter([1, 2, 3]),
                2,
                0,
                true,
                [[1, 2], [3]],
            ],
            [
                $iter([1, 2, 3]),
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                $iter([1, 2, 3]),
                2,
                1,
                true,
                [[1, 2], [2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                2,
                1,
                false,
                [[1, 2], [2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                3,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                3,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                3,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                3,
                2,
                false,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                4,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                4,
                0,
                false,
                [],
            ],
            [
                $iter([1, 2, 3]),
                4,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                4,
                1,
                false,
                [],
            ],
            [
                $iter([1, 2, 3]),
                4,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                4,
                2,
                false,
                [],
            ],
            [
                $iter([1, 2, 3]),
                4,
                3,
                true,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3]),
                4,
                3,
                false,
                [],
            ],
            [
                $iter([1, 2, 3, 4]),
                2,
                0,
                true,
                [[1, 2], [3, 4]],
            ],
            [
                $iter([1, 2, 3, 4]),
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                $iter([1, 2, 3, 4]),
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                $iter([1, 2, 3, 4]),
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                $iter([1, 2, 3, 4]),
                3,
                0,
                true,
                [[1, 2, 3], [4]],
            ],
            [
                $iter([1, 2, 3, 4]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3, 4]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4]],
            ],
            [
                $iter([1, 2, 3, 4]),
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3, 4]),
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                $iter([1, 2, 3, 4]),
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                2,
                0,
                true,
                [[1, 2], [3, 4], [5]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                3,
                0,
                true,
                [[1, 2, 3], [4, 5]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                $iter([1, 2, 3, 4, 5]),
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                $iter([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                0,
                true,
                [[1.1, 2.2], [3.3, 4.4], [5.5]],
            ],
            [
                $iter([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                0,
                false,
                [[1.1, 2.2], [3.3, 4.4]],
            ],
            [
                $iter([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                1,
                true,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                $iter([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                1,
                false,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                $iter(['1', '2', '3', '4', '5']),
                2,
                0,
                true,
                [['1', '2'], ['3', '4'], ['5']],
            ],
            [
                $iter(['1', '2', '3', '4', '5']),
                2,
                0,
                false,
                [['1', '2'], ['3', '4']],
            ],
            [
                $iter(['1', '2', '3', '4', '5']),
                2,
                1,
                true,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                $iter(['1', '2', '3', '4', '5']),
                2,
                1,
                false,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                $iter([[1], [2], [3], [4], [5]]),
                2,
                0,
                true,
                [[[1], [2]], [[3], [4]], [[5]]],
            ],
            [
                $iter([[1], [2], [3], [4], [5]]),
                2,
                0,
                false,
                [[[1], [2]], [[3], [4]]],
            ],
            [
                $iter([[1], [2], [3], [4], [5]]),
                2,
                1,
                true,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                $iter([[1], [2], [3], [4], [5]]),
                2,
                1,
                false,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                $iter([true, true, false, false]),
                2,
                0,
                true,
                [[true, true], [false, false]],
            ],
            [
                $iter([true, true, false, false]),
                2,
                0,
                false,
                [[true, true], [false, false]],
            ],
            [
                $iter([true, true, false, false]),
                2,
                1,
                true,
                [[true, true], [true, false], [false, false]],
            ],
            [
                $iter([true, true, false, false]),
                2,
                1,
                false,
                [[true, true], [true, false], [false, false]],
            ],
            [
                $iter([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                0,
                true,
                [[1, 2.2, '3'], [[4], true, null], ['test data']],
            ],
            [
                $iter([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                0,
                false,
                [[1, 2.2, '3'], [[4], true, null]],
            ],
            [
                $iter([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                1,
                true,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                $iter([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                1,
                false,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                $iter([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                2,
                true,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                $iter([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                2,
                false,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                $iter(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                0,
                true,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                $iter(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                0,
                false,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                $iter(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                1,
                true,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                $iter(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                1,
                false,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9], [9, 10]],
            ],
            [
                $iter([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]),
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForTraversables
     * @param \Traversable $data
     * @param int $chunkSize
     * @param int $overlapSize
     * @param bool $excludeIncompleteTail
     * @param array $expected
     */
    public function testTraversables(\Traversable $data, int $chunkSize, int $overlapSize, bool $excludeIncompleteTail, array $expected): void
    {
        // Given
        $result = [];

        // When
        foreach (Single::chunkwiseOverlap($data, $chunkSize, $overlapSize, $excludeIncompleteTail) as $datum) {
            $result[] = $datum;
        }

        // Then
        $this->assertEquals($expected, $result);
    }

    public function dataProviderForTraversables(): array
    {
        $trav = static function (array $data) {
            return new IteratorAggregateFixture($data);
        };

        return [
            [
                $trav([]),
                1,
                0,
                true,
                [],
            ],
            [
                $trav([]),
                1,
                0,
                false,
                [],
            ],
            [
                $trav([]),
                2,
                0,
                true,
                [],
            ],
            [
                $trav([]),
                2,
                0,
                false,
                [],
            ],
            [
                $trav([]),
                2,
                1,
                true,
                [],
            ],
            [
                $trav([]),
                2,
                1,
                false,
                [],
            ],
            [
                $trav([1]),
                1,
                0,
                true,
                [[1]],
            ],
            [
                $trav([1]),
                1,
                0,
                false,
                [[1]],
            ],
            [
                $trav([1]),
                1,
                0,
                true,
                [[1]],
            ],
            [
                $trav([1]),
                1,
                0,
                false,
                [[1]],
            ],
            [
                $trav([1]),
                2,
                0,
                true,
                [[1]],
            ],
            [
                $trav([1]),
                2,
                0,
                false,
                [],
            ],
            [
                $trav([1]),
                2,
                1,
                true,
                [[1]],
            ],
            [
                $trav([1]),
                2,
                1,
                false,
                [],
            ],
            [
                $trav([1]),
                3,
                0,
                true,
                [[1]],
            ],
            [
                $trav([1]),
                3,
                0,
                false,
                [],
            ],
            [
                $trav([1]),
                3,
                1,
                true,
                [[1]],
            ],
            [
                $trav([1]),
                3,
                1,
                false,
                [],
            ],
            [
                $trav([1]),
                3,
                2,
                true,
                [[1]],
            ],
            [
                $trav([1]),
                3,
                2,
                false,
                [],
            ],
            [
                $trav([1, 2]),
                1,
                0,
                true,
                [[1], [2]],
            ],
            [
                $trav([1, 2]),
                1,
                0,
                false,
                [[1], [2]],
            ],
            [
                $trav([1, 2]),
                2,
                0,
                true,
                [[1, 2]],
            ],
            [
                $trav([1, 2]),
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                $trav([1, 2]),
                2,
                1,
                true,
                [[1, 2]],
            ],
            [
                $trav([1, 2]),
                2,
                1,
                false,
                [[1, 2]],
            ],
            [
                $trav([1, 2]),
                3,
                0,
                true,
                [[1, 2]],
            ],
            [
                $trav([1, 2]),
                3,
                0,
                false,
                [],
            ],
            [
                $trav([1, 2]),
                3,
                1,
                true,
                [[1, 2]],
            ],
            [
                $trav([1, 2]),
                3,
                1,
                false,
                [],
            ],
            [
                $trav([1, 2]),
                3,
                2,
                true,
                [[1, 2]],
            ],
            [
                $trav([1, 2]),
                3,
                2,
                false,
                [],
            ],
            [
                $trav([1, 2, 3]),
                1,
                0,
                true,
                [[1], [2], [3]],
            ],
            [
                $trav([1, 2, 3]),
                1,
                0,
                false,
                [[1], [2], [3]],
            ],
            [
                $trav([1, 2, 3]),
                2,
                0,
                true,
                [[1, 2], [3]],
            ],
            [
                $trav([1, 2, 3]),
                2,
                0,
                false,
                [[1, 2]],
            ],
            [
                $trav([1, 2, 3]),
                2,
                1,
                true,
                [[1, 2], [2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                2,
                1,
                false,
                [[1, 2], [2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                3,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                3,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                3,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                3,
                2,
                false,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                4,
                0,
                true,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                4,
                0,
                false,
                [],
            ],
            [
                $trav([1, 2, 3]),
                4,
                1,
                true,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                4,
                1,
                false,
                [],
            ],
            [
                $trav([1, 2, 3]),
                4,
                2,
                true,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                4,
                2,
                false,
                [],
            ],
            [
                $trav([1, 2, 3]),
                4,
                3,
                true,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3]),
                4,
                3,
                false,
                [],
            ],
            [
                $trav([1, 2, 3, 4]),
                2,
                0,
                true,
                [[1, 2], [3, 4]],
            ],
            [
                $trav([1, 2, 3, 4]),
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                $trav([1, 2, 3, 4]),
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                $trav([1, 2, 3, 4]),
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4]],
            ],
            [
                $trav([1, 2, 3, 4]),
                3,
                0,
                true,
                [[1, 2, 3], [4]],
            ],
            [
                $trav([1, 2, 3, 4]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3, 4]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4]],
            ],
            [
                $trav([1, 2, 3, 4]),
                3,
                1,
                false,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3, 4]),
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                $trav([1, 2, 3, 4]),
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                2,
                0,
                true,
                [[1, 2], [3, 4], [5]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                2,
                0,
                false,
                [[1, 2], [3, 4]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                2,
                1,
                true,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                2,
                1,
                false,
                [[1, 2], [2, 3], [3, 4], [4, 5]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                3,
                0,
                true,
                [[1, 2, 3], [4, 5]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                3,
                0,
                false,
                [[1, 2, 3]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                3,
                2,
                true,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                $trav([1, 2, 3, 4, 5]),
                3,
                2,
                false,
                [[1, 2, 3], [2, 3, 4], [3, 4, 5]],
            ],
            [
                $trav([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                0,
                true,
                [[1.1, 2.2], [3.3, 4.4], [5.5]],
            ],
            [
                $trav([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                0,
                false,
                [[1.1, 2.2], [3.3, 4.4]],
            ],
            [
                $trav([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                1,
                true,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                $trav([1.1, 2.2, 3.3, 4.4, 5.5]),
                2,
                1,
                false,
                [[1.1, 2.2], [2.2, 3.3], [3.3, 4.4], [4.4, 5.5]],
            ],
            [
                $trav(['1', '2', '3', '4', '5']),
                2,
                0,
                true,
                [['1', '2'], ['3', '4'], ['5']],
            ],
            [
                $trav(['1', '2', '3', '4', '5']),
                2,
                0,
                false,
                [['1', '2'], ['3', '4']],
            ],
            [
                $trav(['1', '2', '3', '4', '5']),
                2,
                1,
                true,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                $trav(['1', '2', '3', '4', '5']),
                2,
                1,
                false,
                [['1', '2'], ['2', '3'], ['3', '4'], ['4', '5']],
            ],
            [
                $trav([[1], [2], [3], [4], [5]]),
                2,
                0,
                true,
                [[[1], [2]], [[3], [4]], [[5]]],
            ],
            [
                $trav([[1], [2], [3], [4], [5]]),
                2,
                0,
                false,
                [[[1], [2]], [[3], [4]]],
            ],
            [
                $trav([[1], [2], [3], [4], [5]]),
                2,
                1,
                true,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                $trav([[1], [2], [3], [4], [5]]),
                2,
                1,
                false,
                [[[1], [2]], [[2], [3]], [[3], [4]], [[4], [5]]],
            ],
            [
                $trav([true, true, false, false]),
                2,
                0,
                true,
                [[true, true], [false, false]],
            ],
            [
                $trav([true, true, false, false]),
                2,
                0,
                false,
                [[true, true], [false, false]],
            ],
            [
                $trav([true, true, false, false]),
                2,
                1,
                true,
                [[true, true], [true, false], [false, false]],
            ],
            [
                $trav([true, true, false, false]),
                2,
                1,
                false,
                [[true, true], [true, false], [false, false]],
            ],
            [
                $trav([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                0,
                true,
                [[1, 2.2, '3'], [[4], true, null], ['test data']],
            ],
            [
                $trav([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                0,
                false,
                [[1, 2.2, '3'], [[4], true, null]],
            ],
            [
                $trav([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                1,
                true,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                $trav([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                1,
                false,
                [[1, 2.2, '3'], ['3', [4], true], [true, null, 'test data']],
            ],
            [
                $trav([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                2,
                true,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                $trav([1, 2.2, '3', [4], true, null, 'test data']),
                3,
                2,
                false,
                [[1, 2.2, '3'], [2.2, '3', [4]], ['3', [4], true], [[4], true, null], [true, null, 'test data']],
            ],
            [
                $trav(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                0,
                true,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                $trav(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                0,
                false,
                [['Ross', 'Rachel'], ['Chandler', 'Monica'], ['Joey', 'Phoebe']],
            ],
            [
                $trav(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                1,
                true,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                $trav(['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe']),
                2,
                1,
                false,
                [['Ross', 'Rachel'], ['Rachel', 'Chandler'], ['Chandler', 'Monica'], ['Monica', 'Joey'], ['Joey', 'Phoebe']],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]),
                3,
                1,
                true,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9], [9, 10]],
            ],
            [
                $trav([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]),
                3,
                1,
                false,
                [[1, 2, 3], [3, 4, 5], [5, 6, 7], [7, 8, 9]],
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForChunkSizeError
     * @param iterable $data
     * @param int $chunkSize
     * @param int $overlapSize
     * @param bool $excludeIncompleteTail
     */
    public function testChunkSizeError(iterable $data, int $chunkSize, int $overlapSize, bool $excludeIncompleteTail): void
    {
        // Then
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("Chunk size must be ≥ 1. Got {$chunkSize}");

        // When
        foreach (Single::chunkwiseOverlap($data, $chunkSize, $overlapSize, $excludeIncompleteTail) as $_) {
            break;
        }
    }

    public function dataProviderForChunkSizeError(): array
    {
        return [
            [
                [],
                0,
                0,
                true,
            ],
            [
                [],
                0,
                0,
                false,
            ],
            [
                [],
                -1,
                0,
                true,
            ],
            [
                [],
                -1,
                0,
                false,
            ],
            [
                [1],
                0,
                0,
                true,
            ],
            [
                [1],
                0,
                0,
                false,
            ],
            [
                [1],
                -1,
                0,
                true,
            ],
            [
                [1],
                -1,
                0,
                false,
            ],
            [
                [1, 2],
                0,
                0,
                true,
            ],
            [
                [1, 2],
                0,
                0,
                false,
            ],
            [
                [1, 2, 3],
                0,
                0,
                true,
            ],
            [
                [1, 2, 3],
                0,
                0,
                false,
            ],
            [
                [1, 2, 3],
                -1,
                0,
                true,
            ],
            [
                [1, 2, 3],
                -1,
                0,
                false,
            ],
            [
                [1, 2, 3],
                -2,
                0,
                true,
            ],
            [
                [1, 2, 3],
                -2,
                0,
                false,
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                0,
                0,
                true,
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                0,
                0,
                false,
            ],
            [
                ['1', '2', '3', '4', '5'],
                -1,
                0,
                true,
            ],
            [
                ['1', '2', '3', '4', '5'],
                -1,
                0,
                false,
            ],
            [
                [[1], [2], [3], [4], [5]],
                -2,
                0,
                true,
            ],
            [
                [[1], [2], [3], [4], [5]],
                -2,
                0,
                false,
            ],
            [
                [true, true, false, false],
                0,
                0,
                true,
            ],
            [
                [true, true, false, false],
                0,
                0,
                false,
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                -1,
                0,
                true,
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                -1,
                0,
                false,
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                -2,
                0,
                true,
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                -2,
                0,
                false,
            ],
        ];
    }

    /**
     * @dataProvider dataProviderForOverlapSizeError
     * @param iterable $data
     * @param int $chunkSize
     * @param int $overlapSize
     * @param bool $excludeIncompleteTail
     */
    public function testOverlapSizeError(iterable $data, int $chunkSize, int $overlapSize, bool $excludeIncompleteTail): void
    {
        // Then
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("Overlap size must be less than chunk size");

        // When
        foreach (Single::chunkwiseOverlap($data, $chunkSize, $overlapSize, $excludeIncompleteTail) as $_) {
            break;
        }
    }

    public function dataProviderForOverlapSizeError(): array
    {
        return [
            [
                [],
                1,
                1,
                true,
            ],
            [
                [],
                1,
                1,
                false,
            ],
            [
                [],
                1,
                2,
                true,
            ],
            [
                [],
                1,
                2,
                false,
            ],
            [
                [1],
                1,
                1,
                true,
            ],
            [
                [1],
                1,
                1,
                false,
            ],
            [
                [1],
                1,
                2,
                true,
            ],
            [
                [1],
                1,
                2,
                false,
            ],
            [
                [1, 2],
                1,
                1,
                true,
            ],
            [
                [1, 2],
                1,
                1,
                false,
            ],
            [
                [1, 2],
                1,
                2,
                true,
            ],
            [
                [1, 2],
                1,
                2,
                false,
            ],
            [
                [1, 2, 3],
                1,
                1,
                true,
            ],
            [
                [1, 2, 3],
                1,
                1,
                false,
            ],
            [
                [1, 2, 3],
                1,
                2,
                true,
            ],
            [
                [1, 2, 3],
                1,
                2,
                false,
            ],
            [
                [1, 2, 3],
                2,
                2,
                true,
            ],
            [
                [1, 2, 3],
                2,
                2,
                false,
            ],
            [
                [1, 2, 3],
                2,
                3,
                true,
            ],
            [
                [1, 2, 3],
                2,
                3,
                false,
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                1,
                1,
                true,
            ],
            [
                [1.1, 2.2, 3.3, 4.4, 5.5],
                1,
                1,
                false,
            ],
            [
                ['1', '2', '3', '4', '5'],
                2,
                2,
                true,
            ],
            [
                ['1', '2', '3', '4', '5'],
                2,
                2,
                false,
            ],
            [
                [[1], [2], [3], [4], [5]],
                1,
                1,
                true,
            ],
            [
                [[1], [2], [3], [4], [5]],
                1,
                1,
                false,
            ],
            [
                [true, true, false, false],
                2,
                2,
                true,
            ],
            [
                [true, true, false, false],
                2,
                2,
                false,
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                1,
                1,
                true,
            ],
            [
                [1, 2.2, '3', [4], true, null, 'test data'],
                1,
                1,
                false,
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                2,
                2,
                true,
            ],
            [
                ['Ross', 'Rachel', 'Chandler', 'Monica', 'Joey', 'Phoebe'],
                2,
                2,
                false,
            ],
        ];
    }

    /**
     * @test iterator_to_array
     * @dataProvider dataProviderForArray
     * @param array $data
     * @param int $chunkSize
     * @param int $overlapSize
     * @param bool $excludeIncompleteTail
     * @param array $expected
     */
    public function testIteratorToArray(array $data, int $chunkSize, int $overlapSize, bool $excludeIncompleteTail, array $expected): void
    {
        // Given
        $iterator = Single::chunkwiseOverlap($data, $chunkSize, $overlapSize, $excludeIncompleteTail);

        // When
        $result = iterator_to_array($iterator);

        // Then
        $this->assertEquals($expected, $result);
    }
}
