﻿using System;
using System.Collections.Generic;
using System.Net;
using Ductus.FluentDocker.Model.Common;
using Ductus.FluentDocker.Model.Containers;

namespace Ductus.FluentDocker.Services
{
  public interface IContainerService : IService
  {
    /// <summary>
    ///   The container id of the running container.
    /// </summary>
    string Id { get; }

    /// <summary>
    /// The instance id if multiple instances of same container. May be <see cref="string.Empty"/>.
    /// </summary>
    string InstanceId { get; }

    /// <summary>
    /// If part of a service, the name of the service is in this property. Otherwise <see cref="string.Empty"/>
    /// </summary>
    string Service { get; }

    /// <summary>
    ///   The <see cref="System.Uri" /> to the docker daemon in control of this service.
    /// </summary>
    DockerUri DockerHost { get; }

    /// <summary>
    ///   When set to true the container is stopped automatically on <see cref="IDisposable.Dispose()" />.
    /// </summary>
    bool StopOnDispose { get; set; }

    /// <summary>
    ///   When set to true the container is removed automatically on <see cref="IDisposable.Dispose()" />.
    /// </summary>
    bool RemoveOnDispose { get; set; }

    /// <summary>
    /// Optionally exposes a custom resolver. When this is set, it will override the default container endpoint IP resolver.
    /// </summary>
    /// <value>A custom resolver.</value>
    /// <remarks>
    /// If the resolver returns null it will use the built-in behavior. It is true, when no custom resolver has 
    /// been set on the service.
    /// </remarks>
    Func<Dictionary<string, HostIpEndpoint[]>, string, Uri, IPEndPoint> CustomEndpointResolver { get; }


    /// <summary>
    ///   Determines if this container is based on a windows image or linux image.
    /// </summary>
    bool IsWindowsContainer { get; }

    /// <summary>
    ///   Paths to where certificates resides for this service.
    /// </summary>
    ICertificatePaths Certificates { get; }

    /// <summary>
    /// The image the running container is based on.
    /// </summary>
    IContainerImageService Image { get; }

    /// <summary>
    ///   Gets the configuration from the docker host for this container.
    /// </summary>
    /// <param name="fresh">If a new copy is wanted or a cached one. If non has been requested it will fetch one and cache it.</param>
    /// <remarks>
    ///   This is not cached, thus it will go to the docker daemon each time.
    /// </remarks>
    Container GetConfiguration(bool fresh = false);

    /// <summary>
    ///   Overridden to handle fluent access.
    /// </summary>
    /// <returns></returns>
    new IContainerService Start();

    /// <summary>
    ///   Gets all volumes attached to this container.
    /// </summary>
    /// <returns>A list with zero or more volumes.</returns>
    IList<IVolumeService> GetVolumes();

    /// <summary>
    ///   Gets all networks that this container is attached to.
    /// </summary>
    /// <returns>A list with one or more networks.</returns>
    IList<INetworkService> GetNetworks();
  }
}
