<?php
declare(strict_types=1);

namespace App\Model\Table;

use ArrayObject;
use Cake\Event\Event;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\ORM\TableRegistry;
use Cake\Validation\Validator;

/**
 * AttachmentsLinks Model
 *
 * @property \App\Model\Table\AttachmentsTable|\Cake\ORM\Association\BelongsTo $Attachments
 * @property \App\Model\Table\ForeignsTable|\Cake\ORM\Association\BelongsTo $Foreigns
 * @method \App\Model\Entity\AttachmentsLink get($primaryKey, $options = [])
 * @method \App\Model\Entity\AttachmentsLink newEntity($data = null, array $options = [])
 * @method \App\Model\Entity\AttachmentsLink[] newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\AttachmentsLink|bool save(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\AttachmentsLink|bool saveOrFail(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\AttachmentsLink patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method \App\Model\Entity\AttachmentsLink[] patchEntities($entities, array $data, array $options = [])
 * @method \App\Model\Entity\AttachmentsLink findOrCreate($search, callable $callback = null, $options = [])
 */
class AttachmentsLinksTable extends Table
{
    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('attachments_links');
        $this->setDisplayField('id');
        $this->setPrimaryKey('id');

        $this->belongsTo('Attachments', [
            'foreignKey' => 'attachment_id',
        ]);
        $this->belongsTo('Profiles', [
            'className' => 'Profiles',
            'conditions' => ['class' => 'Profile'],
            'foreignKey' => 'foreign_id',
            'joinType' => 'INNER',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->integer('id')
            ->allowEmptyString('id', 'create');

        $validator
            ->scalar('class')
            ->maxLength('class', 7)
            ->requirePresence('class', 'create')
            ->notEmptyString('class');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn(['attachment_id'], 'Attachments'));

        return $rules;
    }

    /**
     * Aftersave event handler
     *
     * @param \Cake\Event\Event $event Event object
     * @param \App\Model\Entity\AttachmentsLink $entity Entity object
     * @param \ArrayObject $options Options
     * @return void
     */
    public function afterSave(Event $event, \App\Model\Entity\AttachmentsLink $entity, ArrayObject $options)
    {
        if ($entity->class == 'Profile') {
            $profilesToUpdate = [$entity->foreign_id];
            $previousProfileId = $entity->getOriginal('foreign_id');
            if ($entity->foreign_id != $previousProfileId) {
                $profilesToUpdate[] = $previousProfileId;
            }
            /** @var \App\Model\Table\ProfilesTable $ProfilesTable */
            $ProfilesTable = TableRegistry::get('Profiles');
            $ProfilesTable->updateAttachmentCount($profilesToUpdate);
        }
    }

    /**
     * Afterdelete event handler
     *
     * @param \Cake\Event\Event $event Event object
     * @param \App\Model\Entity\AttachmentsLink $entity Entity object
     * @param \ArrayObject $options Options
     * @return void
     */
    public function afterDelete(Event $event, \App\Model\Entity\AttachmentsLink $entity, ArrayObject $options)
    {
        if ($entity->class == 'Profile') {
            /** @var \App\Model\Table\ProfilesTable $ProfilesTable */
            $ProfilesTable = TableRegistry::get('Profiles');
            $ProfilesTable->updateAttachmentCount([$entity->foreign_id]);
        }
    }

    /**
     * Link profile and attachment
     *
     * @param int $profileId Profile id
     * @param string $attachmentId Attachment id
     * @return bool
     */
    public function linkProfile($profileId, $attachmentId)
    {
        if (!$this->exists(['attachment_id' => $attachmentId, 'class' => 'Profile', 'foreign_id' => $profileId])) {
            $attachemntsLink = $this->newEntity(
                ['attachment_id' => $attachmentId, 'class' => 'Profile', 'foreign_id' => $profileId]
            );

            return (bool)$this->save($attachemntsLink);
        } else {
            return false;
        }
    }
}
