---
jupytext:
  formats: ipynb,md:myst
  text_representation:
    extension: .md
    format_name: myst
    format_version: 0.13
    jupytext_version: 1.14.4
kernelspec:
  display_name: Python 3 (ipykernel)
  language: python
  name: python3
---

+++ {"id": "ab415756"}

# Classify text (MRPC) with Albert

This tutorial contains complete code to fine-tune Albert to perform binary classification on (MRPC) dataset. 
In addition to training a model, you will learn how to preprocess text into an appropriate format.

In this notebook, you will:

- Load the MRPC dataset from HuggingFace
- Load Albert Model using tf-transformers
- Build train and validation dataset (on the fly) feature preparation using
tokenizer from tf-transformers.
- Build your own model by combining Albert with a classifier
- Train your own model, fine-tuning Albert as part of that
- Save your model and use it to classify sentences
- Use the end-to-end (preprocessing + inference) in production setup

If you're new to working with the MNLI dataset, please see [MRPC](https://huggingface.co/datasets/glue/viewer/mrpc) for more details.

```{code-cell} ipython3
---
colab:
  base_uri: https://localhost:8080/
id: cb427407
outputId: 65b8c4b4-71ef-4420-e6d7-38daa99887b4
---
!pip install tf-transformers

!pip install sentencepiece

!pip install tensorflow-text

!pip install transformers

!pip install wandb

!pip install datasets
```

```{code-cell} ipython3

```

```{code-cell} ipython3
---
colab:
  base_uri: https://localhost:8080/
id: 03110bb6
outputId: 591f2643-b4c6-4ddb-c1d6-8372631c8c2f
---
import os
os.environ['TF_CPP_MIN_LOG_LEVEL'] = '3' # Supper TF warnings

import tensorflow as tf
import tensorflow_text as tf_text
import datasets
import wandb

print("Tensorflow version", tf.__version__)
print("Tensorflow text version", tf_text.__version__)
print("Devices", tf.config.list_physical_devices())

from tf_transformers.models import AlbertModel, Classification_Model, AlbertTokenizerTFText
from tf_transformers.core import Trainer
from tf_transformers.optimization import create_optimizer
from tf_transformers.losses.loss_wrapper import get_1d_classification_loss
```

```{code-cell} ipython3
:id: 24a3ff45


```

+++ {"id": "13fc720c"}

### Load Model, Optimizer , Trainer

Our Trainer expects ```model```, ```optimizer``` and ```loss``` to be a function.

```{code-cell} ipython3
:id: dfd76273

# Load Model
def get_model(model_name, num_classes, is_training, use_dropout):
  """Get Model"""

  def model_fn():
    model = AlbertModel.from_pretrained(model_name)
    model = Classification_Model(model, num_classes=num_classes, is_training=is_training, use_dropout=use_dropout)
    return model.get_model()
  return model_fn

# Load Optimizer
def get_optimizer(learning_rate, examples, batch_size, epochs, use_constant_lr=False):
    """Get optimizer"""
    steps_per_epoch = int(examples / batch_size)
    num_train_steps = steps_per_epoch * epochs
    warmup_steps = int(0.1 * num_train_steps)

    def optimizer_fn():
        optimizer, learning_rate_fn = create_optimizer(learning_rate, num_train_steps, warmup_steps, use_constant_lr=use_constant_lr)
        return optimizer

    return optimizer_fn

# Load trainer
def get_trainer(distribution_strategy, num_gpus=0, tpu_address=None):
    """Get Trainer"""
    trainer = Trainer(distribution_strategy, num_gpus=num_gpus, tpu_address=tpu_address)
    return trainer
```

```{code-cell} ipython3
:id: fd16d457


```

+++ {"id": "8fb97a79"}

### Prepare Data for Training

We will make use of ```Tensorflow Text``` based tokenizer to do ```on-the-fly``` preprocessing, without having any
overhead of pre prepapre the data in the form of ```pickle```, ```numpy``` or ```tfrecords```.

```{code-cell} ipython3
:id: e723e859

# Load dataset
def load_dataset(dataset, tokenizer_layer, max_seq_len, batch_size, drop_remainder):
    """
    Args:
      dataset; HuggingFace dataset
      tokenizer_layer: tf-transformers tokenizer
      max_seq_len: int (maximum sequence length of text)
      batch_size: int (batch_size)
      drop_remainder: bool (to drop remaining batch_size, when its uneven)
    """
    def parse(item):
        input_ids = tokenizer_layer({'text': item['sentence1'] + '  ' + item['sentence2']})
        # Truncate to max_seq_len-2 (2 is for CLS and SEP)
        input_ids = input_ids[:, :max_seq_len-2]
        # Add CLS and SEP
        input_ids = tf_text.combine_segments(
                      [input_ids], start_of_sequence_id=tokenizer_layer.cls_token_id, end_of_segment_id=tokenizer_layer.sep_token_id
                  )[0]
        input_ids, input_mask = tf_text.pad_model_inputs(input_ids, max_seq_length=max_seq_len)

        result = {}
        result['input_ids'] = input_ids
        result['input_mask'] = input_mask
        result['input_type_ids'] = tf.zeros_like(input_ids)

        labels = {}
        labels['labels'] = tf.expand_dims(item['label'], 1)
        return result, labels

    tfds_dict = dataset.to_dict()
    tfdataset = tf.data.Dataset.from_tensor_slices(tfds_dict).shuffle(100)

    tfdataset = tfdataset.batch(batch_size, drop_remainder=drop_remainder)
    tfdataset = tfdataset.map(parse, num_parallel_calls =tf.data.AUTOTUNE)
    
    # Shard
    options = tf.data.Options()
    options.experimental_distribute.auto_shard_policy = tf.data.experimental.AutoShardPolicy.AUTO
    tfdataset = tfdataset.with_options(options)
    
    return tfdataset
```

```{code-cell} ipython3
:id: 61e76654


```

+++ {"id": "323938e7"}

### Prepare Dataset

1. Set necessay hyperparameters.
2. Prepare ```train dataset```, ```validation dataset```.
3. Load ```model```, ```optimizer```, ```loss``` and ```trainer```.

```{code-cell} ipython3
---
colab:
  base_uri: https://localhost:8080/
  height: 461
  referenced_widgets: [ebef5bec9f6d4fd98b6f4a30e7d56f23, 7c1514a7db444b309592eb41d0f3c4dc,
    d2bf411d06e741b29030693c0a4ee660, 27c76b8276f5465a91257b30074acf81, 64b33e4d4f294147a17092625d5ad01e,
    729b2975e0ad4675b72c20eb8bb33350, 6a837bbe38cd425586160a46c24da023, e0929753196042d6b7079351983daf03,
    4393d78a009042e1bbe50ce4832316c4, 2ca2a9e05a2d4c1eaaf925dcc75bd89d, 9a200a9429e44116bba5ffedc66151fb,
    5e4db260686a470188037bef58a050ec, 684ff46403ce4e68a18479cf9d6ae8a0, 2f82f138149240649ebe20c0870acb7c,
    bb93283717e94e4287de3ad67c7d1710, 0577783a504345e2aff88169f4128a0d, 7a290825c1404c94bf9a198796c37664,
    2e3d009296c7489a8f29dcc55ce66489, 6f8465878fc64b7a97a1e113c7b19303, 885bcfc398644a40bede1a85320e3775,
    58f2bfd77c1848da82cf92f5c8b2c26c, 6a15a1d56b9644fe9cf9a7c483dabe8d, d08661620f894441b17a6be00b7bb92c,
    d04eff5aa72a4489bdf88b65c6c87d8d, 3745c2d5c16e4f91aae323253a91a5ac, bd36aca9f4834fbcb9f30d664ead8d85,
    9e962cb6ffc5433ca2432289c0bb8991, 7ace0f2c50ed4d4289df7cce0c112f45, 2c7089c24b6a41ed8148ba3b659fb3bb,
    6548c93ca96f4c06b9fb2f77289f6a17, 5b139638d9674095997bb8c2901968b7, 1bd089494a394611bed02586d5bcd2d8,
    fb0413b015a3462d9cb1a3da80800a6b, a1bfc36fdd3c4e11878a321ab15525cd, 7992325b7d4c4982a7b1acaa1e807ee5,
    7517cf45faa248e69c51f548c56656d7, e65d2d4f646749f3b709bc6180e505e0, 8a5f2cfa34cc41d99a9cb11cebaa0e6d,
    f89af68b3f104c8a9f509342f04ae409, 0063f28fa49645b497a677b7e5d3555b, 081135059fe5455c93ec1e2ffcf4e1f1,
    e2658779fbac468aa398ac4ce1c718ca, db8022d612324398b5f81e927ebfbd51, dffadcc90d634de780dadf8e510939c5,
    ae700d77887349838973277622f392c2, 0a866b45faae4e528d72dffff83cd836, 54daa704479f45069ca262407e6fd30d,
    de8d9162b86e4e4bb382b66930340a58, 2e0f78cfb08347319f1fafa2091ca949, 0b332f6736844446a62015a8efaf0959,
    239c24e2d8fc42499fd30ae9709a7d4c, c4bc7f8d13004704bb13b8dd01d45c82, f417f5cb758b445ba929b16aa08128b6,
    6234e673d13e4f07b4954604d38a50d7, 690da86d5ffa43ea9f81e4a3ae4593c9, e768affcde1b485fa2871f68d4f6aa4d,
    cbaff1e89ad84f738cb372ab1f3a4064, a0dd7d26a016453e9ffb8b94e3785ae9, 35370392cbea490eb04bd78debd4272b,
    8f57eae29b81417ba44570a2f83461b0, c4c4cd4aaa3442298cd44f5eb7731b38, 791e83ff986e49c4acb321095beb18d4,
    dc8070536ddc41249cf45371a0eec544, d5930cf57ebf4380b769df08596a433f, 50a1ad0b966e497ea8d56e1ac5bdb181,
    282f07ce7339461892bb5f939b721f47, 89340f2e7a954a14be20ce6a0fca070f, fa3f440de7e84d3e8e5b9fec38c29deb,
    a85167072b23446f981557a3a89e82ab, 47dcbe197b894723bfe365e79f58daa6, b69675debee24f929b47f23bbfbb8c6e,
    98b91c41813e48278bc6ae9cbb025792, a379ce09c4b047ca9c4988bd16fef098, dd070ad376bc479cbcd5bdb0c0a83754,
    445aa129eab347cc85c0fca871154177, 4797b6508ef741859d8a89d51bdffa37, 664e7071e1cb4dc0ac3a6520b884f074,
    f5ca6c4b408c41779cc402980f7762ae, e92c375d26d041638c03f0272ef9dbd7, e0c4fb3bb5a34f22b8eb71b4e435ea7b,
    f60906af0baa4b48871386fb7573d9a8, 672e685b6a4142f6be0672188a6876fa, 31e26077de624e18ab56c2d7f1c7cebd,
    6aceef1bf45f459eb6d2f2c34ae2fba1, 5cab4a5266b1445384cca198951e1518, 427e1fa630564981baaedd7f06b9661e,
    01980d3856ab4b25b77e0f48a030bfd7, 85eb7debb99340558ee3a30e7cf7c627, e0bd643b987a41cf84ecb9aadf510e59,
    1f59edb74c30475b84a9d7794646f8de, adb9bde491114253801c25382a195045, 2853638adf484f028e540cdf29a8b73f,
    def031454173470b8b4aff77d0ae51de, 283402403cd7474cb2c3a0d98370ff08, ad7e603cc5a547b9b6397e843221088d,
    bc376186bb564befaac976f2e769b54d, 6f844935f29140bab08109f75bd7b535, d472d7065d7d48018fc251904f9dd2b6,
    bd36b9651c454533a0c10e752103651f, 649a98480b7a4bb29ab76af22ce72b28, d78ecfd6b3ff4f3a94853c0d2d0ec887,
    5c9d8ab5a70544ef83b740cf1466bc1a, 92740a6e120d4e419e64e9d947721706, 7fe7e8b1c1524ad4b845a03f4f46a0fa,
    1545b669aa9544f79c17751b57b5f9d6, a2505b016bea40f9b377b7d8c5a09397, 53d64180ebfb4fefab836c78a223966b,
    e5e8633369b342ccbd21450b0ed27c62, 6ed0a6cd53de46e1ac21148745fa2a5c, 6c2a2e21827a44bba5bec226f5a28751,
    b6e392807cf04130afd7fd83be88a9dd, 631dc167f75c44c39bc0b93bd766098f, e5bd09dd55ff4b14ade7ff4c5f2aa73c,
    2bc7ad7511d84bcb93fb6d3498676230, bd6e969f60744b17b66bfa30e24099a6, ae9dda02dd61440e9720cff6216c15f7,
    0a8eab71676b4cbc985777a19e8f7db6, f5d46be1019a4d1a80ef41d0372e5257, 1f43bfceacd44e7db19869f49e65df2a,
    5de97647ae4e4dfdbb09ce5581e0c759, 48212a11bb3643a39ef710a6c2f4d3b1, dbb1ee14dea343a8844eaefa78131f03,
    83afc86f2b234f9f8746b3c4a3a6fc8c, 131cd5ded8f54eb4914c8b5b83a7cc2d, ae5a50bf5d7249d49d00e820eb3d6720,
    5d29bceb58784a6ca19d80a5f8830602, 24ba9613cd884e3a96cf0171a142b09b, ff6da38c7a6f4b049bfe76ef8c816849,
    fdb03ab85cb048b4abd44fedfa5a1189, b9cfb27e8ac5465bb80d56acd9e3717d, 1b94ab751eba44409908947f621ca14a,
    706d571854fc42219a148d3ec93f3883, f221a5cc0d214a26bebae13f3a06cdcb, 937fbc8951714ada8df456dd9b4fdd53,
    bd41b4ee03cb48f0b55ad0c02d44b558, 68661902e31649c8ad26a5d8eb8ff0d0, cef6d29e30da4c318f4922afa81da81a,
    08167be5b83f462bb909756b7889ce64, e292354e28024c1aa28fa3e17f9dbdb3, 58de085243184b7985244b5e80e8d03b,
    4afbb47247fe49a1a24b0cb0e5820ce5, edbd54b7b31244cb856c1621bb134f43, 3d75a02372a44fd7b32d565a076c53ef]
id: 15308c97
outputId: f0ea97bb-39b6-46a5-835e-f387533eb1bf
---
# Data configs
dataset_name = 'mrpc'
model_name  = 'albert-base-v2'
max_seq_len = 128
batch_size  = 8

# Model configs
learning_rate = 1e-5
epochs = 4
num_classes = 2
model_checkpoint_dir = 'MODELS/mrpc_albert_model'

# Load HF dataset
dataset = datasets.load_dataset('glue', dataset_name)
# Load tokenizer from tf-transformers
tokenizer_layer = AlbertTokenizerTFText.from_pretrained(model_name)
# Train Dataset
train_dataset = load_dataset(dataset['train'], tokenizer_layer, max_seq_len, batch_size, drop_remainder=True)
# Validation Dataset
validation_dataset = load_dataset(dataset['validation'], tokenizer_layer, max_seq_len, batch_size, drop_remainder=False)

# Total train examples
total_train_examples = dataset['train'].num_rows
steps_per_epoch = total_train_examples // batch_size

# model
model_fn =  get_model(model_name, num_classes, is_training=True, use_dropout=True)
# optimizer
optimizer_fn = get_optimizer(learning_rate, total_train_examples, batch_size, epochs)
# trainer
trainer = get_trainer(distribution_strategy='mirrored', num_gpus=1)
# loss
loss_fn = get_1d_classification_loss()
```

```{code-cell} ipython3
:id: 9eddbdd5


```

+++ {"id": "6f9b325d"}

### Wandb configuration

```{code-cell} ipython3
:id: cbee371d

project = "TUTORIALS"
display_name = "mrpc_albert_base_v2"
wandb.init(project=project, name=display_name)
```

```{code-cell} ipython3
:id: 75de739f


```

### Accuracy Callback

```{code-cell} ipython3
:id: a3e80b87

import tqdm
from sklearn.metrics import accuracy_score

METRICS = [tf.keras.metrics.Accuracy(name="accuracy", dtype=None)]


class AccuracyCallback:
    def __init__(self, label_column: str, prediction_column: str) -> None:

        self.label_column = label_column
        self.prediction_column = prediction_column
        self.metrics = METRICS

    def __call__(self, trainer_kwargs):

        validation_dataset_distributed = iter(
            trainer_kwargs["validation_dataset_distributed"]
        )
        model = trainer_kwargs["model"]
        wandb = trainer_kwargs["wandb"]
        step = trainer_kwargs["global_step"]
        strategy = trainer_kwargs["strategy"]
        epoch = trainer_kwargs["epoch"]
        epochs = trainer_kwargs["epochs"]
        validation_steps = trainer_kwargs["validation_steps"]

        if validation_dataset_distributed is None:
            raise ValueError(
                "No validation dataset has been provided either in the trainer class, \
                                 or when callback is initialized. Please provide a validation dataset"
            )

        @tf.function
        def validate_run(dist_inputs):
            batch_inputs, batch_labels = dist_inputs
            model_outputs = model(batch_inputs)
            return tf.argmax(
                model_outputs[self.prediction_column], axis=1
            ), tf.reduce_max(model_outputs[self.prediction_column], axis=1)

        P_ids_flattened = []
        O_ids_flattened = []
        # This is a hack to make tqdm to print colour bar
        # TODO: fix it .
        pbar = tqdm.trange(validation_steps, colour="magenta", unit="batch")
        for step_counter in pbar:
            dist_inputs = next(validation_dataset_distributed)
            predicted_ids, predicted_probs = strategy.run(
                validate_run, args=(dist_inputs,)
            )
            predicted_ids = tf.concat(
                trainer.distribution_strategy.experimental_local_results(predicted_ids),
                axis=0,
            )
            predicted_probs = tf.concat(
                trainer.distribution_strategy.experimental_local_results(
                    predicted_probs
                ),
                axis=0,
            )

            # 1 in tuple of dist_inputs
            batch_labels = dist_inputs[1]
            original_ids = tf.squeeze(
                tf.concat(
                    trainer.distribution_strategy.experimental_local_results(
                        batch_labels[self.label_column]
                    ),
                    axis=0,
                ),
                axis=1,
            )
            P_ids_flattened.extend(predicted_ids)
            O_ids_flattened.extend(original_ids)
            metric_result = {}
            for metric in self.metrics:
                metric.update_state(original_ids, predicted_ids)
                metric_result[metric.name] = metric.result().numpy()
            pbar.set_description(
                "Callback: Epoch {}/{} --- Step {}/{} ".format(
                    epoch, epochs, step_counter, validation_steps
                )
            )
            pbar.set_postfix(**metric_result)
        # Result over whole dataset and reset
        metrics_result = {}
        for metric in self.metrics:
            metrics_result[metric.name] = metric.result().numpy()
            metric.reset_state()
        if wandb:
            wandb.log(metrics_result, step=step)

        metrics_result['acc_sklearn'] = accuracy_score(O_ids_flattened, P_ids_flattened)
        return metrics_result
```

```{code-cell} ipython3
:id: 391e1283


```

### Train :-)

```{code-cell} ipython3
---
colab:
  base_uri: https://localhost:8080/
id: 7b857bfa
outputId: 55c53e36-c14e-4a7e-9811-ad28af57e089
---
accuracy_callback = AccuracyCallback(label_column='labels', 
                                    prediction_column='class_logits')
history = trainer.run(
    model_fn=model_fn,
    optimizer_fn=optimizer_fn,
    train_dataset=train_dataset,
    train_loss_fn=loss_fn,
    epochs=epochs,
    steps_per_epoch=steps_per_epoch,
    model_checkpoint_dir=model_checkpoint_dir,
    batch_size=batch_size,
    validation_dataset=validation_dataset,
    validation_loss_fn=loss_fn,
    callbacks=[accuracy_callback],
    wandb=None
)
```

+++ {"id": "1f8029cc"}

### Visualize Tensorboard

```{code-cell} ipython3
---
colab:
  base_uri: https://localhost:8080/
  height: 821
  resources:
    https://localhost:6006/?tensorboardColab=true:
      data: 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
      headers: [[content-type, text/html; charset=utf-8]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/environment:
      data: eyJ2ZXJzaW9uIjogIjIuNy4wIiwgImRhdGFfbG9jYXRpb24iOiAiTU9ERUxTL21ycGNfYWxiZXJ0X21vZGVsL2xvZ3MiLCAid2luZG93X3RpdGxlIjogIiIsICJleHBlcmltZW50X25hbWUiOiAiIiwgImV4cGVyaW1lbnRfZGVzY3JpcHRpb24iOiAiIiwgImNyZWF0aW9uX3RpbWUiOiAwLjAsICJkZWJ1ZyI6IHsiZGF0YV9wcm92aWRlciI6ICJHcnBjRGF0YVByb3ZpZGVyKGFkZHI9J2xvY2FsaG9zdDo0MTYwMycpIiwgImZsYWdzIjogeyJsb2dkaXIiOiAiTU9ERUxTL21ycGNfYWxiZXJ0X21vZGVsL2xvZ3MiLCAibG9nZGlyX3NwZWMiOiAiIiwgImhvc3QiOiBudWxsLCAiYmluZF9hbGwiOiBmYWxzZSwgInBvcnQiOiBudWxsLCAicmV1c2VfcG9ydCI6IGZhbHNlLCAibG9hZF9mYXN0IjogImF1dG8iLCAiZXh0cmFfZGF0YV9zZXJ2ZXJfZmxhZ3MiOiAiIiwgImdycGNfY3JlZHNfdHlwZSI6ICJsb2NhbCIsICJncnBjX2RhdGFfcHJvdmlkZXIiOiAiIiwgInB1cmdlX29ycGhhbmVkX2RhdGEiOiB0cnVlLCAiZGIiOiAiIiwgImRiX2ltcG9ydCI6IGZhbHNlLCAiaW5zcGVjdCI6IGZhbHNlLCAidmVyc2lvbl90YiI6IGZhbHNlLCAidGFnIjogIiIsICJldmVudF9maWxlIjogIiIsICJwYXRoX3ByZWZpeCI6ICIiLCAid2luZG93X3RpdGxlIjogIiIsICJtYXhfcmVsb2FkX3RocmVhZHMiOiAxLCAicmVsb2FkX2ludGVydmFsIjogNS4wLCAicmVsb2FkX3Rhc2siOiAiYXV0byIsICJyZWxvYWRfbXVsdGlmaWxlIjogbnVsbCwgInJlbG9hZF9tdWx0aWZpbGVfaW5hY3RpdmVfc2VjcyI6IDg2NDAwLCAiZ2VuZXJpY19kYXRhIjogImF1dG8iLCAic2FtcGxlc19wZXJfcGx1Z2luIjoge30sICJjdXN0b21fcHJlZGljdF9mbiI6ICIiLCAid2l0X2RhdGFfZGlyIjogIiIsICJfX3RlbnNvcmJvYXJkX3N1YmNvbW1hbmQiOiAic2VydmUifX19
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/plugin/scalars/scalars?run=eval&tag=accuracy:
      data: W1sxNjQyMjMwNjg2LjA2OTQzNywgMSwgMC44NTc4NDMxNjA2MjkyNzI1XSwgWzE2NDIyMzA3NDYuMDE5NzI5LCAyLCAwLjg1MDQ5MDIxMjQ0MDQ5MDddLCBbMTY0MjIzMDgwNS45NTcxMTYsIDMsIDAuODY3NjQ3MDUxODExMjE4M10sIFsxNjQyMjMwODY1Ljk2MDY1NSwgNCwgMC44NzI1NDg5OTc0MDIxOTEyXV0=
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/plugin/scalars/scalars?run=eval&tag=loss:
      data: W1sxNjQyMjMwNjgwLjQ2ODgyMiwgNDAwLCAwLjM1MDc2MzkxNjk2OTI5OTNdLCBbMTY0MjIzMDc0MC40MTE5OTcsIDgwMCwgMC40NjA2NzEwMDc2MzMyMDkyM10sIFsxNjQyMjMwODAwLjM0ODg5MywgMTIwMCwgMC41ODk5NTE2OTQwMTE2ODgyXSwgWzE2NDIyMzA4NjAuMjg0ODEsIDE2MDAsIDAuNjM5MzQ4ODY0NTU1MzU4OV1d
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/plugin/scalars/scalars?run=train&tag=learning_rate:
      data: W1sxNjQyMjMwNjM2LjA0MjQ5LCAxMDAsIDIuNzU5NTYyNjc2MTQ2NzQzZS0wNl0sIFsxNjQyMjMwNjQ4LjcxNzMwOCwgMjAwLCA3Ljk1MjI4MTUyMTI4NDAyMWUtMDZdLCBbMTY0MjIzMDY2MS4zNzk3NTIsIDMwMCwgOC42MzI2NDA4NTYzNzYzODNlLTA2XSwgWzE2NDIyMzA2NzQuMDUwNzE2LCA0MDAsIDguMDg2NzkwMzMwNzA4MDI3ZS0wNl0sIFsxNjQyMjMwNjk4Ljg1MDQwMiwgNTAwLCA3LjU0MDkzODg5NTU0NDk2ODVlLTA2XSwgWzE2NDIyMzA3MTEuNTI4ODIsIDYwMCwgNi45OTUwODcwMDU2MzQ1NTllLTA2XSwgWzE2NDIyMzA3MjQuMjAzNjQyLCA3MDAsIDYuNDQ5MjM2MDI1MjE4ODUyZS0wNl0sIFsxNjQyMjMwNzM2Ljg3ODQ4NywgODAwLCA1LjkwMzM4NDEzNTMwODQ0M2UtMDZdLCBbMTY0MjIzMDc1OC43Nzk5NDgsIDkwMCwgNS4zNTc1MzI3MDAxNDUzODRlLTA2XSwgWzE2NDIyMzA3NzEuNDUxMzE1LCAxMDAwLCA0LjgxMTY4MDgxMDIzNDk3NWUtMDZdLCBbMTY0MjIzMDc4NC4xMTE2NywgMTEwMCwgNC4yNjU4Mjk4Mjk4MTkyNjhlLTA2XSwgWzE2NDIyMzA3OTYuNzc3Mjg5LCAxMjAwLCAzLjcxOTk3NzcxMjUzNTE4M2UtMDZdLCBbMTY0MjIzMDgxOC43MzAwODUsIDEzMDAsIDMuMTc0MTI2MDQ5OTk4NDQ5ZS0wNl0sIFsxNjQyMjMwODMxLjQxNjIwNCwgMTQwMCwgMi42MjgyNzUwNjk1ODI3NDE3ZS0wNl0sIFsxNjQyMjMwODQ0LjA3NzQwNSwgMTUwMCwgMi4wODI0MjM0MDcwNDYwMDhlLTA2XSwgWzE2NDIyMzA4NTYuNzIwOTc0LCAxNjAwLCAxLjUzNjU3MTc0NDUwOTI3NDFlLTA2XV0=
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/plugin/scalars/scalars?run=train&tag=loss:
      data: W1sxNjQyMjMwNjM2LjA0MTMyOCwgMTAwLCAwLjY4NDcwNjgwNzEzNjUzNTZdLCBbMTY0MjIzMDY0OC43MTYyMTEsIDIwMCwgMC42MTE3MTc0NjI1Mzk2NzI5XSwgWzE2NDIyMzA2NjEuMzc4NjYxLCAzMDAsIDAuNTQ1Nzk2NzUxOTc2MDEzMl0sIFsxNjQyMjMwNjc0LjA0OTU2NSwgNDAwLCAwLjQwOTcxMTAzMzEwNTg1MDJdLCBbMTY0MjIzMDY5OC44NDkyNzUsIDUwMCwgMC4zOTU1MDE1MjQyMDk5NzYyXSwgWzE2NDIyMzA3MTEuNTI3NjgsIDYwMCwgMC40MjI3MzAxNDc4Mzg1OTI1M10sIFsxNjQyMjMwNzI0LjIwMjU1MSwgNzAwLCAwLjM2MDc2NTkwNDE4ODE1NjEzXSwgWzE2NDIyMzA3MzYuODc3NDEyLCA4MDAsIDAuMjc4ODI4NzEwMzE3NjExN10sIFsxNjQyMjMwNzU4Ljc3ODg0LCA5MDAsIDAuMjg0NTQ5OTIxNzUxMDIyMzRdLCBbMTY0MjIzMDc3MS40NTAxMDUsIDEwMDAsIDAuMzM2MjE0MTg0NzYxMDQ3MzZdLCBbMTY0MjIzMDc4NC4xMTA2MDYsIDExMDAsIDAuMjk2ODY0NzQ4MDAxMDk4NjNdLCBbMTY0MjIzMDc5Ni43NzYyMDgsIDEyMDAsIDAuMjA2NzYyNzE2MTc0MTI1NjddLCBbMTY0MjIzMDgxOC43Mjg5NzUsIDEzMDAsIDAuMTYwODE1MzU4MTYxOTI2MjddLCBbMTY0MjIzMDgzMS40MTUwOTYsIDE0MDAsIDAuMTQ1NDE2NzgxMzA2MjY2NzhdLCBbMTY0MjIzMDg0NC4wNzYzMTUsIDE1MDAsIDAuMjM0MTY3OTc4MTY3NTMzODddLCBbMTY0MjIzMDg1Ni43MTk5MzUsIDE2MDAsIDAuMTgxMDA2Mzg2ODc2MTA2MjZdXQ==
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/plugin/scalars/tags:
      data: eyJldmFsIjogeyJsb3NzIjogeyJkaXNwbGF5TmFtZSI6ICIiLCAiZGVzY3JpcHRpb24iOiAiIn0sICJhY2N1cmFjeSI6IHsiZGlzcGxheU5hbWUiOiAiIiwgImRlc2NyaXB0aW9uIjogIiJ9fSwgInRyYWluIjogeyJsb3NzIjogeyJkaXNwbGF5TmFtZSI6ICIiLCAiZGVzY3JpcHRpb24iOiAiIn0sICJsZWFybmluZ19yYXRlIjogeyJkaXNwbGF5TmFtZSI6ICIiLCAiZGVzY3JpcHRpb24iOiAiIn19fQ==
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/plugins_listing:
      data: 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
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/data/runs:
      data: WyJldmFsIiwgInRyYWluIl0=
      headers: [[content-type, application/json]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/font-roboto/Hgo13k-tfSpn0qi1SFdUfZBw1xU1rKptJj_0jans920.woff2:
      data: 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
      headers: [[content-type, font/woff2]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/font-roboto/RxZJdnzeo3R5zSexge8UUZBw1xU1rKptJj_0jans920.woff2:
      data: d09GMgABAAAAACokAA4AAAAAUkQAACnNAAEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAGmQbmXocg0oGYACGTBEMCu1A1wwLg14AATYCJAOHNgQgBYMAByAbO0QF3Bhn2DiAgX12b1EEGwcBhTGLomxQFmT/lwnmGE77wayn0NBAJAPXITeLlQAVKYYKjM1mpr7CgS0HNgpkY1bqRLvLsXy3dA8XPXqvM/yN+w2v2FOlAb85QmOf5P7Az633/wJaSkUwMImTqgk4GDAic6S4MSrGqFakDCocigoYRBqEHnCIYBIGGExRT1Qeqv3690x3z90AwCasSP6ngswqFUVYHrB8VBQAKcYder52r1wzByMHJRZ//0+nNV9g+H/GsveOK0AqSpwZGZI47CReYMFvJOfQ2hTNUVES1lvdXXeyFKh29/XX4ACRY/9vTgMuqbMdO2B5UFAD4VG4vRkzpRE/HAS4Jss5uTZKgIn5b///mp923r+ZD/x22f0pcYRbsj0ne84XpsZN7mQyee9lwgszWcwvZJLFD4WkECjkFyHriuAA87NMWVUCV9VTC6S6tsdX+ApZK4nU+gqn6ipcefja71ffCTv/vpktBbH4Q8OmUzIhiS6SSKLxDYn4I3iKlCraxSKRmLCxMhnQLaUZLPeL70z9PLvdGe4aJpgghNJhdNDfIYfbP4Zrr4IRvQYW1AHHsRm/MoBA8QMAALCA4nacDoQBD4hYsRCpUiHSpUMwMSGyZUPkyoe4rB6qxyYEAgXAC0AAAgCBiIUA4KZAB3a3PfY7jNipySXnETvz4unnEjtv7bILiMECgG+hS5x7+iUX4AR8gRVUUNx1liijpQ3akVwcN9akGiFf5sfC53+NGKbR5WqKVWK9kAti+AS1eOOOyCvDaIwf8afMcFGbPJk65ZRuuRKVi5n34MXC5+eY8DF3ego/YaXaA/kGJCdNqR9aLDwevIQdJ0mKNBnyFChToUqNBoJTtOk4zZgJM5as2bDlwIkLV+48lSpzznk9evW57Y677uk3YNCQYfc98NAjk55Y9MySZa+9sWLVmnUbNiE0xggsAhGnnKBtjyf2QAgPTgoEFh8Jtbt2fBCTGwppEGEglZ5H9iEjajJmypb9zQ7WcvY+F29zpybfh8pFRalEVy+iPrfdcde9rn89b9acef9Z6HrqQ4ueWbLsjRWr1qx3vfW+d9770LXhbZuHCFeD868+CuUUv9RhOBpeRLDSKRtpW+4JClYxVTYdM1P8F5yw4yEH/bl6XJhQCcKlImFqL9vlsPiIpJtgDl7nnTDswuvDCv+DO1fDk/MxTTZl2ozHg0XCE4hnXuaoUGRvMwJjpuxnZEv+3pQmUBk753x56pZFeGbJ8s2IMhXxINvIiBgzZS/fU4ueWbJ8N5ZJIjmllGuC4g0HW6/PDdHeZGVFrFqzPhRHACMh5SpUzmRow4YNGzamMFQ4soqMGDNtbEWyl05HornGI/8uT9miZ5Ysd70RacWqNeumCoS86xhHXc3Jp1y9CINvDOn62prjoEx81Jz3IVIDWX7co1E3yT++FWYmuuNgIacdlV09TGcJJhPrX4ppsGwDXfCvkmLgAFmk6LCYxAftHyxYL1O0P9FCx9PR3lipv92N96FztJm7THzvXYCZF1CGmPHV7zjxlE+yUMyjYlkzeXrN1+XDXy7mZ4SaH3nFQ7Ww4uDmIe7T/PFaC3qFyJcS82v/iTr6GwvR3ze+XD27dfVbnYZQeRFxzohzSVz399nlr3kVWPXOwUJ5dHBrvN8bC/o9RRmDNlxKMCFjXvucAiWWoH0uC9Id0GRlZgrJ2SxOo/NX1BHQbaQBUf96uxZTd36ybZDQD2eu0GhiDfZmfDlc0VzFOlV8wKy9uuc9zoT+etNtsqFheWuKpVn11wnNyFUttlZgbJzVYnwrmDBpqX3O62J0xc3aVeaABaXbnkaGt5Tna0TncyyvCyiVfDTfNg2Tskx1qffMM0NtN69smvOiem3QnIGRMuk1rbqfMN9WYlYX54kVN9Zr843PpJvb6ivMNl+RmEB/BdWcgMMDITBSlFAjGMdJwzpJBRcNOoQYINvHmOh+Zu4HWLNzkAM9wsX9KDcejslxl1SqgpTK+nJ6LJP32jr7AVDWFUr1sbAX9oI4EVeZok7QfSSpICmKkKY4cpSbPOWhQPdQpggVKkKD8tGk3AhUzCkqSosK0Ka8dOguBhTPkHIg8915deAkWMcGli0ohh3l4ECFuVAublSIO+XiQfnlCI7BobVXOYT4INHaqwJUXCUV1+OO6HUPqn/XBA0YhnPfJMSUx7g9IXde/1qA99R9t0XLOL0eivXGOox6LVwgCIKWCEHQPYSoiOMCDwv1DyhKju6lTFGqFKXO+RXrwA1csBuEYRfwgABq5RhDmLJAVAIMYC0Me1CEI+XlzHGKeeAEnBafdFBUHkXlD0UUK7FHQxAEERAEEUvELYg9ALxA1QMUuICHcCSXIboBRsXRb32AMzlHPf3L87pFpik149XgSKLrYiABJiQbu7XX0EZ3qpa5pRm10HWgNpbmvXY2psKEBVBRiYumxxD0yfF+4RFhcOKf8uTBydDgQG9QA2iNNAqnhUWBFCuRLAAGylcAEIhW6rsQekPBdeKnxE2kSncIhFMQTwLtqlUHw4S5y9CmoHva/VBPrVxRoSAQJgJKgoic9kRheIYBERIBcwcACx1JTRqgQScM5u6itHBr0qhBsxat2rTrgDEjYrt2VZyzqFi6XHNdtxtQ2CIOaKHU/e2ri+Ee7MoA2fSUxbwhIBp/6EsBkrpI3jbygQfuiQiOrDHBHxAwqwyQgDwGALBLnUWCBAxYjpb9+Roy/wk3QM6CbfsB0CABSxQAnySXXv93+42ZtuaTv23HZhtjsRhL87XmG8w3mW8+33q+NN/tvsvue+KI0PgLi4TLN3UaMG7Gus+UbHrWnK8332j4spfb2B4r9owh3GMQ/P1r9sv3jkH4J/6/8X/35zeIF4eOyoczmc/Yz+9yx8tGBoAK8xg3HB/3Xk/VY2LM+/9p0k2ZNuOxJxgy/GvWnHn/WcCUORZ+dZZsb73z3gcbcmziBQDI4H0igCRkjW8HQ0ISbLrkpsva+aYOt3TqF2HAqN40JtW4af1pRo/H5k34z4I0Ty2b8twLDC+t+jetYVr3ydP0WaYvtr323Q9ZfvrtbfojJ/2tNjOyY0BujRxSHRqQl9GXAgoy+mFAYUY/DiiqJdZsXwtAMZBYr30DAKVAYqNqk4BzWWLzgAtZYuuAylpiWbUUUJUldguoyRKXBdRmiScCLuYjazwIlyBQnNDxd6jn4zYgH2sRfLeGyBuBeX8dvQgt3Aq6mTwCBwO5ip6gBxjoGBZbM34NE52ChI4XgbqSgsMohHqFmIhSL8HR1+qELePMETQQxH8ATAWSCRJ80KkVSFyshn4rVqc4xO4K9/sBbZUfGmjTrPCOlAJr8aYOsysMEbR4GDZjo5nqxAmguf2d+5ll4+q6dZTUZq1hMoksN66UXJTBBGyt+DrbhMcLq9Bk+7CpxVTXjuuYlC46w3z6kfH9bpWmwC9ElhFLbSMmAlXH7IyhWaYUCy19n4kkfj+MNwH1CXMxzHzrLGoTEVEJIpwww/SO24xCz4blyGgkPPISNVwJHMS8s9eaLgV7MO1MMFVxzgWKDObEffRpbR65hHZghKBm46hPHQIbxBUaIedU2SrMOQQSCxSYE85BZDigzEa1QKgIKEMqmHOWKIr7/orgvAATAUj2mnDy/ahrDOXUW7VsRjmHFUELlLgbeqsOaSaMtOVts1bo7cfGG5ZmMnzyvz7a9D8A49yfhKY0fT2zRlfuOMrMoba1d2Hf2SfChT0yvB6uDam/YVYHeti3rIR00JWgXBrYWqccXULUgWBDLc56ozkbZOKZwbkbwr43STuwCuPa2d9GGGB7Fc7RbV2Y1ryEAPZ+fo+bAVMVWitQuWZzibW7iEwCHXQ7lilW/mPjcU90+t1SKzITRy0tdDnD32eBJegGqTt8gwv7C7U0By0yLHifOEbuQI/HKbSqiN2A7cIrLxEuI4jzhl62d8SsW0WgmoflnBB4zekZkQIL7kLPmy8SnYVExDCJn/vsvX46iwidi74aH8QGlQbnqrSnHdb+O9sslbarcTLIeXWoS3vjlXrP/Atapqv5ib+Vp+qjuFwuDUd/fyHu9CVTIq+qFWJV1Ca09xxlk3lq/Sq37HDeHFvIRQz0Bit3uYQ2MH0kRGaKWNr6gj0uyh0nEF3uif0c7nh4lCgrKdH9hQwYPB6dSzZHuxICRr/dIPICn1SQxKhh5hC5lEbayfHCibqcyA3ZtYkTVgm64xjTZc9SxrTlX5q0if+LMeMTHtRHRueOGGKjMO15oLHaiPWlWmRl/IO10evXz7Uh09LcSPILgN4V8uqJuvCbsexNLzoP3QgU4zJftrAt4TZuhNhSaFJDq30QNy+xijFVzLR5y1ZKXp6namdX7u3I6Z6K8vco9tBP1UZPnALuwG2CMSEhWTElyCqRQIzcxyntYtKFHuO26n2pAIJzuhqKmVWMk0lxlhMvhrOMcQYnpoV7MSCclFSNxvg5F/MSasrgQr4o9P/8ce7LjPQpQTUxFy4xpt29wJlYCQSLskVnUbXUlJD+kq+gImoiUOysTerfknkgSGBDUDKkls/jNmRXBzLzuE4Pph76s3u6BjIpbNN2/uUtpLEO4NfUee3hd2ICHNJIbu7KwOJmXM0OKEjTZcEy+gJZO1A8QqI9juOkuT8zAuZZP3b47Ea8GRr/Yqom6GrAfgurEO3uc8eXUoGKktCRgBAsVnVIoJf9NmMuK5NrsY9ALjf2gU9eNkQ3qYUTAKnCxlt0ZamUlmPRKIzah/3WyZgfmmfwywWHYariaOMQdaAnLtycQZ5AEUKtcuPbwWIRiIXc0guTOqWrEHyCxSaVinmQAkGenh5YyHy4OjCmRFbrOukQ0opaxEEb9LTnu4pMNA5oajIR6FNAvzNYBLI5H1jCNkosMq20DStOahu6Tl25xsb5RqciLQK1kSpeRs15JKSgo+2DBNpTgyY1mugTZwLBQyFZ2LYikcEqfUfXzD5bqRfbmJc7cYYTstDGs2DiLeG4oBCqhtfubuK8OpzZGwftSZqHgjNcMqO0bGJkQTvYWwXWjfQkKZ/6Gt0O9Ma9RrPA7FkHm4ogchaY4T0BfhuQpl0SlqxIwD6dfNlAQepRTVGp5sm+1YGJbv55UKec+VpxVrICAWlg8rr/IVfIahPZWyD4cFFDlIMc+CTZ15JKxmYxJL5x33PQTi4/jNDXsEHs6OL1DQlR6YioBK1LayaotNggHdb6wZHpOYgdxN2h7EuKiQ2Cu82lamU02Q63JmZzS29vUgECR0IeX+G5RNlpnEnO7QNnchXLXsAOlQQHHeTBg7EsUtguvOiQEKbkgjf0n6GjHfqwIC4SWja8GiY+QtaysAIH+Xtc/S34rotjyJiIgZU5ikRm+iLHHqKCu1qwRWEv3fudKN0MuGkIb7vVjGeHHxCp9OWJ6ErT2plncvoXMmdytfNnJjFy1gw9xNMkd0saBFfI4o1358aFbq/Y7HG+0KmQY85AZYiQxA0RN7R7GoFWI0woIEO6jdfg5/lv1W9L8MdgGrzibDAjUzPbmi3IYPDcUi4SpawuXitn7HSA2yOtc0ts4mgYWjYsiSiVXBuGBQjXZXxxpS2Jq6yBdvXRk6hLpa/aV6B4YBjv08cEdkBW/TjBgnZNauhzxqZs3IZtaqmJYIwCdm2CuAwGScMv6WjknojNJSYEDVznSdIe4CUSKBCkndAmwd2jkRJS/wOiqKUozXfWEQvrk2GMFeh/k3cHmd+e5nwHpxKCSAEShab0a9gp/nOaf2S/o/xG9ll8TwiBm+JxaYSCbbEJObCxpFX4W0prjI5tAu+5849d5//w4G4tCb/Zm21f/T+Nbt3FsPz5tFFX9NlIbH+MUDEgQNPWNDZJoT5NdbIxox4IqtKPpOXydp7MulwVpi68NL3QjJdbr8VparAvCppfbCLx6mT+zMWP3/nLtb88S5po6i/tPz5fgrJign1I+C8ng+NvE7413p9rF168tNQhevfryFZSZJG3V0igtyMl6O9ysaVvgGqGS8vU4x6h4YtDo7tnP42xk5KyqVHRYYBXqWC0NOfkNTdnG6U3N+VkdtbnteAMzOxO65jaGRsbmNknmTiQDx9VYHY0ZGa1N+ST0xqbRVsbs1uw+hYOOkQjG1NTQ3NbbR0zW5Pf7ATySWpUAhnss/zDmi+ftaPolRu2TY+xLj7oy3F1rQgy9SGLmmrRS//lS2yb2xGz9qqistCVV0fiKdba0at0F5p/aiJS2moXkBRu+nbeQdPeSo9s6wkCN9L3MS9ieyHZcj2+9pNhhq58fh6l8yaHGNjLty5eiJdnZuh+NoqrBqvTz2Orv9swifLM8rOdS0p795yfyM/+IJ+ZrP3pVDKTYxpEhh8pOKo1y1L5Ha/zu0tFqbapOo5zFQVfP9S1p9gfZN4cTnie0LXRlfQ8qXwDaslp5pouMkwwuPsMOiE4aBZsOflG+ED4727GZgRNYxN9XVNjqrEZDCT0H52X7Qe9I/6E9zqfNz6qoQo7hPeaXz69V0QVtoQnI+7F0SO60X6TR2fG42gRPegI/N26X+yk+swhIP7btCIQRHWqtgyiJvtGpxYmsjNiZl/SSme/zt4Ji/uYe943oR5EfcllEZB2JjQoNCrSJzHSlR0ZfWs2gLH4Y3HYJ6Hd5x+6VMLVuBHe9WdPeF70sb1S6GFnUulRZzPjmYVAUOK4MXxDn61Pw5dKkhZ+SWJaEr0OQcdji2X+J/qSrn6ayTNrXK+e/51eTOBfc/d+4AuOj/SLTLiYk5FdGavnesQG1Hbfun/wIcp+umecGndz8Pmu/55jhWYX+XVxtsnULo8PN60YzgdNTK5k3ltIvMyZ3AAkf+lj/tJ/txSbX4a/APGYQkhYo4f8GZW4W9QBzMuFp9hX/bT43ghFd/nQxpC+T08fTX56yqdsHZrCmE1KDwtJSF6Kiz+44xkW1xdC9fcLpfWF0kDxtXdsSt3AG95nRWffzaXmLyFkYf0c3xov9MD9o/Po6sQzfuNk2yPGnpfTh58ktDOyE5tANdr8BVvFmjaoYk1lgw6b1+OBR5THfQ94Fx/8+pMaQh1UQ6ifwL0tQ7dm6M75BLKHm4+LQ5CXLRthbUwO33/58Fbd+Zq4GF0TpJCtdsrY6DQxgiXDv0ihT/A8P5cl7t3QuqBkyjQ1KTn3SXBi15Uk3FBBuF2KtIOuspLQaEZA2iKuQyBSJ5M4IjfcFcW5wfM5x+3gjWm7m5JfjmeNTykE/wmZd3no/oT7OI/gcnfKl+2fAYtdlacfU3kzjfOs1Tw9Dtic3BCSj8idAS1FWxWaDccRf9abIzQWp+/BxieuUAY4Fvs7MjriF3Ix8B/aoRRWwiT+2bfdReP76Bm04DfrWNneH9EMik9onGfaNlh0Le5++w/2ZydnfaE8OpE1Vawp0HL9y3Hc3o87gtUlOQUNrM/I29SN5u915eUZwlWyP5KdgzJtdaceGU/Xayq0jHL7rYg1jM/+QN5ab07+HAGdqByHgdsPegDH6nrUXeIA2teCTYJ/A45V8+hSlwlYwl2LgL3B127ta6hQQSejTE5FibMPfNr/6oc0nqOV9RXdiNwYw3YNWTseODkgBdYpsPNZbubQi/z2yPXYgYs7lzpH5DLsHv9+jP02v/J9dXKHGkUNC4hh0kGVWt851nI32nLbW34r7WccHf7nJBTdL39QUjEWHBhliem7iam4kUWM/VI0VWzF54bYrLdoykuh+WAdCb8fK+PiuvyukOrm4/sF1q+vzZfqCbVf7xJpP3caZmzmzhkItfsJtZYTYXYHu3UTaa7vAeS93ec+XGNz99/tivYf+A04luzXg78fz4tu/j75QCEzPykmujAvEVEH65Jr02lyNKefQ3Wlql8fGbLOE13d/MS/sdu3fjfXnfSd/UYPV1NLqlVBTPHgBw12eq/mS/JGElUMPfh2af/CphSmRNIYyekID2g8pnsxAz2DA4ljCfdZB9+sVmxGZE4l7UQVpGQkZAV7WpVSbP0mUzwbI2/umf9Uy0ktmcz+nVCXeYHBoIApdmVs5dfK0KN0MJ1jTb6V4v/+/3HuzUc7UyWlF4qOqYxBmdbeY2f3SIyyvkk0sHaD1eUgfqzRr9041pagnaRgtyT7OrL5i/+YoaCf4SxIlV5R5Dt26/HgsqTGbs3dJ4aWex4fg/DfFl2iB9MrRP+IHiyt2Aep97kfaNXLixA3Hh26BIdHZxoA79hwtwI4nlQAYsD6fAIv+xngqaASOJ5U3m0CvGMTiNs1dj2akplvk56fBM2U/vL+cpAoe/yAT243YP7wGJyNaa6b7M3ugJ5P5WQ7dz8v22AbItuZVvnAMfHeq3to+9sSKQdBtNMoyeD/R+mZme4Ohm42QDoitPNSevRecBYPdQnwwH4mKP7a2KvjnFt4VvzV6NrT2feIeRej4luQNtDocKUHY8xXMX60zvv+tDdcxzH7vnNIrQxuvcTdXMW4RdPdopNboUOSldQOFsf+X5cbbg+my7ABD0s8EaHpN++9V9z60pDUQyvXZ0zppZZHJ/eBk/D6wSNbB68k/HmVn7v8eR/qM8ydUV1FbwpipiUQvRYo3KSLfnG5AgnTQhyxZxLgCbOhu8G3e3y4m0gWxN2lq3Ze91rqXmKC9bGdjZMjvcEp3KHP9s1xfntFf+1DsIwqjmDUx+amJRsHUa/e+yz75Vsdoy+61DBxST+uNIZxF/YMj0Rn33TB5gyz+yK93DxKyKk4NuCBLZLZBDYiqmG4XvkGaaiTjRiGRrC3nlDZWN95kTQz4KQQi6bXidRmn02HhHsPXftVUw8Zq2PFQ3ei90GytP9z2iNCwEeeYYw9tWygNcxf7xxFBsbZA4HOnkG2QU4iZFhlT2Dv3SvRihZgE2D3CgGfQC8atsGlPWTfDXTy8S8lM1A2ASxOXEz88yar7JnAPu63nJfifq1kn1sVUvizxdmUfWc7q7+3Pq8/lp57B0io0K83MgPXKFSEDbjFl1xhlNSesZcn8F9wV1LuxpQT417qJp6jpvWBxfE/69JjN4KT+CgLgFtk7wRHtA69k9v61ph2h1pkELdhQCugMnrhH2W378pNmskrbMJbXIxjXrXKAVg8rkshfVe2kbzh2JT34fbNoY/9F9iGgW4OVn7GOhru2gd0rjhYxDqWfOkeyZj2PNvkvYG1p7v5evkagdicLkd/d7+bDv60TMsdmz3moqr+17qvcYDCAIWSOMrqnhy+y+6bauvSTuJiE1bh54v8tvhfe6mEf/fWE3aApGjZ9n5TiEqYWF97szYhxCBsdr5efn0LACd9+U1E7I/x/ndm/gy//TFjEV7YHj1bxoitPPcR2FT9cueJm5uemMURt70jqnhIHiQOhV88Ni8+YlkJXoFePnjPuVeD1wZfz6LXu5evKsrntqvjfi68andpd30zh/vZrg52fE2Av9cEYotXNTB/ZtZv2N+wfz+N+XNVQz73hlLXSu/Eq6FrQ69Gb19b6VYC82Eh1t3nBSYJ6hey9CROkMwC7QbbbiyWmTAIRel6hyVJthF20FL+GGGgiABNXYNsoqtHNtFQJ5vo6ZJNwKLDmR6Is1zBZI48KhZ/P/H+5uGHB5f2zz08dPPwq4mXNeI9/2GqVvCeNKrr2i51ILOS2mHH4K9mnrbrgfv7HtgEyYS74nsuj1dxfHnw89259ac93zyDgjycA1KDOL+ojwNpJqQ26eGDmsSU9LxCZpZv1ehEaH1hSV5hflFl1MBQJbWoNL+o9Byl9sGgd1VOXn6RRlp8TNjkXerSXVro5MfQh3eoz+9SQx/CEWFLnBef3f33FfI58uflZXdkVc6r5KhU/HC4LwiZfOKr4hOr26tgNDje0+rE9O3t4Bt9v31oYfoLhFfRs2LzPCzNM1z61G4r3Q1zuxDW0+xMVCRl+rUKPVz7zPPyEZtSlBwTn+NhY0d3SAscjHSuUAqzEjEVD5FMsPaxL5O7pvIiSo5mnekFuDkHJT1SNInkRyJF65EmFD78Ow0Gr+0qOi8T78x2n+m8N1tb115fXu3lauPkFpAZTYsOzHB1drbyaWA0lu8XbK27KFBXdciu8pBAI1Go5fwha4GmB33OJmMpxmN9zmV9zuSJoUGj8dvO4DkgQDKzTohOiwsOb4rpCFBOpuoaKSm08wV1pmRlXyhiaHnvs/JMlbI92pxNMQX7U4pOoHU4egIb30YCh4WrNVsLPJzAEY9FF+vzNvsutDETaXQd4n7l8Do86ZxA1eAlM10985qMQgM3bTVDQ4Ib5INKA+/V2qsgShuhXKhZOl8ZGlZZuVRdQ8lU1TdUVdE3pZBNg4zIID2c7jjuyFhg+I/7xy4IH/tlPWINXZV+ifuxkQEkGTsVgq6uWh+1uSKzsCr5bEiwv7dDbuRZEBgJs2Z4H7XTUSCeONrnWx+fVnglJpjOVCqTKqCwEgszW5PO9J3QV9E6PSbjnwFuIKwGu0XkhkFUn5CA5DmlUpoJMUJswgu8vSulNNZMbWgLcIvA4LRa4/w9P8f1Z+0w4FiQf59gbM40MSecMjcm6poagzbaUC5WEtxGq8Jn6RnKycgaHrHEHsoBpLKXMCFfEh4tDBxU70v3htT6BxuNLt4eqqm9O1zXifVxC7OycAkBLytvtzBLC7fQAKCVIfEqRUYWAcHOXhQPbzdnp2Df1e/efpwemno10dNQecXVzZzMFSWyraZhaMKISRWvjAnQIiGN33b7lu0RFVXWy1GmwPljo/uF75+3VgyMFcsEs5BTumOytJzw4Do1jEgggs2RjdXr2V2fbuJS3lK0OTQUSJorJwl3Xhst8HMoaCrZqh4ArVMDptOGyYd8CQ52mRutevc4Gv85c7D0mLlq8Lbo96oojSX65avg5sS44Ef21kk24Fhbi2vbiUWpz3PTYxGI27KeX9mcuj3f16Ij5q0fuZsoeZJo21VqlWXattAzZtV6wklh6GHSMTVvZ3uSooFLR6ZVppVT4oS5tauXVQ9mGyy8RH7nXiKazdkyWeNXq2s32971k109Apxco5z0vgiV7PSMvghnpDHHkdlN9EP2Lc6c8zXMbIrmoFGDgfrMabWUHkIm4cHjkUCE7mGo62ahdG3dNyl7V9LIwTOhsaByfO9vzmKbxkT8SnFezqvExFdZ5ZFrkwmMkkgXhebE2IdM89C2M4nWl6VNhjYVWczrdPPQgjXkb6pukZTTVJ6U1xQekTXeAaQow6+zX7e79I1No4xN9EmzWqZNsXU3CYaR3KUETWNjGzamzLVRHA8bFhT7Tw9XEMtm2t35ALnkU3NqsnxLtfq0t4zXqma7V5yNZZpukk6XlOOz+oEUJfT9tdxQEf3iHJfY0sHRrNHXx/Fb2Ma03mh2iGlsAhuiZsC3UTi2ibOklBdpIbQXCXKXN8c3Crv9Mvg7PeEwcNtIYb9vIK/GGT7Xy51TcFttsGsGXE784Jd7+TODRbS96R4K85voRlYrd05RDc25QNpQ0aLGUOcRMeZ7bkdGx/YbvpPoF87WjN5YekbPQCPzbgwts1dHoM+eniUcc2NfRGQsqKh84BkuryqrGuTjFy6E3QEy7slxrQl0L+8EemtSnMC0vC5RTp54WkFOR89HRw9Uae/Ck4q9JFOCTrWJnTCSK+MITyr31LzkrHLCSR4EG8XizVwYaGHFXA54BswA91eIN3NOQ4tr53ICg2agXTrO3C4JLS1dzAA3/2lgaSeZ22Wh1fcX+yBE7YyC73dikrljPqb84eJNX/8l4EzHeMbE+AXSY3yOe0RHB/rGRCFucEp0u6DpUIbwoYwm2HW95UX9rtguhJbQ/1cOQu3KLj9cx5W2inAPCfGH9P0pcPwQB9Ke354yH1IH759/xH5TGAfFH/kf9j9/uY2zzjffL8UPdLbCchm/u208JBNICr4x6JplVXLYYt+xiWz5qAhfL2/9ue45ZqDncpXLT/vzmYz0uG4oObvzx+8NN+eHRuJI8oBbZa8+R1MFHn98IlP+bIbNpDKnhJbvVeKqEqcWh9wmuIa+YTTXfRvYon2xpqWopLaltqKmpqO46GJ7bR9PCy0Kre1poUdgLgDPY5z+j4KHpud5z7rbdQGcMaE/7lIX+7bmwDtJuXzjnJ1w6SI5PTcjLuZKRV5qezri04u1jqIlauiR9EhkOHr0yIXiLA9eb0P9EBozP47eVMEsbW2sUnzyaf15ebOs7tG1Y8XJqiQDNSp0tNm0jdgkdZ+LgXNEaqRaulYK5VJsQV5dTEhKulakGjW4kpaaVkkTP6S65UKUp/wdbp1rdRhxkJT32gVrqroaLknM7MSYljpG7uUG5Nyp/54tvADh0sIYa582i6MGRrpEFWWiuab6KRJcy7vdejvud/wYOrmrQ3UPZzdAdj4bsl16trMHJwj9C8BBaPxKa5K4nayl8ATWSLdXfZuqajai9urlaXVLWl1S43gaFezUNHGc2viWGFVqfMapNI6ZqJrQkGIdrVlpSR2gMlMI5Rq69DmzV4hdMrcHAWfs9BAoTZU2Z769bOXxZc3VFkp4xWibBOFYwgTorrQA9CHSRO6XW+RWuU1ulzugk+IJJrd2XG6lfjva1JwnrQ15Fhg+vshoU78zxce0UticGgUkldh2f/wL0iv1vW3a8KS1TM8CWeMproOsp/4470mj6lkw1MdTho+p9Irw0VTODQiyMjyVlWFRVoYll3JHw5maAiAoawJL1qzs8owCRFg7UwQYHKuvA6APmyGAR8X+5eSiA+FGlKvISqHXVEyywqAtG9PLQDYOESOUrdi5bKecB7mT9W/92UnbzKds/CivQ1ggaPNaTYebto+Dm7It2LtszSNuSJ/mqPEUqaYzG67KzmDhcq440LVTrjHdCbAH3C3KLoZujDGxdgHfzSH/3ziKTf8HIG18azVlTW7R07J2d0c5mZEt3MkFd2eAu7W3sVJe7p0CX/6/fltthFVFKkqjtj7zaoWWRHyaxBAL0BcngJzxrUs1ANWoinudxTTyo7X3vEkF7WDJOkHMB/f2PmpRAYPiGEZh1PFXRQ6uOCwmCQHcLjO1QlaXT8roV1cmYLFRH/qIMoDdb6ZdyDqrc40JgDyupesAej3axsPANaHW0d+K3v6VKQO4dWcnBYyNfnCmBlndj15UYmvdLQVZXYXCAbDvSi53l78mgAvp6tvmI7ycB8vFRn4rC7Z0d8UzgaupqRsZLwDkzv5TIUDPRtu4pZzR/x9ttS/uo2IB5q++zRLVtCeAC/F3TemP0Fvzeym4EC8U3sW+Oa/B+37nEQDoFmu8ZrzdTlxV63fOfcsBAMDIT4LbAYC5ZvPd/8f+n1vebbmzALigAAAQwHHeYgTAdW6gdaFbBSKcajPz+Ekgi2VtdCuFUcG/XvOq0KvaX/LtBzg0FzbxQEo8IZXZxItGvw3ZH5eQQ0tmykBTWTCTZmJNLIkKSSU0YkCCXm33OCStrZMrQacrTnHJSMkVWjMprt2WUOdV1jUFdIKyYhLzf/dFofSrNUJPXZ0h23k0yS4yQ7itdzJmqjhwsrzqj+7MMqlnKY2qS+yyhGbcFLoA6XqJo95gFYoY6USEG+HNc6lmNUzcTbHsuFSqhFJgWYx5103ZxjzZymZTZ8QGj8RAxo2ShcMjb9pOU86KrQLkSLnRmOFGDjONFpx1CXp+s6dvOVx4h3IVL7nbxFUagep8f8S7NVocxKxEfnWDR6/hXkQ87T9Z9YNLZnCf9Dlmsfx8zbHCJMebeqYquSWXCc/YpjXvmnpUiazbSnKTQegpCAFh2s9hSjah52vufYbz9A+ryVFgrtCbZYzt0mfeGYLrgbJalzUNMqomgVWMVFks67y0EFM46+Y3I3DNNWVxTUwuiOvSaiYFqW2Ab7tDuU1RShGhKY6YnJTioazeKCeihEYwu6wmG9tUK49HpautZqJ1h+zsKPQcWAqIKVEnqsSSmJtqnhheK9M0WhgtmepO47uVyu7QWpqtDIeIjQmvctt4GOq3VGnMpi5Rs9OaD+OCoIJ9ijAlxEZ3q8K2cSvUZp3SmC0KHW3jbeojAD4qtIcFXFQPgB+g0B3g59viFAADqeUBeIDyWIQYj2NR/GIqMalKLI7FOYHJ8JDbG+VnZwxJhEixogQLFIRKghIFiqMigSxCBQ3lf2Jj4XzJMV2HhIZtGOJsxPx3x1+U6Iz5JTk2Ivg0hJqUYJ7IBqMJo7HA0wrlnUoclChnBYvwhxO5lcrUnXqV0epC08uiW50qEoH8CHRHjrfInPkG3P3JiRAlkIUK83VE+Guys6hlxhiJAQu2q5B9cEhhYPBIf8/JTwAA
      headers: [[content-type, font/woff2]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/font-roboto/d-6IYplOFocCacKzxwXSOJBw1xU1rKptJj_0jans920.woff2:
      data: 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
      headers: [[content-type, font/woff2]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/font-roboto/oMMgfZMQthOryQo9n22dcuvvDin1pK8aKteLpeZ5c0A.woff2:
      data: d09GMgABAAAAACn8AA4AAAAAUjgAACmjAAEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAGmQbmUAcg0oGYACGTBEMCu1810YLg14AATYCJAOHNgQgBYJ0ByAbL0RFB2LYOGCA7DduwB2DjQOQz3hJ9n+ZwI0hWB/YxYSYMBRSFENBsWqrAkWbJBHpUKamYBECLd7YUy2OVYvlKX1dMf05+T/7LtU4wMKUW3v6cz1MLF0FGyHJLEG01r/VMHufgpZJIoGKBHQMKsYkFpgU4tsDmtv9xsaIGoiNiIJRtJKSElLSIdGjQuiRIypl0JtBlKISJSqZZnH/Oq0ZWaYZyYoPCFpukb7+N+xuwk6iBRZMI8WdlLQH1F3R7l157zoCyXjA9AD36nWBrP07PZhwgBLSeMnLEQ4VqJsqlv8siGuabsED3LeKFnxrugO3hMah9NJJ/ipRkvNo0bWSH6xvN8TX2qedvrqte1s1/yDEW6dCUB4tkIxQEeoDTSYMiiaAuBEGJLK/dSEgoaIi0Ua4CGXj/7vX2aa3YewLW7L1Fw6EC+fgxB/7NWWpMKZaUyFMhaxEyArLmA7J/62Zdje5MjlgYyvcOTf7ZycwswkX9oD2CklKoSNIzN4VWFXVET5/gKgKaO9USZgKCaxcreurQxDS1KS6SqUxAJqz+B+OtbQOoul9mcsgjITAKKN32318dn4Bhdn5t38EFHq8pCEUmUMECIAIFw4RKRIiRgxEnDiIBEkQpSqgdDmAQKCAVcCaECAgAiDACvyk0ex3gJounDg3gOACJy4k2DnDicusvN3gBBrAILTCpecT3AAH9C0NYpDo+7rNUJuyf/BUQ/cM42KIV0v11FyNhkyiUDj0XDKjjuX1lEnr9K7FjMnEPpZVifqZhCWKr4SbqzTQwLYaKjSBrueqpWpPjRbN9WixjymuFFH/CwqFBhYDRux4XXDRJZfdwE+IMBGibhEnQYq021SoUqPpHm069BkyZsKMhXQZsmTrQkH1yGNPPNWtR68+/Z4b8cJLr0x66515C1atWbdh05Z9BxC3mkNDHAcJeFI4TeKEYI8bP9Y44ePEm77r41+ewEGxRUrtUWZUuIsOetH6bTJql3HbzKjlpPZkximHvICSPiqPeMwTngZTLZpmhlleMxe8ac9b3jHPAmuss8EmW8F2u3bYZS/Yb9vBNIRJvask5+7FELPwimHcaoZ46yTaJ9WCSTY0DsjmwGD82Y7DoWmOfu1MGuY6fi64i8JTF1OyuErRYMJBFtuDaaussAMsMLJnmS0vuQHreu+JDYwyxjgTDXPAYsWQxpkTT6ltykCFu+glcU3S4tKxymCyyA7etO0t75hnoWEGSu1TRoW76JHEG97yjnkWzo2mioUhQSbYYOmYZg2ss8EmWw1jsBYFTyY55CV9Yp999tlnP46GGajhpolWtE5hDEphgsHaYdbBBptsWWTntaOdq59hH5i0BI2pTbJ9tW44FNXel24XIjwwjRl9q3SNYW6MMqPqdvrNMVySVP2Trj0U3CvhlYipV1w9cpuCyIgMoYoJ6uAoNADzvug45Zj7yoBXfrjWup4FNpUP90w83Gjmo+puO/m8JAbcN3D4dUpCw6N96EFl/RgLIB76ETisxR/w6aPfR7D+NjfOfbeHOYYeHrwM6cPqaG0C4EDa+qeN8E9UN/umlSM27Cex9S8bW/9cmENaN4plychcK2z/rnfLbrbSThvtiLBzSrZHuO7Wmpvu9U1ldbZfCTb4hlrtIxcAHO5g65ppXrrnhMPC1kvnK5fWHKkuAKxYS2ygcV0rlvjEtGOxglUm6USA2od7bZ7ZA0zgcMNXu6Vq96DjVnaFuUnt8GvYm+2/bLjrmx9odH8OHtYJcBgN/b8kxzn07RsPkp7oNsv+PxY7vLXhbI0B2O6kbBr78YIaGQ9mNpD/ZDn01If3Bh8OOu7XP4HYqD7kLYGVOfcyTk47aKYl0X7mMjiMDjPg7M42er6j3/abnabFTbZeexnmm/4ZPQMO37lXsuKw1GmniQNWEawkgpUo+P+g8bqExlXC6NwiDU+RkuNUiKeoeU+7RxcXffIw9p5jytx58d4L0uW4KO/t67osuGHVFr1lzLkJFBG7OIqjOKlO+kwUPh53gWwuEnGJGNdJ7wYZ8JOTEBHC5HCLTMRIT5wnSPAYSbKSIiNpslAk1h3SUiqP+NULp2hOtdHQAdF0S1ofe7ErOh/EMyMdczKLF9olnjsqE3JyqeeOygFPyuNJXV4ExVMo3X9j06MfxnOvIEZNoDdJXkZT5mC98R7x1gKc1fs01mxBk9dJBzawOQ5sKDkp7MUhWOFQrjBRrvMoIaKIEOVmyUxYL3rQ4QjYwQIGsIZUnapA3GVBSAIaNA5ND0QYkJFRiSHMgAOcYxIJoiQSJek+IlUaTgrBBg6wgcNx6IRwArCCIg8o0IFB0e4sQGgD2sfstj0Yi/vtsrpEq52XqxUx7wXavU9R5AbVfbV1p6EDtGWnJA/XVOMRsvo8RlPdHad9NxBpoVtpYHLL0QrVHnveFe6uwPf/7jyBq5c52luBaALJx8pX+LgSQAvlHBSU/jQISHSdPDXyK0T8EDNBpjzkeaygcCLzJqgiKV9QtldXCkW9eOkpVFBLmWjZoyokvMYGYuM3q7LDGjTAj7sPXGsYBaVQYdkTHmY1/zyKJHrVqlSqUatOvQaNHgpOvOwmgbx4I2vWolWbdigIDqeB6phIgQHjoSEEYoO3GfWG+xmO9H2b14EtgH09CwceqakEMmDjtbHwH1ywyEvghjwO6F0GGbeagAYru65f6TdFf8NxEZpQA3bCjcYxgDMmoUrc/1DdhozZ9MEhMsFlbnKRv254PFwOt8PD8DLMhvlhdVifRh0CAEy4lSj1fMkew8Zt+fiWtyg/Gs6H67an546+N+CsD3294PDnwOcHM058+/z/7Vt/15TJ8MUb+x/6/XXxtmRzAC4jjHlHw5q/XvcP1+XZL9e/HyHSqDHjJkyKEm3KtBmzXpsTI/YX7U0Txdm2Y9eeffEOMAIAZPHeNcAFkH5MT6DhJhQar4QOSmnAxmukk4d0404Pg1C8IcIZ7jem0m8c6GKCWZ7xmjkieMMCoyyyRBTLbDDlbRLDFh94430klk98Z5Uf/ITIL/6x7f0n3jtUBxqZCCT4yEItBRI1aiOQrFEfAiSN+hJI8bmP1LFAqs99ri4F0n3ua3UrkKW5HwRyNfeLQJ7PPVUzgXzNPRco1NwrgSLNvRYo1jyHQ6BkqJuBg+pwqVfwfijyRYUwvQnHv6TWp0PyKigaNh96H4GBhmyOegPVB40f59xGEvqhc+sAD9WxCKjVXrZAVDdIs68AhHKCotyajCaMMM33GHhE96JFqxBPQEEEoqpC8iJOGGL05OS/t6P7dswwoetDgYYOgzlyg98MT+zFf0ajcoEikxZ3w24atFBdBAG0lP/g72LZuroeHO2D0lGazKLkzmMauyiDCdhr2a5txrOVjSChgd2ova0t77MhhbcusVxATqdNzYG/FtlUbXRQPRO4OeZ3jE2LIpfa374QSXm/jXcJ1RlLGxZ+dBY1t4Brn0S4YIb5A885pd4Nm4nTxHTsncPU/uJFsrvfCo55+xh0RpRwlNVFmRZ5JT0a8vqz7YKjXITu0NETuTeLgWx0wlG1ZKwnE8zEIEPVAl29KW8grf5KvdJSAaHPwFwMFyKZcNGCzUM5o39uRYwWpCG2syOpPpQYJcOaFn+YiakAyoJBWW1PdQ3gky5Lyrv1Qu08YS8gSdFMp6Xw8V//1cwtjV/OeuogqHw10Ysw1TOlp8shZnahTTWfy4ZZH3Bv01/rq0FPiwuORTUH3Rr39SnJaRmeqQkMhbRqn4TlQFNuBzh2XJgnuDqHspyu0HjUXsMzfPUKBo4oQFl/59ra8oQdbCz13uBAOQ+7IIF6czUJA1lU66Bw5NYfWV31oNN7ypngGeWsuTqsEsrhFQEqG1WQZkIcz5vH8rsZjFklUsoKdqUiEv68IaVRHWA1ltQdWoR4IBbvBaMNWXDw7xohT2WjEINN7BucggQF7qiNneY8xhDhaJpbjN+TprJdqRJ68eUpZ5QTOoqiy71R/rGSuGKW/zv5f6bK+V+XqNf9bxu3tMVwufyHCz+PSnKjrh9+fWXoqLsUjumye4LFDWIi7dtumjw+brjklOBI+ScdTjXMTJQ4+jRhyjVAOkeCkt30cx/S2Z8qaXHhEDOrpWN7sCJd6FjeIVEmfARcWahxPLsI1sVRRvQyvF1z0BOqq/mw7dXZ5YiDWzSTRl1gQmV52yFfjGnz20KQxu571VcjSo6/lSSUtDza2mZJtR3ssJefgx5bpVWyM8HDYK/wkP5JPnL6UqVarauNUmfRkTOho3FTlGoP6PIFDRtZeBq6BcMRkGuIiNY4MYmDV0ujHNQz+aSb3NheHTbPCEeU8Mg3wvEIObtW60VLwLt7tL9LwT6HZUmRTfWEN7twFXImIwjFDI64QVhZ3NhBFt8eBbxHMZ8CaIe7H1C8xvXMdyxLPb6VIsSjjQ5Tu6SCqgAr05exOiWSIOmZeqYl2yAKsJ4IfFNStZo5LrDOMVUi2QHIrmqZ8dR/Sd8CWKfN1Wr8Mskb5eGEuUnILSj6pq4YSmagxspkGt6pTX8GqCojM8mUjwKb6kdQiWqQsEttHrQP39bBMb/jp+WMbYfEKRvFj3/DjHhrj/gV5E+5wKkhbQbvSCZBg9oF+S042qt+KCNAGgjS24pSDB1vg0HdlcYmpwORmpccTd0pV+zgEtazSNPfQ0pIHnElz5rKRywGsg04LtCsfnVCrglVCXW2UwSoJrdok1FMXuGCi+3BkEsTB73xIylxRo9pfLYQYxLuUC1A6WAXzk0miNkeiw1OG6sUC1JVAcxx+9QSh5gpb4ljMlXlQ8qQQ3dgkoicYMp93+rZewvB1cgE6qFuCI/w2OaxiJkb+/Jx6REGfgqeugr5MzGoCXlAhbvvgpRRatUGSqa0aT40UTVEIniStK2KkQNSWnadTS44MC0ZcShqn/tNbjSpzhJPWqmVtdIahqMy4JIurfISwFqdDknlN/ZkJWaJOfPmu0hCIViLEiCXeDo1fx31S8zs39RXB2funyM/h/4qJvzk1FE1AlLSBoRckNzDUC2Vie0A3dVydpP2+nlQlXWLmsM6kpxf2qNAnzxXE9I8zbaSdNBbUmBp9qN4v7SdxKEPSQw7OPKYqmGDHU1G25UrXOzQT8UESQ8gg7ZI7KplJNU1l8Ei8eqUd9TGu/Hq7MDh3qFLtVYpmTYB1RVu8jGGn+HIKixbp7pl08klbTB60qEpikY9sjLo9EaglvU96Bz3EvfEcLRjtkf+1rba2RopIJY4orU7uObY3Y9SlGnOR8UZRdWYYRWrEpkS6FgkfdvN4lxqew4RQe5mP6W+wnEkhvXYbXxIZd2qOGV+DxkBE9TT6D3p7Zq2VzJUAseOmluQLiocOOkAKDVWaXR015dYYhhgYrCcD6ml8QGDF/GOWXf4pN3DFIOvh2olo92vyup7stXa0MvaZcUB90LXMpfiqT0HdGwvKlR7KYetkJMa2zJ/eArTc+pPLwuQVoRD7jxII8nWDhElbFkfTa6MUhEyZ1g5yvyhBBlM5pzxUDZLT/37WD/EMrpTwTbt4IhlG7ZotGO0+OI4aZ8Go5MwWJcz21FJpnoSIFY/ccFOPVCPGOEsSkbF/PQIj5jGOrVbVnfwRTqNq6nNasQEQSYH7oLXLiW31UgXpk8J44vqKc0GKe+BUadt3wUGAkJHB7zFtQeGQxccdSieReslSPWf21E+PxPTi3gAyczhLgUOemFaBT28Nb0szb2eFfH0QV74/FTUWTUSxGVUnmvtGSxEF+eUJztVfeYpr5KlNEbYbqQ/w8UBwJAH0HamgKFli1D0/cDgt1aKRsyIiBkajPOqYfpIH2OmzjUXl2DjQBcnYNPzP23nDgeJTLHZjvA3F6eZS+qL+hBHp2CFSjh9r5hEHI5LWf8yYeu+fiXuj2dEpxTy4BSscCAS7mcaFpIjOC3TN15a8XSiiv1d5nw1J3XvVVdSdCXBLyTQ7X6UP791na9WoVPtyKsOfHdGN4V1cnOIkpCaGROZGQLYjJzIqtrI6Io6olJUVVV4RG1VQhxWw8BWVUnfSk9FQ88uQ89O+1zrjciqGmx1dYKSoDTFEzCaH0DKetZ6qpr6NipKhta6Pz/vyl71JjjIgm2Mo3dThbeljyxJqar/UU1W/8NoQ8MMt3tOGvIR8uHF1YdJ9djPRUuCiuyQgv0MsjKrmh7gbl1DWxUY4ZuThDq3YelD8tGWvyGgLy1vY6imbeEGLDOpHO7hsJgJURqr/sUfdMIUeQNS3akvcKdANyLxpG6ttwWvR5SKf+OeDogqLHnnfDSKVObzyPWhfkSuc08aOBScttLk84hUfq/nmw8c5a+vGV3LuuaOM8ilM73oei1b0HiQOciMcqb1WLAxXFpjymrKekl9Wf4AqqVC9FkHJbMIdHx1e6qd9XNg5NrO3gaSJT+J0MR/vZGQBOTgYj7stSKQKXNVXJDY31kSRgv+kF24ebC3cBUl1sW4FzlLSg6fQRYu7hOXE+KiplBr8IY64U2sYRs6DULU6B833EN1i/QIcale8TkFiZTnqB9vXuxNN79muxhd9Q6O2QgmRCCZ0QQ4jJKJnpTGv+v9oztv+iwCmx2v+PFGND0YOBw/gIUOqldmXGhMUkLkg/zYNCQBblntjGSPaFO1s10v9n8avPBleHDEAF69UZnZrVe/kugkTRzewS6vfEa97f0yw/q4iTWYFOobntaUn1ZEjgpQEKYF4W/NhTyv0Yd/jpcJkcNPVxmmV74yjU6gZf4ppenXaNNMkCyKPbn0FNpK3wWEJHdmZqV0gZyeMm+7ch9v5jGOnDmQHhB2JdRa3fDkj3rk9+T/0vjk3+XHvmFPPPktb7jXunTtve7t2X9N52URmJf7Oyv7Z1pSZER6MpvtCdeYGG/3KOJiemfEJYDQapiYQomOZSg1yG6ZmOa28yKIOPdT7qhorWaMNs35ggvr6G+bbL/MCjzP6ah2FK4FBKZTczOTe4A20vPvgNDZoDTV5MABkCq/LlcqJyNXIgcCLfEZqKciVoFpdFmDesA2F/EViFPTiYjRMXd1dPV1vB/kbEJBSleHOp5MjjQ3lda6OCtr5fRIdt1PHpfS7wyfXh9tfyRcLaIbr62anLqFf8lc4Y96mu5478FlTWmhS/jNdM+EJUwrPupOtsbzm0OtfvTfGH6QXPh6JozbrrScJ8y+FHD8bb/cUsVwggG/0BH0/sMvm6XGSpkVD6YkrWxDSTe7c+a4a9+DZXxyDkByItiaGDTEM0iu4ZkhD7Yc//CjBmoBQ54/eN/fc9+rwOayDzexK2zg39L2rRpGqZi8+EBartu5PKFZIS7+CS6kU6AePnpik+v3e5Sop2jiixQhuoqm7aMozvXeY41V+Vld4yx/kg4Gj7aU1KWcbDU7WvOfnqsuEoFnZqNFInyV/QiTCN5nZPE7+yH+3MbPICCNKvWjWSgsBAw5vWhq2g8k+yJtInsAQ44pGvpsA+o93SqOOAdyC3tVlhAy8kBF0JFz/cg210yPCDdP7xAzQPhtibvbpfSGstK9AlbQCoPe0sXS0v49hEVCL/h99hL428m9tBO/9ybefFp41hNCTIjwS8uEGDhR0dy1v7/a8ZixbckIvvJrm+BY0FLoZabrhRNHyKcHKnr6S5R8k83m9Hp++unpHLzlzRK4Kve1aWvrXsVreu2rvX03aKXVzpKkaavJDXQ1abeQV46TqbL0dQ1N9DUkmZmnCHqM41HHOAfqm/OfSS4K9QUz6i84iIkjPTGOf2D+8nwat/5Dv8AorzwlLqI8PxllC1r42qlofn899IsIf8EvSmxaRteT6WtFppluQF5yPm0H5ff4GW+Z9FRNX92p9/Tqd9zSLDHteYSAniPjaOrOBVJNenBsCSkb+ECP+P3O8mhOA355PmvXK+xFQG9UdVZuclGwu1FWoJPdy0Drcvdm9umtLDQhcfhBb1RfTnlOadBspjeo+OCnAYPvBgx+qug+qFaSF+EVYN6xnP1wEkFWlp++LWp8WFre0Fj0DuL/n6Bpnjw7SK47206egNWRk90XG/Don2Vs6T8vNHH1M59b7+U8Qm7CD/ByunG6sXvzNnHWWfcvykeYXlt9MTG7sG530/7m2uuFlxMrYDsd9PbtGZvUMz1npmGdh1sFsoXnUwTWI9iKfiQBd/t2N2CJ3TkBQNsVAKhoMpkPe40M8E5YoCEH5vQGfmL1womKR09WGi7PPrg0Ww9iYdkZ2RlwuuJtMcK7BBcPd0eHRbeWvGxLqAXqTpuelRfB0ybaRol6j5A1YOD7aIqKLv+cyWvA+jXRIN+IR2kxVXfYwMsK5FBsP9eu9rxyT8EF1wEWKFLu8fHLl9BdNE9SX8V1atZO69a2Rae/AqkIm3Q/WqMmGlOf6gEbtwUaPjhT5PVtLjaj4+Mb2sYmWofkGMekvJGec49yn4M576+9691P3cKFKYCFOpyXW+zQ432hn13YruyxGLJW4SftnAZi9gSosc+zc3/mKQ38O58cPrVzBK2+gC5uGYi3jtUUlPUwiB3He414ZJexpXEx0ybG5sFWfT2p3v0h5UmdzDJd5raIsZWDNWGLtFUPFrz/di73Drkl4PxquSd/fvrZZ8lGXN/CzTetITYrO97M7UHP010ByufOzPHYpntF73Xz6mKzJ4BDpzMwa9AAFfGC8hK8HdzdjWMteNXRFMACBfQo4pRhrmuuLiHGQS7OwNEm3nnrpJRAxg9haRk5nUd4rqtuIqKZvwlQvnr5B2uUXmrtq2WzUtE/4/MG9JI3MrBXiNP4+SaMrN1rFFFvsPC1xfjPUEa8HQkh9rGB9Paj1MH9rQ65PE6KOLC3iWNig/KfOSOIf67GfMXB/xbjSAusAv+Zivx/LyekzbMK/KXg2qreJCZVzg+31MwnxlW/AdlVth8r13oHPYLpKIB5W4vz80h43r/Ki+1Cd+ZPEWt0MjDa6fXRBZNgznu4PjDY4xZG61IHmLcUGnc/6sA2/98uWmrBVHSlfireILkspnAaLnllZAl0yOQBRbwNFWycUi36BPjPnrnroX1MokJHw0srermzqWv9SQZVLczV2s7TQEneUZ5Dql5P298gdeshJWLYxKPCkkJjGu9CcAzVhWM+MtgR0kixrIiMLHdQljK7YnBIZnzxIGkQC4Jhzs6hE41tz2Cgh/Lo+7NpxUJ2ijgZPv5e6b/95JiCeNKxKPEnoFBy6JYEhb1NPIvVS5gVLZzFXiHexaRzmaX/ciZ8f/ghfs6K04Vn9lfGr4zXLjxWnLNwwMzSf2XopPCKLdM8jwJjFru5w92zthddj52/PgJiWQHZJkReneNqucKR4OdKFmhZfzz8ZTp1+vM40vFwuUWQlzAud2JkrcqwymhjFIV/JQvyTJmAfZoJ+N8jMli6+a02nbZ7Oy9OjMvxEsgCzcvQNflhOnX6y8jjtpWHAqDxha6IOhen/EC4Jl3l3GUNHSfzEqSk81WCWsLtppJfvWrmXhZQW18D0xx9jJlqUfd4qETYpvvTGjZhQgafkY7vUHZbzg6fnD83/fwUmXuVspTJ2T2FyWx/XN5I63GlfZcnZbwABbhCdX+BCMQCRztfWViCQw9/PXj2dGGquds5NMjdMTw3BIe+XQgKHz2KIifGisKjY8Nj4hPt80dG3cvSssJj0zIL/foHivwzsiOJWbmuZS8G7IqSSZGx4qigIMJUf8Bqf6DHZNCww4C/h2e/370LcLb1uFDWjNsYJkShUiFYQa5XTjpYulI6hJZG+RykMR2lHn1G/U4FpdSmJxRdBX6lWNsa9NC6x7ObirfiPV5RkIjWrAJipbelUZLDuFT7jW430zx3aruBLP+dWNsq+uHFbfTKK0pIdV5CcgnBWvGmZowMl2aNkoelWBhXndFR1ROaeFVbxTsimlGyJ++R1dl1bBUN410Aq2cyj4Fgrmu8HNb/FBQ/6+U2lBQVPp7scCM/ncwvqS/PTLMx1zUxd44L8vR3jjM3NdWx+x/ZUHweXynDVll8vqGu8Cy+qrSEvaLoHKF2hGqkOhSmMkQ1GhlsNVAdHxpUHW03AMeeZr27emF+4UH27lV+DS7CscGKGiJCJQz29eFx0bnJMZJWx7VMAq8YXKAme2iDobSQOUis+FFx0aW3wWgO0UEYcqC8klL2moKzDfUF3Ozpp1v3V+2kexy5NREhQQ9kAgQgV0ofw5o6ReFmsSCgmx4UKCEGsqIq6pK28SC4wVlCKgHc+R6HjHwnh7Q8R6e0TAfHzMz/R0RV7oiJq6iK3lTVcFFVgYvp+i7ZLkXVReHZ4YXVHBxf5HPkoS7PwOXKcRc7RYG/knfv3BUrJFSl43JD3RycnHQ0mlJIwJpxX1dVSZL/2s4dMVXBS4XWZYHBcfn+DjcVjWWPZVwkOdUGE6MqQ5yLrmlIyN9RlhDyzEcMgX0Ilhl4M+GIsHgAlN3P9x6WyuamilfjCKYxdelayjMUwLyFGJzuJu7f7Azd/00mwJAhbe73yGi98ZCQDKdbc06kWRV7kLBOmvpCN67+ssKdiYb3eYVGr2+FWXvjgbbCpuhRX17+o54q1WJqr/CT3soyjK2lp66OOcFOw8bSU+eehStiCXerwfFiqIVPZw+9NYerF8HVNtjDhIKULL/oHJh51tpY/NDFQ1Y9rUcyfuzXrqbD6OMBXsyWwi8tV29iyJ05reyzGtnKR/C9emz+z7jmwVBZRVNJwzdQWUN5mq+koaJpVdXq8iy6yqyPuGeZl0zafNLtoDNDavVsevD9iqSi/0QqwislY/PVKvGcPNHOrWoRVqaAFlPHzQgeHU/j2wGzsc9NTgyZ92j6Owff3tA+2s01vBgVMGTKY54jycVZ/kHFWbEAz1AMxkd3qW0DLcUhD0YUhVTNH5H0YtWKR5o77rVdMbmcxxJzRcTZw0xbWMXicZp+3D2j+FFdc3snw7e8oLHMgYltldFbMUuqftSQXbjauLCk5+wa6OCY4HqvE0/61PbIq0lbQFtao1HfcXILdrzmsE2NTzMz/O++vx1b6H/bI0u5UOXIwbo3kuCqo+5oKoIW9sqoY40/cSYqOg2EFI/8pM01K/L++5oUE/suzH8pJt19ezQ8vNzf80ZfYuKbdGXr7ohI4w7+W0XvRR7GPw5TtsFGMh1ljFc2TS19Wp1Y1R8WkrPSBXKThptdmxZiMZHcSEhC/WAVfhwwcr7SgX+X40luZpeU04PtXvblyhQF/Vsipjjo8ahkBNuMvMzoEu2zdNNvX1KOr35j19ah76bT5CCXUPWWAnKE0fTiaUsooh6RfbKHelRxKnikgFYzB6HsX2YCM9SJNMSz/ssqC65lv/Nj4Uuu0I9mQUnmpKcdU7ILs0tj7zoyKjbVtwuWfCX+/94Sk+OeeUvayNj1EzzGdv7teY9ia8sHYh77OauEECO/xb50l7aRsh7yiJmnvfG/C9dZ/Dohrnh2tKNkLj62dEZufNYms+xIHPOJEGIm8Nu+p8/NrM2ozWX8xuIBPD5hBkHeJqaBPg2BAV2BAap6N+TkBfjlZa3lFAQEZRVBaGPeJTDa9FKrpVTDqUtFAkFYZ/8Es3NeFmJnz1y5JX8NWDfZHZVMQ1FpxC7v+5bDQJfB6aBkGo4ipXRGgI3lECSecFA2DkWl+lJs7ayfQ+JVByVjIiqHQqlwsZ6cv50bQmqFs+U44MppDNzcTU08Xcx4DVxdzc3c3AzhNC+SmMQXyc4XlQTMXu2zkGVMRpKQZChztf9XC2d/bUXwp8PQfa0Tld52Z9mL/U77/zx1HsPQEtwS1xLSQoc5xfXFKYaLvdSFx62CPIJ59CAkLC4MpFsS5LPlYWTu2FLT9tHnTzzqJLnygwU8/RycnM0XSAv1Ko2WUQhjE8Su9mzSeT5JKQc+HL7Q0N3ZbHdZEjBzlLlngRRjHMHL0cV8kbT46iml1KdRxEuEMJrPJpQ6HKgfYi0ferUX0wUKZdmNSQn59VCcXJ7XmJCUXY8UFGB74qPRra19CVHoNsCNDMfvAs3LzhHOtYW1fhg7f0xp0M7SWQZwjxBIXYwyO7JVVStzSjUsmRgQXJwSE1AVgrF9mM76elklVQVpdTVVlTrguZFrzMYI54dKhxF8Y1L8vItyoxKqKrOFflz46uUOuVPolSssPMdp696WUtcIvnMXKp40pVaZpg1GBBC/j6+vaLz42KnYn7Uv2klOxfL31Tv42mf5hISqKoiDwupi0rdVxYRF7oqF4k2b8OHmOv/1LY/vTvvDtz33k7HzTYwLDSgrDo/Lr0Cl7/5RIvvag6xIq8BC80t3JHU0NCUl7urIy6qow/BDX2R/8eCH/sKLD/7dvqB0/Tw9bdbyF/8BbA7/B8ACfv5RXi4s/FISFuuq3TPy/xld+aEw/7/MKFgNXAXhBB65quZ3mSNchXEMzjWjNmffEVy77YdFatbn0cri6o3s6doSBL/VEMh8d5R8AGqv7OXj7Ca4f4GMTaYvg+n0kAfMWAUnGXGbq5WubLETFC6/WIYgF1AL8uFFVasHdK2u0/W6QTfiIbHA5gvygmuF4A/RnOkT3Way54spBkO81cmv3dOk0ApZXJgOiMn5By5dK/gTcmD6RLyZLGPEyQCumQpztelDxEyeZUacZg8r6CL+WCHfqB23tFDzijCT7c2n2ArJAFtxcrcVQzi4EqKJ6RNUM9nAiJOvVkoJxLqiODYDRAJrtvBsknxFNpJyCU2M1VykPk3yDYNVohi3tQoHgVZp6OZCZDnjOkFDskq4nijcegh9PuvE5JKvQ50g+jd1bdeqZ1sdixjTOjCv6diLvkJ1nJdYR4410sawNoJ+zUjOShINpFgzFZyEmXM6HtqB8mpjAeduDMj/r/PZkuf+D7hl/haureDS0u+otzR8X3bZVuu5csq/2umBcedopEW8gHvF349HyjOhk4pHKvnj+bNyaIF8NLGTbM6MmQDumb/1qAmimZ+PBplWJmXh6LqSU9bIxFQDfVc/Ooi7qETTFD9xKpmamfyj8OaZZEJ1OAXGegvRs0KesdMf8OOk8lvar+fGgFrQByzkYE7awz8Q118VeWAeDX/e4QBmRp+xoE2+m7GMlofvvD/oaG9B4fka2H8Se6HrFqJLTH+wgH5KKNfdqN0JIDb2+w0mUSZMl/ar/dZgRnxZN8SfnmUeAnHhOHQfyB3h9xrb8lTF+40BiaA+G/tMsBstPAFki2qMQUS/5tcNCQScSy8XOQVfxRiEALnJ2ldCUowtWCR/bx1fBoCnf+puD8A7of3bP+v8X7X5l3kBy6EAAiwuW1oFsPzgUpZe2gHEfVPfVyJbQ+Pfi3eaa0BJkhD7gF+HchvzHPsXu7Bjn49F5zN8RXCq42hEmtMpgjdPSik1+KfGC9Rhhjki5BGFOh7P2jY+iolml5qNctvK6etiatjdxQ1O45ZT/UftEOE3bqvtRDIu1wZeAkdbRF7T2U5EBo1LadSY/qpN8LQwvoPx7dm69oGa7qbVQwQGv8Rx1KnFcVKDBsR6FEuP0M8z0krZRXo2dGVIpgrQZACxNubaIRKbn2lHsmgpZjlM0jyXZTtIjE+SYmfuBjMu9EnBAg0J7SSxU5jouEQW2Q64r7UQXTsi1rKzExqu1A+X6jlOatkw28nEF02OfXiKPKIS2pJjVzEZbbh3ISyxJ723GZchu0mFSaVLr173282DSdX1rOok5z8RfYdVfiCXH0hnhKAmyqsd8skpPf7wplGBVT7cT4/7Y57eJEJrEuNezyksAufPrcCIunfeVFXdJ2HqpdFkoZ1KqYlMukBm/Ja/KOWwUGBVA2qzUUANCRoyyqbbUiKXJc7FH9nO5zSHfEcye4+oFvHDWEbe54KHMapyEvmBaIey6DQ17eZMJ5Ccps8yPphAoIU0opigjziaqITS2h0IHAZOJg0ztFwK+wTcwrzSCwDrvIzxyBPmHfMMp0JDJH6JwCfgTFgBHAL+5BPz43dD42VNaMz8iDuhIZIjIvBGKEtHgB+QcNKFyZXG6iChNoVd1bfMfWhI0wE4A1wvQzAyX4bCzEmN3qWmgWUYfPyXYZ1S1F1muqjkufMQgMCRPQfeuAniJ9At5KbEXeAZLuxWVeHGxnWjrMG7WLQ1mFfOaLe47BD4atnWdUN2Z13y3lDgbc+HC6sgr7+IwIsjc7PHX98u5CYJ6wVv6hZ2bQUWvb15EHej0/q5rrPikXjbcGDnOhsJmr8hO7LRvBu4V/sNalTIU6RBh6Jrts38i4GTzfts3QMAAAA=
      headers: [[content-type, font/woff2]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/icon_bundle.svg:
      data: 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
      headers: [[content-type, image/svg+xml; charset=utf-8]]
      ok: true
      status: 200
      status_text: ''
    https://localhost:6006/index.js?_file_hash=29a7d03a:
      data: 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