﻿using System;
using System.Data;
using System.Diagnostics;
using System.Globalization;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading;
using System.Threading.Tasks;
using ClickHouse.Ado.Impl.ATG.Insert;
using ClickHouse.Ado.Impl.Data;
using Buffer = System.Buffer;

namespace ClickHouse.Ado.Impl.ColumnTypes;

internal class Date32ColumnType : DateColumnType {
    private static readonly DateTime UnixTimeBase = new(1970, 1, 1, 0, 0, 0, DateTimeKind.Utc);
    private readonly string Date32ColumnTypeAsClickHouseType = "Date32";

    public Date32ColumnType() { }

    public Date32ColumnType(DateTime[] data) : base(data) { }

    public override int Rows => Data?.Length ?? 0;

    internal override async Task Read(ProtocolFormatter formatter, int rows, CancellationToken cToken) {
        var itemSize = Marshal.SizeOf<int>();
        var bytes = await formatter.ReadBytes(itemSize * rows, -1, cToken);
        var xdata = new int[rows];
        Buffer.BlockCopy(bytes, 0, xdata, 0, itemSize * rows);
        Data = xdata.Select(ParseValue).ToArray();
    }

    public override async Task Write(ProtocolFormatter formatter, int rows, CancellationToken cToken) {
        Debug.Assert(Rows == rows, "Row count mismatch!");
        foreach (var d in Data)
            await formatter.WriteBytes(BitConverter.GetBytes((int)(d - UnixTimeBase).TotalDays), cToken);
    }

    public override string AsClickHouseType(ClickHouseTypeUsageIntent usageIntent) => Date32ColumnTypeAsClickHouseType;

    public override void ValueFromConst(Parser.ValueType val) {
        if (val.TypeHint == Parser.ConstType.String)
            Data = new[] { DateTime.ParseExact(ProtocolFormatter.UnescapeStringValue(val.StringValue), new[] { "yyyy-MM-dd" }, null, DateTimeStyles.AssumeLocal) };
        else
            throw new InvalidCastException("Cannot convert numeric value to DateTime.");
    }

    public override void ValueFromParam(ClickHouseParameter parameter) {
        if (parameter.DbType == DbType.Date || parameter.DbType == DbType.DateTime || parameter.DbType == DbType.DateTime2 || parameter.DbType == DbType.DateTimeOffset)
            Data = new[] { (DateTime)Convert.ChangeType(parameter.Value, typeof(DateTime)) };
        else throw new InvalidCastException($"Cannot convert parameter with type {parameter.DbType} to DateTime.");
    }

    private DateTime ParseValue(int value) => UnixTimeBase.AddDays(value);
}