//------------------------------------------------------------------------------
//
//  Mars3D: A source port of the game "Mars - The Ultimate Fighter"
//
//  Copyright (C) 1997 by Engine Technology CO. LTD
//  Copyright (C) 1993-1996 by id Software, Inc.
//  Copyright (C) 2018 by Retro Fans of Mars3D
//  Copyright (C) 2004-2022 by Jim Valavanis
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//  02111-1307, USA.
//
// DESCRIPTION:
// Random number LUT.
//
//------------------------------------------------------------------------------
//  Site  : https://sourceforge.net/projects/mars3d/
//------------------------------------------------------------------------------

{$I Mars3D.inc}

unit m_rnd;

{$IFDEF DEBUGRANDOM}
{$DEFINE DEBUG}
{$ENDIF}

interface

//==============================================================================
// M_Random
//
// Returns a number from 0 to 255,
// from a lookup table.
//
//==============================================================================
function M_Random: integer;

//==============================================================================
// P_Random
//
// As M_Random, but used only by the play simulation.
//
//==============================================================================
function P_Random: integer;

//==============================================================================
// N_Random
//
// JVAL: As P_Random, but used only if no compatibility mode.
//
//==============================================================================
function N_Random: integer;

//==============================================================================
//
// I_Random
//
//==============================================================================
function I_Random: integer;

{$IFNDEF HEXEN}

//==============================================================================
//
// Sys_Random
//
//==============================================================================
function Sys_Random: integer;
{$ENDIF}

//==============================================================================
// C_Random
//
// JVAL: Using custom seed
//
//==============================================================================
function C_Random(var idx: integer): integer;

//==============================================================================
// M_ClearRandom
//
// Fix randoms for demos.
//
//==============================================================================
procedure M_ClearRandom;

//==============================================================================
//
// P_SaveRandom
//
//==============================================================================
procedure P_SaveRandom;

//==============================================================================
//
// P_RestoreRandom
//
//==============================================================================
procedure P_RestoreRandom;

//==============================================================================
// P_RandomFromSeed
//
// JVAL: Random number for seed
//
//==============================================================================
function P_RandomFromSeed(const seed: integer): integer;

var
  rndindex: integer = 0;
  prndindex: integer = 0;
  nrndindex: integer = 0; // JVAL new random index
{$IFNDEF HEXEN}
  sysrndindex: integer = 0;
  sysrndseed: integer = 0;
{$ENDIF}

implementation

uses
  {$IFDEF DEBUG}
  d_delphi,
  g_game,
  {$ENDIF}
  i_system,
  m_stack;

const
  rndtable: array[0..255] of byte = (
      0,   8, 109, 220, 222, 241, 149, 107,  75, 248, 254, 140,  16,  66,
     74,  21, 211,  47,  80, 242, 154,  27, 205, 128, 161,  89,  77,  36,
     95, 110,  85,  48, 212, 140, 211, 249,  22,  79, 200,  50,  28, 188,
     52, 140, 202, 120,  68, 145,  62,  70, 184, 190,  91, 197, 152, 224,
    149, 104,  25, 178, 252, 182, 202, 182, 141, 197,   4,  81, 181, 242,
    145,  42,  39, 227, 156, 198, 225, 193, 219,  93, 122, 175, 249,   0,
    175, 143,  70, 239,  46, 246, 163,  53, 163, 109, 168, 135,   2, 235,
     25,  92,  20, 145, 138,  77,  69, 166,  78, 176, 173, 212, 166, 113,
     94, 161,  41,  50, 239,  49, 111, 164,  70,  60,   2,  37, 171,  75,
    136, 156,  11,  56,  42, 146, 138, 229,  73, 146,  77,  61,  98, 196,
    135, 106,  63, 197, 195,  86,  96, 203, 113, 101, 170, 247, 181, 113,
     80, 250, 108,   7, 255, 237, 129, 226,  79, 107, 112, 166, 103, 241,
     24, 223, 239, 120, 198,  58,  60,  82, 128,   3, 184,  66, 143, 224,
    145, 224,  81, 206, 163,  45,  63,  90, 168, 114,  59,  33, 159,  95,
     28, 139, 123,  98, 125, 196,  15,  70, 194, 253,  54,  14, 109, 226,
     71,  17, 161,  93, 186,  87, 244, 138,  20,  52, 123, 251,  26,  36,
     17,  46,  52, 231, 232,  76,  31, 221,  84,  37, 216, 165, 212, 106,
    197, 242,  98,  43,  39, 175, 254, 145, 190,  84, 118, 222, 187, 136,
    120, 163, 236, 249
  );

{$IFNDEF HEXEN}
const
  SYSRNDSIZE = 10007;

  sysrndtable: array[0..SYSRNDSIZE - 1] of byte = (
    $48, $E2, $74, $70, $96, $82, $7D, $FA, $30, $DC, $CD, $B7, $CB, $A4, $A3,
    $4D, $80, $7E, $A7, $02, $3F, $9D, $2F, $EF, $6C, $50, $32, $BE, $4F, $8F,
    $31, $22, $23, $CA, $8C, $7E, $F2, $9C, $07, $54, $8B, $0C, $6F, $7F, $3A,
    $8C, $A3, $4E, $12, $EA, $7B, $FA, $F1, $C4, $4C, $26, $8F, $52, $01, $42,
    $3C, $89, $02, $13, $2C, $EE, $6F, $BA, $57, $95, $63, $FA, $6D, $2B, $3D,
    $B2, $EF, $6F, $04, $52, $25, $83, $93, $3F, $E7, $A4, $50, $B7, $78, $23,
    $BD, $7B, $45, $12, $99, $2A, $D1, $D3, $82, $06, $BA, $FF, $0A, $A4, $75,
    $34, $83, $6D, $E3, $CB, $28, $E1, $E0, $93, $85, $F5, $4A, $05, $28, $1D,
    $52, $9E, $0F, $BB, $F5, $46, $55, $18, $ED, $F1, $15, $1D, $90, $EB, $87,
    $68, $5A, $70, $C0, $7D, $1C, $90, $B0, $24, $0D, $6D, $36, $49, $37, $63,
    $6B, $D5, $07, $A4, $7B, $5F, $E8, $4A, $AB, $BD, $39, $04, $65, $B4, $0B,
    $CB, $6F, $6C, $6E, $48, $97, $55, $40, $D8, $89, $A8, $C2, $62, $90, $B5,
    $6D, $5A, $77, $9F, $B5, $B3, $CF, $47, $D5, $94, $4B, $6F, $BF, $14, $FF,
    $6F, $BC, $54, $A1, $F9, $80, $B1, $35, $C0, $96, $F6, $A6, $0A, $7B, $B4,
    $44, $85, $CE, $36, $31, $CF, $61, $B9, $1C, $A3, $19, $CE, $EB, $96, $EA,
    $1E, $33, $C4, $58, $EC, $85, $ED, $47, $16, $0D, $D7, $0F, $6E, $FF, $FE,
    $15, $9A, $C6, $DA, $63, $F9, $01, $4B, $F1, $1C, $B5, $FA, $B3, $D9, $44,
    $B0, $8C, $13, $17, $00, $B8, $AF, $04, $BA, $5B, $55, $D5, $0D, $3A, $F7,
    $8E, $4B, $D3, $40, $4C, $AB, $54, $58, $D3, $92, $31, $B3, $33, $27, $46,
    $C5, $12, $4E, $D7, $D9, $85, $40, $D7, $6D, $8C, $65, $3A, $C0, $88, $25,
    $4F, $80, $53, $CB, $5A, $C4, $D2, $C1, $F2, $BE, $17, $B1, $CF, $CB, $72,
    $58, $1A, $DA, $16, $14, $65, $48, $1B, $7A, $D7, $A9, $10, $8A, $45, $1C,
    $64, $38, $26, $28, $50, $6C, $B5, $3A, $17, $B8, $AE, $47, $C0, $4D, $60,
    $23, $94, $10, $1B, $92, $15, $BA, $31, $2F, $39, $73, $FD, $9C, $05, $54,
    $C1, $A5, $EC, $B2, $3D, $44, $8C, $33, $52, $A9, $A0, $02, $20, $C6, $47,
    $9E, $5A, $F0, $FC, $24, $D9, $9C, $D9, $0F, $9B, $4D, $06, $C8, $B1, $30,
    $65, $87, $D8, $08, $E6, $53, $B7, $5C, $13, $36, $04, $0C, $B3, $E5, $01,
    $BF, $A8, $70, $18, $DA, $53, $BF, $87, $DA, $2E, $F2, $69, $58, $6A, $E8,
    $80, $05, $AE, $78, $96, $68, $5D, $30, $58, $2E, $F3, $9F, $00, $51, $5C,
    $7B, $32, $6D, $62, $22, $DA, $FC, $85, $60, $FE, $64, $E6, $59, $C2, $F9,
    $14, $16, $1B, $6F, $E3, $10, $ED, $59, $9F, $B0, $8B, $AA, $3F, $AE, $E1,
    $5D, $6A, $AF, $B4, $BD, $7A, $A9, $C3, $E8, $C4, $D8, $B8, $82, $D5, $4D,
    $32, $CD, $74, $69, $56, $75, $88, $CF, $B1, $E7, $3F, $92, $5D, $F4, $0E,
    $B5, $A4, $06, $72, $44, $AE, $51, $77, $8C, $DA, $A0, $82, $4E, $D6, $B6,
    $D3, $E9, $AA, $65, $C0, $CD, $2B, $D4, $A9, $C5, $9E, $90, $13, $E4, $80,
    $65, $7C, $1C, $AD, $16, $C9, $CF, $6C, $7E, $66, $E7, $0B, $9E, $7A, $3D,
    $1C, $B0, $C0, $60, $7E, $46, $F2, $01, $B5, $F9, $D2, $98, $4E, $45, $35,
    $0B, $49, $89, $A7, $5C, $C2, $4F, $8C, $B9, $5B, $4D, $D7, $69, $24, $CC,
    $A7, $73, $F0, $57, $7A, $F6, $65, $EA, $C7, $98, $36, $0A, $17, $83, $45,
    $6E, $04, $BF, $B3, $22, $B2, $05, $0C, $61, $9B, $40, $0F, $7D, $DD, $58,
    $D9, $8C, $49, $11, $5F, $C6, $E7, $03, $7D, $BD, $C0, $FB, $12, $DE, $0D,
    $53, $E6, $E3, $23, $0E, $5B, $23, $C4, $BA, $19, $F5, $98, $74, $3D, $80,
    $11, $8A, $C1, $B2, $E6, $8F, $19, $4C, $F0, $F5, $46, $21, $23, $63, $D2,
    $00, $CD, $6A, $7C, $F3, $78, $56, $83, $B6, $08, $32, $DD, $A6, $EC, $3C,
    $C2, $7A, $8E, $6B, $57, $85, $46, $B7, $71, $6B, $83, $4D, $A3, $6A, $FA,
    $E1, $23, $B0, $F7, $BE, $32, $60, $2E, $4D, $11, $37, $AD, $37, $AE, $03,
    $C1, $51, $9B, $45, $F4, $FC, $FB, $C1, $74, $A7, $5E, $DA, $8F, $9F, $DC,
    $E3, $12, $58, $D5, $E2, $AB, $D8, $1C, $60, $B2, $FB, $8D, $36, $E6, $A8,
    $44, $8E, $08, $40, $EC, $6F, $68, $CF, $EF, $98, $C3, $D2, $7D, $83, $0B,
    $94, $A0, $C7, $FC, $7F, $DD, $3D, $90, $D7, $2D, $9D, $B7, $BB, $27, $79,
    $EA, $78, $C0, $BF, $73, $6A, $FA, $39, $91, $0E, $5B, $AB, $C5, $FD, $0E,
    $85, $EE, $E9, $FC, $47, $0E, $E2, $55, $1C, $D0, $6C, $3F, $36, $35, $40,
    $B0, $F7, $F4, $AE, $A2, $01, $7A, $05, $32, $99, $9C, $73, $4B, $15, $54,
    $43, $F4, $32, $37, $6F, $66, $8A, $61, $DF, $78, $90, $19, $DE, $10, $BC,
    $65, $8C, $9C, $DE, $43, $65, $D2, $B3, $59, $1A, $20, $39, $37, $4F, $AD,
    $61, $8C, $13, $B6, $12, $E5, $0B, $B6, $A8, $D3, $AB, $A7, $5B, $9D, $05,
    $05, $51, $62, $FD, $38, $EE, $B4, $60, $F8, $5E, $DA, $23, $A7, $1E, $3C,
    $92, $57, $01, $2B, $E4, $39, $81, $75, $5B, $06, $76, $3E, $2D, $FF, $14,
    $40, $35, $C5, $F8, $07, $8D, $1A, $CC, $98, $28, $0F, $86, $79, $76, $98,
    $86, $54, $8F, $CE, $2F, $81, $DF, $FE, $B5, $27, $AF, $05, $4C, $67, $F6,
    $11, $50, $54, $1C, $19, $31, $AE, $9A, $BF, $25, $A5, $A8, $5F, $FB, $13,
    $62, $76, $8E, $F8, $13, $BE, $AD, $24, $86, $8B, $7B, $5B, $9E, $5E, $9E,
    $62, $8D, $B0, $C2, $B7, $5F, $67, $37, $F3, $CD, $7F, $C7, $09, $6D, $D1,
    $FE, $E9, $C3, $0E, $75, $34, $D5, $44, $F8, $13, $2C, $99, $7D, $55, $6C,
    $C3, $6B, $C9, $B4, $43, $94, $E0, $9F, $1C, $08, $98, $69, $D0, $12, $F4,
    $A6, $B1, $E9, $19, $11, $1E, $DA, $87, $32, $8D, $CD, $66, $26, $38, $F0,
    $7E, $02, $E9, $E7, $5A, $DD, $EA, $00, $7E, $F5, $5D, $4E, $0F, $CC, $8B,
    $A8, $16, $77, $0D, $E8, $E6, $7B, $EC, $F1, $55, $F9, $09, $40, $D8, $FF,
    $7E, $1D, $A2, $3C, $BB, $B7, $5E, $72, $8D, $B6, $29, $67, $66, $05, $DE,
    $2D, $81, $F7, $D7, $F6, $BA, $FD, $B6, $E2, $3E, $81, $E5, $6E, $6D, $B6,
    $C9, $76, $84, $60, $F1, $EB, $F0, $EF, $A6, $B8, $6A, $60, $B0, $76, $2B,
    $D1, $3A, $DF, $E6, $B3, $21, $2F, $4F, $FE, $16, $DC, $62, $E7, $61, $4B,
    $B2, $EA, $3D, $7D, $67, $2C, $FF, $D6, $C3, $8B, $90, $60, $50, $E6, $EE,
    $52, $07, $58, $B8, $AD, $17, $97, $FA, $21, $27, $94, $B2, $B2, $27, $13,
    $D9, $B9, $13, $EA, $DC, $3C, $47, $03, $C5, $B4, $30, $61, $9C, $CC, $91,
    $C0, $A5, $A3, $96, $93, $92, $99, $78, $71, $CC, $BF, $62, $3B, $5E, $62,
    $E1, $87, $41, $5B, $12, $19, $F0, $0B, $4A, $80, $9F, $33, $FA, $17, $6E,
    $14, $02, $FD, $7B, $83, $BD, $CC, $A6, $D8, $6F, $24, $65, $F0, $E3, $47,
    $11, $9E, $67, $5D, $6F, $44, $73, $2D, $6B, $7B, $98, $50, $4C, $F4, $A4,
    $29, $D0, $05, $FD, $3E, $F8, $7F, $A1, $A5, $75, $D1, $68, $A7, $F8, $8F,
    $52, $DB, $E6, $CA, $17, $3F, $1F, $FB, $3D, $30, $E0, $4D, $5F, $32, $F2,
    $8D, $7A, $E6, $AF, $1C, $61, $B9, $8B, $99, $AD, $12, $24, $22, $D1, $06,
    $1A, $6B, $1A, $F1, $1A, $02, $CF, $6E, $F0, $7E, $BE, $81, $85, $79, $D0,
    $FC, $18, $85, $88, $71, $1A, $8F, $8F, $39, $83, $48, $90, $5B, $7F, $9B,
    $1E, $5E, $C7, $47, $6A, $57, $12, $12, $F5, $BE, $65, $3C, $87, $D2, $38,
    $28, $1F, $96, $AA, $5B, $A0, $C1, $47, $44, $6E, $1B, $17, $76, $3B, $1D,
    $F2, $41, $C5, $E2, $5B, $DB, $37, $AD, $52, $7C, $6F, $97, $E6, $19, $6B,
    $6D, $83, $E1, $68, $C1, $40, $45, $BA, $58, $08, $B5, $6D, $99, $68, $60,
    $59, $3A, $0B, $20, $0D, $28, $13, $A0, $AB, $C9, $E7, $D6, $80, $E7, $96,
    $64, $1A, $05, $2C, $A8, $D1, $03, $90, $8F, $04, $3C, $CE, $6B, $83, $8C,
    $E3, $63, $E6, $E8, $68, $3E, $C4, $77, $BB, $A4, $15, $56, $85, $10, $2D,
    $70, $A6, $0C, $10, $21, $05, $80, $1E, $7C, $BB, $15, $53, $7A, $8D, $0C,
    $E6, $48, $90, $4B, $42, $DA, $9B, $D1, $37, $6A, $88, $75, $AA, $F3, $BD,
    $E8, $B4, $8E, $98, $A5, $8F, $B1, $80, $18, $50, $79, $F3, $B3, $81, $18,
    $96, $68, $CF, $07, $CB, $46, $FF, $8E, $B2, $0D, $C2, $67, $C9, $FE, $15,
    $07, $A8, $62, $D4, $A5, $96, $74, $F5, $C0, $5E, $59, $08, $23, $88, $53,
    $29, $78, $BA, $85, $FE, $FA, $80, $58, $A3, $8E, $D8, $67, $51, $5F, $80,
    $3D, $5E, $82, $29, $C1, $9A, $79, $88, $27, $22, $11, $46, $FE, $C7, $1C,
    $E7, $FC, $4A, $51, $78, $3D, $63, $CF, $26, $3A, $3D, $19, $5F, $65, $1B,
    $9D, $B9, $82, $EC, $1A, $10, $48, $F8, $68, $A5, $D5, $F9, $74, $2C, $01,
    $46, $73, $18, $9E, $E1, $A7, $85, $29, $8B, $33, $9B, $8A, $3C, $D1, $4E,
    $D3, $8F, $7A, $C6, $30, $0F, $3C, $E0, $63, $73, $E7, $8F, $90, $2C, $39,
    $0E, $2D, $DC, $B1, $28, $56, $9D, $4D, $0A, $E5, $23, $B7, $65, $F9, $27,
    $B5, $D3, $E3, $A5, $E9, $BC, $9E, $DA, $2A, $3F, $36, $13, $FD, $CF, $DB,
    $AE, $07, $8B, $80, $E1, $74, $86, $35, $34, $B0, $7E, $AB, $B5, $EE, $BD,
    $85, $55, $54, $1F, $6D, $F8, $D3, $7F, $18, $24, $F9, $F8, $95, $64, $46,
    $CF, $AD, $B7, $D8, $E5, $2C, $B2, $6B, $07, $41, $1F, $0D, $7F, $6D, $50,
    $FD, $C1, $A1, $61, $93, $0A, $7E, $27, $32, $54, $DB, $88, $92, $FA, $09,
    $75, $F2, $DE, $74, $C5, $FB, $06, $99, $B0, $5D, $E2, $72, $9D, $79, $CE,
    $CC, $84, $9F, $71, $33, $29, $AB, $0C, $56, $D6, $15, $E5, $1D, $68, $B3,
    $57, $D1, $DA, $C5, $31, $10, $98, $13, $78, $61, $40, $22, $B5, $B5, $AE,
    $DB, $4F, $21, $69, $20, $B3, $95, $D0, $27, $43, $6A, $AA, $9E, $42, $02,
    $E4, $EC, $46, $DE, $AC, $16, $DD, $D3, $34, $14, $6C, $52, $CC, $A6, $80,
    $EB, $A6, $EB, $65, $41, $2F, $8F, $2D, $E0, $F6, $74, $96, $24, $FB, $1A,
    $75, $95, $88, $1B, $19, $14, $6A, $12, $89, $78, $15, $68, $2E, $54, $CB,
    $F0, $8C, $05, $F0, $7C, $E2, $39, $0C, $DB, $CD, $9D, $36, $EA, $4F, $61,
    $EB, $A5, $CE, $AC, $83, $61, $EC, $49, $C0, $A0, $58, $1C, $43, $26, $E8,
    $24, $CA, $3E, $A6, $1D, $8B, $5C, $A0, $0B, $15, $83, $9D, $3A, $84, $20,
    $19, $7C, $F0, $45, $10, $BC, $69, $6F, $92, $24, $EB, $41, $1B, $65, $0F,
    $E4, $26, $FF, $71, $F8, $88, $70, $05, $3C, $75, $C4, $E5, $63, $47, $E8,
    $DB, $A7, $E6, $A4, $22, $D6, $86, $D2, $DF, $88, $E2, $4B, $B5, $70, $BF,
    $91, $06, $3B, $3F, $80, $24, $F0, $AD, $4F, $54, $C1, $FA, $43, $37, $0A,
    $F8, $6D, $95, $B9, $FA, $62, $20, $DF, $43, $FC, $65, $88, $FC, $76, $8F,
    $48, $B3, $6E, $97, $37, $D0, $95, $EA, $2D, $97, $1F, $D0, $84, $8D, $3B,
    $EC, $BC, $64, $23, $B7, $B4, $CB, $89, $DA, $66, $B6, $74, $AC, $43, $FA,
    $3A, $B4, $69, $A5, $5A, $75, $F8, $4E, $3D, $32, $04, $5A, $D6, $6B, $7F,
    $EC, $19, $EE, $0B, $78, $97, $64, $51, $94, $BB, $E9, $70, $61, $50, $05,
    $D4, $4A, $B5, $6B, $61, $13, $58, $BA, $82, $71, $EA, $CB, $58, $35, $12,
    $4F, $34, $09, $70, $BF, $F8, $A5, $A0, $1F, $48, $C8, $1D, $16, $7A, $3F,
    $2C, $42, $A6, $5D, $44, $DF, $61, $B4, $92, $21, $DB, $06, $95, $D2, $88,
    $A3, $ED, $26, $88, $B8, $D3, $C4, $6E, $47, $EB, $7B, $1F, $CA, $4D, $2F,
    $E3, $B1, $FE, $DA, $F1, $42, $3A, $CC, $FA, $93, $4E, $6D, $8D, $9F, $D2,
    $39, $4F, $9F, $89, $BB, $68, $43, $43, $B0, $9B, $6C, $76, $AA, $E2, $67,
    $73, $CF, $9D, $F8, $66, $8F, $8F, $98, $91, $71, $4D, $B1, $67, $4B, $D4,
    $D2, $75, $90, $62, $10, $4F, $7C, $D6, $1D, $D4, $46, $28, $C3, $99, $BB,
    $BA, $C7, $3F, $CB, $A4, $6E, $0A, $F0, $2C, $23, $F7, $91, $0C, $F0, $2D,
    $AC, $43, $DB, $8B, $6E, $8A, $43, $9D, $9A, $AB, $DA, $4F, $CC, $53, $E4,
    $1C, $01, $22, $BA, $97, $F8, $0C, $D2, $7E, $51, $4A, $C8, $CD, $76, $7F,
    $18, $65, $C3, $C1, $9A, $52, $6A, $A3, $E8, $C3, $0A, $90, $32, $2E, $0B,
    $EE, $46, $3E, $10, $EC, $AF, $91, $32, $FB, $46, $50, $C8, $3F, $93, $29,
    $EB, $84, $D4, $46, $0A, $87, $94, $3C, $21, $A0, $13, $08, $87, $06, $03,
    $A0, $52, $5F, $47, $B2, $C5, $3B, $BE, $7C, $57, $AE, $4F, $FE, $31, $33,
    $B2, $D4, $40, $88, $B3, $7D, $3C, $26, $EF, $EF, $35, $83, $3D, $A3, $8D,
    $61, $4D, $8B, $72, $A0, $28, $9C, $3D, $89, $A0, $D7, $4C, $71, $D9, $FA,
    $2C, $25, $78, $4C, $49, $91, $28, $2A, $5B, $7D, $77, $48, $4B, $67, $42,
    $C2, $5E, $C6, $35, $85, $C2, $3F, $0F, $98, $DC, $A0, $B2, $58, $F5, $A9,
    $30, $AC, $1D, $E5, $5B, $C2, $5E, $36, $8B, $E1, $AB, $C5, $9C, $53, $0C,
    $79, $62, $87, $00, $32, $03, $D9, $1B, $5D, $05, $B4, $47, $A2, $F7, $E6,
    $40, $CC, $09, $70, $2A, $19, $F7, $1D, $5B, $A3, $16, $AA, $B3, $87, $71,
    $54, $24, $F8, $40, $05, $3B, $74, $24, $89, $DB, $B0, $F6, $71, $68, $04,
    $14, $9F, $19, $64, $78, $93, $3F, $3A, $0A, $F5, $A0, $97, $E9, $01, $42,
    $A1, $B0, $C1, $44, $3A, $E7, $47, $9E, $30, $2A, $46, $E5, $83, $B4, $C6,
    $70, $8C, $52, $D3, $76, $12, $F8, $5A, $0C, $96, $77, $99, $11, $67, $08,
    $D6, $66, $47, $B2, $45, $22, $A8, $C0, $49, $7A, $DC, $C8, $78, $9F, $B2,
    $F5, $1C, $CF, $58, $A0, $07, $4C, $87, $68, $FB, $05, $4E, $45, $09, $15,
    $FB, $59, $8D, $76, $FD, $78, $BE, $76, $BE, $9E, $FC, $E4, $C4, $D7, $3C,
    $F1, $B1, $51, $52, $3C, $D8, $8F, $F3, $96, $8B, $09, $A2, $5F, $56, $9D,
    $FC, $99, $CE, $A7, $FF, $08, $A0, $A6, $40, $FE, $0D, $B4, $46, $B7, $08,
    $55, $61, $4B, $84, $46, $BB, $B2, $D1, $55, $84, $54, $F7, $81, $6E, $18,
    $79, $D0, $DC, $0D, $F0, $A3, $77, $FA, $5C, $82, $10, $7A, $3B, $9A, $FF,
    $8E, $9A, $61, $6A, $9F, $C6, $34, $1B, $70, $5A, $CC, $DA, $88, $BC, $DC,
    $70, $EE, $55, $4F, $C5, $65, $04, $DE, $B5, $6B, $B5, $FA, $A9, $B0, $00,
    $4C, $F6, $CC, $42, $41, $8F, $61, $A4, $62, $AD, $F7, $E0, $71, $4A, $39,
    $DC, $86, $B2, $66, $59, $B2, $89, $19, $44, $D1, $EF, $C2, $80, $9D, $BA,
    $68, $29, $2B, $EA, $96, $9C, $5E, $BC, $8D, $E0, $F9, $1F, $0F, $23, $66,
    $5D, $73, $6E, $53, $2E, $C5, $CC, $B3, $AE, $24, $7A, $D6, $CB, $A5, $E9,
    $9E, $26, $6F, $96, $FC, $90, $D6, $17, $43, $90, $9F, $D0, $05, $94, $C1,
    $38, $71, $CF, $44, $9C, $79, $D2, $19, $C7, $31, $6E, $07, $07, $2D, $66,
    $02, $6D, $D5, $02, $97, $3D, $E5, $93, $3A, $F5, $A2, $5F, $9B, $5F, $78,
    $A1, $5C, $88, $FB, $90, $BF, $A6, $91, $83, $30, $37, $2E, $B5, $C5, $DE,
    $F9, $50, $08, $DE, $BE, $9B, $B3, $27, $73, $8F, $32, $4F, $4F, $30, $06,
    $09, $26, $51, $73, $CD, $13, $C7, $FB, $DE, $9A, $16, $9E, $DE, $D1, $27,
    $E7, $88, $41, $8D, $2D, $64, $79, $6D, $9E, $6F, $98, $15, $F4, $10, $43,
    $BF, $EA, $E0, $CB, $21, $DC, $19, $6A, $CA, $14, $AC, $F6, $58, $BB, $50,
    $66, $17, $79, $1F, $F1, $95, $29, $78, $C9, $2C, $34, $48, $A6, $AB, $79,
    $F7, $9C, $7B, $D6, $8B, $01, $04, $B2, $D1, $55, $26, $28, $89, $6C, $07,
    $A1, $F8, $5A, $91, $0A, $54, $D7, $54, $F8, $E9, $16, $BF, $57, $F6, $56,
    $C8, $80, $97, $99, $89, $DA, $44, $4B, $60, $53, $AC, $CF, $9D, $5A, $06,
    $25, $D7, $30, $13, $41, $50, $AF, $6B, $A7, $2A, $DB, $D9, $6C, $3F, $EA,
    $D2, $C4, $DC, $89, $58, $9F, $BF, $8D, $D7, $0F, $0E, $61, $B6, $74, $23,
    $D1, $C4, $B1, $6E, $0C, $B2, $D1, $C8, $FF, $3C, $F8, $EC, $56, $BE, $7A,
    $FC, $D7, $48, $71, $45, $DA, $76, $37, $AA, $09, $A8, $94, $5D, $C3, $51,
    $88, $E0, $B8, $7A, $00, $68, $9D, $B9, $E5, $1E, $16, $32, $45, $E2, $32,
    $64, $94, $78, $0C, $D9, $40, $5B, $7C, $FF, $0E, $89, $EC, $AB, $09, $10,
    $A4, $30, $EF, $31, $3E, $F4, $69, $53, $0D, $E4, $7B, $2B, $95, $4D, $ED,
    $C7, $7B, $E7, $B2, $6A, $78, $1B, $7A, $65, $3E, $D6, $F5, $4F, $58, $6B,
    $7E, $5B, $71, $5C, $33, $A7, $96, $11, $A8, $B3, $4C, $BF, $B7, $7C, $A0,
    $21, $35, $CB, $82, $BB, $70, $BD, $12, $B7, $CC, $A1, $B0, $B4, $DF, $A5,
    $F2, $05, $E9, $EA, $DE, $32, $83, $05, $F7, $C2, $D8, $FF, $0E, $AB, $DE,
    $54, $DD, $D6, $42, $94, $DA, $A5, $0E, $8E, $D1, $68, $45, $40, $27, $24,
    $E7, $E5, $E6, $2C, $23, $DA, $E1, $60, $AF, $46, $9C, $0F, $7E, $A2, $CA,
    $DD, $5D, $A9, $7A, $28, $20, $38, $5E, $C1, $6E, $31, $B4, $00, $59, $67,
    $BE, $84, $48, $1F, $E0, $25, $E9, $BE, $B4, $ED, $2D, $95, $C4, $13, $7F,
    $FB, $BB, $60, $C1, $22, $26, $67, $BF, $44, $28, $7C, $19, $2B, $BA, $64,
    $E1, $2F, $52, $08, $63, $17, $13, $CA, $DD, $9B, $D6, $2F, $18, $52, $B2,
    $FD, $25, $5E, $E2, $10, $BD, $1F, $16, $76, $F1, $B9, $9A, $9A, $3B, $39,
    $E9, $8C, $19, $A5, $84, $B8, $BE, $45, $35, $89, $84, $EB, $09, $D4, $2C,
    $13, $AC, $0F, $1F, $10, $01, $9F, $28, $B1, $00, $CD, $6E, $09, $58, $2F,
    $A2, $48, $4A, $86, $A1, $CA, $BA, $6F, $00, $D4, $80, $64, $46, $9E, $53,
    $DD, $CC, $A1, $FF, $A1, $7F, $4E, $A0, $92, $FD, $9D, $22, $79, $D1, $1E,
    $4A, $A7, $C4, $7B, $EA, $72, $7E, $94, $1D, $DC, $F7, $A4, $B1, $F0, $71,
    $14, $00, $9C, $FE, $F7, $F8, $1C, $BB, $46, $0A, $42, $5A, $5D, $3B, $18,
    $99, $53, $9E, $BF, $B7, $28, $D9, $85, $17, $EB, $9B, $83, $93, $47, $0B,
    $58, $37, $24, $9E, $BB, $6A, $46, $EB, $01, $AD, $1C, $4F, $DD, $82, $1E,
    $8F, $E0, $2A, $78, $C2, $8B, $38, $DB, $2D, $08, $3A, $AD, $E5, $BB, $E9,
    $07, $53, $59, $AD, $18, $FF, $35, $5F, $F3, $2F, $E2, $93, $E6, $43, $64,
    $DC, $C3, $E7, $97, $CA, $D2, $6D, $71, $A6, $51, $FF, $58, $15, $AB, $70,
    $AF, $31, $DE, $D4, $45, $CB, $AE, $39, $5C, $CF, $97, $1E, $65, $10, $B0,
    $7D, $AD, $F2, $E1, $8D, $99, $A5, $19, $96, $2A, $05, $7D, $6C, $E5, $3E,
    $A9, $77, $5F, $C9, $7D, $5B, $02, $6F, $18, $A8, $D4, $4E, $FE, $18, $45,
    $2A, $79, $58, $60, $74, $C0, $52, $D7, $CA, $78, $79, $C6, $FC, $86, $94,
    $35, $AC, $62, $F6, $9D, $31, $45, $3B, $0A, $5A, $FD, $E9, $94, $18, $28,
    $80, $53, $4C, $34, $F3, $FC, $33, $60, $8A, $D4, $2D, $90, $98, $A6, $D7,
    $C7, $9D, $E0, $BC, $81, $AF, $12, $0E, $51, $4A, $6A, $32, $75, $DF, $EA,
    $09, $4B, $BC, $6F, $42, $C8, $33, $D1, $27, $72, $0E, $A8, $A4, $75, $B7,
    $82, $87, $CE, $AF, $93, $2D, $A8, $27, $39, $28, $6A, $E1, $C5, $7F, $11,
    $B0, $75, $A3, $CF, $38, $91, $40, $10, $BF, $05, $6B, $5A, $38, $FF, $EB,
    $AF, $DE, $62, $9E, $5E, $F8, $0F, $25, $6C, $A9, $60, $6C, $10, $DE, $50,
    $AF, $D0, $C8, $2A, $0D, $50, $62, $C5, $DC, $DB, $01, $91, $3C, $33, $E9,
    $19, $A1, $3A, $59, $A6, $54, $6C, $A9, $29, $2C, $57, $41, $AC, $EC, $77,
    $A4, $D5, $13, $2B, $D3, $52, $0F, $9F, $AE, $28, $10, $EC, $2B, $43, $FF,
    $CB, $85, $CA, $49, $87, $BA, $00, $86, $25, $F6, $CA, $2D, $E5, $C0, $14,
    $DB, $88, $18, $B3, $E1, $0A, $C6, $88, $18, $C0, $93, $ED, $3A, $FB, $2E,
    $BB, $39, $A2, $F3, $78, $4A, $78, $D4, $77, $B2, $E6, $18, $EE, $79, $87,
    $BD, $62, $93, $DC, $A4, $DA, $31, $CC, $D6, $BE, $39, $ED, $8A, $5C, $1D,
    $B2, $C3, $70, $95, $EF, $A1, $21, $F0, $67, $C1, $64, $FD, $42, $4D, $A7,
    $C4, $A3, $91, $37, $8F, $C4, $37, $D1, $01, $43, $7E, $36, $28, $92, $14,
    $D0, $DE, $B1, $66, $AA, $E2, $9B, $7D, $55, $6A, $20, $DC, $76, $2E, $00,
    $D5, $19, $26, $92, $5A, $06, $72, $43, $42, $B1, $32, $B5, $7C, $8A, $3E,
    $50, $B4, $B6, $08, $8A, $54, $87, $13, $44, $FC, $36, $CB, $E1, $C6, $AF,
    $B1, $1A, $F6, $7D, $1B, $A0, $F8, $16, $FF, $12, $BB, $29, $89, $DC, $D3,
    $A7, $7A, $FA, $F0, $93, $8A, $C6, $93, $F8, $3F, $3B, $16, $7D, $0A, $61,
    $63, $67, $96, $BE, $F8, $1F, $85, $0F, $11, $82, $1F, $9B, $1D, $7A, $AF,
    $E6, $C9, $CB, $7D, $9A, $53, $FC, $B4, $DE, $E9, $C3, $9F, $86, $47, $75,
    $03, $C6, $DD, $46, $91, $51, $28, $62, $A6, $14, $0F, $3A, $DF, $FE, $64,
    $7B, $32, $75, $E9, $8D, $8D, $6C, $6D, $9D, $3E, $42, $E2, $81, $93, $DF,
    $8E, $00, $74, $AE, $9F, $4B, $F8, $67, $D7, $B9, $28, $D8, $1F, $01, $70,
    $0F, $14, $94, $82, $A0, $01, $D3, $93, $E1, $CF, $34, $42, $55, $CE, $9F,
    $9E, $B2, $6F, $8F, $02, $27, $59, $CA, $C9, $CB, $5B, $AD, $15, $77, $33,
    $6E, $27, $56, $32, $C3, $F6, $F1, $24, $DB, $1E, $9D, $12, $95, $3E, $6E,
    $BF, $5E, $66, $22, $F4, $2C, $32, $B6, $01, $67, $A2, $7D, $0C, $DC, $E0,
    $05, $00, $ED, $7B, $0D, $48, $9C, $C6, $65, $A7, $34, $5E, $79, $D0, $A6,
    $14, $D8, $8E, $D5, $B6, $05, $A7, $71, $71, $53, $CF, $1E, $C7, $AF, $47,
    $F6, $5B, $84, $FF, $BE, $0B, $25, $63, $43, $10, $6E, $27, $C4, $2A, $82,
    $A9, $62, $0B, $D3, $D5, $EC, $7D, $E3, $98, $90, $0E, $6A, $40, $3C, $20,
    $C7, $6D, $C0, $BF, $EF, $6F, $F2, $96, $41, $17, $72, $D4, $F3, $80, $C2,
    $FC, $F8, $3E, $D8, $CC, $91, $F5, $61, $CE, $FB, $5C, $4B, $3B, $DC, $13,
    $65, $39, $60, $E4, $13, $A9, $32, $50, $73, $01, $B9, $79, $77, $F8, $4F,
    $D6, $BB, $F0, $50, $38, $2A, $66, $39, $11, $C4, $DE, $3E, $0A, $FB, $10,
    $FC, $4D, $8E, $42, $49, $13, $DB, $18, $2F, $68, $FB, $55, $98, $23, $34,
    $46, $4E, $8F, $8F, $CA, $32, $40, $91, $4B, $E8, $0F, $7C, $09, $51, $3F,
    $8B, $89, $93, $37, $02, $0D, $84, $69, $1A, $70, $BA, $40, $19, $BF, $9E,
    $33, $20, $69, $33, $55, $72, $ED, $7B, $07, $55, $51, $AA, $9D, $C1, $CE,
    $8F, $1A, $7C, $84, $DC, $5D, $5F, $14, $47, $55, $E4, $7B, $9D, $26, $31,
    $90, $3F, $4A, $42, $F4, $B4, $17, $ED, $66, $3C, $3E, $D7, $B3, $85, $28,
    $A2, $D0, $04, $40, $E0, $C5, $88, $90, $97, $71, $FE, $4E, $62, $63, $95,
    $2C, $7A, $3C, $A1, $90, $3B, $5C, $FC, $08, $DC, $E0, $B8, $9E, $89, $0E,
    $6E, $E6, $FF, $D4, $73, $D8, $61, $F2, $BB, $E4, $F1, $E5, $47, $CB, $D1,
    $BA, $53, $27, $00, $5B, $A2, $AD, $3A, $FA, $A8, $9D, $70, $AE, $6A, $E1,
    $F2, $66, $52, $CC, $8D, $C6, $51, $8E, $39, $98, $0C, $B1, $08, $D2, $F4,
    $4A, $82, $74, $7C, $B0, $A2, $AB, $D0, $D0, $07, $0C, $CA, $92, $2D, $C5,
    $FD, $C5, $AC, $1F, $0E, $97, $D7, $B4, $C8, $92, $39, $25, $89, $95, $C9,
    $42, $0D, $4D, $DA, $E8, $A9, $24, $13, $0D, $6E, $AB, $FB, $B1, $E9, $E4,
    $EB, $BF, $52, $0E, $67, $05, $77, $C6, $39, $D2, $B4, $99, $C4, $ED, $B0,
    $6F, $77, $DF, $CB, $57, $51, $D3, $67, $A6, $DA, $84, $75, $74, $04, $F1,
    $FE, $D7, $D6, $DB, $7D, $83, $E6, $46, $53, $9C, $E5, $76, $A9, $DA, $D3,
    $26, $77, $64, $A8, $F9, $9E, $F2, $75, $56, $37, $1B, $54, $3B, $91, $E4,
    $A3, $B3, $78, $F0, $6F, $F9, $C7, $EF, $FF, $73, $5D, $22, $E0, $1A, $EB,
    $11, $F0, $ED, $45, $DC, $46, $84, $27, $AA, $E6, $F4, $8F, $2C, $AE, $7C,
    $20, $3A, $EF, $C6, $39, $FB, $8B, $9F, $AE, $21, $9D, $84, $45, $21, $03,
    $5E, $73, $7E, $3D, $3F, $F6, $84, $90, $18, $3E, $12, $59, $95, $D7, $46,
    $ED, $BA, $36, $37, $92, $D9, $96, $05, $51, $F7, $E0, $EC, $C0, $D1, $AD,
    $2B, $FC, $03, $F1, $E6, $6D, $01, $DB, $4D, $19, $F0, $ED, $BD, $97, $87,
    $6D, $9C, $E9, $89, $9B, $22, $C8, $84, $5B, $62, $3D, $CF, $43, $26, $79,
    $1A, $2D, $92, $4C, $F2, $C5, $8F, $C7, $FA, $B7, $6B, $C6, $EC, $73, $45,
    $7D, $78, $A2, $22, $4B, $A2, $CA, $C5, $65, $3E, $C5, $A7, $22, $98, $F0,
    $AB, $EF, $81, $50, $6E, $48, $34, $04, $47, $28, $D5, $C3, $56, $D2, $90,
    $42, $47, $FC, $7E, $BA, $93, $94, $E1, $F4, $6D, $CE, $65, $86, $AF, $21,
    $69, $33, $59, $5F, $8C, $94, $9A, $1B, $F8, $79, $5C, $FF, $20, $AA, $CB,
    $CC, $03, $01, $63, $17, $10, $EF, $6B, $17, $47, $FB, $ED, $FC, $3A, $C4,
    $98, $B5, $FA, $8D, $14, $15, $25, $6D, $B6, $C0, $91, $A0, $A4, $50, $50,
    $8E, $CA, $95, $12, $8A, $3F, $E9, $0B, $6F, $EC, $FB, $F9, $78, $9A, $7F,
    $E2, $1A, $C6, $A9, $AA, $B1, $C3, $76, $9A, $43, $25, $0A, $75, $AD, $75,
    $B4, $BC, $01, $98, $64, $2C, $ED, $83, $9F, $AF, $59, $1B, $3C, $3F, $B5,
    $4F, $76, $68, $95, $DE, $1D, $7A, $8A, $51, $F8, $9E, $6F, $0F, $9E, $41,
    $83, $AD, $55, $9E, $0C, $BD, $C3, $64, $6B, $DD, $78, $A1, $C3, $C2, $A9,
    $87, $B3, $69, $99, $B7, $B7, $CB, $87, $11, $33, $AB, $26, $47, $42, $60,
    $F5, $B1, $BD, $AA, $F4, $17, $2C, $05, $91, $73, $7B, $FE, $CA, $E7, $B7,
    $6F, $E6, $60, $25, $CB, $31, $A1, $77, $BC, $AD, $B7, $FD, $C7, $9E, $0E,
    $FE, $C5, $5C, $A9, $A8, $51, $E5, $28, $4F, $82, $FF, $B4, $C8, $9A, $FE,
    $EF, $31, $73, $69, $81, $72, $DA, $1D, $DB, $3F, $22, $AE, $5A, $6D, $9A,
    $15, $D3, $DA, $B4, $F6, $2F, $47, $0A, $0F, $BA, $DB, $88, $95, $26, $31,
    $6A, $90, $89, $22, $E3, $7E, $C0, $B7, $80, $CD, $00, $ED, $B5, $F8, $8D,
    $3B, $1F, $4F, $D8, $01, $64, $B6, $6F, $4D, $C1, $EF, $97, $A7, $AE, $2D,
    $6F, $C8, $2C, $12, $EA, $70, $99, $44, $1C, $C2, $2B, $47, $8A, $D2, $6E,
    $51, $85, $54, $56, $29, $C2, $94, $CF, $67, $B7, $26, $E5, $93, $70, $00,
    $D2, $F5, $D4, $D3, $A0, $28, $A8, $F3, $32, $AF, $B4, $18, $CE, $27, $DD,
    $B6, $8D, $D0, $1B, $D9, $41, $91, $9B, $A4, $B6, $A6, $7A, $EF, $6C, $A0,
    $D6, $A5, $14, $6F, $43, $CD, $8C, $8B, $C1, $79, $10, $86, $CE, $F0, $DE,
    $98, $3D, $D2, $93, $E3, $CE, $1D, $E7, $8A, $D3, $38, $9C, $E5, $4D, $79,
    $CC, $4F, $42, $CB, $D9, $72, $4A, $66, $B8, $07, $94, $C3, $58, $65, $D3,
    $35, $55, $5E, $D3, $D4, $48, $49, $BF, $AA, $54, $2B, $3C, $4D, $C9, $67,
    $ED, $4A, $01, $6A, $0E, $19, $C9, $28, $1E, $7F, $F1, $7F, $E7, $86, $F4,
    $BA, $4E, $88, $B1, $CE, $AD, $DE, $B6, $16, $B3, $3C, $96, $9E, $0A, $D0,
    $47, $48, $38, $9B, $8D, $51, $7A, $AC, $CF, $0F, $95, $2E, $08, $E6, $E1,
    $6B, $91, $CA, $F1, $3B, $F5, $D2, $BC, $86, $2A, $B4, $15, $C4, $C0, $23,
    $DA, $A6, $E7, $74, $31, $5A, $72, $F9, $A9, $15, $1E, $FA, $2D, $C0, $D3,
    $46, $0E, $0F, $A0, $AA, $91, $57, $A5, $E8, $D4, $D8, $38, $DE, $F2, $66,
    $E9, $0D, $A8, $A4, $3C, $AD, $FD, $29, $38, $E9, $A1, $56, $2A, $0F, $04,
    $B1, $2A, $BA, $5E, $03, $58, $48, $25, $97, $CD, $21, $B0, $4F, $B7, $A4,
    $FD, $7C, $2D, $C1, $D8, $97, $9D, $87, $E9, $5E, $ED, $3D, $50, $E9, $AA,
    $C8, $DF, $B6, $EA, $75, $27, $CC, $B1, $62, $CC, $6A, $FA, $23, $ED, $DF,
    $C0, $6C, $8A, $91, $AA, $BC, $2F, $3D, $26, $6D, $63, $01, $1E, $A5, $83,
    $B8, $9B, $F5, $25, $B3, $F4, $BB, $1F, $E4, $6A, $8B, $C5, $F4, $3D, $29,
    $0B, $EF, $B2, $ED, $93, $66, $83, $8B, $D8, $4A, $B6, $9B, $FC, $D1, $E1,
    $64, $ED, $25, $3E, $B6, $E1, $C5, $2E, $E5, $64, $B7, $03, $2B, $75, $0D,
    $2E, $00, $70, $56, $2F, $34, $B6, $F0, $DF, $D2, $61, $64, $F1, $24, $6D,
    $3E, $CE, $53, $27, $04, $18, $83, $C4, $39, $7F, $5F, $75, $A7, $C1, $7D,
    $66, $41, $E5, $0A, $65, $65, $1F, $A1, $35, $57, $E9, $48, $BB, $26, $3E,
    $EC, $90, $6F, $00, $DE, $AB, $9B, $F0, $25, $21, $93, $D6, $AB, $37, $08,
    $B0, $91, $43, $CE, $02, $3D, $18, $DD, $AA, $E9, $F6, $F8, $D0, $B7, $3E,
    $21, $C9, $AC, $64, $3E, $17, $03, $D9, $A9, $D3, $BC, $F0, $88, $22, $5D,
    $F1, $24, $9B, $49, $DD, $A9, $A5, $23, $99, $F4, $41, $13, $82, $64, $C7,
    $FE, $A4, $D6, $81, $F2, $0B, $37, $38, $58, $F3, $5C, $12, $B9, $CC, $03,
    $5A, $1E, $51, $38, $15, $5E, $2A, $79, $75, $D9, $DE, $54, $95, $7D, $07,
    $5F, $05, $11, $0A, $BD, $1A, $20, $CC, $9D, $0A, $04, $21, $59, $41, $E3,
    $F6, $24, $75, $AC, $66, $75, $72, $5A, $42, $9A, $9D, $11, $C0, $1C, $9D,
    $D6, $57, $E2, $C5, $F5, $F3, $41, $0A, $6B, $11, $65, $BA, $C3, $6E, $B8,
    $38, $54, $51, $3C, $72, $D6, $A5, $00, $CD, $9D, $61, $1B, $C8, $12, $FC,
    $DD, $94, $22, $E7, $18, $46, $C6, $46, $C1, $8F, $DB, $6A, $38, $F7, $E4,
    $CA, $04, $15, $7B, $65, $D2, $50, $82, $E5, $F2, $02, $5B, $E9, $C0, $2B,
    $B5, $34, $9D, $F7, $BB, $D1, $64, $CB, $B3, $4B, $90, $A6, $4F, $8A, $75,
    $F4, $4B, $07, $3F, $37, $19, $5D, $A1, $15, $A8, $64, $58, $B4, $99, $5D,
    $01, $7A, $67, $EC, $D8, $89, $4D, $D3, $5A, $9B, $FE, $B3, $F5, $14, $78,
    $46, $2C, $6C, $71, $DD, $DC, $43, $46, $0A, $97, $CA, $56, $C1, $33, $48,
    $35, $68, $08, $F0, $2B, $8F, $79, $51, $D7, $4F, $C1, $00, $FF, $8E, $2C,
    $60, $DB, $99, $24, $98, $61, $E7, $C3, $22, $6B, $DE, $D7, $7C, $AC, $00,
    $79, $E7, $DD, $D4, $B7, $4B, $97, $68, $E5, $99, $AF, $4A, $98, $4A, $20,
    $53, $63, $D5, $D1, $A4, $D6, $3A, $BB, $A4, $51, $00, $47, $F2, $B7, $3B,
    $DF, $89, $12, $7E, $15, $95, $9F, $CC, $D5, $AA, $F0, $F0, $7C, $6E, $0D,
    $CC, $DF, $DC, $BE, $83, $88, $A7, $6F, $94, $B3, $77, $55, $4C, $EC, $01,
    $C3, $1D, $38, $2A, $DC, $C0, $33, $BD, $0A, $3A, $A1, $69, $E6, $28, $3B,
    $09, $86, $DD, $D1, $68, $53, $3D, $01, $0A, $D1, $70, $11, $A8, $13, $BD,
    $C1, $6C, $B8, $3B, $14, $12, $A8, $21, $64, $85, $F2, $DF, $4F, $9F, $83,
    $58, $3C, $0C, $9C, $D8, $9B, $99, $2E, $9C, $11, $57, $92, $19, $30, $62,
    $44, $4A, $80, $35, $8A, $B2, $E3, $D5, $3A, $DB, $A7, $4B, $07, $88, $C3,
    $29, $FB, $DA, $24, $0D, $1B, $1B, $F9, $74, $4E, $F9, $62, $3E, $FC, $5E,
    $00, $B5, $14, $30, $37, $05, $5E, $A2, $2B, $6B, $4C, $EC, $CE, $8F, $AC,
    $87, $79, $C8, $F3, $4C, $EB, $46, $04, $09, $B9, $65, $A6, $75, $72, $91,
    $0C, $95, $06, $92, $22, $F0, $48, $27, $90, $31, $84, $31, $8C, $15, $82,
    $F6, $FA, $B3, $7D, $21, $85, $91, $DD, $95, $8B, $0B, $BF, $39, $1D, $F4,
    $91, $46, $73, $95, $59, $D2, $84, $4D, $6B, $8B, $AE, $87, $9C, $E8, $99,
    $DB, $3D, $49, $BC, $66, $71, $BF, $40, $D0, $9A, $7B, $19, $89, $00, $22,
    $8E, $1C, $E4, $79, $97, $BA, $A4, $30, $C0, $50, $A9, $C4, $7B, $D2, $32,
    $B2, $09, $88, $50, $73, $71, $7D, $8C, $0D, $25, $73, $18, $1E, $4A, $F5,
    $89, $B8, $C8, $5A, $66, $68, $92, $E3, $29, $4B, $D4, $45, $60, $0C, $ED,
    $43, $1C, $19, $66, $2A, $DC, $87, $4D, $5A, $97, $5D, $AB, $28, $9E, $48,
    $EF, $C0, $77, $F3, $8E, $DC, $E4, $75, $31, $76, $BA, $A5, $2F, $BE, $44,
    $3C, $2F, $0E, $43, $08, $2B, $91, $DB, $B8, $38, $E3, $F0, $E4, $BB, $E2,
    $C9, $67, $0D, $61, $FD, $32, $98, $96, $FE, $28, $F6, $59, $BF, $A8, $B5,
    $8F, $59, $D5, $9F, $6C, $A3, $D1, $CC, $54, $16, $60, $CD, $83, $88, $BC,
    $B5, $D7, $88, $8B, $DC, $54, $63, $38, $90, $40, $2A, $2C, $17, $F9, $35,
    $9D, $BE, $D0, $FC, $FA, $D0, $9B, $4E, $26, $AC, $CC, $2E, $FB, $5B, $28,
    $E7, $E2, $69, $82, $21, $6C, $CA, $17, $04, $2B, $16, $55, $B2, $4E, $7D,
    $46, $AB, $B3, $17, $DD, $D9, $74, $89, $DB, $68, $20, $24, $16, $96, $29,
    $E8, $42, $25, $37, $1F, $A2, $2C, $62, $75, $2D, $4D, $95, $BC, $15, $B3,
    $82, $C5, $B3, $44, $FF, $FF, $57, $F4, $AE, $C4, $72, $2A, $3C, $82, $14,
    $94, $B0, $29, $C9, $F0, $7F, $8F, $38, $94, $6C, $B3, $1E, $C3, $63, $0A,
    $82, $1E, $DF, $42, $3A, $BA, $77, $2B, $23, $A3, $B8, $B0, $A8, $E1, $5A,
    $07, $6B, $45, $79, $8B, $36, $8B, $66, $E9, $8B, $9B, $2A, $8D, $75, $FB,
    $E9, $7B, $65, $B9, $9E, $F9, $55, $57, $F7, $C3, $69, $8F, $45, $B0, $CA,
    $ED, $BE, $E8, $85, $98, $8F, $2D, $87, $CB, $76, $CE, $26, $D5, $E4, $40,
    $BF, $A1, $DD, $7E, $49, $1D, $20, $A3, $04, $FC, $3D, $63, $95, $B8, $AD,
    $59, $2A, $C4, $10, $4E, $8E, $50, $24, $E2, $3E, $19, $23, $54, $D8, $2A,
    $81, $3B, $91, $74, $58, $76, $7D, $B6, $E6, $AE, $CC, $1C, $EE, $1A, $58,
    $70, $EC, $77, $1B, $EF, $5C, $56, $7A, $04, $8D, $33, $61, $75, $C1, $06,
    $B2, $D6, $92, $23, $81, $46, $D9, $BA, $15, $C8, $97, $3C, $B5, $BF, $8F,
    $6E, $A6, $5C, $A2, $08, $F8, $0A, $13, $11, $FD, $20, $42, $5B, $7B, $6F,
    $89, $E1, $0A, $69, $46, $D2, $2A, $FE, $43, $95, $E9, $66, $FE, $22, $4B,
    $81, $8D, $55, $B8, $78, $11, $3F, $1E, $C6, $08, $84, $90, $E3, $DF, $4A,
    $98, $2D, $98, $58, $7D, $C9, $73, $53, $2B, $97, $DA, $47, $93, $7B, $0A,
    $EA, $9D, $92, $36, $3D, $0B, $12, $23, $82, $EC, $E8, $75, $A9, $71, $61,
    $26, $A3, $4B, $6A, $F0, $51, $E8, $2E, $94, $E7, $28, $20, $4B, $02, $74,
    $C3, $CE, $D1, $3F, $B6, $BE, $06, $5E, $CC, $63, $6D, $C8, $48, $F5, $6C,
    $A2, $80, $FC, $BD, $AB, $73, $76, $26, $6D, $23, $C7, $03, $C6, $BB, $A2,
    $EE, $AE, $D1, $7F, $CA, $4A, $45, $3C, $02, $B7, $9D, $00, $CD, $1C, $D5,
    $88, $AE, $E8, $FD, $68, $A5, $50, $2A, $F3, $A4, $73, $89, $CC, $4F, $B2,
    $A1, $53, $25, $2A, $08, $7A, $79, $59, $29, $AC, $FB, $5D, $CE, $CF, $A5,
    $53, $8D, $93, $46, $CC, $66, $98, $1A, $59, $04, $15, $0F, $A4, $95, $09,
    $A9, $9A, $1A, $32, $B6, $92, $A4, $3C, $38, $8D, $E4, $EC, $2D, $DD, $FE,
    $46, $2B, $E0, $88, $DE, $FB, $9C, $1F, $DD, $8F, $68, $0D, $6C, $1B, $F0,
    $CE, $89, $20, $30, $87, $7B, $A6, $56, $B0, $68, $16, $89, $8A, $BF, $47,
    $08, $20, $82, $CB, $36, $EC, $95, $8B, $69, $5F, $AB, $2D, $A3, $E1, $6E,
    $F4, $87, $12, $F9, $F9, $2D, $5A, $CC, $EA, $AA, $0F, $C6, $D2, $75, $C1,
    $CE, $6E, $F2, $79, $42, $23, $D0, $F7, $48, $A6, $0A, $31, $9B, $CE, $B4,
    $B2, $74, $38, $E2, $90, $AA, $AC, $4A, $1A, $27, $6F, $85, $75, $3F, $E4,
    $FE, $C5, $A3, $3A, $F4, $A5, $EA, $06, $29, $FA, $EC, $F3, $16, $76, $69,
    $02, $C9, $46, $BB, $F7, $BA, $CD, $93, $E7, $48, $62, $F2, $12, $A3, $6D,
    $E2, $74, $53, $68, $AD, $B7, $AA, $91, $82, $7A, $C3, $5A, $D9, $0A, $ED,
    $E0, $F7, $0E, $93, $F8, $65, $BA, $0E, $A1, $4D, $8A, $E8, $E0, $F8, $A8,
    $83, $53, $E3, $60, $28, $0C, $3F, $65, $F4, $81, $B6, $B9, $23, $90, $CB,
    $78, $AF, $F6, $48, $E8, $49, $B4, $76, $78, $20, $39, $16, $AA, $F2, $58,
    $66, $F7, $68, $58, $98, $3B, $B6, $A9, $92, $38, $F7, $09, $D4, $33, $D2,
    $28, $E2, $BF, $41, $14, $87, $3C, $A5, $FD, $D8, $A2, $76, $14, $D0, $8E,
    $31, $A0, $CE, $BA, $5A, $83, $7B, $48, $B7, $CB, $BF, $99, $1C, $15, $D2,
    $8A, $5D, $C3, $A2, $53, $4E, $6A, $FF, $69, $E4, $E9, $58, $18, $DC, $08,
    $00, $B5, $74, $A3, $28, $A1, $24, $AC, $2C, $83, $6D, $93, $C6, $63, $0F,
    $E2, $46, $94, $54, $16, $F4, $9D, $42, $FA, $21, $84, $CA, $21, $A8, $31,
    $60, $3C, $CF, $CD, $8D, $61, $CD, $4E, $57, $81, $6D, $98, $9B, $B3, $8B,
    $E6, $C0, $04, $6D, $5C, $4E, $EA, $24, $FB, $4F, $9C, $82, $54, $9A, $E3,
    $5C, $B8, $54, $2E, $0D, $BD, $55, $EB, $7C, $59, $45, $83, $24, $E9, $89,
    $74, $D5, $E6, $64, $61, $26, $21, $E2, $1C, $E1, $BD, $6B, $FB, $DB, $54,
    $20, $01, $EE, $E9, $4E, $AC, $2A, $9C, $A5, $68, $56, $4C, $05, $07, $5D,
    $8C, $03, $B1, $4E, $9D, $73, $32, $F5, $55, $AD, $9C, $42, $2A, $50, $97,
    $08, $B4, $AA, $E2, $BB, $77, $29, $DC, $3C, $B7, $5C, $15, $E2, $A8, $74,
    $37, $47, $4B, $59, $DB, $87, $B5, $AA, $07, $01, $9F, $52, $70, $BC, $79,
    $79, $C3, $0A, $58, $0A, $7E, $04, $89, $04, $36, $FD, $8F, $5E, $3F, $0B,
    $DE, $31, $10, $33, $A2, $5F, $F5, $F2, $4E, $0A, $C1, $16, $DA, $73, $B1,
    $2E, $D8, $2A, $A6, $06, $D8, $94, $64, $EF, $B8, $41, $86, $CB, $D8, $92,
    $48, $1D, $56, $9F, $64, $99, $C4, $6D, $97, $E6, $FE, $D6, $97, $A4, $14,
    $BC, $8A, $D6, $68, $67, $3B, $44, $7E, $24, $B5, $84, $12, $86, $ED, $A7,
    $7B, $E2, $A9, $CC, $6B, $E6, $D8, $13, $96, $5F, $ED, $A0, $CC, $97, $14,
    $BF, $8F, $D8, $3D, $5A, $3E, $FA, $53, $32, $4A, $18, $27, $B1, $1A, $BF,
    $DF, $37, $94, $3E, $EC, $2F, $6D, $F1, $E5, $60, $B2, $95, $B7, $3D, $34,
    $1A, $15, $1F, $C2, $78, $1F, $C1, $64, $71, $90, $60, $F1, $51, $05, $47,
    $C6, $18, $24, $C4, $75, $99, $B0, $BE, $9E, $6F, $D8, $64, $65, $F2, $48,
    $C1, $26, $6F, $CD, $1E, $59, $52, $99, $77, $D6, $1C, $6A, $82, $D9, $35,
    $2C, $FB, $7F, $06, $D9, $3E, $A1, $E9, $A7, $58, $35, $D5, $D3, $23, $B1,
    $E0, $88, $14, $06, $EB, $C3, $BE, $31, $1F, $28, $4E, $85, $12, $2C, $72,
    $1E, $AF, $0E, $67, $24, $3E, $6D, $34, $D1, $29, $99, $3F, $5E, $59, $CB,
    $CA, $4B, $49, $F4, $00, $AC, $25, $D0, $A1, $5D, $9B, $5B, $78, $57, $2C,
    $F5, $2C, $90, $2C, $B0, $08, $07, $BB, $A6, $DD, $F2, $59, $60, $1B, $37,
    $F7, $8A, $A7, $9E, $9E, $9B, $35, $65, $FC, $23, $0F, $23, $87, $D9, $8F,
    $24, $07, $B9, $D0, $B3, $9E, $AD, $3E, $EE, $BB, $9C, $13, $25, $7B, $EB,
    $F4, $D4, $6F, $AD, $2D, $C6, $D3, $22, $47, $F1, $8E, $3E, $F4, $FF, $A9,
    $5A, $8E, $BA, $04, $3F, $A5, $BD, $7C, $32, $50, $14, $C1, $BB, $54, $BD,
    $96, $3D, $39, $3B, $44, $A5, $F2, $96, $60, $BE, $AB, $83, $41, $F8, $E0,
    $9E, $C9, $AB, $D6, $DA, $4A, $23, $11, $9B, $BE, $01, $66, $25, $93, $C3,
    $C0, $44, $B6, $6D, $BD, $69, $BE, $B8, $B1, $93, $40, $3E, $BE, $C4, $BD,
    $7C, $06, $E4, $B4, $AD, $2B, $39, $C2, $D3, $A3, $ED, $E4, $71, $8E, $45,
    $3B, $C9, $1E, $77, $E9, $A6, $B5, $4B, $E7, $FE, $49, $F6, $F4, $00, $BF,
    $D0, $EE, $46, $3F, $E7, $3B, $54, $B5, $80, $19, $93, $0A, $05, $EF, $6A,
    $B9, $09, $09, $35, $D8, $45, $42, $26, $BC, $C6, $8F, $66, $BD, $AD, $7F,
    $F8, $D1, $37, $05, $2A, $05, $F2, $34, $AA, $01, $4C, $C0, $7B, $1B, $EB,
    $65, $6B, $BB, $7B, $8F, $C3, $FB, $D3, $D8, $40, $D0, $AF, $4B, $B1, $32,
    $59, $6A, $AF, $FB, $FD, $77, $4F, $79, $83, $77, $CA, $FE, $2E, $ED, $75,
    $D0, $5C, $50, $86, $4F, $1A, $A2, $84, $39, $7B, $C7, $5C, $53, $D3, $71,
    $73, $F5, $51, $C1, $3F, $28, $9A, $FB, $2E, $5E, $4C, $E1, $E7, $01, $AB,
    $9B, $42, $45, $C6, $3F, $60, $12, $D8, $5F, $DC, $E7, $C8, $2D, $F3, $0C,
    $F0, $1C, $16, $74, $9E, $13, $5A, $28, $FA, $8F, $95, $1C, $35, $53, $45,
    $C1, $3D, $85, $74, $BA, $4D, $42, $34, $84, $1A, $22, $84, $00, $7B, $2D,
    $96, $24, $BA, $2D, $D9, $41, $FE, $75, $42, $AD, $7D, $72, $A8, $BF, $58,
    $B3, $3D, $5C, $50, $2D, $73, $94, $C9, $F0, $CF, $0B, $BC, $8F, $87, $21,
    $8A, $EE, $E6, $2A, $D7, $47, $CB, $13, $E1, $7E, $CF, $50, $F8, $2A, $DC,
    $F7, $03, $E8, $3B, $52, $45, $DA, $7A, $5D, $C5, $63, $92, $F0, $0A, $8B,
    $46, $8C, $1B, $9B, $9B, $10, $E2, $D1, $D3, $C5, $6E, $4B, $6F, $8B, $B2,
    $7E, $FC, $17, $BA, $E1, $C1, $86, $73, $94, $98, $C0, $81, $39, $75, $60,
    $C7, $42, $76, $26, $8C, $63, $7F, $2C, $6B, $F7, $A9, $56, $7E, $7B, $1F,
    $BD, $68, $7C, $26, $16, $32, $8D, $63, $41, $21, $29, $B1, $8F, $F1, $94,
    $00, $6D, $2A, $22, $1F, $7A, $6E, $79, $71, $39, $F5, $A7, $DB, $E8, $9F,
    $32, $74, $2A, $4A, $73, $CC, $91, $57, $E3, $CB, $2F, $8C, $2A, $7D, $BE,
    $BB, $1A, $52, $0C, $DC, $DE, $9D, $B5, $85, $14, $65, $55, $F7, $97, $CB,
    $C2, $13, $19, $97, $F9, $FB, $F8, $B5, $C8, $47, $DA, $B4, $61, $E7, $EE,
    $86, $66, $3D, $F4, $1B, $16, $FB, $6B, $A0, $A8, $CA, $6F, $60, $01, $DC,
    $3C, $AD, $4E, $CE, $5A, $42, $35, $94, $E4, $FC, $9D, $54, $71, $F0, $4A,
    $56, $8C, $2C, $C3, $85, $D2, $A4, $48, $19, $05, $2D, $CA, $69, $C9, $7E,
    $25, $03, $D3, $8B, $1D, $7B, $A8, $C5, $CE, $A1, $53, $0E, $9A, $05, $A7,
    $64, $C3, $F2, $E3, $A2, $37, $56, $E4, $C4, $44, $3D, $29, $9C, $34, $C4,
    $65, $EE, $0A, $BF, $5B, $43, $DF, $77, $34, $06, $96, $23, $A5, $AD, $E9,
    $15, $82, $09, $08, $C8, $95, $F0, $46, $2A, $E0, $EA, $8E, $E1, $9B, $9A,
    $A3, $6D, $8A, $F3, $5F, $58, $98, $86, $6C, $D4, $B8, $7A, $45, $8C, $80,
    $F1, $5A, $72, $0B, $EB, $B9, $D9, $06, $F0, $98, $2A, $02, $18, $90, $F2,
    $60, $27, $7B, $2F, $82, $E2, $35, $07, $10, $D8, $BF, $58, $6F, $2F, $75,
    $C6, $CB, $44, $61, $40, $4F, $2B, $DA, $1B, $61, $53, $FB, $37, $06, $E3,
    $97, $D3, $C4, $97, $BF, $05, $9E, $28, $04, $FD, $10, $94, $F7, $7B, $19,
    $C3, $A7, $2C, $B2, $63, $15, $D3, $5E, $FE, $D4, $76, $92, $9C, $5F, $E4,
    $2F, $76, $D5, $5A, $54, $92, $1B, $21, $2F, $43, $A5, $5D, $A7, $E7, $55,
    $6C, $FA, $6C, $3E, $8E, $A8, $62, $33, $B8, $4E, $AD, $73, $C8, $4C, $21,
    $D9, $79, $42, $8E, $5E, $D1, $E3, $16, $DB, $E7, $8C, $C6, $56, $3D, $A8,
    $D2, $41, $0C, $DA, $F0, $80, $31, $5F, $A6, $1A, $AD, $38, $95, $E8, $E2,
    $02, $0F, $A2, $F2, $8A, $7D, $2C, $F6, $88, $71, $F2, $EF, $12, $57, $BA,
    $A9, $AB, $2A, $46, $F1, $05, $AC, $99, $CC, $D0, $EA, $CE, $A4, $39, $27,
    $C5, $88, $F0, $64, $16, $FE, $D5, $29, $1C, $FB, $9D, $49, $F5, $06, $1E,
    $B2, $D7, $A0, $0C, $99, $12, $65, $4C, $8F, $98, $8D, $38, $3C, $F7, $C3,
    $71, $E7, $04, $74, $13, $F4, $B7, $00, $74, $0C, $EF, $86, $C0, $96, $1E,
    $2D, $14, $A5, $17, $15, $E9, $9C, $77, $64, $01, $1D, $70, $12, $18, $FF,
    $DF, $67, $06, $6E, $00, $53, $2C, $67, $28, $10, $3E, $34, $4A, $B2, $D5,
    $A5, $4A, $98, $5A, $13, $48, $EF, $26, $85, $6E, $A1, $FD, $6E, $F9, $70,
    $33, $E9, $78, $1C, $21, $69, $7B, $C6, $C0, $37, $32, $FB, $8C, $42, $B7,
    $0C, $01, $15, $FC, $B0, $5C, $F4, $9D, $09, $6B, $79, $10, $81, $9C, $2F,
    $EE, $2E, $92, $04, $54, $A6, $EE, $9A, $98, $63, $19, $B4, $C3, $3F, $3C,
    $6C, $A3, $97, $AF, $DB, $59, $11, $88, $51, $AC, $32, $24, $70, $9C, $1D,
    $BE, $9D, $ED, $4A, $83, $AF, $AC, $FB, $9E, $D4, $DF, $B1, $44, $02, $EB,
    $0F, $B1, $7D, $5F, $C8, $8D, $3F, $2B, $8D, $52, $4B, $6E, $2D, $7F, $3A,
    $74, $44, $C7, $8F, $7B, $34, $94, $9C, $5B, $46, $67, $01, $3D, $2F, $13,
    $17, $DE, $55, $D0, $D1, $21, $8A, $8F, $DE, $DF, $27, $30, $12, $43, $B2,
    $70, $76, $21, $77, $9F, $6B, $FE, $CE, $9B, $D1, $E1, $E5, $82, $1D, $01,
    $4D, $1C, $BF, $F6, $AB, $03, $5C, $00, $3B, $E9, $A8, $87, $19, $9E, $F0,
    $ED, $6B, $AE, $9A, $4B, $EB, $B2, $2D, $C2, $72, $B0, $A9, $61, $6C, $2A,
    $3A, $E1, $71, $B8, $9A, $AF, $60, $39, $A2, $FC, $AD, $18, $E1, $D9, $CE,
    $1A, $3D, $91, $F7, $B6, $F0, $C3, $8B, $EF, $66, $F6, $EB, $2F, $D2, $A3,
    $A0, $A3, $7C, $F4, $F3, $03, $1A, $AA, $79, $E3, $BE, $F5, $25, $D8, $22,
    $2F, $E7, $28, $6A, $BA, $EB, $C6, $94, $7A, $F3, $CB, $38, $AC, $7A, $BA,
    $4E, $14, $7C, $2F, $1C, $B7, $22, $E1, $88, $AB, $AD, $F7, $5D, $AF, $5F,
    $2E, $57, $86, $3A, $4E, $F8, $56, $CC, $E2, $79, $87, $76, $3E, $5C, $DD,
    $D3, $34, $74, $C2, $2F, $19, $67, $09, $3C, $F6, $FD, $9C, $67, $F0, $C1,
    $D5, $88, $5E, $C0, $80, $CE, $17, $80, $B7, $16, $D2, $64, $26, $18, $C4,
    $1E, $64, $1D, $C5, $ED, $62, $1C, $AE, $BD, $2F, $79, $AB, $6B, $88, $C8,
    $CA, $C0, $F4, $43, $F5, $F2, $63, $E1, $13, $AB, $94, $38, $21, $ED, $B5,
    $94, $5C, $5D, $FF, $F5, $DC, $BA, $4B, $D9, $3D, $95, $D1, $01, $2F, $94,
    $A1, $A2, $09, $C4, $7D, $A5, $BF, $D5, $19, $E2, $D3, $39, $07, $94, $1E,
    $83, $B6, $57, $8A, $8E, $52, $6B, $27, $2B, $78, $30, $1D, $75, $FC, $F1,
    $72, $0F, $3C, $7E, $B2, $CB, $C4, $38, $D0, $A6, $2E, $77, $F0, $FB, $8D,
    $84, $DD, $66, $D0, $82, $58, $B8, $E5, $03, $9C, $E7, $8B, $20, $A6, $FC,
    $1F, $5A, $2B, $47, $D1, $97, $4F, $67, $04, $FB, $7F, $F4, $D6, $77, $05,
    $DB, $0A, $50, $35, $AD, $CF, $EC, $50, $76, $21, $B5, $72, $E8, $C5, $DC,
    $35, $DA, $0F, $41, $D2, $7D, $FD, $DC, $ED, $BD, $59, $5F, $64, $A4, $E5,
    $F1, $AF, $D1, $C3, $38, $C2, $3F, $A0, $1E, $82, $FA, $57, $ED, $27, $7D,
    $10, $CB, $2B, $D4, $04, $A0, $A5, $C4, $80, $F2, $20, $87, $C5, $4B, $60,
    $43, $F7, $EA, $63, $7D, $3C, $14, $FD, $D1, $5E, $12, $D1, $2D, $2F, $87,
    $63, $D1, $D1, $51, $EB, $BB, $8B, $28, $6F, $28, $F9, $0B, $44, $13, $45,
    $AD, $37, $E3, $28, $0D, $D1, $AC, $EE, $DD, $72, $B6, $BD, $70, $46, $31,
    $D6, $52, $4D, $45, $91, $C0, $EF, $ED, $46, $94, $DB, $3C, $76, $7D, $2C,
    $EF, $D6, $23, $92, $ED, $63, $70, $E3, $DB, $56, $55, $83, $06, $BB, $8C,
    $7B, $48, $F6, $86, $EE, $C3, $9A, $63, $A5, $02, $3F, $2F, $FD, $14, $2D,
    $19, $95, $60, $C6, $83, $56, $51, $F8, $1F, $BC, $13, $29, $83, $6A, $F3,
    $62, $E5, $71, $40, $A8, $DD, $E5, $93, $D2, $6B, $99, $F4, $44, $A0, $1E,
    $44, $5A, $97, $68, $D9, $77, $BA, $A5, $36, $F4, $5F, $35, $4A, $73, $4E,
    $D4, $AD, $48, $C1, $54, $AF, $F9, $D2, $7F, $46, $0F, $8F, $09, $28, $BA,
    $03, $A1, $8E, $FA, $07, $85, $8D, $4E, $A3, $2D, $E4, $2D, $40, $F8, $BB,
    $B2, $0F, $E0, $B2, $4D, $5B, $AE, $32, $61, $9C, $2B, $F1, $1A, $96, $42,
    $01, $F3, $CD, $28, $FC, $9D, $20, $D9, $05, $55, $50, $A3, $BC, $DD, $C4,
    $75, $73, $8A, $D4, $33, $18, $33, $63, $52, $BD, $FB, $A6, $60, $BB, $20,
    $3D, $58, $B8, $63, $25, $D8, $2C, $B9, $0A, $98, $1C, $3B, $06, $BC, $31,
    $A4, $E3, $E0, $54, $C0, $5E, $F4, $86, $A3, $FF, $52, $96, $88, $8D, $95,
    $E2, $B0, $37, $1E, $58, $A3, $E7, $5C, $4E, $A6, $65, $38, $05, $50, $3C,
    $35, $C8, $82, $89, $2F, $12, $40, $24, $CF, $37, $26, $C5, $08, $57, $6B,
    $25, $48, $B8, $BE, $28, $2E, $BF, $6F, $A2, $E2, $2A, $A3, $42, $73, $42,
    $8B, $9F, $CB, $1C, $DE, $93, $20, $37, $19, $4D, $F5, $FC, $2D, $C3, $10,
    $EF, $AC, $74, $FB, $A0, $A3, $A9, $E6, $A2, $A9, $9A, $1C, $19, $D1, $28,
    $48, $AD, $84, $10, $43, $DE, $83, $D6, $30, $21, $7D, $BA, $18, $39, $0B,
    $77, $4B, $9F, $F6, $B4, $31, $13, $1B, $47, $68, $87, $55, $C7, $40, $09,
    $34, $10, $FA, $71, $7B, $A4, $62, $CF, $9E, $18, $A4, $00, $C3, $F9, $72,
    $E4, $F6, $D7, $DC, $FF, $99, $55, $74, $9F, $46, $83, $98, $CA, $9C, $D4,
    $07, $7C, $B6, $69, $0E, $0F, $50, $A3, $F4, $C6, $C3, $14, $83, $DB, $66,
    $53, $BE, $8B, $0A, $4A, $2D, $F8, $89, $F9, $51, $E4, $A9, $D3, $21, $9A,
    $CA, $20, $B0, $87, $24, $15, $4F, $5B, $89, $89, $C8, $AE, $E6, $79, $98,
    $8A, $03, $BF, $3D, $4D, $46, $AC, $EA, $C9, $5E, $76, $58, $98, $2B, $9F,
    $6F, $FB, $50, $C0, $FD, $D5, $30, $69, $CC, $7D, $51, $08, $B0, $86, $62,
    $EC, $74, $BA, $5F, $EC, $B1, $DE, $8E, $4E, $04, $05, $54, $D8, $1F, $82,
    $19, $45, $B4, $C4, $C3, $7A, $33, $A3, $6E, $31, $D7, $F3, $74, $04, $84,
    $62, $F6, $01, $68, $EE, $60, $3B, $64, $85, $A9, $F3, $AE, $90, $49, $5C,
    $3F, $80, $24, $16, $E6, $A0, $DC, $CB, $1D, $C3, $58, $AA, $06, $8C, $65,
    $95, $1F, $67, $C4, $46, $8E, $7A, $10, $C2, $20, $1F, $41, $D3, $61, $A7,
    $C2, $75, $55, $82, $5E, $56, $BE, $0E, $77, $E2, $08, $9B, $4B, $85, $35,
    $6C, $38, $08, $60, $D0, $51, $38, $BF, $88, $9F, $4A, $37, $1C, $E7, $9C,
    $DB, $D7, $E7, $CB, $01, $8E, $41, $3F, $B9, $63, $96, $A3, $C7, $24, $88,
    $EA, $81, $0C, $A9, $DE, $29, $ED, $89, $4B, $9D, $16, $5E, $51, $7B, $7E,
    $0D, $6C, $1A, $03, $83, $37, $E9, $BD, $A9, $FE, $1A, $C5, $5C, $A4, $AF,
    $EE, $34, $73, $4D, $39, $14, $CB, $62, $20, $B9, $E7, $8A, $F3, $C5, $08,
    $46, $DF, $0A, $D8, $65, $60, $06, $C7, $B6, $3A, $D6, $B4, $E4, $30, $12,
    $33, $9B, $18, $59, $2D, $40, $12, $7C, $B8, $F5, $E0, $94, $65, $AD, $C7,
    $6F, $56, $72, $69, $1A, $E8, $42, $7A, $84, $C0, $14, $47, $6C, $76, $E3,
    $72, $26, $4E, $E8, $5B, $E4, $9A, $43, $0C, $86, $C2, $D2, $B8, $2F, $15,
    $F3, $09, $19, $D4, $61, $DE, $5C, $19, $7F, $2F, $B4, $97, $B4, $3A, $73,
    $4E, $A4, $30, $28, $AA, $78, $FE, $A6, $B1, $86, $EF, $7B, $39, $EC, $47,
    $06, $3B, $0A, $08, $A7, $CF, $F5, $90, $D3, $59, $2D, $B0, $F2, $F6, $4C,
    $7A, $58, $38, $77, $E1, $CD, $AC, $B2, $E0, $54, $DC, $7B, $D8, $4A, $A1,
    $39, $3F, $F5, $0D, $67, $E5, $20, $48, $97, $67, $67, $0E, $9F, $85, $3E,
    $BD, $95, $84, $EB, $25, $72, $D0, $ED, $72, $80, $73, $88, $F9, $D8, $37,
    $84, $B3, $B4, $68, $40, $B7, $40, $96, $AE, $F7, $33, $DF, $E2, $0F, $6E,
    $87, $3C, $FE, $4A, $EA, $60, $FB, $70, $1C, $85, $CB, $4E, $BD, $BE, $59,
    $99, $D1, $5F, $5D, $5D, $9C, $8D, $57, $91, $E2, $A0, $43, $38, $09, $F4,
    $A8, $76, $E6, $32, $00, $2F, $50, $4F, $92, $0C, $B8, $B5, $3B, $10, $58,
    $59, $C4, $FA, $BD, $B5, $BD, $36, $CA, $E6, $01, $19, $69, $69, $4B, $9C,
    $88, $30, $9A, $D9, $22, $0E, $3D, $06, $6F, $B5, $B8, $AA, $6D, $C4, $F8,
    $3B, $70, $9D, $72, $7F, $68, $02, $2A, $46, $F5, $B8, $BB, $5D, $9B, $9F,
    $E0, $D5, $93, $0E, $19, $FA, $14, $5F, $64, $9A, $14, $F1, $2E, $6E, $D5,
    $18, $EF, $C9, $C9, $B3, $13, $EE, $92, $6B, $2C, $99, $34, $D7, $82, $A3,
    $BD, $FE, $88, $DF, $F7, $E8, $CD, $82, $E4, $58, $57, $B8, $20, $C9, $A9,
    $41, $60, $40, $BF, $38, $B9, $A8, $C8, $52, $51, $B2, $7E, $3D, $B2, $79,
    $0D, $C9, $78, $11, $37, $CB, $E9, $70, $B6, $7D, $DA, $4D, $80, $15, $24,
    $D6, $E1, $85, $9F, $76, $95, $B4, $97, $44, $0E, $75, $11, $A7, $82, $9F,
    $B9, $1E, $C7, $E2, $7B, $67, $0C, $35, $14, $8A, $F2, $F7, $27, $79, $A8,
    $1E, $A8, $46, $F0, $19, $9E, $33, $46, $35, $97, $07, $8E, $A5, $43, $6F,
    $ED, $8E, $21, $6E, $CC, $B1, $30, $01, $20, $84, $F5, $8E, $69, $E9, $74,
    $C6, $18, $D3, $CB, $FD, $44, $73, $95, $3D, $D2, $D1, $51, $78, $92, $E7,
    $64, $B5, $E3, $A3, $6F, $17, $DE, $58, $25, $09, $26, $96, $CB, $0E, $D8,
    $69, $04, $E6, $32, $77, $AE, $07, $F8, $79, $DA, $AE, $A5, $09, $2F, $FD,
    $F1, $D3, $42, $DE, $3E, $7F, $AF, $B5, $F0, $FE, $FE, $0C, $C6, $0E, $A7,
    $C2, $B1, $0F, $B1, $72, $C9, $8D, $CA, $13, $5A, $12, $DD, $08, $21, $23,
    $6B, $1D, $36, $82, $3F, $33, $4C, $09, $40, $8C, $2C, $CE, $72, $FE, $9B,
    $55, $6F, $83, $19, $F6, $E4, $E3, $26, $DE, $E7, $75, $EE, $2D, $1D, $2F,
    $9A, $29, $14, $87, $A1, $91, $51, $E2, $CF, $6B, $D8, $80, $EE, $F7, $4B,
    $33, $33, $95, $DB, $51, $EB, $BF, $9F, $9B, $04, $75, $24, $25, $6A, $B9,
    $2F, $6A, $5A, $E8, $0E, $0A, $25, $23, $02, $FA, $FD, $15, $7F, $1B, $C6,
    $AD, $69, $68, $12, $BF, $79, $4E, $B4, $8C, $75, $DB, $94, $A5, $F4, $BF,
    $B8, $AD, $BC, $67, $9E, $F1, $11, $9B, $6A, $DF, $6F, $73, $80, $B7, $92,
    $8B, $79, $25, $BE, $B9, $AF, $85, $AA, $C1, $B5, $08, $8E, $CB, $7A, $00,
    $1D, $82, $F6, $00, $82, $35, $96, $9A, $A7, $53, $CD, $ED, $4B, $25, $76,
    $4A, $BF, $A2, $00, $9B, $50, $A8, $75, $5F, $A4, $15, $E4, $5C, $A6, $75,
    $CE, $61, $B8, $53, $79, $B0, $50, $48, $69, $FC, $2F, $6E, $5B, $FC, $DA,
    $CD, $72, $07, $0D, $D6, $87, $4E, $3B, $1E, $C3, $89, $71, $D8, $EB, $3A,
    $0C, $CB, $1C, $E2, $36, $B8, $C5, $BF, $0C, $79, $2F, $F6, $DE, $4E, $DA,
    $3D, $F8, $0C, $7A, $6C, $D2, $3F, $70, $81, $AE, $62, $EC, $01, $EC, $BE,
    $27, $D5, $A5, $9E, $63, $E4, $57, $DD, $97, $30, $AF, $05, $F5, $27, $BB,
    $77, $BF, $68, $12, $69, $07, $3D, $BD, $9D, $4E, $7A, $EC, $24, $37, $8D,
    $C9, $9C, $AA, $E5, $E8, $A5, $19, $0D, $32, $93, $DB, $0D, $45, $8A, $BC,
    $F6, $A0, $92, $38, $B5, $29, $4B, $07, $80, $CC, $50, $E9, $86, $4B, $76,
    $E0, $B2, $8B, $6E, $39, $A6, $E1, $7D, $25, $44, $32, $4C, $B6, $63, $EF,
    $0E, $7A, $8C, $3E, $6B, $24, $E2, $D4, $00, $DE, $7F, $54, $5E, $40, $F2,
    $35, $CE, $15, $BD, $8D, $53, $4E, $D9, $30, $A4, $E3, $B6, $15, $22, $7D,
    $42, $E2, $F8, $17, $D7, $17, $F3, $16, $33, $3D, $B9, $60, $C8, $FE, $71,
    $82, $12, $2B, $06, $F9, $BC, $FB, $8C, $17, $C1, $09, $7D, $24, $D4, $ED,
    $CE, $BB, $C0, $67, $16, $71, $88, $83, $3F, $56, $6C, $20, $8C, $D8, $2F,
    $C5, $D7, $14, $3D, $BF, $5D, $A5, $84, $D8, $81, $75, $25, $34, $50, $2C,
    $51, $20, $B2, $2E, $F6, $DB, $96, $59, $15, $AE, $88, $DF, $D7, $D1, $72,
    $05, $C9, $FB, $81, $8E, $CD, $97, $74, $22, $AC, $F8, $BD, $FD, $53, $0C,
    $48, $B2, $27, $6A, $D1, $CA, $01, $AF, $0E, $93, $B7, $96, $2B, $62, $43,
    $35, $48, $17, $F1, $44, $06, $BE, $22, $6F, $52, $6F, $8F, $05, $EE, $FA,
    $1A, $82, $74, $E2, $91, $D7, $D2, $B5, $01, $67, $75, $D9, $6B, $0F, $A7,
    $41, $02, $C5, $CF, $B6, $46, $97, $9F, $A3, $9F, $16, $62, $81, $4D, $7C,
    $E7, $69, $CE, $19, $C7, $5E, $11, $CB, $01, $4A, $1C, $9C, $28, $D8, $F4,
    $C1, $A2, $94, $E9, $90, $AC, $FB, $AA, $42, $29, $72, $AA, $6D, $71, $3E,
    $C9, $AF, $9F, $FC, $47, $E9, $19, $38, $6B, $FD, $4E, $16, $7B, $FC, $E7,
    $31, $D5, $26, $E5, $27, $52, $38, $69, $75, $43, $E4, $26, $94, $AF, $6D,
    $B1, $80, $AB, $C2, $30, $17, $E0, $9E, $24, $FA, $A6, $75, $4D, $DD, $4A,
    $46, $D7, $79, $5A, $8E, $5C, $79, $1C, $F2, $73, $25, $7C, $72, $BC, $A6,
    $1D, $1F, $B9, $18, $5E, $AF, $D9, $D6, $7B, $77, $89, $9E, $11, $93, $36,
    $96, $BA, $51, $7E, $3A, $8F, $1C, $18, $BF, $A9, $88, $1B, $82, $04, $35,
    $B6, $78, $65, $A4, $83, $C9, $12, $45, $FD, $04, $7A, $49, $CA, $96, $3A,
    $09, $E3, $6A, $01, $04, $CA, $F0, $47, $C8, $4D, $DF, $DD, $58, $96, $75,
    $CC, $C8, $7F, $0E, $01, $2D, $6D, $48, $BA, $22, $22, $25, $C7, $7C, $4E,
    $C5, $F7, $DE, $D6, $B9, $58, $5C, $0B, $0C, $15, $5D, $55, $76, $24, $54,
    $FE, $5E, $0C, $1C, $2B, $A2, $6A, $01, $3A, $42, $0D, $BF, $F4, $6D, $CC,
    $0B, $60, $83, $60, $6B, $30, $14, $FB, $6B, $F5, $F3, $54, $7B, $AC, $D2,
    $6B, $02, $4F, $0D, $3E, $76, $6F, $9F, $1F, $7B, $10, $56, $88, $62, $40,
    $C6, $7A, $86, $31, $9D, $D2, $5F, $BC, $43, $2B, $55, $A1, $00, $22, $94,
    $F3, $F7, $25, $7B, $6C, $C9, $20, $6D, $A4, $A7, $A3, $0B, $78, $16, $69,
    $2B, $FB, $CE, $44, $11, $D3, $52, $ED, $BC, $E3, $41, $2E, $53, $8A, $7E,
    $5F, $F0, $EF, $9E, $47, $6E, $18, $4D, $B4, $EE, $C9, $8E, $7E, $33, $61,
    $0A, $B4, $0A, $39, $D6, $24, $63, $3C, $57, $95, $DA, $B6, $3B, $4C, $B2,
    $90, $AF, $B2, $AB, $87, $31, $4B, $1B, $FE, $C5, $42, $3A, $7F, $5C, $3B,
    $89, $1E, $AD, $30, $C4, $37, $E4, $94, $CA, $4A, $D3, $F3, $61, $E8, $95,
    $D2, $D3, $F9, $C7, $83, $B3, $CA, $D0, $2F, $5A, $AB, $D5, $54, $E4, $0C,
    $4F, $67, $BA, $64, $6E, $6A, $F3, $53, $B2, $94, $23, $BA, $64, $74, $50,
    $D1, $E5, $11, $2A, $6B, $F1, $00, $CC, $E7, $50, $75, $94, $8F, $92, $B8,
    $D6, $59, $D6, $97, $82, $4E, $B0, $AF, $05, $0F, $AC, $A9, $E5, $30, $18,
    $C8, $FB, $6E, $F5, $5F, $41, $F4, $49, $1B, $8E, $34, $36, $23, $C7, $01,
    $87, $80, $4A, $F4, $97, $61, $B6, $5A, $CA, $15, $3A, $13, $F6, $DB, $15,
    $33, $75, $E2, $06, $5D, $C5, $14, $B4, $04, $E0, $89, $64, $29, $C0, $C4,
    $84, $38, $B1, $04, $D3, $10, $6A, $60, $3C, $6B, $07, $72, $A8, $28, $63,
    $B6, $50, $2F, $71, $FD, $6D, $E2, $86, $8D, $34, $6E, $BB, $C4, $8D, $63,
    $DC, $64, $F1, $54, $83, $DD, $54, $81, $5C, $34, $10, $C5, $A7, $62, $38,
    $3F, $7F, $49, $C7, $72, $7A, $94, $7D, $E5, $A0, $2C, $86, $6C, $D4, $09,
    $1C, $A3, $2E, $96, $C9, $53, $E3, $EA, $05, $C1, $AC, $7E, $3F, $14, $DC,
    $44, $A8, $68, $85, $28, $3C, $30, $6F, $68, $DA, $F2, $AD, $54, $83, $98,
    $17, $59, $F9, $CC, $0E, $71, $59, $1D, $2A, $4D, $72, $EF, $D1, $E7, $ED,
    $67, $CD, $3C, $89, $DC, $69, $6C, $07, $80, $A5, $7D, $79, $38, $7B, $6F,
    $17, $F4, $65, $7B, $E0, $E9, $87, $8D, $83, $7A, $28, $6E, $7D, $CB, $71,
    $18, $50, $2E, $96, $31, $41, $F5, $D0, $B0, $15, $5D, $E3, $1C, $A5, $8C,
    $0D, $EB, $44, $E4, $94, $F6, $5E, $25, $59, $0A, $BD, $18, $1D, $F6, $5A,
    $BC, $6B, $C2, $B4, $40, $46, $CF, $08, $31, $13, $15, $E5, $A7, $65, $D1,
    $FC, $65, $9A, $37, $1F, $30, $6D, $1A, $D9, $D2, $F8, $88, $11, $86, $C1,
    $D4, $90, $CB, $CE, $96, $77, $D8, $DF, $84, $EC, $34, $3F, $3E, $8C, $20,
    $DD, $6C, $70, $DE, $8D, $CA, $CF, $04, $AD, $F7, $87, $64, $8B, $2E, $1C,
    $3C, $3D, $3A, $FC, $A6, $E0, $4D, $0C, $94, $59, $86, $CE, $C4, $17, $54,
    $B0, $71
  );
{$ENDIF}

//==============================================================================
// M_Random
//
// Which one is deterministic?
//
//==============================================================================
function M_Random: integer;
begin
  rndindex := (rndindex + 1) and $ff;
  result := rndtable[rndindex];
  {$IFDEF DEBUG}
  printf('M_Random(): %3d, tic=%10d, seed = %3d'#13#10, [Result, gametic, rndindex]);
  {$ENDIF}
end;

//==============================================================================
//
// P_Random
//
//==============================================================================
function P_Random: integer;
begin
  prndindex := (prndindex + 1) and $ff;
  result := rndtable[prndindex];
  {$IFDEF DEBUG}
  printf('P_Random(): %3d, tic=%10d, seed = %3d'#13#10, [Result, gametic, prndindex]);
  {$ENDIF}
end;

//==============================================================================
//
// N_Random
//
//==============================================================================
function N_Random: integer;
begin
  nrndindex := (nrndindex + 1) and $ff;
  result := rndtable[nrndindex];
  {$IFDEF DEBUG}
  printf('N_Random(): %3d, tic=%10d, seed = %3d'#13#10, [Result, gametic, nrndindex]);
  {$ENDIF}
end;

//==============================================================================
//
// I_Random
//
//==============================================================================
function I_Random: integer;
begin
  result := Random(256);
  {$IFDEF DEBUG}
  printf('I_Random(): %3d, tic=%10d'#13#10, [Result, gametic]);
  {$ENDIF}
end;

{$IFNDEF HEXEN}

//==============================================================================
//
// Sys_Random
//
//==============================================================================
function Sys_Random: integer;
begin
  sysrndindex := sysrndindex + sysrndseed + 1;
  if sysrndindex >= SYSRNDSIZE then
    sysrndindex := sysrndindex - SYSRNDSIZE;
  result := sysrndtable[sysrndindex];
end;
{$ENDIF}

//==============================================================================
//
// C_Random
//
//==============================================================================
function C_Random(var idx: integer): integer;
begin
  idx := (idx + 1) and $ff;
  result := rndtable[idx];
  {$IFDEF DEBUG}
  printf('C_Random(): %3d, tic=%10d, seed = %3d'#13#10, [Result, gametic, idx]);
  {$ENDIF}
end;

//==============================================================================
//
// P_RandomFromSeed
//
//==============================================================================
function P_RandomFromSeed(const seed: integer): integer;
begin
  result := rndtable[seed and $ff];
  {$IFDEF DEBUG}
  printf('P_RandomFromSeed(): %3d, tic=%10d, seed = %3d'#13#10, [Result, gametic, seed]);
  {$ENDIF}
end;

var
  stack: TIntegerStack;

//==============================================================================
//
// M_ClearRandom
//
//==============================================================================
procedure M_ClearRandom;
begin
  rndindex := 0;
  prndindex := 0;
  nrndindex := 0;
{$IFNDEF HEXEN}
  sysrndindex := 0;
{$ENDIF}
  stack.Clear;
end;

//==============================================================================
//
// P_SaveRandom
//
//==============================================================================
procedure P_SaveRandom;
begin
  stack.Push(prndindex);
end;

//==============================================================================
//
// P_RestoreRandom
//
//==============================================================================
procedure P_RestoreRandom;
begin
  if not stack.Pop(prndindex) then
    I_DevError('P_RestoreRandom(): Stack is empty!'#13#10);
end;

initialization
  stack := TIntegerStack.Create;
  Randomize;

finalization
  stack.Free;

end.

