# Making textures with noise functions

To modify the appearance of a bitmap, you can apply a noise effect to it, using
either the `noise()` method or the `perlinNoise()` methods. A noise effect can
be likened to the static that appears on an untuned television screen.

To apply a noise effect to a a bitmap, use the `noise()` method. This method
applies a random color value to pixels within a specified area of a bitmap
image.

This method accepts five parameters:

- `randomSeed` (int): The random seed number that determines the pattern.
  Despite its name, this number actually creates the same results if the same
  number is passed. To get a true random result, use the `Math.random()` method
  to pass a random number for this parameter.

- `low` (uint): This parameter refers to the lowest value to be generated for
  each pixel (0 to 255). The default value is 0. Setting this value lower
  results in a darker noise pattern, while setting it to a higher value results
  in a brighter pattern.

- `high` (uint): This parameter refers to the highest value to be generated for
  each pixel (0 to 255). The default value is 255. Setting this value lower
  results in a darker noise pattern, while setting it to a higher value results
  in a brighter pattern.

- `channelOptions` (uint): This parameter specifies to which color channel of
  the bitmap object the noise pattern will be applied. The number can be a
  combination of any of the four color channel ARGB values. The default value
  is 7.

- `grayScale` (Boolean): When set to `true`, this parameter applies the
  `randomSeed` value to the bitmap pixels, effectively washing all color out of
  the image. The alpha channel is not affected by this parameter. The default
  value is `false`.

The following example creates a bitmap image and applies a blue noise pattern to
it:

    package
    {
        import flash.display.Sprite;
        import flash.display.Bitmap;
        import flash.display.BitmapData;
        import flash.display.BitmapDataChannel;

        public class BitmapNoise1 extends Sprite
        {
            public function BitmapNoise1()
            {
                var myBitmap:BitmapData = new BitmapData(250, 250,false, 0xff000000);
                myBitmap.noise(500, 0, 255, BitmapDataChannel.BLUE,false);
                var image:Bitmap = new Bitmap(myBitmap);
                addChild(image);
            }
        }
    }

If you want to create a more organic-looking texture, use the `perlinNoise()`
method. The `perlinNoise()` method produces realistic, organic textures that are
ideal for smoke, clouds, water, fire, or even explosions.

Because it is generated by an algorithm, the `perlinNoise()` method uses less
memory than bitmap-based textures. However, it can still have an impact on
processor usage, slowing down your content and causing the screen to be redrawn
more slowly than the frame rate, especially on older computers. This is mainly
due to the floating-point calculations that need to occur to process the perlin
noise algorithms.

The method accepts nine parameters (the first six are required):

- `baseX` (Number): Determines the x (size) value of patterns created.

- `baseY` (Number): Determines the y (size) value of the patterns created.

- `numOctaves` (uint): Number of octaves or individual noise functions to
  combine to create this noise. Larger numbers of octaves create images with
  greater detail but also require more processing time.

- `randomSeed` (int): The random seed number works exactly the same way as it
  does in the `noise()` function. To get a true random result, use the
  `Math.random()` method to pass a random number for this parameter.

- `stitch` (Boolean): If set to `true`, this method attempts to stitch (or
  smooth) the transition edges of the image to create seamless textures for
  tiling as a bitmap fill.

- `fractalNoise` (Boolean): This parameter relates to the edges of the gradients
  being generated by the method. If set to `true`, the method generates fractal
  noise that smooths the edges of the effect. If set to `false`, it generates
  turbulence. An image with turbulence has visible discontinuities in the
  gradient that can make it better approximate sharper visual effects, like
  flames and ocean waves.

- `channelOptions` (uint): The `channelOptions` parameter works exactly the same
  way as it does in the `noise()` method. It specifies to which color channel
  (of the bitmap) the noise pattern is applied. The number can be a combination
  of any of the four color channel ARGB values. The default value is 7.

- `grayScale` (Boolean): The `grayScale` parameter works exactly the same way as
  it does in the `noise()` method. If set to `true`, it applies the `randomSeed`
  value to the bitmap pixels, effectively washing all color out of the image.
  The default value is `false`.

- `offsets` (Array): An array of points that correspond to x and y offsets for
  each octave. By manipulating the offset values, you can smoothly scroll the
  layers of the image. Each point in the offset array affects a specific octave
  noise function. The default value is `null.`

The following example creates a 150 x 150 pixel BitmapData object that calls the
`perlinNoise()` method to generate a green and blue cloud effect:

    package
    {
        import flash.display.Sprite;
        import flash.display.Bitmap;
        import flash.display.BitmapData;
        import flash.display.BitmapDataChannel;

        public class BitmapNoise2 extends Sprite
        {
            public function BitmapNoise2()
            {
                var myBitmapDataObject:BitmapData =
                    new BitmapData(150, 150, false, 0x00FF0000);

                var seed:Number = Math.floor(Math.random() * 100);
                var channels:uint = BitmapDataChannel.GREEN | BitmapDataChannel.BLUE
                myBitmapDataObject.perlinNoise(100, 80, 6, seed, false, true, channels, false, null);

                var myBitmap:Bitmap = new Bitmap(myBitmapDataObject);
                addChild(myBitmap);
            }
        }
    }
