<?php

/**
 * @package    Joomla.UnitTest
 *
 * @copyright  (C) 2019 Open Source Matters, Inc. <https://www.joomla.org>
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Tests\Unit\Libraries\Cms\Mail;

use Joomla\CMS\Mail\MailHelper;
use Joomla\Tests\Unit\UnitTestCase;

/**
 * Test class for JMailHelper.
 * Generated by PHPUnit on 2011-10-26 at 19:33:00.
 *
 * @package     Joomla.UnitTest
 * @subpackage  Mail
 * @since       1.7.0
 */
class MailHelperTest extends UnitTestCase
{
    /**
     * Test data for testCleanLine method
     *
     * @return  array
     *
     * @since   3.0.0
     */
    public function dataCleanLine(): array
    {
        return [
            ["test\n\nme\r\r", 'testme'],
            ["test%0Ame", 'testme'],
            ["test%0Dme", 'testme'],
        ];
    }

    /**
     * Test for the JMailHelper::cleanLine method.
     *
     * @param   string  $input     The input to clean
     * @param   string  $expected  The expected result
     *
     * @return  void
     *
     * @since   3.0.0
     *
     * @dataProvider  dataCleanLine
     */
    public function testCleanLine($input, $expected)
    {
        $this->assertEquals($expected, MailHelper::cleanLine($input));
    }

    /**
     * Test data for testCleanText method
     *
     * @return  array
     *
     * @since   3.0.0
     */
    public function dataCleanText(): array
    {
        return [
            ["test\nme", "test\nme"],
            ["test%0AconTenT-Type:me", 'testme'],
            ["test%0Dcontent-type:me", 'testme'],
            ["test\ncontent-type:me", 'testme'],
            ["test\n\ncontent-type:me", 'testme'],
            ["test\rcontent-type:me", 'testme'],
            ["test\r\rcontent-type:me", 'testme'],
            // @TODO Should this be included array("test\r\ncoNTent-tYPe:me", 'testme'),

            ["test%0Ato:me", 'testme'],
            ["test%0DTO:me", 'testme'],
            ["test\nTo:me", 'testme'],
            ["test\n\ntO:me", 'testme'],
            ["test\rto:me", 'testme'],
            ["test\r\rto:me", 'testme'],
            // @TODO Should this be included array("test\r\nto:me", 'testme'),

            ["test%0Acc:me", 'testme'],
            ["test%0DCC:me", 'testme'],
            ["test\nCc:me", 'testme'],
            ["test\n\ncC:me", 'testme'],
            ["test\rcc:me", 'testme'],
            ["test\r\rcc:me", 'testme'],
            // @TODO Should this be included array("test\r\ncc:me", 'testme'),

            ["test%0Abcc:me", 'testme'],
            ["test%0DBCC:me", 'testme'],
            ["test\nBCc:me", 'testme'],
            ["test\n\nbcC:me", 'testme'],
            ["test\rbcc:me", 'testme'],
            ["test\r\rbcc:me", 'testme'],
            // @TODO Should this be included array("test\r\nbcc:me", 'testme'),
        ];
    }

    /**
     * Test for the JMailHelper::cleanText method.
     *
     * @param   string  $input     The input to clean
     * @param   string  $expected  The expected result
     *
     * @return  void
     *
     * @since   3.0.0
     *
     * @dataProvider  dataCleanText
     */
    public function testCleanText($input, $expected)
    {
        $this->assertEquals($expected, MailHelper::cleanText($input));
    }

    /**
     * Test data for testCleanBody method
     *
     * @return  array
     *
     * @since   3.0.0
     */
    public function dataCleanBody(): array
    {
        return [
            ["testFrom: Foobar me", "test me"],
            ["testfrom: Foobar me", "testfrom: Foobar me"],
            ["testTo: Foobar me", "test me"],
            ["testto: Foobar me", "testto: Foobar me"],
            ["testCc: Foobar me", "test me"],
            ["testcc: Foobar me", "testcc: Foobar me"],
            ["testBcc: Foobar me", "test me"],
            ["testbcc: Foobar me", "testbcc: Foobar me"],
            ["testSubject: Foobar me", "test me"],
            ["testsubject: Foobar me", "testsubject: Foobar me"],
            ["testContent-type: Foobar me", "test me"],
            ["testcontent-type: Foobar me", "testcontent-type: Foobar me"],
            // @TODO should this be case sensitive
        ];
    }

    /**
     * Test for the JMailHelper::cleanBody method.
     *
     * @param   string  $input     The input to clean
     * @param   string  $expected  The expected result
     *
     * @return  void
     *
     * @since   3.0.0
     *
     * @dataProvider  dataCleanBody
     */
    public function testCleanBody($input, $expected)
    {
        $this->assertEquals($expected, MailHelper::cleanBody($input));
    }

    /**
     * Test data for testCleanSubject method
     *
     * @return  array
     *
     * @since   3.0.0
     */
    public function dataCleanSubject(): array
    {
        return [
            ["testFrom: Foobar me", "test me"],
            ["testfrom: Foobar me", "testfrom: Foobar me"],
            ["testTo: Foobar me", "test me"],
            ["testto: Foobar me", "testto: Foobar me"],
            ["testCc: Foobar me", "test me"],
            ["testcc: Foobar me", "testcc: Foobar me"],
            ["testBcc: Foobar me", "test me"],
            ["testbcc: Foobar me", "testbcc: Foobar me"],
            ["testContent-type: Foobar me", "test me"],
            ["testcontent-type: Foobar me", "testcontent-type: Foobar me"],
            // @TODO should this be case sensitive
        ];
    }

    /**
     * Test for the JMailHelper::cleanSubject method.
     *
     * @param   string  $input     The input to clean
     * @param   string  $expected  The expected result
     *
     * @return  void
     *
     * @since   3.0.0
     *
     * @dataProvider  dataCleanSubject
     */
    public function testCleanSubject($input, $expected)
    {
        $this->assertEquals($expected, MailHelper::cleanSubject($input));
    }

    /**
     * Test data for testCleanAddress method
     *
     * @return  array
     *
     * @since   3.0.0
     */
    public function dataCleanAddress(): array
    {
        return [
            ["testme", "testme"],
            ["test me", "test me"],
            ["test;me", "test;me"],
            ["test,me", "test,me"],
            ["test ;,me", false],
        ];
    }

    /**
     * Test for the JMailHelper::cleanAddress method.
     *
     * @param   string  $input     The input to clean
     * @param   string  $expected  The expected result
     *
     * @return  void
     *
     * @since   3.0.0
     *
     * @dataProvider  dataCleanAddress
     */
    public function testCleanAddress($input, $expected)
    {
        $this->assertEquals($expected, MailHelper::cleanAddress($input));
    }

    /**
     * Test data for testIsEmailAddress method
     *
     * @return  array
     *
     * @since   3.0.0
     */
    public function dataIsEmailAddress(): array
    {
        return [
            ["joe", false],
            ["joe@home", true],
            ["a@b.c", true],
            ["joe@home.com", true],
            ["joe.bob@home.com", true],
            ["joe-bob[at]home.com", false],
            ["joe@his.home.com", true],
            ["joe@his.home.place", true],
            ["joe@home.org", true],
            ["joe@joebob.name", true],
            ["joe.@bob.com", false],
            [".joe@bob.com", false],
            ["joe<>bob@bob.come", false],
            ["joe&bob@bob.com", true],
            ["joebob@bob.edu.co", true],
            ["~joe@bob.com", true],
            ["joe..bob@bob.com", false],
            ["joe$@bob.com", true],
            ["joe+bob@bob.com", true],
            ["o'reilly@there.com", true],
            ["o’reilly@there.com", false],
        ];
    }

    /**
     * Test for the JMailHelper::isEmailAddress method.
     *
     * @param   string  $input     The input to clean
     * @param   string  $expected  The expected result
     *
     * @return  void
     *
     * @since   3.0.0
     *
     * @dataProvider  dataIsEmailAddress
     */
    public function testIsEmailAddress($input, $expected)
    {
        $this->assertEquals($expected, MailHelper::isEmailAddress($input));
    }
}
