#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Define the size of a TAR block */
#define TAR_BLOCK_SIZE 512

/* Define the structure of a TAR header block */
typedef struct tar_header {
    char name[100];       /* Name of the file */
    char mode[8];
    char uid[8];
    char gid[8];
    char size[12];        /* Size of the file */
    char mtime[12];
    char chksum[8];
    char typeflag;
    char linkname[100];
    /* ... other header fields ... */
} tar_header_t;

/* Function to convert octal string to unsigned int */
unsigned int octal_to_uint(char *octal, int size) {
    unsigned int out = 0;
    int i = 0;
    while (i < size && octal[i]) {
        out = (out << 3) + (octal[i] - '0');
        i++;
    }
    return out;
}

/* Main function */
int main(int argc, char *argv[]) {
    if (argc != 2) {
        printf("Usage: %s <tarfile>\n", argv[0]);
        return 1;
    }

    FILE *tar_file = fopen(argv[1], "rb");
    if (!tar_file) {
        perror("Error opening file");
        return 1;
    }

    tar_header_t header;
    size_t bytes_read;

    while ((bytes_read = fread(&header, 1, sizeof(header), tar_file)) > 0) {
        if (bytes_read < sizeof(header)) {
            fprintf(stderr, "Incomplete header read.\n");
            break;
        }

        /* Check for two consecutive empty blocks indicating end of archive */
        if (header.name[0] == '\0') {
            fprintf(stderr, "End of archive.\n");
            break;
        }

        printf("File: %s\n", header.name);

        /* Calculate the size of the file */
        unsigned int size = octal_to_uint(header.size, sizeof(header.size));

        /* Calculate the number of blocks to skip */
        unsigned int blocks_to_skip = (size + TAR_BLOCK_SIZE - 1) / TAR_BLOCK_SIZE;

        /* Seek over the file's data blocks */
        fseek(tar_file, blocks_to_skip * TAR_BLOCK_SIZE, SEEK_CUR);
    }

    fclose(tar_file);
    return 0;
}
