var object;

require('./_prepare');

object = mod('object');

test('isBareObject', function() {
  object.isBareObject('a').should.equal(false);
  return object.isBareObject({
    'a': 'a'
  }).should.equal(true);
});

test('typeOf', function() {
  object.typeOf('s').should.equal('string');
  object.typeOf(0).should.equal('number');
  object.typeOf(false).should.equal('boolean');
  object.typeOf({}).should.equal('object');
  object.typeOf(arguments).should.equal('arguments');
  return object.typeOf([]).should.equal('array');
});

test('empty', function() {
  var o;
  o = {
    a: 1,
    b: 2
  };
  object.empty(o);
  o.should.not.have.property('a');
  return o.should.not.have.property('b');
});

test('fastEmpty', function() {
  var o;
  o = {
    a: 1,
    b: 2
  };
  object.fastEmpty(o);
  o.should.not.have.property('a');
  return o.should.not.have.property('b');
});

test('clone', function() {
  var o;
  object.clone([1])[0].should.equal(1);
  object.clone({
    a: 1
  }).a.should.equal(1);
  o = {
    a: 1
  };
  return object.clone(o).should.not.equal(o);
});

test('clone [include prototype]', function() {
  var C, a, b;
  C = (function() {
    function C(a) {
      this.a = a;
    }

    C.prototype.sayA = function() {
      return this.a + 'a';
    };

    return C;

  })();
  a = new C('a');
  a.sayA().should.equal('aa');
  b = object.clone(a, true);
  b.should.not.equal(a);
  b.constructor.should.equal(C);
  b.a.should.equal('a');
  b.a = 'a2';
  return b.sayA().should.equal('a2a');
});

test('clone [without prototype]', function() {
  var C, a, b;
  C = (function() {
    function C(a) {
      this.a = a;
    }

    C.prototype.sayA = function() {
      return this.a + 'a';
    };

    return C;

  })();
  a = new C('a');
  a.sayA().should.equal('aa');
  b = object.clone(a, false);
  return b.should.equal(a);
});

test('overrideOnto [basic]', function() {
  var onto, what;
  onto = {
    a: 'a',
    b: {
      c: 'c',
      d: {
        e: 'e'
      }
    }
  };
  what = {
    a: 'a2',
    b: {
      c: 'c2',
      d: {
        f: 'f2'
      }
    }
  };
  object.overrideOnto(onto, what);
  onto.a.should.equal('a2');
  onto.b.should.have.property('c');
  onto.b.c.should.equal('c2');
  onto.b.d.should.not.have.property('f');
  return onto.b.d.e.should.equal('e');
});

test('override', function() {
  var onto, onto2, what;
  onto = {
    a: 'a',
    b: {
      c: 'c',
      d: {
        e: 'e'
      }
    }
  };
  what = {
    a: 'a2',
    b: {
      c: 'c2',
      d: {
        f: 'f2'
      }
    }
  };
  onto2 = object.override(onto, what);
  onto2.a.should.equal('a2');
  onto2.b.should.have.property('c');
  onto2.b.c.should.equal('c2');
  onto2.b.d.should.not.have.property('f');
  onto2.b.d.e.should.equal('e');
  return onto.should.not.equal(onto2);
});

(function() {
  var onto, what;
  what = {
    a: 'a2',
    c: function() {},
    z: 'z',
    y: {
      a: 'a'
    }
  };
  onto = {
    a: 'a',
    b: 'b'
  };
  test('appendOnto [basic]', function() {
    object.appendOnto(onto, what);
    onto.a.should.equal('a2');
    onto.b.should.equal('b');
    return onto.z.should.equal('z');
  });
  test("appendOnto [shallow copies instances]", function() {
    onto.c.should.be["instanceof"](Function);
    return onto.c.should.equal(what.c);
  });
  return test("appendOnto [clones objects]", function() {
    onto.should.have.property('y');
    onto.y.a.should.equal('a');
    return onto.y.should.not.equal(what.y);
  });
})();

test('groupProps', function() {
  var groups, obj;
  obj = {
    a1: '1',
    a2: '2',
    b1: '1',
    b2: '2',
    c1: '1',
    c2: '2',
    rest1: '1',
    rest2: '2'
  };
  groups = object.groupProps(obj, {
    a: ['a1', 'a2'],
    b: [/^b[0-9]+$/],
    c: function(key) {
      return key[0] === 'c';
    }
  });
  groups.a.should.have.property('a1');
  groups.a.a1.should.equal('1');
  groups.a.should.have.property('a2');
  groups.b.should.have.property('b1');
  groups.b.should.have.property('b2');
  groups.c.should.have.property('c1');
  groups.c.should.have.property('c2');
  groups.rest.should.have.property('rest1');
  groups.rest.should.have.property('rest1');
  return groups.rest.should.not.have.property('c1');
});

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib2JqZWN0LmpzIiwic291cmNlUm9vdCI6Ii4uXFwuLiIsInNvdXJjZXMiOlsiY29mZmVlXFx0ZXN0XFxvYmplY3QuY29mZmVlIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLElBQUEsTUFBQTs7QVFBQSxPUUFBLENRQVEsWVFBUixDQUFBLENBQUE7O0FRRUEsTVFBQSxHUUFTLEdRQUEsQ1FBSSxRUUFKLENBRlQsQ0FBQTs7QVFJQSxJUUFBLENRQUssY1FBTCxFUUFxQixTUUFBLEdRQUE7QVFFcEIsRVFBQSxNUUFNLENRQUMsWVFBUCxDUUFvQixHUUFwQixDUUF3QixDUUFDLE1RQU0sQ1FBQyxLUUFoQyxDUUFzQyxLUUF0QyxDUUFBLENRQUE7U1FFQSxNUUFNLENRQUMsWVFBUCxDUUFvQjtBUUFBLElRQUMsR1FBQSxFUUFLLEdRQU47R1FBcEIsQ1FBK0IsQ1FBQyxNUUFNLENRQUMsS1FBdkMsQ1FBNkMsSVFBN0MsRVFKb0I7QVFBQSxDUUFyQixDQUpBLENBQUE7O0FRVUEsSVFBQSxDUUFLLFFRQUwsRVFBZSxTUUFBLEdRQUE7QVFFZCxFUUFBLE1RQU0sQ1FBQyxNUUFQLENRQWMsR1FBZCxDUUFrQixDUUFDLE1RQU0sQ1FBQyxLUUExQixDUUFnQyxRUUFoQyxDUUFBLENRQUE7QVFBQSxFUUNBLE1RQU0sQ1FBQyxNUUFQLENRQWMsQ1FBZCxDUUFnQixDUUFDLE1RQU0sQ1FBQyxLUUF4QixDUUE4QixRUUE5QixDUURBLENRQUE7QVFBQSxFUUVBLE1RQU0sQ1FBQyxNUUFQLENRQWMsS1FBZCxDUUFvQixDUUFDLE1RQU0sQ1FBQyxLUUE1QixDUUFrQyxTUUFsQyxDUUZBLENRQUE7QVFBQSxFUUdBLE1RQU0sQ1FBQyxNUUFQLENRQWMsRVFBZCxDUUFpQixDUUFDLE1RQU0sQ1FBQyxLUUF6QixDUUErQixRUUEvQixDUUhBLENRQUE7QVFBQSxFUUlBLE1RQU0sQ1FBQyxNUUFQLENRQWMsU1FBZCxDUUF3QixDUUFDLE1RQU0sQ1FBQyxLUUFoQyxDUUFzQyxXUUF0QyxDUUpBLENRQUE7U1FLQSxNUUFNLENRQUMsTVFBUCxDUUFjLEVRQWQsQ1FBaUIsQ1FBQyxNUUFNLENRQUMsS1FBekIsQ1FBK0IsT1FBL0IsRVFQYztBUUFBLENRQWYsQ0FWQSxDQUFBOztBUW1CQSxJUUFBLENRQUssT1FBTCxFUUFjLFNRQUEsR1FBQTtBUUViLE1RQUEsQ1FBQTtBUUFBLEVRQUEsQ1FBQSxHUUVDO0FRQUEsSVFBQSxDUUFBLEVRQUcsQ1FBSDtBUUFBLElRQ0EsQ1FBQSxFUUFHLENRREg7R1FGRCxDUUFBO0FRQUEsRVFNQSxNUUFNLENRQUMsS1FBUCxDUUFhLENRQWIsQ1FOQSxDUUFBO0FRQUEsRVFRQSxDUUFDLENRQUMsTVFBTSxDUUFDLEdRQUcsQ1FBQyxJUUFJLENRQUMsUVFBbEIsQ1FBMkIsR1FBM0IsQ1FSQSxDUUFBO1NRU0EsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxHUUFHLENRQUMsSVFBSSxDUUFDLFFRQWxCLENRQTJCLEdRQTNCLEVRWGE7QVFBQSxDUUFkLENBbkJBLENBQUE7O0FRZ0NBLElRQUEsQ1FBSyxXUUFMLEVRQWtCLFNRQUEsR1FBQTtBUUVqQixNUUFBLENRQUE7QVFBQSxFUUFBLENRQUEsR1FDQztBUUFBLElRQUEsQ1FBQSxFUUFHLENRQUg7QVFBQSxJUUNBLENRQUEsRVFBRyxDUURIO0dRREQsQ1FBQTtBUUFBLEVRS0EsTVFBTSxDUUFDLFNRQVAsQ1FBaUIsQ1FBakIsQ1FMQSxDUUFBO0FRQUEsRVFPQSxDUUFDLENRQUMsTVFBTSxDUUFDLEdRQUcsQ1FBQyxJUUFJLENRQUMsUVFBbEIsQ1FBMkIsR1FBM0IsQ1FQQSxDUUFBO1NRUUEsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxHUUFHLENRQUMsSVFBSSxDUUFDLFFRQWxCLENRQTJCLEdRQTNCLEVRVmlCO0FRQUEsQ1FBbEIsQ0FoQ0EsQ0FBQTs7QVE0Q0EsSVFBQSxDUUFLLE9RQUwsRVFBYyxTUUFBLEdRQUE7QVFFYixNUUFBLENRQUE7QVFBQSxFUUFBLE1RQU0sQ1FBQyxLUUFQLENRQWEsQ1FBQyxDUUFELENRQWIsQ1FBa0IsQ1FBQSxDUUFBLENRQUUsQ1FBQyxNUUFNLENRQUMsS1FBNUIsQ1FBa0MsQ1FBbEMsQ1FBQSxDUUFBO0FRQUEsRVFDQSxNUUFNLENRQUMsS1FBUCxDUUFhO0FRQUEsSVFBQyxDUUFBLEVRQUUsQ1FBSDtHUUFiLENRQW1CLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxLUUE3QixDUUFtQyxDUUFuQyxDUURBLENRQUE7QVFBQSxFUUdBLENRQUEsR1FBSTtBUUFBLElRQUMsQ1FBQSxFUUFHLENRQUo7R1FISixDUUFBO1NRS0EsTVFBTSxDUUFDLEtRQVAsQ1FBYSxDUUFiLENRQWUsQ1FBQyxNUUFNLENRQUMsR1FBRyxDUUFDLEtRQTNCLENRQWlDLENRQWpDLEVRUGE7QVFBQSxDUUFkLENBNUNBLENBQUE7O0FRcURBLElRQUEsQ1FBSywyQlFBTCxFUUFrQyxTUUFBLEdRQUE7QVFFakMsTVFBQSxPUUFBO0FRQUEsRVFBTTtBUUVRLElRQUEsV1FBRSxDUUFGLEdRQUE7QVFBTSxNUUFMLElRQUMsQ1FBQSxJUUFBLENRQUksQ1FBTjtJUUFBLENRQWI7O0FRQUEsZ0JRRUEsSVFBQSxHUUFNLFNRQUEsR1FBQTthUUFHLElRQUMsQ1FBQSxDUUFELEdRQUssSVFBUjtJUUFBLENRRk4sQ1FBQTs7YVFBQTs7TVFGRCxDUUFBO0FRQUEsRVFNQSxDUUFBLEdRQVEsSVFBQSxDUUFBLENRQUUsR1FBRixDUU5SLENRQUE7QVFBQSxFUVFBLENRQUMsQ1FBQyxJUUFGLENRQUEsQ1FBUSxDUUFDLE1RQU0sQ1FBQyxLUUFoQixDUUFzQixJUUF0QixDUVJBLENRQUE7QVFBQSxFUVVBLENRQUEsR1FBSSxNUUFNLENRQUMsS1FBUCxDUUFhLENRQWIsRVFBZ0IsSVFBaEIsQ1FWSixDUUFBO0FRQUEsRVFZQSxDUUFDLENRQUMsTVFBTSxDUUFDLEdRQUcsQ1FBQyxLUUFiLENRQW1CLENRQW5CLENRWkEsQ1FBQTtBUUFBLEVRY0EsQ1FBQyxDUUFDLFdRQVcsQ1FBQyxNUUFNLENRQUMsS1FBckIsQ1FBMkIsQ1FBM0IsQ1FkQSxDUUFBO0FRQUEsRVFnQkEsQ1FBQyxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsS1FBWCxDUUFpQixHUUFqQixDUWhCQSxDUUFBO0FRQUEsRVFrQkEsQ1FBQyxDUUFDLENRQUYsR1FBTSxJUWxCTixDUUFBO1NRb0JBLENRQUMsQ1FBQyxJUUFGLENRQUEsQ1FBUSxDUUFDLE1RQU0sQ1FBQyxLUUFoQixDUUFzQixLUUF0QixFUXRCaUM7QVFBQSxDUUFsQyxDQXJEQSxDQUFBOztBUTZFQSxJUUFBLENRQUssMkJRQUwsRVFBa0MsU1FBQSxHUUFBO0FRRWpDLE1RQUEsT1FBQTtBUUFBLEVRQU07QVFFUSxJUUFBLFdRQUUsQ1FBRixHUUFBO0FRQU0sTVFBTCxJUUFDLENRQUEsSVFBQSxDUUFJLENRQU47SVFBQSxDUUFiOztBUUFBLGdCUUVBLElRQUEsR1FBTSxTUUFBLEdRQUE7YVFBRyxJUUFDLENRQUEsQ1FBRCxHUUFLLElRQVI7SVFBQSxDUUZOLENRQUE7O2FRQUE7O01RRkQsQ1FBQTtBUUFBLEVRTUEsQ1FBQSxHUUFRLElRQUEsQ1FBQSxDUUFFLEdRQUYsQ1FOUixDUUFBO0FRQUEsRVFRQSxDUUFDLENRQUMsSVFBRixDUUFBLENRQVEsQ1FBQyxNUUFNLENRQUMsS1FBaEIsQ1FBc0IsSVFBdEIsQ1FSQSxDUUFBO0FRQUEsRVFVQSxDUUFBLEdRQUksTVFBTSxDUUFDLEtRQVAsQ1FBYSxDUUFiLEVRQWdCLEtRQWhCLENRVkosQ1FBQTtTUVlBLENRQUMsQ1FBQyxNUUFNLENRQUMsS1FBVCxDUUFlLENRQWYsRVFkaUM7QVFBQSxDUUFsQyxDQTdFQSxDQUFBOztBUTZGQSxJUUFBLENRQUssc0JRQUwsRVFBNkIsU1FBQSxHUUFBO0FRRTVCLE1RQUEsVVFBQTtBUUFBLEVRQUEsSVFBQSxHUUNDO0FRQUEsSVFBQSxDUUFBLEVRQUcsR1FBSDtBUUFBLElRQ0EsQ1FBQSxFUUNDO0FRQUEsTVFBQSxDUUFBLEVRQUcsR1FBSDtBUUFBLE1RQ0EsQ1FBQSxFUUNDO0FRQUEsUVFBQSxDUUFBLEVRQUcsR1FBSDtPUUZEO0tRRkQ7R1FERCxDUUFBO0FRQUEsRVFPQSxJUUFBLEdRQ0M7QVFBQSxJUUFBLENRQUEsRVFBRyxJUUFIO0FRQUEsSVFDQSxDUUFBLEVRQ0M7QVFBQSxNUUFBLENRQUEsRVFBRyxJUUFIO0FRQUEsTVFDQSxDUUFBLEVRQ0M7QVFBQSxRUUFBLENRQUEsRVFBRyxJUUFIO09RRkQ7S1FGRDtHUVJELENRQUE7QVFBQSxFUWNBLE1RQU0sQ1FBQyxZUUFQLENRQW9CLElRQXBCLEVRQTBCLElRQTFCLENRZEEsQ1FBQTtBUUFBLEVRZ0JBLElRQUksQ1FBQyxDUUFDLENRQUMsTVFBTSxDUUFDLEtRQWQsQ1FBb0IsSVFBcEIsQ1FoQkEsQ1FBQTtBUUFBLEVRaUJBLElRQUksQ1FBQyxDUUFDLENRQUMsTVFBTSxDUUFDLElRQUksQ1FBQyxRUUFuQixDUUE0QixHUUE1QixDUWpCQSxDUUFBO0FRQUEsRVFrQkEsSVFBSSxDUUFDLENRQUMsQ1FBQyxDUUFDLENRQUMsTVFBTSxDUUFDLEtRQWhCLENRQXNCLElRQXRCLENRbEJBLENRQUE7QVFBQSxFUW1CQSxJUUFJLENRQUMsQ1FBQyxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsR1FBRyxDUUFDLElRQUksQ1FBQyxRUUF6QixDUUFrQyxHUUFsQyxDUW5CQSxDUUFBO1NRb0JBLElRQUksQ1FBQyxDUUFDLENRQUMsQ1FBQyxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsS1FBbEIsQ1FBd0IsR1FBeEIsRVF0QjRCO0FRQUEsQ1FBN0IsQ0E3RkEsQ0FBQTs7QVFxSEEsSVFBQSxDUUFLLFVRQUwsRVFBaUIsU1FBQSxHUUFBO0FRRWhCLE1RQUEsaUJRQUE7QVFBQSxFUUFBLElRQUEsR1FFQztBUUFBLElRQUEsQ1FBQSxFUUFHLEdRQUg7QVFBQSxJUUVBLENRQUEsRVFFQztBUUFBLE1RQUEsQ1FBQSxFUUFHLEdRQUg7QVFBQSxNUUVBLENRQUEsRVFFQztBUUFBLFFRQUEsQ1FBQSxFUUFHLEdRQUg7T1FKRDtLUUpEO0dRRkQsQ1FBQTtBUUFBLEVRWUEsSVFBQSxHUUVDO0FRQUEsSVFBQSxDUUFBLEVRQUcsSVFBSDtBUUFBLElRRUEsQ1FBQSxFUUVDO0FRQUEsTVFBQSxDUUFBLEVRQUcsSVFBSDtBUUFBLE1RRUEsQ1FBQSxFUUVDO0FRQUEsUVFBQSxDUUFBLEVRQUcsSVFBSDtPUUpEO0tRSkQ7R1FkRCxDUUFBO0FRQUEsRVF5QkEsS1FBQSxHUUFRLE1RQU0sQ1FBQyxRUUFQLENRQWdCLElRQWhCLEVRQXNCLElRQXRCLENRekJSLENRQUE7QVFBQSxFUTJCQSxLUUFLLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxLUUFmLENRQXFCLElRQXJCLENRM0JBLENRQUE7QVFBQSxFUTRCQSxLUUFLLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxJUUFJLENRQUMsUVFBcEIsQ1FBNkIsR1FBN0IsQ1E1QkEsQ1FBQTtBUUFBLEVRNkJBLEtRQUssQ1FBQyxDUUFDLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxLUUFqQixDUUF1QixJUUF2QixDUTdCQSxDUUFBO0FRQUEsRVE4QkEsS1FBSyxDUUFDLENRQUMsQ1FBQyxDUUFDLENRQUMsTVFBTSxDUUFDLEdRQUcsQ1FBQyxJUUFJLENRQUMsUVFBMUIsQ1FBbUMsR1FBbkMsQ1E5QkEsQ1FBQTtBUUFBLEVRK0JBLEtRQUssQ1FBQyxDUUFDLENRQUMsQ1FBQyxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsS1FBbkIsQ1FBeUIsR1FBekIsQ1EvQkEsQ1FBQTtTUWlDQSxJUUFJLENRQUMsTVFBTSxDUUFDLEdRQUcsQ1FBQyxLUUFoQixDUUFzQixLUUF0QixFUW5DZ0I7QVFBQSxDUUFqQixDQXJIQSxDQUFBOztBUTBKRyxDUUFBLFNRQUEsR1FBQTtBUUVGLE1RQUEsVVFBQTtBUUFBLEVRQUEsSVFBQSxHUUVDO0FRQUEsSVFBQSxDUUFBLEVRQUcsSVFBSDtBUUFBLElRRUEsQ1FBQSxFUUFHLFNRQUEsR1FBQSxDUUZIO0FRQUEsSVFJQSxDUUFBLEVRQUcsR1FKSDtBUUFBLElRTUEsQ1FBQSxFUUVDO0FRQUEsTVFBQSxDUUFBLEVRQUcsR1FBSDtLUVJEO0dRRkQsQ1FBQTtBUUFBLEVRWUEsSVFBQSxHUUVDO0FRQUEsSVFBQSxDUUFBLEVRQUcsR1FBSDtBUUFBLElRRUEsQ1FBQSxFUUFHLEdRRkg7R1FkRCxDUUFBO0FRQUEsRVFrQkEsSVFBQSxDUUFLLG9CUUFMLEVRQTJCLFNRQUEsR1FBQTtBUUUxQixJUUFBLE1RQU0sQ1FBQyxVUUFQLENRQWtCLElRQWxCLEVRQXdCLElRQXhCLENRQUEsQ1FBQTtBUUFBLElRRUEsSVFBSSxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsS1FBZCxDUUFvQixJUUFwQixDUUZBLENRQUE7QVFBQSxJUUdBLElRQUksQ1FBQyxDUUFDLENRQUMsTVFBTSxDUUFDLEtRQWQsQ1FBb0IsR1FBcEIsQ1FIQSxDUUFBO1dRSUEsSVFBSSxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsS1FBZCxDUUFvQixHUUFwQixFUU4wQjtFUUFBLENRQTNCLENRbEJBLENRQUE7QVFBQSxFUTBCQSxJUUFBLENRQUssdUNRQUwsRVFBOEMsU1FBQSxHUUFBO0FRRTdDLElRQUEsSVFBSSxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsRVFBRSxDUUFDLFlRQUQsQ1FBaEIsQ1FBNEIsUVFBNUIsQ1FBQSxDUUFBO1dRQ0EsSVFBSSxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsS1FBZCxDUUFvQixJUUFJLENRQUMsQ1FBekIsRVFINkM7RVFBQSxDUUE5QyxDUTFCQSxDUUFBO1NRZ0NBLElRQUEsQ1FBSyw2QlFBTCxFUUFvQyxTUUFBLEdRQUE7QVFFbkMsSVFBQSxJUUFJLENRQUMsTVFBTSxDUUFDLElRQUksQ1FBQyxRUUFqQixDUUEwQixHUUExQixDUUFBLENRQUE7QVFBQSxJUUNBLElRQUksQ1FBQyxDUUFDLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxLUUFoQixDUUFzQixHUUF0QixDUURBLENRQUE7V1FFQSxJUUFJLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxHUUFHLENRQUMsS1FBbEIsQ1FBd0IsSVFBSSxDUUFDLENRQTdCLEVRSm1DO0VRQUEsQ1FBcEMsRVFsQ0U7QVFBQSxDUUFBLENRQUgsQ1FBQSxDQTFKQSxDQUFBOztBUWtNQSxJUUFBLENRQUssWVFBTCxFUUFtQixTUUFBLEdRQUE7QVFFbEIsTVFBQSxXUUFBO0FRQUEsRVFBQSxHUUFBLEdRRUM7QVFBQSxJUUFBLEVRQUEsRVFBSSxHUUFKO0FRQUEsSVFDQSxFUUFBLEVRQUksR1FESjtBUUFBLElRR0EsRVFBQSxFUUFJLEdRSEo7QVFBQSxJUUlBLEVRQUEsRVFBSSxHUUpKO0FRQUEsSVFNQSxFUUFBLEVRQUksR1FOSjtBUUFBLElRT0EsRVFBQSxFUUFJLEdRUEo7QVFBQSxJUVNBLEtRQUEsRVFBTyxHUVRQO0FRQUEsSVFVQSxLUUFBLEVRQU8sR1FWUDtHUUZELENRQUE7QVFBQSxFUWNBLE1RQUEsR1FBUyxNUUFNLENRQUMsVVFBUCxDUUFrQixHUUFsQixFUUVSO0FRQUEsSVFBQSxDUUFBLEVRQUcsQ1FBQyxJUUFELEVRQU8sSVFBUCxDUUFIO0FRQUEsSVFFQSxDUUFBLEVRQUcsQ1FBQyxXUUFELENRRkg7QVFBQSxJUUlBLENRQUEsRVFBRyxTUUFDLEdRQUQsR1FBQTthUUFTLEdRQUksQ1FBQSxDUUFBLENRQUosS1FBVSxJUUFuQjtJUUFBLENRSkg7R1FGUSxDUWRULENRQUE7QVFBQSxFUXNCQSxNUUFNLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxJUUFJLENRQUMsUVFBckIsQ1FBOEIsSVFBOUIsQ1F0QkEsQ1FBQTtBUUFBLEVRdUJBLE1RQU0sQ1FBQyxDUUFDLENRQUMsRVFBRSxDUUFDLE1RQU0sQ1FBQyxLUUFuQixDUUF5QixHUUF6QixDUXZCQSxDUUFBO0FRQUEsRVF5QkEsTVFBTSxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsSVFBSSxDUUFDLFFRQXJCLENRQThCLElRQTlCLENRekJBLENRQUE7QVFBQSxFUTJCQSxNUUFNLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxJUUFJLENRQUMsUVFBckIsQ1FBOEIsSVFBOUIsQ1EzQkEsQ1FBQTtBUUFBLEVRNEJBLE1RQU0sQ1FBQyxDUUFDLENRQUMsTVFBTSxDUUFDLElRQUksQ1FBQyxRUUFyQixDUUE4QixJUUE5QixDUTVCQSxDUUFBO0FRQUEsRVE4QkEsTVFBTSxDUUFDLENRQUMsQ1FBQyxNUUFNLENRQUMsSVFBSSxDUUFDLFFRQXJCLENRQThCLElRQTlCLENROUJBLENRQUE7QVFBQSxFUStCQSxNUUFNLENRQUMsQ1FBQyxDUUFDLE1RQU0sQ1FBQyxJUUFJLENRQUMsUVFBckIsQ1FBOEIsSVFBOUIsQ1EvQkEsQ1FBQTtBUUFBLEVRaUNBLE1RQU0sQ1FBQyxJUUFJLENRQUMsTVFBTSxDUUFDLElRQUksQ1FBQyxRUUF4QixDUUFpQyxPUUFqQyxDUWpDQSxDUUFBO0FRQUEsRVFrQ0EsTVFBTSxDUUFDLElRQUksQ1FBQyxNUUFNLENRQUMsSVFBSSxDUUFDLFFRQXhCLENRQWlDLE9RQWpDLENRbENBLENRQUE7U1FvQ0EsTVFBTSxDUUFDLElRQUksQ1FBQyxNUUFNLENRQUMsR1FBRyxDUUFDLElRQUksQ1FBQyxRUUE1QixDUUFxQyxJUUFyQyxFUXRDa0I7QVFBQSxDUUFuQixDQWxNQSxDQUFBIiwic291cmNlc0NvbnRlbnQiOlsibW9kdWxlLmV4cG9ydHMgPSBhcnJheSA9XG5cblx0IyMjXG5cdFRyaWVzIHRvIHR1cm4gYW55dGhpbmcgaW50byBhbiBhcnJheS5cblx0IyMjXG5cdGZyb206IChyKSAtPlxuXG5cdFx0QXJyYXk6OnNsaWNlLmNhbGwgclxuXG5cdCMjI1xuXHRDbG9uZSBvZiBhbiBhcnJheS4gUHJvcGVydGllcyB3aWxsIGJlIHNoYWxsb3cgY29waWVzLlxuXHQjIyNcblx0c2ltcGxlQ2xvbmU6IChhKSAtPlxuXG5cdFx0YS5zbGljZSAwXG5cblx0c2hhbGxvd0VxdWFsOiAoYTEsIGEyKSAtPlxuXG5cdFx0cmV0dXJuIG5vIHVubGVzcyBBcnJheS5pc0FycmF5KGExKSBhbmQgQXJyYXkuaXNBcnJheShhMikgYW5kIGExLmxlbmd0aCBpcyBhMi5sZW5ndGhcblxuXHRcdGZvciB2YWwsIGkgaW4gYTFcblxuXHRcdFx0cmV0dXJuIG5vIHVubGVzcyBhMltpXSBpcyB2YWxcblxuXHRcdHllc1xuXG5cdHBsdWNrOiAoYSwgaSkgLT5cblxuXHRcdHJldHVybiBhIGlmIGEubGVuZ3RoIDwgMVxuXG5cblx0XHRmb3IgdmFsdWUsIGluZGV4IGluIGFcblxuXHRcdFx0aWYgaW5kZXggPiBpXG5cblx0XHRcdFx0YVtpbmRleCAtIDFdID0gYVtpbmRleF1cblxuXHRcdGEubGVuZ3RoID0gYS5sZW5ndGggLSAxXG5cblx0XHRhXG5cblx0cGx1Y2tJdGVtOiAoYSwgaXRlbSkgLT5cblxuXHRcdHJldHVybiBhIGlmIGEubGVuZ3RoIDwgMVxuXG5cblx0XHRyZW1vdmVkID0gMFxuXG5cdFx0Zm9yIHZhbHVlLCBpbmRleCBpbiBhXG5cblx0XHRcdGlmIHZhbHVlIGlzIGl0ZW1cblxuXHRcdFx0XHRyZW1vdmVkKytcblxuXHRcdFx0XHRjb250aW51ZVxuXG5cdFx0XHRpZiByZW1vdmVkIGlzbnQgMFxuXG5cdFx0XHRcdGFbaW5kZXggLSByZW1vdmVkXSA9IGFbaW5kZXhdXG5cblx0XHRhLmxlbmd0aCA9IGEubGVuZ3RoIC0gcmVtb3ZlZCBpZiByZW1vdmVkID4gMFxuXG5cdFx0YVxuXG5cdHBsdWNrT25lSXRlbTogKGEsIGl0ZW0pIC0+XG5cblx0XHRyZXR1cm4gYSBpZiBhLmxlbmd0aCA8IDFcblxuXHRcdHJlYWNoZWQgPSBub1xuXG5cdFx0Zm9yIHZhbHVlLCBpbmRleCBpbiBhXG5cblx0XHRcdGlmIG5vdCByZWFjaGVkXG5cblx0XHRcdFx0aWYgdmFsdWUgaXMgaXRlbVxuXG5cdFx0XHRcdFx0cmVhY2hlZCA9IHllc1xuXG5cdFx0XHRcdFx0Y29udGludWVcblxuXHRcdFx0ZWxzZVxuXG5cdFx0XHRcdGFbaW5kZXggLSAxXSA9IGFbaW5kZXhdXG5cblx0XHRhLmxlbmd0aCA9IGEubGVuZ3RoIC0gMSBpZiByZWFjaGVkXG5cblx0XHRhXG5cblx0cGx1Y2tCeUNhbGxiYWNrOiAoYSwgY2IpIC0+XG5cblx0XHRyZXR1cm4gYSBpZiBhLmxlbmd0aCA8IDFcblxuXHRcdHJlbW92ZWQgPSAwXG5cblx0XHRmb3IgdmFsdWUsIGluZGV4IGluIGFcblxuXHRcdFx0aWYgY2IgdmFsdWUsIGluZGV4XG5cblx0XHRcdFx0cmVtb3ZlZCsrXG5cblx0XHRcdFx0Y29udGludWVcblxuXHRcdFx0aWYgcmVtb3ZlZCBpc250IDBcblxuXHRcdFx0XHRhW2luZGV4IC0gcmVtb3ZlZF0gPSBhW2luZGV4XVxuXG5cdFx0aWYgcmVtb3ZlZCA+IDBcblxuXHRcdFx0YS5sZW5ndGggPSBhLmxlbmd0aCAtIHJlbW92ZWRcblxuXHRcdGFcblxuXHRwbHVja011bHRpcGxlOiAoYXJyYXksIGluZGV4ZXNUb1JlbW92ZSkgLT5cblxuXHRcdHJldHVybiBhcnJheSBpZiBhcnJheS5sZW5ndGggPCAxXG5cblx0XHRyZW1vdmVkU29GYXIgPSAwXG5cblx0XHRpbmRleGVzVG9SZW1vdmUuc29ydCgpXG5cblx0XHRmb3IgaSBpbiBpbmRleGVzVG9SZW1vdmVcblxuXHRcdFx0QHBsdWNrIGFycmF5LCBpIC0gcmVtb3ZlZFNvRmFyXG5cblx0XHRcdHJlbW92ZWRTb0ZhcisrXG5cblx0XHRhcnJheVxuXG5cdGluamVjdEJ5Q2FsbGJhY2s6IChhLCB0b0luamVjdCwgc2hvdWxkSW5qZWN0KSAtPlxuXG5cdFx0dmFsQSA9IG51bGxcblxuXHRcdHZhbEIgPSBudWxsXG5cblx0XHRsZW4gPSBhLmxlbmd0aFxuXG5cdFx0aWYgbGVuIDwgMVxuXG5cdFx0XHRhLnB1c2ggdG9JbmplY3RcblxuXHRcdFx0cmV0dXJuIGFcblxuXG5cdFx0Zm9yIHZhbCwgaSBpbiBhXG5cblx0XHRcdHZhbEEgPSB2YWxCXG5cblx0XHRcdHZhbEIgPSB2YWxcblxuXHRcdFx0aWYgc2hvdWxkSW5qZWN0IHZhbEEsIHZhbEIsIHRvSW5qZWN0XG5cblx0XHRcdFx0cmV0dXJuIGEuc3BsaWNlIGksIDAsIHRvSW5qZWN0XG5cblx0XHRhLnB1c2ggdG9JbmplY3RcblxuXHRcdGFcblxuXHRpbmplY3RJbkluZGV4OiAoYSwgaW5kZXgsIHRvSW5qZWN0KSAtPlxuXG5cdFx0bGVuID0gYS5sZW5ndGhcblxuXHRcdGkgPSBpbmRleFxuXG5cdFx0aWYgbGVuIDwgMVxuXG5cdFx0XHRhLnB1c2ggdG9JbmplY3RcblxuXHRcdFx0cmV0dXJuIGFcblxuXHRcdHRvUHV0ID0gdG9JbmplY3RcblxuXHRcdHRvUHV0TmV4dCA9IG51bGxcblxuXHRcdGBmb3IoOyBpIDw9IGxlbjsgaSsrKXtcblxuXHRcdFx0dG9QdXROZXh0ID0gYVtpXTtcblxuXHRcdFx0YVtpXSA9IHRvUHV0O1xuXG5cdFx0XHR0b1B1dCA9IHRvUHV0TmV4dDtcblxuXHRcdH1gXG5cblx0XHQjIGFbaV0gPSB0b1B1dFxuXG5cdFx0bnVsbCIsIm1vZHVsZS5leHBvcnRzID0gY2xhc3NpYyA9IHt9XG5cbiMgTGl0dGxlIGhlbHBlciBmb3IgbWl4aW5zIGZyb20gQ29mZmVlU2NyaXB0IEZBUSxcbiMgY291cnRlc3kgb2YgU2V0aGF1cnVzIChodHRwOi8vZ2l0aHViLmNvbS9zZXRoYXVydXMpXG5jbGFzc2ljLmltcGxlbWVudCA9IChtaXhpbnMuLi4sIGNsYXNzUmVmZXJlbmNlKSAtPlxuXG5cdGZvciBtaXhpbiBpbiBtaXhpbnNcblxuXHRcdGNsYXNzUHJvdG8gPSBjbGFzc1JlZmVyZW5jZTo6XG5cblx0XHRmb3IgbWVtYmVyIG9mIG1peGluOjpcblxuXHRcdFx0dW5sZXNzIE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3IgY2xhc3NQcm90bywgbWVtYmVyXG5cblx0XHRcdFx0ZGVzYyA9IE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3IgbWl4aW46OiwgbWVtYmVyXG5cblx0XHRcdFx0T2JqZWN0LmRlZmluZVByb3BlcnR5IGNsYXNzUHJvdG8sIG1lbWJlciwgZGVzY1xuXG5cdGNsYXNzUmVmZXJlbmNlXG5cbmNsYXNzaWMubWl4ID0gKG1peGlucy4uLiwgY2xhc3NSZWZlcmVuY2UpIC0+XG5cblx0Y2xhc3NQcm90byA9IGNsYXNzUmVmZXJlbmNlOjpcblxuXHRjbGFzc1JlZmVyZW5jZS5fX21peGluQ2xvbmVycyA9IFtdXG5cblx0Y2xhc3NSZWZlcmVuY2UuX19hcHBseUNsb25lcnNGb3IgPSAoaW5zdGFuY2UsIGFyZ3MgPSBudWxsKSAtPlxuXG5cdFx0Zm9yIGNsb25lciBpbiBjbGFzc1JlZmVyZW5jZS5fX21peGluQ2xvbmVyc1xuXG5cdFx0XHRjbG9uZXIuYXBwbHkgaW5zdGFuY2UsIGFyZ3NcblxuXHRcdHJldHVyblxuXG5cdGNsYXNzUmVmZXJlbmNlLl9fbWl4aW5Jbml0aWFsaXplcnMgPSBbXVxuXG5cdGNsYXNzUmVmZXJlbmNlLl9faW5pdE1peGluc0ZvciA9IChpbnN0YW5jZSwgYXJncyA9IG51bGwpIC0+XG5cblx0XHRmb3IgaW5pdGlhbGl6ZXIgaW4gY2xhc3NSZWZlcmVuY2UuX19taXhpbkluaXRpYWxpemVyc1xuXG5cdFx0XHRpbml0aWFsaXplci5hcHBseSBpbnN0YW5jZSwgYXJnc1xuXG5cdFx0cmV0dXJuXG5cblx0Y2xhc3NSZWZlcmVuY2UuX19taXhpblF1aXR0ZXJzID0gW11cblxuXHRjbGFzc1JlZmVyZW5jZS5fX2FwcGx5UXVpdHRlcnNGb3IgPSAoaW5zdGFuY2UsIGFyZ3MgPSBudWxsKSAtPlxuXG5cdFx0Zm9yIHF1aXR0ZXIgaW4gY2xhc3NSZWZlcmVuY2UuX19taXhpblF1aXR0ZXJzXG5cblx0XHRcdHF1aXR0ZXIuYXBwbHkgaW5zdGFuY2UsIGFyZ3NcblxuXHRcdHJldHVyblxuXG5cdGZvciBtaXhpbiBpbiBtaXhpbnNcblxuXHRcdHVubGVzcyBtaXhpbi5jb25zdHJ1Y3RvciBpbnN0YW5jZW9mIEZ1bmN0aW9uXG5cblx0XHRcdHRocm93IEVycm9yIFwiTWl4aW4gc2hvdWxkIGJlIGEgZnVuY3Rpb25cIlxuXG5cdFx0Zm9yIG1lbWJlciBvZiBtaXhpbjo6XG5cblx0XHRcdGlmIG1lbWJlci5zdWJzdHIoMCwgMTEpIGlzICdfX2luaXRNaXhpbidcblxuXHRcdFx0XHRjbGFzc1JlZmVyZW5jZS5fX21peGluSW5pdGlhbGl6ZXJzLnB1c2ggbWl4aW46OlttZW1iZXJdXG5cblx0XHRcdFx0Y29udGludWVcblxuXHRcdFx0ZWxzZSBpZiBtZW1iZXIuc3Vic3RyKDAsIDExKSBpcyAnX19jbG9uZXJGb3InXG5cblx0XHRcdFx0Y2xhc3NSZWZlcmVuY2UuX19taXhpbkNsb25lcnMucHVzaCBtaXhpbjo6W21lbWJlcl1cblxuXHRcdFx0XHRjb250aW51ZVxuXG5cdFx0XHRlbHNlIGlmIG1lbWJlci5zdWJzdHIoMCwgMTIpIGlzICdfX3F1aXR0ZXJGb3InXG5cblx0XHRcdFx0Y2xhc3NSZWZlcmVuY2UuX19taXhpblF1aXR0ZXJzLnB1c2ggbWl4aW46OlttZW1iZXJdXG5cblx0XHRcdFx0Y29udGludWVcblxuXHRcdFx0dW5sZXNzIE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3IgY2xhc3NQcm90bywgbWVtYmVyXG5cblx0XHRcdFx0ZGVzYyA9IE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3IgbWl4aW46OiwgbWVtYmVyXG5cblx0XHRcdFx0T2JqZWN0LmRlZmluZVByb3BlcnR5IGNsYXNzUHJvdG8sIG1lbWJlciwgZGVzY1xuXG5cdGNsYXNzUmVmZXJlbmNlIiwiYXJyYXkgPSByZXF1aXJlICcuL2FycmF5J1xyXG5cclxubW9kdWxlLmV4cG9ydHMgPSBjbGFzcyBFbWl0dGVyXHJcblxyXG5cdGNvbnN0cnVjdG9yOiAtPlxyXG5cclxuXHRcdEBfbGlzdGVuZXJzID0ge31cclxuXHJcblx0XHRAX2xpc3RlbmVyc0ZvckFueUV2ZW50ID0gW11cclxuXHJcblx0XHRAX2Rpc2FibGVkRW1pdHRlcnMgPSB7fVxyXG5cclxuXHRvbjogKGV2ZW50TmFtZSwgbGlzdGVuZXIpIC0+XHJcblxyXG5cdFx0dW5sZXNzIEBfbGlzdGVuZXJzW2V2ZW50TmFtZV0/XHJcblxyXG5cdFx0XHRAX2xpc3RlbmVyc1tldmVudE5hbWVdID0gW11cclxuXHJcblx0XHRAX2xpc3RlbmVyc1tldmVudE5hbWVdLnB1c2ggbGlzdGVuZXJcclxuXHJcblx0XHRAXHJcblxyXG5cdG9uY2U6IChldmVudE5hbWUsIGxpc3RlbmVyKSAtPlxyXG5cclxuXHRcdHJhbiA9IG5vXHJcblxyXG5cdFx0Y2IgPSA9PlxyXG5cclxuXHRcdFx0cmV0dXJuIGlmIHJhblxyXG5cclxuXHRcdFx0cmFuID0geWVzXHJcblxyXG5cdFx0XHRkbyBsaXN0ZW5lclxyXG5cclxuXHRcdFx0c2V0VGltZW91dCA9PlxyXG5cclxuXHRcdFx0XHRAcmVtb3ZlRXZlbnQgZXZlbnROYW1lLCBjYlxyXG5cclxuXHRcdFx0LCAwXHJcblxyXG5cdFx0QG9uIGV2ZW50TmFtZSwgY2JcclxuXHJcblx0XHRAXHJcblxyXG5cdG9uQW55RXZlbnQ6IChsaXN0ZW5lcikgLT5cclxuXHJcblx0XHRAX2xpc3RlbmVyc0ZvckFueUV2ZW50LnB1c2ggbGlzdGVuZXJcclxuXHJcblx0XHRAXHJcblxyXG5cdHJlbW92ZUV2ZW50OiAoZXZlbnROYW1lLCBsaXN0ZW5lcikgLT5cclxuXHJcblx0XHRyZXR1cm4gQCB1bmxlc3MgQF9saXN0ZW5lcnNbZXZlbnROYW1lXT9cclxuXHJcblx0XHRhcnJheS5wbHVja09uZUl0ZW0gQF9saXN0ZW5lcnNbZXZlbnROYW1lXSwgbGlzdGVuZXJcclxuXHJcblx0XHRAXHJcblxyXG5cdHJlbW92ZUxpc3RlbmVyczogKGV2ZW50TmFtZSkgLT5cclxuXHJcblx0XHRyZXR1cm4gQCB1bmxlc3MgQF9saXN0ZW5lcnNbZXZlbnROYW1lXT9cclxuXHJcblx0XHRAX2xpc3RlbmVyc1tldmVudE5hbWVdLmxlbmd0aCA9IDBcclxuXHJcblx0XHRAXHJcblxyXG5cdHJlbW92ZUFsbExpc3RlbmVyczogLT5cclxuXHJcblx0XHRmb3IgbmFtZSwgbGlzdGVuZXJzIG9mIEBfbGlzdGVuZXJzXHJcblxyXG5cdFx0XHRsaXN0ZW5lcnMubGVuZ3RoID0gMFxyXG5cclxuXHRcdEBcclxuXHJcblx0X2VtaXQ6IChldmVudE5hbWUsIGRhdGEpIC0+XHJcblxyXG5cdFx0Zm9yIGxpc3RlbmVyIGluIEBfbGlzdGVuZXJzRm9yQW55RXZlbnRcclxuXHJcblx0XHRcdGxpc3RlbmVyLmNhbGwgQCwgZGF0YSwgZXZlbnROYW1lXHJcblxyXG5cdFx0cmV0dXJuIHVubGVzcyBAX2xpc3RlbmVyc1tldmVudE5hbWVdP1xyXG5cclxuXHRcdGZvciBsaXN0ZW5lciBpbiBAX2xpc3RlbmVyc1tldmVudE5hbWVdXHJcblxyXG5cdFx0XHRsaXN0ZW5lci5jYWxsIEAsIGRhdGFcclxuXHJcblx0XHRyZXR1cm5cclxuXHJcblx0IyB0aGlzIG1ha2VzIHN1cmUgdGhhdCBhbGwgdGhlIGNhbGxzIHRvIHRoaXMgY2xhc3MncyBtZXRob2QgJ2ZuTmFtZSdcclxuXHQjIGFyZSB0aHJvdHRsZWRcclxuXHRfdGhyb3R0bGVFbWl0dGVyTWV0aG9kOiAoZm5OYW1lLCB0aW1lID0gMTAwMCkgLT5cclxuXHJcblx0XHRvcmlnaW5hbEZuID0gQFtmbk5hbWVdXHJcblxyXG5cdFx0aWYgdHlwZW9mIG9yaWdpbmFsRm4gaXNudCAnZnVuY3Rpb24nXHJcblxyXG5cdFx0XHR0aHJvdyBFcnJvciBcInRoaXMgY2xhc3MgZG9lcyBub3QgaGF2ZSBhIG1ldGhvZCBjYWxsZWQgJyN7Zm5OYW1lfSdcIlxyXG5cclxuXHRcdGxhc3RDYWxsQXJncyA9IG51bGxcclxuXHRcdHBlbmRpbmcgPSBub1xyXG5cdFx0dGltZXIgPSBudWxsXHJcblxyXG5cdFx0QFtmbk5hbWVdID0gPT5cclxuXHJcblx0XHRcdGxhc3RDYWxsQXJncyA9IGFyZ3VtZW50c1xyXG5cclxuXHRcdFx0ZG8gcGVuZFxyXG5cclxuXHRcdHBlbmQgPSA9PlxyXG5cclxuXHRcdFx0aWYgcGVuZGluZ1xyXG5cclxuXHRcdFx0XHRjbGVhclRpbWVvdXQgdGltZXJcclxuXHJcblx0XHRcdHRpbWVyID0gc2V0VGltZW91dCBydW5JdCwgdGltZVxyXG5cclxuXHRcdFx0cGVuZGluZyA9IHllc1xyXG5cclxuXHRcdHJ1bkl0ID0gPT5cclxuXHJcblx0XHRcdHBlbmRpbmcgPSBub1xyXG5cclxuXHRcdFx0b3JpZ2luYWxGbi5hcHBseSBALCBsYXN0Q2FsbEFyZ3NcclxuXHJcblx0X2Rpc2FibGVFbWl0dGVyOiAoZm5OYW1lKSAtPlxyXG5cclxuXHRcdGlmIEBfZGlzYWJsZWRFbWl0dGVyc1tmbk5hbWVdP1xyXG5cclxuXHRcdFx0dGhyb3cgRXJyb3IgXCIje2ZuTmFtZX0gaXMgYWxyZWFkeSBhIGRpc2FibGVkIGVtaXR0ZXJcIlxyXG5cclxuXHRcdEBfZGlzYWJsZWRFbWl0dGVyc1tmbk5hbWVdID0gQFtmbk5hbWVdXHJcblxyXG5cdFx0QFtmbk5hbWVdID0gLT5cclxuXHJcblx0X2VuYWJsZUVtaXR0ZXI6IChmbk5hbWUpIC0+XHJcblxyXG5cdFx0Zm4gPSBAX2Rpc2FibGVkRW1pdHRlcnNbZm5OYW1lXVxyXG5cclxuXHRcdHVubGVzcyBmbj9cclxuXHJcblx0XHRcdHRocm93IEVycm9yIFwiI3tmbk5hbWV9IGlzIG5vdCBhIGRpc2FibGVkIGVtaXR0ZXJcIlxyXG5cclxuXHRcdEBbZm5OYW1lXSA9IGZuXHJcblxyXG5cdFx0ZGVsZXRlIEBfZGlzYWJsZWRFbWl0dGVyc1tmbk5hbWVdIiwiX2NvbW1vbiA9IHJlcXVpcmUgJy4vX2NvbW1vbidcblxubW9kdWxlLmV4cG9ydHMgPSBvYmplY3QgPVxuXG5cdGlzQmFyZU9iamVjdDogX2NvbW1vbi5pc0JhcmVPYmplY3QuYmluZCBfY29tbW9uXG5cblx0IyMjXG5cdGlmIG9iamVjdCBpcyBhbiBpbnN0YW5jZSBvZiBhIGNsYXNzXG5cdCMjI1xuXHRpc0luc3RhbmNlOiAod2hhdCkgLT5cblxuXHRcdG5vdCBAaXNCYXJlT2JqZWN0IHdoYXRcblxuXHQjIyNcblx0QWxpYXMgdG8gX2NvbW1vbi50eXBlT2Zcblx0IyMjXG5cdHR5cGVPZjogX2NvbW1vbi50eXBlT2YuYmluZCBfY29tbW9uXG5cblx0IyMjXG5cdEFsaWFzIHRvIF9jb21tb24uY2xvbmVcblx0IyMjXG5cdGNsb25lOiBfY29tbW9uLmNsb25lLmJpbmQgX2NvbW1vblxuXG5cdCMjI1xuXHRFbXB0aWVzIGFuIG9iamVjdCBvZiBpdHMgcHJvcGVydGllcy5cblx0IyMjXG5cdGVtcHR5OiAobykgLT5cblxuXHRcdGZvciBwcm9wIG9mIG9cblxuXHRcdFx0ZGVsZXRlIG9bcHJvcF0gaWYgby5oYXNPd25Qcm9wZXJ0eSBwcm9wXG5cblx0XHRvXG5cblx0IyMjXG5cdEVtcHRpZXMgYW4gb2JqZWN0LiBEb2Vzbid0IGNoZWNrIGZvciBoYXNPd25Qcm9wZXJ0eSwgc28gaXQncyBhIHRpbnlcblx0Yml0IGZhc3Rlci4gVXNlIGl0IGZvciBwbGFpbiBvYmplY3RzLlxuXHQjIyNcblx0ZmFzdEVtcHR5OiAobykgLT5cblxuXHRcdGRlbGV0ZSBvW3Byb3BlcnR5XSBmb3IgcHJvcGVydHkgb2Ygb1xuXG5cdFx0b1xuXG5cdCMjI1xuXHRPdmVycmlkZXMgdmFsdWVzIGZvbXIgYG5ld1ZhbHVlc2Agb24gYGJhc2VgLCBhcyBsb25nIGFzIHRoZXlcblx0YWxyZWFkeSBleGlzdCBpbiBiYXNlLlxuXHQjIyNcblx0b3ZlcnJpZGVPbnRvOiAoYmFzZSwgbmV3VmFsdWVzKSAtPlxuXG5cdFx0cmV0dXJuIGJhc2UgaWYgbm90IEBpc0JhcmVPYmplY3QobmV3VmFsdWVzKSBvciBub3QgQGlzQmFyZU9iamVjdChiYXNlKVxuXG5cdFx0Zm9yIGtleSwgb2xkVmFsIG9mIGJhc2VcblxuXHRcdFx0bmV3VmFsID0gbmV3VmFsdWVzW2tleV1cblxuXHRcdFx0Y29udGludWUgaWYgbmV3VmFsIGlzIHVuZGVmaW5lZFxuXG5cdFx0XHRpZiB0eXBlb2YgbmV3VmFsIGlzbnQgJ29iamVjdCcgb3IgQGlzSW5zdGFuY2UgbmV3VmFsXG5cblx0XHRcdFx0YmFzZVtrZXldID0gQGNsb25lIG5ld1ZhbFxuXG5cdFx0XHQjIG5ld1ZhbCBpcyBhIHBsYWluIG9iamVjdFxuXHRcdFx0ZWxzZVxuXG5cdFx0XHRcdGlmIHR5cGVvZiBvbGRWYWwgaXNudCAnb2JqZWN0JyBvciBAaXNJbnN0YW5jZSBvbGRWYWxcblxuXHRcdFx0XHRcdGJhc2Vba2V5XSA9IEBjbG9uZSBuZXdWYWxcblxuXHRcdFx0XHRlbHNlXG5cblx0XHRcdFx0XHRAb3ZlcnJpZGVPbnRvIG9sZFZhbCwgbmV3VmFsXG5cdFx0YmFzZVxuXG5cdCMjI1xuXHRUYWtlcyBhIGNsb25lIG9mICdiYXNlJyBhbmQgcnVucyAjb3ZlcnJpZGVPbnRvIG9uIGl0XG5cdCMjI1xuXHRvdmVycmlkZTogKGJhc2UsIG5ld1ZhbHVlcykgLT5cblxuXHRcdEBvdmVycmlkZU9udG8gQGNsb25lKGJhc2UpLCBuZXdWYWx1ZXNcblxuXHRhcHBlbmQ6IChiYXNlLCB0b0FwcGVuZCkgLT5cblxuXHRcdEBhcHBlbmRPbnRvIEBjbG9uZShiYXNlKSwgdG9BcHBlbmRcblxuXHQjIERlZXAgYXBwZW5kcyB2YWx1ZXMgZnJvbSBgdG9BcHBlbmRgIHRvIGBiYXNlYFxuXHRhcHBlbmRPbnRvOiAoYmFzZSwgdG9BcHBlbmQpIC0+XG5cblx0XHRyZXR1cm4gYmFzZSBpZiBub3QgQGlzQmFyZU9iamVjdCh0b0FwcGVuZCkgb3Igbm90IEBpc0JhcmVPYmplY3QoYmFzZSlcblxuXHRcdGZvciBvd24ga2V5LCBuZXdWYWwgb2YgdG9BcHBlbmRcblxuXHRcdFx0Y29udGludWUgdW5sZXNzIG5ld1ZhbCBpc250IHVuZGVmaW5lZFxuXG5cdFx0XHRpZiB0eXBlb2YgbmV3VmFsIGlzbnQgJ29iamVjdCcgb3IgQGlzSW5zdGFuY2UgbmV3VmFsXG5cblx0XHRcdFx0YmFzZVtrZXldID0gbmV3VmFsXG5cblx0XHRcdGVsc2VcblxuXHRcdFx0XHQjIG5ld1ZhbCBpcyBhIGJhcmUgb2JqZWN0XG5cblx0XHRcdFx0b2xkVmFsID0gYmFzZVtrZXldXG5cblx0XHRcdFx0aWYgdHlwZW9mIG9sZFZhbCBpc250ICdvYmplY3QnIG9yIEBpc0luc3RhbmNlIG9sZFZhbFxuXG5cdFx0XHRcdFx0YmFzZVtrZXldID0gQGNsb25lIG5ld1ZhbFxuXG5cdFx0XHRcdGVsc2VcblxuXHRcdFx0XHRcdEBhcHBlbmRPbnRvIG9sZFZhbCwgbmV3VmFsXG5cblx0XHRiYXNlXG5cblx0IyBHcm91cHNcblx0Z3JvdXBQcm9wczogKG9iaiwgZ3JvdXBzKSAtPlxuXG5cdFx0Z3JvdXBlZCA9IHt9XG5cblx0XHRmb3IgbmFtZSwgZGVmcyBvZiBncm91cHNcblxuXHRcdFx0Z3JvdXBlZFtuYW1lXSA9IHt9XG5cblx0XHRncm91cGVkWydyZXN0J10gPSB7fVxuXG5cdFx0YHRvcDogLy9gXG5cdFx0Zm9yIGtleSwgdmFsIG9mIG9ialxuXG5cdFx0XHRzaG91bGRBZGQgPSBub1xuXG5cdFx0XHRmb3IgbmFtZSwgZGVmcyBvZiBncm91cHNcblxuXHRcdFx0XHR1bmxlc3MgQXJyYXkuaXNBcnJheSBkZWZzXG5cblx0XHRcdFx0XHRkZWZzID0gW2RlZnNdXG5cblx0XHRcdFx0Zm9yIGRlZiBpbiBkZWZzXG5cblx0XHRcdFx0XHRpZiB0eXBlb2YgZGVmIGlzICdzdHJpbmcnXG5cblx0XHRcdFx0XHRcdGlmIGtleSBpcyBkZWZcblxuXHRcdFx0XHRcdFx0XHRzaG91bGRBZGQgPSB5ZXNcblxuXHRcdFx0XHRcdGVsc2UgaWYgZGVmIGluc3RhbmNlb2YgUmVnRXhwXG5cblx0XHRcdFx0XHRcdGlmIGRlZi50ZXN0IGtleVxuXG5cdFx0XHRcdFx0XHRcdHNob3VsZEFkZCA9IHllc1xuXG5cdFx0XHRcdFx0ZWxzZSBpZiBkZWYgaW5zdGFuY2VvZiBGdW5jdGlvblxuXG5cdFx0XHRcdFx0XHRpZiBkZWYga2V5XG5cblx0XHRcdFx0XHRcdFx0c2hvdWxkQWRkID0geWVzXG5cblx0XHRcdFx0XHRlbHNlXG5cblx0XHRcdFx0XHRcdHRocm93IEVycm9yICdHcm91cCBkZWZpbml0aW9ucyBtdXN0IGVpdGhlclxuXHRcdFx0XHRcdFx0YmUgc3RyaW5ncywgcmVnZXhlcywgb3IgZnVuY3Rpb25zLidcblxuXHRcdFx0XHRcdGlmIHNob3VsZEFkZFxuXG5cdFx0XHRcdFx0XHRncm91cGVkW25hbWVdW2tleV0gPSB2YWxcblxuXHRcdFx0XHRcdFx0YGNvbnRpbnVlIHRvcGBcblxuXHRcdFx0Z3JvdXBlZFsncmVzdCddW2tleV0gPSB2YWxcblxuXHRcdGdyb3VwZWQiLCJtb2R1bGUuZXhwb3J0cyA9XHJcblxyXG5cdCMgcGFkcyBhIG51bWJlciB3aXRoIGxlYWRpbmcgemVyb2VzXHJcblx0I1xyXG5cdCMgaHR0cDovL3N0YWNrb3ZlcmZsb3cuY29tL2EvMTAwNzM3ODgvNjA3OTk3XHJcblx0cGFkOiAobiwgd2lkdGgsIHogPSAnMCcpIC0+XHJcblxyXG5cdFx0biA9IG4gKyAnJ1xyXG5cclxuXHRcdGlmIG4ubGVuZ3RoID49IHdpZHRoXHJcblxyXG5cdFx0XHRuXHJcblxyXG5cdFx0ZWxzZVxyXG5cclxuXHRcdFx0bmV3IEFycmF5KHdpZHRoIC0gbi5sZW5ndGggKyAxKS5qb2luKHopICsgbiIsIm1vZHVsZS5leHBvcnRzID0gdXRpbGEgPVxuXG5cdGFycmF5OiByZXF1aXJlICcuL2FycmF5J1xuXHRjbGFzc2ljOiByZXF1aXJlICcuL2NsYXNzaWMnXG5cdG9iamVjdDogcmVxdWlyZSAnLi9vYmplY3QnXG5cdHN0cmluZzogcmVxdWlyZSAnLi9zdHJpbmcnXG5cdEVtaXR0ZXI6IHJlcXVpcmUgJy4vRW1pdHRlciciLCJtb2R1bGUuZXhwb3J0cyA9IGNvbW1vbiA9XG5cblx0IyMjXG5cdENoZWNrcyB0byBzZWUgaWYgbyBpcyBhbiBvYmplY3QsIGFuZCBpdCBpc24ndCBhbiBpbnN0YW5jZVxuXHRvZiBzb21lIGNsYXNzLlxuXHQjIyNcblx0aXNCYXJlT2JqZWN0OiAobykgLT5cblxuXHRcdGlmIG8/IGFuZCBvLmNvbnN0cnVjdG9yIGlzIE9iamVjdFxuXG5cdFx0XHRyZXR1cm4gdHJ1ZVxuXG5cdFx0ZmFsc2VcblxuXHQjIyNcblx0UmV0dXJucyB0eXBlIG9mIGFuIG9iamVjdCwgaW5jbHVkaW5nOlxuXHR1bmRlZmluZWQsIG51bGwsIHN0cmluZywgbnVtYmVyLCBhcnJheSxcblx0YXJndW1lbnRzLCBlbGVtZW50LCB0ZXh0bm9kZSwgd2hpdGVzcGFjZSwgYW5kIG9iamVjdFxuXHQjIyNcblx0dHlwZU9mOiAoaXRlbSkgLT5cblxuXHRcdHJldHVybiAnbnVsbCcgaWYgaXRlbSBpcyBudWxsXG5cblx0XHRyZXR1cm4gdHlwZW9mIGl0ZW0gaWYgdHlwZW9mIGl0ZW0gaXNudCAnb2JqZWN0J1xuXG5cdFx0cmV0dXJuICdhcnJheScgaWYgQXJyYXkuaXNBcnJheSBpdGVtXG5cblx0XHQjIEZyb20gTW9vVG9vbHNcblx0XHQjIC0gZG8gd2UgZXZlbiBuZWVkIHRoaXM/XG5cdFx0aWYgaXRlbS5ub2RlTmFtZVxuXG5cdFx0XHRpZiBpdGVtLm5vZGVUeXBlIGlzIDEgdGhlbiByZXR1cm4gJ2VsZW1lbnQnXG5cdFx0XHRpZiBpdGVtLm5vZGVUeXBlIGlzIDMgdGhlbiByZXR1cm4gKC9cXFMvKS50ZXN0KGl0ZW0ubm9kZVZhbHVlKSA/ICd0ZXh0bm9kZScgOiAnd2hpdGVzcGFjZSdcblxuXHRcdGVsc2UgaWYgdHlwZW9mIGl0ZW0ubGVuZ3RoIGlzICdudW1iZXInXG5cblx0XHRcdGlmIGl0ZW0uY2FsbGVlIHRoZW4gcmV0dXJuICdhcmd1bWVudHMnXG5cblx0XHRyZXR1cm4gdHlwZW9mIGl0ZW1cblxuXHQjIERlZXAgY2xvbmUgb2YgYW55IHZhcmlhYmxlLlxuXHQjIEZyb20gTW9vVG9vbHNcblx0Y2xvbmU6IChpdGVtLCBpbmNsdWRlUHJvdG90eXBlID0gZmFsc2UpIC0+XG5cblx0XHRzd2l0Y2ggY29tbW9uLnR5cGVPZiBpdGVtXG5cblx0XHRcdHdoZW4gJ2FycmF5JyB0aGVuIHJldHVybiBjb21tb24uX2Nsb25lQXJyYXkgaXRlbSwgaW5jbHVkZVByb3RvdHlwZVxuXG5cdFx0XHR3aGVuICdvYmplY3QnIHRoZW4gcmV0dXJuIGNvbW1vbi5fY2xvbmVPYmplY3QgaXRlbSwgaW5jbHVkZVByb3RvdHlwZVxuXG5cdFx0XHRlbHNlIHJldHVybiBpdGVtXG5cblx0IyMjXG5cdERlZXAgY2xvbmUgb2YgYW4gb2JqZWN0LlxuXHRGcm9tIE1vb1Rvb2xzXG5cdCMjI1xuXHRfY2xvbmVPYmplY3Q6IChvLCBpbmNsdWRlUHJvdG90eXBlID0gZmFsc2UpIC0+XG5cblx0XHRpZiBjb21tb24uaXNCYXJlT2JqZWN0IG9cblxuXHRcdFx0Y2xvbmUgPSB7fVxuXG5cdFx0XHRmb3Iga2V5IG9mIG9cblxuXHRcdFx0XHRjbG9uZVtrZXldID0gY29tbW9uLmNsb25lIG9ba2V5XSwgaW5jbHVkZVByb3RvdHlwZVxuXG5cdFx0XHRyZXR1cm4gY2xvbmVcblxuXHRcdGVsc2VcblxuXHRcdFx0cmV0dXJuIG8gdW5sZXNzIGluY2x1ZGVQcm90b3R5cGVcblxuXHRcdFx0cmV0dXJuIG8gaWYgbyBpbnN0YW5jZW9mIEZ1bmN0aW9uXG5cblx0XHRcdGNsb25lID0gT2JqZWN0LmNyZWF0ZSBvLmNvbnN0cnVjdG9yLnByb3RvdHlwZVxuXG5cdFx0XHRmb3Iga2V5IG9mIG9cblxuXHRcdFx0XHRpZiBvLmhhc093blByb3BlcnR5IGtleVxuXG5cdFx0XHRcdFx0Y2xvbmVba2V5XSA9IGNvbW1vbi5jbG9uZSBvW2tleV0sIGluY2x1ZGVQcm90b3R5cGVcblxuXHRcdFx0Y2xvbmVcblxuXHQjIyNcblx0RGVlcCBjbG9uZSBvZiBhbiBhcnJheS5cblx0RnJvbSBNb29Ub29sc1xuXHQjIyNcblx0X2Nsb25lQXJyYXk6IChhLCBpbmNsdWRlUHJvdG90eXBlID0gZmFsc2UpIC0+XG5cblx0XHRpID0gYS5sZW5ndGhcblxuXHRcdGNsb25lID0gbmV3IEFycmF5IGlcblxuXHRcdHdoaWxlIGktLVxuXG5cdFx0XHRjbG9uZVtpXSA9IGNvbW1vbi5jbG9uZSBhW2ldLCBpbmNsdWRlUHJvdG90eXBlXG5cblx0XHRjbG9uZSIsInJlcXVpcmUgJy4vX3ByZXBhcmUnXG5cbmFycmF5ID0gbW9kICdhcnJheSdcblxudGVzdCAnZnJvbScsIC0+XG5cblx0YXJyYXkuZnJvbShbMV0pLnNob3VsZC5iZS5hbi5pbnN0YW5jZU9mIEFycmF5XG5cdGFycmF5LmZyb20oWzFdKVswXS5zaG91bGQuZXF1YWwgMVxuXG4jIHRlc3QgJ2Nsb25lJywgLT5cblxuIyBcdGEgPSBbMCwgMSwgMl1cblxuIyBcdGIgPSBhcnJheS5jbG9uZSBhXG5cbiMgXHRiWzBdLnNob3VsZC5lcXVhbCAwXG4jIFx0YlsxXS5zaG91bGQuZXF1YWwgMVxuXG4jIFx0YlswXSA9IDNcblxuIyBcdGFbMF0uc2hvdWxkLmVxdWFsIDBcblxudGVzdCAncGx1Y2snLCAtPlxuXG5cdGEgPSBbMCwgMSwgMiwgM11cblxuXHRhZnRlciA9IGFycmF5LnBsdWNrIGEsIDFcblxuXHRhZnRlci5sZW5ndGguc2hvdWxkLmVxdWFsIDNcblxuXHRhZnRlclswXS5zaG91bGQuZXF1YWwgMFxuXHRhZnRlclsxXS5zaG91bGQuZXF1YWwgMlxuXHRhZnRlclsyXS5zaG91bGQuZXF1YWwgM1xuXHRhZnRlci5zaG91bGQuZXF1YWwgYVxuXG50ZXN0ICdwbHVja011bHRpcGxlJywgLT5cblxuXHRhID0gWzAsIDEsIDIsIDMsIDQsIDUsIDZdXG5cblx0YXJyYXkucGx1Y2tNdWx0aXBsZSBhLCBbMCwgNCwgMiwgNl1cblxuXHRhLmxlbmd0aC5zaG91bGQuZXF1YWwgM1xuXHRhWzBdLnNob3VsZC5lcXVhbCAxXG5cdGFbMV0uc2hvdWxkLmVxdWFsIDNcblx0YVsyXS5zaG91bGQuZXF1YWwgNVxuXG50ZXN0ICdwbHVja0l0ZW0nLCAtPlxuXG5cdGEgPSBbMCwgMSwgMiwgMywgMiwgNCwgMl1cblxuXHRhcnJheS5wbHVja0l0ZW0gYSwgMlxuXG5cdGFbMF0uc2hvdWxkLmVxdWFsIDBcblx0YVsxXS5zaG91bGQuZXF1YWwgMVxuXHRhWzJdLnNob3VsZC5lcXVhbCAzXG5cdGFbM10uc2hvdWxkLmVxdWFsIDRcblxuXHRhcnJheS5wbHVja0l0ZW0oWzFdLCAyKS5sZW5ndGguc2hvdWxkLmVxdWFsIDFcblxuXG50ZXN0ICdwbHVja09uZUl0ZW0nLCAtPlxuXG5cdGEgPSBbMCwgMSwgMiwgMywgMiwgNCwgMl1cblxuXHRhcnJheS5wbHVja09uZUl0ZW0gYSwgMlxuXG5cdGFbMF0uc2hvdWxkLmVxdWFsIDBcblx0YVsxXS5zaG91bGQuZXF1YWwgMVxuXHRhWzJdLnNob3VsZC5lcXVhbCAzXG5cdGFbM10uc2hvdWxkLmVxdWFsIDJcblx0YVs0XS5zaG91bGQuZXF1YWwgNFxuXHRhWzVdLnNob3VsZC5lcXVhbCAyXG5cblx0YSA9IFsxLCAyXVxuXG5cdGFycmF5LnBsdWNrT25lSXRlbSBhLCAxXG5cblx0YS5sZW5ndGguc2hvdWxkLmVxdWFsIDFcblx0YVswXS5zaG91bGQuZXF1YWwgMlxuXG5cdGFycmF5LnBsdWNrT25lSXRlbShbXSwgMSkubGVuZ3RoLnNob3VsZC5lcXVhbCAwXG5cblx0YXJyYXkucGx1Y2tPbmVJdGVtKFsxXSwgMikubGVuZ3RoLnNob3VsZC5lcXVhbCAxXG5cbnRlc3QgJ3BsY3VrQnlDYWxsYmFjaycsIC0+XG5cblx0YSA9IFswLCAxLCAyLCAzXVxuXG5cdGFycmF5LnBsdWNrQnlDYWxsYmFjayBhLCAodmFsLCBpKSAtPlxuXG5cdFx0cmV0dXJuIHllcyBpZiB2YWwgaXMgMlxuXG5cdFx0cmV0dXJuIG5vXG5cblx0YVswXS5zaG91bGQuZXF1YWwgMFxuXHRhWzFdLnNob3VsZC5lcXVhbCAxXG5cdGFbMl0uc2hvdWxkLmVxdWFsIDNcblxudGVzdCAnaW5qZWN0QnlDYWxsYmFjaycsIC0+XG5cblx0c2hvdWxkSW5qZWN0ID0gKHZhbEEsIHZhbEIsIHRvSW5qZWN0KSAtPlxuXG5cdFx0dW5sZXNzIHZhbEE/XG5cblx0XHRcdHJldHVybiB5ZXMgaWYgdG9JbmplY3QgPD0gdmFsQlxuXG5cdFx0XHRyZXR1cm4gbm9cblxuXHRcdHVubGVzcyB2YWxCP1xuXG5cdFx0XHRyZXR1cm4geWVzIGlmIHZhbEEgPD0gdG9JbmplY3RcblxuXHRcdFx0cmV0dXJuIG5vXG5cblx0XHRyZXR1cm4geWVzIGlmIHZhbEEgPD0gdG9JbmplY3QgPD0gdmFsQlxuXG5cdFx0cmV0dXJuIG5vXG5cblx0YSA9IFswLjUsIDEsIDIuNSwgMi41LCAyLjc1LCAyLjc1LCAzXVxuXG5cdGFycmF5LmluamVjdEJ5Q2FsbGJhY2sgYSwgMCwgc2hvdWxkSW5qZWN0XG5cblx0YVswXS5zaG91bGQuZXF1YWwgMFxuXHRhWzFdLnNob3VsZC5lcXVhbCAwLjVcblx0YVs3XS5zaG91bGQuZXF1YWwgM1xuXG5cdGEgPSBbMC41LCAxLCAyLjUsIDIuNSwgMi43NSwgMi43NSwgM11cblxuXHRhcnJheS5pbmplY3RCeUNhbGxiYWNrIGEsIDIuNywgc2hvdWxkSW5qZWN0XG5cblx0YVswXS5zaG91bGQuZXF1YWwgMC41XG5cdGFbNF0uc2hvdWxkLmVxdWFsIDIuN1xuXHRhWzVdLnNob3VsZC5lcXVhbCAyLjc1XG5cdGFbN10uc2hvdWxkLmVxdWFsIDNcblxuXHRhID0gWzAuNSwgMSwgMi41LCAyLjUsIDIuNzUsIDIuNzUsIDNdXG5cblx0YXJyYXkuaW5qZWN0QnlDYWxsYmFjayBhLCAzLjIsIHNob3VsZEluamVjdFxuXG5cdGFbMF0uc2hvdWxkLmVxdWFsIDAuNVxuXHRhWzRdLnNob3VsZC5lcXVhbCAyLjc1XG5cdGFbNl0uc2hvdWxkLmVxdWFsIDNcblx0YVs3XS5zaG91bGQuZXF1YWwgMy4yIiwicmVxdWlyZSAnLi9fcHJlcGFyZSdcblxub2JqZWN0ID0gbW9kICdvYmplY3QnXG5cbnRlc3QgJ2lzQmFyZU9iamVjdCcsIC0+XG5cblx0b2JqZWN0LmlzQmFyZU9iamVjdCgnYScpLnNob3VsZC5lcXVhbCBmYWxzZVxuXG5cdG9iamVjdC5pc0JhcmVPYmplY3QoeydhJzogJ2EnfSkuc2hvdWxkLmVxdWFsIHRydWVcblxudGVzdCAndHlwZU9mJywgLT5cblxuXHRvYmplY3QudHlwZU9mKCdzJykuc2hvdWxkLmVxdWFsICdzdHJpbmcnXG5cdG9iamVjdC50eXBlT2YoMCkuc2hvdWxkLmVxdWFsICdudW1iZXInXG5cdG9iamVjdC50eXBlT2YoZmFsc2UpLnNob3VsZC5lcXVhbCAnYm9vbGVhbidcblx0b2JqZWN0LnR5cGVPZih7fSkuc2hvdWxkLmVxdWFsICdvYmplY3QnXG5cdG9iamVjdC50eXBlT2YoYXJndW1lbnRzKS5zaG91bGQuZXF1YWwgJ2FyZ3VtZW50cydcblx0b2JqZWN0LnR5cGVPZihbXSkuc2hvdWxkLmVxdWFsICdhcnJheSdcblxudGVzdCAnZW1wdHknLCAtPlxuXG5cdG8gPVxuXG5cdFx0YTogMVxuXHRcdGI6IDJcblxuXG5cdG9iamVjdC5lbXB0eSBvXG5cblx0by5zaG91bGQubm90LmhhdmUucHJvcGVydHkgJ2EnXG5cdG8uc2hvdWxkLm5vdC5oYXZlLnByb3BlcnR5ICdiJ1xuXG50ZXN0ICdmYXN0RW1wdHknLCAtPlxuXG5cdG8gPVxuXHRcdGE6IDFcblx0XHRiOiAyXG5cblxuXHRvYmplY3QuZmFzdEVtcHR5IG9cblxuXHRvLnNob3VsZC5ub3QuaGF2ZS5wcm9wZXJ0eSAnYSdcblx0by5zaG91bGQubm90LmhhdmUucHJvcGVydHkgJ2InXG5cbnRlc3QgJ2Nsb25lJywgLT5cblxuXHRvYmplY3QuY2xvbmUoWzFdKVswXS5zaG91bGQuZXF1YWwgMVxuXHRvYmplY3QuY2xvbmUoe2E6MX0pLmEuc2hvdWxkLmVxdWFsIDFcblxuXHRvID0ge2E6IDF9XG5cblx0b2JqZWN0LmNsb25lKG8pLnNob3VsZC5ub3QuZXF1YWwgb1xuXG50ZXN0ICdjbG9uZSBbaW5jbHVkZSBwcm90b3R5cGVdJywgLT5cblxuXHRjbGFzcyBDXG5cblx0XHRjb25zdHJ1Y3RvcjogKEBhKSAtPlxuXG5cdFx0c2F5QTogLT4gQGEgKyAnYSdcblxuXHRhID0gbmV3IEMgJ2EnXG5cblx0YS5zYXlBKCkuc2hvdWxkLmVxdWFsICdhYSdcblxuXHRiID0gb2JqZWN0LmNsb25lIGEsIHllc1xuXG5cdGIuc2hvdWxkLm5vdC5lcXVhbCBhXG5cblx0Yi5jb25zdHJ1Y3Rvci5zaG91bGQuZXF1YWwgQ1xuXG5cdGIuYS5zaG91bGQuZXF1YWwgJ2EnXG5cblx0Yi5hID0gJ2EyJ1xuXG5cdGIuc2F5QSgpLnNob3VsZC5lcXVhbCAnYTJhJ1xuXG50ZXN0ICdjbG9uZSBbd2l0aG91dCBwcm90b3R5cGVdJywgLT5cblxuXHRjbGFzcyBDXG5cblx0XHRjb25zdHJ1Y3RvcjogKEBhKSAtPlxuXG5cdFx0c2F5QTogLT4gQGEgKyAnYSdcblxuXHRhID0gbmV3IEMgJ2EnXG5cblx0YS5zYXlBKCkuc2hvdWxkLmVxdWFsICdhYSdcblxuXHRiID0gb2JqZWN0LmNsb25lIGEsIG5vXG5cblx0Yi5zaG91bGQuZXF1YWwgYVxuXG50ZXN0ICdvdmVycmlkZU9udG8gW2Jhc2ljXScsIC0+XG5cblx0b250byA9XG5cdFx0YTogJ2EnXG5cdFx0Yjpcblx0XHRcdGM6ICdjJ1xuXHRcdFx0ZDpcblx0XHRcdFx0ZTogJ2UnXG5cblx0d2hhdCA9XG5cdFx0YTogJ2EyJ1xuXHRcdGI6XG5cdFx0XHRjOiAnYzInXG5cdFx0XHRkOlxuXHRcdFx0XHRmOiAnZjInXG5cblx0b2JqZWN0Lm92ZXJyaWRlT250byBvbnRvLCB3aGF0XG5cblx0b250by5hLnNob3VsZC5lcXVhbCAnYTInXG5cdG9udG8uYi5zaG91bGQuaGF2ZS5wcm9wZXJ0eSAnYydcblx0b250by5iLmMuc2hvdWxkLmVxdWFsICdjMidcblx0b250by5iLmQuc2hvdWxkLm5vdC5oYXZlLnByb3BlcnR5ICdmJ1xuXHRvbnRvLmIuZC5lLnNob3VsZC5lcXVhbCAnZSdcblxudGVzdCAnb3ZlcnJpZGUnLCAtPlxuXG5cdG9udG8gPVxuXG5cdFx0YTogJ2EnXG5cblx0XHRiOlxuXG5cdFx0XHRjOiAnYydcblxuXHRcdFx0ZDpcblxuXHRcdFx0XHRlOiAnZSdcblxuXHR3aGF0ID1cblxuXHRcdGE6ICdhMidcblxuXHRcdGI6XG5cblx0XHRcdGM6ICdjMidcblxuXHRcdFx0ZDpcblxuXHRcdFx0XHRmOiAnZjInXG5cblxuXHRvbnRvMiA9IG9iamVjdC5vdmVycmlkZSBvbnRvLCB3aGF0XG5cblx0b250bzIuYS5zaG91bGQuZXF1YWwgJ2EyJ1xuXHRvbnRvMi5iLnNob3VsZC5oYXZlLnByb3BlcnR5ICdjJ1xuXHRvbnRvMi5iLmMuc2hvdWxkLmVxdWFsICdjMidcblx0b250bzIuYi5kLnNob3VsZC5ub3QuaGF2ZS5wcm9wZXJ0eSAnZidcblx0b250bzIuYi5kLmUuc2hvdWxkLmVxdWFsICdlJ1xuXG5cdG9udG8uc2hvdWxkLm5vdC5lcXVhbCBvbnRvMlxuXG5kbyAtPlxuXG5cdHdoYXQgPVxuXG5cdFx0YTogJ2EyJ1xuXG5cdFx0YzogLT5cblxuXHRcdHo6ICd6J1xuXG5cdFx0eTpcblxuXHRcdFx0YTogJ2EnXG5cblx0b250byA9XG5cblx0XHRhOiAnYSdcblxuXHRcdGI6ICdiJ1xuXG5cdHRlc3QgJ2FwcGVuZE9udG8gW2Jhc2ljXScsIC0+XG5cblx0XHRvYmplY3QuYXBwZW5kT250byBvbnRvLCB3aGF0XG5cblx0XHRvbnRvLmEuc2hvdWxkLmVxdWFsICdhMidcblx0XHRvbnRvLmIuc2hvdWxkLmVxdWFsICdiJ1xuXHRcdG9udG8uei5zaG91bGQuZXF1YWwgJ3onXG5cblx0dGVzdCBcImFwcGVuZE9udG8gW3NoYWxsb3cgY29waWVzIGluc3RhbmNlc11cIiwgLT5cblxuXHRcdG9udG8uYy5zaG91bGQuYmUuaW5zdGFuY2VvZiBGdW5jdGlvblxuXHRcdG9udG8uYy5zaG91bGQuZXF1YWwgd2hhdC5jXG5cblxuXHR0ZXN0IFwiYXBwZW5kT250byBbY2xvbmVzIG9iamVjdHNdXCIsIC0+XG5cblx0XHRvbnRvLnNob3VsZC5oYXZlLnByb3BlcnR5ICd5J1xuXHRcdG9udG8ueS5hLnNob3VsZC5lcXVhbCAnYSdcblx0XHRvbnRvLnkuc2hvdWxkLm5vdC5lcXVhbCB3aGF0LnlcblxudGVzdCAnZ3JvdXBQcm9wcycsIC0+XG5cblx0b2JqID1cblxuXHRcdGExOiAnMSdcblx0XHRhMjogJzInXG5cblx0XHRiMTogJzEnXG5cdFx0YjI6ICcyJ1xuXG5cdFx0YzE6ICcxJ1xuXHRcdGMyOiAnMidcblxuXHRcdHJlc3QxOiAnMSdcblx0XHRyZXN0MjogJzInXG5cblx0Z3JvdXBzID0gb2JqZWN0Lmdyb3VwUHJvcHMgb2JqLFxuXG5cdFx0YTogWydhMScsICdhMiddXG5cblx0XHRiOiBbL15iWzAtOV0rJC9dXG5cblx0XHRjOiAoa2V5KSAtPiBrZXlbMF0gaXMgJ2MnXG5cblx0Z3JvdXBzLmEuc2hvdWxkLmhhdmUucHJvcGVydHkgJ2ExJ1xuXHRncm91cHMuYS5hMS5zaG91bGQuZXF1YWwgJzEnXG5cblx0Z3JvdXBzLmEuc2hvdWxkLmhhdmUucHJvcGVydHkgJ2EyJ1xuXG5cdGdyb3Vwcy5iLnNob3VsZC5oYXZlLnByb3BlcnR5ICdiMSdcblx0Z3JvdXBzLmIuc2hvdWxkLmhhdmUucHJvcGVydHkgJ2IyJ1xuXG5cdGdyb3Vwcy5jLnNob3VsZC5oYXZlLnByb3BlcnR5ICdjMSdcblx0Z3JvdXBzLmMuc2hvdWxkLmhhdmUucHJvcGVydHkgJ2MyJ1xuXG5cdGdyb3Vwcy5yZXN0LnNob3VsZC5oYXZlLnByb3BlcnR5ICdyZXN0MSdcblx0Z3JvdXBzLnJlc3Quc2hvdWxkLmhhdmUucHJvcGVydHkgJ3Jlc3QxJ1xuXG5cdGdyb3Vwcy5yZXN0LnNob3VsZC5ub3QuaGF2ZS5wcm9wZXJ0eSAnYzEnIl19