import gi
import smbus2
import CSPB
import os.path
import time
import threading
import queue
from datetime import datetime

gi.require_version('Gtk', '3.0')
from gi.repository import Gtk, GLib, Gdk


class gui2:
    """ 
    A class to generate a graphical user interface and provide i2c 
    communications with the cluster system power board. This class handles the
    callback function generated by the glade layout manager.
    
    Attributes
    ----------
    None
    
    Methods
    -------
    set_power(value):
        Sends the power command value to the cluster system power board.
        
    do_destroy_window(self, widget, data=None):
    do_i2c_device_address_changed
    do_rescan_button_clicked
    do_send_clicked
    do_read_registers_clicked
    do_register_write_enable
    do_write_register_clicked
    do_IOError_close_clicked
    populate_bus_num
    populate_device_num
    init_cspb_command_type_group
    set_widget_references
    get_device_on_bus
    update_power_display
    set_power_indicator_text
    update_progress
    show_busy_cursor
    get_register_data
    update_register_display
    process_shutdown_commands
    get_cspb
    rescan_i2c_bus
    get_available_i2c_bus_numbers
    clear_power_indicators
    clear_slot_command_selection
    clear_register_display
    do_io_error_dialog_destroy
    
    """
    
    def __init__(self):

        self.builder = Gtk.Builder()
        self.builder.add_from_file("Test2.glade")
        self.builder.connect_signals(self)
        
        self.set_widget_references()
        self.populate_bus_num(self.get_available_i2c_bus_numbers())
        self.populate_device_num(self.get_device_on_bus(int(self.bus_combobox.get_active_text())))
        self.init_cspb_command_type_group()
        
        self.gui = self.builder.get_object("CSPB Command Window")
        self.gtk_box = self.builder.get_object("gtk_box")
        self.gui.show()
        
    def main(self):
        Gtk.main()
        
    def do_destroy_window(self, widget, data=None):
        print("destroy_window()")
        Gtk.main_quit()
        
    def do_i2c_device_address_changed(self, widget, data=None):
        print("do_i2c_device_address_changed()")
        self.update_power_display()
        
    def do_rescan_button_clicked(self, widget, data=None):
        print("do_rescan_button_clicked()")
        self.clear_power_indicators()
        self.clear_slot_command_selection()
        self.rescan_i2c_bus()
        
    def do_send_clicked(self, widget, data=None):
        print("do_send_clicked()")
        
        command_set = 0; # decimal value representing the slots selected.
        number_of_slots_selected = 0
        for slot in range (1, 5):
            command = "slot_command_"
            command += str(slot)
            slot_command = self.builder.get_object(command)
            slot_command_value = int(slot_command.get_active())
            if (slot_command_value == 1):
                number_of_slots_selected += 1
                self.set_power_indicator_text(slot-1, " ")
            shift_command = slot_command_value << slot-1
            command_set = command_set + shift_command

        try:
            cspb = self.get_cspb()
            time.sleep(.01)
        except Exception as e:
            print("do_send_clicked(): input error! ", e)
            return
        
        # Determine which command to send to the CSPB.
        if (self.rb_shutdown.get_active() == True):
            if (number_of_slots_selected == 0):
                return
            shutdown_timeout = cspb.read_register(CSPB.CSPB.SHTDWN_TIME_OUT_RGSTR_ADDR) # already in seconds.
            power_down_hold_delay = (cspb.read_register(CSPB.CSPB.PWR_DWN_SGNL_DRTN_RGSTR_ADDR)*20)/1000 # convert from miliseconds to seconds.
            max_trys = number_of_slots_selected*(shutdown_timeout + power_down_hold_delay)+1
            print("Max trys ", max_trys, "command_set: ", str(command_set))
                    
            cspb.shutdown(command_set)
            time.sleep(.1)
            
            work_thread = threading.Thread(target=self.process_shutdown_commands, args=(cspb, max_trys))
            work_thread.start()
            
        elif (self.rb_signal_only.get_active() == True):
            if (number_of_slots_selected == 0):
                return
            cspb.signal_shutdown(command_set)
        elif (self.rb_power.get_active() == True):
            cspb.set_power(command_set)
            time.sleep(.1)
            self.update_power_display()

                            
    def do_read_registers_clicked(self, widget, data=None):
        print("do_read_registers_clicked()")
        
        bus_id = int(self.bus_combobox.get_active_text())
        i2c_addr = int(self.addr_combobox.get_active_text())
        self.data_queue = queue.Queue()
        work_thread = threading.Thread(target=self.get_register_data, args=(bus_id, i2c_addr, self.data_queue))
        work_thread.start()
           
    def do_register_write_enable(self, widget, data=None):
        print("do_register_write_enable()", widget.get_name())
        
        if (widget.get_name() == "enable_0"):
            self.enable_box = self.builder.get_object('register_write_enable_0')
            self.input_box = self.builder.get_object('register_address_input_0')
        elif (widget.get_name() == "enable_1"):
            self.enable_box = self.builder.get_object('register_write_enable_1')
            self.input_box = self.builder.get_object('register_address_input_1')
        elif (widget.get_name() == "enable_2"):
            self.enable_box = self.builder.get_object('register_write_enable_2')
            self.input_box = self.builder.get_object('register_address_input_2')            
        elif (widget.get_name() == "enable_3"):
            self.enable_box = self.builder.get_object('register_write_enable_3')
            self.input_box = self.builder.get_object('register_address_input_3')            
        elif (widget.get_name() == "enable_4"):
            self.enable_box = self.builder.get_object('register_write_enable_4')
            self.input_box = self.builder.get_object('register_address_input_4')            
        elif (widget.get_name() == "enable_5"):
            self.enable_box = self.builder.get_object('register_write_enable_5')
            self.input_box = self.builder.get_object('register_address_input_5')            
            
        if self.enable_box.get_active():
            self.input_box.set_visible(True)
        else:
            self.input_box.set_visible(False)
        
        
    def do_write_register_clicked(self, widget, data=None):
        print("do_write_register_clicked()")
        
        try:
            cspb = self.get_cspb()
        except Exception as e:
            print("do_write_register_clicked(): ", e)
            return
            
        write_enable_prefix = "register_write_enable_"
        input_box_prefix = "register_address_input_"
        for register in range (0, 6):
            write_enable_ID = write_enable_prefix +str(register)
            write_enable_object = self.builder.get_object(write_enable_ID)
            if write_enable_object.get_active():
                input_box_ID = input_box_prefix +str(register)
                print("input box id: ", input_box_ID, "register: ", register)
                input_box_object = self.builder.get_object(input_box_ID)
                input_buffer = input_box_object.get_buffer()
                cspb.write_register(register, int(input_buffer.get_text()))
                time.sleep(.1) # allow bus to clear before next command.
        self.do_read_registers_clicked(widget, data=None)
        
    def do_IOError_close_clicked(self, widget, data=None):
        print("do_IOError_close_clicked")
    
        self.io_error_dialog.hide();

    def populate_bus_num(self, list_store):
        print("populate_bus_num()")
        
        combobox = self.builder.get_object('Bus_Number_Combobox')
        combobox.set_model(list_store)
        combobox.set_active(0)
        
    def populate_device_num(self, list_store):
        print("populate_device_num()")
        
        self.addr_combobox.set_model(list_store)
        self.addr_combobox.set_active_id(None)

        GLib.idle_add(self.show_busy_cursor, False)
        
    def init_cspb_command_type_group(self):
        self.rb_shutdown.set_active(True)
        
    def set_widget_references(self):
        self.rb_shutdown = self.builder.get_object('command_type_shutdown')
        self.rb_signal_only = self.builder.get_object('command_type_shutdown_signal_only')
        self.rb_power = self.builder.get_object('command_type_power')
        self.bus_combobox = self.builder.get_object('Bus_Number_Combobox')
        self.addr_combobox = self.builder.get_object('Device_Addr_Combobox')
        self.progress_bar = self.builder.get_object('progress_bar')
        self.progress_bar.set_size_request(100,30)
        self.io_error_dialog = self.builder.get_object('register_IO_error_dialog')
        
    def get_device_on_bus(self, busNum):
        devices = Gtk.ListStore(str)
        try:
            bus = smbus2.SMBus(busNum)
            for addr in range (8, 178):
                GLib.idle_add(self.update_progress, (addr+1)/178)
                try:
                    bus.write_quick(addr)
                    devices.append([str(addr)])
                except IOError:
                    pass
        except:
            pass
        GLib.idle_add(self.update_progress, 0)
        return devices
    
    def update_power_display(self):
        print("update_power_display()")
        
        for nuber_of_trys in range (0, 3): #  make 3 attempts to get power data.
            try:
                cspb = self.get_cspb()
                power_state = cspb.read_register(CSPB.CSPB.PWR_STTS_RGSTR_ADDR)
                print("power state ", str(power_state))
                for indicator_num in range (0,4):
                    if (power_state & (1 << indicator_num) >= 1):
                        self.set_power_indicator_text(indicator_num, "On")
                    else:
                        self.set_power_indicator_text(indicator_num, "Off")
                return
            except Exception as e:
                print("display error", e)
                time.sleep(.5) # delay before trying again.
            
    def set_power_indicator_text(self, indicator_num, text):
        print("set_power_indicator_text()")
        
        display_name = "power_indicator_" +str(indicator_num)
        display_object = self.builder.get_object(display_name)
        display_object.set_label(text)
        
    def update_progress(self, fraction):
        print("update_progress()")
        
        self.progress_bar.set_fraction(fraction)
        return False
    
    def show_busy_cursor(self, visable):
        print("show_busy_cursor()")
        
        if (visable == True):
            self.gui.get_window().set_cursor(Gdk.Cursor(Gdk.CursorType.WATCH))
        else:
            self.gui.get_window().set_cursor(Gdk.Cursor(Gdk.CursorType.ARROW))
    
    def get_register_data(self, bus_id, i2c_addr, data_queue):
        print("get_register_data()")

        GLib.idle_add(self.show_busy_cursor, True)
        GLib.idle_add(self.clear_register_display)
        try:
            cspb = self.get_cspb()
            number_of_registers_to_retreive = 6 # The frist 6 registers are user programable.
            for register_num in range (0,number_of_registers_to_retreive):
                data_queue.put(cspb.read_register(register_num))
                GLib.idle_add(self.update_progress, (register_num+1)/number_of_registers_to_retreive)
                time.sleep(.1) # Give i2c bus time to stablize.
        except Exception as e:
            print("get_register_data() error: ", e)
            GLib.idle_add(self.update_progress, 0)
            GLib.idle_add(self.show_busy_cursor, False)
            GLib.idle_add(self.io_error_dialog.show)
            # TODO - Popup an error message.
            return
        GLib.idle_add(self.update_progress, 0)
        GLib.idle_add(self.update_register_display, data_queue)
        GLib.idle_add(self.show_busy_cursor, False)
        return
                    
    def update_register_display(self, data_queue):
        print("update_register_display()")
        
        display_name_prefix = "register_value_display"
        number_of_registers = 6 # The frist 6 registers are user programable.
        for register_num in range (0,number_of_registers):
            display_name = display_name_prefix + str(register_num)
            display_object = self.builder.get_object(display_name)
            display_buffer = display_object.get_buffer()
            display_buffer.set_text(str(self.data_queue.get()))
        return False
    
    def process_shutdown_commands(self, cspb, max_trys):
        print("process_shutdown_commands()")
    
        in_shutdown = True
        attempts = 0
        GLib.idle_add(self.show_busy_cursor, (True))
        while (in_shutdown == True):
            print("get is_shutting_down", str(datetime.now()))
            attempts += 1
            print("attempts ", str(attempts))
            GLib.idle_add(self.update_progress, (attempts/max_trys))
            if (attempts > max_trys):
                print("Max trys reached")
                in_shutdown = False
            try:
                is_shutting_down = cspb.read_register(CSPB.CSPB.IN_SHUTDOWN_RGSTR_ADDR) # any read command will do
                if (is_shutting_down):
                    pass # just wait
                else:
                    in_shutdown = False        
            except IOError:
                print("IOError")
                print("in_shutdown = ", in_shutdown)
            time.sleep(1)
        GLib.idle_add(self.update_progress, 0)
        GLib.idle_add(self.update_power_display)
        GLib.idle_add(self.show_busy_cursor, (False))
        
    def get_cspb(self):
        print("get_cspb()")
        
        bus_combobox = self.builder.get_object('Bus_Number_Combobox')
        bus_number = int(bus_combobox.get_active_text())
        addr_combobox = self.builder.get_object('Device_Addr_Combobox')
        i2c_address = int(addr_combobox.get_active_text())    
        cspb = CSPB.CSPB(bus_number, i2c_address)
        return cspb
    
            
    def rescan_i2c_bus(self):
        print("rescan_i2c_bus()")
        
        GLib.idle_add(self.show_busy_cursor, True)
        self.populate_bus_num(self.get_available_i2c_bus_numbers())
        self.populate_device_num(self.get_device_on_bus(int(self.bus_combobox.get_active_text())))

    def get_available_i2c_bus_numbers(self):
        print("get_available_i2c_bus_numbers()")
        store = Gtk.ListStore(str)
        for i2c_num in range (0, 10):
            file_path = "/dev/i2c-"
            file_path  += str(i2c_num)
            if os.path.exists(file_path):
                store.append([str(i2c_num)])
        return store
    
    def clear_power_indicators(self):
        print("clear_power_indicators()")
        
        for indicator_num in range (0,4):
            self.set_power_indicator_text(indicator_num, " ")

    def clear_slot_command_selection(self):
        print("clear_slot_command_selection()")
        
        for slot in range (1, 5):
            command = "slot_command_"
            command += str(slot)
            slot_command = self.builder.get_object(command)
            slot_command.set_active(False)
            
    def clear_register_display(self):
        print("clear_register_display()")

        display_name_prefix = "register_value_display"
        number_of_registers = 6 # The frist 6 registers are user programable.
        for register_num in range (0,number_of_registers):
            display_name = display_name_prefix + str(register_num)
            display_object = self.builder.get_object(display_name)
            display_buffer = display_object.get_buffer()
            display_buffer.set_text("")
        return False
    
    def do_io_error_dialog_destroy(self, widget, data=None):
        print("do_io_error_dialog_destroy()")
        
        self.io_error_dialog.hide()
        return True # Prevent dialog from being destroyed.
            
if __name__ == '__main__':

    app = gui2()
    app.main()