# Copyright 2020 Gerard L. Muir 
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
#  in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and or sell
#  copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
# SOFTWARE.
import gi
import smbus
from cspb.CSPB import CSPB
import os.path
import time
import threading
import sys
import getopt
from datetime import datetime
import logging

try:
    import queue
except:
    import Queue as queue

gi.require_version('Gtk', '3.0')
from gi.repository import Gtk, GLib, Gdk


class cspb_gui:
    """ 
    A class to generate a graphical user interface and provide i2c 
    communications with the cluster system power board. This class handles the
    callback function generated by the glade layout manager.
    
    Attributes
    ----------
    None
    
    Methods
    -------
    set_power(value):
        Sends the power command value to the cluster system power board.
        
    do_destroy_window(self, widget, data=None):
        Glade callback for application termination through window closure.
        
    do_i2c_device_address_changed(self, widget, data=None):
        Glade callback for i2c address combo box data change event.
    
    do_rescan_button_clicked(self, widget, data=None):
        Glade callback for rescan button.
    
    do_send_clicked(self, widget, data=None):
        Glade callback for send button.
        
    do_read_registers_clicked(self, widget, data=None):
        Glade callback for read registers button.
    
    do_register_write_enable(self, widget, data=None):
        Glade callback for all write enable check boxes.
    
    do_write_register_clicked(self, widget, data=None):
        Glade callback for write registers button.
    
    do_IOError_close_clicked(self, widget, data=None):
        Glade callback for dialog box closure.
    
    populate_bus_num(self, list_store):
        Populates the bus number combo box with identified buss numbers.
         
    populate_device_num(self, list_store):
        Populates the device number combo box with identified device numbers.
        
    init_cspb_command_type_group(self):
        Initializes the command button group to the shutdown command.
        
    set_widget_references(self):
        Creates class attributes for quick access to Glade components.
        
    get_device_on_bus(self, busNum):
        Returns a list of i2c device addresses found on the specified buss.
        
    update_power_display(self):
        Updates the current power state display for the specified device.
        
    set_power_indicator_text
        Sets the power indicator field to the specified state of 'On' or 'Off'
        
    update_progress
        Updates the progress bar display to the specified level.
        
    show_busy_cursor
        Show or hide the busy cursor.
        
    get_register_data
        Retrieves the CSPB register data and updates the register display. 
        
    update_register_display
        Update the register display with data from the query queue.
        
    process_shutdown_commands
        Sends the shutdown command to the CSPB and monitors the command progress.
        
    get_cspb
        Returns a CSPB communication object.
        
    rescan_i2c_bus
        Scans the i2c bus for i2c device addresses.
        
    get_available_i2c_bus_numbers
        Scans the host Linux system for i2c buss numbers.
          
    clear_power_indicators
        Sets the power indicator fields to a blank string.
        
    clear_slot_command_selection
        Sets all command selection check boxes to unselected.
        
    clear_register_display
        Clears data from the register display fields.
        
    do_io_error_dialog_destroy(self, widget, data=None):
        Glade callback that overrides the dialog destroy process and hides the
        error dialog box.
    
    """
    
    def __init__(self, opts):
        """
        Initializes and displays the graphical user interface. Also attempts to
        populate the display with any discovered i2c buses numbers.
    
        Attributes
        ----------
        self: cspb_gui
            the cspb_gui object.    
        """
        for opt, arg in opts:
            if opt == '-v':
                print('logging config')
                logging.basicConfig(format='%(asctime)s %(levelname)s %(message)s', datefmt='%m/%d/%Y %I:%M:%S %p', stream=sys.stdout, level=logging.DEBUG)
                
        logging.info('__int__()')
            
        self.builder = Gtk.Builder()
        root_dir = os.path.dirname(os.path.abspath(__file__))
        glade_file_path = os.path.join(root_dir, "cspb_gui.glade")
        self.builder.add_from_file(glade_file_path)
        self.builder.connect_signals(self)
        
        self.set_widget_references()
        self.populate_bus_num(self.get_available_i2c_bus_numbers())
        self.populate_device_num(self.get_device_on_bus(int(self.bus_combobox.get_active_text())))
        self.init_cspb_command_type_group()
        
        self.gui = self.builder.get_object("CSPB Command Window")
        self.gtk_box = self.builder.get_object("gtk_box")
        self.gui.set_title("CSPB Control Center")
        self.gui.show()
        
    def main(self):
        """
        Checks for program options and starts the user interface.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        argv: list 
            the list of options and arguments to the program.

        Returns
        -------
        None

        """
        logging.debug('main()')
        Gtk.main()
        
    def do_destroy_window(self, widget, data=None):
        """
        Glade callback for application termination through window closure.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None

        """
        logging.debug("destroy_window()")
        Gtk.main_quit()
        
    def do_i2c_device_address_changed(self, widget, data=None):
        """
        Glade callback for i2c address combo box data change event.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None

        """
        logging.debug("do_i2c_device_address_changed()")
        self.update_power_display()
        
    def do_rescan_button_clicked(self, widget, data=None):
        """
         Glade callback for rescan button.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None

        """            
        logging.debug("do_rescan_button_clicked()")
        self.clear_power_indicators()
        self.clear_slot_command_selection()
        self.rescan_i2c_bus()
        
    def do_send_clicked(self, widget, data=None):
        """
         Glade callback for the send button.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None

        """
        logging.debug("do_send_clicked()")
        
        command_set = 0; # decimal value representing the slots selected.
        number_of_slots_selected = 0
        for slot in range (1, 5):
            command = "slot_command_"
            command += str(slot)
            slot_command = self.builder.get_object(command)
            slot_command_value = int(slot_command.get_active())
            if (slot_command_value == 1):
                number_of_slots_selected += 1
                self.set_power_indicator_text(slot-1, " ")
            shift_command = slot_command_value << slot-1
            command_set = command_set + shift_command

        try:
            cspb = self.get_cspb()
            time.sleep(.01)
        except Exception as e:
            logging.error("do_send_clicked(): input error! %s", e)
            return
        
        # Determine which command to send to the CSPB.
        if (self.rb_shutdown.get_active() == True):
            if (number_of_slots_selected == 0):
                return
            shutdown_timeout = cspb.read_register(CSPB.SHTDWN_TIME_OUT_RGSTR_ADDR) # already in seconds.
            power_down_hold_delay = (cspb.read_register(CSPB.PWR_DWN_SGNL_DRTN_RGSTR_ADDR)*20)/1000 # convert from miliseconds to seconds.
            max_tries = number_of_slots_selected*(shutdown_timeout + power_down_hold_delay)+1
            logging.debug("Max tries %s command_set: %s", max_tries, str(command_set))
                    
            cspb.shutdown(command_set)
            time.sleep(.1)
            
            work_thread = threading.Thread(target=self.process_shutdown_commands, args=(cspb, max_tries))
            work_thread.start()
            
        elif (self.rb_signal_only.get_active() == True):
            if (number_of_slots_selected == 0):
                return
            cspb.signal_shutdown(command_set)
        elif (self.rb_power.get_active() == True):
            cspb.set_power(command_set)
            time.sleep(.1)
            self.update_power_display()

                            
    def do_read_registers_clicked(self, widget, data=None):
        """
         Glade callback for 
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None
        """
        logging.debug("do_read_registers_clicked()")
        
        self.data_queue = queue.Queue()
        work_thread = threading.Thread(target=self.get_register_data)
        work_thread.start()
           
    def do_register_write_enable(self, widget, data=None):
        """
         Glade callback for 
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None
        """
        logging.debug("do_register_write_enable() %s", widget.get_name())
        
        if (widget.get_name() == "enable_0"):
            self.enable_box = self.builder.get_object('register_write_enable_0')
            self.input_box = self.builder.get_object('register_address_input_0')
        elif (widget.get_name() == "enable_1"):
            self.enable_box = self.builder.get_object('register_write_enable_1')
            self.input_box = self.builder.get_object('register_address_input_1')
        elif (widget.get_name() == "enable_2"):
            self.enable_box = self.builder.get_object('register_write_enable_2')
            self.input_box = self.builder.get_object('register_address_input_2')            
        elif (widget.get_name() == "enable_3"):
            self.enable_box = self.builder.get_object('register_write_enable_3')
            self.input_box = self.builder.get_object('register_address_input_3')            
        elif (widget.get_name() == "enable_4"):
            self.enable_box = self.builder.get_object('register_write_enable_4')
            self.input_box = self.builder.get_object('register_address_input_4')            
        elif (widget.get_name() == "enable_5"):
            self.enable_box = self.builder.get_object('register_write_enable_5')
            self.input_box = self.builder.get_object('register_address_input_5')            
            
        if self.enable_box.get_active():
            self.input_box.set_visible(True)
        else:
            self.input_box.set_visible(False)
        
        
    def do_write_register_clicked(self, widget, data=None):
        """
         Glade callback for 
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None
        """
        logging.debug("do_write_register_clicked()")
        
        try:
            cspb = self.get_cspb()
        except Exception as e:
            logging.debug("do_write_register_clicked(): %s", e)
            return
            
        write_enable_prefix = "register_write_enable_"
        input_box_prefix = "register_address_input_"
        for register in range (0, 6):
            write_enable_ID = write_enable_prefix +str(register)
            write_enable_object = self.builder.get_object(write_enable_ID)
            if write_enable_object.get_active():
                input_box_ID = input_box_prefix +str(register)
                logging.debug("input box id: %s", input_box_ID, "register: %s", register)
                input_box_object = self.builder.get_object(input_box_ID)
                input_buffer = input_box_object.get_buffer()
                cspb.write_register(register, int(input_buffer.get_text()))
                time.sleep(.1) # allow bus to clear before next command.
        self.do_read_registers_clicked(widget, data=None)
        
    def do_IOError_close_clicked(self, widget, data=None):
        """
         Glade callback for 
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        None
        """
        logging.debug("do_IOError_close_clicked")
    
        self.io_error_dialog.hide();

    def populate_bus_num(self, list_store):
        """ 
        Populates the Bus Number combo box with the provided i2c bus numbers.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        list_store: ListStore
            a list containing system i2c bus numbers. 

        Returns
        -------
        None
        """
        logging.debug("populate_bus_num()")
        
        combobox = self.builder.get_object('Bus_Number_Combobox')
        combobox.set_model(list_store)
        combobox.set_active(0)
        
    def populate_device_num(self, list_store):
        """ 
        Populates the Device Number combo box with the provide i2c device addresses.
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        list_store : ListStore
            a list of i2c device numbers found on the i2c bus.
            
        Returns
        -------
        None
        """
        logging.debug("populate_device_num()")
        
        self.addr_combobox.set_model(list_store)
        self.addr_combobox.set_active_id(None)

        GLib.idle_add(self.show_busy_cursor, False)
        
    def init_cspb_command_type_group(self):
        """ 
        Initializes the command type group selection to display 'shutdown'.
         
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        self.rb_shutdown.set_active(True)
        
    def set_widget_references(self):
        """ 
        Creates a set of class attributes pointing to Glade display objects for
        easy access through out this class.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        self.rb_shutdown = self.builder.get_object('command_type_shutdown')
        self.rb_signal_only = self.builder.get_object('command_type_shutdown_signal_only')
        self.rb_power = self.builder.get_object('command_type_power')
        self.bus_combobox = self.builder.get_object('Bus_Number_Combobox')
        self.addr_combobox = self.builder.get_object('Device_Addr_Combobox')
        self.progress_bar = self.builder.get_object('progress_bar')
        self.progress_bar.set_size_request(100,30)
        self.io_error_dialog = self.builder.get_object('register_IO_error_dialog')
        
    def get_device_on_bus(self, busNum):        
        """ 
        Returns a listStore containing i2c device numbers found on the specified
        i2c bus.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        devices : ListStore
        """
        devices = Gtk.ListStore(str)
        try:
            bus = smbus.SMBus(busNum)
            for addr in range (8, 178):
                GLib.idle_add(self.update_progress, (addr+1)/178)
                try:
                    bus.write_quick(addr)
                    devices.append([str(addr)])
                except IOError:
                    pass
        except:
            pass
        GLib.idle_add(self.update_progress, 0)
        return devices
    
    def update_power_display(self):
        """ 
        Queries the power board power register in and updates the GUI with the
        current power state of each slot.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        logging.debug("update_power_display()")
        
        cspb = self.get_cspb()
        for number_of_tries in range (0, 3): #  make 3 attempts to get power data.
            try:
                power_state = cspb.read_register(CSPB.PWR_STTS_RGSTR_ADDR)
                logging.debug("power state %s", str(power_state))
                for indicator_num in range (0,4):
                    if (power_state & (1 << indicator_num) >= 1):
                        self.set_power_indicator_text(indicator_num, "On")
                    else:
                        self.set_power_indicator_text(indicator_num, "Off")
                break
            except Exception as e:
                logging.error("display error %s", e)
                time.sleep(.5) # delay before trying again.
            
    def set_power_indicator_text(self, indicator_num, text):
        """ 
        Sets the text of the specified slot power indicator field to the
        specified value.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        indicator_num : int
            the power indicator field number (0-3).
            
        text : string
            the text to be displayed. (nominally 'On' or 'Off')
            
        Returns
        -------
        None
        """
        logging.debug("set_power_indicator_text()")
        
        display_name = "power_indicator_" +str(indicator_num)
        display_object = self.builder.get_object(display_name)
        display_object.set_label(text)
        
    def update_progress(self, fraction):
        """ 
        Causes the progress bar to "fill in" the portion of the bar specified
        by fraction. The value of fraction should be between 0.0 and 1.0.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
        
        fraction : float
            the decimal percentage 

        Returns
        -------
        False
        """
        logging.debug("update_progress()")
        
        self.progress_bar.set_fraction(fraction)
        return False
    
    def show_busy_cursor(self, visible):
        """ 
        Set the program cursor to the busy cursor or the standard cursor.
         
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        visible : boolean
            The state of the busy cursor. True to display the busy cursor.

        Returns
        -------
        None
        """
        logging.debug("show_busy_cursor()")
        
        if (visible == True):
            self.gui.get_window().set_cursor(Gdk.Cursor(Gdk.CursorType.WATCH))
        else:
            self.gui.get_window().set_cursor(Gdk.Cursor(Gdk.CursorType.ARROW))
    
    def get_register_data(self):
        """ 
        Retrieves the power board EEPROM and Virtual register data while
        updating the progress bar. Then updates register display fields.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        Returns
        -------
        None
        """
        logging.debug("get_register_data()")

        GLib.idle_add(self.show_busy_cursor, True)
        GLib.idle_add(self.clear_register_display)
        try:
            cspb = self.get_cspb()
            number_of_registers_to_retreive = 6 # The first 6 registers are user programmable.
            for register_num in range (0,number_of_registers_to_retreive):
                self.data_queue.put(cspb.read_register(register_num))
                GLib.idle_add(self.update_progress, (register_num+1)/number_of_registers_to_retreive)
                time.sleep(.1) # Give i2c bus time to stabilize.
        except Exception as e:
            logging.error("get_register_data() error: %s", e)
            GLib.idle_add(self.update_progress, 0)
            GLib.idle_add(self.show_busy_cursor, False)
            GLib.idle_add(self.io_error_dialog.show)
            # TODO - Popup an error message.
            return
        GLib.idle_add(self.update_progress, 0)
        GLib.idle_add(self.update_register_display)
        GLib.idle_add(self.show_busy_cursor, False)
        return
                    
    def update_register_display(self):
        """ 
        Updates the register display fields with data available in the data queue.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        logging.debug("update_register_display()")
        
        display_name_prefix = "register_value_display"
        number_of_registers = 6 # The first 6 registers are user programmable.
        for register_num in range (0,number_of_registers):
            display_name = display_name_prefix + str(register_num)
            display_object = self.builder.get_object(display_name)
            display_buffer = display_object.get_buffer()
            display_buffer.set_text(str(self.data_queue.get()))
        return False
    
    def process_shutdown_commands(self, cspb, max_tries):
        """ 
        Monitors the power board's 'in shutdown register' while updating the
        progress bar. 
         
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        logging.debug("process_shutdown_commands()")
    
        in_shutdown = True
        attempts = 0
        GLib.idle_add(self.show_busy_cursor, (True))
        while (in_shutdown == True):
            logging.debug("get is_shutting_down %s", str(datetime.now()))
            attempts += 1
            logging.debug("attempts %s", str(attempts))
            GLib.idle_add(self.update_progress, (attempts/max_tries))
            if (attempts > max_tries):
                logging.debug("Max tries reached.")
                in_shutdown = False
            try:
                is_shutting_down = cspb.read_register(CSPB.IN_SHUTDOWN_RGSTR_ADDR) # any read command will do
                if (is_shutting_down):
                    pass # just wait
                else:
                    in_shutdown = False        
            except IOError:
                logging.error("IOError")
                logging.debug("in_shutdown = %s", in_shutdown)
            time.sleep(1)
        GLib.idle_add(self.update_progress, 0)
        GLib.idle_add(self.update_power_display)
        GLib.idle_add(self.show_busy_cursor, (False))
        
    def get_cspb(self):
        """ 
        Returns a CSPB driver object to communicate with the power board.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        csbp : CSPB
            the CSPB communication driver.
        """
        logging.debug("get_cspb()")
        
        bus_combobox = self.builder.get_object('Bus_Number_Combobox')
        bus_number = int(bus_combobox.get_active_text())
        addr_combobox = self.builder.get_object('Device_Addr_Combobox')
        i2c_address = int(addr_combobox.get_active_text())
        cspb = CSPB(bus_number, i2c_address)
        return cspb
    
            
    def rescan_i2c_bus(self):
        """ 
        Rescans the system for available i2c bus numbers and updates the i2c
        bus number combo box and i2c address combo box.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        logging.debug("rescan_i2c_bus()")
        
        GLib.idle_add(self.show_busy_cursor, True)
        self.populate_bus_num(self.get_available_i2c_bus_numbers())
        self.populate_device_num(self.get_device_on_bus(int(self.bus_combobox.get_active_text())))

    def get_available_i2c_bus_numbers(self):
        """ 
        Returns a list store containing the available i2c bus numbers on the
        system.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        store : ListStore
            the list of available i2c bus numbers.
        """
        logging.debug("get_available_i2c_bus_numbers()")
        store = Gtk.ListStore(str)
        for i2c_num in range (0, 10):
            file_path = "/dev/i2c-"
            file_path  += str(i2c_num)
            if os.path.exists(file_path):
                store.append([str(i2c_num)])
        return store
    
    def clear_power_indicators(self):
        """ 
        Sets all power indicator text fields to a blank string.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        logging.debug("clear_power_indicators()")
        
        for indicator_num in range (0,4):
            self.set_power_indicator_text(indicator_num, " ")

    def clear_slot_command_selection(self):
        """ 
        Clears the check mark from all slot command check boxes.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        logging.debug("clear_slot_command_selection()")
        
        for slot in range (1, 5):
            command = "slot_command_"
            command += str(slot)
            slot_command = self.builder.get_object(command)
            slot_command.set_active(False)
            
    def clear_register_display(self):
        """ 
        Clears all text values from each register text box.
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.

        Returns
        -------
        None
        """
        logging.debug("clear_register_display()")

        display_name_prefix = "register_value_display"
        number_of_registers = 6 # The first 6 registers are user programmable.
        for register_num in range (0,number_of_registers):
            display_name = display_name_prefix + str(register_num)
            display_object = self.builder.get_object(display_name)
            display_buffer = display_object.get_buffer()
            display_buffer.set_text("")
        return False
    
    def do_io_error_dialog_destroy(self, widget, data=None):
        """
         Glade callback for 
        
        Parameters
        ----------
        self : cspb_gui
            this cspb_gui object.
            
        widget: GtkWidget 
            the GTK widget.
            
        data : gpointer
            the GTK widget user data.

        Returns
        -------
        True
        """
        logging.debug("do_io_error_dialog_destroy()")
        
        self.io_error_dialog.hide()
        return True # Prevent dialog from being destroyed.

            
if __name__ == '__main__':
    
    try:
        opts, args = getopt.getopt(sys.argv[1:],"v")
    except getopt.GetoptError:
        print('cspb_gui.py -v')
        sys.exit(2)                
    
    app = cspb_gui(opts)
    app.main()