//
//  Color.swift
//  CostumeKit
//
//  Created by Jake Marsh on 8/7/16.
//  Copyright © 2016 Jake Marsh. All rights reserved.
//

import Foundation

public typealias Color = String

extension Color : ColorConvertible {
  public var colorValue: UIColor {
    return UIColor(hexString: self)!
  }

  public static let none = UIColor.clear
  public static let white = UIColor.white
  public static let black = UIColor.black
}

//  SwiftHEXColors.swift
//
// Copyright (c) 2014 Doan Truong Thi
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#if os(iOS) || os(tvOS)
  import UIKit
  typealias SWColor = UIColor
#else
  import Cocoa
  typealias SWColor = NSColor
#endif

private extension Int {
  func duplicate4bits() -> Int {
    return (self << 4) + self
  }
}

/// An extension of UIColor (on iOS) or NSColor (on OSX) providing HEX color handling.
public extension SWColor {
  /**
   Create non-autoreleased color with in the given hex string. Alpha will be set as 1 by default.
   
   - parameter hexString: The hex string, with or without the hash character.
   - returns: A color with the given hex string.
   */
  public convenience init?(hexString: String) {
    self.init(hexString: hexString, alpha: 1.0)
  }
  
  private convenience init?(hex3: Int, alpha: Float) {
    self.init(red:   CGFloat( ((hex3 & 0xF00) >> 8).duplicate4bits() ) / 255.0,
              green: CGFloat( ((hex3 & 0x0F0) >> 4).duplicate4bits() ) / 255.0,
              blue:  CGFloat( ((hex3 & 0x00F) >> 0).duplicate4bits() ) / 255.0, alpha: CGFloat(alpha))
  }
  
  private convenience init?(hex6: Int, alpha: Float) {
    self.init(red:   CGFloat( (hex6 & 0xFF0000) >> 16 ) / 255.0,
              green: CGFloat( (hex6 & 0x00FF00) >> 8 ) / 255.0,
              blue:  CGFloat( (hex6 & 0x0000FF) >> 0 ) / 255.0, alpha: CGFloat(alpha))
  }
  
  /**
   Create non-autoreleased color with in the given hex string and alpha.
   
   - parameter hexString: The hex string, with or without the hash character.
   - parameter alpha: The alpha value, a floating value between 0 and 1.
   - returns: A color with the given hex string and alpha.
   */
  public convenience init?(hexString: String, alpha: Float) {
    var hex = hexString
    
    // Check for hash and remove the hash
    if hex.hasPrefix("#") {
      hex = hex.substring(from: hex.index(hex.startIndex, offsetBy: 1))
    }
    
    guard let hexVal = Int(hex, radix: 16) else {
      self.init()
      return nil
    }
    
    switch hex.characters.count {
    case 3:
      self.init(hex3: hexVal, alpha: alpha)
    case 6:
      self.init(hex6: hexVal, alpha: alpha)
    default:
      // Note:
      // The swift 1.1 compiler is currently unable to destroy partially initialized classes in all cases,
      // so it disallows formation of a situation where it would have to.  We consider this a bug to be fixed
      // in future releases, not a feature. -- Apple Forum
      self.init()
      return nil
    }
  }
  
  /**
   Create non-autoreleased color with in the given hex value. Alpha will be set as 1 by default.
   
   - parameter hex: The hex value. For example: 0xff8942 (no quotation).
   - returns: A color with the given hex value
   */
  public convenience init?(hex: Int) {
    self.init(hex: hex, alpha: 1.0)
  }
  
  /**
   Create non-autoreleased color with in the given hex value and alpha
   
   - parameter hex: The hex value. For example: 0xff8942 (no quotation).
   - parameter alpha: The alpha value, a floating value between 0 and 1.
   - returns: color with the given hex value and alpha
   */
  public convenience init?(hex: Int, alpha: Float) {
    if (0x000000 ... 0xFFFFFF) ~= hex {
      self.init(hex6: hex , alpha: alpha)
    } else {
      self.init()
      return nil
    }
  }
}
