/*
 * Copyright (c) 2023 Isaac Chou
 * 
 * This software is licensed under the MIT License that can be 
 * found in the LICENSE file at the top of the source tree
 */
const mat4 = glMatrix.mat4;
const vec3 = glMatrix.vec3;

const texture_map = new Map();
const keyboard = new Set();
const mouse = new Set();
const key_map = new Map([["Escape",256],["Enter",257],["ArrowRight",262],["ArrowLeft",263],
                        ["ArrowDown",264],["ArrowUp",265],["ShiftLeft",340],["ShiftRight",344]]);
class Renderer 
{
  constructor(gl, camera) {
    this.gl = gl;
    this.camera = camera;
    this.shaderProgram = this.initShaderProgram();
    this.gl.useProgram(this.shaderProgram);
    this.shape_map = new Map();
  }

  setup() {
    const gl = this.gl;
    const shaderProgram = this.shaderProgram;

    // a directional light vector pointing from the light source
    const light_direction = vec3.create();
    vec3.normalize(light_direction, [-1, -3, 0]);
    const light_ambient = 0.6;
    gl.uniform1f(gl.getUniformLocation(shaderProgram, "light.ambient"), light_ambient);
    gl.uniform3fv(gl.getUniformLocation(shaderProgram, "light.direction"), light_direction);
    gl.enable(gl.DEPTH_TEST);
    gl.clearColor(0.2, 0.3, 0.3, 1.0);
    gl.clearDepth(1.0);
    gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
    this.reshape();
  }

  reshape() {
    const canvas = document.querySelector("canvas");
    canvas.width = document.documentElement.clientWidth;
    canvas.height = document.documentElement.clientHeight;
    this.gl.viewport(0, 0, canvas.width, canvas.height);
    
    const projection = mat4.create();
    const fovy = window.matchMedia("(orientation: portrait)").matches ? 90.0 : 60.0
    mat4.perspective(projection, glMatrix.glMatrix.toRadian(fovy), canvas.width/canvas.height, 0.1, 600.0);
    this.gl.uniformMatrix4fv(this.gl.getUniformLocation(this.shaderProgram, "projection"), false, new Float32Array(projection));
  }

  draw() {
    const gl = this.gl;
    gl.clearColor(0.2, 0.3, 0.3, 1.0);
    gl.clearDepth(1.0);
    gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

    const view = this.camera.get_view_matrix();
    gl.uniformMatrix4fv(gl.getUniformLocation(this.shaderProgram, "view"), false, new Float32Array(view));
    for (let shape of this.shape_map.values()) {
      shape.draw(this.shaderProgram, shape.trans);
    }
  }

  add_texture(id, width, height, image) {
    const gl = this.gl;
    const txtr = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, txtr);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.REPEAT);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.REPEAT);        
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGB, width, height, 0, gl.RGB, gl.UNSIGNED_BYTE, image);
    texture_map.set(id, txtr);
  }

  add_shape(shape_id, descriptor) {
    this.shape_map.set(shape_id, new Shape(this.gl, descriptor));
  }

  update_shape(shape_id, trans) {
    if(this.shape_map.has(shape_id)) {
      this.shape_map.get(shape_id).trans = trans;
    }
  }

  remove_shape(shape_id) {
    if(this.shape_map.has(shape_id)) {
      this.shape_map.delete(shape_id);
    }
  }

  initShaderProgram()
  { 
    const gl = this.gl;

    // Vertex shader
    const vsSource = `#version 300 es
      layout(location = 0) in vec3 pos;
      layout(location = 1) in vec2 t;
      layout(location = 2) in vec3 n;

      out vec3 normal;
      out vec2 txtr_pos;

      uniform mat4 model;
      uniform mat4 view;
      uniform mat4 projection;

      void main()
      {
        normal = mat3(transpose(inverse(model))) * n;
        txtr_pos = t;
        gl_Position = projection * view * model * vec4(pos, 1.0);
      }    
    `;

    // Fragment shader
    const fsSource = `#version 300 es
      precision mediump float;

      struct Light {
        float ambient;
        vec3 direction;
      };
      in vec3 normal;
      in vec2 txtr_pos;
      out vec4 clr;

      uniform sampler2D txtr;
      uniform Light light;

      void main()
      {
        float light = max(dot(normal, -light.direction), 0.0) + light.ambient;
        clr = vec4(light * texture(txtr, txtr_pos).rgb, 1.0);
      }  
    `;
    const vertexShader = this.loadShader(gl.VERTEX_SHADER, vsSource);
    const fragmentShader = this.loadShader(gl.FRAGMENT_SHADER, fsSource);

    const shaderProgram = gl.createProgram();
    gl.attachShader(shaderProgram, vertexShader);
    gl.attachShader(shaderProgram, fragmentShader);
    gl.linkProgram(shaderProgram);
    if (!gl.getProgramParameter(shaderProgram, gl.LINK_STATUS)) {
      alert(`Failed to link program: ${gl.getProgramInfoLog(shaderProgram)}`);
      return null;
    }
    return shaderProgram;
  }

  loadShader(type, source)
  {
    const gl = this.gl;
    const shader = gl.createShader(type);
    gl.shaderSource(shader, source);
    gl.compileShader(shader);
    if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {
      alert(`Failed to compile shader: ${gl.getShaderInfoLog(shader)}`);
      gl.deleteShader(shader);
      return null;
    }
    return shader;
  }
}

class Shape
{
  constructor(gl, d) {
    this.gl = gl;    
    this.trans = d.trans;
    if (Object.hasOwn(d, 'default_texture')) {
      this.default_texture = texture_map.get(d.default_texture);
    }
    
    if (Object.hasOwn(d, 'child')) {
      // compound shape
      this.child = new Array();
      for (let s of d.child) {
        this.child.push(new Shape(gl, s));
      }
      return;
    } else {
      // simple shape
      this.mesh = d.mesh;
      this.face_index = d.face_index;
      this.textures = Object.hasOwn(d, "textures") ? d.textures : [];
    }

    // setup shape vertices
    this.vao = gl.createVertexArray();
    gl.bindVertexArray(this.vao);

    this.buffer = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, this.buffer);

    const mesh = this.mesh;
    const num_vertices = mesh.length / 5;
    const normal = new Array();
    for (let n = 0; n < num_vertices; n += 3) {
      let i = n * 5;
      //const mesh = shape.mesh;
      const p1 = vec3.fromValues(mesh[i++], mesh[i++], mesh[i++]); i += 2;
      const p2 = vec3.fromValues(mesh[i++], mesh[i++], mesh[i++]); i += 2;
      const p3 = vec3.fromValues(mesh[i++], mesh[i++], mesh[i++]);

      // one normal vector for each vertex!!!
      const norm = vec3.create();
      vec3.subtract(p2, p2, p1);
      vec3.subtract(p3, p3, p1);
      vec3.normalize(norm, vec3.cross(norm, p2, p3));
      normal.push(norm[0], norm[1], norm[2]);
      normal.push(norm[0], norm[1], norm[2]);
      normal.push(norm[0], norm[1], norm[2]);
    }
    gl.bufferData(gl.ARRAY_BUFFER, num_vertices * (5 + 3) * 4, gl.STATIC_DRAW);
    gl.bufferSubData(gl.ARRAY_BUFFER, 0, new Float32Array(this.mesh));
    gl.bufferSubData(gl.ARRAY_BUFFER, num_vertices * 5 * 4, new Float32Array(normal));

    // pos buffer (location = 0 in vertex shader)
    gl.vertexAttribPointer(0, 3, gl.FLOAT, false, 5 * 4, 0);
    gl.enableVertexAttribArray(0);

    // texture coordinates (location = 1)
    gl.vertexAttribPointer(1, 2, gl.FLOAT, false, 5 * 4, 3 * 4);
    gl.enableVertexAttribArray(1);

    // normals (location = 2)
    gl.vertexAttribPointer(2, 3, gl.FLOAT, false, 3 * 4, num_vertices * 5 * 4);
    gl.enableVertexAttribArray(2);
  }

  draw(shaderProgram, model) {
    const gl = this.gl;
    if (Object.hasOwn(this, 'child')) {
      for (let s of this.child) {
        const m = mat4.create();
        mat4.multiply(m, model, s.trans)
        s.draw(shaderProgram, m);
      }    
    } 
    else 
    { // draw a simple shape
      gl.uniformMatrix4fv(gl.getUniformLocation(shaderProgram, "model"), false, new Float32Array(model));
      gl.bindVertexArray(this.vao);
      const num_vertices = this.mesh.length / 5;
      for (let i = 0; i < this.face_index.length; i++)
      {          
        if (!Object.hasOwn(this, "default_texture") && i >= this.textures.length) {
          // skip the face if no texture specified
          // no wire-frame drawing mode support in WebGL
          continue;
        }        
        const index = this.face_index[i];
        const n = (i == this.face_index.length - 1) ? num_vertices - index : this.face_index[i + 1] - index;
        const txtr = (i < this.textures.length) ? texture_map.get(this.textures[i]) : this.default_texture;
        gl.activeTexture(gl.TEXTURE0);
        gl.bindTexture(gl.TEXTURE_2D, txtr);    
        gl.uniform1i(gl.getUniformLocation(shaderProgram, "txtr"), 0);
        gl.drawArrays(gl.TRIANGLES, index, n);
      }
    }
  }
}

const game_client = new (class {
  constructor() {
    const canvas = document.querySelector("canvas");
    const gl = canvas.getContext("webgl2");
    if (gl === null) {
      alert("Failed to initialize WebGL");
      return;
    }

    this.player_id = -1;

    this.camera = new (class {
      constructor() {
        this.eye = vec3.create();
        this.target = vec3.create();
        this.trans = mat4.create();
        this.pos_buffer = new Array();
      }

      setup(eye, target, follow) {
        // camera always follows player for now
        this.eye = eye;
        this.target = target;
      }

      update(trans) {
        this.trans = trans;
      }

      get_stabilized_pos(pos) {
        const n = 90;
        if (this.pos_buffer.length >= n) {
          // remove the oldest (first) position
          this.pos_buffer.shift();
        }
        this.pos_buffer.push(pos);
        const p = this.pos_buffer.reduce((acc, cur) => vec3.add(acc, cur, acc));
        return vec3.scale(p, p, 1.0 / this.pos_buffer.length);
      }

      get_view_matrix() {
        let eye = vec3.create();
        vec3.transformMat4(eye, this.eye, this.trans);
        eye = this.get_stabilized_pos(eye);
        
        const target = vec3.create();
        vec3.transformMat4(target, this.target, this.trans);
        
        const m = mat4.create();
        mat4.lookAt(m, eye, target, [0, 1, 0]);
        return m;
      }
    });

    this.renderer = new Renderer(gl, this.camera);
    this.renderer.setup();
    window.addEventListener('resize', function() { 
      game_client.renderer.reshape(); 
    }, false);

    this.cursor_last_x = 0;
    this.cursor_last_y = 0;
    this.cursor_cur_x = 0;
    this.cursor_cur_y = 0;

    const hostname = location.hostname;
    const port = "9001"; // change this if the game server is listening on a different port
    const url = hostname + ":" + port
    this.socket = this.connect(url);
    this.init_ctrls();
  }

  init_ctrls() {  
    document.body.addEventListener("touchstart", function(e){
      e.preventDefault();
      e.stopImmediatePropagation();
      game_client.cursor_cur_x = e.touches[0].clientX;
      game_client.cursor_cur_y = e.touches[0].clientY;
      game_client.cursor_last_x = e.touches[0].clientX;
      game_client.cursor_last_y = e.touches[0].clientY;
    }, { passive: false });
    document.body.addEventListener("touchmove", function(e){
      game_client.cursor_cur_x = e.touches[0].clientX;
      game_client.cursor_cur_y = e.touches[0].clientY;
    }, false);
    document.body.addEventListener("touchcancel", function(e){
      game_client.cursor_cur_x = 0;
      game_client.cursor_cur_y = 0;
      game_client.cursor_last_x = 0;
      game_client.cursor_last_y = 0;
    }, false);
    document.body.addEventListener("touchend", function(e){
      game_client.cursor_cur_x = 0;
      game_client.cursor_cur_y = 0;
      game_client.cursor_last_x = 0;
      game_client.cursor_last_y = 0;
    }, false);    
    
    document.body.onkeydown = function(e){
      if(key_map.has(e.code)) {
        keyboard.add(key_map.get(e.code));
      }
    };
  
    document.body.onkeyup = function(e){
      if(key_map.has(e.code)) {
        keyboard.delete(key_map.get(e.code));
      }
    };
  
    document.body.onmousedown = function(e){
      if (e.button == 0) mouse.add(0);
      else if (e.button == 2) mouse.add(1);
    }
    
    document.body.onmouseup = function(e){
      if (e.button == 0) mouse.delete(0);
      else if (e.button == 2) mouse.delete(1);
    }
  
    document.body.onmouseleave = function(e){
      mouse.clear();
    }  
  }

  connect(url) {
    const socket = new WebSocket("ws://" + url);
    socket.onerror = (event) => { alert("Failed to connect to game server @ " + url); }
    socket.onopen = (event) => {};  
    socket.onmessage = (event) => {
      const n = event.data.length;
      const msg = JSON.parse(event.data);
      switch (msg.cmd) {
        // initial setup messages
        case "set_player_id":
          this.player_id = msg.player_id;
          break;
        case "setup_camera":
          this.camera.setup(msg.eye, msg.target, msg.follow);
          break;
        case "add_texture":
          const image = Uint8Array.from(atob(msg.data), (c) => c.charCodeAt(0));
          this.renderer.add_texture(msg.id, msg.width, msg.height, image);
          break;
        
        // messages in an update cycle
        case "get_controller":
          const keys = [];
          for(let key of keyboard.values()) {
            keys.push(key);
          }
          const buttons = [];
          for(let button of mouse.values()) {
            buttons.push(button);
          }
          
          const delta_x = game_client.cursor_cur_x - game_client.cursor_last_x; 
          const delta_y = game_client.cursor_cur_y - game_client.cursor_last_y;
          // don't move and turn at the same time
          if ((delta_x * delta_x) > (delta_y * delta_y)) {
            // x movement is more dominant
            if (delta_x < -0.5) keys.push(key_map.get("ArrowLeft"));
            if (delta_x > 0.5) keys.push(key_map.get("ArrowRight"));
          } else {
            // y movement is more dominant
            if (delta_y > 0.5) keys.push(key_map.get("ArrowDown"));
            if (delta_y < -0.5) keys.push(key_map.get("ArrowUp"));
          }

          const ctlr = {
            "keyboard": keys,
            "mouse": buttons,
            "cursor_cur_pos": [0,0],
            "cursor_last_pos": [0,0],
            "cursor_scroll_pos": [0,0]
          };
          socket.send(JSON.stringify(ctlr));
          break;
        case "set_player_transform":
          if (this.player_id == msg.player_id) {
            this.camera.update(msg.trans);
          }
          break;
        case "add_shape":
          this.renderer.add_shape(msg.shape_id, msg.descriptor);
          break;
        case "update_shape":
          this.renderer.update_shape(msg.shape_id, msg.trans);
          break;
        case "remove_shape":
          this.renderer.remove_shape(msg.shape_id);
          break;      
        case "end_update":
          socket.send(JSON.stringify({"continue": true}));
          // reder one frame
          this.renderer.draw();
          break;

        // terminating message
        case "end":
          break; 
      }
    };
    return socket;
  }
});
