////////////////////////////////////////////////////////////////////////////////
/// DISCLAIMER
///
/// Copyright 2021 ArangoDB GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is ArangoDB GmbH, Cologne, Germany
///
/// @author Andrey Abramov
////////////////////////////////////////////////////////////////////////////////

#pragma once

#include <frozen/map.h>
#include <frozen/set.h>

#include <cstdint>

namespace irs::utf8_utils {

inline constexpr frozen::set<uint32_t, 10> kWhiteSpaceTable = {
  0x9, 0x20, 0x85, 0xa0, 0x1680, 0x2000, 0x2028, 0x202f, 0x205f, 0x3000};

inline constexpr frozen::map<uint32_t, uint16_t, 3566> kGeneralCategoryTable = {
  {0x00, 0x4363},     {0x20, 0x5a73},     {0x21, 0x506f},    {0x24, 0x5363},
  {0x25, 0x506f},     {0x28, 0x5073},     {0x29, 0x5065},    {0x2a, 0x506f},
  {0x2b, 0x536d},     {0x2c, 0x506f},     {0x2d, 0x5064},    {0x2e, 0x506f},
  {0x30, 0x4e64},     {0x3a, 0x506f},     {0x3c, 0x536d},    {0x3f, 0x506f},
  {0x41, 0x4c75},     {0x5b, 0x5073},     {0x5c, 0x506f},    {0x5d, 0x5065},
  {0x5e, 0x536b},     {0x5f, 0x5063},     {0x60, 0x536b},    {0x61, 0x4c6c},
  {0x7b, 0x5073},     {0x7c, 0x536d},     {0x7d, 0x5065},    {0x7e, 0x536d},
  {0x7f, 0x4363},     {0xa0, 0x5a73},     {0xa1, 0x506f},    {0xa2, 0x5363},
  {0xa6, 0x536f},     {0xa7, 0x506f},     {0xa8, 0x536b},    {0xa9, 0x536f},
  {0xaa, 0x4c6f},     {0xab, 0x5069},     {0xac, 0x536d},    {0xad, 0x4366},
  {0xae, 0x536f},     {0xaf, 0x536b},     {0xb0, 0x536f},    {0xb1, 0x536d},
  {0xb2, 0x4e6f},     {0xb4, 0x536b},     {0xb5, 0x4c6c},    {0xb6, 0x506f},
  {0xb8, 0x536b},     {0xb9, 0x4e6f},     {0xba, 0x4c6f},    {0xbb, 0x5066},
  {0xbc, 0x4e6f},     {0xbf, 0x506f},     {0xc0, 0x4c75},    {0xd7, 0x536d},
  {0xd8, 0x4c75},     {0xdf, 0x4c6c},     {0xf7, 0x536d},    {0xf8, 0x4c6c},
  {0x100, 0x4c75},    {0x101, 0x4c6c},    {0x102, 0x4c75},   {0x103, 0x4c6c},
  {0x104, 0x4c75},    {0x105, 0x4c6c},    {0x106, 0x4c75},   {0x107, 0x4c6c},
  {0x108, 0x4c75},    {0x109, 0x4c6c},    {0x10a, 0x4c75},   {0x10b, 0x4c6c},
  {0x10c, 0x4c75},    {0x10d, 0x4c6c},    {0x10e, 0x4c75},   {0x10f, 0x4c6c},
  {0x110, 0x4c75},    {0x111, 0x4c6c},    {0x112, 0x4c75},   {0x113, 0x4c6c},
  {0x114, 0x4c75},    {0x115, 0x4c6c},    {0x116, 0x4c75},   {0x117, 0x4c6c},
  {0x118, 0x4c75},    {0x119, 0x4c6c},    {0x11a, 0x4c75},   {0x11b, 0x4c6c},
  {0x11c, 0x4c75},    {0x11d, 0x4c6c},    {0x11e, 0x4c75},   {0x11f, 0x4c6c},
  {0x120, 0x4c75},    {0x121, 0x4c6c},    {0x122, 0x4c75},   {0x123, 0x4c6c},
  {0x124, 0x4c75},    {0x125, 0x4c6c},    {0x126, 0x4c75},   {0x127, 0x4c6c},
  {0x128, 0x4c75},    {0x129, 0x4c6c},    {0x12a, 0x4c75},   {0x12b, 0x4c6c},
  {0x12c, 0x4c75},    {0x12d, 0x4c6c},    {0x12e, 0x4c75},   {0x12f, 0x4c6c},
  {0x130, 0x4c75},    {0x131, 0x4c6c},    {0x132, 0x4c75},   {0x133, 0x4c6c},
  {0x134, 0x4c75},    {0x135, 0x4c6c},    {0x136, 0x4c75},   {0x137, 0x4c6c},
  {0x139, 0x4c75},    {0x13a, 0x4c6c},    {0x13b, 0x4c75},   {0x13c, 0x4c6c},
  {0x13d, 0x4c75},    {0x13e, 0x4c6c},    {0x13f, 0x4c75},   {0x140, 0x4c6c},
  {0x141, 0x4c75},    {0x142, 0x4c6c},    {0x143, 0x4c75},   {0x144, 0x4c6c},
  {0x145, 0x4c75},    {0x146, 0x4c6c},    {0x147, 0x4c75},   {0x148, 0x4c6c},
  {0x14a, 0x4c75},    {0x14b, 0x4c6c},    {0x14c, 0x4c75},   {0x14d, 0x4c6c},
  {0x14e, 0x4c75},    {0x14f, 0x4c6c},    {0x150, 0x4c75},   {0x151, 0x4c6c},
  {0x152, 0x4c75},    {0x153, 0x4c6c},    {0x154, 0x4c75},   {0x155, 0x4c6c},
  {0x156, 0x4c75},    {0x157, 0x4c6c},    {0x158, 0x4c75},   {0x159, 0x4c6c},
  {0x15a, 0x4c75},    {0x15b, 0x4c6c},    {0x15c, 0x4c75},   {0x15d, 0x4c6c},
  {0x15e, 0x4c75},    {0x15f, 0x4c6c},    {0x160, 0x4c75},   {0x161, 0x4c6c},
  {0x162, 0x4c75},    {0x163, 0x4c6c},    {0x164, 0x4c75},   {0x165, 0x4c6c},
  {0x166, 0x4c75},    {0x167, 0x4c6c},    {0x168, 0x4c75},   {0x169, 0x4c6c},
  {0x16a, 0x4c75},    {0x16b, 0x4c6c},    {0x16c, 0x4c75},   {0x16d, 0x4c6c},
  {0x16e, 0x4c75},    {0x16f, 0x4c6c},    {0x170, 0x4c75},   {0x171, 0x4c6c},
  {0x172, 0x4c75},    {0x173, 0x4c6c},    {0x174, 0x4c75},   {0x175, 0x4c6c},
  {0x176, 0x4c75},    {0x177, 0x4c6c},    {0x178, 0x4c75},   {0x17a, 0x4c6c},
  {0x17b, 0x4c75},    {0x17c, 0x4c6c},    {0x17d, 0x4c75},   {0x17e, 0x4c6c},
  {0x181, 0x4c75},    {0x183, 0x4c6c},    {0x184, 0x4c75},   {0x185, 0x4c6c},
  {0x186, 0x4c75},    {0x188, 0x4c6c},    {0x189, 0x4c75},   {0x18c, 0x4c6c},
  {0x18e, 0x4c75},    {0x192, 0x4c6c},    {0x193, 0x4c75},   {0x195, 0x4c6c},
  {0x196, 0x4c75},    {0x199, 0x4c6c},    {0x19c, 0x4c75},   {0x19e, 0x4c6c},
  {0x19f, 0x4c75},    {0x1a1, 0x4c6c},    {0x1a2, 0x4c75},   {0x1a3, 0x4c6c},
  {0x1a4, 0x4c75},    {0x1a5, 0x4c6c},    {0x1a6, 0x4c75},   {0x1a8, 0x4c6c},
  {0x1a9, 0x4c75},    {0x1aa, 0x4c6c},    {0x1ac, 0x4c75},   {0x1ad, 0x4c6c},
  {0x1ae, 0x4c75},    {0x1b0, 0x4c6c},    {0x1b1, 0x4c75},   {0x1b4, 0x4c6c},
  {0x1b5, 0x4c75},    {0x1b6, 0x4c6c},    {0x1b7, 0x4c75},   {0x1b9, 0x4c6c},
  {0x1bb, 0x4c6f},    {0x1bc, 0x4c75},    {0x1bd, 0x4c6c},   {0x1c0, 0x4c6f},
  {0x1c4, 0x4c75},    {0x1c5, 0x4c74},    {0x1c6, 0x4c6c},   {0x1c7, 0x4c75},
  {0x1c8, 0x4c74},    {0x1c9, 0x4c6c},    {0x1ca, 0x4c75},   {0x1cb, 0x4c74},
  {0x1cc, 0x4c6c},    {0x1cd, 0x4c75},    {0x1ce, 0x4c6c},   {0x1cf, 0x4c75},
  {0x1d0, 0x4c6c},    {0x1d1, 0x4c75},    {0x1d2, 0x4c6c},   {0x1d3, 0x4c75},
  {0x1d4, 0x4c6c},    {0x1d5, 0x4c75},    {0x1d6, 0x4c6c},   {0x1d7, 0x4c75},
  {0x1d8, 0x4c6c},    {0x1d9, 0x4c75},    {0x1da, 0x4c6c},   {0x1db, 0x4c75},
  {0x1dc, 0x4c6c},    {0x1de, 0x4c75},    {0x1df, 0x4c6c},   {0x1e0, 0x4c75},
  {0x1e1, 0x4c6c},    {0x1e2, 0x4c75},    {0x1e3, 0x4c6c},   {0x1e4, 0x4c75},
  {0x1e5, 0x4c6c},    {0x1e6, 0x4c75},    {0x1e7, 0x4c6c},   {0x1e8, 0x4c75},
  {0x1e9, 0x4c6c},    {0x1ea, 0x4c75},    {0x1eb, 0x4c6c},   {0x1ec, 0x4c75},
  {0x1ed, 0x4c6c},    {0x1ee, 0x4c75},    {0x1ef, 0x4c6c},   {0x1f1, 0x4c75},
  {0x1f2, 0x4c74},    {0x1f3, 0x4c6c},    {0x1f4, 0x4c75},   {0x1f5, 0x4c6c},
  {0x1f6, 0x4c75},    {0x1f9, 0x4c6c},    {0x1fa, 0x4c75},   {0x1fb, 0x4c6c},
  {0x1fc, 0x4c75},    {0x1fd, 0x4c6c},    {0x1fe, 0x4c75},   {0x1ff, 0x4c6c},
  {0x200, 0x4c75},    {0x201, 0x4c6c},    {0x202, 0x4c75},   {0x203, 0x4c6c},
  {0x204, 0x4c75},    {0x205, 0x4c6c},    {0x206, 0x4c75},   {0x207, 0x4c6c},
  {0x208, 0x4c75},    {0x209, 0x4c6c},    {0x20a, 0x4c75},   {0x20b, 0x4c6c},
  {0x20c, 0x4c75},    {0x20d, 0x4c6c},    {0x20e, 0x4c75},   {0x20f, 0x4c6c},
  {0x210, 0x4c75},    {0x211, 0x4c6c},    {0x212, 0x4c75},   {0x213, 0x4c6c},
  {0x214, 0x4c75},    {0x215, 0x4c6c},    {0x216, 0x4c75},   {0x217, 0x4c6c},
  {0x218, 0x4c75},    {0x219, 0x4c6c},    {0x21a, 0x4c75},   {0x21b, 0x4c6c},
  {0x21c, 0x4c75},    {0x21d, 0x4c6c},    {0x21e, 0x4c75},   {0x21f, 0x4c6c},
  {0x220, 0x4c75},    {0x221, 0x4c6c},    {0x222, 0x4c75},   {0x223, 0x4c6c},
  {0x224, 0x4c75},    {0x225, 0x4c6c},    {0x226, 0x4c75},   {0x227, 0x4c6c},
  {0x228, 0x4c75},    {0x229, 0x4c6c},    {0x22a, 0x4c75},   {0x22b, 0x4c6c},
  {0x22c, 0x4c75},    {0x22d, 0x4c6c},    {0x22e, 0x4c75},   {0x22f, 0x4c6c},
  {0x230, 0x4c75},    {0x231, 0x4c6c},    {0x232, 0x4c75},   {0x233, 0x4c6c},
  {0x23a, 0x4c75},    {0x23c, 0x4c6c},    {0x23d, 0x4c75},   {0x23f, 0x4c6c},
  {0x241, 0x4c75},    {0x242, 0x4c6c},    {0x243, 0x4c75},   {0x247, 0x4c6c},
  {0x248, 0x4c75},    {0x249, 0x4c6c},    {0x24a, 0x4c75},   {0x24b, 0x4c6c},
  {0x24c, 0x4c75},    {0x24d, 0x4c6c},    {0x24e, 0x4c75},   {0x24f, 0x4c6c},
  {0x294, 0x4c6f},    {0x295, 0x4c6c},    {0x2b0, 0x4c6d},   {0x2c2, 0x536b},
  {0x2c6, 0x4c6d},    {0x2d2, 0x536b},    {0x2e0, 0x4c6d},   {0x2e5, 0x536b},
  {0x2ec, 0x4c6d},    {0x2ed, 0x536b},    {0x2ee, 0x4c6d},   {0x2ef, 0x536b},
  {0x300, 0x4d6e},    {0x370, 0x4c75},    {0x371, 0x4c6c},   {0x372, 0x4c75},
  {0x373, 0x4c6c},    {0x374, 0x4c6d},    {0x375, 0x536b},   {0x376, 0x4c75},
  {0x377, 0x4c6c},    {0x378, 0x436e},    {0x37a, 0x4c6d},   {0x37b, 0x4c6c},
  {0x37e, 0x506f},    {0x37f, 0x4c75},    {0x380, 0x436e},   {0x384, 0x536b},
  {0x386, 0x4c75},    {0x387, 0x506f},    {0x388, 0x4c75},   {0x38b, 0x436e},
  {0x38c, 0x4c75},    {0x38d, 0x436e},    {0x38e, 0x4c75},   {0x390, 0x4c6c},
  {0x391, 0x4c75},    {0x3a2, 0x436e},    {0x3a3, 0x4c75},   {0x3ac, 0x4c6c},
  {0x3cf, 0x4c75},    {0x3d0, 0x4c6c},    {0x3d2, 0x4c75},   {0x3d5, 0x4c6c},
  {0x3d8, 0x4c75},    {0x3d9, 0x4c6c},    {0x3da, 0x4c75},   {0x3db, 0x4c6c},
  {0x3dc, 0x4c75},    {0x3dd, 0x4c6c},    {0x3de, 0x4c75},   {0x3df, 0x4c6c},
  {0x3e0, 0x4c75},    {0x3e1, 0x4c6c},    {0x3e2, 0x4c75},   {0x3e3, 0x4c6c},
  {0x3e4, 0x4c75},    {0x3e5, 0x4c6c},    {0x3e6, 0x4c75},   {0x3e7, 0x4c6c},
  {0x3e8, 0x4c75},    {0x3e9, 0x4c6c},    {0x3ea, 0x4c75},   {0x3eb, 0x4c6c},
  {0x3ec, 0x4c75},    {0x3ed, 0x4c6c},    {0x3ee, 0x4c75},   {0x3ef, 0x4c6c},
  {0x3f4, 0x4c75},    {0x3f5, 0x4c6c},    {0x3f6, 0x536d},   {0x3f7, 0x4c75},
  {0x3f8, 0x4c6c},    {0x3f9, 0x4c75},    {0x3fb, 0x4c6c},   {0x3fd, 0x4c75},
  {0x430, 0x4c6c},    {0x460, 0x4c75},    {0x461, 0x4c6c},   {0x462, 0x4c75},
  {0x463, 0x4c6c},    {0x464, 0x4c75},    {0x465, 0x4c6c},   {0x466, 0x4c75},
  {0x467, 0x4c6c},    {0x468, 0x4c75},    {0x469, 0x4c6c},   {0x46a, 0x4c75},
  {0x46b, 0x4c6c},    {0x46c, 0x4c75},    {0x46d, 0x4c6c},   {0x46e, 0x4c75},
  {0x46f, 0x4c6c},    {0x470, 0x4c75},    {0x471, 0x4c6c},   {0x472, 0x4c75},
  {0x473, 0x4c6c},    {0x474, 0x4c75},    {0x475, 0x4c6c},   {0x476, 0x4c75},
  {0x477, 0x4c6c},    {0x478, 0x4c75},    {0x479, 0x4c6c},   {0x47a, 0x4c75},
  {0x47b, 0x4c6c},    {0x47c, 0x4c75},    {0x47d, 0x4c6c},   {0x47e, 0x4c75},
  {0x47f, 0x4c6c},    {0x480, 0x4c75},    {0x481, 0x4c6c},   {0x482, 0x536f},
  {0x483, 0x4d6e},    {0x488, 0x4d65},    {0x48a, 0x4c75},   {0x48b, 0x4c6c},
  {0x48c, 0x4c75},    {0x48d, 0x4c6c},    {0x48e, 0x4c75},   {0x48f, 0x4c6c},
  {0x490, 0x4c75},    {0x491, 0x4c6c},    {0x492, 0x4c75},   {0x493, 0x4c6c},
  {0x494, 0x4c75},    {0x495, 0x4c6c},    {0x496, 0x4c75},   {0x497, 0x4c6c},
  {0x498, 0x4c75},    {0x499, 0x4c6c},    {0x49a, 0x4c75},   {0x49b, 0x4c6c},
  {0x49c, 0x4c75},    {0x49d, 0x4c6c},    {0x49e, 0x4c75},   {0x49f, 0x4c6c},
  {0x4a0, 0x4c75},    {0x4a1, 0x4c6c},    {0x4a2, 0x4c75},   {0x4a3, 0x4c6c},
  {0x4a4, 0x4c75},    {0x4a5, 0x4c6c},    {0x4a6, 0x4c75},   {0x4a7, 0x4c6c},
  {0x4a8, 0x4c75},    {0x4a9, 0x4c6c},    {0x4aa, 0x4c75},   {0x4ab, 0x4c6c},
  {0x4ac, 0x4c75},    {0x4ad, 0x4c6c},    {0x4ae, 0x4c75},   {0x4af, 0x4c6c},
  {0x4b0, 0x4c75},    {0x4b1, 0x4c6c},    {0x4b2, 0x4c75},   {0x4b3, 0x4c6c},
  {0x4b4, 0x4c75},    {0x4b5, 0x4c6c},    {0x4b6, 0x4c75},   {0x4b7, 0x4c6c},
  {0x4b8, 0x4c75},    {0x4b9, 0x4c6c},    {0x4ba, 0x4c75},   {0x4bb, 0x4c6c},
  {0x4bc, 0x4c75},    {0x4bd, 0x4c6c},    {0x4be, 0x4c75},   {0x4bf, 0x4c6c},
  {0x4c0, 0x4c75},    {0x4c2, 0x4c6c},    {0x4c3, 0x4c75},   {0x4c4, 0x4c6c},
  {0x4c5, 0x4c75},    {0x4c6, 0x4c6c},    {0x4c7, 0x4c75},   {0x4c8, 0x4c6c},
  {0x4c9, 0x4c75},    {0x4ca, 0x4c6c},    {0x4cb, 0x4c75},   {0x4cc, 0x4c6c},
  {0x4cd, 0x4c75},    {0x4ce, 0x4c6c},    {0x4d0, 0x4c75},   {0x4d1, 0x4c6c},
  {0x4d2, 0x4c75},    {0x4d3, 0x4c6c},    {0x4d4, 0x4c75},   {0x4d5, 0x4c6c},
  {0x4d6, 0x4c75},    {0x4d7, 0x4c6c},    {0x4d8, 0x4c75},   {0x4d9, 0x4c6c},
  {0x4da, 0x4c75},    {0x4db, 0x4c6c},    {0x4dc, 0x4c75},   {0x4dd, 0x4c6c},
  {0x4de, 0x4c75},    {0x4df, 0x4c6c},    {0x4e0, 0x4c75},   {0x4e1, 0x4c6c},
  {0x4e2, 0x4c75},    {0x4e3, 0x4c6c},    {0x4e4, 0x4c75},   {0x4e5, 0x4c6c},
  {0x4e6, 0x4c75},    {0x4e7, 0x4c6c},    {0x4e8, 0x4c75},   {0x4e9, 0x4c6c},
  {0x4ea, 0x4c75},    {0x4eb, 0x4c6c},    {0x4ec, 0x4c75},   {0x4ed, 0x4c6c},
  {0x4ee, 0x4c75},    {0x4ef, 0x4c6c},    {0x4f0, 0x4c75},   {0x4f1, 0x4c6c},
  {0x4f2, 0x4c75},    {0x4f3, 0x4c6c},    {0x4f4, 0x4c75},   {0x4f5, 0x4c6c},
  {0x4f6, 0x4c75},    {0x4f7, 0x4c6c},    {0x4f8, 0x4c75},   {0x4f9, 0x4c6c},
  {0x4fa, 0x4c75},    {0x4fb, 0x4c6c},    {0x4fc, 0x4c75},   {0x4fd, 0x4c6c},
  {0x4fe, 0x4c75},    {0x4ff, 0x4c6c},    {0x500, 0x4c75},   {0x501, 0x4c6c},
  {0x502, 0x4c75},    {0x503, 0x4c6c},    {0x504, 0x4c75},   {0x505, 0x4c6c},
  {0x506, 0x4c75},    {0x507, 0x4c6c},    {0x508, 0x4c75},   {0x509, 0x4c6c},
  {0x50a, 0x4c75},    {0x50b, 0x4c6c},    {0x50c, 0x4c75},   {0x50d, 0x4c6c},
  {0x50e, 0x4c75},    {0x50f, 0x4c6c},    {0x510, 0x4c75},   {0x511, 0x4c6c},
  {0x512, 0x4c75},    {0x513, 0x4c6c},    {0x514, 0x4c75},   {0x515, 0x4c6c},
  {0x516, 0x4c75},    {0x517, 0x4c6c},    {0x518, 0x4c75},   {0x519, 0x4c6c},
  {0x51a, 0x4c75},    {0x51b, 0x4c6c},    {0x51c, 0x4c75},   {0x51d, 0x4c6c},
  {0x51e, 0x4c75},    {0x51f, 0x4c6c},    {0x520, 0x4c75},   {0x521, 0x4c6c},
  {0x522, 0x4c75},    {0x523, 0x4c6c},    {0x524, 0x4c75},   {0x525, 0x4c6c},
  {0x526, 0x4c75},    {0x527, 0x4c6c},    {0x528, 0x4c75},   {0x529, 0x4c6c},
  {0x52a, 0x4c75},    {0x52b, 0x4c6c},    {0x52c, 0x4c75},   {0x52d, 0x4c6c},
  {0x52e, 0x4c75},    {0x52f, 0x4c6c},    {0x530, 0x436e},   {0x531, 0x4c75},
  {0x557, 0x436e},    {0x559, 0x4c6d},    {0x55a, 0x506f},   {0x560, 0x436e},
  {0x561, 0x4c6c},    {0x588, 0x436e},    {0x589, 0x506f},   {0x58a, 0x5064},
  {0x58b, 0x436e},    {0x58d, 0x536f},    {0x58f, 0x5363},   {0x590, 0x436e},
  {0x591, 0x4d6e},    {0x5be, 0x5064},    {0x5bf, 0x4d6e},   {0x5c0, 0x506f},
  {0x5c1, 0x4d6e},    {0x5c3, 0x506f},    {0x5c4, 0x4d6e},   {0x5c6, 0x506f},
  {0x5c7, 0x4d6e},    {0x5c8, 0x436e},    {0x5d0, 0x4c6f},   {0x5eb, 0x436e},
  {0x5f0, 0x4c6f},    {0x5f3, 0x506f},    {0x5f5, 0x436e},   {0x600, 0x4366},
  {0x606, 0x536d},    {0x609, 0x506f},    {0x60b, 0x5363},   {0x60c, 0x506f},
  {0x60e, 0x536f},    {0x610, 0x4d6e},    {0x61b, 0x506f},   {0x61c, 0x4366},
  {0x61d, 0x436e},    {0x61e, 0x506f},    {0x620, 0x4c6f},   {0x640, 0x4c6d},
  {0x641, 0x4c6f},    {0x64b, 0x4d6e},    {0x660, 0x4e64},   {0x66a, 0x506f},
  {0x66e, 0x4c6f},    {0x670, 0x4d6e},    {0x671, 0x4c6f},   {0x6d4, 0x506f},
  {0x6d5, 0x4c6f},    {0x6d6, 0x4d6e},    {0x6dd, 0x4366},   {0x6de, 0x536f},
  {0x6df, 0x4d6e},    {0x6e5, 0x4c6d},    {0x6e7, 0x4d6e},   {0x6e9, 0x536f},
  {0x6ea, 0x4d6e},    {0x6ee, 0x4c6f},    {0x6f0, 0x4e64},   {0x6fa, 0x4c6f},
  {0x6fd, 0x536f},    {0x6ff, 0x4c6f},    {0x700, 0x506f},   {0x70e, 0x436e},
  {0x70f, 0x4366},    {0x710, 0x4c6f},    {0x711, 0x4d6e},   {0x712, 0x4c6f},
  {0x730, 0x4d6e},    {0x74b, 0x436e},    {0x74d, 0x4c6f},   {0x7a6, 0x4d6e},
  {0x7b1, 0x4c6f},    {0x7b2, 0x436e},    {0x7c0, 0x4e64},   {0x7ca, 0x4c6f},
  {0x7eb, 0x4d6e},    {0x7f4, 0x4c6d},    {0x7f6, 0x536f},   {0x7f7, 0x506f},
  {0x7fa, 0x4c6d},    {0x7fb, 0x436e},    {0x800, 0x4c6f},   {0x816, 0x4d6e},
  {0x81a, 0x4c6d},    {0x81b, 0x4d6e},    {0x824, 0x4c6d},   {0x825, 0x4d6e},
  {0x828, 0x4c6d},    {0x829, 0x4d6e},    {0x82e, 0x436e},   {0x830, 0x506f},
  {0x83f, 0x436e},    {0x840, 0x4c6f},    {0x859, 0x4d6e},   {0x85c, 0x436e},
  {0x85e, 0x506f},    {0x85f, 0x436e},    {0x8a0, 0x4c6f},   {0x8b5, 0x436e},
  {0x8e3, 0x4d6e},    {0x903, 0x4d63},    {0x904, 0x4c6f},   {0x93a, 0x4d6e},
  {0x93b, 0x4d63},    {0x93c, 0x4d6e},    {0x93d, 0x4c6f},   {0x93e, 0x4d63},
  {0x941, 0x4d6e},    {0x949, 0x4d63},    {0x94d, 0x4d6e},   {0x94e, 0x4d63},
  {0x950, 0x4c6f},    {0x951, 0x4d6e},    {0x958, 0x4c6f},   {0x962, 0x4d6e},
  {0x964, 0x506f},    {0x966, 0x4e64},    {0x970, 0x506f},   {0x971, 0x4c6d},
  {0x972, 0x4c6f},    {0x981, 0x4d6e},    {0x982, 0x4d63},   {0x984, 0x436e},
  {0x985, 0x4c6f},    {0x98d, 0x436e},    {0x98f, 0x4c6f},   {0x991, 0x436e},
  {0x993, 0x4c6f},    {0x9a9, 0x436e},    {0x9aa, 0x4c6f},   {0x9b1, 0x436e},
  {0x9b2, 0x4c6f},    {0x9b3, 0x436e},    {0x9b6, 0x4c6f},   {0x9ba, 0x436e},
  {0x9bc, 0x4d6e},    {0x9bd, 0x4c6f},    {0x9be, 0x4d63},   {0x9c1, 0x4d6e},
  {0x9c5, 0x436e},    {0x9c7, 0x4d63},    {0x9c9, 0x436e},   {0x9cb, 0x4d63},
  {0x9cd, 0x4d6e},    {0x9ce, 0x4c6f},    {0x9cf, 0x436e},   {0x9d7, 0x4d63},
  {0x9d8, 0x436e},    {0x9dc, 0x4c6f},    {0x9de, 0x436e},   {0x9df, 0x4c6f},
  {0x9e2, 0x4d6e},    {0x9e4, 0x436e},    {0x9e6, 0x4e64},   {0x9f0, 0x4c6f},
  {0x9f2, 0x5363},    {0x9f4, 0x4e6f},    {0x9fa, 0x536f},   {0x9fb, 0x5363},
  {0x9fc, 0x436e},    {0xa01, 0x4d6e},    {0xa03, 0x4d63},   {0xa04, 0x436e},
  {0xa05, 0x4c6f},    {0xa0b, 0x436e},    {0xa0f, 0x4c6f},   {0xa11, 0x436e},
  {0xa13, 0x4c6f},    {0xa29, 0x436e},    {0xa2a, 0x4c6f},   {0xa31, 0x436e},
  {0xa32, 0x4c6f},    {0xa34, 0x436e},    {0xa35, 0x4c6f},   {0xa37, 0x436e},
  {0xa38, 0x4c6f},    {0xa3a, 0x436e},    {0xa3c, 0x4d6e},   {0xa3d, 0x436e},
  {0xa3e, 0x4d63},    {0xa41, 0x4d6e},    {0xa43, 0x436e},   {0xa47, 0x4d6e},
  {0xa49, 0x436e},    {0xa4b, 0x4d6e},    {0xa4e, 0x436e},   {0xa51, 0x4d6e},
  {0xa52, 0x436e},    {0xa59, 0x4c6f},    {0xa5d, 0x436e},   {0xa5e, 0x4c6f},
  {0xa5f, 0x436e},    {0xa66, 0x4e64},    {0xa70, 0x4d6e},   {0xa72, 0x4c6f},
  {0xa75, 0x4d6e},    {0xa76, 0x436e},    {0xa81, 0x4d6e},   {0xa83, 0x4d63},
  {0xa84, 0x436e},    {0xa85, 0x4c6f},    {0xa8e, 0x436e},   {0xa8f, 0x4c6f},
  {0xa92, 0x436e},    {0xa93, 0x4c6f},    {0xaa9, 0x436e},   {0xaaa, 0x4c6f},
  {0xab1, 0x436e},    {0xab2, 0x4c6f},    {0xab4, 0x436e},   {0xab5, 0x4c6f},
  {0xaba, 0x436e},    {0xabc, 0x4d6e},    {0xabd, 0x4c6f},   {0xabe, 0x4d63},
  {0xac1, 0x4d6e},    {0xac6, 0x436e},    {0xac7, 0x4d6e},   {0xac9, 0x4d63},
  {0xaca, 0x436e},    {0xacb, 0x4d63},    {0xacd, 0x4d6e},   {0xace, 0x436e},
  {0xad0, 0x4c6f},    {0xad1, 0x436e},    {0xae0, 0x4c6f},   {0xae2, 0x4d6e},
  {0xae4, 0x436e},    {0xae6, 0x4e64},    {0xaf0, 0x506f},   {0xaf1, 0x5363},
  {0xaf2, 0x436e},    {0xaf9, 0x4c6f},    {0xafa, 0x436e},   {0xb01, 0x4d6e},
  {0xb02, 0x4d63},    {0xb04, 0x436e},    {0xb05, 0x4c6f},   {0xb0d, 0x436e},
  {0xb0f, 0x4c6f},    {0xb11, 0x436e},    {0xb13, 0x4c6f},   {0xb29, 0x436e},
  {0xb2a, 0x4c6f},    {0xb31, 0x436e},    {0xb32, 0x4c6f},   {0xb34, 0x436e},
  {0xb35, 0x4c6f},    {0xb3a, 0x436e},    {0xb3c, 0x4d6e},   {0xb3d, 0x4c6f},
  {0xb3e, 0x4d63},    {0xb3f, 0x4d6e},    {0xb40, 0x4d63},   {0xb41, 0x4d6e},
  {0xb45, 0x436e},    {0xb47, 0x4d63},    {0xb49, 0x436e},   {0xb4b, 0x4d63},
  {0xb4d, 0x4d6e},    {0xb4e, 0x436e},    {0xb56, 0x4d6e},   {0xb57, 0x4d63},
  {0xb58, 0x436e},    {0xb5c, 0x4c6f},    {0xb5e, 0x436e},   {0xb5f, 0x4c6f},
  {0xb62, 0x4d6e},    {0xb64, 0x436e},    {0xb66, 0x4e64},   {0xb70, 0x536f},
  {0xb71, 0x4c6f},    {0xb72, 0x4e6f},    {0xb78, 0x436e},   {0xb82, 0x4d6e},
  {0xb83, 0x4c6f},    {0xb84, 0x436e},    {0xb85, 0x4c6f},   {0xb8b, 0x436e},
  {0xb8e, 0x4c6f},    {0xb91, 0x436e},    {0xb92, 0x4c6f},   {0xb96, 0x436e},
  {0xb99, 0x4c6f},    {0xb9b, 0x436e},    {0xb9c, 0x4c6f},   {0xb9d, 0x436e},
  {0xb9e, 0x4c6f},    {0xba0, 0x436e},    {0xba3, 0x4c6f},   {0xba5, 0x436e},
  {0xba8, 0x4c6f},    {0xbab, 0x436e},    {0xbae, 0x4c6f},   {0xbba, 0x436e},
  {0xbbe, 0x4d63},    {0xbc0, 0x4d6e},    {0xbc1, 0x4d63},   {0xbc3, 0x436e},
  {0xbc6, 0x4d63},    {0xbc9, 0x436e},    {0xbca, 0x4d63},   {0xbcd, 0x4d6e},
  {0xbce, 0x436e},    {0xbd0, 0x4c6f},    {0xbd1, 0x436e},   {0xbd7, 0x4d63},
  {0xbd8, 0x436e},    {0xbe6, 0x4e64},    {0xbf0, 0x4e6f},   {0xbf3, 0x536f},
  {0xbf9, 0x5363},    {0xbfa, 0x536f},    {0xbfb, 0x436e},   {0xc00, 0x4d6e},
  {0xc01, 0x4d63},    {0xc04, 0x436e},    {0xc05, 0x4c6f},   {0xc0d, 0x436e},
  {0xc0e, 0x4c6f},    {0xc11, 0x436e},    {0xc12, 0x4c6f},   {0xc29, 0x436e},
  {0xc2a, 0x4c6f},    {0xc3a, 0x436e},    {0xc3d, 0x4c6f},   {0xc3e, 0x4d6e},
  {0xc41, 0x4d63},    {0xc45, 0x436e},    {0xc46, 0x4d6e},   {0xc49, 0x436e},
  {0xc4a, 0x4d6e},    {0xc4e, 0x436e},    {0xc55, 0x4d6e},   {0xc57, 0x436e},
  {0xc58, 0x4c6f},    {0xc5b, 0x436e},    {0xc60, 0x4c6f},   {0xc62, 0x4d6e},
  {0xc64, 0x436e},    {0xc66, 0x4e64},    {0xc70, 0x436e},   {0xc78, 0x4e6f},
  {0xc7f, 0x536f},    {0xc80, 0x436e},    {0xc81, 0x4d6e},   {0xc82, 0x4d63},
  {0xc84, 0x436e},    {0xc85, 0x4c6f},    {0xc8d, 0x436e},   {0xc8e, 0x4c6f},
  {0xc91, 0x436e},    {0xc92, 0x4c6f},    {0xca9, 0x436e},   {0xcaa, 0x4c6f},
  {0xcb4, 0x436e},    {0xcb5, 0x4c6f},    {0xcba, 0x436e},   {0xcbc, 0x4d6e},
  {0xcbd, 0x4c6f},    {0xcbe, 0x4d63},    {0xcbf, 0x4d6e},   {0xcc0, 0x4d63},
  {0xcc5, 0x436e},    {0xcc6, 0x4d6e},    {0xcc7, 0x4d63},   {0xcc9, 0x436e},
  {0xcca, 0x4d63},    {0xccc, 0x4d6e},    {0xcce, 0x436e},   {0xcd5, 0x4d63},
  {0xcd7, 0x436e},    {0xcde, 0x4c6f},    {0xcdf, 0x436e},   {0xce0, 0x4c6f},
  {0xce2, 0x4d6e},    {0xce4, 0x436e},    {0xce6, 0x4e64},   {0xcf0, 0x436e},
  {0xcf1, 0x4c6f},    {0xcf3, 0x436e},    {0xd01, 0x4d6e},   {0xd02, 0x4d63},
  {0xd04, 0x436e},    {0xd05, 0x4c6f},    {0xd0d, 0x436e},   {0xd0e, 0x4c6f},
  {0xd11, 0x436e},    {0xd12, 0x4c6f},    {0xd3b, 0x436e},   {0xd3d, 0x4c6f},
  {0xd3e, 0x4d63},    {0xd41, 0x4d6e},    {0xd45, 0x436e},   {0xd46, 0x4d63},
  {0xd49, 0x436e},    {0xd4a, 0x4d63},    {0xd4d, 0x4d6e},   {0xd4e, 0x4c6f},
  {0xd4f, 0x436e},    {0xd57, 0x4d63},    {0xd58, 0x436e},   {0xd5f, 0x4c6f},
  {0xd62, 0x4d6e},    {0xd64, 0x436e},    {0xd66, 0x4e64},   {0xd70, 0x4e6f},
  {0xd76, 0x436e},    {0xd79, 0x536f},    {0xd7a, 0x4c6f},   {0xd80, 0x436e},
  {0xd82, 0x4d63},    {0xd84, 0x436e},    {0xd85, 0x4c6f},   {0xd97, 0x436e},
  {0xd9a, 0x4c6f},    {0xdb2, 0x436e},    {0xdb3, 0x4c6f},   {0xdbc, 0x436e},
  {0xdbd, 0x4c6f},    {0xdbe, 0x436e},    {0xdc0, 0x4c6f},   {0xdc7, 0x436e},
  {0xdca, 0x4d6e},    {0xdcb, 0x436e},    {0xdcf, 0x4d63},   {0xdd2, 0x4d6e},
  {0xdd5, 0x436e},    {0xdd6, 0x4d6e},    {0xdd7, 0x436e},   {0xdd8, 0x4d63},
  {0xde0, 0x436e},    {0xde6, 0x4e64},    {0xdf0, 0x436e},   {0xdf2, 0x4d63},
  {0xdf4, 0x506f},    {0xdf5, 0x436e},    {0xe01, 0x4c6f},   {0xe31, 0x4d6e},
  {0xe32, 0x4c6f},    {0xe34, 0x4d6e},    {0xe3b, 0x436e},   {0xe3f, 0x5363},
  {0xe40, 0x4c6f},    {0xe46, 0x4c6d},    {0xe47, 0x4d6e},   {0xe4f, 0x506f},
  {0xe50, 0x4e64},    {0xe5a, 0x506f},    {0xe5c, 0x436e},   {0xe81, 0x4c6f},
  {0xe83, 0x436e},    {0xe84, 0x4c6f},    {0xe85, 0x436e},   {0xe87, 0x4c6f},
  {0xe89, 0x436e},    {0xe8a, 0x4c6f},    {0xe8b, 0x436e},   {0xe8d, 0x4c6f},
  {0xe8e, 0x436e},    {0xe94, 0x4c6f},    {0xe98, 0x436e},   {0xe99, 0x4c6f},
  {0xea0, 0x436e},    {0xea1, 0x4c6f},    {0xea4, 0x436e},   {0xea5, 0x4c6f},
  {0xea6, 0x436e},    {0xea7, 0x4c6f},    {0xea8, 0x436e},   {0xeaa, 0x4c6f},
  {0xeac, 0x436e},    {0xead, 0x4c6f},    {0xeb1, 0x4d6e},   {0xeb2, 0x4c6f},
  {0xeb4, 0x4d6e},    {0xeba, 0x436e},    {0xebb, 0x4d6e},   {0xebd, 0x4c6f},
  {0xebe, 0x436e},    {0xec0, 0x4c6f},    {0xec5, 0x436e},   {0xec6, 0x4c6d},
  {0xec7, 0x436e},    {0xec8, 0x4d6e},    {0xece, 0x436e},   {0xed0, 0x4e64},
  {0xeda, 0x436e},    {0xedc, 0x4c6f},    {0xee0, 0x436e},   {0xf00, 0x4c6f},
  {0xf01, 0x536f},    {0xf04, 0x506f},    {0xf13, 0x536f},   {0xf14, 0x506f},
  {0xf15, 0x536f},    {0xf18, 0x4d6e},    {0xf1a, 0x536f},   {0xf20, 0x4e64},
  {0xf2a, 0x4e6f},    {0xf34, 0x536f},    {0xf35, 0x4d6e},   {0xf36, 0x536f},
  {0xf37, 0x4d6e},    {0xf38, 0x536f},    {0xf39, 0x4d6e},   {0xf3a, 0x5073},
  {0xf3b, 0x5065},    {0xf3c, 0x5073},    {0xf3d, 0x5065},   {0xf3e, 0x4d63},
  {0xf40, 0x4c6f},    {0xf48, 0x436e},    {0xf49, 0x4c6f},   {0xf6d, 0x436e},
  {0xf71, 0x4d6e},    {0xf7f, 0x4d63},    {0xf80, 0x4d6e},   {0xf85, 0x506f},
  {0xf86, 0x4d6e},    {0xf88, 0x4c6f},    {0xf8d, 0x4d6e},   {0xf98, 0x436e},
  {0xf99, 0x4d6e},    {0xfbd, 0x436e},    {0xfbe, 0x536f},   {0xfc6, 0x4d6e},
  {0xfc7, 0x536f},    {0xfcd, 0x436e},    {0xfce, 0x536f},   {0xfd0, 0x506f},
  {0xfd5, 0x536f},    {0xfd9, 0x506f},    {0xfdb, 0x436e},   {0x1000, 0x4c6f},
  {0x102b, 0x4d63},   {0x102d, 0x4d6e},   {0x1031, 0x4d63},  {0x1032, 0x4d6e},
  {0x1038, 0x4d63},   {0x1039, 0x4d6e},   {0x103b, 0x4d63},  {0x103d, 0x4d6e},
  {0x103f, 0x4c6f},   {0x1040, 0x4e64},   {0x104a, 0x506f},  {0x1050, 0x4c6f},
  {0x1056, 0x4d63},   {0x1058, 0x4d6e},   {0x105a, 0x4c6f},  {0x105e, 0x4d6e},
  {0x1061, 0x4c6f},   {0x1062, 0x4d63},   {0x1065, 0x4c6f},  {0x1067, 0x4d63},
  {0x106e, 0x4c6f},   {0x1071, 0x4d6e},   {0x1075, 0x4c6f},  {0x1082, 0x4d6e},
  {0x1083, 0x4d63},   {0x1085, 0x4d6e},   {0x1087, 0x4d63},  {0x108d, 0x4d6e},
  {0x108e, 0x4c6f},   {0x108f, 0x4d63},   {0x1090, 0x4e64},  {0x109a, 0x4d63},
  {0x109d, 0x4d6e},   {0x109e, 0x536f},   {0x10a0, 0x4c75},  {0x10c6, 0x436e},
  {0x10c7, 0x4c75},   {0x10c8, 0x436e},   {0x10cd, 0x4c75},  {0x10ce, 0x436e},
  {0x10d0, 0x4c6f},   {0x10fb, 0x506f},   {0x10fc, 0x4c6d},  {0x10fd, 0x4c6f},
  {0x1249, 0x436e},   {0x124a, 0x4c6f},   {0x124e, 0x436e},  {0x1250, 0x4c6f},
  {0x1257, 0x436e},   {0x1258, 0x4c6f},   {0x1259, 0x436e},  {0x125a, 0x4c6f},
  {0x125e, 0x436e},   {0x1260, 0x4c6f},   {0x1289, 0x436e},  {0x128a, 0x4c6f},
  {0x128e, 0x436e},   {0x1290, 0x4c6f},   {0x12b1, 0x436e},  {0x12b2, 0x4c6f},
  {0x12b6, 0x436e},   {0x12b8, 0x4c6f},   {0x12bf, 0x436e},  {0x12c0, 0x4c6f},
  {0x12c1, 0x436e},   {0x12c2, 0x4c6f},   {0x12c6, 0x436e},  {0x12c8, 0x4c6f},
  {0x12d7, 0x436e},   {0x12d8, 0x4c6f},   {0x1311, 0x436e},  {0x1312, 0x4c6f},
  {0x1316, 0x436e},   {0x1318, 0x4c6f},   {0x135b, 0x436e},  {0x135d, 0x4d6e},
  {0x1360, 0x506f},   {0x1369, 0x4e6f},   {0x137d, 0x436e},  {0x1380, 0x4c6f},
  {0x1390, 0x536f},   {0x139a, 0x436e},   {0x13a0, 0x4c75},  {0x13f6, 0x436e},
  {0x13f8, 0x4c6c},   {0x13fe, 0x436e},   {0x1400, 0x5064},  {0x1401, 0x4c6f},
  {0x166d, 0x506f},   {0x166f, 0x4c6f},   {0x1680, 0x5a73},  {0x1681, 0x4c6f},
  {0x169b, 0x5073},   {0x169c, 0x5065},   {0x169d, 0x436e},  {0x16a0, 0x4c6f},
  {0x16eb, 0x506f},   {0x16ee, 0x4e6c},   {0x16f1, 0x4c6f},  {0x16f9, 0x436e},
  {0x1700, 0x4c6f},   {0x170d, 0x436e},   {0x170e, 0x4c6f},  {0x1712, 0x4d6e},
  {0x1715, 0x436e},   {0x1720, 0x4c6f},   {0x1732, 0x4d6e},  {0x1735, 0x506f},
  {0x1737, 0x436e},   {0x1740, 0x4c6f},   {0x1752, 0x4d6e},  {0x1754, 0x436e},
  {0x1760, 0x4c6f},   {0x176d, 0x436e},   {0x176e, 0x4c6f},  {0x1771, 0x436e},
  {0x1772, 0x4d6e},   {0x1774, 0x436e},   {0x1780, 0x4c6f},  {0x17b4, 0x4d6e},
  {0x17b6, 0x4d63},   {0x17b7, 0x4d6e},   {0x17be, 0x4d63},  {0x17c6, 0x4d6e},
  {0x17c7, 0x4d63},   {0x17c9, 0x4d6e},   {0x17d4, 0x506f},  {0x17d7, 0x4c6d},
  {0x17d8, 0x506f},   {0x17db, 0x5363},   {0x17dc, 0x4c6f},  {0x17dd, 0x4d6e},
  {0x17de, 0x436e},   {0x17e0, 0x4e64},   {0x17ea, 0x436e},  {0x17f0, 0x4e6f},
  {0x17fa, 0x436e},   {0x1800, 0x506f},   {0x1806, 0x5064},  {0x1807, 0x506f},
  {0x180b, 0x4d6e},   {0x180e, 0x4366},   {0x180f, 0x436e},  {0x1810, 0x4e64},
  {0x181a, 0x436e},   {0x1820, 0x4c6f},   {0x1843, 0x4c6d},  {0x1844, 0x4c6f},
  {0x1878, 0x436e},   {0x1880, 0x4c6f},   {0x18a9, 0x4d6e},  {0x18aa, 0x4c6f},
  {0x18ab, 0x436e},   {0x18b0, 0x4c6f},   {0x18f6, 0x436e},  {0x1900, 0x4c6f},
  {0x191f, 0x436e},   {0x1920, 0x4d6e},   {0x1923, 0x4d63},  {0x1927, 0x4d6e},
  {0x1929, 0x4d63},   {0x192c, 0x436e},   {0x1930, 0x4d63},  {0x1932, 0x4d6e},
  {0x1933, 0x4d63},   {0x1939, 0x4d6e},   {0x193c, 0x436e},  {0x1940, 0x536f},
  {0x1941, 0x436e},   {0x1944, 0x506f},   {0x1946, 0x4e64},  {0x1950, 0x4c6f},
  {0x196e, 0x436e},   {0x1970, 0x4c6f},   {0x1975, 0x436e},  {0x1980, 0x4c6f},
  {0x19ac, 0x436e},   {0x19b0, 0x4c6f},   {0x19ca, 0x436e},  {0x19d0, 0x4e64},
  {0x19da, 0x4e6f},   {0x19db, 0x436e},   {0x19de, 0x536f},  {0x1a00, 0x4c6f},
  {0x1a17, 0x4d6e},   {0x1a19, 0x4d63},   {0x1a1b, 0x4d6e},  {0x1a1c, 0x436e},
  {0x1a1e, 0x506f},   {0x1a20, 0x4c6f},   {0x1a55, 0x4d63},  {0x1a56, 0x4d6e},
  {0x1a57, 0x4d63},   {0x1a58, 0x4d6e},   {0x1a5f, 0x436e},  {0x1a60, 0x4d6e},
  {0x1a61, 0x4d63},   {0x1a62, 0x4d6e},   {0x1a63, 0x4d63},  {0x1a65, 0x4d6e},
  {0x1a6d, 0x4d63},   {0x1a73, 0x4d6e},   {0x1a7d, 0x436e},  {0x1a7f, 0x4d6e},
  {0x1a80, 0x4e64},   {0x1a8a, 0x436e},   {0x1a90, 0x4e64},  {0x1a9a, 0x436e},
  {0x1aa0, 0x506f},   {0x1aa7, 0x4c6d},   {0x1aa8, 0x506f},  {0x1aae, 0x436e},
  {0x1ab0, 0x4d6e},   {0x1abe, 0x4d65},   {0x1abf, 0x436e},  {0x1b00, 0x4d6e},
  {0x1b04, 0x4d63},   {0x1b05, 0x4c6f},   {0x1b34, 0x4d6e},  {0x1b35, 0x4d63},
  {0x1b36, 0x4d6e},   {0x1b3b, 0x4d63},   {0x1b3c, 0x4d6e},  {0x1b3d, 0x4d63},
  {0x1b42, 0x4d6e},   {0x1b43, 0x4d63},   {0x1b45, 0x4c6f},  {0x1b4c, 0x436e},
  {0x1b50, 0x4e64},   {0x1b5a, 0x506f},   {0x1b61, 0x536f},  {0x1b6b, 0x4d6e},
  {0x1b74, 0x536f},   {0x1b7d, 0x436e},   {0x1b80, 0x4d6e},  {0x1b82, 0x4d63},
  {0x1b83, 0x4c6f},   {0x1ba1, 0x4d63},   {0x1ba2, 0x4d6e},  {0x1ba6, 0x4d63},
  {0x1ba8, 0x4d6e},   {0x1baa, 0x4d63},   {0x1bab, 0x4d6e},  {0x1bae, 0x4c6f},
  {0x1bb0, 0x4e64},   {0x1bba, 0x4c6f},   {0x1be6, 0x4d6e},  {0x1be7, 0x4d63},
  {0x1be8, 0x4d6e},   {0x1bea, 0x4d63},   {0x1bed, 0x4d6e},  {0x1bee, 0x4d63},
  {0x1bef, 0x4d6e},   {0x1bf2, 0x4d63},   {0x1bf4, 0x436e},  {0x1bfc, 0x506f},
  {0x1c00, 0x4c6f},   {0x1c24, 0x4d63},   {0x1c2c, 0x4d6e},  {0x1c34, 0x4d63},
  {0x1c36, 0x4d6e},   {0x1c38, 0x436e},   {0x1c3b, 0x506f},  {0x1c40, 0x4e64},
  {0x1c4a, 0x436e},   {0x1c4d, 0x4c6f},   {0x1c50, 0x4e64},  {0x1c5a, 0x4c6f},
  {0x1c78, 0x4c6d},   {0x1c7e, 0x506f},   {0x1c80, 0x436e},  {0x1cc0, 0x506f},
  {0x1cc8, 0x436e},   {0x1cd0, 0x4d6e},   {0x1cd3, 0x506f},  {0x1cd4, 0x4d6e},
  {0x1ce1, 0x4d63},   {0x1ce2, 0x4d6e},   {0x1ce9, 0x4c6f},  {0x1ced, 0x4d6e},
  {0x1cee, 0x4c6f},   {0x1cf2, 0x4d63},   {0x1cf4, 0x4d6e},  {0x1cf5, 0x4c6f},
  {0x1cf7, 0x436e},   {0x1cf8, 0x4d6e},   {0x1cfa, 0x436e},  {0x1d00, 0x4c6c},
  {0x1d2c, 0x4c6d},   {0x1d6b, 0x4c6c},   {0x1d78, 0x4c6d},  {0x1d79, 0x4c6c},
  {0x1d9b, 0x4c6d},   {0x1dc0, 0x4d6e},   {0x1df6, 0x436e},  {0x1dfc, 0x4d6e},
  {0x1e00, 0x4c75},   {0x1e01, 0x4c6c},   {0x1e02, 0x4c75},  {0x1e03, 0x4c6c},
  {0x1e04, 0x4c75},   {0x1e05, 0x4c6c},   {0x1e06, 0x4c75},  {0x1e07, 0x4c6c},
  {0x1e08, 0x4c75},   {0x1e09, 0x4c6c},   {0x1e0a, 0x4c75},  {0x1e0b, 0x4c6c},
  {0x1e0c, 0x4c75},   {0x1e0d, 0x4c6c},   {0x1e0e, 0x4c75},  {0x1e0f, 0x4c6c},
  {0x1e10, 0x4c75},   {0x1e11, 0x4c6c},   {0x1e12, 0x4c75},  {0x1e13, 0x4c6c},
  {0x1e14, 0x4c75},   {0x1e15, 0x4c6c},   {0x1e16, 0x4c75},  {0x1e17, 0x4c6c},
  {0x1e18, 0x4c75},   {0x1e19, 0x4c6c},   {0x1e1a, 0x4c75},  {0x1e1b, 0x4c6c},
  {0x1e1c, 0x4c75},   {0x1e1d, 0x4c6c},   {0x1e1e, 0x4c75},  {0x1e1f, 0x4c6c},
  {0x1e20, 0x4c75},   {0x1e21, 0x4c6c},   {0x1e22, 0x4c75},  {0x1e23, 0x4c6c},
  {0x1e24, 0x4c75},   {0x1e25, 0x4c6c},   {0x1e26, 0x4c75},  {0x1e27, 0x4c6c},
  {0x1e28, 0x4c75},   {0x1e29, 0x4c6c},   {0x1e2a, 0x4c75},  {0x1e2b, 0x4c6c},
  {0x1e2c, 0x4c75},   {0x1e2d, 0x4c6c},   {0x1e2e, 0x4c75},  {0x1e2f, 0x4c6c},
  {0x1e30, 0x4c75},   {0x1e31, 0x4c6c},   {0x1e32, 0x4c75},  {0x1e33, 0x4c6c},
  {0x1e34, 0x4c75},   {0x1e35, 0x4c6c},   {0x1e36, 0x4c75},  {0x1e37, 0x4c6c},
  {0x1e38, 0x4c75},   {0x1e39, 0x4c6c},   {0x1e3a, 0x4c75},  {0x1e3b, 0x4c6c},
  {0x1e3c, 0x4c75},   {0x1e3d, 0x4c6c},   {0x1e3e, 0x4c75},  {0x1e3f, 0x4c6c},
  {0x1e40, 0x4c75},   {0x1e41, 0x4c6c},   {0x1e42, 0x4c75},  {0x1e43, 0x4c6c},
  {0x1e44, 0x4c75},   {0x1e45, 0x4c6c},   {0x1e46, 0x4c75},  {0x1e47, 0x4c6c},
  {0x1e48, 0x4c75},   {0x1e49, 0x4c6c},   {0x1e4a, 0x4c75},  {0x1e4b, 0x4c6c},
  {0x1e4c, 0x4c75},   {0x1e4d, 0x4c6c},   {0x1e4e, 0x4c75},  {0x1e4f, 0x4c6c},
  {0x1e50, 0x4c75},   {0x1e51, 0x4c6c},   {0x1e52, 0x4c75},  {0x1e53, 0x4c6c},
  {0x1e54, 0x4c75},   {0x1e55, 0x4c6c},   {0x1e56, 0x4c75},  {0x1e57, 0x4c6c},
  {0x1e58, 0x4c75},   {0x1e59, 0x4c6c},   {0x1e5a, 0x4c75},  {0x1e5b, 0x4c6c},
  {0x1e5c, 0x4c75},   {0x1e5d, 0x4c6c},   {0x1e5e, 0x4c75},  {0x1e5f, 0x4c6c},
  {0x1e60, 0x4c75},   {0x1e61, 0x4c6c},   {0x1e62, 0x4c75},  {0x1e63, 0x4c6c},
  {0x1e64, 0x4c75},   {0x1e65, 0x4c6c},   {0x1e66, 0x4c75},  {0x1e67, 0x4c6c},
  {0x1e68, 0x4c75},   {0x1e69, 0x4c6c},   {0x1e6a, 0x4c75},  {0x1e6b, 0x4c6c},
  {0x1e6c, 0x4c75},   {0x1e6d, 0x4c6c},   {0x1e6e, 0x4c75},  {0x1e6f, 0x4c6c},
  {0x1e70, 0x4c75},   {0x1e71, 0x4c6c},   {0x1e72, 0x4c75},  {0x1e73, 0x4c6c},
  {0x1e74, 0x4c75},   {0x1e75, 0x4c6c},   {0x1e76, 0x4c75},  {0x1e77, 0x4c6c},
  {0x1e78, 0x4c75},   {0x1e79, 0x4c6c},   {0x1e7a, 0x4c75},  {0x1e7b, 0x4c6c},
  {0x1e7c, 0x4c75},   {0x1e7d, 0x4c6c},   {0x1e7e, 0x4c75},  {0x1e7f, 0x4c6c},
  {0x1e80, 0x4c75},   {0x1e81, 0x4c6c},   {0x1e82, 0x4c75},  {0x1e83, 0x4c6c},
  {0x1e84, 0x4c75},   {0x1e85, 0x4c6c},   {0x1e86, 0x4c75},  {0x1e87, 0x4c6c},
  {0x1e88, 0x4c75},   {0x1e89, 0x4c6c},   {0x1e8a, 0x4c75},  {0x1e8b, 0x4c6c},
  {0x1e8c, 0x4c75},   {0x1e8d, 0x4c6c},   {0x1e8e, 0x4c75},  {0x1e8f, 0x4c6c},
  {0x1e90, 0x4c75},   {0x1e91, 0x4c6c},   {0x1e92, 0x4c75},  {0x1e93, 0x4c6c},
  {0x1e94, 0x4c75},   {0x1e95, 0x4c6c},   {0x1e9e, 0x4c75},  {0x1e9f, 0x4c6c},
  {0x1ea0, 0x4c75},   {0x1ea1, 0x4c6c},   {0x1ea2, 0x4c75},  {0x1ea3, 0x4c6c},
  {0x1ea4, 0x4c75},   {0x1ea5, 0x4c6c},   {0x1ea6, 0x4c75},  {0x1ea7, 0x4c6c},
  {0x1ea8, 0x4c75},   {0x1ea9, 0x4c6c},   {0x1eaa, 0x4c75},  {0x1eab, 0x4c6c},
  {0x1eac, 0x4c75},   {0x1ead, 0x4c6c},   {0x1eae, 0x4c75},  {0x1eaf, 0x4c6c},
  {0x1eb0, 0x4c75},   {0x1eb1, 0x4c6c},   {0x1eb2, 0x4c75},  {0x1eb3, 0x4c6c},
  {0x1eb4, 0x4c75},   {0x1eb5, 0x4c6c},   {0x1eb6, 0x4c75},  {0x1eb7, 0x4c6c},
  {0x1eb8, 0x4c75},   {0x1eb9, 0x4c6c},   {0x1eba, 0x4c75},  {0x1ebb, 0x4c6c},
  {0x1ebc, 0x4c75},   {0x1ebd, 0x4c6c},   {0x1ebe, 0x4c75},  {0x1ebf, 0x4c6c},
  {0x1ec0, 0x4c75},   {0x1ec1, 0x4c6c},   {0x1ec2, 0x4c75},  {0x1ec3, 0x4c6c},
  {0x1ec4, 0x4c75},   {0x1ec5, 0x4c6c},   {0x1ec6, 0x4c75},  {0x1ec7, 0x4c6c},
  {0x1ec8, 0x4c75},   {0x1ec9, 0x4c6c},   {0x1eca, 0x4c75},  {0x1ecb, 0x4c6c},
  {0x1ecc, 0x4c75},   {0x1ecd, 0x4c6c},   {0x1ece, 0x4c75},  {0x1ecf, 0x4c6c},
  {0x1ed0, 0x4c75},   {0x1ed1, 0x4c6c},   {0x1ed2, 0x4c75},  {0x1ed3, 0x4c6c},
  {0x1ed4, 0x4c75},   {0x1ed5, 0x4c6c},   {0x1ed6, 0x4c75},  {0x1ed7, 0x4c6c},
  {0x1ed8, 0x4c75},   {0x1ed9, 0x4c6c},   {0x1eda, 0x4c75},  {0x1edb, 0x4c6c},
  {0x1edc, 0x4c75},   {0x1edd, 0x4c6c},   {0x1ede, 0x4c75},  {0x1edf, 0x4c6c},
  {0x1ee0, 0x4c75},   {0x1ee1, 0x4c6c},   {0x1ee2, 0x4c75},  {0x1ee3, 0x4c6c},
  {0x1ee4, 0x4c75},   {0x1ee5, 0x4c6c},   {0x1ee6, 0x4c75},  {0x1ee7, 0x4c6c},
  {0x1ee8, 0x4c75},   {0x1ee9, 0x4c6c},   {0x1eea, 0x4c75},  {0x1eeb, 0x4c6c},
  {0x1eec, 0x4c75},   {0x1eed, 0x4c6c},   {0x1eee, 0x4c75},  {0x1eef, 0x4c6c},
  {0x1ef0, 0x4c75},   {0x1ef1, 0x4c6c},   {0x1ef2, 0x4c75},  {0x1ef3, 0x4c6c},
  {0x1ef4, 0x4c75},   {0x1ef5, 0x4c6c},   {0x1ef6, 0x4c75},  {0x1ef7, 0x4c6c},
  {0x1ef8, 0x4c75},   {0x1ef9, 0x4c6c},   {0x1efa, 0x4c75},  {0x1efb, 0x4c6c},
  {0x1efc, 0x4c75},   {0x1efd, 0x4c6c},   {0x1efe, 0x4c75},  {0x1eff, 0x4c6c},
  {0x1f08, 0x4c75},   {0x1f10, 0x4c6c},   {0x1f16, 0x436e},  {0x1f18, 0x4c75},
  {0x1f1e, 0x436e},   {0x1f20, 0x4c6c},   {0x1f28, 0x4c75},  {0x1f30, 0x4c6c},
  {0x1f38, 0x4c75},   {0x1f40, 0x4c6c},   {0x1f46, 0x436e},  {0x1f48, 0x4c75},
  {0x1f4e, 0x436e},   {0x1f50, 0x4c6c},   {0x1f58, 0x436e},  {0x1f59, 0x4c75},
  {0x1f5a, 0x436e},   {0x1f5b, 0x4c75},   {0x1f5c, 0x436e},  {0x1f5d, 0x4c75},
  {0x1f5e, 0x436e},   {0x1f5f, 0x4c75},   {0x1f60, 0x4c6c},  {0x1f68, 0x4c75},
  {0x1f70, 0x4c6c},   {0x1f7e, 0x436e},   {0x1f80, 0x4c6c},  {0x1f88, 0x4c74},
  {0x1f90, 0x4c6c},   {0x1f98, 0x4c74},   {0x1fa0, 0x4c6c},  {0x1fa8, 0x4c74},
  {0x1fb0, 0x4c6c},   {0x1fb5, 0x436e},   {0x1fb6, 0x4c6c},  {0x1fb8, 0x4c75},
  {0x1fbc, 0x4c74},   {0x1fbd, 0x536b},   {0x1fbe, 0x4c6c},  {0x1fbf, 0x536b},
  {0x1fc2, 0x4c6c},   {0x1fc5, 0x436e},   {0x1fc6, 0x4c6c},  {0x1fc8, 0x4c75},
  {0x1fcc, 0x4c74},   {0x1fcd, 0x536b},   {0x1fd0, 0x4c6c},  {0x1fd4, 0x436e},
  {0x1fd6, 0x4c6c},   {0x1fd8, 0x4c75},   {0x1fdc, 0x436e},  {0x1fdd, 0x536b},
  {0x1fe0, 0x4c6c},   {0x1fe8, 0x4c75},   {0x1fed, 0x536b},  {0x1ff0, 0x436e},
  {0x1ff2, 0x4c6c},   {0x1ff5, 0x436e},   {0x1ff6, 0x4c6c},  {0x1ff8, 0x4c75},
  {0x1ffc, 0x4c74},   {0x1ffd, 0x536b},   {0x1fff, 0x436e},  {0x2000, 0x5a73},
  {0x200b, 0x4366},   {0x2010, 0x5064},   {0x2016, 0x506f},  {0x2018, 0x5069},
  {0x2019, 0x5066},   {0x201a, 0x5073},   {0x201b, 0x5069},  {0x201d, 0x5066},
  {0x201e, 0x5073},   {0x201f, 0x5069},   {0x2020, 0x506f},  {0x2028, 0x5a6c},
  {0x2029, 0x5a70},   {0x202a, 0x4366},   {0x202f, 0x5a73},  {0x2030, 0x506f},
  {0x2039, 0x5069},   {0x203a, 0x5066},   {0x203b, 0x506f},  {0x203f, 0x5063},
  {0x2041, 0x506f},   {0x2044, 0x536d},   {0x2045, 0x5073},  {0x2046, 0x5065},
  {0x2047, 0x506f},   {0x2052, 0x536d},   {0x2053, 0x506f},  {0x2054, 0x5063},
  {0x2055, 0x506f},   {0x205f, 0x5a73},   {0x2060, 0x4366},  {0x2065, 0x436e},
  {0x2066, 0x4366},   {0x2070, 0x4e6f},   {0x2071, 0x4c6d},  {0x2072, 0x436e},
  {0x2074, 0x4e6f},   {0x207a, 0x536d},   {0x207d, 0x5073},  {0x207e, 0x5065},
  {0x207f, 0x4c6d},   {0x2080, 0x4e6f},   {0x208a, 0x536d},  {0x208d, 0x5073},
  {0x208e, 0x5065},   {0x208f, 0x436e},   {0x2090, 0x4c6d},  {0x209d, 0x436e},
  {0x20a0, 0x5363},   {0x20bf, 0x436e},   {0x20d0, 0x4d6e},  {0x20dd, 0x4d65},
  {0x20e1, 0x4d6e},   {0x20e2, 0x4d65},   {0x20e5, 0x4d6e},  {0x20f1, 0x436e},
  {0x2100, 0x536f},   {0x2102, 0x4c75},   {0x2103, 0x536f},  {0x2107, 0x4c75},
  {0x2108, 0x536f},   {0x210a, 0x4c6c},   {0x210b, 0x4c75},  {0x210e, 0x4c6c},
  {0x2110, 0x4c75},   {0x2113, 0x4c6c},   {0x2114, 0x536f},  {0x2115, 0x4c75},
  {0x2116, 0x536f},   {0x2118, 0x536d},   {0x2119, 0x4c75},  {0x211e, 0x536f},
  {0x2124, 0x4c75},   {0x2125, 0x536f},   {0x2126, 0x4c75},  {0x2127, 0x536f},
  {0x2128, 0x4c75},   {0x2129, 0x536f},   {0x212a, 0x4c75},  {0x212e, 0x536f},
  {0x212f, 0x4c6c},   {0x2130, 0x4c75},   {0x2134, 0x4c6c},  {0x2135, 0x4c6f},
  {0x2139, 0x4c6c},   {0x213a, 0x536f},   {0x213c, 0x4c6c},  {0x213e, 0x4c75},
  {0x2140, 0x536d},   {0x2145, 0x4c75},   {0x2146, 0x4c6c},  {0x214a, 0x536f},
  {0x214b, 0x536d},   {0x214c, 0x536f},   {0x214e, 0x4c6c},  {0x214f, 0x536f},
  {0x2150, 0x4e6f},   {0x2160, 0x4e6c},   {0x2183, 0x4c75},  {0x2184, 0x4c6c},
  {0x2185, 0x4e6c},   {0x2189, 0x4e6f},   {0x218a, 0x536f},  {0x218c, 0x436e},
  {0x2190, 0x536d},   {0x2195, 0x536f},   {0x219a, 0x536d},  {0x219c, 0x536f},
  {0x21a0, 0x536d},   {0x21a1, 0x536f},   {0x21a3, 0x536d},  {0x21a4, 0x536f},
  {0x21a6, 0x536d},   {0x21a7, 0x536f},   {0x21ae, 0x536d},  {0x21af, 0x536f},
  {0x21ce, 0x536d},   {0x21d0, 0x536f},   {0x21d2, 0x536d},  {0x21d3, 0x536f},
  {0x21d4, 0x536d},   {0x21d5, 0x536f},   {0x21f4, 0x536d},  {0x2300, 0x536f},
  {0x2308, 0x5073},   {0x2309, 0x5065},   {0x230a, 0x5073},  {0x230b, 0x5065},
  {0x230c, 0x536f},   {0x2320, 0x536d},   {0x2322, 0x536f},  {0x2329, 0x5073},
  {0x232a, 0x5065},   {0x232b, 0x536f},   {0x237c, 0x536d},  {0x237d, 0x536f},
  {0x239b, 0x536d},   {0x23b4, 0x536f},   {0x23dc, 0x536d},  {0x23e2, 0x536f},
  {0x23fb, 0x436e},   {0x2400, 0x536f},   {0x2427, 0x436e},  {0x2440, 0x536f},
  {0x244b, 0x436e},   {0x2460, 0x4e6f},   {0x249c, 0x536f},  {0x24ea, 0x4e6f},
  {0x2500, 0x536f},   {0x25b7, 0x536d},   {0x25b8, 0x536f},  {0x25c1, 0x536d},
  {0x25c2, 0x536f},   {0x25f8, 0x536d},   {0x2600, 0x536f},  {0x266f, 0x536d},
  {0x2670, 0x536f},   {0x2768, 0x5073},   {0x2769, 0x5065},  {0x276a, 0x5073},
  {0x276b, 0x5065},   {0x276c, 0x5073},   {0x276d, 0x5065},  {0x276e, 0x5073},
  {0x276f, 0x5065},   {0x2770, 0x5073},   {0x2771, 0x5065},  {0x2772, 0x5073},
  {0x2773, 0x5065},   {0x2774, 0x5073},   {0x2775, 0x5065},  {0x2776, 0x4e6f},
  {0x2794, 0x536f},   {0x27c0, 0x536d},   {0x27c5, 0x5073},  {0x27c6, 0x5065},
  {0x27c7, 0x536d},   {0x27e6, 0x5073},   {0x27e7, 0x5065},  {0x27e8, 0x5073},
  {0x27e9, 0x5065},   {0x27ea, 0x5073},   {0x27eb, 0x5065},  {0x27ec, 0x5073},
  {0x27ed, 0x5065},   {0x27ee, 0x5073},   {0x27ef, 0x5065},  {0x27f0, 0x536d},
  {0x2800, 0x536f},   {0x2900, 0x536d},   {0x2983, 0x5073},  {0x2984, 0x5065},
  {0x2985, 0x5073},   {0x2986, 0x5065},   {0x2987, 0x5073},  {0x2988, 0x5065},
  {0x2989, 0x5073},   {0x298a, 0x5065},   {0x298b, 0x5073},  {0x298c, 0x5065},
  {0x298d, 0x5073},   {0x298e, 0x5065},   {0x298f, 0x5073},  {0x2990, 0x5065},
  {0x2991, 0x5073},   {0x2992, 0x5065},   {0x2993, 0x5073},  {0x2994, 0x5065},
  {0x2995, 0x5073},   {0x2996, 0x5065},   {0x2997, 0x5073},  {0x2998, 0x5065},
  {0x2999, 0x536d},   {0x29d8, 0x5073},   {0x29d9, 0x5065},  {0x29da, 0x5073},
  {0x29db, 0x5065},   {0x29dc, 0x536d},   {0x29fc, 0x5073},  {0x29fd, 0x5065},
  {0x29fe, 0x536d},   {0x2b00, 0x536f},   {0x2b30, 0x536d},  {0x2b45, 0x536f},
  {0x2b47, 0x536d},   {0x2b4d, 0x536f},   {0x2b74, 0x436e},  {0x2b76, 0x536f},
  {0x2b96, 0x436e},   {0x2b98, 0x536f},   {0x2bba, 0x436e},  {0x2bbd, 0x536f},
  {0x2bc9, 0x436e},   {0x2bca, 0x536f},   {0x2bd2, 0x436e},  {0x2bec, 0x536f},
  {0x2bf0, 0x436e},   {0x2c00, 0x4c75},   {0x2c2f, 0x436e},  {0x2c30, 0x4c6c},
  {0x2c5f, 0x436e},   {0x2c60, 0x4c75},   {0x2c61, 0x4c6c},  {0x2c62, 0x4c75},
  {0x2c65, 0x4c6c},   {0x2c67, 0x4c75},   {0x2c68, 0x4c6c},  {0x2c69, 0x4c75},
  {0x2c6a, 0x4c6c},   {0x2c6b, 0x4c75},   {0x2c6c, 0x4c6c},  {0x2c6d, 0x4c75},
  {0x2c71, 0x4c6c},   {0x2c72, 0x4c75},   {0x2c73, 0x4c6c},  {0x2c75, 0x4c75},
  {0x2c76, 0x4c6c},   {0x2c7c, 0x4c6d},   {0x2c7e, 0x4c75},  {0x2c81, 0x4c6c},
  {0x2c82, 0x4c75},   {0x2c83, 0x4c6c},   {0x2c84, 0x4c75},  {0x2c85, 0x4c6c},
  {0x2c86, 0x4c75},   {0x2c87, 0x4c6c},   {0x2c88, 0x4c75},  {0x2c89, 0x4c6c},
  {0x2c8a, 0x4c75},   {0x2c8b, 0x4c6c},   {0x2c8c, 0x4c75},  {0x2c8d, 0x4c6c},
  {0x2c8e, 0x4c75},   {0x2c8f, 0x4c6c},   {0x2c90, 0x4c75},  {0x2c91, 0x4c6c},
  {0x2c92, 0x4c75},   {0x2c93, 0x4c6c},   {0x2c94, 0x4c75},  {0x2c95, 0x4c6c},
  {0x2c96, 0x4c75},   {0x2c97, 0x4c6c},   {0x2c98, 0x4c75},  {0x2c99, 0x4c6c},
  {0x2c9a, 0x4c75},   {0x2c9b, 0x4c6c},   {0x2c9c, 0x4c75},  {0x2c9d, 0x4c6c},
  {0x2c9e, 0x4c75},   {0x2c9f, 0x4c6c},   {0x2ca0, 0x4c75},  {0x2ca1, 0x4c6c},
  {0x2ca2, 0x4c75},   {0x2ca3, 0x4c6c},   {0x2ca4, 0x4c75},  {0x2ca5, 0x4c6c},
  {0x2ca6, 0x4c75},   {0x2ca7, 0x4c6c},   {0x2ca8, 0x4c75},  {0x2ca9, 0x4c6c},
  {0x2caa, 0x4c75},   {0x2cab, 0x4c6c},   {0x2cac, 0x4c75},  {0x2cad, 0x4c6c},
  {0x2cae, 0x4c75},   {0x2caf, 0x4c6c},   {0x2cb0, 0x4c75},  {0x2cb1, 0x4c6c},
  {0x2cb2, 0x4c75},   {0x2cb3, 0x4c6c},   {0x2cb4, 0x4c75},  {0x2cb5, 0x4c6c},
  {0x2cb6, 0x4c75},   {0x2cb7, 0x4c6c},   {0x2cb8, 0x4c75},  {0x2cb9, 0x4c6c},
  {0x2cba, 0x4c75},   {0x2cbb, 0x4c6c},   {0x2cbc, 0x4c75},  {0x2cbd, 0x4c6c},
  {0x2cbe, 0x4c75},   {0x2cbf, 0x4c6c},   {0x2cc0, 0x4c75},  {0x2cc1, 0x4c6c},
  {0x2cc2, 0x4c75},   {0x2cc3, 0x4c6c},   {0x2cc4, 0x4c75},  {0x2cc5, 0x4c6c},
  {0x2cc6, 0x4c75},   {0x2cc7, 0x4c6c},   {0x2cc8, 0x4c75},  {0x2cc9, 0x4c6c},
  {0x2cca, 0x4c75},   {0x2ccb, 0x4c6c},   {0x2ccc, 0x4c75},  {0x2ccd, 0x4c6c},
  {0x2cce, 0x4c75},   {0x2ccf, 0x4c6c},   {0x2cd0, 0x4c75},  {0x2cd1, 0x4c6c},
  {0x2cd2, 0x4c75},   {0x2cd3, 0x4c6c},   {0x2cd4, 0x4c75},  {0x2cd5, 0x4c6c},
  {0x2cd6, 0x4c75},   {0x2cd7, 0x4c6c},   {0x2cd8, 0x4c75},  {0x2cd9, 0x4c6c},
  {0x2cda, 0x4c75},   {0x2cdb, 0x4c6c},   {0x2cdc, 0x4c75},  {0x2cdd, 0x4c6c},
  {0x2cde, 0x4c75},   {0x2cdf, 0x4c6c},   {0x2ce0, 0x4c75},  {0x2ce1, 0x4c6c},
  {0x2ce2, 0x4c75},   {0x2ce3, 0x4c6c},   {0x2ce5, 0x536f},  {0x2ceb, 0x4c75},
  {0x2cec, 0x4c6c},   {0x2ced, 0x4c75},   {0x2cee, 0x4c6c},  {0x2cef, 0x4d6e},
  {0x2cf2, 0x4c75},   {0x2cf3, 0x4c6c},   {0x2cf4, 0x436e},  {0x2cf9, 0x506f},
  {0x2cfd, 0x4e6f},   {0x2cfe, 0x506f},   {0x2d00, 0x4c6c},  {0x2d26, 0x436e},
  {0x2d27, 0x4c6c},   {0x2d28, 0x436e},   {0x2d2d, 0x4c6c},  {0x2d2e, 0x436e},
  {0x2d30, 0x4c6f},   {0x2d68, 0x436e},   {0x2d6f, 0x4c6d},  {0x2d70, 0x506f},
  {0x2d71, 0x436e},   {0x2d7f, 0x4d6e},   {0x2d80, 0x4c6f},  {0x2d97, 0x436e},
  {0x2da0, 0x4c6f},   {0x2da7, 0x436e},   {0x2da8, 0x4c6f},  {0x2daf, 0x436e},
  {0x2db0, 0x4c6f},   {0x2db7, 0x436e},   {0x2db8, 0x4c6f},  {0x2dbf, 0x436e},
  {0x2dc0, 0x4c6f},   {0x2dc7, 0x436e},   {0x2dc8, 0x4c6f},  {0x2dcf, 0x436e},
  {0x2dd0, 0x4c6f},   {0x2dd7, 0x436e},   {0x2dd8, 0x4c6f},  {0x2ddf, 0x436e},
  {0x2de0, 0x4d6e},   {0x2e00, 0x506f},   {0x2e02, 0x5069},  {0x2e03, 0x5066},
  {0x2e04, 0x5069},   {0x2e05, 0x5066},   {0x2e06, 0x506f},  {0x2e09, 0x5069},
  {0x2e0a, 0x5066},   {0x2e0b, 0x506f},   {0x2e0c, 0x5069},  {0x2e0d, 0x5066},
  {0x2e0e, 0x506f},   {0x2e17, 0x5064},   {0x2e18, 0x506f},  {0x2e1a, 0x5064},
  {0x2e1b, 0x506f},   {0x2e1c, 0x5069},   {0x2e1d, 0x5066},  {0x2e1e, 0x506f},
  {0x2e20, 0x5069},   {0x2e21, 0x5066},   {0x2e22, 0x5073},  {0x2e23, 0x5065},
  {0x2e24, 0x5073},   {0x2e25, 0x5065},   {0x2e26, 0x5073},  {0x2e27, 0x5065},
  {0x2e28, 0x5073},   {0x2e29, 0x5065},   {0x2e2a, 0x506f},  {0x2e2f, 0x4c6d},
  {0x2e30, 0x506f},   {0x2e3a, 0x5064},   {0x2e3c, 0x506f},  {0x2e40, 0x5064},
  {0x2e41, 0x506f},   {0x2e42, 0x5073},   {0x2e43, 0x436e},  {0x2e80, 0x536f},
  {0x2e9a, 0x436e},   {0x2e9b, 0x536f},   {0x2ef4, 0x436e},  {0x2f00, 0x536f},
  {0x2fd6, 0x436e},   {0x2ff0, 0x536f},   {0x2ffc, 0x436e},  {0x3000, 0x5a73},
  {0x3001, 0x506f},   {0x3004, 0x536f},   {0x3005, 0x4c6d},  {0x3006, 0x4c6f},
  {0x3007, 0x4e6c},   {0x3008, 0x5073},   {0x3009, 0x5065},  {0x300a, 0x5073},
  {0x300b, 0x5065},   {0x300c, 0x5073},   {0x300d, 0x5065},  {0x300e, 0x5073},
  {0x300f, 0x5065},   {0x3010, 0x5073},   {0x3011, 0x5065},  {0x3012, 0x536f},
  {0x3014, 0x5073},   {0x3015, 0x5065},   {0x3016, 0x5073},  {0x3017, 0x5065},
  {0x3018, 0x5073},   {0x3019, 0x5065},   {0x301a, 0x5073},  {0x301b, 0x5065},
  {0x301c, 0x5064},   {0x301d, 0x5073},   {0x301e, 0x5065},  {0x3020, 0x536f},
  {0x3021, 0x4e6c},   {0x302a, 0x4d6e},   {0x302e, 0x4d63},  {0x3030, 0x5064},
  {0x3031, 0x4c6d},   {0x3036, 0x536f},   {0x3038, 0x4e6c},  {0x303b, 0x4c6d},
  {0x303c, 0x4c6f},   {0x303d, 0x506f},   {0x303e, 0x536f},  {0x3040, 0x436e},
  {0x3041, 0x4c6f},   {0x3097, 0x436e},   {0x3099, 0x4d6e},  {0x309b, 0x536b},
  {0x309d, 0x4c6d},   {0x309f, 0x4c6f},   {0x30a0, 0x5064},  {0x30a1, 0x4c6f},
  {0x30fb, 0x506f},   {0x30fc, 0x4c6d},   {0x30ff, 0x4c6f},  {0x3100, 0x436e},
  {0x3105, 0x4c6f},   {0x312e, 0x436e},   {0x3131, 0x4c6f},  {0x318f, 0x436e},
  {0x3190, 0x536f},   {0x3192, 0x4e6f},   {0x3196, 0x536f},  {0x31a0, 0x4c6f},
  {0x31bb, 0x436e},   {0x31c0, 0x536f},   {0x31e4, 0x436e},  {0x31f0, 0x4c6f},
  {0x3200, 0x536f},   {0x321f, 0x436e},   {0x3220, 0x4e6f},  {0x322a, 0x536f},
  {0x3248, 0x4e6f},   {0x3250, 0x536f},   {0x3251, 0x4e6f},  {0x3260, 0x536f},
  {0x3280, 0x4e6f},   {0x328a, 0x536f},   {0x32b1, 0x4e6f},  {0x32c0, 0x536f},
  {0x32ff, 0x436e},   {0x3300, 0x536f},   {0x3400, 0x4c6f},  {0x4db6, 0x436e},
  {0x4dc0, 0x536f},   {0x4e00, 0x4c6f},   {0x9fd6, 0x436e},  {0xa000, 0x4c6f},
  {0xa015, 0x4c6d},   {0xa016, 0x4c6f},   {0xa48d, 0x436e},  {0xa490, 0x536f},
  {0xa4c7, 0x436e},   {0xa4d0, 0x4c6f},   {0xa4f8, 0x4c6d},  {0xa4fe, 0x506f},
  {0xa500, 0x4c6f},   {0xa60c, 0x4c6d},   {0xa60d, 0x506f},  {0xa610, 0x4c6f},
  {0xa620, 0x4e64},   {0xa62a, 0x4c6f},   {0xa62c, 0x436e},  {0xa640, 0x4c75},
  {0xa641, 0x4c6c},   {0xa642, 0x4c75},   {0xa643, 0x4c6c},  {0xa644, 0x4c75},
  {0xa645, 0x4c6c},   {0xa646, 0x4c75},   {0xa647, 0x4c6c},  {0xa648, 0x4c75},
  {0xa649, 0x4c6c},   {0xa64a, 0x4c75},   {0xa64b, 0x4c6c},  {0xa64c, 0x4c75},
  {0xa64d, 0x4c6c},   {0xa64e, 0x4c75},   {0xa64f, 0x4c6c},  {0xa650, 0x4c75},
  {0xa651, 0x4c6c},   {0xa652, 0x4c75},   {0xa653, 0x4c6c},  {0xa654, 0x4c75},
  {0xa655, 0x4c6c},   {0xa656, 0x4c75},   {0xa657, 0x4c6c},  {0xa658, 0x4c75},
  {0xa659, 0x4c6c},   {0xa65a, 0x4c75},   {0xa65b, 0x4c6c},  {0xa65c, 0x4c75},
  {0xa65d, 0x4c6c},   {0xa65e, 0x4c75},   {0xa65f, 0x4c6c},  {0xa660, 0x4c75},
  {0xa661, 0x4c6c},   {0xa662, 0x4c75},   {0xa663, 0x4c6c},  {0xa664, 0x4c75},
  {0xa665, 0x4c6c},   {0xa666, 0x4c75},   {0xa667, 0x4c6c},  {0xa668, 0x4c75},
  {0xa669, 0x4c6c},   {0xa66a, 0x4c75},   {0xa66b, 0x4c6c},  {0xa66c, 0x4c75},
  {0xa66d, 0x4c6c},   {0xa66e, 0x4c6f},   {0xa66f, 0x4d6e},  {0xa670, 0x4d65},
  {0xa673, 0x506f},   {0xa674, 0x4d6e},   {0xa67e, 0x506f},  {0xa67f, 0x4c6d},
  {0xa680, 0x4c75},   {0xa681, 0x4c6c},   {0xa682, 0x4c75},  {0xa683, 0x4c6c},
  {0xa684, 0x4c75},   {0xa685, 0x4c6c},   {0xa686, 0x4c75},  {0xa687, 0x4c6c},
  {0xa688, 0x4c75},   {0xa689, 0x4c6c},   {0xa68a, 0x4c75},  {0xa68b, 0x4c6c},
  {0xa68c, 0x4c75},   {0xa68d, 0x4c6c},   {0xa68e, 0x4c75},  {0xa68f, 0x4c6c},
  {0xa690, 0x4c75},   {0xa691, 0x4c6c},   {0xa692, 0x4c75},  {0xa693, 0x4c6c},
  {0xa694, 0x4c75},   {0xa695, 0x4c6c},   {0xa696, 0x4c75},  {0xa697, 0x4c6c},
  {0xa698, 0x4c75},   {0xa699, 0x4c6c},   {0xa69a, 0x4c75},  {0xa69b, 0x4c6c},
  {0xa69c, 0x4c6d},   {0xa69e, 0x4d6e},   {0xa6a0, 0x4c6f},  {0xa6e6, 0x4e6c},
  {0xa6f0, 0x4d6e},   {0xa6f2, 0x506f},   {0xa6f8, 0x436e},  {0xa700, 0x536b},
  {0xa717, 0x4c6d},   {0xa720, 0x536b},   {0xa722, 0x4c75},  {0xa723, 0x4c6c},
  {0xa724, 0x4c75},   {0xa725, 0x4c6c},   {0xa726, 0x4c75},  {0xa727, 0x4c6c},
  {0xa728, 0x4c75},   {0xa729, 0x4c6c},   {0xa72a, 0x4c75},  {0xa72b, 0x4c6c},
  {0xa72c, 0x4c75},   {0xa72d, 0x4c6c},   {0xa72e, 0x4c75},  {0xa72f, 0x4c6c},
  {0xa732, 0x4c75},   {0xa733, 0x4c6c},   {0xa734, 0x4c75},  {0xa735, 0x4c6c},
  {0xa736, 0x4c75},   {0xa737, 0x4c6c},   {0xa738, 0x4c75},  {0xa739, 0x4c6c},
  {0xa73a, 0x4c75},   {0xa73b, 0x4c6c},   {0xa73c, 0x4c75},  {0xa73d, 0x4c6c},
  {0xa73e, 0x4c75},   {0xa73f, 0x4c6c},   {0xa740, 0x4c75},  {0xa741, 0x4c6c},
  {0xa742, 0x4c75},   {0xa743, 0x4c6c},   {0xa744, 0x4c75},  {0xa745, 0x4c6c},
  {0xa746, 0x4c75},   {0xa747, 0x4c6c},   {0xa748, 0x4c75},  {0xa749, 0x4c6c},
  {0xa74a, 0x4c75},   {0xa74b, 0x4c6c},   {0xa74c, 0x4c75},  {0xa74d, 0x4c6c},
  {0xa74e, 0x4c75},   {0xa74f, 0x4c6c},   {0xa750, 0x4c75},  {0xa751, 0x4c6c},
  {0xa752, 0x4c75},   {0xa753, 0x4c6c},   {0xa754, 0x4c75},  {0xa755, 0x4c6c},
  {0xa756, 0x4c75},   {0xa757, 0x4c6c},   {0xa758, 0x4c75},  {0xa759, 0x4c6c},
  {0xa75a, 0x4c75},   {0xa75b, 0x4c6c},   {0xa75c, 0x4c75},  {0xa75d, 0x4c6c},
  {0xa75e, 0x4c75},   {0xa75f, 0x4c6c},   {0xa760, 0x4c75},  {0xa761, 0x4c6c},
  {0xa762, 0x4c75},   {0xa763, 0x4c6c},   {0xa764, 0x4c75},  {0xa765, 0x4c6c},
  {0xa766, 0x4c75},   {0xa767, 0x4c6c},   {0xa768, 0x4c75},  {0xa769, 0x4c6c},
  {0xa76a, 0x4c75},   {0xa76b, 0x4c6c},   {0xa76c, 0x4c75},  {0xa76d, 0x4c6c},
  {0xa76e, 0x4c75},   {0xa76f, 0x4c6c},   {0xa770, 0x4c6d},  {0xa771, 0x4c6c},
  {0xa779, 0x4c75},   {0xa77a, 0x4c6c},   {0xa77b, 0x4c75},  {0xa77c, 0x4c6c},
  {0xa77d, 0x4c75},   {0xa77f, 0x4c6c},   {0xa780, 0x4c75},  {0xa781, 0x4c6c},
  {0xa782, 0x4c75},   {0xa783, 0x4c6c},   {0xa784, 0x4c75},  {0xa785, 0x4c6c},
  {0xa786, 0x4c75},   {0xa787, 0x4c6c},   {0xa788, 0x4c6d},  {0xa789, 0x536b},
  {0xa78b, 0x4c75},   {0xa78c, 0x4c6c},   {0xa78d, 0x4c75},  {0xa78e, 0x4c6c},
  {0xa78f, 0x4c6f},   {0xa790, 0x4c75},   {0xa791, 0x4c6c},  {0xa792, 0x4c75},
  {0xa793, 0x4c6c},   {0xa796, 0x4c75},   {0xa797, 0x4c6c},  {0xa798, 0x4c75},
  {0xa799, 0x4c6c},   {0xa79a, 0x4c75},   {0xa79b, 0x4c6c},  {0xa79c, 0x4c75},
  {0xa79d, 0x4c6c},   {0xa79e, 0x4c75},   {0xa79f, 0x4c6c},  {0xa7a0, 0x4c75},
  {0xa7a1, 0x4c6c},   {0xa7a2, 0x4c75},   {0xa7a3, 0x4c6c},  {0xa7a4, 0x4c75},
  {0xa7a5, 0x4c6c},   {0xa7a6, 0x4c75},   {0xa7a7, 0x4c6c},  {0xa7a8, 0x4c75},
  {0xa7a9, 0x4c6c},   {0xa7aa, 0x4c75},   {0xa7ae, 0x436e},  {0xa7b0, 0x4c75},
  {0xa7b5, 0x4c6c},   {0xa7b6, 0x4c75},   {0xa7b7, 0x4c6c},  {0xa7b8, 0x436e},
  {0xa7f7, 0x4c6f},   {0xa7f8, 0x4c6d},   {0xa7fa, 0x4c6c},  {0xa7fb, 0x4c6f},
  {0xa802, 0x4d6e},   {0xa803, 0x4c6f},   {0xa806, 0x4d6e},  {0xa807, 0x4c6f},
  {0xa80b, 0x4d6e},   {0xa80c, 0x4c6f},   {0xa823, 0x4d63},  {0xa825, 0x4d6e},
  {0xa827, 0x4d63},   {0xa828, 0x536f},   {0xa82c, 0x436e},  {0xa830, 0x4e6f},
  {0xa836, 0x536f},   {0xa838, 0x5363},   {0xa839, 0x536f},  {0xa83a, 0x436e},
  {0xa840, 0x4c6f},   {0xa874, 0x506f},   {0xa878, 0x436e},  {0xa880, 0x4d63},
  {0xa882, 0x4c6f},   {0xa8b4, 0x4d63},   {0xa8c4, 0x4d6e},  {0xa8c5, 0x436e},
  {0xa8ce, 0x506f},   {0xa8d0, 0x4e64},   {0xa8da, 0x436e},  {0xa8e0, 0x4d6e},
  {0xa8f2, 0x4c6f},   {0xa8f8, 0x506f},   {0xa8fb, 0x4c6f},  {0xa8fc, 0x506f},
  {0xa8fd, 0x4c6f},   {0xa8fe, 0x436e},   {0xa900, 0x4e64},  {0xa90a, 0x4c6f},
  {0xa926, 0x4d6e},   {0xa92e, 0x506f},   {0xa930, 0x4c6f},  {0xa947, 0x4d6e},
  {0xa952, 0x4d63},   {0xa954, 0x436e},   {0xa95f, 0x506f},  {0xa960, 0x4c6f},
  {0xa97d, 0x436e},   {0xa980, 0x4d6e},   {0xa983, 0x4d63},  {0xa984, 0x4c6f},
  {0xa9b3, 0x4d6e},   {0xa9b4, 0x4d63},   {0xa9b6, 0x4d6e},  {0xa9ba, 0x4d63},
  {0xa9bc, 0x4d6e},   {0xa9bd, 0x4d63},   {0xa9c1, 0x506f},  {0xa9ce, 0x436e},
  {0xa9cf, 0x4c6d},   {0xa9d0, 0x4e64},   {0xa9da, 0x436e},  {0xa9de, 0x506f},
  {0xa9e0, 0x4c6f},   {0xa9e5, 0x4d6e},   {0xa9e6, 0x4c6d},  {0xa9e7, 0x4c6f},
  {0xa9f0, 0x4e64},   {0xa9fa, 0x4c6f},   {0xa9ff, 0x436e},  {0xaa00, 0x4c6f},
  {0xaa29, 0x4d6e},   {0xaa2f, 0x4d63},   {0xaa31, 0x4d6e},  {0xaa33, 0x4d63},
  {0xaa35, 0x4d6e},   {0xaa37, 0x436e},   {0xaa40, 0x4c6f},  {0xaa43, 0x4d6e},
  {0xaa44, 0x4c6f},   {0xaa4c, 0x4d6e},   {0xaa4d, 0x4d63},  {0xaa4e, 0x436e},
  {0xaa50, 0x4e64},   {0xaa5a, 0x436e},   {0xaa5c, 0x506f},  {0xaa60, 0x4c6f},
  {0xaa70, 0x4c6d},   {0xaa71, 0x4c6f},   {0xaa77, 0x536f},  {0xaa7a, 0x4c6f},
  {0xaa7b, 0x4d63},   {0xaa7c, 0x4d6e},   {0xaa7d, 0x4d63},  {0xaa7e, 0x4c6f},
  {0xaab0, 0x4d6e},   {0xaab1, 0x4c6f},   {0xaab2, 0x4d6e},  {0xaab5, 0x4c6f},
  {0xaab7, 0x4d6e},   {0xaab9, 0x4c6f},   {0xaabe, 0x4d6e},  {0xaac0, 0x4c6f},
  {0xaac1, 0x4d6e},   {0xaac2, 0x4c6f},   {0xaac3, 0x436e},  {0xaadb, 0x4c6f},
  {0xaadd, 0x4c6d},   {0xaade, 0x506f},   {0xaae0, 0x4c6f},  {0xaaeb, 0x4d63},
  {0xaaec, 0x4d6e},   {0xaaee, 0x4d63},   {0xaaf0, 0x506f},  {0xaaf2, 0x4c6f},
  {0xaaf3, 0x4c6d},   {0xaaf5, 0x4d63},   {0xaaf6, 0x4d6e},  {0xaaf7, 0x436e},
  {0xab01, 0x4c6f},   {0xab07, 0x436e},   {0xab09, 0x4c6f},  {0xab0f, 0x436e},
  {0xab11, 0x4c6f},   {0xab17, 0x436e},   {0xab20, 0x4c6f},  {0xab27, 0x436e},
  {0xab28, 0x4c6f},   {0xab2f, 0x436e},   {0xab30, 0x4c6c},  {0xab5b, 0x536b},
  {0xab5c, 0x4c6d},   {0xab60, 0x4c6c},   {0xab66, 0x436e},  {0xab70, 0x4c6c},
  {0xabc0, 0x4c6f},   {0xabe3, 0x4d63},   {0xabe5, 0x4d6e},  {0xabe6, 0x4d63},
  {0xabe8, 0x4d6e},   {0xabe9, 0x4d63},   {0xabeb, 0x506f},  {0xabec, 0x4d63},
  {0xabed, 0x4d6e},   {0xabee, 0x436e},   {0xabf0, 0x4e64},  {0xabfa, 0x436e},
  {0xac00, 0x4c6f},   {0xd7a4, 0x436e},   {0xd7b0, 0x4c6f},  {0xd7c7, 0x436e},
  {0xd7cb, 0x4c6f},   {0xd7fc, 0x436e},   {0xd800, 0x4373},  {0xe000, 0x436f},
  {0xf900, 0x4c6f},   {0xfa6e, 0x436e},   {0xfa70, 0x4c6f},  {0xfada, 0x436e},
  {0xfb00, 0x4c6c},   {0xfb07, 0x436e},   {0xfb13, 0x4c6c},  {0xfb18, 0x436e},
  {0xfb1d, 0x4c6f},   {0xfb1e, 0x4d6e},   {0xfb1f, 0x4c6f},  {0xfb29, 0x536d},
  {0xfb2a, 0x4c6f},   {0xfb37, 0x436e},   {0xfb38, 0x4c6f},  {0xfb3d, 0x436e},
  {0xfb3e, 0x4c6f},   {0xfb3f, 0x436e},   {0xfb40, 0x4c6f},  {0xfb42, 0x436e},
  {0xfb43, 0x4c6f},   {0xfb45, 0x436e},   {0xfb46, 0x4c6f},  {0xfbb2, 0x536b},
  {0xfbc2, 0x436e},   {0xfbd3, 0x4c6f},   {0xfd3e, 0x5065},  {0xfd3f, 0x5073},
  {0xfd40, 0x436e},   {0xfd50, 0x4c6f},   {0xfd90, 0x436e},  {0xfd92, 0x4c6f},
  {0xfdc8, 0x436e},   {0xfdf0, 0x4c6f},   {0xfdfc, 0x5363},  {0xfdfd, 0x536f},
  {0xfdfe, 0x436e},   {0xfe00, 0x4d6e},   {0xfe10, 0x506f},  {0xfe17, 0x5073},
  {0xfe18, 0x5065},   {0xfe19, 0x506f},   {0xfe1a, 0x436e},  {0xfe20, 0x4d6e},
  {0xfe30, 0x506f},   {0xfe31, 0x5064},   {0xfe33, 0x5063},  {0xfe35, 0x5073},
  {0xfe36, 0x5065},   {0xfe37, 0x5073},   {0xfe38, 0x5065},  {0xfe39, 0x5073},
  {0xfe3a, 0x5065},   {0xfe3b, 0x5073},   {0xfe3c, 0x5065},  {0xfe3d, 0x5073},
  {0xfe3e, 0x5065},   {0xfe3f, 0x5073},   {0xfe40, 0x5065},  {0xfe41, 0x5073},
  {0xfe42, 0x5065},   {0xfe43, 0x5073},   {0xfe44, 0x5065},  {0xfe45, 0x506f},
  {0xfe47, 0x5073},   {0xfe48, 0x5065},   {0xfe49, 0x506f},  {0xfe4d, 0x5063},
  {0xfe50, 0x506f},   {0xfe53, 0x436e},   {0xfe54, 0x506f},  {0xfe58, 0x5064},
  {0xfe59, 0x5073},   {0xfe5a, 0x5065},   {0xfe5b, 0x5073},  {0xfe5c, 0x5065},
  {0xfe5d, 0x5073},   {0xfe5e, 0x5065},   {0xfe5f, 0x506f},  {0xfe62, 0x536d},
  {0xfe63, 0x5064},   {0xfe64, 0x536d},   {0xfe67, 0x436e},  {0xfe68, 0x506f},
  {0xfe69, 0x5363},   {0xfe6a, 0x506f},   {0xfe6c, 0x436e},  {0xfe70, 0x4c6f},
  {0xfe75, 0x436e},   {0xfe76, 0x4c6f},   {0xfefd, 0x436e},  {0xfeff, 0x4366},
  {0xff00, 0x436e},   {0xff01, 0x506f},   {0xff04, 0x5363},  {0xff05, 0x506f},
  {0xff08, 0x5073},   {0xff09, 0x5065},   {0xff0a, 0x506f},  {0xff0b, 0x536d},
  {0xff0c, 0x506f},   {0xff0d, 0x5064},   {0xff0e, 0x506f},  {0xff10, 0x4e64},
  {0xff1a, 0x506f},   {0xff1c, 0x536d},   {0xff1f, 0x506f},  {0xff21, 0x4c75},
  {0xff3b, 0x5073},   {0xff3c, 0x506f},   {0xff3d, 0x5065},  {0xff3e, 0x536b},
  {0xff3f, 0x5063},   {0xff40, 0x536b},   {0xff41, 0x4c6c},  {0xff5b, 0x5073},
  {0xff5c, 0x536d},   {0xff5d, 0x5065},   {0xff5e, 0x536d},  {0xff5f, 0x5073},
  {0xff60, 0x5065},   {0xff61, 0x506f},   {0xff62, 0x5073},  {0xff63, 0x5065},
  {0xff64, 0x506f},   {0xff66, 0x4c6f},   {0xff70, 0x4c6d},  {0xff71, 0x4c6f},
  {0xff9e, 0x4c6d},   {0xffa0, 0x4c6f},   {0xffbf, 0x436e},  {0xffc2, 0x4c6f},
  {0xffc8, 0x436e},   {0xffca, 0x4c6f},   {0xffd0, 0x436e},  {0xffd2, 0x4c6f},
  {0xffd8, 0x436e},   {0xffda, 0x4c6f},   {0xffdd, 0x436e},  {0xffe0, 0x5363},
  {0xffe2, 0x536d},   {0xffe3, 0x536b},   {0xffe4, 0x536f},  {0xffe5, 0x5363},
  {0xffe7, 0x436e},   {0xffe8, 0x536f},   {0xffe9, 0x536d},  {0xffed, 0x536f},
  {0xffef, 0x436e},   {0xfff9, 0x4366},   {0xfffc, 0x536f},  {0xfffe, 0x436e},
  {0x10000, 0x4c6f},  {0x1000c, 0x436e},  {0x1000d, 0x4c6f}, {0x10027, 0x436e},
  {0x10028, 0x4c6f},  {0x1003b, 0x436e},  {0x1003c, 0x4c6f}, {0x1003e, 0x436e},
  {0x1003f, 0x4c6f},  {0x1004e, 0x436e},  {0x10050, 0x4c6f}, {0x1005e, 0x436e},
  {0x10080, 0x4c6f},  {0x100fb, 0x436e},  {0x10100, 0x506f}, {0x10103, 0x436e},
  {0x10107, 0x4e6f},  {0x10134, 0x436e},  {0x10137, 0x536f}, {0x10140, 0x4e6c},
  {0x10175, 0x4e6f},  {0x10179, 0x536f},  {0x1018a, 0x4e6f}, {0x1018c, 0x536f},
  {0x1018d, 0x436e},  {0x10190, 0x536f},  {0x1019c, 0x436e}, {0x101a0, 0x536f},
  {0x101a1, 0x436e},  {0x101d0, 0x536f},  {0x101fd, 0x4d6e}, {0x101fe, 0x436e},
  {0x10280, 0x4c6f},  {0x1029d, 0x436e},  {0x102a0, 0x4c6f}, {0x102d1, 0x436e},
  {0x102e0, 0x4d6e},  {0x102e1, 0x4e6f},  {0x102fc, 0x436e}, {0x10300, 0x4c6f},
  {0x10320, 0x4e6f},  {0x10324, 0x436e},  {0x10330, 0x4c6f}, {0x10341, 0x4e6c},
  {0x10342, 0x4c6f},  {0x1034a, 0x4e6c},  {0x1034b, 0x436e}, {0x10350, 0x4c6f},
  {0x10376, 0x4d6e},  {0x1037b, 0x436e},  {0x10380, 0x4c6f}, {0x1039e, 0x436e},
  {0x1039f, 0x506f},  {0x103a0, 0x4c6f},  {0x103c4, 0x436e}, {0x103c8, 0x4c6f},
  {0x103d0, 0x506f},  {0x103d1, 0x4e6c},  {0x103d6, 0x436e}, {0x10400, 0x4c75},
  {0x10428, 0x4c6c},  {0x10450, 0x4c6f},  {0x1049e, 0x436e}, {0x104a0, 0x4e64},
  {0x104aa, 0x436e},  {0x10500, 0x4c6f},  {0x10528, 0x436e}, {0x10530, 0x4c6f},
  {0x10564, 0x436e},  {0x1056f, 0x506f},  {0x10570, 0x436e}, {0x10600, 0x4c6f},
  {0x10737, 0x436e},  {0x10740, 0x4c6f},  {0x10756, 0x436e}, {0x10760, 0x4c6f},
  {0x10768, 0x436e},  {0x10800, 0x4c6f},  {0x10806, 0x436e}, {0x10808, 0x4c6f},
  {0x10809, 0x436e},  {0x1080a, 0x4c6f},  {0x10836, 0x436e}, {0x10837, 0x4c6f},
  {0x10839, 0x436e},  {0x1083c, 0x4c6f},  {0x1083d, 0x436e}, {0x1083f, 0x4c6f},
  {0x10856, 0x436e},  {0x10857, 0x506f},  {0x10858, 0x4e6f}, {0x10860, 0x4c6f},
  {0x10877, 0x536f},  {0x10879, 0x4e6f},  {0x10880, 0x4c6f}, {0x1089f, 0x436e},
  {0x108a7, 0x4e6f},  {0x108b0, 0x436e},  {0x108e0, 0x4c6f}, {0x108f3, 0x436e},
  {0x108f4, 0x4c6f},  {0x108f6, 0x436e},  {0x108fb, 0x4e6f}, {0x10900, 0x4c6f},
  {0x10916, 0x4e6f},  {0x1091c, 0x436e},  {0x1091f, 0x506f}, {0x10920, 0x4c6f},
  {0x1093a, 0x436e},  {0x1093f, 0x506f},  {0x10940, 0x436e}, {0x10980, 0x4c6f},
  {0x109b8, 0x436e},  {0x109bc, 0x4e6f},  {0x109be, 0x4c6f}, {0x109c0, 0x4e6f},
  {0x109d0, 0x436e},  {0x109d2, 0x4e6f},  {0x10a00, 0x4c6f}, {0x10a01, 0x4d6e},
  {0x10a04, 0x436e},  {0x10a05, 0x4d6e},  {0x10a07, 0x436e}, {0x10a0c, 0x4d6e},
  {0x10a10, 0x4c6f},  {0x10a14, 0x436e},  {0x10a15, 0x4c6f}, {0x10a18, 0x436e},
  {0x10a19, 0x4c6f},  {0x10a34, 0x436e},  {0x10a38, 0x4d6e}, {0x10a3b, 0x436e},
  {0x10a3f, 0x4d6e},  {0x10a40, 0x4e6f},  {0x10a48, 0x436e}, {0x10a50, 0x506f},
  {0x10a59, 0x436e},  {0x10a60, 0x4c6f},  {0x10a7d, 0x4e6f}, {0x10a7f, 0x506f},
  {0x10a80, 0x4c6f},  {0x10a9d, 0x4e6f},  {0x10aa0, 0x436e}, {0x10ac0, 0x4c6f},
  {0x10ac8, 0x536f},  {0x10ac9, 0x4c6f},  {0x10ae5, 0x4d6e}, {0x10ae7, 0x436e},
  {0x10aeb, 0x4e6f},  {0x10af0, 0x506f},  {0x10af7, 0x436e}, {0x10b00, 0x4c6f},
  {0x10b36, 0x436e},  {0x10b39, 0x506f},  {0x10b40, 0x4c6f}, {0x10b56, 0x436e},
  {0x10b58, 0x4e6f},  {0x10b60, 0x4c6f},  {0x10b73, 0x436e}, {0x10b78, 0x4e6f},
  {0x10b80, 0x4c6f},  {0x10b92, 0x436e},  {0x10b99, 0x506f}, {0x10b9d, 0x436e},
  {0x10ba9, 0x4e6f},  {0x10bb0, 0x436e},  {0x10c00, 0x4c6f}, {0x10c49, 0x436e},
  {0x10c80, 0x4c75},  {0x10cb3, 0x436e},  {0x10cc0, 0x4c6c}, {0x10cf3, 0x436e},
  {0x10cfa, 0x4e6f},  {0x10d00, 0x436e},  {0x10e60, 0x4e6f}, {0x10e7f, 0x436e},
  {0x11000, 0x4d63},  {0x11001, 0x4d6e},  {0x11002, 0x4d63}, {0x11003, 0x4c6f},
  {0x11038, 0x4d6e},  {0x11047, 0x506f},  {0x1104e, 0x436e}, {0x11052, 0x4e6f},
  {0x11066, 0x4e64},  {0x11070, 0x436e},  {0x1107f, 0x4d6e}, {0x11082, 0x4d63},
  {0x11083, 0x4c6f},  {0x110b0, 0x4d63},  {0x110b3, 0x4d6e}, {0x110b7, 0x4d63},
  {0x110b9, 0x4d6e},  {0x110bb, 0x506f},  {0x110bd, 0x4366}, {0x110be, 0x506f},
  {0x110c2, 0x436e},  {0x110d0, 0x4c6f},  {0x110e9, 0x436e}, {0x110f0, 0x4e64},
  {0x110fa, 0x436e},  {0x11100, 0x4d6e},  {0x11103, 0x4c6f}, {0x11127, 0x4d6e},
  {0x1112c, 0x4d63},  {0x1112d, 0x4d6e},  {0x11135, 0x436e}, {0x11136, 0x4e64},
  {0x11140, 0x506f},  {0x11144, 0x436e},  {0x11150, 0x4c6f}, {0x11173, 0x4d6e},
  {0x11174, 0x506f},  {0x11176, 0x4c6f},  {0x11177, 0x436e}, {0x11180, 0x4d6e},
  {0x11182, 0x4d63},  {0x11183, 0x4c6f},  {0x111b3, 0x4d63}, {0x111b6, 0x4d6e},
  {0x111bf, 0x4d63},  {0x111c1, 0x4c6f},  {0x111c5, 0x506f}, {0x111ca, 0x4d6e},
  {0x111cd, 0x506f},  {0x111ce, 0x436e},  {0x111d0, 0x4e64}, {0x111da, 0x4c6f},
  {0x111db, 0x506f},  {0x111dc, 0x4c6f},  {0x111dd, 0x506f}, {0x111e0, 0x436e},
  {0x111e1, 0x4e6f},  {0x111f5, 0x436e},  {0x11200, 0x4c6f}, {0x11212, 0x436e},
  {0x11213, 0x4c6f},  {0x1122c, 0x4d63},  {0x1122f, 0x4d6e}, {0x11232, 0x4d63},
  {0x11234, 0x4d6e},  {0x11235, 0x4d63},  {0x11236, 0x4d6e}, {0x11238, 0x506f},
  {0x1123e, 0x436e},  {0x11280, 0x4c6f},  {0x11287, 0x436e}, {0x11288, 0x4c6f},
  {0x11289, 0x436e},  {0x1128a, 0x4c6f},  {0x1128e, 0x436e}, {0x1128f, 0x4c6f},
  {0x1129e, 0x436e},  {0x1129f, 0x4c6f},  {0x112a9, 0x506f}, {0x112aa, 0x436e},
  {0x112b0, 0x4c6f},  {0x112df, 0x4d6e},  {0x112e0, 0x4d63}, {0x112e3, 0x4d6e},
  {0x112eb, 0x436e},  {0x112f0, 0x4e64},  {0x112fa, 0x436e}, {0x11300, 0x4d6e},
  {0x11302, 0x4d63},  {0x11304, 0x436e},  {0x11305, 0x4c6f}, {0x1130d, 0x436e},
  {0x1130f, 0x4c6f},  {0x11311, 0x436e},  {0x11313, 0x4c6f}, {0x11329, 0x436e},
  {0x1132a, 0x4c6f},  {0x11331, 0x436e},  {0x11332, 0x4c6f}, {0x11334, 0x436e},
  {0x11335, 0x4c6f},  {0x1133a, 0x436e},  {0x1133c, 0x4d6e}, {0x1133d, 0x4c6f},
  {0x1133e, 0x4d63},  {0x11340, 0x4d6e},  {0x11341, 0x4d63}, {0x11345, 0x436e},
  {0x11347, 0x4d63},  {0x11349, 0x436e},  {0x1134b, 0x4d63}, {0x1134e, 0x436e},
  {0x11350, 0x4c6f},  {0x11351, 0x436e},  {0x11357, 0x4d63}, {0x11358, 0x436e},
  {0x1135d, 0x4c6f},  {0x11362, 0x4d63},  {0x11364, 0x436e}, {0x11366, 0x4d6e},
  {0x1136d, 0x436e},  {0x11370, 0x4d6e},  {0x11375, 0x436e}, {0x11480, 0x4c6f},
  {0x114b0, 0x4d63},  {0x114b3, 0x4d6e},  {0x114b9, 0x4d63}, {0x114ba, 0x4d6e},
  {0x114bb, 0x4d63},  {0x114bf, 0x4d6e},  {0x114c1, 0x4d63}, {0x114c2, 0x4d6e},
  {0x114c4, 0x4c6f},  {0x114c6, 0x506f},  {0x114c7, 0x4c6f}, {0x114c8, 0x436e},
  {0x114d0, 0x4e64},  {0x114da, 0x436e},  {0x11580, 0x4c6f}, {0x115af, 0x4d63},
  {0x115b2, 0x4d6e},  {0x115b6, 0x436e},  {0x115b8, 0x4d63}, {0x115bc, 0x4d6e},
  {0x115be, 0x4d63},  {0x115bf, 0x4d6e},  {0x115c1, 0x506f}, {0x115d8, 0x4c6f},
  {0x115dc, 0x4d6e},  {0x115de, 0x436e},  {0x11600, 0x4c6f}, {0x11630, 0x4d63},
  {0x11633, 0x4d6e},  {0x1163b, 0x4d63},  {0x1163d, 0x4d6e}, {0x1163e, 0x4d63},
  {0x1163f, 0x4d6e},  {0x11641, 0x506f},  {0x11644, 0x4c6f}, {0x11645, 0x436e},
  {0x11650, 0x4e64},  {0x1165a, 0x436e},  {0x11680, 0x4c6f}, {0x116ab, 0x4d6e},
  {0x116ac, 0x4d63},  {0x116ad, 0x4d6e},  {0x116ae, 0x4d63}, {0x116b0, 0x4d6e},
  {0x116b6, 0x4d63},  {0x116b7, 0x4d6e},  {0x116b8, 0x436e}, {0x116c0, 0x4e64},
  {0x116ca, 0x436e},  {0x11700, 0x4c6f},  {0x1171a, 0x436e}, {0x1171d, 0x4d6e},
  {0x11720, 0x4d63},  {0x11722, 0x4d6e},  {0x11726, 0x4d63}, {0x11727, 0x4d6e},
  {0x1172c, 0x436e},  {0x11730, 0x4e64},  {0x1173a, 0x4e6f}, {0x1173c, 0x506f},
  {0x1173f, 0x536f},  {0x11740, 0x436e},  {0x118a0, 0x4c75}, {0x118c0, 0x4c6c},
  {0x118e0, 0x4e64},  {0x118ea, 0x4e6f},  {0x118f3, 0x436e}, {0x118ff, 0x4c6f},
  {0x11900, 0x436e},  {0x11ac0, 0x4c6f},  {0x11af9, 0x436e}, {0x12000, 0x4c6f},
  {0x1239a, 0x436e},  {0x12400, 0x4e6c},  {0x1246f, 0x436e}, {0x12470, 0x506f},
  {0x12475, 0x436e},  {0x12480, 0x4c6f},  {0x12544, 0x436e}, {0x13000, 0x4c6f},
  {0x1342f, 0x436e},  {0x14400, 0x4c6f},  {0x14647, 0x436e}, {0x16800, 0x4c6f},
  {0x16a39, 0x436e},  {0x16a40, 0x4c6f},  {0x16a5f, 0x436e}, {0x16a60, 0x4e64},
  {0x16a6a, 0x436e},  {0x16a6e, 0x506f},  {0x16a70, 0x436e}, {0x16ad0, 0x4c6f},
  {0x16aee, 0x436e},  {0x16af0, 0x4d6e},  {0x16af5, 0x506f}, {0x16af6, 0x436e},
  {0x16b00, 0x4c6f},  {0x16b30, 0x4d6e},  {0x16b37, 0x506f}, {0x16b3c, 0x536f},
  {0x16b40, 0x4c6d},  {0x16b44, 0x506f},  {0x16b45, 0x536f}, {0x16b46, 0x436e},
  {0x16b50, 0x4e64},  {0x16b5a, 0x436e},  {0x16b5b, 0x4e6f}, {0x16b62, 0x436e},
  {0x16b63, 0x4c6f},  {0x16b78, 0x436e},  {0x16b7d, 0x4c6f}, {0x16b90, 0x436e},
  {0x16f00, 0x4c6f},  {0x16f45, 0x436e},  {0x16f50, 0x4c6f}, {0x16f51, 0x4d63},
  {0x16f7f, 0x436e},  {0x16f8f, 0x4d6e},  {0x16f93, 0x4c6d}, {0x16fa0, 0x436e},
  {0x1b000, 0x4c6f},  {0x1b002, 0x436e},  {0x1bc00, 0x4c6f}, {0x1bc6b, 0x436e},
  {0x1bc70, 0x4c6f},  {0x1bc7d, 0x436e},  {0x1bc80, 0x4c6f}, {0x1bc89, 0x436e},
  {0x1bc90, 0x4c6f},  {0x1bc9a, 0x436e},  {0x1bc9c, 0x536f}, {0x1bc9d, 0x4d6e},
  {0x1bc9f, 0x506f},  {0x1bca0, 0x4366},  {0x1bca4, 0x436e}, {0x1d000, 0x536f},
  {0x1d0f6, 0x436e},  {0x1d100, 0x536f},  {0x1d127, 0x436e}, {0x1d129, 0x536f},
  {0x1d165, 0x4d63},  {0x1d167, 0x4d6e},  {0x1d16a, 0x536f}, {0x1d16d, 0x4d63},
  {0x1d173, 0x4366},  {0x1d17b, 0x4d6e},  {0x1d183, 0x536f}, {0x1d185, 0x4d6e},
  {0x1d18c, 0x536f},  {0x1d1aa, 0x4d6e},  {0x1d1ae, 0x536f}, {0x1d1e9, 0x436e},
  {0x1d200, 0x536f},  {0x1d242, 0x4d6e},  {0x1d245, 0x536f}, {0x1d246, 0x436e},
  {0x1d300, 0x536f},  {0x1d357, 0x436e},  {0x1d360, 0x4e6f}, {0x1d372, 0x436e},
  {0x1d400, 0x4c75},  {0x1d41a, 0x4c6c},  {0x1d434, 0x4c75}, {0x1d44e, 0x4c6c},
  {0x1d455, 0x436e},  {0x1d456, 0x4c6c},  {0x1d468, 0x4c75}, {0x1d482, 0x4c6c},
  {0x1d49c, 0x4c75},  {0x1d49d, 0x436e},  {0x1d49e, 0x4c75}, {0x1d4a0, 0x436e},
  {0x1d4a2, 0x4c75},  {0x1d4a3, 0x436e},  {0x1d4a5, 0x4c75}, {0x1d4a7, 0x436e},
  {0x1d4a9, 0x4c75},  {0x1d4ad, 0x436e},  {0x1d4ae, 0x4c75}, {0x1d4b6, 0x4c6c},
  {0x1d4ba, 0x436e},  {0x1d4bb, 0x4c6c},  {0x1d4bc, 0x436e}, {0x1d4bd, 0x4c6c},
  {0x1d4c4, 0x436e},  {0x1d4c5, 0x4c6c},  {0x1d4d0, 0x4c75}, {0x1d4ea, 0x4c6c},
  {0x1d504, 0x4c75},  {0x1d506, 0x436e},  {0x1d507, 0x4c75}, {0x1d50b, 0x436e},
  {0x1d50d, 0x4c75},  {0x1d515, 0x436e},  {0x1d516, 0x4c75}, {0x1d51d, 0x436e},
  {0x1d51e, 0x4c6c},  {0x1d538, 0x4c75},  {0x1d53a, 0x436e}, {0x1d53b, 0x4c75},
  {0x1d53f, 0x436e},  {0x1d540, 0x4c75},  {0x1d545, 0x436e}, {0x1d546, 0x4c75},
  {0x1d547, 0x436e},  {0x1d54a, 0x4c75},  {0x1d551, 0x436e}, {0x1d552, 0x4c6c},
  {0x1d56c, 0x4c75},  {0x1d586, 0x4c6c},  {0x1d5a0, 0x4c75}, {0x1d5ba, 0x4c6c},
  {0x1d5d4, 0x4c75},  {0x1d5ee, 0x4c6c},  {0x1d608, 0x4c75}, {0x1d622, 0x4c6c},
  {0x1d63c, 0x4c75},  {0x1d656, 0x4c6c},  {0x1d670, 0x4c75}, {0x1d68a, 0x4c6c},
  {0x1d6a6, 0x436e},  {0x1d6a8, 0x4c75},  {0x1d6c1, 0x536d}, {0x1d6c2, 0x4c6c},
  {0x1d6db, 0x536d},  {0x1d6dc, 0x4c6c},  {0x1d6e2, 0x4c75}, {0x1d6fb, 0x536d},
  {0x1d6fc, 0x4c6c},  {0x1d715, 0x536d},  {0x1d716, 0x4c6c}, {0x1d71c, 0x4c75},
  {0x1d735, 0x536d},  {0x1d736, 0x4c6c},  {0x1d74f, 0x536d}, {0x1d750, 0x4c6c},
  {0x1d756, 0x4c75},  {0x1d76f, 0x536d},  {0x1d770, 0x4c6c}, {0x1d789, 0x536d},
  {0x1d78a, 0x4c6c},  {0x1d790, 0x4c75},  {0x1d7a9, 0x536d}, {0x1d7aa, 0x4c6c},
  {0x1d7c3, 0x536d},  {0x1d7c4, 0x4c6c},  {0x1d7ca, 0x4c75}, {0x1d7cb, 0x4c6c},
  {0x1d7cc, 0x436e},  {0x1d7ce, 0x4e64},  {0x1d800, 0x536f}, {0x1da00, 0x4d6e},
  {0x1da37, 0x536f},  {0x1da3b, 0x4d6e},  {0x1da6d, 0x536f}, {0x1da75, 0x4d6e},
  {0x1da76, 0x536f},  {0x1da84, 0x4d6e},  {0x1da85, 0x536f}, {0x1da87, 0x506f},
  {0x1da8c, 0x436e},  {0x1da9b, 0x4d6e},  {0x1daa0, 0x436e}, {0x1daa1, 0x4d6e},
  {0x1dab0, 0x436e},  {0x1e800, 0x4c6f},  {0x1e8c5, 0x436e}, {0x1e8c7, 0x4e6f},
  {0x1e8d0, 0x4d6e},  {0x1e8d7, 0x436e},  {0x1ee00, 0x4c6f}, {0x1ee04, 0x436e},
  {0x1ee05, 0x4c6f},  {0x1ee20, 0x436e},  {0x1ee21, 0x4c6f}, {0x1ee23, 0x436e},
  {0x1ee24, 0x4c6f},  {0x1ee25, 0x436e},  {0x1ee27, 0x4c6f}, {0x1ee28, 0x436e},
  {0x1ee29, 0x4c6f},  {0x1ee33, 0x436e},  {0x1ee34, 0x4c6f}, {0x1ee38, 0x436e},
  {0x1ee39, 0x4c6f},  {0x1ee3a, 0x436e},  {0x1ee3b, 0x4c6f}, {0x1ee3c, 0x436e},
  {0x1ee42, 0x4c6f},  {0x1ee43, 0x436e},  {0x1ee47, 0x4c6f}, {0x1ee48, 0x436e},
  {0x1ee49, 0x4c6f},  {0x1ee4a, 0x436e},  {0x1ee4b, 0x4c6f}, {0x1ee4c, 0x436e},
  {0x1ee4d, 0x4c6f},  {0x1ee50, 0x436e},  {0x1ee51, 0x4c6f}, {0x1ee53, 0x436e},
  {0x1ee54, 0x4c6f},  {0x1ee55, 0x436e},  {0x1ee57, 0x4c6f}, {0x1ee58, 0x436e},
  {0x1ee59, 0x4c6f},  {0x1ee5a, 0x436e},  {0x1ee5b, 0x4c6f}, {0x1ee5c, 0x436e},
  {0x1ee5d, 0x4c6f},  {0x1ee5e, 0x436e},  {0x1ee5f, 0x4c6f}, {0x1ee60, 0x436e},
  {0x1ee61, 0x4c6f},  {0x1ee63, 0x436e},  {0x1ee64, 0x4c6f}, {0x1ee65, 0x436e},
  {0x1ee67, 0x4c6f},  {0x1ee6b, 0x436e},  {0x1ee6c, 0x4c6f}, {0x1ee73, 0x436e},
  {0x1ee74, 0x4c6f},  {0x1ee78, 0x436e},  {0x1ee79, 0x4c6f}, {0x1ee7d, 0x436e},
  {0x1ee7e, 0x4c6f},  {0x1ee7f, 0x436e},  {0x1ee80, 0x4c6f}, {0x1ee8a, 0x436e},
  {0x1ee8b, 0x4c6f},  {0x1ee9c, 0x436e},  {0x1eea1, 0x4c6f}, {0x1eea4, 0x436e},
  {0x1eea5, 0x4c6f},  {0x1eeaa, 0x436e},  {0x1eeab, 0x4c6f}, {0x1eebc, 0x436e},
  {0x1eef0, 0x536d},  {0x1eef2, 0x436e},  {0x1f000, 0x536f}, {0x1f02c, 0x436e},
  {0x1f030, 0x536f},  {0x1f094, 0x436e},  {0x1f0a0, 0x536f}, {0x1f0af, 0x436e},
  {0x1f0b1, 0x536f},  {0x1f0c0, 0x436e},  {0x1f0c1, 0x536f}, {0x1f0d0, 0x436e},
  {0x1f0d1, 0x536f},  {0x1f0f6, 0x436e},  {0x1f100, 0x4e6f}, {0x1f10d, 0x436e},
  {0x1f110, 0x536f},  {0x1f12f, 0x436e},  {0x1f130, 0x536f}, {0x1f16c, 0x436e},
  {0x1f170, 0x536f},  {0x1f19b, 0x436e},  {0x1f1e6, 0x536f}, {0x1f203, 0x436e},
  {0x1f210, 0x536f},  {0x1f23b, 0x436e},  {0x1f240, 0x536f}, {0x1f249, 0x436e},
  {0x1f250, 0x536f},  {0x1f252, 0x436e},  {0x1f300, 0x536f}, {0x1f3fb, 0x536b},
  {0x1f400, 0x536f},  {0x1f57a, 0x436e},  {0x1f57b, 0x536f}, {0x1f5a4, 0x436e},
  {0x1f5a5, 0x536f},  {0x1f6d1, 0x436e},  {0x1f6e0, 0x536f}, {0x1f6ed, 0x436e},
  {0x1f6f0, 0x536f},  {0x1f6f4, 0x436e},  {0x1f700, 0x536f}, {0x1f774, 0x436e},
  {0x1f780, 0x536f},  {0x1f7d5, 0x436e},  {0x1f800, 0x536f}, {0x1f80c, 0x436e},
  {0x1f810, 0x536f},  {0x1f848, 0x436e},  {0x1f850, 0x536f}, {0x1f85a, 0x436e},
  {0x1f860, 0x536f},  {0x1f888, 0x436e},  {0x1f890, 0x536f}, {0x1f8ae, 0x436e},
  {0x1f910, 0x536f},  {0x1f919, 0x436e},  {0x1f980, 0x536f}, {0x1f985, 0x436e},
  {0x1f9c0, 0x536f},  {0x1f9c1, 0x436e},  {0x20000, 0x4c6f}, {0x2a6d7, 0x436e},
  {0x2a700, 0x4c6f},  {0x2b735, 0x436e},  {0x2b740, 0x4c6f}, {0x2b81e, 0x436e},
  {0x2b820, 0x4c6f},  {0x2cea2, 0x436e},  {0x2f800, 0x4c6f}, {0x2fa1e, 0x436e},
  {0xe0001, 0x4366},  {0xe0002, 0x436e},  {0xe0020, 0x4366}, {0xe0080, 0x436e},
  {0xe0100, 0x4d6e},  {0xe01f0, 0x436e},  {0xf0000, 0x436f}, {0xffffe, 0x436e},
  {0x100000, 0x436f}, {0x10fffe, 0x436e},
};

}  // namespace irs::utf8_utils
