// audio_primitives.hpp
/*
  neogfx C++ App/Game Engine
  Copyright (c) 2021 Leigh Johnston.  All Rights Reserved.
  
  This program is free software: you can redistribute it and / or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <neogfx/neogfx.hpp>

#include <array>
#include <bit>
#include <boost/lexical_cast.hpp>

#pragma once

namespace neogfx
{
    enum class note : std::uint32_t
    {
        MIDI0   = 0,
        MIDI1   = 1,
        MIDI2   = 2,
        MIDI3   = 3,
        MIDI4   = 4,
        MIDI5   = 5,
        MIDI6   = 6,
        MIDI7   = 7,
        MIDI8   = 8,
        MIDI9   = 9,
        MIDI10  = 10,
        MIDI11  = 11,
        MIDI12  = 12,
        MIDI13  = 13,
        MIDI14  = 14,
        MIDI15  = 15,
        MIDI16  = 16,
        MIDI17  = 17,
        MIDI18  = 18,
        MIDI19  = 19,
        MIDI20  = 20,

        A0      = 21,
        Bb0     = 22,
        B0      = 23,
        C1      = 24,
        Db1     = 25,
        D1      = 26,
        Eb1     = 27,
        E1      = 28,
        F1      = 29,
        Gb1     = 30,
        G1      = 31,
        Ab1     = 32,
        A1      = 33,
        Bb1     = 34,
        B1      = 35,
        C2      = 36,
        Db2     = 37,
        D2      = 38,
        Eb2     = 39,
        E2      = 40,
        F2      = 41,
        Gb2     = 42,
        G2      = 43,
        Ab2     = 44,
        A2      = 45,
        Bb2     = 46,
        B2      = 47,
        C3      = 48,
        Db3     = 49,
        D3      = 50,
        Eb3     = 51,
        E3      = 52,
        F3      = 53,
        Gb3     = 54,
        G3      = 55,
        Ab3     = 56,
        A3      = 57,
        Bb3     = 58,
        B3      = 59,
        C4      = 60,
        Db4     = 61,
        D4      = 62,
        Eb4     = 63,
        E4      = 64,
        F4      = 65,
        Gb4     = 66,
        G4      = 67,
        Ab4     = 68,
        A4      = 69,
        Bb4     = 70,
        B4      = 71,
        C5      = 72,
        Db5     = 73,
        D5      = 74,
        Eb5     = 75,
        E5      = 76,
        F5      = 77,
        Gb5     = 78,
        G5      = 79,
        Ab5     = 80,
        A5      = 81,
        Bb5     = 82,
        B5      = 83,
        C6      = 84,
        Db6     = 85,
        D6      = 86,
        Eb6     = 87,
        E6      = 88,
        F6      = 89,
        Gb6     = 90,
        G6      = 91,
        Ab6     = 92,
        A6      = 93,
        Bb6     = 94,
        B6      = 95,
        C7      = 96,
        Db7     = 97,
        D7      = 98,
        Eb7     = 99,
        E7      = 100,
        F7      = 101,
        Gb7     = 102,
        G7      = 103,
        Ab7     = 104,
        A7      = 105,
        Bb7     = 106,
        B7      = 107,
        C8      = 108,
        Db8     = 109,
        D8      = 110,
        Eb8     = 111,
        E8      = 112,
        F8      = 113,
        Gb8     = 114,
        G8      = 115,
        Ab8     = 116,
        A8      = 117,
        Bb8     = 118,
        B8      = 119,
        C9      = 120,
        Db9     = 121,
        D9      = 122,
        Eb9     = 123,
        E9      = 124,
        F9      = 125,
        Gb9     = 126,
        G9      = 127,
        Ab9     = 128
    };

    inline std::string to_string(note const& aNote)
    {
        static std::unordered_map<note, std::string> const sMap =
        {
            { note::MIDI0   , "MIDI0"  },
            { note::MIDI1   , "MIDI1"  },
            { note::MIDI2   , "MIDI2"  },
            { note::MIDI3   , "MIDI3"  },
            { note::MIDI4   , "MIDI4"  },
            { note::MIDI5   , "MIDI5"  },
            { note::MIDI6   , "MIDI6"  },
            { note::MIDI7   , "MIDI7"  },
            { note::MIDI8   , "MIDI8"  },
            { note::MIDI9   , "MIDI9"  },
            { note::MIDI10  , "MIDI10"  },
            { note::MIDI11  , "MIDI11"  },
            { note::MIDI12  , "MIDI12"  },
            { note::MIDI13  , "MIDI13"  },
            { note::MIDI14  , "MIDI14"  },
            { note::MIDI15  , "MIDI15"  },
            { note::MIDI16  , "MIDI16"  },
            { note::MIDI17  , "MIDI17"  },
            { note::MIDI18  , "MIDI18"  },
            { note::MIDI19  , "MIDI19"  },
            { note::MIDI20  , "MIDI20"  },

            { note::A0  , "A0"  },
            { note::Bb0 , "Bb0" },
            { note::B0  , "B0"  },
            { note::C1  , "C1"  },
            { note::Db1 , "Db1" },
            { note::D1  , "D1"  },
            { note::Eb1 , "Eb1" },
            { note::E1  , "E1"  },
            { note::F1  , "F1"  },
            { note::Gb1 , "Gb1" },
            { note::G1  , "G1"  },
            { note::Ab1 , "Ab1" },
            { note::A1  , "A1"  },
            { note::Bb1 , "Bb1" },
            { note::B1  , "B1"  },
            { note::C2  , "C2"  },
            { note::Db2 , "Db2" },
            { note::D2  , "D2"  },
            { note::Eb2 , "Eb2" },
            { note::E2  , "E2"  },
            { note::F2  , "F2"  },
            { note::Gb2 , "Gb2" },
            { note::G2  , "G2"  },
            { note::Ab2 , "Ab2" },
            { note::A2  , "A2"  },
            { note::Bb2 , "Bb2" },
            { note::B2  , "B2"  },
            { note::C3  , "C3"  },
            { note::Db3 , "Db3" },
            { note::D3  , "D3"  },
            { note::Eb3 , "Eb3" },
            { note::E3  , "E3"  },
            { note::F3  , "F3"  },
            { note::Gb3 , "Gb3" },
            { note::G3  , "G3"  },
            { note::Ab3 , "Ab3" },
            { note::A3  , "A3"  },
            { note::Bb3 , "Bb3" },
            { note::B3  , "B3"  },
            { note::C4  , "C4"  },
            { note::Db4 , "Db4" },
            { note::D4  , "D4"  },
            { note::Eb4 , "Eb4" },
            { note::E4  , "E4"  },
            { note::F4  , "F4"  },
            { note::Gb4 , "Gb4" },
            { note::G4  , "G4"  },
            { note::Ab4 , "Ab4" },
            { note::A4  , "A4"  },
            { note::Bb4 , "Bb4" },
            { note::B4  , "B4"  },
            { note::C5  , "C5"  },
            { note::Db5 , "Db5" },
            { note::D5  , "D5"  },
            { note::Eb5 , "Eb5" },
            { note::E5  , "E5"  },
            { note::F5  , "F5"  },
            { note::Gb5 , "Gb5" },
            { note::G5  , "G5"  },
            { note::Ab5 , "Ab5" },
            { note::A5  , "A5"  },
            { note::Bb5 , "Bb5" },
            { note::B5  , "B5"  },
            { note::C6  , "C6"  },
            { note::Db6 , "Db6" },
            { note::D6  , "D6"  },
            { note::Eb6 , "Eb6" },
            { note::E6  , "E6"  },
            { note::F6  , "F6"  },
            { note::Gb6 , "Gb6" },
            { note::G6  , "G6"  },
            { note::Ab6 , "Ab6" },
            { note::A6  , "A6"  },
            { note::Bb6 , "Bb6" },
            { note::B6  , "B6"  },
            { note::C7  , "C7"  },
            { note::Db7 , "Db7" },
            { note::D7  , "D7"  },
            { note::Eb7 , "Eb7" },
            { note::E7  , "E7"  },
            { note::F7  , "F7"  },
            { note::Gb7 , "Gb7" },
            { note::G7  , "G7"  },
            { note::Ab7 , "Ab7" },
            { note::A7  , "A7"  },
            { note::Bb7 , "Bb7" },
            { note::B7  , "B7"  },
            { note::C8  , "C8"  },
            { note::Db8 , "Db8" },
            { note::D8  , "D8"  },
            { note::Eb8 , "Eb8" },
            { note::E8  , "E8"  },
            { note::F8  , "F8"  },
            { note::Gb8 , "Gb8" },
            { note::G8  , "G8"  },
            { note::Ab8 , "Ab8" },
            { note::A8  , "A8"  },
            { note::Bb8 , "Bb8" },
            { note::B8  , "B8"  },
            { note::C9  , "C9"  },
            { note::Db9 , "Db9" },
            { note::D9  , "D9"  },
            { note::Eb9 , "Eb9" },
            { note::E9  , "E9"  },
            { note::F9  , "F9"  },
            { note::Gb9 , "Gb9" },
            { note::G9  , "G9"  },
            { note::Ab9 , "Ab9" }
        };

        return sMap.at(aNote);
    }

    inline note string_to_note(std::string const& aNote)
    {
        if (std::isdigit(aNote.at(0)))
            return static_cast<note>(boost::lexical_cast<std::uint32_t>(aNote));

        static std::unordered_map<std::string, note> const sMap =
        {
            { "A0" , note::A0  },
            { "A#0", note::Bb0 },
            { "Bb0", note::Bb0 },
            { "B0" , note::B0  },
            { "C1" , note::C1  },
            { "C#1", note::Db1 },
            { "Db1", note::Db1 },
            { "D1" , note::D1  },
            { "D#1", note::Eb1 },
            { "Eb1", note::Eb1 },
            { "E1" , note::E1  },
            { "F1" , note::F1  },
            { "F#1", note::Gb1 },
            { "Gb1", note::Gb1 },
            { "G1" , note::G1  },
            { "G#1", note::Ab1 },
            { "Ab1", note::Ab1 },
            { "A1" , note::A1  },
            { "A#1", note::Bb1 },
            { "Bb1", note::Bb1 },
            { "B1" , note::B1  },
            { "C2" , note::C2  },
            { "C#2", note::Db2 },
            { "Db2", note::Db2 },
            { "D2" , note::D2  },
            { "D#2", note::Eb2 },
            { "Eb2", note::Eb2 },
            { "E2" , note::E2  },
            { "F2" , note::F2  },
            { "F#2", note::Gb2 },
            { "Gb2", note::Gb2 },
            { "G2" , note::G2  },
            { "G#2", note::Ab2 },
            { "Ab2", note::Ab2 },
            { "A2" , note::A2  },
            { "A#2", note::Bb2 },
            { "Bb2", note::Bb2 },
            { "B2" , note::B2  },
            { "C3" , note::C3  },
            { "C#3", note::Db3 },
            { "Db3", note::Db3 },
            { "D3" , note::D3  },
            { "D#3", note::Eb3 },
            { "Eb3", note::Eb3 },
            { "E3" , note::E3  },
            { "F3" , note::F3  },
            { "F#3", note::Gb3 },
            { "Gb3", note::Gb3 },
            { "G3" , note::G3  },
            { "G#3", note::Ab3 },
            { "Ab3", note::Ab3 },
            { "A3" , note::A3  },
            { "A#3", note::Bb3 },
            { "Bb3", note::Bb3 },
            { "B3" , note::B3  },
            { "C4" , note::C4  },
            { "C#4", note::Db4 },
            { "Db4", note::Db4 },
            { "D4" , note::D4  },
            { "D#4", note::Eb4 },
            { "Eb4", note::Eb4 },
            { "E4" , note::E4  },
            { "F4" , note::F4  },
            { "F#4", note::Gb4 },
            { "Gb4", note::Gb4 },
            { "G4" , note::G4  },
            { "G#4", note::Ab4 },
            { "Ab4", note::Ab4 },
            { "A4" , note::A4  },
            { "A#4", note::Bb4 },
            { "Bb4", note::Bb4 },
            { "B4" , note::B4  },
            { "C5" , note::C5  },
            { "C#5", note::Db5 },
            { "Db5", note::Db5 },
            { "D5" , note::D5  },
            { "D#5", note::Eb5 },
            { "Eb5", note::Eb5 },
            { "E5" , note::E5  },
            { "F5" , note::F5  },
            { "F#5", note::Gb5 },
            { "Gb5", note::Gb5 },
            { "G5" , note::G5  },
            { "G#5", note::Ab5 },
            { "Ab5", note::Ab5 },
            { "A5" , note::A5  },
            { "A#5", note::Bb5 },
            { "Bb5", note::Bb5 },
            { "B5" , note::B5  },
            { "C6" , note::C6  },
            { "C#6", note::Db6 },
            { "Db6", note::Db6 },
            { "D6" , note::D6  },
            { "D#6", note::Eb6 },
            { "Eb6", note::Eb6 },
            { "E6" , note::E6  },
            { "F6" , note::F6  },
            { "F#6", note::Gb6 },
            { "Gb6", note::Gb6 },
            { "G6" , note::G6  },
            { "G#6", note::Ab6 },
            { "Ab6", note::Ab6 },
            { "A6" , note::A6  },
            { "A#6", note::Bb6 },
            { "Bb6", note::Bb6 },
            { "B6" , note::B6  },
            { "C7" , note::C7  },
            { "C#7", note::Db7 },
            { "Db7", note::Db7 },
            { "D7" , note::D7  },
            { "D#7", note::Eb7 },
            { "Eb7", note::Eb7 },
            { "E7" , note::E7  },
            { "F7" , note::F7  },
            { "F#7", note::Gb7 },
            { "Gb7", note::Gb7 },
            { "G7" , note::G7  },
            { "G#7", note::Ab7 },
            { "Ab7", note::Ab7 },
            { "A7" , note::A7  },
            { "A#7", note::Bb7 },
            { "Bb7", note::Bb7 },
            { "B7" , note::B7  },
            { "C8" , note::C8  },
            { "C#8", note::Db8 },
            { "Db8", note::Db8 },
            { "D8" , note::D8  },
            { "D#8", note::Eb8 },
            { "Eb8", note::Eb8 },
            { "E8" , note::E8  },
            { "F8" , note::F8  },
            { "F#8", note::Gb8 },
            { "Gb8", note::Gb8 },
            { "G8" , note::G8  },
            { "G#8", note::Ab8 },
            { "Ab8", note::Ab8 },
            { "A8" , note::A8  },
            { "A#8", note::Bb8 },
            { "Bb8", note::Bb8 },
            { "B8" , note::B8  },
            { "C9" , note::C9  },
            { "C#9", note::Db9 },
            { "Db9", note::Db9 },
            { "D9" , note::D9  },
            { "D#9", note::Eb9 },
            { "Eb9", note::Eb9 },
            { "E9" , note::E9  },
            { "F9" , note::F9  },
            { "F#9", note::Gb9 },
            { "Gb9", note::Gb9 },
            { "G9" , note::G9  },
            { "G#9", note::Ab9 },
            { "Ab9", note::Ab9 }
        };

        return sMap.at(aNote);
    }

    inline float frequency(neogfx::note note)
    {
        return std::pow(2.0f, (static_cast<float>(note) - 69.0f) / 12.0f) * 440.0f;
    }

    template <neogfx::note Note>
    inline float frequency()
    {
        return frequency(Note);
    }

    enum class instrument : std::uint32_t
    {
        PureTone            = 0,
        AcousticGrandPiano  = 1,
        BrightAcousticPiano = 2,
        ElectricGrandPiano  = 3,
        HonkytonkPiano      = 4,
        ElectricPiano1      = 5,
        ElectricPiano2      = 6,
        Harpsichord         = 7,
        Clavi               = 8,  
        Celesta             = 9,
        Glockenspiel        = 10,
        MusicBox            = 11,
        Vibraphone          = 12,
        Marimba             = 13,
        Xylophone           = 14,
        TubularBells        = 15,
        Dulcimer            = 16,
        DrawbarOrgan        = 17,
        PercussiveOrgan     = 18,
        RockOrgan           = 19,
        ChurchOrgan         = 20,
        ReedOrgan           = 21,
        Accordion           = 22,
        Harmonica           = 23,
        TangoAccordion      = 24,
        AcousticGuitarNylon = 25,
        AcousticGuitarSteel = 26,
        ElectricGuitarJazz  = 27,
        ElectricGuitarClean = 28,
        ElectricGuitarMuted = 29,
        OverdrivenGuitar    = 30,
        DistortionGuitar    = 31,
        Guitarharmonics     = 32,
        AcousticBass        = 33,
        ElectricBassFinger  = 34,
        ElectricBassPick    = 35,
        FretlessBass        = 36,
        SlapBass1           = 37,
        SlapBass2           = 38,
        SynthBass1          = 39,
        SynthBass2          = 40,
        Violin              = 41,
        Viola               = 42,
        Cello               = 43,
        Contrabass          = 44,
        TremoloStrings      = 45,
        PizzicatoStrings    = 46,
        OrchestralHarp      = 47,
        Timpani             = 48,
        StringEnsemble1     = 49,
        StringEnsemble2     = 50,
        SynthStrings1       = 51,
        SynthStrings2       = 52,
        ChoirAahs           = 53,
        VoiceOohs           = 54,
        SynthVoice          = 55,
        OrchestraHit        = 56,
        Trumpet             = 57,
        Trombone            = 58,
        Tuba                = 59,
        MutedTrumpet        = 60,
        FrenchHorn          = 61,
        BrassSection        = 62,
        SynthBrass1         = 63,
        SynthBrass2         = 64,
        SopranoSax          = 65,
        AltoSax             = 66,
        TenorSax            = 67,
        BaritoneSax         = 68,
        Oboe                = 69,
        EnglishHorn         = 70,
        Bassoon             = 71,
        Clarinet            = 72,
        Piccolo             = 73,
        Flute               = 74,
        Recorder            = 75,
        PanFlute            = 76,
        BlownBottle         = 77,
        Shakuhachi          = 78,
        Whistle             = 79,
        Ocarina             = 80,
        Lead1               = 81,
        Lead2               = 82,
        Lead3               = 83,
        Lead4               = 84,
        Lead5               = 85,
        Lead6               = 86,
        Lead7               = 87,
        Lead8               = 88,
        Pad1                = 89,
        Pad2                = 90,
        Pad3                = 91,
        Pad4                = 92,
        Pad5                = 93,
        Pad6                = 94,
        Pad7                = 95,
        Pad8                = 96,
        FX1                 = 97,
        FX2                 = 98,
        FX3                 = 99,
        FX4                 = 100,
        FX5                 = 101,
        FX6                 = 102,
        FX7                 = 103,
        FX8                 = 104,
        Sitar               = 105,
        Banjo               = 106,
        Shamisen            = 107,
        Koto                = 108,
        Kalimba             = 109,
        Bagpipe             = 110,
        Fiddle              = 111,
        Shanai              = 112,
        TinkleBell          = 113,
        Agogo               = 114,
        SteelDrums          = 115,
        Woodblock           = 116,
        TaikoDrum           = 117,
        MelodicTom          = 118,
        SynthDrum           = 119,
        ReverseCymbal       = 120,
        GuitarFretNoise     = 121,
        BreathNoise         = 122,
        Seashore            = 123,
        BirdTweet           = 124,
        TelephoneRing       = 125,
        Helicopter          = 126,
        Applause            = 127,
        Gunshot             = 128
    };

    enum class percussion_instrument
    {
        AcousticBassDrum    = 35,
        BassDrum1           = 36,
        SideStick           = 37,
        AcousticSnare       = 38,
        HandClap            = 39,
        ElectricSnare       = 40,
        LowFloorTom         = 41,
        ClosedHiHat         = 42,
        HighFloorTom        = 43,
        PedalHiHat          = 44,
        LowTom              = 45,
        OpenHiHat           = 46,
        LowMidTom           = 47,
        HiMidTom            = 48,
        CrashCymbal1        = 49,
        HighTom             = 50,
        RideCymbal1         = 51,
        ChineseCymbal       = 52,
        RideBell            = 53,
        Tambourine          = 54,
        SplashCymbal        = 55,
        Cowbell             = 56,
        CrashCymbal2        = 57,
        Vibraslap           = 58,
        RideCymbal2         = 59,
        HiBongo             = 60,
        LowBongo            = 61,
        MuteHiConga         = 62,
        OpenHiConga         = 63,
        LowConga            = 64,
        HighTimbale         = 65,
        LowTimbale          = 66,
        HighAgogo           = 67,
        LowAgogo            = 68,
        Cabasa              = 69,
        Maracas             = 70,
        ShortWhistle        = 71,
        LongWhistle         = 72,
        ShortGuiro          = 73,
        LongGuiro           = 74,
        Claves              = 75,
        HiWoodBlock         = 76,
        LowWoodBlock        = 77,
        MuteCuica           = 78,
        OpenCuica           = 79,
        MuteTriangle        = 80,
        OpenTriangle        = 81
    };

    inline std::string to_string(neogfx::instrument instrument)
    {
        static std::unordered_map<neogfx::instrument, std::string> const sMap = 
        {
            { neogfx::instrument::AcousticGrandPiano,   "Acoustic Grand Piano" },
            { neogfx::instrument::BrightAcousticPiano,  "Bright Acoustic Piano" },
            { neogfx::instrument::ElectricGrandPiano,   "Electric Grand Piano" },
            { neogfx::instrument::HonkytonkPiano,       "Honky - tonk Piano" },
            { neogfx::instrument::ElectricPiano1,       "Electric Piano 1" },
            { neogfx::instrument::ElectricPiano2,       "Electric Piano 2" },
            { neogfx::instrument::Harpsichord,          "Harpsichord" },
            { neogfx::instrument::Clavi,                "Clavi" },
            { neogfx::instrument::Celesta,              "Celesta" },
            { neogfx::instrument::Glockenspiel,         "Glockenspiel" },
            { neogfx::instrument::MusicBox,             "Music Box" },
            { neogfx::instrument::Vibraphone,           "Vibraphone" },
            { neogfx::instrument::Marimba,              "Marimba" },
            { neogfx::instrument::Xylophone,            "Xylophone" },
            { neogfx::instrument::TubularBells,         "Tubular Bells" },
            { neogfx::instrument::Dulcimer,             "Dulcimer" },
            { neogfx::instrument::DrawbarOrgan,         "Drawbar Organ" },
            { neogfx::instrument::PercussiveOrgan,      "Percussive Organ" },
            { neogfx::instrument::RockOrgan,            "Rock Organ" },
            { neogfx::instrument::ChurchOrgan,          "Church Organ" },
            { neogfx::instrument::ReedOrgan,            "Reed Organ" },
            { neogfx::instrument::Accordion,            "Accordion" },
            { neogfx::instrument::Harmonica,            "Harmonica" },
            { neogfx::instrument::TangoAccordion,       "Tango Accordion" },
            { neogfx::instrument::AcousticGuitarNylon,       "Acoustic Guitar(nylon)" },
            { neogfx::instrument::AcousticGuitarSteel,       "Acoustic Guitar(steel)" },
            { neogfx::instrument::ElectricGuitarJazz,       "Electric Guitar(jazz)" },
            { neogfx::instrument::ElectricGuitarClean,       "Electric Guitar(clean)" },
            { neogfx::instrument::ElectricGuitarMuted,       "Electric Guitar(muted)" },
            { neogfx::instrument::OverdrivenGuitar,     "Overdriven Guitar" },
            { neogfx::instrument::DistortionGuitar,     "Distortion Guitar" },
            { neogfx::instrument::Guitarharmonics,      "Guitar harmonics" },
            { neogfx::instrument::AcousticBass,         "Acoustic Bass" },
            { neogfx::instrument::ElectricBassFinger,   "Electric Bass(finger)" },
            { neogfx::instrument::ElectricBassPick,     "Electric Bass(pick)" },
            { neogfx::instrument::FretlessBass,         "Fretless Bass" },
            { neogfx::instrument::SlapBass1,            "Slap Bass 1" },
            { neogfx::instrument::SlapBass2,            "Slap Bass 2" },
            { neogfx::instrument::SynthBass1,           "Synth Bass 1" },
            { neogfx::instrument::SynthBass2,           "Synth Bass 2" },
            { neogfx::instrument::Violin,               "Violin" },
            { neogfx::instrument::Viola,                "Viola" },
            { neogfx::instrument::Cello,                "Cello" },
            { neogfx::instrument::Contrabass,           "Contrabass" },
            { neogfx::instrument::TremoloStrings,       "Tremolo Strings" },
            { neogfx::instrument::PizzicatoStrings,     "Pizzicato Strings" },
            { neogfx::instrument::OrchestralHarp,       "Orchestral Harp" },
            { neogfx::instrument::Timpani,              "Timpani" },
            { neogfx::instrument::StringEnsemble1,      "String Ensemble 1" },
            { neogfx::instrument::StringEnsemble2,      "String Ensemble 2" },
            { neogfx::instrument::SynthStrings1,        "SynthStrings 1" },
            { neogfx::instrument::SynthStrings2,        "SynthStrings 2" },
            { neogfx::instrument::ChoirAahs,            "Choir Aahs" },
            { neogfx::instrument::VoiceOohs,            "Voice Oohs" },
            { neogfx::instrument::SynthVoice,           "Synth Voice" },
            { neogfx::instrument::OrchestraHit,         "Orchestra Hit" },
            { neogfx::instrument::Trumpet,              "Trumpet" },
            { neogfx::instrument::Trombone,             "Trombone" },
            { neogfx::instrument::Tuba,                 "Tuba" },
            { neogfx::instrument::MutedTrumpet,         "Muted Trumpet" },
            { neogfx::instrument::FrenchHorn,           "French Horn" },
            { neogfx::instrument::BrassSection,         "Brass Section" },
            { neogfx::instrument::SynthBrass1,          "SynthBrass 1" },
            { neogfx::instrument::SynthBrass2,          "SynthBrass 2" },
            { neogfx::instrument::SopranoSax,           "Soprano Sax" },
            { neogfx::instrument::AltoSax,              "Alto Sax" },
            { neogfx::instrument::TenorSax,             "Tenor Sax" },
            { neogfx::instrument::BaritoneSax,          "Baritone Sax" },
            { neogfx::instrument::Oboe,                 "Oboe" },
            { neogfx::instrument::EnglishHorn,          "English Horn" },
            { neogfx::instrument::Bassoon,              "Bassoon" },
            { neogfx::instrument::Clarinet,             "Clarinet" },
            { neogfx::instrument::Piccolo,              "Piccolo" },
            { neogfx::instrument::Flute,                "Flute" },
            { neogfx::instrument::Recorder,             "Recorder" },
            { neogfx::instrument::PanFlute,             "Pan Flute" },
            { neogfx::instrument::BlownBottle,          "Blown Bottle" },
            { neogfx::instrument::Shakuhachi,           "Shakuhachi" },
            { neogfx::instrument::Whistle,              "Whistle" },
            { neogfx::instrument::Ocarina,              "Ocarina" },
            { neogfx::instrument::Lead1,                "Lead 1 (square)" },
            { neogfx::instrument::Lead2,                "Lead 2 (sawtooth)" },
            { neogfx::instrument::Lead3,                "Lead 3 (calliope)" },
            { neogfx::instrument::Lead4,                "Lead 4 (chiff)" },
            { neogfx::instrument::Lead5,                "Lead 5 (charang)" },
            { neogfx::instrument::Lead6,                "Lead 6 (voice)" },
            { neogfx::instrument::Lead7,                "Lead 7 (fifths)" },
            { neogfx::instrument::Lead8,                "Lead 8 (bass + lead)" },
            { neogfx::instrument::Pad1,                 "Pad 1 (new age)" },
            { neogfx::instrument::Pad2,                 "Pad 2 (warm)" },
            { neogfx::instrument::Pad3,                 "Pad 3 (polysynth)" },
            { neogfx::instrument::Pad4,                 "Pad 4 (choir)" },
            { neogfx::instrument::Pad5,                 "Pad 5 (bowed)" },
            { neogfx::instrument::Pad6,                 "Pad 6 (metallic)" },
            { neogfx::instrument::Pad7,                 "Pad 7 (halo)" },
            { neogfx::instrument::Pad8,                 "Pad 8 (sweep)" },
            { neogfx::instrument::FX1,                  "FX 1 (rain)" },
            { neogfx::instrument::FX2,                  "FX 2 (soundtrack)" },
            { neogfx::instrument::FX3,                  "FX 3 (crystal)" },
            { neogfx::instrument::FX4,                  "FX 4 (atmosphere)" },
            { neogfx::instrument::FX5,                  "FX 5 (brightness)" },
            { neogfx::instrument::FX6,                  "FX 6 (goblins)" },
            { neogfx::instrument::FX7,                  "FX 7 (echoes)" },
            { neogfx::instrument::FX8,                  "FX 8 (sci - fi)" },
            { neogfx::instrument::Sitar,                "Sitar" },
            { neogfx::instrument::Banjo,                "Banjo" },
            { neogfx::instrument::Shamisen,             "Shamisen" },
            { neogfx::instrument::Koto,                 "Koto" },
            { neogfx::instrument::Kalimba,              "Kalimba" },
            { neogfx::instrument::Bagpipe,              "Bag pipe" },
            { neogfx::instrument::Fiddle,               "Fiddle" },
            { neogfx::instrument::Shanai,               "Shanai" },
            { neogfx::instrument::TinkleBell,           "Tinkle Bell" },
            { neogfx::instrument::Agogo,                "Agogo" },
            { neogfx::instrument::SteelDrums,           "Steel Drums" },
            { neogfx::instrument::Woodblock,            "Woodblock" },
            { neogfx::instrument::TaikoDrum,            "Taiko Drum" },
            { neogfx::instrument::MelodicTom,           "Melodic Tom" },
            { neogfx::instrument::SynthDrum,            "Synth Drum" },
            { neogfx::instrument::ReverseCymbal,        "Reverse Cymbal" },
            { neogfx::instrument::GuitarFretNoise,      "Guitar Fret Noise" },
            { neogfx::instrument::BreathNoise,          "Breath Noise" },
            { neogfx::instrument::Seashore,             "Seashore" },
            { neogfx::instrument::BirdTweet,            "Bird Tweet" },
            { neogfx::instrument::TelephoneRing,        "Telephone Ring" },
            { neogfx::instrument::Helicopter,           "Helicopter" },
            { neogfx::instrument::Applause,             "Applause" },
            { neogfx::instrument::Gunshot,              "Gunshot" }
        };

        return sMap.at(instrument);
    }  
     
    inline std::string to_string(neogfx::percussion_instrument instrument)
    {
        static std::unordered_map<percussion_instrument, std::string> const sMap =
        {
            { neogfx::percussion_instrument::AcousticBassDrum   , "Acoustic Bass Drum" },
            { neogfx::percussion_instrument::BassDrum1          , "Bass Drum 1" },
            { neogfx::percussion_instrument::SideStick          , "Side Stick" },
            { neogfx::percussion_instrument::AcousticSnare      , "Acoustic Snare" },
            { neogfx::percussion_instrument::HandClap           , "Hand Clap" },
            { neogfx::percussion_instrument::ElectricSnare      , "Electric Snare" },
            { neogfx::percussion_instrument::LowFloorTom        , "Low Floor Tom" },
            { neogfx::percussion_instrument::ClosedHiHat        , "Closed Hi Hat" },
            { neogfx::percussion_instrument::HighFloorTom       , "High Floor Tom" },
            { neogfx::percussion_instrument::PedalHiHat         , "Pedal Hi - Hat" },
            { neogfx::percussion_instrument::LowTom             , "Low Tom" },
            { neogfx::percussion_instrument::OpenHiHat          , "Open Hi - Hat" },
            { neogfx::percussion_instrument::LowMidTom          , "Low - Mid Tom" },
            { neogfx::percussion_instrument::HiMidTom           , "Hi - Mid Tom" },
            { neogfx::percussion_instrument::CrashCymbal1       , "Crash Cymbal 1" },
            { neogfx::percussion_instrument::HighTom            , "High Tom" },
            { neogfx::percussion_instrument::RideCymbal1        , "Ride Cymbal 1" },
            { neogfx::percussion_instrument::ChineseCymbal      , "Chinese Cymbal" },
            { neogfx::percussion_instrument::RideBell           , "Ride Bell" },
            { neogfx::percussion_instrument::Tambourine         , "Tambourine" },
            { neogfx::percussion_instrument::SplashCymbal       , "Splash Cymbal" },
            { neogfx::percussion_instrument::Cowbell            , "Cowbell" },
            { neogfx::percussion_instrument::CrashCymbal2       , "Crash Cymbal 2" },
            { neogfx::percussion_instrument::Vibraslap          , "Vibraslap" },
            { neogfx::percussion_instrument::RideCymbal2        , "Ride Cymbal 2" },
            { neogfx::percussion_instrument::HiBongo            , "Hi Bongo" },
            { neogfx::percussion_instrument::LowBongo           , "Low Bongo" },
            { neogfx::percussion_instrument::MuteHiConga        , "Mute Hi Conga" },
            { neogfx::percussion_instrument::OpenHiConga        , "Open Hi Conga" },
            { neogfx::percussion_instrument::LowConga           , "Low Conga" },
            { neogfx::percussion_instrument::HighTimbale        , "High Timbale" },
            { neogfx::percussion_instrument::LowTimbale         , "Low Timbale" },
            { neogfx::percussion_instrument::HighAgogo          , "High Agogo" },
            { neogfx::percussion_instrument::LowAgogo           , "Low Agogo" },
            { neogfx::percussion_instrument::Cabasa             , "Cabasa" },
            { neogfx::percussion_instrument::Maracas            , "Maracas" },
            { neogfx::percussion_instrument::ShortWhistle       , "Short Whistle" },
            { neogfx::percussion_instrument::LongWhistle        , "Long Whistle" },
            { neogfx::percussion_instrument::ShortGuiro         , "Short Guiro" },
            { neogfx::percussion_instrument::LongGuiro          , "Long Guiro" },
            { neogfx::percussion_instrument::Claves             , "Claves" },
            { neogfx::percussion_instrument::HiWoodBlock        , "Hi Wood Block" },
            { neogfx::percussion_instrument::LowWoodBlock       , "Low Wood Block" },
            { neogfx::percussion_instrument::MuteCuica          , "Mute Cuica" },
            { neogfx::percussion_instrument::OpenCuica          , "Open Cuica" },
            { neogfx::percussion_instrument::MuteTriangle       , "Mute Triangle" },
            { neogfx::percussion_instrument::OpenTriangle       , "Open Triangle" }
        };

        return sMap.at(instrument);
    }

    enum class audio_channel : std::uint64_t
    {
        None                               = 0x0000000000000000ULL,
        Mono                               = 0x0000000000000001ULL,
        FrontLeft                          = 0x0000000000000002ULL,
        FrontRight                         = 0x0000000000000004ULL,
        FrontCenter                        = 0x0000000000000008ULL,
        Lfe                                = 0x0000000000000010ULL,
        BackLeft                           = 0x0000000000000020ULL,
        BackRight                          = 0x0000000000000040ULL,
        FrontLeftCenter                    = 0x0000000000000080ULL,
        FrontRightCenter                   = 0x0000000000000100ULL,
        BackCenter                         = 0x0000000000000200ULL,
        SideLeft                           = 0x0000000000000400ULL,
        SideRight                          = 0x0000000000000800ULL,
        TopCenter                          = 0x0000000000001000ULL,
        TopFrontLeft                       = 0x0000000000002000ULL,
        TopFrontCenter                     = 0x0000000000004000ULL,
        TopFrontRight                      = 0x0000000000008000ULL,
        TopBackLeft                        = 0x0000000000010000ULL,
        TopBackCenter                      = 0x0000000000020000ULL,
        TopBackRight                       = 0x0000000000040000ULL,
        Aux0                               = 0x0000000000080000ULL,
        Aux1                               = 0x0000000000100000ULL,
        Aux2                               = 0x0000000000200000ULL,
        Aux3                               = 0x0000000000400000ULL,
        Aux4                               = 0x0000000000800000ULL,
        Aux5                               = 0x0000000001000000ULL,
        Aux6                               = 0x0000000002000000ULL,
        Aux7                               = 0x0000000004000000ULL,
        Aux8                               = 0x0000000008000000ULL,
        Aux9                               = 0x0000000010000000ULL,
        Aux10                              = 0x0000000020000000ULL,
        Aux11                              = 0x0000000040000000ULL,
        Aux12                              = 0x0000000080000000ULL,
        Aux13                              = 0x0000000100000000ULL,
        Aux14                              = 0x0000000200000000ULL,
        Aux15                              = 0x0000000400000000ULL,
        Aux16                              = 0x0000000800000000ULL,
        Aux17                              = 0x0000001000000000ULL,
        Aux18                              = 0x0000002000000000ULL,
        Aux19                              = 0x0000004000000000ULL,
        Aux20                              = 0x0000008000000000ULL,
        Aux21                              = 0x0000010000000000ULL,
        Aux22                              = 0x0000020000000000ULL,
        Aux23                              = 0x0000040000000000ULL,
        Aux24                              = 0x0000080000000000ULL,
        Aux25                              = 0x0000100000000000ULL,
        Aux26                              = 0x0000200000000000ULL,
        Aux27                              = 0x0000400000000000ULL,
        Aux28                              = 0x0000800000000000ULL,
        Aux29                              = 0x0001000000000000ULL,
        Aux30                              = 0x0002000000000000ULL,
        Aux31                              = 0x0004000000000000ULL,
                                           
        Left                               = FrontLeft,
        Right                              = FrontRight
    };

    inline audio_channel operator&(audio_channel lhs, audio_channel rhs)
    {
        return static_cast<audio_channel>(static_cast<std::uint64_t>(lhs) & static_cast<std::uint64_t>(rhs));
    }

    inline audio_channel operator|(audio_channel lhs, audio_channel rhs)
    {
        return static_cast<audio_channel>(static_cast<std::uint64_t>(lhs) | static_cast<std::uint64_t>(rhs));
    }

    inline audio_channel operator^(audio_channel lhs, audio_channel rhs)
    {
        return static_cast<audio_channel>(static_cast<std::uint64_t>(lhs) ^ static_cast<std::uint64_t>(rhs));
    }

    inline audio_channel operator~(audio_channel lhs)
    {
        return static_cast<audio_channel>(~static_cast<std::uint64_t>(lhs));
    }

    inline std::uint64_t channel_count(audio_channel channels)
    {
        return static_cast<std::uint64_t>(std::popcount(static_cast<std::uint64_t>(channels)));
    }

    enum class audio_stream_format : std::uint32_t
    {
        Unknown = 0,

        Pcm     = 1
    };

    enum class audio_sample_format : std::uint32_t
    {
        Unknown = 0,

        U8      = 1,
        S16     = 2,
        S24     = 3,
        S32     = 4,
        F32     = 5
    };

    typedef std::uint64_t audio_sample_rate;

	enum class audio_standard_sample_rate : std::uint64_t
	{
        Rate48000   = 48000,
        Rate44100   = 44100,

        Rate32000   = 32000,
        Rate24000   = 24000,
        Rate22050   = 22050,

        Rate88200   = 88200,
        Rate96000   = 96000,
        Rate176400  = 176400,
        Rate192000  = 192000,

        Rate16000   = 16000,
        Rate11025   = 11250,
        Rate8000    = 8000,

        Rate352800  = 352800,
        Rate384000  = 384000,
	};

    struct audio_data_format
    {
        typedef audio_data_format abstract_type;

        audio_sample_format sampleFormat;
        std::uint32_t channels;
        audio_sample_rate sampleRate;
    };

    typedef std::uint64_t audio_sample_index;
    typedef std::uint64_t audio_sample_count;

    typedef std::uint64_t audio_frame_index;
    typedef std::uint64_t audio_frame_count;

    template <typename SampleType, std::size_t Channels>
    using audio_frame = std::array<SampleType, Channels>;

    struct adsr_envelope
    {
        float attack;
        float decay;
        float sustain;
        float release;
    };
}