﻿using System;
using System.IO;
using System.Security.Cryptography;
using System.Text;
using HsnSoft.Base.DependencyInjection;
using Microsoft.Extensions.Options;

namespace HsnSoft.Base.Security.Encryption;

/// <summary>
/// Can be used to simply encrypt/decrypt texts.
/// </summary>
public class StringEncryptionService : IStringEncryptionService, ITransientDependency
{
    public StringEncryptionService(IOptions<BaseStringEncryptionOptions> options)
    {
        Options = options.Value;
    }

    protected BaseStringEncryptionOptions Options { get; }

    public virtual string Encrypt(string plainText, string passPhrase = null, byte[] salt = null)
    {
        if (plainText == null)
        {
            return null;
        }

        if (passPhrase == null)
        {
            passPhrase = Options.DefaultPassPhrase;
        }

        if (salt == null)
        {
            salt = Options.DefaultSalt;
        }

        var plainTextBytes = Encoding.UTF8.GetBytes(plainText);
        using (var password = new Rfc2898DeriveBytes(passPhrase, salt))
        {
            var keyBytes = password.GetBytes(Options.Keysize / 8);
            using (var symmetricKey = Aes.Create())
            {
                symmetricKey.Mode = CipherMode.CBC;
                using (var encryptor = symmetricKey.CreateEncryptor(keyBytes, Options.InitVectorBytes))
                {
                    using (var memoryStream = new MemoryStream())
                    {
                        using (var cryptoStream = new CryptoStream(memoryStream, encryptor, CryptoStreamMode.Write))
                        {
                            cryptoStream.Write(plainTextBytes, 0, plainTextBytes.Length);
                            cryptoStream.FlushFinalBlock();
                            var cipherTextBytes = memoryStream.ToArray();
                            return Convert.ToBase64String(cipherTextBytes);
                        }
                    }
                }
            }
        }
    }

    public virtual string Decrypt(string cipherText, string passPhrase = null, byte[] salt = null)
    {
        if (string.IsNullOrEmpty(cipherText))
        {
            return null;
        }

        if (passPhrase == null)
        {
            passPhrase = Options.DefaultPassPhrase;
        }

        if (salt == null)
        {
            salt = Options.DefaultSalt;
        }

        var cipherTextBytes = Convert.FromBase64String(cipherText);
        using (var password = new Rfc2898DeriveBytes(passPhrase, salt))
        {
            var keyBytes = password.GetBytes(Options.Keysize / 8);
            using (var symmetricKey = Aes.Create())
            {
                symmetricKey.Mode = CipherMode.CBC;
                using (var decryptor = symmetricKey.CreateDecryptor(keyBytes, Options.InitVectorBytes))
                {
                    using (var memoryStream = new MemoryStream(cipherTextBytes))
                    {
                        using (var cryptoStream = new CryptoStream(memoryStream, decryptor, CryptoStreamMode.Read))
                        {
                            var plainTextBytes = new byte[cipherTextBytes.Length];
                            var totalReadCount = 0;
                            while (totalReadCount < cipherTextBytes.Length)
                            {
                                var buffer = new byte[cipherTextBytes.Length];
                                var readCount = cryptoStream.Read(buffer, 0, buffer.Length);
                                if (readCount == 0)
                                {
                                    break;
                                }

                                for (var i = 0; i < readCount; i++)
                                {
                                    plainTextBytes[i + totalReadCount] = buffer[i];
                                }

                                totalReadCount += readCount;
                            }

                            return Encoding.UTF8.GetString(plainTextBytes, 0, totalReadCount);
                        }
                    }
                }
            }
        }
    }
}