<?php
namespace app;

use app\common\exception\UploadException;
use think\addons\AddonsException;
use think\db\exception\DataNotFoundException;
use think\db\exception\ModelNotFoundException;
use think\exception\Handle;
use think\exception\HttpException;
use think\exception\HttpResponseException;
use think\exception\ValidateException;
use think\Response;
use Throwable;

/**
 * 应用异常处理类
 */
class ExceptionHandle extends Handle
{
    /**
     * 不需要记录信息（日志）的异常类列表
     * @var array
     */
    protected $ignoreReport = [
        HttpException::class,
        HttpResponseException::class,
        ModelNotFoundException::class,
        DataNotFoundException::class,
        ValidateException::class,
    ];

    /**
     * 记录异常信息（包括日志或者其它方式记录）
     *
     * @access public
     * @param  Throwable $exception
     * @return void
     */
    public function report(Throwable $exception): void
    {
        // 使用内置的方式记录异常日志
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @access public
     * @param \think\Request   $request
     * @param Throwable $e
     * @return Response
     */
    public function render($request, Throwable $e): Response
    {
        // 添加自定义异常处理机制
        if ($e instanceof ValidateException && $request->isAjax()) {
            return json(['code' => -1000, 'msg' => $e->getMessage(), 'data' => []]);
        }

        // 插件异常处理
        if ($e instanceof AddonsException) {
            if ($request->isAjax()) {
                return json(['code' => -1000, 'msg' => $e->getMessage(), 'data' => []], $e->getStatusCode());
            } else {
                return response($e->getMessage(), $e->getStatusCode());
            }
        }

        // 文件上传异常处理
        if ($e instanceof UploadException) {
            if ($request->isAjax()) {
                return json(['code' => -1000, 'msg' => $e->getMessage(), 'data' => []], 200);
            } else {
                return response($e->getMessage(), $e->getStatusCode());
            }
        }
        if ($e instanceof \think\Exception && $request->isAjax()) {
            return json(['code' => -1000, 'msg' => $e->getMessage(), 'data' => []], 200);
        }

        // 其他错误交给系统处理
        return parent::render($request, $e);
    }
}
