/*
 * Copyright (C) 2019-2024 HERE Europe B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * License-Filename: LICENSE
 */

package com.here.offscreenmap;

import android.content.Context;
import android.os.Bundle;
import android.view.ViewTreeObserver;
import android.widget.Button;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;

import com.here.sdk.core.engine.SDKNativeEngine;
import com.here.sdk.core.engine.SDKOptions;
import com.here.sdk.core.errors.InstantiationErrorException;
import com.here.sdk.mapview.MapScheme;

import java.util.ArrayList;
import java.util.List;

/**
 * Shows four images generated by a {@link com.here.sdk.mapview.MapSurface} to an offscreen
 * drawing surface.
 * <p>
 * The 'Redraw' button regenerates all four images with random zoom.
 */
public class MainActivity extends AppCompatActivity implements ViewTreeObserver.OnGlobalLayoutListener {
    private static final String TAG = MainActivity.class.getSimpleName();

    private List<MapPanel> mapPanels = new ArrayList<>(4);

    Button redrawButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // Usually, you need to initialize the HERE SDK only once during the lifetime of an application.
        initializeHERESDK();

        setContentView(R.layout.activity_main);

        redrawButton = findViewById(R.id.redrawButton);

        // Use OnGlobalLayoutListener to initialize map renderers only after
        // layout has finished and size of image views is known.
        redrawButton.getViewTreeObserver().addOnGlobalLayoutListener(this);

        redrawButton.setOnClickListener(v -> {
            for (MapPanel panel : mapPanels) {
                panel.redraw();
            }
        });
    }

    @Override
    public void onGlobalLayout() {
        redrawButton.getViewTreeObserver().removeOnGlobalLayoutListener(this);
        initMapRenderers();
    }

    private void initMapRenderers(){
        mapPanels.add(new MapPanel(findViewById(R.id.imageView1), MapScheme.NORMAL_DAY));
        mapPanels.add(new MapPanel(findViewById(R.id.imageView2), MapScheme.LITE_DAY));
        mapPanels.add(new MapPanel(findViewById(R.id.imageView3), MapScheme.SATELLITE));
        mapPanels.add(new MapPanel(findViewById(R.id.imageView4), MapScheme.LITE_HYBRID_NIGHT));
    }

    private void initializeHERESDK() {
        if (SDKNativeEngine.getSharedInstance() != null) {
            return;
        }
        // Set your credentials for the HERE SDK.
        String accessKeyID = "YOUR_ACCESS_KEY_ID";
        String accessKeySecret = "YOUR_ACCESS_KEY_SECRET";
        SDKOptions options = new SDKOptions(accessKeyID, accessKeySecret);
        try {
            Context context = this;
            SDKNativeEngine.makeSharedInstance(context, options);
        } catch (InstantiationErrorException e) {
            throw new RuntimeException("Initialization of HERE SDK failed: " + e.error.name());
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        for (MapPanel panel : mapPanels) {
            panel.onPause();
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        for (MapPanel panel : mapPanels) {
            panel.onResume();
        }
    }

    @Override
    protected void onDestroy() {
        for (MapPanel panel : mapPanels) {
            panel.onDestroy();
        }
        disposeHERESDK();
        super.onDestroy();
    }

    @Override
    protected void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
    }
    
    private void disposeHERESDK() {
        // Free HERE SDK resources before the application shuts down.
        // Usually, this should be called only on application termination.
        // Afterwards, the HERE SDK is no longer usable unless it is initialized again.
        SDKNativeEngine sdkNativeEngine = SDKNativeEngine.getSharedInstance();
        if (sdkNativeEngine != null) {
            sdkNativeEngine.dispose();
            // For safety reasons, we explicitly set the shared instance to null to avoid situations,
            // where a disposed instance is accidentally reused.
            SDKNativeEngine.setSharedInstance(null);
        }
    }
}
