/*
 * Copyright (C) 2024 HERE Europe B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * License-Filename: LICENSE
 */

import heresdk
import SwiftUI

/// Encapsulates multiple offscreen map views for generating
/// map images and exposing them to the UI.
///
/// Designed to be used as environment object to share state
/// among SwiftUI views.
class OffscreenMapsModel: ObservableObject {

    enum GeneratorState {
        case initializing
        case ready
        case busy
    }

    /// Map image generators.
    private var offscreenMapViewWrappers: [String : OffscreenMapViewWrapper] = [:]

    /// Holds generated images for each named generator. Initially empty.
    @Published var mapImages: [String : UIImage] = [:]

    /// Holds state of each map image generator.
    @Published var mapStatus: [String : GeneratorState] = [:]

    /// Initializes a new named map image generator.
    ///
    /// - Parameter name: The name of the image generator.
    /// - Parameter size: The dimensions of the image to be generated.
    /// - Parameter mapScheme: The map scheme to use for the map.
    func initMapImageGenerator(named name: String, size: CGSize, mapScheme: MapScheme) {
        mapStatus[name] = .initializing
        offscreenMapViewWrappers[name] = OffscreenMapViewWrapper(size: Size2D(width: size.width, height: size.height), mapScheme: mapScheme) {
            self.mapStatus[name] = .ready
        }
    }

    /// Refreshes all generated images by randomly changing zoom of the map.
    func refreshAll() {
        for key in offscreenMapViewWrappers.keys {
            generateMapImage(generatorName: key)
        }
    }

    private func generateMapImage(generatorName name: String) {
        guard let mapWrapper = offscreenMapViewWrappers[name] else {
            return
        }

        let isGenerating = mapWrapper.generateImage(mapTransformer: { map in
            map.camera.zoomTo(zoomLevel: Double.random(in: 2..<20))
        }) { image in
            self.mapImages[name] = image
            self.mapStatus[name] = .ready
        }

        if isGenerating {
            mapStatus[name] = .busy
        }
    }
}

/// Shows an image generated by offscreen `MapView`.
/// Additionally shows the current state of the image generator.
struct OffscreenMapView: View {
    @EnvironmentObject var offscreenMaps: OffscreenMapsModel

    let name: String
    let mapScheme: MapScheme

    init(named name: String, mapScheme: MapScheme = .normalDay) {
        self.name = name
        self.mapScheme = mapScheme
    }

    var body: some View {
        GeometryReader { geometry in
            if let image = offscreenMaps.mapImages[name]?.cgImage {
                Image(image, scale: UIScreen.main.scale, label: Text(verbatim: "Map Image"))
                    .aspectRatio(contentMode: .fill)
            } else {
                // Initially, just show some icon. The important part is initializing
                // the offscreen map renderer in `onAppear()`, as we have to know the
                // size of the images to be generated.
                Image(systemName: "globe")
                    .aspectRatio(contentMode: .fill)
                    .imageScale(.large)
                    .frame(maxWidth: .infinity, maxHeight: .infinity)
                    .onAppear() {
                        // Init offscreen map view once this view appears and we know its size.
                        offscreenMaps.initMapImageGenerator(named: name, size: geometry.size, mapScheme: mapScheme)
                    }
            }
            if let state = offscreenMaps.mapStatus[name] {
                switch state {
                case .initializing:
                    Text(verbatim: "Initializing")
                        .foregroundColor(Color.red)
                        .frame(maxWidth: .infinity)
                case .ready:
                    Text(verbatim: "Ready")
                        .fontWeight(.bold)
                        .foregroundColor(Color.green)
                        .frame(maxWidth: .infinity)
                case .busy:
                    Text(verbatim: "Refreshing")
                        .fontWeight(.bold)
                        .foregroundColor(Color.orange)
                        .frame(maxWidth: .infinity)
                }
            }
        }
    }
}

struct OffscreenMapsExampleView: View {
    @EnvironmentObject var offscreenMaps: OffscreenMapsModel

    var body: some View {
        VStack {
            HStack {
                OffscreenMapView(named: "One")
                OffscreenMapView(named: "Two", mapScheme: .normalNight)
            }

            Button("Refresh") {
                offscreenMaps.refreshAll()
            }

            HStack {
                OffscreenMapView(named: "Three", mapScheme: .hybridDay)
                OffscreenMapView(named: "Four", mapScheme: .liteDay)
            }
        }
    }
}

struct ContentView: View {
    var body: some View {
        OffscreenMapsExampleView().environmentObject(OffscreenMapsModel())
    }
}

struct ContentView_Previews: PreviewProvider {
    static var previews: some View {
        ContentView()
    }
}
