/*
 * Copyright (C) 2019-2024 HERE Europe B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * License-Filename: LICENSE
 */

package com.here;

import static junit.framework.TestCase.assertEquals;
import static org.mockito.ArgumentMatchers.anyDouble;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.withSettings;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.here.sdk.core.Angle;
import com.here.sdk.core.GeoCoordinates;
import com.here.sdk.core.GeoCoordinatesUpdate;
import com.here.sdk.core.GeoOrientation;
import com.here.sdk.mapview.MapCamera;
import com.here.sdk.mapview.MapCameraAnimation;
import com.here.sdk.mapview.MapCameraAnimationFactory;
import com.here.sdk.mapview.MapView;
import com.here.time.Duration;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;

/**
 * The HERE SDK's Mocking library provides constructor helpers that assist in initializing
 * struct-like classes. These helpers automatically initialize public non-static fields whose
 * names match the constructor parameter names. When parameter and field names do not match,
 * a custom implementation can be provided.
 * 
 * To provide a custom implementation, you must either:
 * - Implement the `ConstructorHelper` interface, or
 * - Extend the `ConstructorHelperImpl` class and set the `ConstructorMockHelperInstance` field to this implementation.
 * 
 * The `ConstructorHelper` interface contains methods named `_constructor(...)`, with one method
 * for each constructor in the class where it is nested. Each `_constructor(...)` method has the same
 * parameters as the related constructor and returns an instance of the class.
 * 
 * The `ConstructorHelperImpl` class implements the `ConstructorHelper` interface and provides default
 * implementations as described above. It is useful to extend this class and override only specific
 * `_constructor(...)` methods as needed. Each constructor, except the parameterless one, calls the
 * related version of `ConstructorMockHelperInstance._constructor(...)` and copies all public fields
 * from the returned instance to the current instance.
 * 
 * Rule of thumb: When the constructor contains parameter names that are not exposed as members, then use the
 * ConstructorMockHelperInstance approach. Otherwise, not.
 */
class GeoCoordinatesUpdateMockHelper extends GeoCoordinatesUpdate.ConstructorHelperImpl {

    /**
     * The `GeoCoordinatesUpdate` class does not have a `coordinates` field. Instead, it has `latitude`,
     * `longitude`, and `altitude` fields. As a result, the default implementation of 
     * `GeoCoordinatesUpdate.ConstructorHelperImpl` cannot initialize these fields. Therefore, 
     * it is necessary to initialize them using custom logic.
     */
    public GeoCoordinatesUpdate _constructor(@NonNull final GeoCoordinates coordinates) {
        // Each class of the HERE SDK's Mock Library contains a parameterless constructor even if the original class doesn't.
        // This constructor can be used to create instances of the class and initialize
        // fields with default values.
        GeoCoordinatesUpdate _result = new GeoCoordinatesUpdate();
        _result.latitude = coordinates.latitude;
        _result.longitude = coordinates.longitude;
        _result.altitude = coordinates.altitude;
        // All public fields are copied in constructor
        // "GeoCoordinatesUpdate(@NonNull final GeoCoordinates coordinates)"
        return _result;
    }
}

@RunWith(MockitoJUnitRunner.class)
public class TestBasicTypes {

    // These tests show examples of how the HERE SDK can be accessed in a unit test scenario.

    @Before
    public void SetUp() {
        // Use custom logic for constructors of "GeoCoordinatesUpdate" class.
        GeoCoordinatesUpdate.ConstructorMockHelperInstance = new GeoCoordinatesUpdateMockHelper();
    }

    @After
    public void TearDown() {
        // Reset back to default implementation
        GeoCoordinatesUpdate.ConstructorMockHelperInstance = new GeoCoordinatesUpdate.ConstructorHelperImpl();
    }

    @Test
    public void testAutoGeneratedConstructor() {
        // "GeoCoordinates" has constructor with parameters "latitude" and "longitude" and
        // it has fields with the same names, so in the example below mentioned fields are
        // initalized by the constructor's parameters and altitude has default value.
        GeoCoordinates geoCoordinates = new GeoCoordinates(10.0, 20.0);
        assertEquals(geoCoordinates.latitude, 10.0);
        assertEquals(geoCoordinates.longitude, 20.0);
    }

    @Test
    public void testSelfImplementedConstructor() {
        GeoCoordinates geoCoordinates = new GeoCoordinates(5.0, 10.0, 20.0);
        // This constructor uses "GeoCoordinatesUpdateMockHelper" which is set in "SetUp".
        GeoCoordinatesUpdate geoCoordinatesUpdate = new GeoCoordinatesUpdate(geoCoordinates);

        assertEquals(geoCoordinatesUpdate.latitude, 5.0);
        assertEquals(geoCoordinatesUpdate.longitude, 10.0);
        assertEquals(geoCoordinatesUpdate.altitude, 20.0);
    }

    @Test
    public void testNonStaticMethod() {
        Angle angleMock = mock(Angle.class);

        when(angleMock.getDegrees()).thenReturn(10.0);

        assertEquals(10.0, angleMock.getDegrees());
        verify(angleMock, times(1)).getDegrees();
        verifyNoMoreInteractions(angleMock);
    }

    @Test
    public void testStaticMethod() {
        Angle angleMock = mock(Angle.class);
        when(angleMock.getDegrees()).thenReturn(10.0);

        // Each HERE SDK class with static methods contains helper code to make mocking easier.
        // Add heresdk-xxx.jar to access these additional mock helper instances.
        Angle.StaticMockHelperInstance = mock(Angle.StaticMockHelper.class);
        when(Angle.StaticMockHelperInstance.fromRadians(anyDouble())).thenReturn(angleMock);

        // Test static creation of Angle class. Static HERE SDK classes require a StaticMockHelperInstance.
        assertEquals(10.0, Angle.fromRadians(0.174533).getDegrees());

        verify(Angle.StaticMockHelperInstance, times(1)).fromRadians(anyDouble());
        verify(angleMock, times(1)).getDegrees();
        verifyNoMoreInteractions(Angle.StaticMockHelperInstance);
        verifyNoMoreInteractions(angleMock);
    }

    @Test
    public void testMapView() {
        GeoCoordinates targetCoordinates = mock(GeoCoordinates.class);
        GeoOrientation orientationAtTarget = mock(GeoOrientation.class);
        double distanceInMeters = 5000.0;
        double zoomLevel = 1000.0;

        double bowFactor = 1;
        Duration duration = mock(Duration.class);
        GeoCoordinatesUpdate geoCoordinatesUpdate = mock(GeoCoordinatesUpdate.class);

        MapView mapView = mock(MapView.class);

        // When a mock is declared as lenient, then none of its stubbings will be checked for 'unnecessary stubbing'.
        MapCamera mapCamera = mock(MapCamera.class, withSettings().lenient());
        MapCamera.State state = new MapCamera.State(targetCoordinates, orientationAtTarget, distanceInMeters, zoomLevel);

        MapCameraAnimation mapCameraAnimation = mock(MapCameraAnimation.class);
        MapCameraAnimationFactory.StaticMockHelperInstance = mock(MapCameraAnimationFactory.StaticMockHelper.class);

        when(mapView.getWidth()).thenReturn(100);
        when(mapView.getHeight()).thenReturn(100);
        when(mapCamera.getState()).thenReturn(state);
        when(MapCameraAnimationFactory.StaticMockHelperInstance.flyTo(
                geoCoordinatesUpdate, bowFactor, duration)).thenReturn(mapCameraAnimation);

        // This verifies that the HERE SDK's MapView can be mocked as expected.
        assertEquals(100, mapView.getWidth());
        assertEquals(100, mapView.getHeight());

        // This verifies that the HERE SDK's MapCamera can be mocked as expected.
        assertEquals(state, mapCamera.getState());
        assertEquals(mapCameraAnimation, MapCameraAnimationFactory.StaticMockHelperInstance.flyTo(
                geoCoordinatesUpdate, bowFactor, duration));
        verify(mapView, times(1)).getWidth();
        verify(mapView, times(1)).getHeight();
        verify(mapCamera, times(1)).getState();
    }
}
