package com.graphaware.nlp.dsl;

import com.graphaware.nlp.NLPIntegrationTest;
import com.graphaware.nlp.dsl.request.PipelineSpecification;
import com.graphaware.nlp.stub.StubTextProcessor;
import org.junit.Test;
import org.neo4j.graphdb.Transaction;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

import static org.junit.Assert.*;

public class TextProcessorsProcedureTest extends NLPIntegrationTest {
    private static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-mm-dd hh:mm:ss");

    @Test
    public void testGetPipelineInformationsProcedure() {
        executeInTransaction("CALL ga.nlp.processor.getPipelines", (result -> {
            assertFalse(result.hasNext());
        }));
    }

    @Test
    public void testAddPipeline() {
        clearDb();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name:'custom-1', textProcessor:'" + StubTextProcessor.class.getName() +"', processingSteps:{tokenize:true,ner:true,dependency:true},excludedNER:['MONEY','MISC']})", emptyConsumer());
        assertTrue(checkConfigurationContainsKey(STORE_KEY + "PIPELINE_custom-1"));
        PipelineSpecification pipelineSpecification = getNLPManager().getConfiguration()
                .loadPipeline("custom-1");
        assertNotNull(pipelineSpecification);
        assertTrue(pipelineSpecification.hasProcessingStep("tokenize"));
        assertTrue(pipelineSpecification.hasProcessingStep("dependency"));
        assertTrue(pipelineSpecification.getExcludedNER().contains("MONEY"));
        assertTrue(pipelineSpecification.getExcludedNER().contains("MISC"));
    }

    @Test(expected = RuntimeException.class)
    public void testAddPipelineOnNonExistentProcessorShouldFail() {
        clearDb();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name:'non-exist', textProcessor:'non-processor'})", (result) -> {
            //
        });
    }

    @Test
    public void removePipelineTest() {
        clearDb();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name:'custom-1', textProcessor:'" + StubTextProcessor.class.getName() +"'})", emptyConsumer());
        assertTrue(checkConfigurationContainsKey(STORE_KEY + "PIPELINE_custom-1"));
//        assertTrue(getNLPManager().getTextProcessorsManager().getTextProcessor(StubTextProcessor.class.getName()).getPipelines().contains("custom-1"));
        executeInTransaction("CALL ga.nlp.processor.removePipeline('custom-1', '"+StubTextProcessor.class.getName()+"')", emptyConsumer());
//        assertFalse(getNLPManager().getTextProcessorsManager().getTextProcessor(StubTextProcessor.class.getName()).getPipelines().contains("custom-1"));
        assertFalse(checkConfigurationContainsKey(STORE_KEY + "PIPELINE_custom-1"));
    }

    @Test
    public void refreshPipelineTest() throws Exception {
        clearDb();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name: $p0, textProcessor: $p1, processingSteps: {ner:true}})", buildSeqParameters("hello", StubTextProcessor.class.getName()), emptyConsumer());
        assertTrue(checkConfigurationContainsKey(STORE_KEY + "PIPELINE_hello"));
        Date created = DATE_FORMAT.parse(getNLPManager().getConfiguration().loadPipeline("hello").getCreatedAt());
        Thread.sleep(2000);
        executeInTransaction("CALL ga.nlp.refreshPipeline('hello')", emptyConsumer());
        assertTrue(checkConfigurationContainsKey(STORE_KEY + "PIPELINE_hello"));

        assertTrue(DATE_FORMAT.parse(getNLPManager().getConfiguration().loadPipeline("hello").getCreatedAt()).after(created));
    }

    @Test
    public void testGetPipelineInfosWorksWithAndWithoutAPipelineNameParameter() {
        clearDb();
        removeCustomPipelines();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name:'custom-1', textProcessor:'" + StubTextProcessor.class.getName() +"'})", emptyConsumer());
        executeInTransaction("CALL ga.nlp.processor.getPipelines", (result -> {
            assertTrue(result.hasNext());
            assertEquals(1, result.stream().count());
        }));

        executeInTransaction("CALL ga.nlp.processor.getPipelines('custom-1')", (result -> {
            assertTrue(result.hasNext());
            assertEquals(1, result.stream().count());
        }));
    }

    @Test
    public void testAddPipelineWithCustomWhiteList() {
        clearDb();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name:'custom-1', textProcessor:'" + StubTextProcessor.class.getName() +"', whitelist:'hello,i,be,work,IBM,ibm', processingSteps:{tokenize:true,ner:true,dependency:true},excludedNER:['MONEY','MISC']})", emptyConsumer());
        assertTrue(checkConfigurationContainsKey(STORE_KEY + "PIPELINE_custom-1"));
        PipelineSpecification pipelineSpecification = getNLPManager().getConfiguration()
                .loadPipeline("custom-1");
        assertEquals("hello,i,be,work,IBM,ibm", pipelineSpecification.getWhitelist());
    }

    @Test
    public void testAddingPipelineWithCustomSentimentModel() {
        clearDb();
        removeCustomPipelines();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name:'custom-1', textProcessor:'" + StubTextProcessor.class.getName() +"', processingSteps:{customSentiment:'my-model'}})", emptyConsumer());
        PipelineSpecification pipelineSpecification = getNLPManager().getConfiguration().loadPipeline("custom-1");
        assertEquals("my-model", pipelineSpecification.getProcessingStepAsString("customSentiment"));
    }

    @Test
    public void testCustomPipelineWithSentimentModelShouldDisplayModelNameInPipelineInfo() {
        clearDb();
        removeCustomPipelines();
        executeInTransaction("CALL ga.nlp.processor.addPipeline({name:'custom-1', textProcessor:'" + StubTextProcessor.class.getName() +"', processingSteps:{customSentiment:'my-model'}})", emptyConsumer());
        PipelineSpecification pipelineSpecification = getNLPManager().getConfiguration().loadPipeline("custom-1");
        assertEquals("my-model", pipelineSpecification.getProcessingStepAsString("customSentiment"));
        executeInTransaction("CALL ga.nlp.processor.getPipelines('custom-1')", (result -> {
            assertTrue(result.hasNext());
            while (result.hasNext()) {
                Map<String, Object> record = result.next();
                System.out.println(record);
                Map<String, Object> specs = (Map<String, Object>) record.get("processingSteps");
                assertEquals("my-model", specs.get("customSentiment"));
            }
        }));
    }

    @Test
    public void testCustomPipelineWithTrainedModelStoreModelInConfig() {
        clearDb();
        removeCustomPipelines();
        String modelsPath = getClass().getClassLoader().getResource("").getPath();
        executeInTransaction("CALL ga.nlp.config.model.workdir({p0})", buildSeqParameters(modelsPath), emptyConsumer());
        String q = "CALL ga.nlp.processor.train({textProcessor: '" + StubTextProcessor.class.getName() + "', modelIdentifier: \"testmodel\", alg: \"ner\", inputFile: 'model.tsv', trainingParameters: {iter: 10}})";
        executeInTransaction(q, emptyConsumer());
        assertNotNull(getNLPManager().getConfiguration().getModelPaths("testmodel"));
    }

    private void removeCustomPipelines() {
        try (Transaction tx = getDatabase().beginTx()) {
            getNLPManager().getConfiguration().loadCustomPipelines().forEach(pipelineSpecification -> {
                getNLPManager().getConfiguration().removePipeline(pipelineSpecification.getName(), pipelineSpecification.getTextProcessor());
            });
            tx.success();
        }
    }
}
