/*
 * Copyright (c) 2013-2018 GraphAware
 *
 * This file is part of the GraphAware Framework.
 *
 * GraphAware Framework is free software: you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. You should have received a copy of
 * the GNU General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */
package com.graphaware.nlp.ml.textrank;

import com.graphaware.nlp.annotation.NLPModuleExtension;
import com.graphaware.nlp.dsl.request.TextRankPostprocessRequest;
import com.graphaware.nlp.dsl.request.TextRankRequest;
import com.graphaware.nlp.dsl.result.SingleResult;
import com.graphaware.nlp.extension.AbstractExtension;
import com.graphaware.nlp.extension.NLPExtension;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.logging.Log;
import com.graphaware.common.log.LoggerFactory;

import java.util.stream.Collectors;

@NLPModuleExtension(name = "TextRankProcessor")
public class TextRankProcessor extends AbstractExtension implements NLPExtension {

    private static final Log LOG = LoggerFactory.getLogger(TextRankProcessor.class);

    public SingleResult computeAndStore(TextRankRequest request) {
        TextRankResult result = compute(request);
        TextRankPersister persister = new TextRankPersister(Label.label(request.getKeywordLabel()));
        Node attachedNode = request.getNode() != null ? request.getNode() : request.getMotherNode();
        persister.persistKeywords(result.getResult(), attachedNode);

        return result.getStatus().equals(TextRankResult.TextRankStatus.SUCCESS)
                ? SingleResult.success()
                : SingleResult.fail();
    }

    public TextRankResult compute(TextRankRequest request) {
        TextRank.Builder textrankBuilder = new TextRank.Builder(getDatabase(), getNLPManager().getConfiguration());
        if (request.getStopWords() != null 
                && !request.getStopWords().isEmpty()) {
            textrankBuilder.setStopwords(request.getStopWords());
        }
        textrankBuilder.removeStopWords(request.isDoStopwords())
                .respectDirections(request.isRespectDirections())
                .respectSentences(request.isRespectSentences())
                .useDependencies(request.isUseDependencies())
                .useDependenciesForCooccurrences(request.isUseDependenciesForCooccurrences())
                //.setCooccurrenceWindow(request.getCooccurrenceWindow())
                .setTopXTags(request.getTopXTags())
                .setCleanKeywords(request.isCleanKeywords())
                .setKeywordLabel(request.getKeywordLabel())
                .setAdmittedPOSs(request.getAdmittedPOSs())
                .setForbiddenPOSs(request.getForbiddenPOSs())
                .setForbiddenNEs(request.getForbiddenNEs());
        
        TextRank textRank = textrankBuilder.build();
        TextRankResult result = textRank.evaluate(request.getNodes(),
                request.getLanguage(),
                request.getIterations(), 
                request.getDamp(), 
                request.getThreshold());
        LOG.info("AnnotatedText with ID(s) " + request.getNodes().stream().map(el -> String.valueOf(el.getId())).collect(Collectors.joining(", ")) + " processed. Result: " + result.getStatus());

        return result;
    }

    public SingleResult postprocess(TextRankPostprocessRequest request) {
        LOG.info("Starting TextRank post-processing ...");
        TextRank.Builder textrankBuilder = new TextRank.Builder(getDatabase(), getNLPManager().getConfiguration());
        textrankBuilder.setKeywordLabel(request.getKeywordLabel());
        if (!textrankBuilder.build().postProcess(request.getMethod(), request.getAnnotatedText()))
            return SingleResult.fail();
        LOG.info("TextRank post-processing completed.");
        return SingleResult.success();
    }
}
