<?php
/**
 * @filesource modules/bp/views/history.php
 *
 * @copyright 2016 Goragod.com
 * @license https://www.kotchasan.com/license/
 *
 * @see https://www.kotchasan.com/
 */

namespace Bp\History;

use Kotchasan\DataTable;
use Kotchasan\Date;
use Kotchasan\Http\Request;
use \Bp\Calculator\Model as Calculator;

/**
 * module=bp-history
 *
 * @author Goragod Wiriya <admin@goragod.com>
 *
 * @since 1.0
 */
class View extends \Gcms\View
{
    /**
     * @var object
     */
    private $category;
    /**
     * @var array
     */
    private $sys = [];
    /**
     * @var array
     */
    private $dia = [];
    /**
     * ตาราง Bp
     *
     * @param Request $request
     * @param object $profile
     *
     * @return string
     */
    public function render(Request $request, $profile)
    {
        $params = [
            'family_id' => $profile->id,
            'member_id' => $profile->member_id,
            'from' => $request->request('from', date('Y-m-d', strtotime('-7 days')))->date(),
            'to' => $request->request('to', date('Y-m-d'))->date(),
            'tag' => $request->request('tag')->topic()
        ];
        $this->category = \Bp\Category\Model::init($profile->member_id);
        // URL สำหรับส่งให้ตาราง
        $uri = $request->createUriWithGlobals(WEB_URL.'index.php');
        // ตาราง
        $table = new DataTable([
            /* Uri */
            'uri' => $uri,
            /* Model */
            'model' => \Bp\History\Model::toDataTable($params),
            /* รายการต่อหน้า */
            'perPage' => $request->cookie('bpHistoryperPage', 30)->toInt(),
            /* เรียงลำดับ */
            'sort' => $request->cookie('bpHistorysort', 'create_date desc')->toString(),
            /* ฟังก์ชั่นจัดรูปแบบการแสดงผลแถวของตาราง */
            'onRow' => [$this, 'onRow'],
            /* ฟังก์ชั่นแสดงผล Footer */
            'onCreateFooter' => [$this, 'onCreateFooter'],
            /* คอลัมน์ที่ไม่ต้องแสดงผล */
            'hideColumns' => ['id'],
            /* ตั้งค่าการกระทำของของตัวเลือกต่างๆ ด้านล่างตาราง ซึ่งจะใช้ร่วมกับการขีดถูกเลือกแถว */
            'action' => 'index.php/bp/model/history/action',
            'actionCallback' => 'dataTableActionCallback',
            'actions' => [
                [
                    'id' => 'action',
                    'class' => 'ok',
                    'text' => '{LNG_With selected}',
                    'options' => [
                        'delete' => '{LNG_Delete}'
                    ]
                ]
            ],
            /* ตัวเลือกด้านบนของตาราง ใช้จำกัดผลลัพท์การ query */
            'filters' => [
                [
                    'type' => 'date',
                    'name' => 'from',
                    'text' => '{LNG_from}',
                    'value' => $params['from']
                ],
                [
                    'type' => 'date',
                    'name' => 'to',
                    'text' => '{LNG_to}',
                    'value' => $params['to']
                ],
                [
                    'name' => 'tag',
                    'text' => '{LNG_Tag}',
                    'options' => [0 => '{LNG_all items}'] + $this->category->toSelect('tag'),
                    'value' => $params['tag']
                ]
            ],
            /* ส่วนหัวของตาราง และการเรียงลำดับ (thead) */
            'headers' => [
                'create_date' => [
                    'text' => '{LNG_Date}',
                    'sort' => 'create_date'
                ],
                'sys1' => [
                    'text' => 'SYS 1',
                    'class' => 'center'
                ],
                'dia1' => [
                    'text' => 'DIA 1',
                    'class' => 'center'
                ],
                'pulse1' => [
                    'text' => '{LNG_Pulse} 1',
                    'class' => 'center'
                ],
                'sys2' => [
                    'text' => 'SYS 2',
                    'class' => 'center'
                ],
                'dia2' => [
                    'text' => 'DIA 2',
                    'class' => 'center'
                ],
                'pulse2' => [
                    'text' => '{LNG_Pulse} 2',
                    'class' => 'center'
                ],
                'height' => [
                    'text' => '{LNG_Height}',
                    'class' => 'center'
                ],
                'weight' => [
                    'text' => '{LNG_Weight}',
                    'class' => 'center'
                ],
                'bmi' => [
                    'text' => '{LNG_BMI}',
                    'class' => 'center'
                ],
                'waist' => [
                    'text' => '{LNG_Waist size}',
                    'class' => 'center'
                ],
                'temperature' => [
                    'text' => '{LNG_Temperature}',
                    'class' => 'center'
                ],
                'tag' => [
                    'text' => '{LNG_Tag}',
                    'class' => 'center',
                    'sort' => 'tag'
                ]
            ],
            /* รูปแบบการแสดงผลของคอลัมน์ (tbody) */
            'cols' => [
                'create_date' => [
                    'class' => 'nowrap'
                ],
                'sys1' => [
                    'class' => 'center'
                ],
                'dia1' => [
                    'class' => 'center'
                ],
                'pulse1' => [
                    'class' => 'center'
                ],
                'sys2' => [
                    'class' => 'center'
                ],
                'dia2' => [
                    'class' => 'center'
                ],
                'pulse2' => [
                    'class' => 'center'
                ],
                'height' => [
                    'class' => 'center nowrap'
                ],
                'weight' => [
                    'class' => 'center nowrap'
                ],
                'bmi' => [
                    'class' => 'center nowrap'
                ],
                'waist' => [
                    'class' => 'center nowrap'
                ],
                'temperature' => [
                    'class' => 'center nowrap'
                ],
                'tag' => [
                    'class' => 'center nowrap'
                ]
            ],
            /* ปุ่มแสดงในแต่ละแถว */
            'buttons' => [
                [
                    'class' => 'icon-edit button green',
                    'href' => $uri->createBackUri(['module' => 'bp-record', 'id' => ':id']),
                    'text' => '{LNG_Edit}'
                ]
            ]
        ]);
        // save cookie
        setcookie('bpHistoryperPage', $table->perPage, time() + 2592000, '/', HOST, HTTPS, true);
        setcookie('bpHistorysort', $table->sort, time() + 2592000, '/', HOST, HTTPS, true);
        // คืนค่า HTML
        $content = $table->render();
        $content .= '<footer class=float_bottom_menu>';
        $content .= '<a class=bp-family title="{LNG_a family member}" href="'.WEB_URL.'index.php?module=bp-family"><span class=icon-users></span></a>';
        $content .= '<a class=bp-record title="{LNG_Record}" href="'.WEB_URL.'index.php?module=bp-record&amp;family_id='.$profile->id.'"><span class=icon-new></span></a>';
        $content .= '<a class=bp-report title="{LNG_Report}" href="'.WEB_URL.'index.php?module=bp-report&amp;id='.$profile->id.'"><span class=icon-stats></span></a>';
        $content .= '</footer>';
        return $content;
    }

    /**
     * จัดรูปแบบการแสดงผลในแต่ละแถว
     *
     * @param array  $item ข้อมูลแถว
     * @param int    $o    ID ของข้อมูล
     * @param object $prop กำหนด properties ของ TR
     *
     * @return array คืนค่า $item กลับไป
     */
    public function onRow($item, $o, $prop)
    {
        if (!empty($item['sys1'])) {
            $this->sys[] = $item['sys1'];
        }
        if (!empty($item['sys2'])) {
            $this->sys[] = $item['sys2'];
        }
        if (!empty($item['dia1'])) {
            $this->dia[] = $item['dia1'];
        }
        if (!empty($item['dia2'])) {
            $this->dia[] = $item['dia2'];
        }
        $item['tag'] = $this->category->get('tag', $item['tag']);
        $item['create_date'] = Date::format($item['create_date']);
        if (empty($item['height']) || empty($item['weight'])) {
            $item['bmi'] = '-';
        } else {
            $height = $item['height'] / 100;
            $bmi = round($item['weight'] / ($height * $height), 2);
            $item['bmi'] = '<span class=color-'.\Bp\Calculator\Model::bmiColor($bmi).'>'.$bmi.'</span>';
        }
        $item['height'] = empty($item['height']) ? '-' : $item['height'].' {LNG_Cm.}';
        $item['weight'] = empty($item['weight']) ? '-' : $item['weight'].' {LNG_Kg.}';
        $item['waist'] = empty($item['waist']) ? '-' : $item['waist'].' {LNG_Cm.}';
        $item['temperature'] = empty($item['temperature']) ? '-' : $item['temperature'].' ℃';
        $item['sys1'] = empty($item['sys1']) ? '-' : '<span class=color-'.Calculator::bpColor($item['sys1'], 0).'>'.$item['sys1'].'</span>';
        $item['sys2'] = empty($item['sys2']) ? '-' : '<span class=color-'.Calculator::bpColor($item['sys2'], 0).'>'.$item['sys2'].'</span>';
        $item['dia1'] = empty($item['dia1']) ? '-' : '<span class=color-'.Calculator::bpColor(0, $item['dia1']).'>'.$item['dia1'].'</span>';
        $item['dia2'] = empty($item['dia2']) ? '-' : '<span class=color-'.Calculator::bpColor(0, $item['dia2']).'>'.$item['dia2'].'</span>';
        return $item;
    }

    /**
     * ฟังก์ชั่นสร้างแถวของ footer
     *
     * @return string
     */
    public function onCreateFooter()
    {
        $tr = '<tr>';
        $tr .= '<td class=right>{LNG_Average}</td>';
        $tr .= '<th class="check-column"><a class="checkall icon-uncheck" title="{LNG_Select all}"></a></th>';
        if (empty($this->sys) || empty($this->sys)) {
            $tr .= '<td colspan=4></td>';
        } else {
            $sys = floor(array_sum($this->sys) / count($this->sys));
            $dia = floor(array_sum($this->dia) / count($this->dia));
            $tr .= '<td class="center color-'.Calculator::bpColor($sys, 0).'" colspan=2>'.$sys.'</td>';
            $tr .= '<td class="center color-'.Calculator::bpColor(0, $dia).'" colspan=2>'.$dia.'</td>';
        }
        $tr .= '<td colspan=9></td></tr>';
        return $tr;
    }
}
