#!/usr/bin/env python
# -*- coding: utf-8 -*- 

#
# Copyright 2013, 2014, 2016, 2017 Guenter Bartsch
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# random tokenization / nlp pre-processing related stuff
#

import sys
import re
import unittest
import logging
from num2words import num2words

def detect_latin1 (fn):

    f = open(fn)

    l1ucnt = 0

    for line in f:

        for c in line:

            o = ord(c)
            if o == 0xc4 or o == 0xd6 or o == 0xdc or o == 0xe4 or o==0xf6 or o==0xfc or o==0xdf:
                l1ucnt += 1


    f.close()

    return l1ucnt > 1

def compress_ws (s):

        vc = True

        res = ''

        for c in s:

                if c == ' ':
                        vc = False
                else:
                        if vc:
                                res = res + c
                        else:
                                res = res + ' ' + c
                        vc = True

        return res 

def isgalnum (s):

    for c in s:
        if c.isalnum():
            continue

        if c == u'%' or c ==u'§':
            continue

        return False

    return True

#####################################################################
#
# tokenizer commons
#
#####################################################################

PERCENT_PATTERN = re.compile(r"(?:(?<=\s|[\(\)\[\];:])|^)[-]?\d+[,.]?\d*\s*[%](?=\s|[\(\)\[\];:.,!?…-]|$)")
NUMBER_PATTERN = re.compile(r"(?:(?<=\s|[\(\)\[\];:])|^)[-]?\d+[,.]?\d*(?=\s|[\(\)\[\];:.,!?…-]|$)")

PUNCTUATION = [
    u',', u'.', u';', 
    u'?', u'!', 
    u'+', u'-', u'*', u'#', u'=', u'|',
    u'/', u'\\',  
    u'[', u']', u'(', u')', u'»', u'«', u'<', u'>', 
    u'\'', u'"', u'”',u'…',u'ʻ',
]

MACRO_PUNCTUATION = [
    u'@', u':'
]

#####################################################################
#
# english tokenizer
#
#####################################################################

# word replacement table
wrt_en = { u'0'          : u'zero',
           u'1'          : u'one',
           u'colour'     : u'color',
           u'okay'       : u'ok',
         }

symb_abbrev_norm_en = [
                       (u'\ufeff'  , u' '),
                       (u'\u2019'  , u"'"),
                       (u'\xa0'    , u' '),
                       (u'\u203a'  , u' '),
                       (u'\u2039'  , u' '),
                       (u'&'       , u'and'),
                       (u'\xa020'  , u' '),
                       (u'„'       , u' '),
                       (u'“'       , u' '),
                       (u'$'       , u'dollar ')
                      ]

def spellout_number_en (m):

    numstr = m.group(0)

    if numstr[0].isspace():
        numstr = numstr[1:]
        res = ' '
    else:
        res = ''

    if numstr[0] == '-':
        res += 'minus '
        numstr = numstr[1:].strip()

    if not '.' in numstr:
        numstr = numstr.replace(',','.')

    if numstr.endswith('%'):
        numstr = numstr[:len(numstr)-1]
        percent = True
    else:
        percent = False

    parts = numstr.split ('.')

    # print repr(parts)

    res += num2words(int(parts[0]))

    if len(parts)>1 and len(parts[1])>0:

        # spell out fractional part in digits

        res += ' point ' + num2words(int(parts[1]))

    if percent:
        res += ' percent'

    return res

APOSTROPHE_S_PATTERN1 = re.compile(r"\w\w+[']s\s")
APOSTROPHE_S_PATTERN2 = re.compile(r"\w\w+[']s$")
APOSTROPHE_S_PATTERN3 = re.compile(r"\w\w+[']s[,.;:]")

def protect_apostrophe_s (m):

    s = m.group(0)

    return s.replace(u"'s", u"✓s")

APOSTROPHE_NT_PATTERN1 = re.compile(r"\wn[']t\s")
APOSTROPHE_NT_PATTERN2 = re.compile(r"\wn[']t$")
APOSTROPHE_NT_PATTERN3 = re.compile(r"\wn[']t[,.;:]")

def protect_apostrophe_nt (m):

    s = m.group(0)

    return s.replace(u"n't", u"n✓t")

APOSTROPHE_D_PATTERN1 = re.compile(r"\w[']d\s")
APOSTROPHE_D_PATTERN2 = re.compile(r"\w[']d$")
APOSTROPHE_D_PATTERN3 = re.compile(r"\w[']d[,.;:]")

def protect_apostrophe_d (m):

    s = m.group(0)

    return s.replace(u"'d", u"✓d")

APOSTROPHE_RE_PATTERN1 = re.compile(r"\w[']re\s")
APOSTROPHE_RE_PATTERN2 = re.compile(r"\w[']re$")
APOSTROPHE_RE_PATTERN3 = re.compile(r"\w[']re[,.;:]")

def protect_apostrophe_re (m):

    s = m.group(0)

    return s.replace(u"'re", u"✓re")

APOSTROPHE_LL_PATTERN1 = re.compile(r"\w[']ll\s")
APOSTROPHE_LL_PATTERN2 = re.compile(r"\w[']ll$")
APOSTROPHE_LL_PATTERN3 = re.compile(r"\w[']ll[,.;:]")

def protect_apostrophe_ll (m):

    s = m.group(0)

    return s.replace(u"'ll", u"✓ll")

def tokenize_en (s, keep_punctuation=False, keep_macros=False, keep_underscores=True):

    global wrt_en, symb_abbrev_norm_en

    for san in symb_abbrev_norm_en:
        srch = san[0]
        repl = san[1]

        s = s.replace (srch, repl)

    s = s.lower()

    # deal with numbers
    s = PERCENT_PATTERN.sub(spellout_number_en, s)
    s = NUMBER_PATTERN.sub(spellout_number_en, s)

    # deal with apostrophe-s
    s = APOSTROPHE_S_PATTERN1.sub(protect_apostrophe_s, s)
    s = APOSTROPHE_S_PATTERN2.sub(protect_apostrophe_s, s)
    s = APOSTROPHE_S_PATTERN3.sub(protect_apostrophe_s, s)

    # deal with apostrophe-nt
    s = APOSTROPHE_NT_PATTERN1.sub(protect_apostrophe_nt, s)
    s = APOSTROPHE_NT_PATTERN2.sub(protect_apostrophe_nt, s)
    s = APOSTROPHE_NT_PATTERN3.sub(protect_apostrophe_nt, s)

    # deal with apostrophe-d (I'd we'd)
    s = APOSTROPHE_D_PATTERN1.sub(protect_apostrophe_d, s)
    s = APOSTROPHE_D_PATTERN2.sub(protect_apostrophe_d, s)
    s = APOSTROPHE_D_PATTERN3.sub(protect_apostrophe_d, s)

    # deal with apostrophe-re (we're they're)
    s = APOSTROPHE_RE_PATTERN1.sub(protect_apostrophe_re, s)
    s = APOSTROPHE_RE_PATTERN2.sub(protect_apostrophe_re, s)
    s = APOSTROPHE_RE_PATTERN3.sub(protect_apostrophe_re, s)

    # deal with apostrophe-ll (we'll they'll)
    s = APOSTROPHE_LL_PATTERN1.sub(protect_apostrophe_ll, s)
    s = APOSTROPHE_LL_PATTERN2.sub(protect_apostrophe_ll, s)
    s = APOSTROPHE_LL_PATTERN3.sub(protect_apostrophe_ll, s)

    # I'm, I've
    s = s.replace (u"i'm", u'i✓m')
    s = s.replace (u"i've", u'i✓ve')

    # deal with punctuation
    if keep_punctuation:
        for p in PUNCTUATION:
            s = s.replace (p, u' ' + p + u' ')
        if not keep_macros:
            for p in MACRO_PUNCTUATION:
                s = s.replace (p, u' ' + p + u' ')
    else:
        for p in PUNCTUATION:
            s = s.replace(p,' ')
        if not keep_macros:
            for p in MACRO_PUNCTUATION:
                s = s.replace(p,' ')
        if not keep_underscores:
            s = s.replace('_',' ')

    # re-insert apostrophes
    s = s.replace (u'✓', u"'")

    res = []

    words = re.split ('\s+', s)

    # print repr(words)

    for word in words:

        w = word.rstrip().replace(u'–',u'')
        if len(w) > 0:

            if w in wrt_en:
                w = wrt_en[w]
        
                words2 = re.split('\s+', w)
                for w2 in words2:
                    res.append(w2)

            else:
                res.append (w)

    return res

#####################################################################
#
# french tokenizer
#
#####################################################################

# word replacement table
wrt_fr = { u'0'          : u'zéro',
           u'1'          : u'un',
         }

symb_abbrev_norm_fr = [
                       (u'\ufeff'  , u' '),
                       (u'\u2019'  , u"'"),
                       (u'\xa0'    , u' '),
                       (u'\u203a'  , u' '),
                       (u'\u2039'  , u' '),
                       (u'&'       , u'et'),
                       (u'\xa020'  , u' '),
                       (u'„'       , u' '),
                       (u'“'       , u' '),
                       (u'«'       , u' '),
                       (u'»'       , u' '),
                       (u'$'       , u'dollar '),
                       (u'€'       , u'euro ')
                      ]

# based on code for german below

w1_fr = u"zéro un deux trois quatre cinq six sept huit neuf dix onze douze treize quatorze quinze seize dix-sept dix-huit dix-neuf".split()
w2_fr = u"vingt trente quarante cinquante soixante".split()

def nombre_en_mots(n, z=False):
    if n < 0: raise ValueError
    if n == 0 and z: return ''
    if n < 20: return w1_fr[n]
    if n < 70:
        w = w2_fr[(n - 20) // 10]
        if n % 10 == 1:
            w = w + u' et un'
        if n % 10 > 1:
            w = w + '-' + w1_fr[n % 10]
        return w
    if n == 71:
        return "soixante et onze"
    if n < 80:
        w = w1_fr[n - 60]
        w = "soixante-" + w
        return w
    if n == 80:
        return "quatre-vingts"
    if n < 100:
        return "quatre-vingt-" + w1_fr[(n - 80)]
    if n < 1000:
        if n // 100 == 1:
            return "cent " + nombre_en_mots(n % 100, z=True)
        return w1_fr[n // 100] + " cent " + nombre_en_mots(n % 100, z=True)
    if n < 2000:
        return "mille " + nombre_en_mots(n % 1000, z=True)
    if n < 1000000:
        return nombre_en_mots(n // 1000) + " mille " + nombre_en_mots(n % 1000, z=True)
    if n < 1000000000:
        m = n // 1000000
        suff = ' millions ' if m>1 else ' million '
        return nombre_en_mots(m) + suff + nombre_en_mots(n % 1000000, z=True)
    if n < 1000000000:
        m = n // 1000000000
        suff = ' milliards ' if m>1 else ' milliard '
        return nombre_en_mots(m) + suff + nombre_en_mots(n % 1000000000, z=True)
    # raise ValueError
    logging.warn('nombre_en_mots: cannot handle %s' % n)
    return str(n)

def spellout_number_fr (m):

    numstr = m.group(0)

    # print "spellout number:", numstr

    if numstr[0].isspace():
        numstr = numstr[1:]
        res = ' '
    else:
        res = ''

    if numstr[0] == '-':
        res += 'moins '
        numstr = numstr[1:].strip()

    if not '.' in numstr:
        numstr = numstr.replace(',','.')

    if numstr.endswith('%'):
        numstr = numstr[:len(numstr)-1].strip()
        percent = True
    else:
        percent = False

    parts = numstr.split ('.')

    # print repr(parts)

    res += nombre_en_mots(int(parts[0]))

    if len(parts)>1 and len(parts[1])>0:

        res += ' virgule'

        # spell out leading zeroes
        fractional_str = parts[1]
        for c in fractional_str:
            if c == '0':
                res += u' zéro'
                fractional_str = fractional_str[1:]
            else:
                break

        # spell out (simpler) fractional part as a whole, otherwise spell by
        # digit
        if len(fractional_str)>0:
            if int(fractional_str) < 1000:
                res += ' ' + nombre_en_mots(int(fractional_str))
            else:
                for c in fractional_str:
                    res += ' ' + nombre_en_mots(int(c))

    if percent:
        res += ' pour cent'

    return res

APOSTROPHE_ELISION_PATTERN1 = re.compile(r"(^|[^\w])(?:[cdjlmnst]|jusqu|lorsqu|presqu|puisqu|qu|quelqu|quoiqu)['][^'.,;]")
APOSTROPHE_ELISION_PATTERN2 = re.compile(r"(^|[^\w]|[✓])(?:aujourd|entr)['][^'.,;]")
PROTECT_DASH_PATTERN1 = re.compile(r"[a-zàéèêëîïùü](-[a-zàéèêëîïùü])+")

def protect_elision (m):

    s = m.group(0)

    return s.replace(u"'", u"✓")

def protect_dash (m):

    s = m.group(0)

    return s.replace(u"-", u"⥬")

def tokenize_fr (s, keep_punctuation=False, keep_macros=False, keep_underscores=True):

    global wrt_fr, symb_abbrev_norm_fr

    for san in symb_abbrev_norm_fr:
        srch = san[0]
        repl = san[1]

        s = s.replace (srch, repl)

    s = s.lower()

    # deal with numbers
    s = PERCENT_PATTERN.sub(spellout_number_fr, s)
    s = NUMBER_PATTERN.sub(spellout_number_fr, s)

    # deal with all dashes between words (including numbers)
    s = PROTECT_DASH_PATTERN1.sub(protect_dash, s)

    # deal with elision
    s = APOSTROPHE_ELISION_PATTERN1.sub(protect_elision, s)
    s = APOSTROPHE_ELISION_PATTERN2.sub(protect_elision, s)

    # deal with punctuation
    if keep_punctuation:
        for p in PUNCTUATION:
            s = s.replace (p, u' ' + p + u' ')
        if not keep_macros:
            for p in MACRO_PUNCTUATION:
                s = s.replace (p, u' ' + p + u' ')
    else:
        for p in PUNCTUATION:
            s = s.replace(p,' ')
        if not keep_macros:
            for p in MACRO_PUNCTUATION:
                s = s.replace(p,' ')
        if not keep_underscores:
            s = s.replace('_',' ')

    # deal with digits that may appear after splitting punctuation.
    s = PERCENT_PATTERN.sub(spellout_number_fr, s)
    s = NUMBER_PATTERN.sub(spellout_number_fr, s)

    # re-insert apostrophes & dashes
    s = s.replace (u'✓', u"'")
    s = s.replace (u'⥬', u"-")

    res = []

    words = re.split ('\s+', s)

    # print repr(words)

    for word in words:

        w = word.rstrip().replace(u'–',u'')
        if len(w) > 0:

            if w in wrt_fr:
                w = wrt_fr[w]

                words2 = re.split('\s+', w)
                for w2 in words2:
                    res.append(w2)

            else:
                res.append (w)

    return res


#####################################################################
#
# german tokenizer
#
#####################################################################

# word replacement table
wrt = { u'0'             : u'null',
        u'10%'           : u'zehn prozent',
        u'100%'          : u'hundert prozent',
        u'1700'          : u'siebzehnhundert',
        u'1825'          : u'achtzehnhundertfünfundzwanzig',
        u'1949'          : u'neunzehnhundertneunundvierzig',
        u'1960ER'        : u'neunzehnhundertsechziger',
        u'1970ER'        : u'neunzehnhundertsiebziger',
        u'1970ERJAHREN'  : u'neunzehnhundertsiebziger jahren',
        u'1977'          : u'neunzehnhundertsiebenundsiebzig',
        u'1979'          : u'neunzehnhundertneunundsiebzig',
        u'1980ER'        : u'neunzehnhundertachtziger',
        u'1983'          : u'neunzehnhundertdreiundachtzig',
        u'1984'          : u'neunzehnhundertvierundachtzig',
        u'1989'          : u'neunzehnhundertneunundachtzig',
        u'1990'          : u'neunzehnhundertneunzig',
        u'1990ER'        : u'neunzehnhundertneunziger',
        u'1991'          : u'neunzehnhunderteinundneunzig',
        u'1993'          : u'neunzehnhundertdreiundneunzig',
        u'1998'          : u'neunzehnhundertachtundneunzig',
        u'2%'            : u'zwei prozent', 
        u'40%'           : u'vierzig prozent', 
        u'80%'           : u'achtzig prozent',
        u'800'           : u'achthundert',
        u'80ER'          : u'achtziger',

        # old spellings extracted from wiktionary

        u'abbat'                       : u'abbot',
        u'abbauprozeß'                 : u'abbauprozess',
        u'abbiß'                       : u'abbiss',
        u'abbißt'                      : u'abbisst',
        u'abblaßt'                     : u'abblasst',
        u'abblaßte'                    : u'abblasste',
        u'abblaßten'                   : u'abblassten',
        u'abblaßtest'                  : u'abblasstest',
        u'abblaßtet'                   : u'abblasstet',
        u'abdaßle'                     : u'abdassle',
        u'abdaßler'                    : u'abdassler',
        u'abdaßlest'                   : u'abdasslest',
        u'abdaßlet'                    : u'abdasslet',
        u'abdaßlung'                   : u'abdasslung',
        u'abdroßle'                    : u'abdrossle',
        u'abdroßlest'                  : u'abdrosslest',
        u'abdroßlet'                   : u'abdrosslet',
        u'abdroßlung'                  : u'abdrosslung',
        u'abend'                       : u'abend',
        u'abendimbiß'                  : u'abendimbiss',
        u'abeßt'                       : u'abesst',
        u'abfalladung'                 : u'abfallladung',
        u'abfallager'                  : u'abfalllager',
        u'abfallaster'                 : u'abfalllaster',
        u'abfalleerung'                : u'abfallleerung',
        u'abfaßen'                     : u'abfassen',
        u'abfaßt'                      : u'abfasst',
        u'abfaßte'                     : u'abfasste',
        u'abfaßten'                    : u'abfassten',
        u'abfaßtest'                   : u'abfasstest',
        u'abfaßtet'                    : u'abfasstet',
        u'abfloß'                      : u'abfloss',
        u'abfloßt'                     : u'abflosst',
        u'abfluß'                      : u'abfluss',
        u'abflußbecken'                : u'abflussbecken',
        u'abflußgebiet'                : u'abflussgebiet',
        u'abflußgraben'                : u'abflussgraben',
        u'abflußhahn'                  : u'abflusshahn',
        u'abflußkanal'                 : u'abflusskanal',
        u'abflußkoeffizient'           : u'abflusskoeffizient',
        u'abflußleitung'               : u'abflussleitung',
        u'abflußloch'                  : u'abflussloch',
        u'abflußlos'                   : u'abflusslos',
        u'abflußlose'                  : u'abflusslose',
        u'abflußlosem'                 : u'abflusslosem',
        u'abflußlosen'                 : u'abflusslosen',
        u'abflußloser'                 : u'abflussloser',
        u'abflußloses'                 : u'abflussloses',
        u'abflußregime'                : u'abflussregime',
        u'abflußreiniger'              : u'abflussreiniger',
        u'abflußrinne'                 : u'abflussrinne',
        u'abflußrinnen'                : u'abflussrinnen',
        u'abflußrohr'                  : u'abflussrohr',
        u'abflußrohre'                 : u'abflussrohre',
        u'abflußrohren'                : u'abflussrohren',
        u'abflußrohres'                : u'abflussrohres',
        u'abflußrohrs'                 : u'abflussrohrs',
        u'abflußsystem'                : u'abflusssystem',
        u'abflußventil'                : u'abflussventil',
        u'abfreßt'                     : u'abfresst',
        u'abfrißt'                     : u'abfrisst',
        u'abgabeschluß'                : u'abgabeschluss',
        u'abgeblaßt'                   : u'abgeblasst',
        u'abgefaßt'                    : u'abgefasst',
        u'abgeküßt'                    : u'abgeküsst',
        u'abgepaßt'                    : u'abgepasst',
        u'abgepaßte'                   : u'abgepasste',
        u'abgepaßtem'                  : u'abgepasstem',
        u'abgepaßten'                  : u'abgepassten',
        u'abgepaßter'                  : u'abgepasster',
        u'abgepaßtes'                  : u'abgepasstes',
        u'abgepreßt'                   : u'abgepresst',
        u'abgeschloßnere'              : u'abgeschlossnere',
        u'abgeschloßnerem'             : u'abgeschlossnerem',
        u'abgeschloßneren'             : u'abgeschlossneren',
        u'abgeschloßnerer'             : u'abgeschlossnerer',
        u'abgeschloßneres'             : u'abgeschlossneres',
        u'abgoß'                       : u'abgoss',
        u'abgoßt'                      : u'abgosst',
        u'abgrundhäßlich'              : u'abgrundhässlich',
        u'abguß'                       : u'abguss',
        u'abgußsammlung'               : u'abgusssammlung',
        u'abißt'                       : u'abisst',
        u'abkühlungsprozeß'            : u'abkühlungsprozess',
        u'abküßt'                      : u'abküsst',
        u'abküßte'                     : u'abküsste',
        u'abküßten'                    : u'abküssten',
        u'abküßtest'                   : u'abküsstest',
        u'abküßtet'                    : u'abküsstet',
        u'ablaß'                       : u'ablass',
        u'ablaßbrief'                  : u'ablassbrief',
        u'ablaßbriefe'                 : u'ablassbriefe',
        u'ablaßbriefen'                : u'ablassbriefen',
        u'ablaßbriefes'                : u'ablassbriefes',
        u'ablaßbriefs'                 : u'ablassbriefs',
        u'ablaßhahn'                   : u'ablasshahn',
        u'ablaßhandel'                 : u'ablasshandel',
        u'ablaßhandels'                : u'ablasshandels',
        u'ablaßhändler'                : u'ablasshändler',
        u'ablaßjahr'                   : u'ablassjahr',
        u'ablaßkrämer'                 : u'ablasskrämer',
        u'ablaßlehre'                  : u'ablasslehre',
        u'ablaßpraxis'                 : u'ablasspraxis',
        u'ablaßprediger'               : u'ablassprediger',
        u'ablaßrohr'                   : u'ablassrohr',
        u'ablaßschraube'               : u'ablassschraube',
        u'ablaßstreit'                 : u'ablassstreit',
        u'ablaßt'                      : u'ablasst',
        u'ablaßventil'                 : u'ablassventil',
        u'ablaßvorrichtung'            : u'ablassvorrichtung',
        u'ablaßwesen'                  : u'ablasswesen',
        u'ablaßzettel'                 : u'ablasszettel',
        u'ablehnungsbeschluß'          : u'ablehnungsbeschluss',
        u'abläßt'                      : u'ablässt',
        u'ablösungsprozeß'             : u'ablösungsprozess',
        u'abmeßt'                      : u'abmesst',
        u'abmißt'                      : u'abmisst',
        u'abnabelungsprozeß'           : u'abnabelungsprozess',
        u'abpaßt'                      : u'abpasst',
        u'abpaßte'                     : u'abpasste',
        u'abpaßten'                    : u'abpassten',
        u'abpaßtest'                   : u'abpasstest',
        u'abpaßtet'                    : u'abpasstet',
        u'abpreßt'                     : u'abpresst',
        u'abpreßte'                    : u'abpresste',
        u'abpreßten'                   : u'abpressten',
        u'abpreßtest'                  : u'abpresstest',
        u'abpreßtet'                   : u'abpresstet',
        u'abriß'                       : u'abriss',
        u'abrißantrag'                 : u'abrissantrag',
        u'abrißantrage'                : u'abrissantrage',
        u'abrißantrages'               : u'abrissantrages',
        u'abrißantrags'                : u'abrissantrags',
        u'abrißanträge'                : u'abrissanträge',
        u'abrißanträgen'               : u'abrissanträgen',
        u'abrißarbeiten'               : u'abrissarbeiten',
        u'abrißbagger'                 : u'abrissbagger',
        u'abrißbirne'                  : u'abrissbirne',
        u'abrißbirnen'                 : u'abrissbirnen',
        u'abrißfirma'                  : u'abrissfirma',
        u'abrißgenehmigung'            : u'abrissgenehmigung',
        u'abrißhaus'                   : u'abrisshaus',
        u'abrißobjekt'                 : u'abrissobjekt',
        u'abrißstopp'                  : u'abrissstopp',
        u'abrißt'                      : u'abrisst',
        u'abrißunternehmen'            : u'abrissunternehmen',
        u'abschiedskuß'                : u'abschiedskuss',
        u'abschloß'                    : u'abschloss',
        u'abschloßt'                   : u'abschlosst',
        u'abschluß'                    : u'abschluss',
        u'abschlußabend'               : u'abschlussabend',
        u'abschlußarbeit'              : u'abschlussarbeit',
        u'abschlußarbeiten'            : u'abschlussarbeiten',
        u'abschlußball'                : u'abschlussball',
        u'abschlußballe'               : u'abschlussballe',
        u'abschlußballes'              : u'abschlussballes',
        u'abschlußballs'               : u'abschlussballs',
        u'abschlußbericht'             : u'abschlussbericht',
        u'abschlußberichte'            : u'abschlussberichte',
        u'abschlußberichten'           : u'abschlussberichten',
        u'abschlußberichtes'           : u'abschlussberichtes',
        u'abschlußberichts'            : u'abschlussberichts',
        u'abschlußbilanz'              : u'abschlussbilanz',
        u'abschlußbälle'               : u'abschlussbälle',
        u'abschlußbällen'              : u'abschlussbällen',
        u'abschlußdatum'               : u'abschlussdatum',
        u'abschlußdebatte'             : u'abschlussdebatte',
        u'abschlußerklärung'           : u'abschlusserklärung',
        u'abschlußexamen'              : u'abschlussexamen',
        u'abschlußfahrt'               : u'abschlussfahrt',
        u'abschlußfeier'               : u'abschlussfeier',
        u'abschlußfeiern'              : u'abschlussfeiern',
        u'abschlußfest'                : u'abschlussfest',
        u'abschlußgala'                : u'abschlussgala',
        u'abschlußjahr'                : u'abschlussjahr',
        u'abschlußjahre'               : u'abschlussjahre',
        u'abschlußjahren'              : u'abschlussjahren',
        u'abschlußjahres'              : u'abschlussjahres',
        u'abschlußjahrs'               : u'abschlussjahrs',
        u'abschlußkappe'               : u'abschlusskappe',
        u'abschlußkappen'              : u'abschlusskappen',
        u'abschlußklasse'              : u'abschlussklasse',
        u'abschlußkonzert'             : u'abschlusskonzert',
        u'abschlußkundgebung'          : u'abschlusskundgebung',
        u'abschlußparty'               : u'abschlussparty',
        u'abschlußphase'               : u'abschlussphase',
        u'abschlußprojekt'             : u'abschlussprojekt',
        u'abschlußprüfung'             : u'abschlussprüfung',
        u'abschlußprüfungen'           : u'abschlussprüfungen',
        u'abschlußrechnung'            : u'abschlussrechnung',
        u'abschlußrede'                : u'abschlussrede',
        u'abschlußreden'               : u'abschlussreden',
        u'abschlußsitzung'             : u'abschlusssitzung',
        u'abschlußspiel'               : u'abschlussspiel',
        u'abschlußtraining'            : u'abschlusstraining',
        u'abschlußtrainings'           : u'abschlusstrainings',
        u'abschlußtreffen'             : u'abschlusstreffen',
        u'abschlußveranstaltung'       : u'abschlussveranstaltung',
        u'abschlußvorstellung'         : u'abschlussvorstellung',
        u'abschlußzahlung'             : u'abschlusszahlung',
        u'abschlußzensur'              : u'abschlusszensur',
        u'abschlußzeremonie'           : u'abschlusszeremonie',
        u'abschlußzeugnis'             : u'abschlusszeugnis',
        u'abschlußzeugnisse'           : u'abschlusszeugnisse',
        u'abschlußzeugnissen'          : u'abschlusszeugnissen',
        u'abschlußzeugnisses'          : u'abschlusszeugnisses',
        u'abschmiß'                    : u'abschmiss',
        u'abschmißt'                   : u'abschmisst',
        u'abschoß'                     : u'abschoss',
        u'abschoßt'                    : u'abschosst',
        u'abschuß'                     : u'abschuss',
        u'abschußbasen'                : u'abschussbasen',
        u'abschußbasis'                : u'abschussbasis',
        u'abschußbefehl'               : u'abschussbefehl',
        u'abschußbefehle'              : u'abschussbefehle',
        u'abschußbefehlen'             : u'abschussbefehlen',
        u'abschußbefehles'             : u'abschussbefehles',
        u'abschußbefehls'              : u'abschussbefehls',
        u'abschußbereit'               : u'abschussbereit',
        u'abschußbewilligung'          : u'abschussbewilligung',
        u'abschußkontingent'           : u'abschusskontingent',
        u'abschußliste'                : u'abschussliste',
        u'abschußlisten'               : u'abschusslisten',
        u'abschußplan'                 : u'abschussplan',
        u'abschußprämie'               : u'abschussprämie',
        u'abschußprämien'              : u'abschussprämien',
        u'abschußquote'                : u'abschussquote',
        u'abschußrampe'                : u'abschussrampe',
        u'abschußrampen'               : u'abschussrampen',
        u'abschußsilo'                 : u'abschusssilo',
        u'abschußvorrichtung'          : u'abschussvorrichtung',
        u'abschußziffer'               : u'abschussziffer',
        u'abszeß'                      : u'abszess',
        u'abwägungsbeschluß'           : u'abwägungsbeschluss',
        u'acajounuß'                   : u'acajounuss',
        u'accusativ'                   : u'akkusativ',
        u'accusativs'                  : u'akkusativs',
        u'achillessehnenriß'           : u'achillessehnenriss',
        u'achselsproß'                 : u'achselspross',
        u'achtkläßler'                 : u'achtklässler',
        u'achtkläßlerin'               : u'achtklässlerin',
        u'activ'                       : u'aktiv',
        u'activs'                      : u'aktivs',
        u'aderlaß'                     : u'aderlass',
        u'aderlaßmännchen'             : u'aderlassmännchen',
        u'adhäsionsprozeß'             : u'adhäsionsprozess',
        u'adhäsionsverschluß'          : u'adhäsionsverschluss',
        u'adreßbeschaffung'            : u'adressbeschaffung',
        u'adreßbestand'                : u'adressbestand',
        u'adreßbuch'                   : u'adressbuch',
        u'adreßbuche'                  : u'adressbuche',
        u'adreßbuches'                 : u'adressbuches',
        u'adreßbuchs'                  : u'adressbuchs',
        u'adreßbuchverlag'             : u'adressbuchverlag',
        u'adreßbücher'                 : u'adressbücher',
        u'adreßbüchern'                : u'adressbüchern',
        u'adreßbüchlein'               : u'adressbüchlein',
        u'adreßbüro'                   : u'adressbüro',
        u'adreßliste'                  : u'adressliste',
        u'adreßsammlung'               : u'adresssammlung',
        u'adreßverzeichnis'            : u'adressverzeichnis',
        u'adreßänderung'               : u'adressänderung',
        u'adventivsproß'               : u'adventivspross',
        u'aegypten'                    : u'ägypten',
        u'aethers'                     : u'äthers',
        u'affengeschiß'                : u'affengeschiss',
        u'africa'                      : u'afrika',
        u'africaner'                   : u'afrikaner',
        u'africanerin'                 : u'afrikanerin',
        u'africanisch'                 : u'afrikanisch',
        u'after-shave'                 : u'aftershave',
        u'afterrüßler'                 : u'afterrüssler',
        u'agrarüberschuß'              : u'agrarüberschuss',
        u'agribusineß'                 : u'agribusiness',
        u'agrobusineß'                 : u'agrobusiness',
        u'ahnenpaß'                    : u'ahnenpass',
        u'airhosteß'                   : u'airhostess',
        u'aktionsausschuß'             : u'aktionsausschuss',
        u'aktivprozeß'                 : u'aktivprozess',
        u'aktschluß'                   : u'aktschluss',
        u'akzelerationsprozeß'         : u'akzelerationsprozess',
        u'akzeß'                       : u'akzess',
        u'alfalfasproß'                : u'alfalfaspross',
        u'alkoholeinfluß'              : u'alkoholeinfluss',
        u'alkoholexzeß'                : u'alkoholexzess',
        u'alkoholgenuß'                : u'alkoholgenuss',
        u'alkoholmißbrauch'            : u'alkoholmissbrauch',
        u'alkoholmißbrauche'           : u'alkoholmissbrauche',
        u'alkoholmißbrauches'          : u'alkoholmissbrauches',
        u'alkoholmißbrauchs'           : u'alkoholmissbrauchs',
        u'alkoholmißbräuche'           : u'alkoholmissbräuche',
        u'alkoholmißbräuchen'          : u'alkoholmissbräuchen',
        u'allelujah'                   : u'alleluja',
        u'allergiepaß'                 : u'allergiepass',
        u'allerseelenablaß'            : u'allerseelenablass',
        u'alpenpaß'                    : u'alpenpass',
        u'alptraumes'                  : u'albtraumes',
        u'altarkuß'                    : u'altarkuss',
        u'alterthum'                   : u'altertum',
        u'alterthums'                  : u'altertums',
        u'alterungsprozeß'             : u'alterungsprozess',
        u'altwienerisch'               : u'alt-wienerisch',
        u'amboß'                       : u'amboss',
        u'amboßbahn'                   : u'ambossbahn',
        u'amboßeinsatz'                : u'ambosseinsatz',
        u'amboßfuß'                    : u'ambossfuß',
        u'amboßklotz'                  : u'ambossklotz',
        u'amboßrüttler'                : u'ambossrüttler',
        u'amboßstock'                  : u'ambossstock',
        u'ambößchen'                   : u'ambösschen',
        u'america'                     : u'amerika',
        u'americaner'                  : u'amerikaner',
        u'americanerin'                : u'amerikanerin',
        u'americanisch'                : u'amerikanisch',
        u'amtsmißbrauch'               : u'amtsmissbrauch',
        u'analoganschluß'              : u'analoganschluss',
        u'analogieschluß'              : u'analogieschluss',
        u'anbiß'                       : u'anbiss',
        u'anbißt'                      : u'anbisst',
        u'anbraßt'                     : u'anbrasst',
        u'anbraßte'                    : u'anbrasste',
        u'anbraßten'                   : u'anbrassten',
        u'anbraßtest'                  : u'anbrasstest',
        u'anbraßtet'                   : u'anbrasstet',
        u'aneinanderfeßle'             : u'aneinanderfessle',
        u'aneinanderfeßlest'           : u'aneinanderfesslest',
        u'aneinanderfeßlet'            : u'aneinanderfesslet',
        u'aneinandergepaßt'            : u'aneinandergepasst',
        u'aneinandergepreßt'           : u'aneinandergepresst',
        u'aneinanderpaßt'              : u'aneinanderpasst',
        u'aneinanderpaßte'             : u'aneinanderpasste',
        u'aneinanderpaßten'            : u'aneinanderpassten',
        u'aneinanderpaßtest'           : u'aneinanderpasstest',
        u'aneinanderpaßtet'            : u'aneinanderpasstet',
        u'aneinanderpreßt'             : u'aneinanderpresst',
        u'aneinanderpreßte'            : u'aneinanderpresste',
        u'aneinanderpreßten'           : u'aneinanderpressten',
        u'aneinanderpreßtest'          : u'aneinanderpresstest',
        u'aneinanderpreßtet'           : u'aneinanderpresstet',
        u'aneinanderschloß'            : u'aneinanderschloss',
        u'aneinanderschloßt'           : u'aneinanderschlosst',
        u'aneßt'                       : u'anesst',
        u'anfaßt'                      : u'anfasst',
        u'anfaßte'                     : u'anfasste',
        u'anfaßten'                    : u'anfassten',
        u'anfaßtest'                   : u'anfasstest',
        u'anfaßtet'                    : u'anfasstet',
        u'anfeßle'                     : u'anfessle',
        u'anfeßlest'                   : u'anfesslest',
        u'anfeßlet'                    : u'anfesslet',
        u'anfloß'                      : u'anfloss',
        u'anfloßt'                     : u'anflosst',
        u'anfreßt'                     : u'anfresst',
        u'anfrißt'                     : u'anfrisst',
        u'angebraßt'                   : u'angebrasst',
        u'angefaßt'                    : u'angefasst',
        u'angelruthe'                  : u'angelrute',
        u'angemeßnere'                 : u'angemessnere',
        u'angemeßnerem'                : u'angemessnerem',
        u'angemeßneren'                : u'angemessneren',
        u'angemeßnerer'                : u'angemessnerer',
        u'angemeßneres'                : u'angemessneres',
        u'angepaßt'                    : u'angepasst',
        u'angepaßte'                   : u'angepasste',
        u'angepaßtem'                  : u'angepasstem',
        u'angepaßten'                  : u'angepassten',
        u'angepaßter'                  : u'angepasster',
        u'angepaßtere'                 : u'angepasstere',
        u'angepaßterem'                : u'angepassterem',
        u'angepaßteren'                : u'angepassteren',
        u'angepaßterer'                : u'angepassterer',
        u'angepaßteres'                : u'angepassteres',
        u'angepaßtes'                  : u'angepasstes',
        u'angepaßteste'                : u'angepassteste',
        u'angepaßtestem'               : u'angepasstestem',
        u'angepaßtesten'               : u'angepasstesten',
        u'angepaßtester'               : u'angepasstester',
        u'angepaßtestes'               : u'angepasstestes',
        u'angepaßtheit'                : u'angepasstheit',
        u'angepißt'                    : u'angepisst',
        u'angepreßt'                   : u'angepresst',
        u'angepreßte'                  : u'angepresste',
        u'angepreßtem'                 : u'angepresstem',
        u'angepreßten'                 : u'angepressten',
        u'angepreßter'                 : u'angepresster',
        u'angepreßtes'                 : u'angepresstes',
        u'anglisierungsprozeß'         : u'anglisierungsprozess',
        u'angoß'                       : u'angoss',
        u'angoßt'                      : u'angosst',
        u'anguß'                       : u'anguss',
        u'anhängeschloß'               : u'anhängeschloss',
        u'anißt'                       : u'anisst',
        u'anlaß'                       : u'anlass',
        u'anlaßfarbe'                  : u'anlassfarbe',
        u'anlaßlos'                    : u'anlasslos',
        u'anlaßlose'                   : u'anlasslose',
        u'anlaßlosem'                  : u'anlasslosem',
        u'anlaßlosen'                  : u'anlasslosen',
        u'anlaßloser'                  : u'anlassloser',
        u'anlaßloses'                  : u'anlassloses',
        u'anlaßt'                      : u'anlasst',
        u'anläßlich'                   : u'anlässlich',
        u'anläßt'                      : u'anlässt',
        u'anmeldeschluß'               : u'anmeldeschluss',
        u'anmeßt'                      : u'anmesst',
        u'anmißt'                      : u'anmisst',
        u'annahmeschluß'               : u'annahmeschluss',
        u'anpassungsprozeß'            : u'anpassungsprozess',
        u'anpaßbar'                    : u'anpassbar',
        u'anpaßt'                      : u'anpasst',
        u'anpaßte'                     : u'anpasste',
        u'anpaßten'                    : u'anpassten',
        u'anpaßtest'                   : u'anpasstest',
        u'anpaßtet'                    : u'anpasstet',
        u'anpißt'                      : u'anpisst',
        u'anpißte'                     : u'anpisste',
        u'anpißten'                    : u'anpissten',
        u'anpißtest'                   : u'anpisstest',
        u'anpißtet'                    : u'anpisstet',
        u'anpreßrolle'                 : u'anpressrolle',
        u'anpreßt'                     : u'anpresst',
        u'anpreßte'                    : u'anpresste',
        u'anpreßten'                   : u'anpressten',
        u'anpreßtest'                  : u'anpresstest',
        u'anpreßtet'                   : u'anpresstet',
        u'anquaßle'                    : u'anquassle',
        u'anquaßlest'                  : u'anquasslest',
        u'anquaßlet'                   : u'anquasslet',
        u'anraßle'                     : u'anrassle',
        u'anraßlest'                   : u'anrasslest',
        u'anraßlet'                    : u'anrasslet',
        u'anriß'                       : u'anriss',
        u'anrißt'                      : u'anrisst',
        u'anschiß'                     : u'anschiss',
        u'anschißt'                    : u'anschisst',
        u'anschloß'                    : u'anschloss',
        u'anschloßt'                   : u'anschlosst',
        u'anschluß'                    : u'anschluss',
        u'anschlußbahn'                : u'anschlussbahn',
        u'anschlußbillett'             : u'anschlussbillett',
        u'anschlußbus'                 : u'anschlussbus',
        u'anschlußdose'                : u'anschlussdose',
        u'anschlußfertig'              : u'anschlussfertig',
        u'anschlußflug'                : u'anschlussflug',
        u'anschlußgebühr'              : u'anschlussgebühr',
        u'anschlußgleis'               : u'anschlussgleis',
        u'anschlußgoal'                : u'anschlussgoal',
        u'anschlußkabel'               : u'anschlusskabel',
        u'anschlußkonkurs'             : u'anschlusskonkurs',
        u'anschlußlinie'               : u'anschlusslinie',
        u'anschlußprojekt'             : u'anschlussprojekt',
        u'anschlußstelle'              : u'anschlussstelle',
        u'anschlußstrecke'             : u'anschlussstrecke',
        u'anschlußstutzen'             : u'anschlussstutzen',
        u'anschlußstück'               : u'anschlussstück',
        u'anschlußtor'                 : u'anschlusstor',
        u'anschlußtreffer'             : u'anschlusstreffer',
        u'anschlußtreffern'            : u'anschlusstreffern',
        u'anschlußtreffers'            : u'anschlusstreffers',
        u'anschlußverfahren'           : u'anschlussverfahren',
        u'anschlußverfahrens'          : u'anschlussverfahrens',
        u'anschlußzug'                 : u'anschlusszug',
        u'anschlußzuge'                : u'anschlusszuge',
        u'anschlußzuges'               : u'anschlusszuges',
        u'anschlußzugs'                : u'anschlusszugs',
        u'anschlußzwang'               : u'anschlusszwang',
        u'anschlußzüge'                : u'anschlusszüge',
        u'anschlußzügen'               : u'anschlusszügen',
        u'anschmiß'                    : u'anschmiss',
        u'anschmißt'                   : u'anschmisst',
        u'anschovis'                   : u'anchovis',
        u'anschoß'                     : u'anschoss',
        u'anschoßt'                    : u'anschosst',
        u'anschuß'                     : u'anschuss',
        u'antheil'                     : u'anteil',
        u'anwaltsprozeß'               : u'anwaltsprozess',
        u'arachisnuß'                  : u'arachisnuss',
        u'arbeitsausschuß'             : u'arbeitsausschuss',
        u'arbeitsgerichtsprozeß'       : u'arbeitsgerichtsprozess',
        u'arbeitsgeräth'               : u'arbeitsgerät',
        u'arbeitsprozeß'               : u'arbeitsprozess',
        u'arbeitsschluß'               : u'arbeitsschluss',
        u'arekanuß'                    : u'arekanuss',
        u'armguß'                      : u'armguss',
        u'armuth'                      : u'armut',
        u'artilleriebeschuß'           : u'artilleriebeschuss',
        u'arzneimittelmißbrauch'       : u'arzneimittelmissbrauch',
        u'as'                          : u'ass',
        u'aschantinuß'                 : u'aschantinuss',
        u'asteroïden'                  : u'asteroiden',
        u'asylmißbrauch'               : u'asylmissbrauch',
        u'asylmißbrauche'              : u'asylmissbrauche',
        u'asylmißbrauches'             : u'asylmissbrauches',
        u'asylmißbrauchs'              : u'asylmissbrauchs',
        u'asylmißbräuche'              : u'asylmissbräuche',
        u'asylmißbräuchen'             : u'asylmissbräuchen',
        u'athem'                       : u'atem',
        u'atommüllager'                : u'atommülllager',
        u'atommüllagern'               : u'atommülllagern',
        u'atommüllagers'               : u'atommülllagers',
        u'atommüllagerung'             : u'atommülllagerung',
        u'atommüllagerungen'           : u'atommülllagerungen',
        u'attractionssphäre'           : u'attraktionssphäre',
        u'aufbauprozeß'                : u'aufbauprozess',
        u'aufbereitungsprozeß'         : u'aufbereitungsprozess',
        u'aufbeßre'                    : u'aufbessre',
        u'aufbeßrest'                  : u'aufbessrest',
        u'aufbeßret'                   : u'aufbessret',
        u'aufbiß'                      : u'aufbiss',
        u'aufbißt'                     : u'aufbisst',
        u'aufbraßt'                    : u'aufbrasst',
        u'aufbraßte'                   : u'aufbrasste',
        u'aufbraßten'                  : u'aufbrassten',
        u'aufbraßtest'                 : u'aufbrasstest',
        u'aufbraßtet'                  : u'aufbrasstet',
        u'aufeinanderbiß'              : u'aufeinanderbiss',
        u'aufeinanderbißt'             : u'aufeinanderbisst',
        u'aufeinandergepaßt'           : u'aufeinandergepasst',
        u'aufeinandergepaßte'          : u'aufeinandergepasste',
        u'aufeinandergepaßtem'         : u'aufeinandergepasstem',
        u'aufeinandergepaßten'         : u'aufeinandergepassten',
        u'aufeinandergepaßter'         : u'aufeinandergepasster',
        u'aufeinandergepaßtes'         : u'aufeinandergepasstes',
        u'aufeinandergepreßt'          : u'aufeinandergepresst',
        u'aufeinanderpaßt'             : u'aufeinanderpasst',
        u'aufeinanderpaßte'            : u'aufeinanderpasste',
        u'aufeinanderpaßten'           : u'aufeinanderpassten',
        u'aufeinanderpaßtest'          : u'aufeinanderpasstest',
        u'aufeinanderpaßtet'           : u'aufeinanderpasstet',
        u'aufeinanderpreßt'            : u'aufeinanderpresst',
        u'aufeinanderpreßte'           : u'aufeinanderpresste',
        u'aufeinanderpreßten'          : u'aufeinanderpressten',
        u'aufeinanderpreßtest'         : u'aufeinanderpresstest',
        u'aufeinanderpreßtet'          : u'aufeinanderpresstet',
        u'aufeßt'                      : u'aufesst',
        u'auffaßt'                     : u'auffasst',
        u'auffaßte'                    : u'auffasste',
        u'auffaßten'                   : u'auffassten',
        u'auffaßtest'                  : u'auffasstest',
        u'auffaßtet'                   : u'auffasstet',
        u'auffreßt'                    : u'auffresst',
        u'auffrißt'                    : u'auffrisst',
        u'aufgebraßt'                  : u'aufgebrasst',
        u'aufgefaßt'                   : u'aufgefasst',
        u'aufgehißt'                   : u'aufgehisst',
        u'aufgehißte'                  : u'aufgehisste',
        u'aufgehißtem'                 : u'aufgehisstem',
        u'aufgehißten'                 : u'aufgehissten',
        u'aufgehißter'                 : u'aufgehisster',
        u'aufgehißtes'                 : u'aufgehisstes',
        u'aufgepaßt'                   : u'aufgepasst',
        u'aufgepreßt'                  : u'aufgepresst',
        u'aufgoß'                      : u'aufgoss',
        u'aufgoßt'                     : u'aufgosst',
        u'aufguß'                      : u'aufguss',
        u'aufgußbeutel'                : u'aufgussbeutel',
        u'aufgußtierchen'              : u'aufgusstierchen',
        u'aufhißt'                     : u'aufhisst',
        u'aufhißte'                    : u'aufhisste',
        u'aufhißten'                   : u'aufhissten',
        u'aufhißtest'                  : u'aufhisstest',
        u'aufhißtet'                   : u'aufhisstet',
        u'aufißt'                      : u'aufisst',
        u'aufklärungsbeschluß'         : u'aufklärungsbeschluss',
        u'auflaßt'                     : u'auflasst',
        u'aufläßt'                     : u'auflässt',
        u'auflösungsprozeß'            : u'auflösungsprozess',
        u'aufmeßt'                     : u'aufmesst',
        u'aufmißt'                     : u'aufmisst',
        u'aufpaßt'                     : u'aufpasst',
        u'aufpaßte'                    : u'aufpasste',
        u'aufpaßten'                   : u'aufpassten',
        u'aufpaßtest'                  : u'aufpasstest',
        u'aufpaßtet'                   : u'aufpasstet',
        u'aufpraßle'                   : u'aufprassle',
        u'aufpraßlest'                 : u'aufprasslest',
        u'aufpraßlet'                  : u'aufprasslet',
        u'aufpraßlung'                 : u'aufprasslung',
        u'aufpreßt'                    : u'aufpresst',
        u'aufpreßte'                   : u'aufpresste',
        u'aufpreßten'                  : u'aufpressten',
        u'aufpreßtest'                 : u'aufpresstest',
        u'aufpreßtet'                  : u'aufpresstet',
        u'aufriß'                      : u'aufriss',
        u'aufrißt'                     : u'aufrisst',
        u'aufrißzeichnung'             : u'aufrisszeichnung',
        u'aufschloß'                   : u'aufschloss',
        u'aufschloßt'                  : u'aufschlosst',
        u'aufschluß'                   : u'aufschluss',
        u'aufschlußarbeiten'           : u'aufschlussarbeiten',
        u'aufschlußreich'              : u'aufschlussreich',
        u'aufschlußreiche'             : u'aufschlussreiche',
        u'aufschlußreichem'            : u'aufschlussreichem',
        u'aufschlußreichen'            : u'aufschlussreichen',
        u'aufschlußreicher'            : u'aufschlussreicher',
        u'aufschlußreichere'           : u'aufschlussreichere',
        u'aufschlußreicherem'          : u'aufschlussreicherem',
        u'aufschlußreicheren'          : u'aufschlussreicheren',
        u'aufschlußreicherer'          : u'aufschlussreicherer',
        u'aufschlußreicheres'          : u'aufschlussreicheres',
        u'aufschlußreiches'            : u'aufschlussreiches',
        u'aufschlußreichste'           : u'aufschlussreichste',
        u'aufschlußreichstem'          : u'aufschlussreichstem',
        u'aufschlußreichsten'          : u'aufschlussreichsten',
        u'aufschlußreichster'          : u'aufschlussreichster',
        u'aufschlußreichstes'          : u'aufschlussreichstes',
        u'aufschlüßle'                 : u'aufschlüssle',
        u'aufschlüßlest'               : u'aufschlüsslest',
        u'aufschlüßlet'                : u'aufschlüsslet',
        u'aufschlüßlung'               : u'aufschlüsslung',
        u'aufschmiß'                   : u'aufschmiss',
        u'aufschmißt'                  : u'aufschmisst',
        u'aufschoß'                    : u'aufschoss',
        u'aufschoßt'                   : u'aufschosst',
        u'aufsproß'                    : u'aufspross',
        u'aufsproßt'                   : u'aufsprosst',
        u'augsproß'                    : u'augspross',
        u'ausbeßre'                    : u'ausbessre',
        u'ausbeßrest'                  : u'ausbessrest',
        u'ausbeßret'                   : u'ausbessret',
        u'ausbiß'                      : u'ausbiss',
        u'ausbißt'                     : u'ausbisst',
        u'ausblaßt'                    : u'ausblasst',
        u'ausblaßte'                   : u'ausblasste',
        u'ausblaßten'                  : u'ausblassten',
        u'ausblaßtest'                 : u'ausblasstest',
        u'ausblaßtet'                  : u'ausblasstet',
        u'auseinanderfloß'             : u'auseinanderfloss',
        u'auseinanderfloßt'            : u'auseinanderflosst',
        u'auseinanderriß'              : u'auseinanderriss',
        u'auseinanderrißt'             : u'auseinanderrisst',
        u'auseßt'                      : u'ausesst',
        u'ausfirnißt'                  : u'ausfirnisst',
        u'ausfirnißte'                 : u'ausfirnisste',
        u'ausfirnißten'                : u'ausfirnissten',
        u'ausfirnißtest'               : u'ausfirnisstest',
        u'ausfirnißtet'                : u'ausfirnisstet',
        u'ausfloß'                     : u'ausfloss',
        u'ausfloßt'                    : u'ausflosst',
        u'ausfluß'                     : u'ausfluss',
        u'ausflußgeschwindigkeit'      : u'ausflussgeschwindigkeit',
        u'ausflußhahn'                 : u'ausflusshahn',
        u'ausflußloch'                 : u'ausflussloch',
        u'ausflußmenge'                : u'ausflussmenge',
        u'ausflußrohr'                 : u'ausflussrohr',
        u'ausflußöffnung'              : u'ausflussöffnung',
        u'ausfreßt'                    : u'ausfresst',
        u'ausfrißt'                    : u'ausfrisst',
        u'ausfuhrüberschuß'            : u'ausfuhrüberschuss',
        u'ausgeblaßt'                  : u'ausgeblasst',
        u'ausgefirnißt'                : u'ausgefirnisst',
        u'ausgelaßner'                 : u'ausgelassner',
        u'ausgelaßnere'                : u'ausgelassnere',
        u'ausgelaßnerem'               : u'ausgelassnerem',
        u'ausgelaßneren'               : u'ausgelassneren',
        u'ausgelaßnerer'               : u'ausgelassnerer',
        u'ausgelaßneres'               : u'ausgelassneres',
        u'ausgepreßt'                  : u'ausgepresst',
        u'ausgethan'                   : u'ausgetan',
        u'ausgoß'                      : u'ausgoss',
        u'ausgoßt'                     : u'ausgosst',
        u'ausguß'                      : u'ausguss',
        u'ausgußbecken'                : u'ausgussbecken',
        u'ausgußrohr'                  : u'ausgussrohr',
        u'ausgußwasser'                : u'ausgusswasser',
        u'ausißt'                      : u'ausisst',
        u'auskopierprozeß'             : u'auskopierprozess',
        u'auslaß'                      : u'auslass',
        u'auslaßhahn'                  : u'auslasshahn',
        u'auslaßt'                     : u'auslasst',
        u'auslaßventil'                : u'auslassventil',
        u'ausländerhaß'                : u'ausländerhass',
        u'ausläßt'                     : u'auslässt',
        u'ausmeßt'                     : u'ausmesst',
        u'ausmißt'                     : u'ausmisst',
        u'auspreßt'                    : u'auspresst',
        u'auspreßte'                   : u'auspresste',
        u'auspreßten'                  : u'auspressten',
        u'auspreßtest'                 : u'auspresstest',
        u'auspreßtet'                  : u'auspresstet',
        u'ausriß'                      : u'ausriss',
        u'ausrißt'                     : u'ausrisst',
        u'ausschiß'                    : u'ausschiss',
        u'ausschißt'                   : u'ausschisst',
        u'ausschloß'                   : u'ausschloss',
        u'ausschloßt'                  : u'ausschlosst',
        u'ausschluß'                   : u'ausschluss',
        u'ausschlußkriterien'          : u'ausschlusskriterien',
        u'ausschlußkriterium'          : u'ausschlusskriterium',
        u'ausschlußkriteriums'         : u'ausschlusskriteriums',
        u'ausschlußverfahren'          : u'ausschlussverfahren',
        u'ausschlußverfahrens'         : u'ausschlussverfahrens',
        u'ausschoß'                    : u'ausschoss',
        u'ausschoßt'                   : u'ausschosst',
        u'ausschuß'                    : u'ausschuss',
        u'ausschußberatung'            : u'ausschussberatung',
        u'ausschußmitglied'            : u'ausschussmitglied',
        u'ausschußmitgliede'           : u'ausschussmitgliede',
        u'ausschußmitglieder'          : u'ausschussmitglieder',
        u'ausschußmitgliedern'         : u'ausschussmitgliedern',
        u'ausschußmitgliedes'          : u'ausschussmitgliedes',
        u'ausschußmitglieds'           : u'ausschussmitglieds',
        u'ausschußquote'               : u'ausschussquote',
        u'ausschußsitzung'             : u'ausschusssitzung',
        u'ausschußvorsitzende'         : u'ausschussvorsitzende',
        u'ausschußvorsitzendem'        : u'ausschussvorsitzendem',
        u'ausschußvorsitzenden'        : u'ausschussvorsitzenden',
        u'ausschußvorsitzender'        : u'ausschussvorsitzender',
        u'ausschußware'                : u'ausschussware',
        u'austerfäßchen'               : u'austerfässchen',
        u'autobahnanschluß'            : u'autobahnanschluss',
        u'autobahnanschlußstelle'      : u'autobahnanschlussstelle',
        u'autotelephon'                : u'autotelefon',
        u'autotelephone'               : u'autotelefone',
        u'autotelephonen'              : u'autotelefonen',
        u'autotelephons'               : u'autotelefons',
        u'autumnalkatarr'              : u'autumnalkatarrh',
        u'autumnalkatarre'             : u'autumnalkatarrhe',
        u'autumnalkatarren'            : u'autumnalkatarrhen',
        u'autumnalkatarrs'             : u'autumnalkatarrhs',
        u'außengrenzen'                : u'aussengrenzen',
        u'außerbewußt'                 : u'außerbewusst',
        u'außerbewußte'                : u'außerbewusste',
        u'außerbewußtem'               : u'außerbewusstem',
        u'außerbewußten'               : u'außerbewussten',
        u'außerbewußter'               : u'außerbewusster',
        u'außerbewußtes'               : u'außerbewusstes',
        u'babies'                      : u'babys',
        u'backsteingothik'             : u'backsteingotik',
        u'bahnanschluß'                : u'bahnanschluss',
        u'bajonettanschluß'            : u'bajonettanschluss',
        u'bajonettverschluß'           : u'bajonettverschluss',
        u'balancirt'                   : u'balanciert',
        u'balkanisierungsprozeß'       : u'balkanisierungsprozess',
        u'ballettanz'                  : u'balletttanz',
        u'ballettheater'               : u'balletttheater',
        u'ballettänzer'                : u'balletttänzer',
        u'ballettänzerin'              : u'balletttänzerin',
        u'ballettänzerinnen'           : u'balletttänzerinnen',
        u'ballettänzern'               : u'balletttänzern',
        u'ballettänzers'               : u'balletttänzers',
        u'bambussproß'                 : u'bambusspross',
        u'bandpaßfilter'               : u'bandpassfilter',
        u'bandpaßfiltern'              : u'bandpassfiltern',
        u'bandpaßfilters'              : u'bandpassfilters',
        u'baroneß'                     : u'baroness',
        u'barriquefaß'                 : u'barriquefass',
        u'basküleverschluß'            : u'basküleverschluss',
        u'bauchschuß'                  : u'bauchschuss',
        u'bauernschenke'               : u'bauernschänke',
        u'baukostenzuschuß'            : u'baukostenzuschuss',
        u'baumkänguruh'                : u'baumkänguru',
        u'bauzuschuß'                  : u'bauzuschuss',
        u'baß'                         : u'bass',
        u'baßbariton'                  : u'bassbariton',
        u'baßbläser'                   : u'bassbläser',
        u'baßbläserin'                 : u'bassbläserin',
        u'baßbuffo'                    : u'bassbuffo',
        u'baßdrum'                     : u'bassdrum',
        u'baßflöte'                    : u'bassflöte',
        u'baßflöten'                   : u'bassflöten',
        u'baßgeige'                    : u'bassgeige',
        u'baßgeigen'                   : u'bassgeigen',
        u'baßgeiger'                   : u'bassgeiger',
        u'baßgeigerin'                 : u'bassgeigerin',
        u'baßgitarre'                  : u'bassgitarre',
        u'baßgitarren'                 : u'bassgitarren',
        u'baßhorn'                     : u'basshorn',
        u'baßinstrument'               : u'bassinstrument',
        u'baßklarinette'               : u'bassklarinette',
        u'baßlage'                     : u'basslage',
        u'baßlaute'                    : u'basslaute',
        u'baßlinie'                    : u'basslinie',
        u'baßpartie'                   : u'basspartie',
        u'baßposaune'                  : u'bassposaune',
        u'baßposaunen'                 : u'bassposaunen',
        u'baßsaite'                    : u'basssaite',
        u'baßschlüssel'                : u'bassschlüssel',
        u'baßschlüsseln'               : u'bassschlüsseln',
        u'baßschlüssels'               : u'bassschlüssels',
        u'baßsolo'                     : u'basssolo',
        u'baßstimme'                   : u'bassstimme',
        u'baßstimmen'                  : u'bassstimmen',
        u'baßsänger'                   : u'basssänger',
        u'baßsängern'                  : u'basssängern',
        u'baßsängers'                  : u'basssängers',
        u'baßtiefe'                    : u'basstiefe',
        u'baßton'                      : u'basston',
        u'baßtuba'                     : u'basstuba',
        u'baßtölpel'                   : u'basstölpel',
        u'baßtölpeln'                  : u'basstölpeln',
        u'baßtölpels'                  : u'basstölpels',
        u'beauties'                    : u'beautys',
        u'beckmeßre'                   : u'beckmessre',
        u'becreutzig'                  : u'bekreuzig',
        u'becreutzige'                 : u'bekreuzige',
        u'becreutzigen'                : u'bekreuzigen',
        u'becreutzigend'               : u'bekreuzigend',
        u'becreutzigest'               : u'bekreuzigest',
        u'becreutziget'                : u'bekreuziget',
        u'becreutzigst'                : u'bekreuzigst',
        u'becreutzigt'                 : u'bekreuzigt',
        u'becreutzigte'                : u'bekreuzigte',
        u'becreutzigten'               : u'bekreuzigten',
        u'becreutzigtest'              : u'bekreuzigtest',
        u'becreutzigtet'               : u'bekreuzigtet',
        u'beeinfluß'                   : u'beeinfluss',
        u'beeinflußbar'                : u'beeinflussbar',
        u'beeinflußbare'               : u'beeinflussbare',
        u'beeinflußbarem'              : u'beeinflussbarem',
        u'beeinflußbaren'              : u'beeinflussbaren',
        u'beeinflußbarer'              : u'beeinflussbarer',
        u'beeinflußbarere'             : u'beeinflussbarere',
        u'beeinflußbarerem'            : u'beeinflussbarerem',
        u'beeinflußbareren'            : u'beeinflussbareren',
        u'beeinflußbarerer'            : u'beeinflussbarerer',
        u'beeinflußbareres'            : u'beeinflussbareres',
        u'beeinflußbares'              : u'beeinflussbares',
        u'beeinflußbarkeit'            : u'beeinflussbarkeit',
        u'beeinflußbarste'             : u'beeinflussbarste',
        u'beeinflußbarstem'            : u'beeinflussbarstem',
        u'beeinflußbarsten'            : u'beeinflussbarsten',
        u'beeinflußbarster'            : u'beeinflussbarster',
        u'beeinflußbarstes'            : u'beeinflussbarstes',
        u'beeinflußt'                  : u'beeinflusst',
        u'beeinflußte'                 : u'beeinflusste',
        u'beeinflußtem'                : u'beeinflusstem',
        u'beeinflußten'                : u'beeinflussten',
        u'beeinflußter'                : u'beeinflusster',
        u'beeinflußtere'               : u'beeinflusstere',
        u'beeinflußterem'              : u'beeinflussterem',
        u'beeinflußteren'              : u'beeinflussteren',
        u'beeinflußterer'              : u'beeinflussterer',
        u'beeinflußteres'              : u'beeinflussteres',
        u'beeinflußtes'                : u'beeinflusstes',
        u'beeinflußtest'               : u'beeinflusstest',
        u'beeinflußteste'              : u'beeinflussteste',
        u'beeinflußtestem'             : u'beeinflusstestem',
        u'beeinflußtesten'             : u'beeinflusstesten',
        u'beeinflußtester'             : u'beeinflusstester',
        u'beeinflußtestes'             : u'beeinflusstestes',
        u'beeinflußtet'                : u'beeinflusstet',
        u'befaß'                       : u'befass',
        u'befaßt'                      : u'befasst',
        u'befaßte'                     : u'befasste',
        u'befaßten'                    : u'befassten',
        u'befaßtest'                   : u'befasstest',
        u'befaßtet'                    : u'befasstet',
        u'befliß'                      : u'befliss',
        u'beflißner'                   : u'beflissner',
        u'beflißnere'                  : u'beflissnere',
        u'beflißnerem'                 : u'beflissnerem',
        u'beflißneren'                 : u'beflissneren',
        u'beflißnerer'                 : u'beflissnerer',
        u'beflißneres'                 : u'beflissneres',
        u'beflißt'                     : u'beflisst',
        u'begnadigungserlaß'           : u'begnadigungserlass',
        u'begoß'                       : u'begoss',
        u'begoßt'                      : u'begosst',
        u'begrüßungskuß'               : u'begrüßungskuss',
        u'behend'                      : u'behänd',
        u'behende'                     : u'behände',
        u'behendigkeit'                : u'behändigkeit',
        u'behennuß'                    : u'behennuss',
        u'behältniß'                   : u'behältnis',
        u'beimeßt'                     : u'beimesst',
        u'beimißt'                     : u'beimisst',
        u'beinschuß'                   : u'beinschuss',
        u'beischloß'                   : u'beischloss',
        u'beischloßt'                  : u'beischlosst',
        u'beischluß'                   : u'beischluss',
        u'beischoß'                    : u'beischoss',
        u'beischoßt'                   : u'beischosst',
        u'belaß'                       : u'belass',
        u'belaßt'                      : u'belasst',
        u'beleidigungsprozeß'          : u'beleidigungsprozess',
        u'belemmern'                   : u'belämmern',
        u'belemmernd'                  : u'belämmernd',
        u'belemmert'                   : u'belämmert',
        u'belkanto'                    : u'belcanto',
        u'belkantos'                   : u'belcantos',
        u'beläßt'                      : u'belässt',
        u'bemeßt'                      : u'bemesst',
        u'bemiß'                       : u'bemiss',
        u'bemißt'                      : u'bemisst',
        u'bemißtrauen'                 : u'bemisstrauen',
        u'bemißtrauend'                : u'bemisstrauend',
        u'bemißtraut'                  : u'bemisstraut',
        u'bennuß'                      : u'bennuss',
        u'benutzer:asgar'              : u'xxxxx',
        u'benutzer:balû/aal'           : u'ål',
        u'benäß'                       : u'benäss',
        u'benäßt'                      : u'benässt',
        u'benäßte'                     : u'benässte',
        u'benäßten'                    : u'benässten',
        u'benäßtest'                   : u'benässtest',
        u'benäßtet'                    : u'benässtet',
        u'bepiß'                       : u'bepiss',
        u'bepißt'                      : u'bepisst',
        u'bepißte'                     : u'bepisste',
        u'bepißten'                    : u'bepissten',
        u'bepißtest'                   : u'bepisstest',
        u'bepißtet'                    : u'bepisstet',
        u'bequaßle'                    : u'bequassle',
        u'bequaßlest'                  : u'bequasslest',
        u'bequaßlet'                   : u'bequasslet',
        u'berathung'                   : u'beratung',
        u'beratungsausschuß'           : u'beratungsausschuss',
        u'bergkänguruh'                : u'bergkänguru',
        u'beschiß'                     : u'beschiss',
        u'beschißt'                    : u'beschisst',
        u'beschloß'                    : u'beschloss',
        u'beschloßt'                   : u'beschlosst',
        u'beschluß'                    : u'beschluss',
        u'beschlußcharakter'           : u'beschlusscharakter',
        u'beschlußfassend'             : u'beschlussfassend',
        u'beschlußfassung'             : u'beschlussfassung',
        u'beschlußfähig'               : u'beschlussfähig',
        u'beschlußfähige'              : u'beschlussfähige',
        u'beschlußfähigem'             : u'beschlussfähigem',
        u'beschlußfähigen'             : u'beschlussfähigen',
        u'beschlußfähiger'             : u'beschlussfähiger',
        u'beschlußfähiges'             : u'beschlussfähiges',
        u'beschlußfähigkeit'           : u'beschlussfähigkeit',
        u'beschlußlage'                : u'beschlusslage',
        u'beschlußorgan'               : u'beschlussorgan',
        u'beschlußrecht'               : u'beschlussrecht',
        u'beschlußreif'                : u'beschlussreif',
        u'beschlußunfähig'             : u'beschlussunfähig',
        u'beschlußunfähigkeit'         : u'beschlussunfähigkeit',
        u'beschmiß'                    : u'beschmiss',
        u'beschmißt'                   : u'beschmisst',
        u'beschoß'                     : u'beschoss',
        u'beschoßt'                    : u'beschosst',
        u'beschuß'                     : u'beschuss',
        u'beschußstempel'              : u'beschussstempel',
        u'besorgniß'                   : u'besorgnis',
        u'bestandtheile'               : u'bestandteile',
        u'bestelliste'                 : u'bestellliste',
        u'bestgehaßt'                  : u'bestgehasst',
        u'bestußt'                     : u'bestusst',
        u'bestußte'                    : u'bestusste',
        u'bestußtem'                   : u'bestusstem',
        u'bestußten'                   : u'bestussten',
        u'bestußter'                   : u'bestusster',
        u'bestußtere'                  : u'bestusstere',
        u'bestußterem'                 : u'bestussterem',
        u'bestußteren'                 : u'bestussteren',
        u'bestußterer'                 : u'bestussterer',
        u'bestußteres'                 : u'bestussteres',
        u'bestußtes'                   : u'bestusstes',
        u'bestußteste'                 : u'bestussteste',
        u'bestußtestem'                : u'bestusstestem',
        u'bestußtesten'                : u'bestusstesten',
        u'bestußtester'                : u'bestusstester',
        u'bestußtestes'                : u'bestusstestes',
        u'betelnuß'                    : u'betelnuss',
        u'betelnußpalme'               : u'betelnusspalme',
        u'betonguß'                    : u'betonguss',
        u'betreßt'                     : u'betresst',
        u'betreßte'                    : u'betresste',
        u'betreßtem'                   : u'betresstem',
        u'betreßten'                   : u'betressten',
        u'betreßter'                   : u'betresster',
        u'betreßtes'                   : u'betresstes',
        u'betriebsausschuß'            : u'betriebsausschuss',
        u'betriebspaß'                 : u'betriebspass',
        u'betriebsschluß'              : u'betriebsschluss',
        u'bettuch'                     : u'betttuch',
        u'bettuches'                   : u'betttuches',
        u'bettuchs'                    : u'betttuchs',
        u'bettücher'                   : u'betttücher',
        u'bettüchern'                  : u'betttüchern',
        u'bevorschuß'                  : u'bevorschuss',
        u'bevorschußt'                 : u'bevorschusst',
        u'bevorschußte'                : u'bevorschusste',
        u'bevorschußten'               : u'bevorschussten',
        u'bevorschußtest'              : u'bevorschusstest',
        u'bevorschußtet'               : u'bevorschusstet',
        u'bevölkerungsüberschuß'       : u'bevölkerungsüberschuss',
        u'beweisbeschluß'              : u'beweisbeschluss',
        u'bewußt'                      : u'bewusst',
        u'bewußte'                     : u'bewusste',
        u'bewußtem'                    : u'bewusstem',
        u'bewußten'                    : u'bewussten',
        u'bewußter'                    : u'bewusster',
        u'bewußtere'                   : u'bewusstere',
        u'bewußterem'                  : u'bewussterem',
        u'bewußteren'                  : u'bewussteren',
        u'bewußterer'                  : u'bewussterer',
        u'bewußteres'                  : u'bewussteres',
        u'bewußtes'                    : u'bewusstes',
        u'bewußteste'                  : u'bewussteste',
        u'bewußtestem'                 : u'bewusstestem',
        u'bewußtesten'                 : u'bewusstesten',
        u'bewußtester'                 : u'bewusstester',
        u'bewußtestes'                 : u'bewusstestes',
        u'bewußtheit'                  : u'bewusstheit',
        u'bewußtlos'                   : u'bewusstlos',
        u'bewußtlose'                  : u'bewusstlose',
        u'bewußtlosem'                 : u'bewusstlosem',
        u'bewußtlosen'                 : u'bewusstlosen',
        u'bewußtloses'                 : u'bewusstloses',
        u'bewußtlosigkeit'             : u'bewusstlosigkeit',
        u'bewußtmachung'               : u'bewusstmachung',
        u'bewußtsein'                  : u'bewusstsein',
        u'bewußtseins'                 : u'bewusstseins',
        u'bewußtseinsbildung'          : u'bewusstseinsbildung',
        u'bewußtseinseinengung'        : u'bewusstseinseinengung',
        u'bewußtseinsentwicklung'      : u'bewusstseinsentwicklung',
        u'bewußtseinsentwicklungen'    : u'bewusstseinsentwicklungen',
        u'bewußtseinserweiternd'       : u'bewusstseinserweiternd',
        u'bewußtseinserweiterung'      : u'bewusstseinserweiterung',
        u'bewußtseinsinhalt'           : u'bewusstseinsinhalt',
        u'bewußtseinslage'             : u'bewusstseinslage',
        u'bewußtseinslenkung'          : u'bewusstseinslenkung',
        u'bewußtseinsschwelle'         : u'bewusstseinsschwelle',
        u'bewußtseinsspaltung'         : u'bewusstseinsspaltung',
        u'bewußtseinsstörung'          : u'bewusstseinsstörung',
        u'bewußtseinstrübung'          : u'bewusstseinstrübung',
        u'bewußtseinsverschiebung'     : u'bewusstseinsverschiebung',
        u'bewußtseinsveränderung'      : u'bewusstseinsveränderung',
        u'bewußtseinsveränderungen'    : u'bewusstseinsveränderungen',
        u'bewußtseinszustand'          : u'bewusstseinszustand',
        u'bewußtwerdung'               : u'bewusstwerdung',
        u'bewäßre'                     : u'bewässre',
        u'bewäßrest'                   : u'bewässrest',
        u'bewäßret'                    : u'bewässret',
        u'bewäßrung'                   : u'bewässrung',
        u'bewäßrungsanlage'            : u'bewässrungsanlage',
        u'bewäßrungsgraben'            : u'bewässrungsgraben',
        u'bewäßrungskanal'             : u'bewässrungskanal',
        u'bewäßrungspumpe'             : u'bewässrungspumpe',
        u'bewäßrungssystem'            : u'bewässrungssystem',
        u'bey'                         : u'bei',
        u'beziehungsstreß'             : u'beziehungsstress',
        u'bezirksausschuß'             : u'bezirksausschuss',
        u'bezuschuß'                   : u'bezuschuss',
        u'bezuschußt'                  : u'bezuschusst',
        u'bezuschußte'                 : u'bezuschusste',
        u'bezuschußten'                : u'bezuschussten',
        u'bezuschußtest'               : u'bezuschusstest',
        u'bezuschußtet'                : u'bezuschusstet',
        u'beßre'                       : u'bessre',
        u'beßrest'                     : u'bessrest',
        u'beßret'                      : u'bessret',
        u'bierbaß'                     : u'bierbass',
        u'bierfaß'                     : u'bierfass',
        u'bierschiß'                   : u'bierschiss',
        u'bildtelephon'                : u'bildtelefon',
        u'bildtelephone'               : u'bildtelefone',
        u'bildtelephonen'              : u'bildtelefonen',
        u'bildtelephons'               : u'bildtelefons',
        u'bildungsprozeß'              : u'bildungsprozess',
        u'bilwiß'                      : u'bilwiss',
        u'binnenschiffahrt'            : u'binnenschifffahrt',
        u'biographen'                  : u'biografen',
        u'biographien'                 : u'biografien',
        u'biographische'               : u'biografische',
        u'bios-paßwort'                : u'bios-passwort',
        u'birkenmeth'                  : u'birkenmet',
        u'birkenruthe'                 : u'birkenrute',
        u'birkenschößling'             : u'birkenschössling',
        u'birkenschößlinge'            : u'birkenschösslinge',
        u'birkenschößlingen'           : u'birkenschösslingen',
        u'birkenschößlings'            : u'birkenschösslings',
        u'birkentheer'                 : u'birkenteer',
        u'biß'                         : u'biss',
        u'bißchen'                     : u'bisschen',
        u'bißchens'                    : u'bisschens',
        u'bißfest'                     : u'bissfest',
        u'bißfeste'                    : u'bissfeste',
        u'bißfestem'                   : u'bissfestem',
        u'bißfesten'                   : u'bissfesten',
        u'bißfester'                   : u'bissfester',
        u'bißfestere'                  : u'bissfestere',
        u'bißfesterem'                 : u'bissfesterem',
        u'bißfesteren'                 : u'bissfesteren',
        u'bißfesterer'                 : u'bissfesterer',
        u'bißfesteres'                 : u'bissfesteres',
        u'bißfestes'                   : u'bissfestes',
        u'bißfesteste'                 : u'bissfesteste',
        u'bißfestestem'                : u'bissfestestem',
        u'bißfestesten'                : u'bissfestesten',
        u'bißfestester'                : u'bissfestester',
        u'bißfestestes'                : u'bissfestestes',
        u'bißspur'                     : u'bissspur',
        u'bißspuren'                   : u'bissspuren',
        u'bißt'                        : u'bisst',
        u'bißverletzung'               : u'bissverletzung',
        u'bißverletzungen'             : u'bissverletzungen',
        u'bißwunde'                    : u'bisswunde',
        u'bißwunden'                   : u'bisswunden',
        u'blattschuß'                  : u'blattschuss',
        u'blaß'                        : u'blass',
        u'blaßblau'                    : u'blassblau',
        u'blaßblauen'                  : u'blassblauen',
        u'blaßen'                      : u'blassen',
        u'blaßgelben'                  : u'blassgelben',
        u'blaßgesicht'                 : u'blassgesicht',
        u'blaßgesichtig'               : u'blassgesichtig',
        u'blaßgrün'                    : u'blassgrün',
        u'blaßgrüner'                  : u'blassgrüner',
        u'blaßgrünliche'               : u'blassgrünliche',
        u'blaßrosa'                    : u'blassrosa',
        u'blaßroten'                   : u'blassroten',
        u'bleßgänse'                   : u'blessgänse',
        u'bleßgänsen'                  : u'blessgänsen',
        u'bleßhuhn'                    : u'blesshuhn',
        u'bleßhuhne'                   : u'blesshuhne',
        u'bleßhuhnes'                  : u'blesshuhnes',
        u'bleßhuhns'                   : u'blesshuhns',
        u'bleßhühner'                  : u'blesshühner',
        u'bleßhühnern'                 : u'blesshühnern',
        u'blutdruckmeßgerät'           : u'blutdruckmessgerät',
        u'blutdruckmeßgeräte'          : u'blutdruckmessgeräte',
        u'blutdruckmeßgeräten'         : u'blutdruckmessgeräten',
        u'blutdruckmeßgerätes'         : u'blutdruckmessgerätes',
        u'blutdruckmeßgeräts'          : u'blutdruckmessgeräts',
        u'bluterguß'                   : u'bluterguss',
        u'bläßbock'                    : u'blässbock',
        u'bläßgans'                    : u'blässgans',
        u'bläßgänse'                   : u'blässgänse',
        u'bläßgänsen'                  : u'blässgänsen',
        u'bläßhuhn'                    : u'blässhuhn',
        u'bläßhuhne'                   : u'blässhuhne',
        u'bläßhuhnes'                  : u'blässhuhnes',
        u'bläßhuhns'                   : u'blässhuhns',
        u'bläßhühner'                  : u'blässhühner',
        u'bläßhühnern'                 : u'blässhühnern',
        u'bläßlich'                    : u'blässlich',
        u'boarding-house'              : u'boardinghouse',
        u'boarding-houses'             : u'boardinghouses',
        u'bodenriß'                    : u'bodenriss',
        u'bodies'                      : u'bodys',
        u'bogenschuß'                  : u'bogenschuss',
        u'bolzenschußapparat'          : u'bolzenschussapparat',
        u'bombenschuß'                 : u'bombenschuss',
        u'bootskompaß'                 : u'bootskompass',
        u'boß'                         : u'boss',
        u'boßle'                       : u'bossle',
        u'boßlest'                     : u'bosslest',
        u'boßlet'                      : u'bosslet',
        u'bramahschloß'                : u'bramahschloss',
        u'brandschutzabschluß'         : u'brandschutzabschluss',
        u'branntweinschenke'           : u'branntweinschänke',
        u'bravur'                      : u'bravour',
        u'bravurös'                    : u'bravourös',
        u'bravuröse'                   : u'bravouröse',
        u'bravurösem'                  : u'bravourösem',
        u'bravurösen'                  : u'bravourösen',
        u'bravuröser'                  : u'bravouröser',
        u'bravurösere'                 : u'bravourösere',
        u'bravuröserem'                : u'bravouröserem',
        u'bravuröseren'                : u'bravouröseren',
        u'bravuröserer'                : u'bravouröserer',
        u'bravuröseres'                : u'bravouröseres',
        u'bravuröses'                  : u'bravouröses',
        u'bravuröseste'                : u'bravouröseste',
        u'bravurösestem'               : u'bravourösestem',
        u'bravurösesten'               : u'bravourösesten',
        u'bravurösester'               : u'bravourösester',
        u'bravurösestes'               : u'bravourösestes',
        u'braß'                        : u'brass',
        u'brechnuß'                    : u'brechnuss',
        u'brennessel'                  : u'brennnessel',
        u'brennesseln'                 : u'brennnesseln',
        u'brennesselstoff'             : u'brennnesselstoff',
        u'brennesselstoffe'            : u'brennnesselstoffe',
        u'brennesselstoffen'           : u'brennnesselstoffen',
        u'brennesselstoffes'           : u'brennnesselstoffes',
        u'brennesselstoffs'            : u'brennnesselstoffs',
        u'brennesseltee'               : u'brennnesseltee',
        u'brennesseltees'              : u'brennnesseltees',
        u'brillenpaß'                  : u'brillenpass',
        u'brod'                        : u'brot',
        u'bronzeguß'                   : u'bronzeguss',
        u'bruderhaß'                   : u'bruderhass',
        u'bruderkuß'                   : u'bruderkuss',
        u'brummbaß'                    : u'brummbass',
        u'brüßler'                     : u'brüssler',
        u'brüßlerin'                   : u'brüsslerin',
        u'brüßlerinnen'                : u'brüsslerinnen',
        u'brüßlern'                    : u'brüsslern',
        u'brüßlers'                    : u'brüsslers',
        u'buchstabenschloß'            : u'buchstabenschloss',
        u'buggies'                     : u'buggys',
        u'bumbaß'                      : u'bumbass',
        u'bundeswahlausschuß'          : u'bundeswahlausschuss',
        u'buntbartschloß'              : u'buntbartschloss',
        u'buschkänguruh'               : u'buschkänguru',
        u'busineß'                     : u'business',
        u'butike'                      : u'boutique',
        u'butiken'                     : u'boutiquen',
        u'butterfaß'                   : u'butterfass',
        u'bußle'                       : u'bussle',
        u'bußlest'                     : u'busslest',
        u'bußlet'                      : u'busslet',
        u'bänderriß'                   : u'bänderriss',
        u'böllerschuß'                 : u'böllerschuss',
        u'börsenschluß'                : u'börsenschluss',
        u'büchsenschuß'                : u'büchsenschuss',
        u'bügelschloß'                 : u'bügelschloss',
        u'bügelverschluß'              : u'bügelverschluss',
        u'büroschluß'                  : u'büroschluss',
        u'cairo'                       : u'kairo',
        u'cairos'                      : u'kairos',
        u'calender'                    : u'kalender',
        u'calendern'                   : u'kalendern',
        u'calenders'                   : u'kalenders',
        u'calfater'                    : u'kalfater',
        u'calfatere'                   : u'kalfatere',
        u'calfaterest'                 : u'kalfaterest',
        u'calfateret'                  : u'kalfateret',
        u'calfatern'                   : u'kalfatern',
        u'calfaternd'                  : u'kalfaternd',
        u'calfaterst'                  : u'kalfaterst',
        u'calfatert'                   : u'kalfatert',
        u'calfaterte'                  : u'kalfaterte',
        u'calfaterten'                 : u'kalfaterten',
        u'calfatertest'                : u'kalfatertest',
        u'calfatertet'                 : u'kalfatertet',
        u'calfatre'                    : u'kalfatre',
        u'calfatrest'                  : u'kalfatrest',
        u'calfatret'                   : u'kalfatret',
        u'calif'                       : u'kalif',
        u'califen'                     : u'kalifen',
        u'californien'                 : u'kalifornien',
        u'californiens'                : u'kaliforniens',
        u'californier'                 : u'kalifornier',
        u'californierin'               : u'kalifornierin',
        u'californierinnen'            : u'kalifornierinnen',
        u'californiern'                : u'kaliforniern',
        u'californiers'                : u'kaliforniers',
        u'californisch'                : u'kalifornisch',
        u'californische'               : u'kalifornische',
        u'californischem'              : u'kalifornischem',
        u'californischen'              : u'kalifornischen',
        u'californischer'              : u'kalifornischer',
        u'californisches'              : u'kalifornisches',
        u'campagne'                    : u'kampagne',
        u'campagnen'                   : u'kampagnen',
        u'campanien'                   : u'kampanien',
        u'campaniens'                  : u'kampaniens',
        u'canada'                      : u'kanada',
        u'canadas'                     : u'kanadas',
        u'canadier'                    : u'kanadier',
        u'canadierin'                  : u'kanadierin',
        u'canadierinnen'               : u'kanadierinnen',
        u'canadiern'                   : u'kanadiern',
        u'canadiers'                   : u'kanadiers',
        u'canadisch'                   : u'kanadisch',
        u'canadische'                  : u'kanadische',
        u'canadischem'                 : u'kanadischem',
        u'canadischen'                 : u'kanadischen',
        u'canadischer'                 : u'kanadischer',
        u'canadisches'                 : u'kanadisches',
        u'canal'                       : u'kanal',
        u'canale'                      : u'kanale',
        u'canales'                     : u'kanales',
        u'canals'                      : u'kanals',
        u'candis'                      : u'kandis',
        u'cannibale'                   : u'kannibale',
        u'cannibalen'                  : u'kannibalen',
        u'cannibalin'                  : u'kannibalin',
        u'cannibalinnen'               : u'kannibalinnen',
        u'cannibalismus'               : u'kannibalismus',
        u'canäle'                      : u'kanäle',
        u'canälen'                     : u'kanälen',
        u'capital'                     : u'kapital',
        u'capitale'                    : u'kapitale',
        u'capitalen'                   : u'kapitalen',
        u'capitalien'                  : u'kapitalien',
        u'capitalismus'                : u'kapitalismus',
        u'capitals'                    : u'kapitals',
        u'caricatur'                   : u'karikatur',
        u'caricaturen'                 : u'karikaturen',
        u'carlsruhe'                   : u'karlsruhe',
        u'carlsruhes'                  : u'karlsruhes',
        u'carracas'                    : u'caracas',
        u'carricatur'                  : u'karikatur',
        u'carricaturen'                : u'karikaturen',
        u'cashewnuß'                   : u'cashewnuss',
        u'casse'                       : u'kasse',
        u'cassel'                      : u'kassel',
        u'cassels'                     : u'kassels',
        u'cassen'                      : u'kassen',
        u'castilien'                   : u'kastilien',
        u'catholicismus'               : u'katholizismus',
        u'catholisch'                  : u'katholisch',
        u'catholische'                 : u'katholische',
        u'catholischem'                : u'katholischem',
        u'catholischen'                : u'katholischen',
        u'catholischer'                : u'katholischer',
        u'catholisches'                : u'katholisches',
        u'catholizismus'               : u'katholizismus',
        u'centralkörper'               : u'zentralkörper',
        u'centren'                     : u'zentren',
        u'centrum'                     : u'zentrum',
        u'centrums'                    : u'zentrums',
        u'chaiberpaß'                  : u'chaiberpass',
        u'christenthum'                : u'christentum',
        u'christenthums'               : u'christentums',
        u'churmark'                    : u'kurmark',
        u'cigarette'                   : u'zigarette',
        u'cigaretten'                  : u'zigaretten',
        u'cigarre'                     : u'zigarre',
        u'cigarren'                    : u'zigarren',
        u'citat'                       : u'zitat',
        u'citate'                      : u'zitate',
        u'citaten'                     : u'zitaten',
        u'citates'                     : u'zitates',
        u'citats'                      : u'zitats',
        u'cities'                      : u'citys',
        u'citrone'                     : u'zitrone',
        u'citronen'                    : u'zitronen',
        u'civil'                       : u'zivil',
        u'civils'                      : u'zivils',
        u'clarinette'                  : u'klarinette',
        u'clarinetten'                 : u'klarinetten',
        u'clarinettist'                : u'klarinettist',
        u'clarinettisten'              : u'klarinettisten',
        u'clarinettistin'              : u'klarinettistin',
        u'clarinettistinnen'           : u'klarinettistinnen',
        u'classicismus'                : u'klassizismus',
        u'clausel'                     : u'klausel',
        u'clauseln'                    : u'klauseln',
        u'clavier'                     : u'klavier',
        u'claviere'                    : u'klaviere',
        u'clavieren'                   : u'klavieren',
        u'clavieres'                   : u'klavieres',
        u'claviers'                    : u'klaviers',
        u'cleverneß'                   : u'cleverness',
        u'cliché'                      : u'klischee',
        u'coblenz'                     : u'koblenz',
        u'cobolentz'                   : u'koblenz',
        u'cocosnuß'                    : u'kokosnuss',
        u'cocosnüsse'                  : u'kokosnüsse',
        u'cocosnüssen'                 : u'kokosnüssen',
        u'collision'                   : u'kollision',
        u'collisionen'                 : u'kollisionen',
        u'colonialware'                : u'kolonialware',
        u'colonialwaren'               : u'kolonialwaren',
        u'colonie'                     : u'kolonie',
        u'colonien'                    : u'kolonien',
        u'colorieren'                  : u'kolorieren',
        u'colour'                      : u'color',
        u'columbianisch'               : u'kolumbianisch',
        u'columbianische'              : u'kolumbianische',
        u'columbianischem'             : u'kolumbianischem',
        u'columbianischen'             : u'kolumbianischen',
        u'columbianischer'             : u'kolumbianischer',
        u'columbianisches'             : u'kolumbianisches',
        u'columbien'                   : u'kolumbien',
        u'columbiens'                  : u'kolumbiens',
        u'combination'                 : u'kombination',
        u'combinationen'               : u'kombinationen',
        u'combinatorik'                : u'kombinatorik',
        u'comma'                       : u'komma',
        u'commas'                      : u'kommas',
        u'commata'                     : u'kommata',
        u'communismus'                 : u'kommunismus',
        u'communist'                   : u'kommunist',
        u'communisten'                 : u'kommunisten',
        u'communistin'                 : u'kommunistin',
        u'communistinnen'              : u'kommunistinnen',
        u'communistisch'               : u'kommunistisch',
        u'communistische'              : u'kommunistische',
        u'communistischem'             : u'kommunistischem',
        u'communistischen'             : u'kommunistischen',
        u'communistischer'             : u'kommunistischer',
        u'communistisches'             : u'kommunistisches',
        u'comparativ'                  : u'komparativ',
        u'comparative'                 : u'komparative',
        u'comparativen'                : u'komparativen',
        u'comparativs'                 : u'komparativs',
        u'competent'                   : u'kompetent',
        u'competente'                  : u'kompetente',
        u'competentem'                 : u'kompetentem',
        u'competenten'                 : u'kompetenten',
        u'competenter'                 : u'kompetenter',
        u'competentere'                : u'kompetentere',
        u'competenterem'               : u'kompetenterem',
        u'competenteren'               : u'kompetenteren',
        u'competenterer'               : u'kompetenterer',
        u'competenteres'               : u'kompetenteres',
        u'competentes'                 : u'kompetentes',
        u'competenteste'               : u'kompetenteste',
        u'competentestem'              : u'kompetentestem',
        u'competentesten'              : u'kompetentesten',
        u'competentester'              : u'kompetentester',
        u'competentestes'              : u'kompetentestes',
        u'competenz'                   : u'kompetenz',
        u'competenzen'                 : u'kompetenzen',
        u'complexität'                 : u'komplexität',
        u'componist'                   : u'komponist',
        u'componisten'                 : u'komponisten',
        u'componistin'                 : u'komponistin',
        u'componistinnen'              : u'komponistinnen',
        u'composition'                 : u'komposition',
        u'compositionen'               : u'kompositionen',
        u'compromisse'                 : u'kompromisse',
        u'compromissen'                : u'kompromissen',
        u'compromisses'                : u'kompromisses',
        u'compromiß'                   : u'kompromiss',
        u'compromißlos'                : u'kompromisslos',
        u'compromißlose'               : u'kompromisslose',
        u'compromißlosem'              : u'kompromisslosem',
        u'compromißlosen'              : u'kompromisslosen',
        u'compromißloser'              : u'kompromissloser',
        u'compromißloses'              : u'kompromissloses',
        u'comödie'                     : u'komödie',
        u'comödien'                    : u'komödien',
        u'concentrirte'                : u'konzentrierte',
        u'concert'                     : u'konzert',
        u'concerte'                    : u'konzerte',
        u'concerten'                   : u'konzerten',
        u'concertes'                   : u'konzertes',
        u'concerts'                    : u'konzerts',
        u'conclusion'                  : u'konklusion',
        u'conclusionen'                : u'konklusionen',
        u'concurrenz'                  : u'konkurrenz',
        u'congresse'                   : u'kongresse',
        u'congressen'                  : u'kongressen',
        u'congresses'                  : u'kongresses',
        u'congreß'                     : u'kongress',
        u'congreßsaal'                 : u'kongresssaal',
        u'congreßsaale'                : u'kongresssaale',
        u'congreßsaales'               : u'kongresssaales',
        u'congreßsaals'                : u'kongresssaals',
        u'congreßsäle'                 : u'kongresssäle',
        u'congreßsälen'                : u'kongresssälen',
        u'congruenz'                   : u'kongruenz',
        u'congruenzen'                 : u'kongruenzen',
        u'conjunctiv'                  : u'konjunktiv',
        u'conjunctive'                 : u'konjunktive',
        u'conjunctiven'                : u'konjunktiven',
        u'conjunctivs'                 : u'konjunktivs',
        u'consequent'                  : u'konsequent',
        u'consequente'                 : u'konsequente',
        u'consequentem'                : u'konsequentem',
        u'consequenten'                : u'konsequenten',
        u'consequenter'                : u'konsequenter',
        u'consequentere'               : u'konsequentere',
        u'consequenterem'              : u'konsequenterem',
        u'consequenteren'              : u'konsequenteren',
        u'consequenterer'              : u'konsequenterer',
        u'consequenteres'              : u'konsequenteres',
        u'consequentes'                : u'konsequentes',
        u'consequenteste'              : u'konsequenteste',
        u'consequentestem'             : u'konsequentestem',
        u'consequentesten'             : u'konsequentesten',
        u'consequentester'             : u'konsequentester',
        u'consequentestes'             : u'konsequentestes',
        u'consequenz'                  : u'konsequenz',
        u'consequenzen'                : u'konsequenzen',
        u'consonant'                   : u'konsonant',
        u'consonanten'                 : u'konsonanten',
        u'consonantenhäufung'          : u'konsonantenhäufung',
        u'consonanz'                   : u'konsonanz',
        u'consonanzen'                 : u'konsonanzen',
        u'constante'                   : u'konstante',
        u'constanten'                  : u'konstanten',
        u'constanz'                    : u'konstanz',
        u'construction'                : u'konstruktion',
        u'constructionen'              : u'konstruktionen',
        u'contrabasses'                : u'kontrabasses',
        u'contrabaß'                   : u'kontrabass',
        u'contrabässe'                 : u'kontrabässe',
        u'contrabässen'                : u'kontrabässen',
        u'contrapunct'                 : u'kontrapunkt',
        u'contrapuncte'                : u'kontrapunkte',
        u'contrapuncten'               : u'kontrapunkten',
        u'contrapunctes'               : u'kontrapunktes',
        u'contrapuncts'                : u'kontrapunkts',
        u'convention'                  : u'konvention',
        u'conventionen'                : u'konventionen',
        u'conversation'                : u'konversation',
        u'conversationen'              : u'konversationen',
        u'coolneß'                     : u'coolness',
        u'cooperation'                 : u'kooperation',
        u'cooperationen'               : u'kooperationen',
        u'copie'                       : u'kopie',
        u'copien'                      : u'kopien',
        u'cordilleren'                 : u'kordilleren',
        u'cormoran'                    : u'kormoran',
        u'cormorane'                   : u'kormorane',
        u'cormoranen'                  : u'kormoranen',
        u'cormorans'                   : u'kormorans',
        u'corollar'                    : u'korollar',
        u'corollare'                   : u'korollare',
        u'corollaren'                  : u'korollaren',
        u'corollars'                   : u'korollars',
        u'correctur'                   : u'korrektur',
        u'correcturen'                 : u'korrekturen',
        u'costaricaner'                : u'costa-ricaner',
        u'costaricanerin'              : u'costa-ricanerin',
        u'costaricanerinnen'           : u'costa-ricanerinnen',
        u'costaricanern'               : u'costa-ricanern',
        u'costaricaners'               : u'costa-ricaners',
        u'costaricanisch'              : u'costa-ricanisch',
        u'costaricanische'             : u'costa-ricanische',
        u'costaricanischem'            : u'costa-ricanischem',
        u'costaricanischen'            : u'costa-ricanischen',
        u'costaricanischer'            : u'costa-ricanischer',
        u'costaricanisches'            : u'costa-ricanisches',
        u'counteß'                     : u'countess',
        u'cretin'                      : u'kretin',
        u'cretins'                     : u'kretins',
        u'creutz'                      : u'kreuz',
        u'creutze'                     : u'kreuze',
        u'creutzen'                    : u'kreuzen',
        u'creutzes'                    : u'kreuzes',
        u'creutzig'                    : u'kreuzig',
        u'creutzige'                   : u'kreuzige',
        u'creutzigen'                  : u'kreuzigen',
        u'creutzigend'                 : u'kreuzigend',
        u'creutzigest'                 : u'kreuzigest',
        u'creutziget'                  : u'kreuziget',
        u'creutzigst'                  : u'kreuzigst',
        u'creutzigt'                   : u'kreuzigt',
        u'creutzigte'                  : u'kreuzigte',
        u'creutzigten'                 : u'kreuzigten',
        u'creutzigtest'                : u'kreuzigtest',
        u'creutzigtet'                 : u'kreuzigtet',
        u'creutzigung'                 : u'kreuzigung',
        u'creutzigungen'               : u'kreuzigungen',
        u'crocodil'                    : u'krokodil',
        u'crocodile'                   : u'krokodile',
        u'crocodilen'                  : u'krokodilen',
        u'crocodils'                   : u'krokodils',
        u'crystallinischer'            : u'kristallinischer',
        u'cuba'                        : u'kuba',
        u'cubaner'                     : u'kubaner',
        u'cubanerin'                   : u'kubanerin',
        u'cubanerinnen'                : u'kubanerinnen',
        u'cubanern'                    : u'kubanern',
        u'cubaners'                    : u'kubaners',
        u'cubanisch'                   : u'kubanisch',
        u'cubanische'                  : u'kubanische',
        u'cubanischem'                 : u'kubanischem',
        u'cubanischen'                 : u'kubanischen',
        u'cubanischer'                 : u'kubanischer',
        u'cubanisches'                 : u'kubanisches',
        u'cubas'                       : u'kubas',
        u'cultur'                      : u'kultur',
        u'culturen'                    : u'kulturen',
        u'cursiv'                      : u'kursiv',
        u'cursive'                     : u'kursive',
        u'cursivem'                    : u'kursivem',
        u'cursiven'                    : u'kursiven',
        u'cursiver'                    : u'kursiver',
        u'cursives'                    : u'kursives',
        u'curvatur'                    : u'kurvatur',
        u'cuscus'                      : u'kuskus',
        u'czar'                        : u'zar',
        u'czaren'                      : u'zaren',
        u'cöln'                        : u'köln',
        u'cölns'                       : u'kölns',
        u'cöthen'                      : u'köthen',
        u'cöthens'                     : u'köthens',
        u'dachgeschoß'                 : u'dachgeschoss',
        u'dachgeschoßwohnung'          : u'dachgeschosswohnung',
        u'dachgeschoßwohnungen'        : u'dachgeschosswohnungen',
        u'daddies'                     : u'daddys',
        u'dahinfloß'                   : u'dahinfloss',
        u'dahinfloßt'                  : u'dahinflosst',
        u'dalaßt'                      : u'dalasst',
        u'daläßt'                      : u'dalässt',
        u'dammriß'                     : u'dammriss',
        u'dampfroß'                    : u'dampfross',
        u'dampfschiffahrt'             : u'dampfschifffahrt',
        u'dampfschiffahrten'           : u'dampfschifffahrten',
        u'dampfschiffahrtsgesellschaft' : u'dampfschifffahrtsgesellschaft',
        u'dampfschiffahrtsgesellschaften' : u'dampfschifffahrtsgesellschaften',
        u'danaidenfaß'                 : u'danaidenfass',
        u'danebenschoß'                : u'danebenschoss',
        u'danebenschoßt'               : u'danebenschosst',
        u'dargethan'                   : u'dargetan',
        u'darmriß'                     : u'darmriss',
        u'darmverschluß'               : u'darmverschluss',
        u'darthue'                     : u'dartue',
        u'darthuen'                    : u'dartuen',
        u'darthuend'                   : u'dartuend',
        u'darthuest'                   : u'dartuest',
        u'darthuet'                    : u'dartuet',
        u'darthun'                     : u'dartun',
        u'darthust'                    : u'dartust',
        u'darthut'                     : u'dartut',
        u'darzuthun'                   : u'darzutun',
        u'dasz'                        : u'dass',
        u'datenflußdiagramm'           : u'datenflussdiagramm',
        u'datenflußdiagramme'          : u'datenflussdiagramme',
        u'datenflußdiagrammen'         : u'datenflussdiagrammen',
        u'datenflußdiagramms'          : u'datenflussdiagramms',
        u'davonschoß'                  : u'davonschoss',
        u'davonschoßt'                 : u'davonschosst',
        u'davysch'                     : u'davysch',
        u'daß'                         : u'dass',
        u'daß-satz'                    : u'dass-satz',
        u'daß-satze'                   : u'dass-satze',
        u'daß-satzes'                  : u'dass-satzes',
        u'daß-sätze'                   : u'dass-sätze',
        u'daß-sätzen'                  : u'dass-sätzen',
        u'daßsatz'                     : u'dasssatz',
        u'daßsatzes'                   : u'dasssatzes',
        u'daßsätze'                    : u'dasssätze',
        u'daßsätzen'                   : u'dasssätzen',
        u'december'                    : u'dezember',
        u'decembern'                   : u'dezembern',
        u'decembers'                   : u'dezembers',
        u'deflationsprozeß'            : u'deflationsprozess',
        u'delikateßgeschäft'           : u'delikatessgeschäft',
        u'delikateßgurke'              : u'delikatessgurke',
        u'delikateßhering'             : u'delikatesshering',
        u'delikateßheringe'            : u'delikatessheringe',
        u'delikateßheringen'           : u'delikatessheringen',
        u'delikateßherings'            : u'delikatessherings',
        u'delikateßladen'              : u'delikatessladen',
        u'delikateßmajonäse'           : u'delikatessmajonäse',
        u'delikateßmayonnaise'         : u'delikatessmayonnaise',
        u'delikateßsenf'               : u'delikatesssenf',
        u'delikateßware'               : u'delikatessware',
        u'demographische'              : u'demografische',
        u'dengelamboß'                 : u'dengelamboss',
        u'denkprozeß'                  : u'denkprozess',
        u'deplaciert'                  : u'deplatziert',
        u'deplacierte'                 : u'deplatzierte',
        u'deplaciertem'                : u'deplatziertem',
        u'deplacierten'                : u'deplatzierten',
        u'deplacierter'                : u'deplatzierter',
        u'deplaciertere'               : u'deplatziertere',
        u'deplacierterem'              : u'deplatzierterem',
        u'deplacierteren'              : u'deplatzierteren',
        u'deplacierterer'              : u'deplatzierterer',
        u'deplacierteres'              : u'deplatzierteres',
        u'deplaciertes'                : u'deplatziertes',
        u'deplacierteste'              : u'deplatzierteste',
        u'deplaciertestem'             : u'deplatziertestem',
        u'deplaciertesten'             : u'deplatziertesten',
        u'deplaciertester'             : u'deplatziertester',
        u'deplaciertestes'             : u'deplatziertestes',
        u'deplaziert'                  : u'deplatziert',
        u'deplazierte'                 : u'deplatzierte',
        u'deplaziertem'                : u'deplatziertem',
        u'deplazierten'                : u'deplatzierten',
        u'deplazierter'                : u'deplatzierter',
        u'deplaziertere'               : u'deplatziertere',
        u'deplazierterem'              : u'deplatzierterem',
        u'deplazierteren'              : u'deplatzierteren',
        u'deplazierterer'              : u'deplatzierterer',
        u'deplazierteres'              : u'deplatzierteres',
        u'deplaziertes'                : u'deplatziertes',
        u'deplazierteste'              : u'deplatzierteste',
        u'deplaziertestem'             : u'deplatziertestem',
        u'deplaziertesten'             : u'deplatziertesten',
        u'deplaziertester'             : u'deplatziertester',
        u'deplaziertestes'             : u'deplatziertestes',
        u'deputiertenkongreß'          : u'deputiertenkongress',
        u'desachalandiren'             : u'desachalandieren',
        u'detailbeseßner'              : u'detailbesessner',
        u'detailbeseßnere'             : u'detailbesessnere',
        u'detailbeseßnerem'            : u'detailbesessnerem',
        u'detailbeseßneren'            : u'detailbesessneren',
        u'detailbeseßnerer'            : u'detailbesessnerer',
        u'detailbeseßneres'            : u'detailbesessneres',
        u'deutschenhaß'                : u'deutschenhass',
        u'deßhalb'                     : u'deshalb',
        u'dhau'                        : u'dau',
        u'dhaus'                       : u'daus',
        u'dichterroß'                  : u'dichterross',
        u'dienstbeflißner'             : u'dienstbeflissner',
        u'dienstbeflißnere'            : u'dienstbeflissnere',
        u'dienstbeflißnerem'           : u'dienstbeflissnerem',
        u'dienstbeflißneren'           : u'dienstbeflissneren',
        u'dienstbeflißnerer'           : u'dienstbeflissnerer',
        u'dienstbeflißneres'           : u'dienstbeflissneres',
        u'dienstschluß'                : u'dienstschluss',
        u'diethyläther'                : u'diethylether',
        u'diethyläthers'               : u'diethylethers',
        u'differential-regulator'      : u'differenzial-regulator',
        u'differentialrechnung'        : u'differenzialrechnung',
        u'differenzierungsprozeß'      : u'differenzierungsprozess',
        u'diplomatenpaß'               : u'diplomatenpass',
        u'direktschuß'                 : u'direktschuss',
        u'disstreß'                    : u'disstress',
        u'distanzschuß'                : u'distanzschuss',
        u'ditto'                       : u'dito',
        u'diß'                         : u'diss',
        u'dißt'                        : u'disst',
        u'dißte'                       : u'disste',
        u'dißten'                      : u'dissten',
        u'dißtest'                     : u'disstest',
        u'dißtet'                      : u'disstet',
        u'diättip'                     : u'diättipp',
        u'diättips'                    : u'diättipps',
        u'doctor'                      : u'doktor',
        u'donaudampfschiffahrtsgesellschaft' : u'donaudampfschifffahrtsgesellschaft',
        u'donaudampfschiffahrtsgesellschaftskapitän' : u'donaudampfschifffahrtsgesellschaftskapitän',
        u'donaudampfschiffahrtsgesellschaftskapitäne' : u'donaudampfschifffahrtsgesellschaftskapitäne',
        u'donaudampfschiffahrtsgesellschaftskapitänen' : u'donaudampfschifffahrtsgesellschaftskapitänen',
        u'donaudampfschiffahrtsgesellschaftskapitäns' : u'donaudampfschifffahrtsgesellschaftskapitäns',
        u'doppelpaß'                   : u'doppelpass',
        u'doppelpaßspiel'              : u'doppelpassspiel',
        u'drankfaß'                    : u'drankfass',
        u'drauflosschoß'               : u'drauflosschoss',
        u'drauflosschoßt'              : u'drauflosschosst',
        u'drehschuß'                   : u'drehschuss',
        u'dreipaß'                     : u'dreipass',
        u'drey'                        : u'drei',
        u'dreye'                       : u'dreie',
        u'dreyen'                      : u'dreien',
        u'dreyer'                      : u'dreier',
        u'dreß'                        : u'dress',
        u'dreßt'                       : u'dresst',
        u'dreßte'                      : u'dresste',
        u'dreßten'                     : u'dressten',
        u'dreßtest'                    : u'dresstest',
        u'dreßtet'                     : u'dresstet',
        u'drittklaßler'                : u'drittklassler',
        u'drittklaßlerin'              : u'drittklasslerin',
        u'drittkläßler'                : u'drittklässler',
        u'drittkläßlerin'              : u'drittklässlerin',
        u'drittkläßlern'               : u'drittklässlern',
        u'drittkläßlers'               : u'drittklässlers',
        u'drittplazierte'              : u'drittplatzierte',
        u'drittplaziertem'             : u'drittplatziertem',
        u'drittplazierten'             : u'drittplatzierten',
        u'drittplazierter'             : u'drittplatzierter',
        u'drogeneinfluß'               : u'drogeneinfluss',
        u'drogenmißbrauch'             : u'drogenmissbrauch',
        u'droßle'                      : u'drossle',
        u'droßlest'                    : u'drosslest',
        u'droßlet'                     : u'drosslet',
        u'droßlung'                    : u'drosslung',
        u'dum-dum-geschoß'             : u'dum-dum-geschoss',
        u'dumdumgeschoß'               : u'dumdumgeschoss',
        u'dummies'                     : u'dummys',
        u'durchbiß'                    : u'durchbiss',
        u'durchbißt'                   : u'durchbisst',
        u'durcheinandereßt'            : u'durcheinanderesst',
        u'durcheinanderißt'            : u'durcheinanderisst',
        u'durchfloß'                   : u'durchfloss',
        u'durchfloßt'                  : u'durchflosst',
        u'durchfluß'                   : u'durchfluss',
        u'durchflußgeschwindigkeit'    : u'durchflussgeschwindigkeit',
        u'durchflußmenge'              : u'durchflussmenge',
        u'durchflußmengen'             : u'durchflussmengen',
        u'durchflußmesser'             : u'durchflussmesser',
        u'durchfreßner'                : u'durchfressner',
        u'durchfreßnere'               : u'durchfressnere',
        u'durchfreßnerem'              : u'durchfressnerem',
        u'durchfreßneren'              : u'durchfressneren',
        u'durchfreßnerer'              : u'durchfressnerer',
        u'durchfreßneres'              : u'durchfressneres',
        u'durchfreßt'                  : u'durchfresst',
        u'durchfrißt'                  : u'durchfrisst',
        u'durchgepreßt'                : u'durchgepresst',
        u'durchgeseßner'               : u'durchgesessner',
        u'durchgeseßnere'              : u'durchgesessnere',
        u'durchgeseßnerem'             : u'durchgesessnerem',
        u'durchgeseßneren'             : u'durchgesessneren',
        u'durchgeseßnerer'             : u'durchgesessnerer',
        u'durchgeseßneres'             : u'durchgesessneres',
        u'durchgoß'                    : u'durchgoss',
        u'durchgoßt'                   : u'durchgosst',
        u'durchißt'                    : u'durchisst',
        u'durchlaß'                    : u'durchlass',
        u'durchlaßt'                   : u'durchlasst',
        u'durchläßt'                   : u'durchlässt',
        u'durchnumerieren'             : u'durchnummerieren',
        u'durchnäß'                    : u'durchnäss',
        u'durchnäßt'                   : u'durchnässt',
        u'durchnäßte'                  : u'durchnässte',
        u'durchnäßten'                 : u'durchnässten',
        u'durchnäßter'                 : u'durchnässter',
        u'durchnäßtest'                : u'durchnässtest',
        u'durchnäßtet'                 : u'durchnässtet',
        u'durchschuß'                  : u'durchschuss',
        u'durchtelephonieren'          : u'durchtelefonieren',
        u'dußlig'                      : u'dusslig',
        u'dußlige'                     : u'dusslige',
        u'dußligem'                    : u'dussligem',
        u'dußligen'                    : u'dussligen',
        u'dußliger'                    : u'dussliger',
        u'dußligere'                   : u'dussligere',
        u'dußligerem'                  : u'dussligerem',
        u'dußligeren'                  : u'dussligeren',
        u'dußligerer'                  : u'dussligerer',
        u'dußligeres'                  : u'dussligeres',
        u'dußliges'                    : u'dussliges',
        u'dußligste'                   : u'dussligste',
        u'dußligstem'                  : u'dussligstem',
        u'dußligsten'                  : u'dussligsten',
        u'dußligster'                  : u'dussligster',
        u'dußligstes'                  : u'dussligstes',
        u'dünnschiß'                   : u'dünnschiss',
        u'e-baß'                       : u'e-bass',
        u'egypten'                     : u'ägypten',
        u'egyptens'                    : u'ägyptens',
        u'eheprozeß'                   : u'eheprozess',
        u'ehescheidungsprozeß'         : u'ehescheidungsprozess',
        u'ehprozeß'                    : u'ehprozess',
        u'ehrprozeß'                   : u'ehrprozess',
        u'eigenthum'                   : u'eigentum',
        u'eigenthume'                  : u'eigentume',
        u'eigenthumen'                 : u'eigentumen',
        u'eigenthumes'                 : u'eigentumes',
        u'eigenthums'                  : u'eigentums',
        u'eigenthümer'                 : u'eigentümer',
        u'eigenthümern'                : u'eigentümern',
        u'eigenthümers'                : u'eigentümers',
        u'einbleuen'                   : u'einbläuen',
        u'einfangprozeß'               : u'einfangprozess',
        u'einfluß'                     : u'einfluss',
        u'einflußbereich'              : u'einflussbereich',
        u'einflußbereiche'             : u'einflussbereiche',
        u'einflußbereichen'            : u'einflussbereichen',
        u'einflußbereiches'            : u'einflussbereiches',
        u'einflußbereichs'             : u'einflussbereichs',
        u'einflußreich'                : u'einflussreich',
        u'einflußreiche'               : u'einflussreiche',
        u'einflußreichem'              : u'einflussreichem',
        u'einflußreichen'              : u'einflussreichen',
        u'einflußreicher'              : u'einflussreicher',
        u'einflußreichere'             : u'einflussreichere',
        u'einflußreicherem'            : u'einflussreicherem',
        u'einflußreicheren'            : u'einflussreicheren',
        u'einflußreicherer'            : u'einflussreicherer',
        u'einflußreicheres'            : u'einflussreicheres',
        u'einflußreiches'              : u'einflussreiches',
        u'einflußreichste'             : u'einflussreichste',
        u'einflußreichstem'            : u'einflussreichstem',
        u'einflußreichsten'            : u'einflussreichsten',
        u'einflußreichster'            : u'einflussreichster',
        u'einflußreichstes'            : u'einflussreichstes',
        u'einflußsphäre'               : u'einflusssphäre',
        u'eingefaßte'                  : u'eingefasste',
        u'eingekremt'                  : u'eingecremt',
        u'einkrem'                     : u'eincrem',
        u'einkreme'                    : u'eincreme',
        u'einkremen'                   : u'eincremen',
        u'einkremend'                  : u'eincremend',
        u'einkremest'                  : u'eincremest',
        u'einkremet'                   : u'eincremet',
        u'einkremst'                   : u'eincremst',
        u'einkremt'                    : u'eincremt',
        u'einkremte'                   : u'eincremte',
        u'einkremten'                  : u'eincremten',
        u'einkremtest'                 : u'eincremtest',
        u'einkremtet'                  : u'eincremtet',
        u'einlaß'                      : u'einlass',
        u'einlaßt'                     : u'einlasst',
        u'einläßt'                     : u'einlässt',
        u'einnäßte'                    : u'einnässte',
        u'einnäßten'                   : u'einnässten',
        u'einnäßtest'                  : u'einnässtest',
        u'einnäßtet'                   : u'einnässtet',
        u'einschiß'                    : u'einschiss',
        u'einschißt'                   : u'einschisst',
        u'einschloß'                   : u'einschloss',
        u'einschluß'                   : u'einschluss',
        u'einschmelzungsprozeß'        : u'einschmelzungsprozess',
        u'einschuß'                    : u'einschuss',
        u'einschußloch'                : u'einschussloch',
        u'einschußloche'               : u'einschussloche',
        u'einschußloches'              : u'einschussloches',
        u'einschußlochs'               : u'einschusslochs',
        u'einschußlöcher'              : u'einschusslöcher',
        u'einschußlöchern'             : u'einschusslöchern',
        u'einsendeschluß'              : u'einsendeschluss',
        u'einzukremen'                 : u'einzucremen',
        u'eisschnellauf'               : u'eisschnelllauf',
        u'eisschnellaufe'              : u'eisschnelllaufe',
        u'eisschnellaufen'             : u'eisschnelllaufen',
        u'eisschnellaufens'            : u'eisschnelllaufens',
        u'eisschnellaufes'             : u'eisschnelllaufes',
        u'eisschnellaufs'              : u'eisschnelllaufs',
        u'eisschnelläufer'             : u'eisschnellläufer',
        u'eisschnelläuferin'           : u'eisschnellläuferin',
        u'eisschnelläuferinnen'        : u'eisschnellläuferinnen',
        u'eisschnelläufern'            : u'eisschnellläufern',
        u'eisschnelläufers'            : u'eisschnellläufers',
        u'electro'                     : u'elektro',
        u'elephant'                    : u'elefant',
        u'elephanten'                  : u'elefanten',
        u'elsaß'                       : u'elsass',
        u'elsaß-lothringen'            : u'elsass-lothringen',
        u'elsaß-lothringens'           : u'elsass-lothringens',
        u'emanzipationsprozeß'         : u'emanzipationsprozess',
        u'emporriß'                    : u'emporriss',
        u'emporrißt'                   : u'emporrisst',
        u'energiepaß'                  : u'energiepass',
        u'engpaß'                      : u'engpass',
        u'entfeßle'                    : u'entfessle',
        u'entfeßlest'                  : u'entfesslest',
        u'entfeßlet'                   : u'entfesslet',
        u'entfloß'                     : u'entfloss',
        u'entfloßt'                    : u'entflosst',
        u'entlaß'                      : u'entlass',
        u'entlaßt'                     : u'entlasst',
        u'entläßt'                     : u'entlässt',
        u'entriß'                      : u'entriss',
        u'entrißt'                     : u'entrisst',
        u'entschloß'                   : u'entschloss',
        u'entschloßt'                  : u'entschlosst',
        u'entschluß'                   : u'entschluss',
        u'entschlußkraft'              : u'entschlusskraft',
        u'entschlüßle'                 : u'entschlüssle',
        u'entschlüßlest'               : u'entschlüsslest',
        u'entschlüßlet'                : u'entschlüsslet',
        u'entspannungsprozeß'          : u'entspannungsprozess',
        u'entsproß'                    : u'entspross',
        u'entsproßt'                   : u'entsprosst',
        u'entstehungsprozeß'           : u'entstehungsprozess',
        u'entweder-oder'               : u'entweder-oder',
        u'entwickelungsprozeß'         : u'entwickelungsprozess',
        u'entwicklungsprozeß'          : u'entwicklungsprozess',
        u'entwäßre'                    : u'entwässre',
        u'entzweiriß'                  : u'entzweiriss',
        u'entzweirißt'                 : u'entzweirisst',
        u'erblaß'                      : u'erblass',
        u'erblaßt'                     : u'erblasst',
        u'erblaßte'                    : u'erblasste',
        u'erblaßten'                   : u'erblassten',
        u'erblaßtest'                  : u'erblasstest',
        u'erblaßtet'                   : u'erblasstet',
        u'erdgeschoßwohnung'           : u'erdgeschosswohnung',
        u'erdgeschoßwohnungen'         : u'erdgeschosswohnungen',
        u'erdnuß'                      : u'erdnuss',
        u'erdnußbutter'                : u'erdnussbutter',
        u'erdnußflip'                  : u'erdnussflip',
        u'erdnußflips'                 : u'erdnussflips',
        u'erdroßle'                    : u'erdrossle',
        u'erdroßlest'                  : u'erdrosslest',
        u'erdroßlet'                   : u'erdrosslet',
        u'ereigniß'                    : u'ereignis',
        u'erfaß'                       : u'erfass',
        u'erfaßbarer'                  : u'erfassbarer',
        u'erfaßt'                      : u'erfasst',
        u'erfaßte'                     : u'erfasste',
        u'erfaßten'                    : u'erfassten',
        u'erfaßtest'                   : u'erfasstest',
        u'erfaßtet'                    : u'erfasstet',
        u'erforderniß'                 : u'erfordernis',
        u'ergoß'                       : u'ergoss',
        u'ergoßt'                      : u'ergosst',
        u'erkenntnisprozeß'            : u'erkenntnisprozess',
        u'erkenntniß'                  : u'erkenntnis',
        u'erlaß'                       : u'erlass',
        u'erlaßt'                      : u'erlasst',
        u'erläßt'                      : u'erlässt',
        u'ermeßt'                      : u'ermesst',
        u'ermiß'                       : u'ermiss',
        u'ermißt'                      : u'ermisst',
        u'erpreß'                      : u'erpress',
        u'erpreßt'                     : u'erpresst',
        u'erpreßte'                    : u'erpresste',
        u'erpreßten'                   : u'erpressten',
        u'erpreßtest'                  : u'erpresstest',
        u'erpreßtet'                   : u'erpresstet',
        u'errathen'                    : u'erraten',
        u'erschloß'                    : u'erschloss',
        u'erschloßt'                   : u'erschlosst',
        u'erschoß'                     : u'erschoss',
        u'erschoßt'                    : u'erschosst',
        u'ersproß'                     : u'erspross',
        u'ersproßt'                    : u'ersprosst',
        u'erstklaßler'                 : u'erstklassler',
        u'erstklaßlerin'               : u'erstklasslerin',
        u'erstklaßlerinnen'            : u'erstklasslerinnen',
        u'erstklaßlern'                : u'erstklasslern',
        u'erstklaßlers'                : u'erstklasslers',
        u'erstkläßler'                 : u'erstklässler',
        u'erstkläßlerin'               : u'erstklässlerin',
        u'erstkläßlerinnen'            : u'erstklässlerinnen',
        u'erstkläßlern'                : u'erstklässlern',
        u'erstkläßlers'                : u'erstklässlers',
        u'erstplazierte'               : u'erstplatzierte',
        u'erstplaziertem'              : u'erstplatziertem',
        u'erstplazierten'              : u'erstplatzierten',
        u'erstplazierter'              : u'erstplatzierter',
        u'erziehungsprozeß'            : u'erziehungsprozess',
        u'eur'                         : u'euro',
        u'existentiell'                : u'existenziell',
        u'expreß'                      : u'express',
        u'expreßzug'                   : u'expresszug',
        u'expreßzuge'                  : u'expresszuge',
        u'expreßzuges'                 : u'expresszuges',
        u'expreßzugs'                  : u'expresszugs',
        u'expreßzüge'                  : u'expresszüge',
        u'expreßzügen'                 : u'expresszügen',
        u'exzeß'                       : u'exzess',
        u'eßbar'                       : u'essbar',
        u'eßbare'                      : u'essbare',
        u'eßbarem'                     : u'essbarem',
        u'eßbaren'                     : u'essbaren',
        u'eßbarer'                     : u'essbarer',
        u'eßbarerem'                   : u'essbarerem',
        u'eßbareren'                   : u'essbareren',
        u'eßbarerer'                   : u'essbarerer',
        u'eßbareres'                   : u'essbareres',
        u'eßbares'                     : u'essbares',
        u'eßbarste'                    : u'essbarste',
        u'eßbarstem'                   : u'essbarstem',
        u'eßbarsten'                   : u'essbarsten',
        u'eßbarster'                   : u'essbarster',
        u'eßbarstes'                   : u'essbarstes',
        u'eßbesteck'                   : u'essbesteck',
        u'eßbestecke'                  : u'essbestecke',
        u'eßbestecken'                 : u'essbestecken',
        u'eßbesteckes'                 : u'essbesteckes',
        u'eßbestecks'                  : u'essbestecks',
        u'eßecke'                      : u'essecke',
        u'eßecken'                     : u'essecken',
        u'eßgeschirr'                  : u'essgeschirr',
        u'eßgewohnheit'                : u'essgewohnheit',
        u'eßkastanie'                  : u'esskastanie',
        u'eßkultur'                    : u'esskultur',
        u'eßkulturen'                  : u'esskulturen',
        u'eßküche'                     : u'essküche',
        u'eßlust'                      : u'esslust',
        u'eßlöffel'                    : u'esslöffel',
        u'eßlöffeln'                   : u'esslöffeln',
        u'eßlöffels'                   : u'esslöffels',
        u'eßlöffelweise'               : u'esslöffelweise',
        u'eßlöffelweisem'              : u'esslöffelweisem',
        u'eßlöffelweisen'              : u'esslöffelweisen',
        u'eßlöffelweiser'              : u'esslöffelweiser',
        u'eßlöffelweises'              : u'esslöffelweises',
        u'eßpapier'                    : u'esspapier',
        u'eßpapiers'                   : u'esspapiers',
        u'eßstube'                     : u'essstube',
        u'eßstäbchen'                  : u'essstäbchen',
        u'eßstörung'                   : u'essstörung',
        u'eßstörungen'                 : u'essstörungen',
        u'eßsucht'                     : u'esssucht',
        u'eßt'                         : u'esst',
        u'eßtisch'                     : u'esstisch',
        u'eßtische'                    : u'esstische',
        u'eßtischen'                   : u'esstischen',
        u'eßtisches'                   : u'esstisches',
        u'eßtischs'                    : u'esstischs',
        u'eßzimmer'                    : u'esszimmer',
        u'eßzimmern'                   : u'esszimmern',
        u'eßzimmers'                   : u'esszimmers',
        u'eßzimmerstuhl'               : u'esszimmerstuhl',
        u'eßzimmerstuhle'              : u'esszimmerstuhle',
        u'eßzimmerstuhles'             : u'esszimmerstuhles',
        u'eßzimmerstuhls'              : u'esszimmerstuhls',
        u'eßzimmertür'                 : u'esszimmertür',
        u'fabrikationsprozeß'          : u'fabrikationsprozess',
        u'fahrniß'                     : u'fahrnis',
        u'fahrradschloß'               : u'fahrradschloss',
        u'fairneß'                     : u'fairness',
        u'fangschuß'                   : u'fangschuss',
        u'farbphoto'                   : u'farbfoto',
        u'farbphotos'                  : u'farbfotos',
        u'fassette'                    : u'facette',
        u'fassetten'                   : u'facetten',
        u'fassettenartig'              : u'facettenartig',
        u'fassettenartige'             : u'facettenartige',
        u'fassettenartigem'            : u'facettenartigem',
        u'fassettenartigen'            : u'facettenartigen',
        u'fassettenartiger'            : u'facettenartiger',
        u'fassettenartiges'            : u'facettenartiges',
        u'fassettenauge'               : u'facettenauge',
        u'fassettenaugen'              : u'facettenaugen',
        u'fassettenauges'              : u'facettenauges',
        u'fassettenreich'              : u'facettenreich',
        u'fassettenreiche'             : u'facettenreiche',
        u'fassettenreichem'            : u'facettenreichem',
        u'fassettenreichen'            : u'facettenreichen',
        u'fassettenreicher'            : u'facettenreicher',
        u'fassettenreichere'           : u'facettenreichere',
        u'fassettenreicherem'          : u'facettenreicherem',
        u'fassettenreicheren'          : u'facettenreicheren',
        u'fassettenreicherer'          : u'facettenreicherer',
        u'fassettenreicheres'          : u'facettenreicheres',
        u'fassettenreiches'            : u'facettenreiches',
        u'fassettenreichste'           : u'facettenreichste',
        u'fassettenreichstem'          : u'facettenreichstem',
        u'fassettenreichsten'          : u'facettenreichsten',
        u'fassettenreichster'          : u'facettenreichster',
        u'fassettenreichstes'          : u'facettenreichstes',
        u'fassettenreichtum'           : u'facettenreichtum',
        u'fassettenreichtums'          : u'facettenreichtums',
        u'faustschluß'                 : u'faustschluss',
        u'faß'                         : u'fass',
        u'faßbar'                      : u'fassbar',
        u'faßbare'                     : u'fassbare',
        u'faßbarem'                    : u'fassbarem',
        u'faßbaren'                    : u'fassbaren',
        u'faßbarer'                    : u'fassbarer',
        u'faßbarere'                   : u'fassbarere',
        u'faßbarerem'                  : u'fassbarerem',
        u'faßbareren'                  : u'fassbareren',
        u'faßbarerer'                  : u'fassbarerer',
        u'faßbareres'                  : u'fassbareres',
        u'faßbares'                    : u'fassbares',
        u'faßbarste'                   : u'fassbarste',
        u'faßbarstem'                  : u'fassbarstem',
        u'faßbarsten'                  : u'fassbarsten',
        u'faßbarster'                  : u'fassbarster',
        u'faßbarstes'                  : u'fassbarstes',
        u'faßbier'                     : u'fassbier',
        u'faßbiere'                    : u'fassbiere',
        u'faßbieren'                   : u'fassbieren',
        u'faßbieres'                   : u'fassbieres',
        u'faßbiers'                    : u'fassbiers',
        u'faßbinder'                   : u'fassbinder',
        u'faßbindern'                  : u'fassbindern',
        u'faßbinders'                  : u'fassbinders',
        u'faßboden'                    : u'fassboden',
        u'faßbombe'                    : u'fassbombe',
        u'faßbomben'                   : u'fassbomben',
        u'faßdaube'                    : u'fassdaube',
        u'faßdauben'                   : u'fassdauben',
        u'faßgeschmack'                : u'fassgeschmack',
        u'faßgeschmackes'              : u'fassgeschmackes',
        u'faßgeschmacks'               : u'fassgeschmacks',
        u'faßküfer'                    : u'fassküfer',
        u'faßküfern'                   : u'fassküfern',
        u'faßküfers'                   : u'fassküfers',
        u'faßlich'                     : u'fasslich',
        u'faßliche'                    : u'fassliche',
        u'faßlichem'                   : u'fasslichem',
        u'faßlichen'                   : u'fasslichen',
        u'faßlicher'                   : u'fasslicher',
        u'faßlichere'                  : u'fasslichere',
        u'faßlicherem'                 : u'fasslicherem',
        u'faßlicheren'                 : u'fasslicheren',
        u'faßlicherer'                 : u'fasslicherer',
        u'faßlicheres'                 : u'fasslicheres',
        u'faßliches'                   : u'fassliches',
        u'faßlichste'                  : u'fasslichste',
        u'faßlichstem'                 : u'fasslichstem',
        u'faßlichsten'                 : u'fasslichsten',
        u'faßlichster'                 : u'fasslichster',
        u'faßlichstes'                 : u'fasslichstes',
        u'faßmaler'                    : u'fassmaler',
        u'faßmalern'                   : u'fassmalern',
        u'faßmalers'                   : u'fassmalers',
        u'faßt'                        : u'fasst',
        u'faßte'                       : u'fasste',
        u'faßten'                      : u'fassten',
        u'faßtest'                     : u'fasstest',
        u'faßtet'                      : u'fasstet',
        u'faßtrichter'                 : u'fasstrichter',
        u'faßtrichtern'                : u'fasstrichtern',
        u'faßtrichters'                : u'fasstrichters',
        u'faßwein'                     : u'fasswein',
        u'faßweine'                    : u'fassweine',
        u'faßweinen'                   : u'fassweinen',
        u'faßweines'                   : u'fassweines',
        u'faßweins'                    : u'fassweins',
        u'fehlpaß'                     : u'fehlpass',
        u'fehlschluß'                  : u'fehlschluss',
        u'fehlschuß'                   : u'fehlschuss',
        u'fellos'                      : u'felllos',
        u'fernschuß'                   : u'fernschuss',
        u'fertigungsprozeß'            : u'fertigungsprozess',
        u'fettusche'                   : u'fetttusche',
        u'feuerschutzabschluß'         : u'feuerschutzabschluss',
        u'feßle'                       : u'fessle',
        u'feßlest'                     : u'fesslest',
        u'feßlet'                      : u'fesslet',
        u'fieberphantasien'            : u'fieberfantasien',
        u'figurbewußt'                 : u'figurbewusst',
        u'figurbewußte'                : u'figurbewusste',
        u'figurbewußtem'               : u'figurbewusstem',
        u'figurbewußten'               : u'figurbewussten',
        u'figurbewußter'               : u'figurbewusster',
        u'figurbewußtere'              : u'figurbewusstere',
        u'figurbewußterem'             : u'figurbewussterem',
        u'figurbewußteren'             : u'figurbewussteren',
        u'figurbewußterer'             : u'figurbewussterer',
        u'figurbewußteres'             : u'figurbewussteres',
        u'figurbewußtes'               : u'figurbewusstes',
        u'figurbewußteste'             : u'figurbewussteste',
        u'figurbewußtestem'            : u'figurbewusstestem',
        u'figurbewußtesten'            : u'figurbewusstesten',
        u'figurbewußtester'            : u'figurbewusstester',
        u'figurbewußtestes'            : u'figurbewusstestes',
        u'filmriß'                     : u'filmriss',
        u'firniß'                      : u'firniss',
        u'firnißt'                     : u'firnisst',
        u'firnißte'                    : u'firnisste',
        u'firnißten'                   : u'firnissten',
        u'firnißtest'                  : u'firnisstest',
        u'firnißtet'                   : u'firnisstet',
        u'fitneß'                      : u'fitness',
        u'fitneß-studio'               : u'fitness-studio',
        u'fitneß-studios'              : u'fitness-studios',
        u'fitneßcenter'                : u'fitnesscenter',
        u'fitneßcentern'               : u'fitnesscentern',
        u'fitneßcenters'               : u'fitnesscenters',
        u'fitneßgerät'                 : u'fitnessgerät',
        u'fitneßgeräte'                : u'fitnessgeräte',
        u'fitneßgeräten'               : u'fitnessgeräten',
        u'fitneßgerätes'               : u'fitnessgerätes',
        u'fitneßgeräts'                : u'fitnessgeräts',
        u'fitneßraum'                  : u'fitnessraum',
        u'fitneßraume'                 : u'fitnessraume',
        u'fitneßraumes'                : u'fitnessraumes',
        u'fitneßraums'                 : u'fitnessraums',
        u'fitneßräume'                 : u'fitnessräume',
        u'fitneßräumen'                : u'fitnessräumen',
        u'fitneßstudio'                : u'fitnessstudio',
        u'fitneßstudios'               : u'fitnessstudios',
        u'fitneßtraining'              : u'fitnesstraining',
        u'fitneßtrainings'             : u'fitnesstrainings',
        u'flachschuß'                  : u'flachschuss',
        u'flohbiß'                     : u'flohbiss',
        u'floß'                        : u'floss',
        u'floßt'                       : u'flosst',
        u'flussschiffahrt'             : u'flussschifffahrt',
        u'fluß'                        : u'fluss',
        u'flußaal'                     : u'flussaal',
        u'flußaale'                    : u'flussaale',
        u'flußaalen'                   : u'flussaalen',
        u'flußaales'                   : u'flussaales',
        u'flußaals'                    : u'flussaals',
        u'flußabwärts'                 : u'flussabwärts',
        u'flußarm'                     : u'flussarm',
        u'flußarme'                    : u'flussarme',
        u'flußarmen'                   : u'flussarmen',
        u'flußarmes'                   : u'flussarmes',
        u'flußarms'                    : u'flussarms',
        u'flußaufwärts'                : u'flussaufwärts',
        u'flußbad'                     : u'flussbad',
        u'flußbett'                    : u'flussbett',
        u'flußbette'                   : u'flussbette',
        u'flußbetten'                  : u'flussbetten',
        u'flußdenken'                  : u'flussdenken',
        u'flußdenkens'                 : u'flussdenkens',
        u'flußfisch'                   : u'flussfisch',
        u'flußfische'                  : u'flussfische',
        u'flußfischen'                 : u'flussfischen',
        u'flußfischerei'               : u'flussfischerei',
        u'flußfisches'                 : u'flussfisches',
        u'flußfischs'                  : u'flussfischs',
        u'flußgebiet'                  : u'flussgebiet',
        u'flußgebietseinheit'          : u'flussgebietseinheit',
        u'flußgebietseinheiten'        : u'flussgebietseinheiten',
        u'flußgeist'                   : u'flussgeist',
        u'flußgeiste'                  : u'flussgeiste',
        u'flußgeister'                 : u'flussgeister',
        u'flußgeistern'                : u'flussgeistern',
        u'flußgeistes'                 : u'flussgeistes',
        u'flußgeists'                  : u'flussgeists',
        u'flußgott'                    : u'flussgott',
        u'flußgrund'                   : u'flussgrund',
        u'flußgrunde'                  : u'flussgrunde',
        u'flußgrundes'                 : u'flussgrundes',
        u'flußgrunds'                  : u'flussgrunds',
        u'flußgründe'                  : u'flussgründe',
        u'flußgründen'                 : u'flussgründen',
        u'flußhafen'                   : u'flusshafen',
        u'flußhafens'                  : u'flusshafens',
        u'flußhäfen'                   : u'flusshäfen',
        u'flußhändler'                 : u'flusshändler',
        u'flußhändlern'                : u'flusshändlern',
        u'flußhändlers'                : u'flusshändlers',
        u'flußkrebs'                   : u'flusskrebs',
        u'flußkrebse'                  : u'flusskrebse',
        u'flußkrebsen'                 : u'flusskrebsen',
        u'flußkrebses'                 : u'flusskrebses',
        u'flußland'                    : u'flussland',
        u'flußlauf'                    : u'flusslauf',
        u'flußlaufe'                   : u'flusslaufe',
        u'flußlaufes'                  : u'flusslaufes',
        u'flußlaufs'                   : u'flusslaufs',
        u'flußläufe'                   : u'flussläufe',
        u'flußläufen'                  : u'flussläufen',
        u'flußmarschboden'             : u'flussmarschboden',
        u'flußmarschbodens'            : u'flussmarschbodens',
        u'flußmarschböden'             : u'flussmarschböden',
        u'flußmitte'                   : u'flussmitte',
        u'flußmitten'                  : u'flussmitten',
        u'flußmündung'                 : u'flussmündung',
        u'flußmündungen'               : u'flussmündungen',
        u'flußname'                    : u'flussname',
        u'flußnamen'                   : u'flussnamen',
        u'flußnamens'                  : u'flussnamens',
        u'flußniederung'               : u'flussniederung',
        u'flußniederungen'             : u'flussniederungen',
        u'flußpferd'                   : u'flusspferd',
        u'flußpferde'                  : u'flusspferde',
        u'flußpferden'                 : u'flusspferden',
        u'flußpferdes'                 : u'flusspferdes',
        u'flußpferdfamilienidyll'      : u'flusspferdfamilienidyll',
        u'flußpferdparade'             : u'flusspferdparade',
        u'flußpferds'                  : u'flusspferds',
        u'flußpferdzähnen'             : u'flusspferdzähnen',
        u'flußregenpfeifer'            : u'flussregenpfeifer',
        u'flußregenpfeifern'           : u'flussregenpfeifern',
        u'flußregenpfeifers'           : u'flussregenpfeifers',
        u'flußschiff'                  : u'flussschiff',
        u'flußschiffahrt'              : u'flussschifffahrt',
        u'flußschiffe'                 : u'flussschiffe',
        u'flußschiffen'                : u'flussschiffen',
        u'flußschiffes'                : u'flussschiffes',
        u'flußschifffahrt'             : u'flussschifffahrt',
        u'flußschiffs'                 : u'flussschiffs',
        u'flußschwamm'                 : u'flussschwamm',
        u'flußseeschwalbe'             : u'flussseeschwalbe',
        u'flußseeschwalben'            : u'flussseeschwalben',
        u'flußspat'                    : u'flussspat',
        u'flußspate'                   : u'flussspate',
        u'flußspaten'                  : u'flussspaten',
        u'flußspates'                  : u'flussspates',
        u'flußspats'                   : u'flussspats',
        u'flußspäte'                   : u'flussspäte',
        u'flußspäten'                  : u'flussspäten',
        u'flußstraße'                  : u'flussstraße',
        u'flußstreifen'                : u'flussstreifen',
        u'flußsystem'                  : u'flusssystem',
        u'flußsäure'                   : u'flusssäure',
        u'flußtäler'                   : u'flusstäler',
        u'flußufer'                    : u'flussufer',
        u'flußufern'                   : u'flussufern',
        u'flußufers'                   : u'flussufers',
        u'flußwasser'                  : u'flusswasser',
        u'flußwassern'                 : u'flusswassern',
        u'flußwassers'                 : u'flusswassers',
        u'flußwässer'                  : u'flusswässer',
        u'flußwässern'                 : u'flusswässern',
        u'flüßchen'                    : u'flüsschen',
        u'flüßchens'                   : u'flüsschens',
        u'formiren'                    : u'formieren',
        u'fotograph'                   : u'fotograf',
        u'fotographen'                 : u'fotografen',
        u'fotographieren'              : u'fotografieren',
        u'frachtschiffahrt'            : u'frachtschifffahrt',
        u'fraktionsbeschluß'           : u'fraktionsbeschluss',
        u'frauenhaß'                   : u'frauenhass',
        u'frauenüberschuß'             : u'frauenüberschuss',
        u'freilaßt'                    : u'freilasst',
        u'freiläßt'                    : u'freilässt',
        u'fremdenhaß'                  : u'fremdenhass',
        u'freytag'                     : u'freitag',
        u'freytage'                    : u'freitage',
        u'freytagen'                   : u'freitagen',
        u'freytages'                   : u'freitages',
        u'freytags'                    : u'freitags',
        u'freßaal'                     : u'fressaal',
        u'freßaale'                    : u'fressaale',
        u'freßaalen'                   : u'fressaalen',
        u'freßaales'                   : u'fressaales',
        u'freßaals'                    : u'fressaals',
        u'freßfeind'                   : u'fressfeind',
        u'freßfeinde'                  : u'fressfeinde',
        u'freßfeinden'                 : u'fressfeinden',
        u'freßfeindes'                 : u'fressfeindes',
        u'freßfeinds'                  : u'fressfeinds',
        u'freßluke'                    : u'fressluke',
        u'freßluken'                   : u'fressluken',
        u'freßsack'                    : u'fresssack',
        u'freßsacke'                   : u'fresssacke',
        u'freßsackes'                  : u'fresssackes',
        u'freßsacks'                   : u'fresssacks',
        u'freßsäcke'                   : u'fresssäcke',
        u'freßsäcken'                  : u'fresssäcken',
        u'freßsüchtig'                 : u'fresssüchtig',
        u'freßsüchtige'                : u'fresssüchtige',
        u'freßsüchtigem'               : u'fresssüchtigem',
        u'freßsüchtigen'               : u'fresssüchtigen',
        u'freßsüchtiger'               : u'fresssüchtiger',
        u'freßsüchtigere'              : u'fresssüchtigere',
        u'freßsüchtigerem'             : u'fresssüchtigerem',
        u'freßsüchtigeren'             : u'fresssüchtigeren',
        u'freßsüchtigerer'             : u'fresssüchtigerer',
        u'freßsüchtigeres'             : u'fresssüchtigeres',
        u'freßsüchtiges'               : u'fresssüchtiges',
        u'freßsüchtigste'              : u'fresssüchtigste',
        u'freßsüchtigstem'             : u'fresssüchtigstem',
        u'freßsüchtigsten'             : u'fresssüchtigsten',
        u'freßsüchtigster'             : u'fresssüchtigster',
        u'freßsüchtigstes'             : u'fresssüchtigstes',
        u'friedensprozeß'              : u'friedensprozess',
        u'friedensschluß'              : u'friedensschluss',
        u'friteuse'                    : u'fritteuse',
        u'fritieren'                   : u'frittieren',
        u'friß'                        : u'friss',
        u'frißt'                       : u'frisst',
        u'froschbiß'                   : u'froschbiss',
        u'frotté'                      : u'frottee',
        u'frottés'                     : u'frottees',
        u'frottéstoff'                 : u'frotteestoff',
        u'frottéstoffe'                : u'frotteestoffe',
        u'frottéstoffen'               : u'frotteestoffen',
        u'frottéstoffes'               : u'frotteestoffes',
        u'frottéstoffs'                : u'frotteestoffs',
        u'fussballandesmeister'        : u'fußballlandesmeister',
        u'fussballandesmeisterin'      : u'fußballlandesmeisterin',
        u'fussballandesmeisterinnen'   : u'fußballlandesmeisterinnen',
        u'fussballandesmeistern'       : u'fußballlandesmeistern',
        u'fussballandesmeisters'       : u'fußballlandesmeisters',
        u'fussballegende'              : u'fußballlegende',
        u'fussballegenden'             : u'fußballlegenden',
        u'fussballehrbuch'             : u'fußballlehrbuch',
        u'fussballehrbuche'            : u'fußballlehrbuche',
        u'fussballehrbuches'           : u'fußballlehrbuches',
        u'fussballehrbuchs'            : u'fußballlehrbuchs',
        u'fussballehrbücher'           : u'fußballlehrbücher',
        u'fussballehrbüchern'          : u'fußballlehrbüchern',
        u'fussballehrer'               : u'fußballlehrer',
        u'fussballehrerin'             : u'fußballlehrerin',
        u'fussballehrerinnen'          : u'fußballlehrerinnen',
        u'fussballehrern'              : u'fußballlehrern',
        u'fussballehrers'              : u'fußballlehrers',
        u'fussballiga'                 : u'fußballliga',
        u'fussballänderspiel'          : u'fußballländerspiel',
        u'fussballänderspiele'         : u'fußballländerspiele',
        u'fussballänderspielen'        : u'fußballländerspielen',
        u'fussballänderspieles'        : u'fußballländerspieles',
        u'fussballänderspiels'         : u'fußballländerspiels',
        u'fußballandesmeister'         : u'fußballlandesmeister',
        u'fußballegende'               : u'fußballlegende',
        u'fußballegenden'              : u'fußballlegenden',
        u'fußballehrbuch'              : u'fußballlehrbuch',
        u'fußballehrer'                : u'fußballlehrer',
        u'fußballiga'                  : u'fußballliga',
        u'fußballänderspiel'           : u'fußballländerspiel',
        u'fußkuß'                      : u'fußkuss',
        u'fußle'                       : u'fussle',
        u'fußlest'                     : u'fusslest',
        u'fußlet'                      : u'fusslet',
        u'fußmeßstation'               : u'fussmessstation',
        u'fußmeßstationen'             : u'fussmessstationen',
        u'fäulnisprozeß'               : u'fäulnisprozess',
        u'fäßchen'                     : u'fässchen',
        u'fön'                         : u'föhn',
        u'föne'                        : u'föhne',
        u'fönen'                       : u'föhnen',
        u'fönend'                      : u'föhnend',
        u'fönes'                       : u'föhnes',
        u'fönest'                      : u'föhnest',
        u'fönet'                       : u'föhnet',
        u'föns'                        : u'föhns',
        u'fönst'                       : u'föhnst',
        u'fönt'                        : u'föhnt',
        u'fönte'                       : u'föhnte',
        u'fönten'                      : u'föhnten',
        u'föntest'                     : u'föhntest',
        u'föntet'                      : u'föhntet',
        u'füllaut'                     : u'fülllaut',
        u'fürbaß'                      : u'fürbass',
        u'fürstenthum'                 : u'fürstentum',
        u'gangsterboß'                 : u'gangsterboss',
        u'gasanschluß'                 : u'gasanschluss',
        u'gassenschenke'               : u'gassenschänke',
        u'gastwirth'                   : u'gastwirt',
        u'gastwirthe'                  : u'gastwirte',
        u'gastwirthen'                 : u'gastwirten',
        u'gastwirthes'                 : u'gastwirtes',
        u'gastwirths'                  : u'gastwirts',
        u'gebirgspaß'                  : u'gebirgspass',
        u'gebiß'                       : u'gebiss',
        u'gecreutzigt'                 : u'gekreuzigt',
        u'gedißt'                      : u'gedisst',
        u'gedreßt'                     : u'gedresst',
        u'gefaßt'                      : u'gefasst',
        u'gefaßte'                     : u'gefasste',
        u'gefaßtem'                    : u'gefasstem',
        u'gefaßten'                    : u'gefassten',
        u'gefaßter'                    : u'gefasster',
        u'gefaßtere'                   : u'gefasstere',
        u'gefaßterem'                  : u'gefassterem',
        u'gefaßteren'                  : u'gefassteren',
        u'gefaßterer'                  : u'gefassterer',
        u'gefaßteres'                  : u'gefassteres',
        u'gefaßtes'                    : u'gefasstes',
        u'gefaßteste'                  : u'gefassteste',
        u'gefaßtestem'                 : u'gefasstestem',
        u'gefaßtesten'                 : u'gefasstesten',
        u'gefaßtester'                 : u'gefasstester',
        u'gefaßtestes'                 : u'gefasstestes',
        u'gefängniß'                   : u'gefängnis',
        u'gefönt'                      : u'geföhnt',
        u'gehaßt'                      : u'gehasst',
        u'gehaßter'                    : u'gehasster',
        u'geheimniß'                   : u'geheimnis',
        u'geheimtip'                   : u'geheimtipp',
        u'geheimtips'                  : u'geheimtipps',
        u'gehißt'                      : u'gehisst',
        u'gehußt'                      : u'gehusst',
        u'gekremt'                     : u'gecremt',
        u'geküßt'                      : u'geküsst',
        u'gelaß'                       : u'gelass',
        u'gelaßner'                    : u'gelassner',
        u'gelaßnere'                   : u'gelassnere',
        u'gelaßnerem'                  : u'gelassnerem',
        u'gelaßneren'                  : u'gelassneren',
        u'gelaßnerer'                  : u'gelassnerer',
        u'gelaßneres'                  : u'gelassneres',
        u'geldabfluß'                  : u'geldabfluss',
        u'geldzufluß'                  : u'geldzufluss',
        u'gelöthet'                    : u'gelötet',
        u'gemeinderath'                : u'gemeinderat',
        u'gemeinderathe'               : u'gemeinderate',
        u'gemeinderathes'              : u'gemeinderates',
        u'gemeinderaths'               : u'gemeinderats',
        u'gemeinderäthe'               : u'gemeinderäte',
        u'gemeinderäthen'              : u'gemeinderäten',
        u'gemeinfaßlich'               : u'gemeinfasslich',
        u'gemeinfaßliche'              : u'gemeinfassliche',
        u'gemeinfaßlichem'             : u'gemeinfasslichem',
        u'gemeinfaßlichen'             : u'gemeinfasslichen',
        u'gemeinfaßlicher'             : u'gemeinfasslicher',
        u'gemeinfaßlichere'            : u'gemeinfasslichere',
        u'gemeinfaßlicherem'           : u'gemeinfasslicherem',
        u'gemeinfaßlicheren'           : u'gemeinfasslicheren',
        u'gemeinfaßlicherer'           : u'gemeinfasslicherer',
        u'gemeinfaßlicheres'           : u'gemeinfasslicheres',
        u'gemeinfaßliches'             : u'gemeinfassliches',
        u'gemeinfaßlichste'            : u'gemeinfasslichste',
        u'gemeinfaßlichstem'           : u'gemeinfasslichstem',
        u'gemeinfaßlichsten'           : u'gemeinfasslichsten',
        u'gemeinfaßlichster'           : u'gemeinfasslichster',
        u'gemeinfaßlichstes'           : u'gemeinfasslichstes',
        u'gemeynt'                     : u'gemeint',
        u'gemißt'                      : u'gemisst',
        u'gemse'                       : u'gämse',
        u'gemsen'                      : u'gämsen',
        u'gemshorn'                    : u'gämshorn',
        u'gemshorne'                   : u'gämshorne',
        u'gemshornes'                  : u'gämshornes',
        u'gemshorns'                   : u'gämshorns',
        u'gemshörner'                  : u'gämshörner',
        u'gemshörnern'                 : u'gämshörnern',
        u'gemußt'                      : u'gemusst',
        u'gemußtes'                    : u'gemusstes',
        u'gemüth'                      : u'gemüt',
        u'gemüthe'                     : u'gemüte',
        u'gemüther'                    : u'gemüter',
        u'gemüthern'                   : u'gemütern',
        u'gemüthes'                    : u'gemütes',
        u'gemüths'                     : u'gemüts',
        u'generalstabsausschuß'        : u'generalstabsausschuss',
        u'genesungsprozeß'             : u'genesungsprozess',
        u'genickschuß'                 : u'genickschuss',
        u'genoß'                       : u'genoss',
        u'genuß'                       : u'genuss',
        u'genußfreude'                 : u'genussfreude',
        u'genußmensch'                 : u'genussmensch',
        u'genußmenschen'               : u'genussmenschen',
        u'genußmittel'                 : u'genussmittel',
        u'genußmitteln'                : u'genussmitteln',
        u'genußmittels'                : u'genussmittels',
        u'genußreiches'                : u'genussreiches',
        u'genußspecht'                 : u'genussspecht',
        u'genußspechte'                : u'genussspechte',
        u'genußspechten'               : u'genussspechten',
        u'genußspechtes'               : u'genussspechtes',
        u'genußspechts'                : u'genussspechts',
        u'genußsucht'                  : u'genusssucht',
        u'genußsüchtig'                : u'genusssüchtig',
        u'genußsüchtige'               : u'genusssüchtige',
        u'genußsüchtigem'              : u'genusssüchtigem',
        u'genußsüchtigen'              : u'genusssüchtigen',
        u'genußsüchtiger'              : u'genusssüchtiger',
        u'genußsüchtigere'             : u'genusssüchtigere',
        u'genußsüchtigerem'            : u'genusssüchtigerem',
        u'genußsüchtigeren'            : u'genusssüchtigeren',
        u'genußsüchtigerer'            : u'genusssüchtigerer',
        u'genußsüchtigeres'            : u'genusssüchtigeres',
        u'genußsüchtiges'              : u'genusssüchtiges',
        u'genußsüchtigste'             : u'genusssüchtigste',
        u'genußsüchtigstem'            : u'genusssüchtigstem',
        u'genußsüchtigsten'            : u'genusssüchtigsten',
        u'genußsüchtigster'            : u'genusssüchtigster',
        u'genußsüchtigstes'            : u'genusssüchtigstes',
        u'genäßt'                      : u'genässt',
        u'genüßlich'                   : u'genüsslich',
        u'genüßliche'                  : u'genüssliche',
        u'genüßlichem'                 : u'genüsslichem',
        u'genüßlichen'                 : u'genüsslichen',
        u'genüßlicher'                 : u'genüsslicher',
        u'genüßlichere'                : u'genüsslichere',
        u'genüßlicherem'               : u'genüsslicherem',
        u'genüßlicheren'               : u'genüsslicheren',
        u'genüßlicherer'               : u'genüsslicherer',
        u'genüßlicheres'               : u'genüsslicheres',
        u'genüßliches'                 : u'genüssliches',
        u'genüßlichste'                : u'genüsslichste',
        u'genüßlichstem'               : u'genüsslichstem',
        u'genüßlichsten'               : u'genüsslichsten',
        u'genüßlichster'               : u'genüsslichster',
        u'genüßlichstes'               : u'genüsslichstes',
        u'geograph'                    : u'geograf',
        u'geographen'                  : u'geografen',
        u'geographie'                  : u'geografie',
        u'geographiebuch'              : u'geografiebuch',
        u'geographieklasse'            : u'geografieklasse',
        u'geographiestunde'            : u'geografiestunde',
        u'gepaßt'                      : u'gepasst',
        u'gepißt'                      : u'gepisst',
        u'gepreßt'                     : u'gepresst',
        u'gepreßte'                    : u'gepresste',
        u'gepreßtem'                   : u'gepresstem',
        u'gepreßten'                   : u'gepressten',
        u'gepreßter'                   : u'gepresster',
        u'gepreßtes'                   : u'gepresstes',
        u'gerauht'                     : u'geraut',
        u'gerichtsbeschluß'            : u'gerichtsbeschluss',
        u'gerichtsprozeß'              : u'gerichtsprozess',
        u'geröllawine'                 : u'gerölllawine',
        u'geröllawinen'                : u'gerölllawinen',
        u'gesamtabschluß'              : u'gesamtabschluss',
        u'geschaßt'                    : u'geschasst',
        u'geschirreiniger'             : u'geschirrreiniger',
        u'geschirreinigern'            : u'geschirrreinigern',
        u'geschirreinigers'            : u'geschirrreinigers',
        u'geschiß'                     : u'geschiss',
        u'geschlußfolgert'             : u'geschlussfolgert',
        u'geschneuzt'                  : u'geschnäuzt',
        u'geschoß'                     : u'geschoss',
        u'geschoßflächenzahl'          : u'geschossflächenzahl',
        u'geschoßflächenzahlen'        : u'geschossflächenzahlen',
        u'geschoßhöhe'                 : u'geschosshöhe',
        u'geschoßhöhen'                : u'geschosshöhen',
        u'geschoßpost'                 : u'geschosspost',
        u'geschoßschwarm'              : u'geschossschwarm',
        u'geschoßwohnung'              : u'geschosswohnung',
        u'geschoßwohnungen'            : u'geschosswohnungen',
        u'geschäftsprozeß'             : u'geschäftsprozess',
        u'gesetzesbeschluß'            : u'gesetzesbeschluss',
        u'gespieen'                    : u'gespien',
        u'gestreßt'                    : u'gestresst',
        u'gethan'                      : u'getan',
        u'gethier'                     : u'getier',
        u'gethiere'                    : u'getiere',
        u'gethieres'                   : u'getieres',
        u'gethiers'                    : u'getiers',
        u'gethürmt'                    : u'getürmt',
        u'gewehrschuß'                 : u'gewehrschuss',
        u'gewinnummer'                 : u'gewinnnummer',
        u'gewinnummern'                : u'gewinnnummern',
        u'gewissensbiß'                : u'gewissensbiss',
        u'gewiß'                       : u'gewiss',
        u'gewißheit'                   : u'gewissheit',
        u'gewißheiten'                 : u'gewissheiten',
        u'gewißlich'                   : u'gewisslich',
        u'gewußt'                      : u'gewusst',
        u'gewußte'                     : u'gewusste',
        u'gewußtem'                    : u'gewusstem',
        u'gewußten'                    : u'gewussten',
        u'gewußter'                    : u'gewusster',
        u'gewußtes'                    : u'gewusstes',
        u'giebt'                       : u'gibt',
        u'glasfluß'                    : u'glasfluss',
        u'gletscherfluß'               : u'gletscherfluss',
        u'glimmstengel'                : u'glimmstängel',
        u'glimmstengeln'               : u'glimmstängeln',
        u'glimmstengels'               : u'glimmstängels',
        u'gnadenschuß'                 : u'gnadenschuss',
        u'goettingen'                  : u'göttingen',
        u'gothe'                       : u'gote',
        u'gothen'                      : u'goten',
        u'gothik'                      : u'gotik',
        u'gothisch'                    : u'gotisch',
        u'gothische'                   : u'gotische',
        u'gothischem'                  : u'gotischem',
        u'gothischen'                  : u'gotischen',
        u'gothischer'                  : u'gotischer',
        u'gothisches'                  : u'gotisches',
        u'gothland'                    : u'gotland',
        u'gothlands'                   : u'gotlands',
        u'goß'                         : u'goss',
        u'grausig-gräßlich-grausame'   : u'grausig-grässlich-grausame',
        u'grellila'                    : u'grelllila',
        u'grenzfluß'                   : u'grenzfluss',
        u'greuel'                      : u'gräuel',
        u'greuel-wort'                 : u'gräuelwort',
        u'greuelbilder'                : u'gräuelbilder',
        u'greueln'                     : u'gräueln',
        u'greuelreichen'               : u'gräuelreichen',
        u'greuels'                     : u'gräuels',
        u'greueltat'                   : u'gräueltat',
        u'greueltaten'                 : u'gräueltaten',
        u'greuelwort'                  : u'gräuelwort',
        u'greuelwortes'                : u'gräuelwortes',
        u'greuelwörtern'               : u'gräuelwörtern',
        u'greulich'                    : u'gräulich',
        u'greuliche'                   : u'gräuliche',
        u'greulichem'                  : u'gräulichem',
        u'greulichen'                  : u'gräulichen',
        u'greulicher'                  : u'gräulicher',
        u'greulichere'                 : u'gräulichere',
        u'greulicherem'                : u'gräulicherem',
        u'greulicheren'                : u'gräulicheren',
        u'greulicherer'                : u'gräulicherer',
        u'greulicheres'                : u'gräulicheres',
        u'greuliches'                  : u'gräuliches',
        u'greulichste'                 : u'gräulichste',
        u'greulichstem'                : u'gräulichstem',
        u'greulichsten'                : u'gräulichsten',
        u'greulichster'                : u'gräulichster',
        u'greulichstes'                : u'gräulichstes',
        u'grisli'                      : u'grizzly',
        u'grislibär'                   : u'grizzlybär',
        u'grislibären'                 : u'grizzlybären',
        u'grislibärin'                 : u'grizzlybärin',
        u'grislis'                     : u'grizzlys',
        u'gross'                       : u'groß',
        u'grosse'                      : u'große',
        u'grossen'                     : u'großen',
        u'grosses'                     : u'großes',
        u'grossschiffahrt'             : u'großschifffahrt',
        u'grossschiffahrtsweg'         : u'großschifffahrtsweg',
        u'grossschiffahrtswege'        : u'großschifffahrtswege',
        u'grossschiffahrtswegen'       : u'großschifffahrtswegen',
        u'grossschiffahrtsweges'       : u'großschifffahrtsweges',
        u'grossschiffahrtswegs'        : u'großschifffahrtswegs',
        u'groszherzogthum'             : u'großherzogtum',
        u'grottenhäßlich'              : u'grottenhässlich',
        u'grottenhäßliche'             : u'grottenhässliche',
        u'grottenhäßlichem'            : u'grottenhässlichem',
        u'grottenhäßlichen'            : u'grottenhässlichen',
        u'grottenhäßlicher'            : u'grottenhässlicher',
        u'grottenhäßliches'            : u'grottenhässliches',
        u'großfürstenthum'             : u'großfürstentum',
        u'großschiffahrtsstraße'       : u'großschifffahrtsstraße',
        u'großschiffahrtsstraßen'      : u'großschifffahrtsstraßen',
        u'großschiffahrtsweg'          : u'großschifffahrtsweg',
        u'großschiffahrtswege'         : u'großschifffahrtswege',
        u'großschiffahrtswegen'        : u'großschifffahrtswegen',
        u'großschiffahrtsweges'        : u'großschifffahrtsweges',
        u'großschiffahrtswegs'         : u'großschifffahrtswegs',
        u'grundbaß'                    : u'grundbass',
        u'grundriß'                    : u'grundriss',
        u'gränze'                      : u'grenze',
        u'gränzen'                     : u'grenzen',
        u'gräßlich'                    : u'grässlich',
        u'gräßliche'                   : u'grässliche',
        u'gräßlichem'                  : u'grässlichem',
        u'gräßlichen'                  : u'grässlichen',
        u'gräßlicher'                  : u'grässlicher',
        u'gräßlichere'                 : u'grässlichere',
        u'gräßlicherem'                : u'grässlicherem',
        u'gräßlicheren'                : u'grässlicheren',
        u'gräßlicherer'                : u'grässlicherer',
        u'gräßlicheres'                : u'grässlicheres',
        u'gräßliches'                  : u'grässliches',
        u'gräßlichste'                 : u'grässlichste',
        u'gräßlichstem'                : u'grässlichstem',
        u'gräßlichsten'                : u'grässlichsten',
        u'gräßlichster'                : u'grässlichster',
        u'gräßlichstes'                : u'grässlichstes',
        u'gummimantelgeschoß'          : u'gummimantelgeschoss',
        u'gurkenpaß'                   : u'gurkenpass',
        u'gutenachtkuß'                : u'gutenachtkuss',
        u'guß'                         : u'guss',
        u'gußbeton'                    : u'gussbeton',
        u'gußeisen'                    : u'gusseisen',
        u'gußeisens'                   : u'gusseisens',
        u'gußeisern'                   : u'gusseisern',
        u'gußeiserne'                  : u'gusseiserne',
        u'gußeisernem'                 : u'gusseisernem',
        u'gußeisernen'                 : u'gusseisernen',
        u'gußeiserner'                 : u'gusseiserner',
        u'gußeisernes'                 : u'gusseisernes',
        u'gußform'                     : u'gussform',
        u'gußstahl'                    : u'gussstahl',
        u'gußteil'                     : u'gussteil',
        u'gußteile'                    : u'gussteile',
        u'gußteilen'                   : u'gussteilen',
        u'gußteiles'                   : u'gussteiles',
        u'gußteils'                    : u'gussteils',
        u'gärprozeß'                   : u'gärprozess',
        u'gärungsprozeß'               : u'gärungsprozess',
        u'gäßchen'                     : u'gässchen',
        u'gäßchens'                    : u'gässchens',
        u'gürtelthier'                 : u'gürteltier',
        u'gürtelthiere'                : u'gürteltiere',
        u'gürtelthieren'               : u'gürteltieren',
        u'gürtelthieres'               : u'gürteltieres',
        u'gürtelthiers'                : u'gürteltiers',
        u'gürthelthier'                : u'gürteltier',
        u'gürthelthiere'               : u'gürteltiere',
        u'gürthelthieren'              : u'gürteltieren',
        u'güterschiffahrt'             : u'güterschifffahrt',
        u'haarriß'                     : u'haarriss',
        u'hafenschenke'                : u'hafenschänke',
        u'halbgeschoß'                 : u'halbgeschoss',
        u'hallelujah'                  : u'halleluja',
        u'handelsbilanzüberschuß'      : u'handelsbilanzüberschuss',
        u'handelung'                   : u'handlung',
        u'handies'                     : u'handys',
        u'handkuß'                     : u'handkuss',
        u'haraß'                       : u'harass',
        u'haselnuß'                    : u'haselnuss',
        u'haselnußstauden'             : u'haselnussstauden',
        u'haselnußstrauch'             : u'haselnussstrauch',
        u'haselnußstrauche'            : u'haselnussstrauche',
        u'haselnußstrauches'           : u'haselnussstrauches',
        u'haselnußstrauchs'            : u'haselnussstrauchs',
        u'haselnußsträucher'           : u'haselnusssträucher',
        u'haselnußsträuchern'          : u'haselnusssträuchern',
        u'haselruthe'                  : u'haselrute',
        u'haselruthen'                 : u'haselruten',
        u'hasenkänguruh'               : u'hasenkänguru',
        u'hauptschiffahrtsstrasse'     : u'hauptschifffahrtsstraße',
        u'hauptschiffahrtsstrassen'    : u'hauptschifffahrtsstraßen',
        u'hauptschulabschluß'          : u'hauptschulabschluss',
        u'hausanschluß'                : u'hausanschluss',
        u'haushaltsausschuß'           : u'haushaltsausschuss',
        u'haß'                         : u'hass',
        u'haßentbrannt'                : u'hassentbrannt',
        u'haßerfüllt'                  : u'hasserfüllt',
        u'haßerfüllte'                 : u'hasserfüllte',
        u'haßerfülltem'                : u'hasserfülltem',
        u'haßerfüllten'                : u'hasserfüllten',
        u'haßfigur'                    : u'hassfigur',
        u'haßfiguren'                  : u'hassfiguren',
        u'haßkommentar'                : u'hasskommentar',
        u'haßkommentare'               : u'hasskommentare',
        u'haßkommentaren'              : u'hasskommentaren',
        u'haßkommentars'               : u'hasskommentars',
        u'haßliebe'                    : u'hassliebe',
        u'haßmaske'                    : u'hassmaske',
        u'haßmasken'                   : u'hassmasken',
        u'haßobjekt'                   : u'hassobjekt',
        u'haßobjekte'                  : u'hassobjekte',
        u'haßobjekten'                 : u'hassobjekten',
        u'haßobjektes'                 : u'hassobjektes',
        u'haßobjekts'                  : u'hassobjekts',
        u'haßprediger'                 : u'hassprediger',
        u'haßpredigern'                : u'hasspredigern',
        u'haßpredigers'                : u'hasspredigers',
        u'haßpredigt'                  : u'hasspredigt',
        u'haßpredigten'                : u'hasspredigten',
        u'haßt'                        : u'hasst',
        u'haßte'                       : u'hasste',
        u'haßten'                      : u'hassten',
        u'haßtest'                     : u'hasstest',
        u'haßtet'                      : u'hasstet',
        u'haßtirade'                   : u'hasstirade',
        u'haßtiraden'                  : u'hasstiraden',
        u'hefepreßwein'                : u'hefepresswein',
        u'hefepreßweine'               : u'hefepressweine',
        u'hefepreßweinen'              : u'hefepressweinen',
        u'hefepreßweines'              : u'hefepressweines',
        u'hefepreßweins'               : u'hefepressweins',
        u'heidenthum'                  : u'heidentum',
        u'heilprozeß'                  : u'heilprozess',
        u'heilungsprozeß'              : u'heilungsprozess',
        u'heimath'                     : u'heimat',
        u'heimathen'                   : u'heimaten',
        u'heimatschuß'                 : u'heimatschuss',
        u'heissen'                     : u'heißen',
        u'heisst'                      : u'heißt',
        u'hellicht'                    : u'helllicht',
        u'hellichte'                   : u'helllichte',
        u'hellichtem'                  : u'helllichtem',
        u'hellichten'                  : u'helllichten',
        u'hellichter'                  : u'helllichter',
        u'hellichtes'                  : u'helllichtes',
        u'hellila'                     : u'helllila',
        u'herausriß'                   : u'herausriss',
        u'herausrißt'                  : u'herausrisst',
        u'hercules'                    : u'herkules',
        u'herscht'                     : u'herrscht',
        u'herstellungsprozeß'          : u'herstellungsprozess',
        u'herumtelephoniere'           : u'herumtelefoniere',
        u'herumtelephonieren'          : u'herumtelefonieren',
        u'herumtelephonierend'         : u'herumtelefonierend',
        u'herumtelephonierende'        : u'herumtelefonierende',
        u'herumtelephonierendem'       : u'herumtelefonierendem',
        u'herumtelephonierenden'       : u'herumtelefonierenden',
        u'herumtelephonierender'       : u'herumtelefonierender',
        u'herumtelephonierendes'       : u'herumtelefonierendes',
        u'herumtelephonierest'         : u'herumtelefonierest',
        u'herumtelephonieret'          : u'herumtelefonieret',
        u'herumtelephonierst'          : u'herumtelefonierst',
        u'herumtelephoniert'           : u'herumtelefoniert',
        u'herumtelephonierte'          : u'herumtelefonierte',
        u'herumtelephonierten'         : u'herumtelefonierten',
        u'herumtelephoniertest'        : u'herumtelefoniertest',
        u'herumtelephoniertet'         : u'herumtelefoniertet',
        u'herumzutelephonieren'        : u'herumzutelefonieren',
        u'herunterfloß'                : u'herunterfloss',
        u'herunterlaßt'                : u'herunterlasst',
        u'herunterläßt'                : u'herunterlässt',
        u'herzas'                      : u'herzass',
        u'hexenprozeß'                 : u'hexenprozess',
        u'hexenschuß'                  : u'hexenschuss',
        u'hindurchfloß'                : u'hindurchfloss',
        u'hineinbiß'                   : u'hineinbiss',
        u'hineinbißt'                  : u'hineinbisst',
        u'hinriß'                      : u'hinriss',
        u'hinterlaß'                   : u'hinterlass',
        u'hinterlaßt'                  : u'hinterlasst',
        u'hinterläßt'                  : u'hinterlässt',
        u'hiß'                         : u'hiss',
        u'hißt'                        : u'hisst',
        u'hißte'                       : u'hisste',
        u'hißten'                      : u'hissten',
        u'hißtest'                     : u'hisstest',
        u'hißtet'                      : u'hisstet',
        u'hobbies'                     : u'hobbys',
        u'hochgenuß'                   : u'hochgenuss',
        u'hochschulabschluß'           : u'hochschulabschluss',
        u'hochzeitsphoto'              : u'hochzeitsfoto',
        u'holzfaß'                     : u'holzfass',
        u'hosteß'                      : u'hostess',
        u'hoteltelephon'               : u'hoteltelefon',
        u'hoteltelephone'              : u'hoteltelefone',
        u'hoteltelephonen'             : u'hoteltelefonen',
        u'hoteltelephons'              : u'hoteltelefons',
        u'hundebiß'                    : u'hundebiss',
        u'huß'                         : u'huss',
        u'hußt'                        : u'husst',
        u'hußte'                       : u'husste',
        u'hußten'                      : u'hussten',
        u'hußtest'                     : u'husstest',
        u'hußtet'                      : u'husstet',
        u'händler-dhau'                : u'händler-dau',
        u'häresieprozeß'               : u'häresieprozess',
        u'häßlich'                     : u'hässlich',
        u'häßliche'                    : u'hässliche',
        u'häßlichem'                   : u'hässlichem',
        u'häßlichen'                   : u'hässlichen',
        u'häßlicher'                   : u'hässlicher',
        u'häßlichere'                  : u'hässlichere',
        u'häßlicherem'                 : u'hässlicherem',
        u'häßlicheren'                 : u'hässlicheren',
        u'häßlicherer'                 : u'hässlicherer',
        u'häßlicheres'                 : u'hässlicheres',
        u'häßliches'                   : u'hässliches',
        u'häßlichkeit'                 : u'hässlichkeit',
        u'häßlichkeiten'               : u'hässlichkeiten',
        u'häßlichste'                  : u'hässlichste',
        u'häßlichstem'                 : u'hässlichstem',
        u'häßlichsten'                 : u'hässlichsten',
        u'häßlichster'                 : u'hässlichster',
        u'häßlichstes'                 : u'hässlichstes',
        u'hülfelehrer'                 : u'hilfslehrer',
        u'hülfslehrer'                 : u'hilfslehrer',
        u'i-punkt'                     : u'i-punkt',
        u'i-punkte'                    : u'i-punkte',
        u'i-punkten'                   : u'i-punkten',
        u'i-punktes'                   : u'i-punktes',
        u'i-punkts'                    : u'i-punkts',
        u'i-tüpfelchen'                : u'i-tüpfelchen',
        u'imbiß'                       : u'imbiss',
        u'imbißbude'                   : u'imbissbude',
        u'imbißbuden'                  : u'imbissbuden',
        u'imbißstand'                  : u'imbissstand',
        u'imbißstande'                 : u'imbissstande',
        u'imbißstandes'                : u'imbissstandes',
        u'imbißstands'                 : u'imbissstands',
        u'imbißstände'                 : u'imbissstände',
        u'imbißständen'                : u'imbissständen',
        u'imperfect'                   : u'imperfekt',
        u'imperfecte'                  : u'imperfekte',
        u'imperfecten'                 : u'imperfekten',
        u'imperfects'                  : u'imperfekts',
        u'impfpaß'                     : u'impfpass',
        u'incompetent'                 : u'inkompetent',
        u'incompetente'                : u'inkompetente',
        u'incompetentem'               : u'inkompetentem',
        u'incompetenten'               : u'inkompetenten',
        u'incompetenter'               : u'inkompetenter',
        u'incompetentere'              : u'inkompetentere',
        u'incompetenterem'             : u'inkompetenterem',
        u'incompetenteren'             : u'inkompetenteren',
        u'incompetenterer'             : u'inkompetenterer',
        u'incompetenteres'             : u'inkompetenteres',
        u'incompetentes'               : u'inkompetentes',
        u'incompetenteste'             : u'inkompetenteste',
        u'incompetentestem'            : u'inkompetentestem',
        u'incompetentesten'            : u'inkompetentesten',
        u'incompetentester'            : u'inkompetentester',
        u'incompetentestes'            : u'inkompetentestes',
        u'incompetenz'                 : u'inkompetenz',
        u'incompetenzen'               : u'inkompetenzen',
        u'inconsequent'                : u'inkonsequent',
        u'inconsequente'               : u'inkonsequente',
        u'inconsequentem'              : u'inkonsequentem',
        u'inconsequenten'              : u'inkonsequenten',
        u'inconsequenter'              : u'inkonsequenter',
        u'inconsequentere'             : u'inkonsequentere',
        u'inconsequenterem'            : u'inkonsequenterem',
        u'inconsequenteren'            : u'inkonsequenteren',
        u'inconsequenterer'            : u'inkonsequenterer',
        u'inconsequenteres'            : u'inkonsequenteres',
        u'inconsequentes'              : u'inkonsequentes',
        u'inconsequenteste'            : u'inkonsequenteste',
        u'inconsequentestem'           : u'inkonsequentestem',
        u'inconsequentesten'           : u'inkonsequentesten',
        u'inconsequentester'           : u'inkonsequentester',
        u'inconsequentestes'           : u'inkonsequentestes',
        u'inconsequenz'                : u'inkonsequenz',
        u'inconsequenzen'              : u'inkonsequenzen',
        u'indicativ'                   : u'indikativ',
        u'indicative'                  : u'indikative',
        u'indicativen'                 : u'indikativen',
        u'indicativs'                  : u'indikativs',
        u'indizienprozeß'              : u'indizienprozess',
        u'innovationspotential'        : u'innovationspotenzial',
        u'inquisitionsprozeß'          : u'inquisitionsprozess',
        u'integrationsprozeß'          : u'integrationsprozess',
        u'intercity-expreß'            : u'intercity-express',
        u'intercityexpreß'             : u'intercityexpress',
        u'interressanter'              : u'interessanter',
        u'ionisationsmeßröhre'         : u'ionisationsmessröhre',
        u'ionisationsmeßröhren'        : u'ionisationsmessröhren',
        u'ionisierungsprozeß'          : u'ionisierungsprozess',
        u'iß'                          : u'iss',
        u'ißt'                         : u'isst',
        u'jagdschloß'                  : u'jagdschloss',
        u'jahresabschluß'              : u'jahresabschluss',
        u'jahresüberschuß'             : u'jahresüberschuss',
        u'jahrn'                       : u'jahren',
        u'jauchefaß'                   : u'jauchefass',
        u'jauchenfaß'                  : u'jauchenfass',
        u'job-sharing'                 : u'jobsharing',
        u'jockei'                      : u'jockey',
        u'jockeis'                     : u'jockeys',
        u'joga'                        : u'yoga',
        u'jogakurs'                    : u'yogakurs',
        u'jogakurse'                   : u'yogakurse',
        u'jogakursen'                  : u'yogakursen',
        u'jogakurses'                  : u'yogakurses',
        u'jogalehrer'                  : u'yogalehrer',
        u'jogalehrerin'                : u'yogalehrerin',
        u'jogalehrerinnen'             : u'yogalehrerinnen',
        u'jogalehrern'                 : u'yogalehrern',
        u'jogalehrers'                 : u'yogalehrers',
        u'jogas'                       : u'yogas',
        u'jogaschule'                  : u'yogaschule',
        u'jogaschulen'                 : u'yogaschulen',
        u'jogaübung'                   : u'yogaübung',
        u'jogaübungen'                 : u'yogaübungen',
        u'judenhaß'                    : u'judenhass',
        u'judenthum'                   : u'judentum',
        u'julierpaß'                   : u'julierpass',
        u'jäheit'                      : u'jähheit',
        u'jüdisch-deutsch'             : u'jüdischdeutsch',
        u'kabelanschluß'               : u'kabelanschluss',
        u'kabinettsbeschluß'           : u'kabinettsbeschluss',
        u'kabrio'                      : u'cabrio',
        u'kalbsnuß'                    : u'kalbsnuss',
        u'kalligraphischen'            : u'kalligrafischen',
        u'kalorienbewußt'              : u'kalorienbewusst',
        u'kalorienbewußte'             : u'kalorienbewusste',
        u'kalorienbewußtem'            : u'kalorienbewusstem',
        u'kalorienbewußten'            : u'kalorienbewussten',
        u'kalorienbewußter'            : u'kalorienbewusster',
        u'kalorienbewußtere'           : u'kalorienbewusstere',
        u'kalorienbewußterem'          : u'kalorienbewussterem',
        u'kalorienbewußteren'          : u'kalorienbewussteren',
        u'kalorienbewußterer'          : u'kalorienbewussterer',
        u'kalorienbewußteres'          : u'kalorienbewussteres',
        u'kalorienbewußtes'            : u'kalorienbewusstes',
        u'kalorienbewußteste'          : u'kalorienbewussteste',
        u'kalorienbewußtestem'         : u'kalorienbewusstestem',
        u'kalorienbewußtesten'         : u'kalorienbewusstesten',
        u'kalorienbewußtester'         : u'kalorienbewusstester',
        u'kalorienbewußtestes'         : u'kalorienbewusstestes',
        u'kaltgepreßt'                 : u'kaltgepresst',
        u'kalvinismus'                 : u'calvinismus',
        u'kalvinist'                   : u'calvinist',
        u'kalvinisten'                 : u'calvinisten',
        u'kalvinistin'                 : u'calvinistin',
        u'kalvinistinnen'              : u'calvinistinnen',
        u'kalvinistisch'               : u'calvinistisch',
        u'kalvinistische'              : u'calvinistische',
        u'kalvinistischem'             : u'calvinistischem',
        u'kalvinistischen'             : u'calvinistischen',
        u'kalvinistischer'             : u'calvinistischer',
        u'kalvinistischere'            : u'calvinistischere',
        u'kalvinistischerem'           : u'calvinistischerem',
        u'kalvinistischeren'           : u'calvinistischeren',
        u'kalvinistischerer'           : u'calvinistischerer',
        u'kalvinistischeres'           : u'calvinistischeres',
        u'kalvinistisches'             : u'calvinistisches',
        u'kalvinistischste'            : u'calvinistischste',
        u'kalvinistischstem'           : u'calvinistischstem',
        u'kalvinistischster'           : u'calvinistischster',
        u'kalvinistischstes'           : u'calvinistischstes',
        u'kameraverschluß'             : u'kameraverschluss',
        u'kamerunnuß'                  : u'kamerunnuss',
        u'kammacher'                   : u'kammmacher',
        u'kammachergeschäft'           : u'kammmachergeschäft',
        u'kammachergeschäfts'          : u'kammmachergeschäfts',
        u'kammachermeister'            : u'kammmachermeister',
        u'kammachern'                  : u'kammmachern',
        u'kammachers'                  : u'kammmachers',
        u'kammaschine'                 : u'kammmaschine',
        u'kammolch'                    : u'kammmolch',
        u'kammolche'                   : u'kammmolche',
        u'kammolchen'                  : u'kammmolchen',
        u'kammolches'                  : u'kammmolches',
        u'kammolchs'                   : u'kammmolchs',
        u'kammuschel'                  : u'kammmuschel',
        u'kammuscheln'                 : u'kammmuscheln',
        u'kaninchenkänguruh'           : u'kaninchenkänguru',
        u'kaninchenkänguruhs'          : u'kaninchenkängurus',
        u'kanonenschuß'                : u'kanonenschuss',
        u'kanonenschußartiger'         : u'kanonenschussartiger',
        u'kanzelmißbrauch'             : u'kanzelmissbrauch',
        u'kanzelmißbrauche'            : u'kanzelmissbrauche',
        u'kanzelmißbrauches'           : u'kanzelmissbrauches',
        u'kanzelmißbrauchs'            : u'kanzelmissbrauchs',
        u'kanzelmißbrauchsparagraph'   : u'kanzelmissbrauchsparagraph',
        u'kanzelmißbrauchsparagraphen' : u'kanzelmissbrauchsparagraphen',
        u'kapitalzufluß'               : u'kapitalzufluss',
        u'kapselriß'                   : u'kapselriss',
        u'karakter'                    : u'charakter',
        u'karaktere'                   : u'charaktere',
        u'karakteren'                  : u'charakteren',
        u'karakters'                   : u'charakters',
        u'karamel'                     : u'karamell',
        u'karamelfarben'               : u'karamellfarben',
        u'karamelfarbene'              : u'karamellfarbene',
        u'karamelfarbenem'             : u'karamellfarbenem',
        u'karamelfarbenen'             : u'karamellfarbenen',
        u'karamelfarbener'             : u'karamellfarbener',
        u'karamelfarbenes'             : u'karamellfarbenes',
        u'karamelkarawane'             : u'karamellkarawane',
        u'karamels'                    : u'karamells',
        u'karfreytag'                  : u'karfreitag',
        u'karfreytage'                 : u'karfreitage',
        u'karfreytagen'                : u'karfreitagen',
        u'karfreytages'                : u'karfreitages',
        u'karfreytags'                 : u'karfreitags',
        u'kargo'                       : u'cargo',
        u'kargos'                      : u'cargos',
        u'karoas'                      : u'karoass',
        u'kartentelephon'              : u'kartentelefon',
        u'kartentelephone'             : u'kartentelefone',
        u'kartentelephonen'            : u'kartentelefonen',
        u'kartentelephons'             : u'kartentelefons',
        u'katarr'                      : u'katarrh',
        u'katarre'                     : u'katarrhe',
        u'katarren'                    : u'katarrhen',
        u'katarrs'                     : u'katarrhs',
        u'katholicismus'               : u'katholizismus',
        u'kaßler'                      : u'kassler',
        u'kaßlern'                     : u'kasslern',
        u'kaßlers'                     : u'kasslers',
        u'kehlkopfverschlußlaut'       : u'kehlkopfverschlusslaut',
        u'kehlkopfverschlußlaute'      : u'kehlkopfverschlusslaute',
        u'kehlkopfverschlußlauten'     : u'kehlkopfverschlusslauten',
        u'kehlkopfverschlußlautes'     : u'kehlkopfverschlusslautes',
        u'kehlkopfverschlußlauts'      : u'kehlkopfverschlusslauts',
        u'kennummer'                   : u'kennnummer',
        u'kennummern'                  : u'kennnummern',
        u'ketschup'                    : u'ketchup',
        u'ketschupflasche'             : u'ketchupflasche',
        u'ketschupflaschen'            : u'ketchupflaschen',
        u'ketschups'                   : u'ketchups',
        u'ketzerprozeß'                : u'ketzerprozess',
        u'keß'                         : u'kess',
        u'keßheit'                     : u'kessheit',
        u'khaiberpaß'                  : u'khaiberpass',
        u'khyberpaß'                   : u'khyberpass',
        u'kindesmißbrauch'             : u'kindesmissbrauch',
        u'kindesmißbrauche'            : u'kindesmissbrauche',
        u'kindesmißbrauches'           : u'kindesmissbrauches',
        u'kindesmißbrauchs'            : u'kindesmissbrauchs',
        u'kindesmißbräuche'            : u'kindesmissbräuche',
        u'kindesmißbräuchen'           : u'kindesmissbräuchen',
        u'kindesmißhandlung'           : u'kindesmisshandlung',
        u'kindesmißhandlungen'         : u'kindesmisshandlungen',
        u'kinematograph'               : u'kinematograf',
        u'kinematographen-theater'     : u'kinematografen-theater',
        u'kirchthurm'                  : u'kirchturm',
        u'kißchen'                     : u'kisschen',
        u'kißchens'                    : u'kisschens',
        u'klappult'                    : u'klapppult',
        u'klappulte'                   : u'klapppulte',
        u'klappulten'                  : u'klapppulten',
        u'klappultes'                  : u'klapppultes',
        u'klappults'                   : u'klapppults',
        u'klassenbewußtsein'           : u'klassenbewusstsein',
        u'klassenhaß'                  : u'klassenhass',
        u'klatschenaß'                 : u'klatschenass',
        u'klatschnaß'                  : u'klatschnass',
        u'klausenpaß'                  : u'klausenpass',
        u'klaß'                        : u'klass',
        u'klaßleiter'                  : u'klassleiter',
        u'klaßleitern'                 : u'klassleitern',
        u'klaßleiters'                 : u'klassleiters',
        u'klemmappe'                   : u'klemmmappe',
        u'klemmappen'                  : u'klemmmappen',
        u'klemmass'                    : u'klemmmaß',
        u'klemmasse'                   : u'klemmmaße',
        u'klemmassen'                  : u'klemmmaßen',
        u'klemmasses'                  : u'klemmmaßes',
        u'klemmaß'                     : u'klemmmaß',
        u'klemmaße'                    : u'klemmmaße',
        u'klemmaßen'                   : u'klemmmaßen',
        u'klemmaßes'                   : u'klemmmaßes',
        u'klettenhaftverschluß'        : u'klettenhaftverschluss',
        u'klettenverschluß'            : u'klettenverschluss',
        u'klettverschluß'              : u'klettverschluss',
        u'klitschenaß'                 : u'klitschenass',
        u'klitschnaß'                  : u'klitschnass',
        u'knacknuß'                    : u'knacknuss',
        u'knallaut'                    : u'knalllaut',
        u'knallila'                    : u'knalllila',
        u'kofferschloß'                : u'kofferschloss',
        u'kohlenstoffaser'             : u'kohlenstofffaser',
        u'kohlenstoffasern'            : u'kohlenstofffasern',
        u'kokillenguß'                 : u'kokillenguss',
        u'kokosnuß'                    : u'kokosnuss',
        u'kolanuß'                     : u'kolanuss',
        u'kollektivbewußtsein'         : u'kollektivbewusstsein',
        u'kollier'                     : u'collier',
        u'kolofonium'                  : u'kolophonium',
        u'kolofoniums'                 : u'kolophoniums',
        u'koloß'                       : u'koloss',
        u'koloßkopf'                   : u'kolosskopf',
        u'kombinationschloß'           : u'kombinationschloss',
        u'kommiß'                      : u'kommiss',
        u'kommißbrot'                  : u'kommissbrot',
        u'kommißbrote'                 : u'kommissbrote',
        u'kommißbroten'                : u'kommissbroten',
        u'kommißbrotes'                : u'kommissbrotes',
        u'kommißbrots'                 : u'kommissbrots',
        u'kommißstiefel'               : u'kommissstiefel',
        u'kommißstimme'                : u'kommissstimme',
        u'kommißstimmen'               : u'kommissstimmen',
        u'kommißzeit'                  : u'kommisszeit',
        u'kommunikee'                  : u'kommuniqué',
        u'kompaß'                      : u'kompass',
        u'kompaßnadel'                 : u'kompassnadel',
        u'kompaßnadeln'                : u'kompassnadeln',
        u'kompaßrose'                  : u'kompassrose',
        u'kompaßrosen'                 : u'kompassrosen',
        u'komplice'                    : u'komplize',
        u'komplicen'                   : u'komplizen',
        u'kompotteller'                : u'kompottteller',
        u'kompromiß'                   : u'kompromiss',
        u'kompromißbereit'             : u'kompromissbereit',
        u'kompromißbereite'            : u'kompromissbereite',
        u'kompromißbereitem'           : u'kompromissbereitem',
        u'kompromißbereiten'           : u'kompromissbereiten',
        u'kompromißbereiter'           : u'kompromissbereiter',
        u'kompromißbereitere'          : u'kompromissbereitere',
        u'kompromißbereiterem'         : u'kompromissbereiterem',
        u'kompromißbereiteren'         : u'kompromissbereiteren',
        u'kompromißbereiterer'         : u'kompromissbereiterer',
        u'kompromißbereiteres'         : u'kompromissbereiteres',
        u'kompromißbereites'           : u'kompromissbereites',
        u'kompromißbereiteste'         : u'kompromissbereiteste',
        u'kompromißbereitestem'        : u'kompromissbereitestem',
        u'kompromißbereitesten'        : u'kompromissbereitesten',
        u'kompromißbereitester'        : u'kompromissbereitester',
        u'kompromißbereitestes'        : u'kompromissbereitestes',
        u'kompromißbereitschaft'       : u'kompromissbereitschaft',
        u'kompromißfähig'              : u'kompromissfähig',
        u'kompromißfähigkeit'          : u'kompromissfähigkeit',
        u'kompromißkandidat'           : u'kompromisskandidat',
        u'kompromißkandidatin'         : u'kompromisskandidatin',
        u'kompromißler'                : u'kompromissler',
        u'kompromißlerin'              : u'kompromisslerin',
        u'kompromißlerinnen'           : u'kompromisslerinnen',
        u'kompromißlerisch'            : u'kompromisslerisch',
        u'kompromißlern'               : u'kompromisslern',
        u'kompromißlers'               : u'kompromisslers',
        u'kompromißlos'                : u'kompromisslos',
        u'kompromißlose'               : u'kompromisslose',
        u'kompromißlosem'              : u'kompromisslosem',
        u'kompromißlosen'              : u'kompromisslosen',
        u'kompromißloser'              : u'kompromissloser',
        u'kompromißlosere'             : u'kompromisslosere',
        u'kompromißloserem'            : u'kompromissloserem',
        u'kompromißloseren'            : u'kompromissloseren',
        u'kompromißloserer'            : u'kompromissloserer',
        u'kompromißloseres'            : u'kompromissloseres',
        u'kompromißloses'              : u'kompromissloses',
        u'kompromißloseste'            : u'kompromissloseste',
        u'kompromißlosestem'           : u'kompromisslosestem',
        u'kompromißlosesten'           : u'kompromisslosesten',
        u'kompromißlosester'           : u'kompromisslosester',
        u'kompromißlosestes'           : u'kompromisslosestes',
        u'kompromißlosigkeit'          : u'kompromisslosigkeit',
        u'kompromißlösung'             : u'kompromisslösung',
        u'kompromißversuch'            : u'kompromissversuch',
        u'kompromißvorschlag'          : u'kompromissvorschlag',
        u'kompromißvorschlage'         : u'kompromissvorschlage',
        u'kompromißvorschlages'        : u'kompromissvorschlages',
        u'kompromißvorschlags'         : u'kompromissvorschlags',
        u'kompromißvorschläge'         : u'kompromissvorschläge',
        u'kompromißvorschlägen'        : u'kompromissvorschlägen',
        u'komputer'                    : u'computer',
        u'komputern'                   : u'computern',
        u'komputers'                   : u'computers',
        u'komteß'                      : u'komtess',
        u'konferenzbeschluß'           : u'konferenzbeschluss',
        u'kongreß'                     : u'kongress',
        u'kongreßabgeordnete'          : u'kongressabgeordnete',
        u'kongreßabgeordnetem'         : u'kongressabgeordnetem',
        u'kongreßabgeordneten'         : u'kongressabgeordneten',
        u'kongreßabgeordneter'         : u'kongressabgeordneter',
        u'kongreßhalle'                : u'kongresshalle',
        u'kongreßmitgliedes'           : u'kongressmitgliedes',
        u'kongreßpolen'                : u'kongresspolen',
        u'kongreßpolens'               : u'kongresspolens',
        u'kongreßsaal'                 : u'kongresssaal',
        u'kongreßsaale'                : u'kongresssaale',
        u'kongreßsaales'               : u'kongresssaales',
        u'kongreßsaals'                : u'kongresssaals',
        u'kongreßstadt'                : u'kongressstadt',
        u'kongreßstädte'               : u'kongressstädte',
        u'kongreßstädten'              : u'kongressstädten',
        u'kongreßsäle'                 : u'kongresssäle',
        u'kongreßsälen'                : u'kongresssälen',
        u'kongreßteilnehmer'           : u'kongressteilnehmer',
        u'kongreßteilnehmerin'         : u'kongressteilnehmerin',
        u'kongreßteilnehmerinnen'      : u'kongressteilnehmerinnen',
        u'kongreßteilnehmern'          : u'kongressteilnehmern',
        u'kongreßteilnehmers'          : u'kongressteilnehmers',
        u'kongreßwahl'                 : u'kongresswahl',
        u'kongreßwahlen'               : u'kongresswahlen',
        u'kongreßzentrum'              : u'kongresszentrum',
        u'konkursprozeß'               : u'konkursprozess',
        u'kontainer'                   : u'container',
        u'kontainern'                  : u'containern',
        u'kontainers'                  : u'containers',
        u'kontrabaß'                   : u'kontrabass',
        u'kontraktabschluß'            : u'kontraktabschluss',
        u'kontrollampe'                : u'kontrolllampe',
        u'kontrollampen'               : u'kontrolllampen',
        u'kontrolleuchte'              : u'kontrollleuchte',
        u'kontrolleuchten'             : u'kontrollleuchten',
        u'kontrolliste'                : u'kontrollliste',
        u'konzernjahresabschluß'       : u'konzernjahresabschluss',
        u'kopfnuß'                     : u'kopfnuss',
        u'kopfschuß'                   : u'kopfschuss',
        u'koppelschloß'                : u'koppelschloss',
        u'kostenbewußt'                : u'kostenbewusst',
        u'kostenbewußtsein'            : u'kostenbewusstsein',
        u'koth'                        : u'kot',
        u'kothe'                       : u'kote',
        u'kothen'                      : u'koten',
        u'kothes'                      : u'kotes',
        u'koths'                       : u'kots',
        u'kottbus'                     : u'cottbus',
        u'kraß'                        : u'krass',
        u'kraßheit'                    : u'krassheit',
        u'kreisprozeß'                 : u'kreisprozess',
        u'krem'                        : u'creme',
        u'kremartig'                   : u'cremeartig',
        u'kremartige'                  : u'cremeartige',
        u'kremartigem'                 : u'cremeartigem',
        u'kremartigen'                 : u'cremeartigen',
        u'kremartiger'                 : u'cremeartiger',
        u'kremartiges'                 : u'cremeartiges',
        u'kreme'                       : u'creme',
        u'kremeartig'                  : u'cremeartig',
        u'kremeartige'                 : u'cremeartige',
        u'kremeartigem'                : u'cremeartigem',
        u'kremeartigen'                : u'cremeartigen',
        u'kremeartiger'                : u'cremeartiger',
        u'kremeartiges'                : u'cremeartiges',
        u'kremefarben'                 : u'cremefarben',
        u'kremefarbene'                : u'cremefarbene',
        u'kremefarbenem'               : u'cremefarbenem',
        u'kremefarbenen'               : u'cremefarbenen',
        u'kremefarbener'               : u'cremefarbener',
        u'kremefarbenes'               : u'cremefarbenes',
        u'kremefarbig'                 : u'cremefarbig',
        u'kremefarbige'                : u'cremefarbige',
        u'kremefarbigem'               : u'cremefarbigem',
        u'kremefarbigen'               : u'cremefarbigen',
        u'kremefarbiger'               : u'cremefarbiger',
        u'kremefarbiges'               : u'cremefarbiges',
        u'kremen'                      : u'cremen',
        u'kremend'                     : u'cremend',
        u'kremest'                     : u'cremest',
        u'kremet'                      : u'cremet',
        u'kremfarben'                  : u'cremefarben',
        u'kremfarbene'                 : u'cremefarbene',
        u'kremfarbenem'                : u'cremefarbenem',
        u'kremfarbenen'                : u'cremefarbenen',
        u'kremfarbener'                : u'cremefarbener',
        u'kremfarbenes'                : u'cremefarbenes',
        u'kremfarbig'                  : u'cremefarbig',
        u'kremfarbige'                 : u'cremefarbige',
        u'kremfarbigem'                : u'cremefarbigem',
        u'kremfarbigen'                : u'cremefarbigen',
        u'kremfarbiger'                : u'cremefarbiger',
        u'kremfarbiges'                : u'cremefarbiges',
        u'kremig'                      : u'cremig',
        u'kremige'                     : u'cremige',
        u'kremigem'                    : u'cremigem',
        u'kremigen'                    : u'cremigen',
        u'kremiger'                    : u'cremiger',
        u'kremigere'                   : u'cremigere',
        u'kremigerem'                  : u'cremigerem',
        u'kremigeren'                  : u'cremigeren',
        u'kremigerer'                  : u'cremigerer',
        u'kremigeres'                  : u'cremigeres',
        u'kremiges'                    : u'cremiges',
        u'kremigste'                   : u'cremigste',
        u'kremigstem'                  : u'cremigstem',
        u'kremigsten'                  : u'cremigsten',
        u'kremigster'                  : u'cremigster',
        u'kremigstes'                  : u'cremigstes',
        u'kremst'                      : u'cremst',
        u'kremt'                       : u'cremt',
        u'kremte'                      : u'cremte',
        u'kremten'                     : u'cremten',
        u'kremtest'                    : u'cremtest',
        u'kremtet'                     : u'cremtet',
        u'kreppapier'                  : u'krepppapier',
        u'kreppapiere'                 : u'krepppapiere',
        u'kreppapieren'                : u'krepppapieren',
        u'kreppapieres'                : u'krepppapieres',
        u'kreppapiers'                 : u'krepppapiers',
        u'kreutz'                      : u'kreuz',
        u'kreutze'                     : u'kreuze',
        u'kreutzen'                    : u'kreuzen',
        u'kreutzes'                    : u'kreuzes',
        u'kreuzas'                     : u'kreuzass',
        u'kreuzbandriß'                : u'kreuzbandriss',
        u'kriegsverbrecherprozeß'      : u'kriegsverbrecherprozess',
        u'krimgothisch'                : u'krimgotisch',
        u'kriminalprozeß'              : u'kriminalprozess',
        u'kristalleuchter'             : u'kristallleuchter',
        u'kristallinse'                : u'kristalllinse',
        u'kristallisationsprozeß'      : u'kristallisationsprozess',
        u'kristalluster'               : u'kristallluster',
        u'kristallustern'              : u'kristalllustern',
        u'kristallusters'              : u'kristalllusters',
        u'kristallüster'               : u'kristalllüster',
        u'kristallüstern'              : u'kristalllüstern',
        u'kristallüsters'              : u'kristalllüsters',
        u'krißlig'                     : u'krisslig',
        u'krißlige'                    : u'krisslige',
        u'krißligem'                   : u'krissligem',
        u'krißligen'                   : u'krissligen',
        u'krißliger'                   : u'krissliger',
        u'krißligere'                  : u'krissligere',
        u'krißligerem'                 : u'krissligerem',
        u'krißligeren'                 : u'krissligeren',
        u'krißligerer'                 : u'krissligerer',
        u'krißligeres'                 : u'krissligeres',
        u'krißliges'                   : u'krissliges',
        u'krißligste'                  : u'krissligste',
        u'krißligstem'                 : u'krissligstem',
        u'krißligsten'                 : u'krissligsten',
        u'krißligster'                 : u'krissligster',
        u'krißligstes'                 : u'krissligstes',
        u'kronprinzeßlich'             : u'kronprinzesslich',
        u'kroß'                        : u'kross',
        u'krummachen'                  : u'krummmachen',
        u'kulturausschuß'              : u'kulturausschuss',
        u'kunststoffaden'              : u'kunststofffaden',
        u'kunststoffadens'             : u'kunststofffadens',
        u'kunststoffaser'              : u'kunststofffaser',
        u'kunststoffenster'            : u'kunststofffenster',
        u'kunststoffenstern'           : u'kunststofffenstern',
        u'kunststoffensters'           : u'kunststofffensters',
        u'kunststoffolie'              : u'kunststofffolie',
        u'kunststoffolien'             : u'kunststofffolien',
        u'kunststoffspritzgußteil'     : u'kunststoffspritzgussteil',
        u'kunststoffspritzgußteile'    : u'kunststoffspritzgussteile',
        u'kunststoffspritzgußteilen'   : u'kunststoffspritzgussteilen',
        u'kunststoffspritzgußteiles'   : u'kunststoffspritzgussteiles',
        u'kunststoffspritzgußteils'    : u'kunststoffspritzgussteils',
        u'kunststoffäden'              : u'kunststofffäden',
        u'kupee'                       : u'coupé',
        u'kurzfaßte'                   : u'kurzfasste',
        u'kurzfaßten'                  : u'kurzfassten',
        u'kurzfaßtest'                 : u'kurzfasstest',
        u'kurzfaßtet'                  : u'kurzfasstet',
        u'kurzpaß'                     : u'kurzpass',
        u'kurzpaßspiel'                : u'kurzpassspiel',
        u'kurzpaßspieles'              : u'kurzpassspieles',
        u'kurzpaßspiels'               : u'kurzpassspiels',
        u'kurzschloß'                  : u'kurzschloss',
        u'kurzschloßt'                 : u'kurzschlosst',
        u'kurzschluß'                  : u'kurzschluss',
        u'kurzschlußhandlung'          : u'kurzschlusshandlung',
        u'kurzschlußhandlungen'        : u'kurzschlusshandlungen',
        u'kurzschlußlüftung'           : u'kurzschlusslüftung',
        u'kurzschlußlüftungen'         : u'kurzschlusslüftungen',
        u'kurzschlußreaktion'          : u'kurzschlussreaktion',
        u'kurzschlußreaktionen'        : u'kurzschlussreaktionen',
        u'kurzschlußtat'               : u'kurzschlusstat',
        u'kurzschlußtaten'             : u'kurzschlusstaten',
        u'kuß'                         : u'kuss',
        u'kußecht'                     : u'kussecht',
        u'kußechte'                    : u'kussechte',
        u'kußechtem'                   : u'kussechtem',
        u'kußechten'                   : u'kussechten',
        u'kußechter'                   : u'kussechter',
        u'kußechtere'                  : u'kussechtere',
        u'kußechterem'                 : u'kussechterem',
        u'kußechteren'                 : u'kussechteren',
        u'kußechterer'                 : u'kussechterer',
        u'kußechteres'                 : u'kussechteres',
        u'kußechtes'                   : u'kussechtes',
        u'kußechteste'                 : u'kussechteste',
        u'kußechtestem'                : u'kussechtestem',
        u'kußechtesten'                : u'kussechtesten',
        u'kußechtester'                : u'kussechtester',
        u'kußechtestes'                : u'kussechtestes',
        u'kußhand'                     : u'kusshand',
        u'kußhändchen'                 : u'kusshändchen',
        u'kußhände'                    : u'kusshände',
        u'kußhänden'                   : u'kusshänden',
        u'kußkrankheit'                : u'kusskrankheit',
        u'kußkrankheiten'              : u'kusskrankheiten',
        u'kußzahl'                     : u'kusszahl',
        u'kußzahlen'                   : u'kusszahlen',
        u'kwaß'                        : u'kwass',
        u'kämmaschine'                 : u'kämmmaschine',
        u'känguruh'                    : u'känguru',
        u'känguruhapfel'               : u'känguruapfel',
        u'känguruhdorn'                : u'kängurudorn',
        u'känguruhfleisch'             : u'kängurufleisch',
        u'känguruhfleische'            : u'kängurufleische',
        u'känguruhfleisches'           : u'kängurufleisches',
        u'känguruhfleischs'            : u'kängurufleischs',
        u'känguruhinsel'               : u'känguruinsel',
        u'känguruhleder'               : u'känguruleder',
        u'känguruhledern'              : u'känguruledern',
        u'känguruhleders'              : u'känguruleders',
        u'känguruhmaus'                : u'kängurumaus',
        u'känguruhratte'               : u'kängururatte',
        u'känguruhs'                   : u'kängurus',
        u'königsschloß'                : u'königsschloss',
        u'königthum'                   : u'königtum',
        u'königthums'                  : u'königtums',
        u'königthümer'                 : u'königtümer',
        u'königthümern'                : u'königtümern',
        u'körperbewußtsein'            : u'körperbewusstsein',
        u'küßchen'                     : u'küsschen',
        u'küßchens'                    : u'küsschens',
        u'küßt'                        : u'küsst',
        u'küßte'                       : u'küsste',
        u'küßten'                      : u'küssten',
        u'küßtest'                     : u'küsstest',
        u'küßtet'                      : u'küsstet',
        u'ladenschluß'                 : u'ladenschluss',
        u'ladenschlußgesetz'           : u'ladenschlussgesetz',
        u'ladenschlußzeit'             : u'ladenschlusszeit',
        u'ladies'                      : u'ladys',
        u'lambertsnuß'                 : u'lambertsnuss',
        u'lamellenverschluß'           : u'lamellenverschluss',
        u'landcadett'                  : u'landkadett',
        u'landcadetten'                : u'landkadetten',
        u'landesverrath'               : u'landesverrat',
        u'landesverrathe'              : u'landesverrate',
        u'landesverrathes'             : u'landesverrates',
        u'landesverraths'              : u'landesverrats',
        u'landschloß'                  : u'landschloss',
        u'langstengelig'               : u'langstängelig',
        u'langstengelige'              : u'langstängelige',
        u'langstengeligem'             : u'langstängeligem',
        u'langstengeligen'             : u'langstängeligen',
        u'langstengeliger'             : u'langstängeliger',
        u'langstengeliges'             : u'langstängeliges',
        u'langstenglig'                : u'langstänglig',
        u'langstenglige'               : u'langstänglige',
        u'langstengligem'              : u'langstängligem',
        u'langstengligen'              : u'langstängligen',
        u'langstengliger'              : u'langstängliger',
        u'langstengliges'              : u'langstängliges',
        u'lattenschuß'                 : u'lattenschuss',
        u'laufpaß'                     : u'laufpass',
        u'laugenfaß'                   : u'laugenfass',
        u'laß'                         : u'lass',
        u'laßt'                        : u'lasst',
        u'lebensgenuß'                 : u'lebensgenuss',
        u'lebensprozeß'                : u'lebensprozess',
        u'leberabszeß'                 : u'leberabszess',
        u'legir'                       : u'legier',
        u'legire'                      : u'legiere',
        u'legiren'                     : u'legieren',
        u'legirend'                    : u'legierend',
        u'legirest'                    : u'legierest',
        u'legiret'                     : u'legieret',
        u'legirst'                     : u'legierst',
        u'legirt'                      : u'legiert',
        u'legirte'                     : u'legierte',
        u'legirten'                    : u'legierten',
        u'legirtest'                   : u'legiertest',
        u'legirtet'                    : u'legiertet',
        u'lehrabschluß'                : u'lehrabschluss',
        u'lehrabschlußzeugnis'         : u'lehrabschlusszeugnis',
        u'lehrprozeß'                  : u'lehrprozess',
        u'leichenblaß'                 : u'leichenblass',
        u'leichenpaß'                  : u'leichenpass',
        u'leistungspotential'          : u'leistungspotenzial',
        u'lenkradschloß'               : u'lenkradschloss',
        u'lenkungsausschuß'            : u'lenkungsausschuss',
        u'lernprozeß'                  : u'lernprozess',
        u'liebesschloß'                : u'liebesschloss',
        u'lieblingsrößlein'            : u'lieblingsrösslein',
        u'lieferengpaß'                : u'lieferengpass',
        u'linksschuß'                  : u'linksschuss',
        u'lithograph'                  : u'lithograf',
        u'lithographien'               : u'lithografien',
        u'lithographierten'            : u'lithografierten',
        u'litteratur'                  : u'literatur',
        u'lobbies'                     : u'lobbys',
        u'locativ'                     : u'lokativ',
        u'locative'                    : u'lokative',
        u'locativen'                   : u'lokativen',
        u'locativs'                    : u'lokativs',
        u'lorbaß'                      : u'lorbass',
        u'loslaßt'                     : u'loslasst',
        u'losläßt'                     : u'loslässt',
        u'losschoß'                    : u'losschoss',
        u'losschoßt'                   : u'losschosst',
        u'loth'                        : u'lot',
        u'lothe'                       : u'lote',
        u'lothen'                      : u'loten',
        u'lothes'                      : u'lotes',
        u'loths'                       : u'lots',
        u'luftschiffahrt'              : u'luftschifffahrt',
        u'luftschloß'                  : u'luftschloss',
        u'lukmanierpaß'                : u'lukmanierpass',
        u'lustschloß'                  : u'lustschloss',
        u'läßlich'                     : u'lässlich',
        u'läßliche'                    : u'lässliche',
        u'läßlichem'                   : u'lässlichem',
        u'läßlichen'                   : u'lässlichen',
        u'läßlicher'                   : u'lässlicher',
        u'läßlichere'                  : u'lässlichere',
        u'läßlicherem'                 : u'lässlicherem',
        u'läßlicheren'                 : u'lässlicheren',
        u'läßlicherer'                 : u'lässlicherer',
        u'läßlicheres'                 : u'lässlicheres',
        u'läßliches'                   : u'lässliches',
        u'läßlichste'                  : u'lässlichste',
        u'läßlichstem'                 : u'lässlichstem',
        u'läßlichsten'                 : u'lässlichsten',
        u'läßlichster'                 : u'lässlichster',
        u'läßlichstes'                 : u'lässlichstes',
        u'läßt'                        : u'lässt',
        u'löthe'                       : u'löte',
        u'löthen'                      : u'löten',
        u'löthend'                     : u'lötend',
        u'löthest'                     : u'lötest',
        u'löthet'                      : u'lötet',
        u'löthete'                     : u'lötete',
        u'lötheten'                    : u'löteten',
        u'löthetest'                   : u'lötetest',
        u'löthetet'                    : u'lötetet',
        u'lötschenpaß'                 : u'lötschenpass',
        u'macedonien'                  : u'mazedonien',
        u'macedoniens'                 : u'mazedoniens',
        u'macedonisch'                 : u'mazedonisch',
        u'macedonische'                : u'mazedonische',
        u'macedonischem'               : u'mazedonischem',
        u'macedonischen'               : u'mazedonischen',
        u'macedonischer'               : u'mazedonischer',
        u'macedonisches'               : u'mazedonisches',
        u'machtmißbrauch'              : u'machtmissbrauch',
        u'machtmißbrauche'             : u'machtmissbrauche',
        u'machtmißbrauches'            : u'machtmissbrauches',
        u'machtmißbrauchs'             : u'machtmissbrauchs',
        u'machtmißbräuche'             : u'machtmissbräuche',
        u'machtmißbräuchen'            : u'machtmissbräuchen',
        u'maerz'                       : u'märz',
        u'maffia'                      : u'mafia',
        u'maffiaboss'                  : u'mafiaboss',
        u'maffiabosse'                 : u'mafiabosse',
        u'maffiabossen'                : u'mafiabossen',
        u'maffiabosses'                : u'mafiabosses',
        u'maffiaboß'                   : u'mafiaboss',
        u'maffias'                     : u'mafias',
        u'mafiaboß'                    : u'mafiaboss',
        u'magen-darm-katarr'           : u'magen-darm-katarrh',
        u'magen-darm-katarre'          : u'magen-darm-katarrhe',
        u'magen-darm-katarren'         : u'magen-darm-katarrhen',
        u'magen-darm-katarres'         : u'magen-darm-katarrhes',
        u'magen-darm-katarrs'          : u'magen-darm-katarrhs',
        u'magistratsbeschluß'          : u'magistratsbeschluss',
        u'majonäse'                    : u'mayonnaise',
        u'majoritätsbeschluß'          : u'majoritätsbeschluss',
        u'malojapaß'                   : u'malojapass',
        u'maläse'                      : u'malaise',
        u'maläsen'                     : u'malaisen',
        u'mammutprozeß'                : u'mammutprozess',
        u'markenbewußt'                : u'markenbewusst',
        u'markenbewußtsein'            : u'markenbewusstsein',
        u'marschkompaß'                : u'marschkompass',
        u'masculina'                   : u'maskulina',
        u'masculinum'                  : u'maskulinum',
        u'masculinums'                 : u'maskulinums',
        u'masurka'                     : u'mazurka',
        u'masurkas'                    : u'mazurkas',
        u'masurken'                    : u'mazurken',
        u'matrosenschenke'             : u'matrosenschänke',
        u'maulkorberlaß'               : u'maulkorberlass',
        u'maulthier'                   : u'maultier',
        u'maulthiere'                  : u'maultiere',
        u'maulthieren'                 : u'maultieren',
        u'maulthieres'                 : u'maultieres',
        u'maulthiers'                  : u'maultiers',
        u'medicin'                     : u'medizin',
        u'medicinen'                   : u'medizinen',
        u'meeranschluß'                : u'meeranschluss',
        u'mehrentheils'                : u'mehrenteils',
        u'meinungsbildungsprozeß'      : u'meinungsbildungsprozess',
        u'meldeschluß'                 : u'meldeschluss',
        u'meniskusriß'                 : u'meniskusriss',
        u'menschenhaß'                 : u'menschenhass',
        u'mercur'                      : u'merkur',
        u'mercurs'                     : u'merkurs',
        u'metallampe'                  : u'metalllampe',
        u'metallegierung'              : u'metalllegierung',
        u'metallegierungen'            : u'metalllegierungen',
        u'metalleuchte'                : u'metallleuchte',
        u'metalleuchter'               : u'metallleuchter',
        u'metallfaß'                   : u'metallfass',
        u'metalloïde'                  : u'metalloide',
        u'meyn'                        : u'mein',
        u'meyne'                       : u'meine',
        u'meynen'                      : u'meinen',
        u'meynend'                     : u'meinend',
        u'meynest'                     : u'meinest',
        u'meynet'                      : u'meinet',
        u'meynst'                      : u'meinst',
        u'meynt'                       : u'meint',
        u'meynte'                      : u'meinte',
        u'meynten'                     : u'meinten',
        u'meyntest'                    : u'meintest',
        u'meyntet'                     : u'meintet',
        u'meßbar'                      : u'messbar',
        u'meßbare'                     : u'messbare',
        u'meßbarem'                    : u'messbarem',
        u'meßbaren'                    : u'messbaren',
        u'meßbarer'                    : u'messbarer',
        u'meßbarere'                   : u'messbarere',
        u'meßbarerem'                  : u'messbarerem',
        u'meßbareren'                  : u'messbareren',
        u'meßbarerer'                  : u'messbarerer',
        u'meßbareres'                  : u'messbareres',
        u'meßbares'                    : u'messbares',
        u'meßbarkeit'                  : u'messbarkeit',
        u'meßbarste'                   : u'messbarste',
        u'meßbarstem'                  : u'messbarstem',
        u'meßbarsten'                  : u'messbarsten',
        u'meßbarster'                  : u'messbarster',
        u'meßbarstes'                  : u'messbarstes',
        u'meßbecher'                   : u'messbecher',
        u'meßbechern'                  : u'messbechern',
        u'meßbechers'                  : u'messbechers',
        u'meßbetrag'                   : u'messbetrag',
        u'meßbetrage'                  : u'messbetrage',
        u'meßbetrages'                 : u'messbetrages',
        u'meßbetrags'                  : u'messbetrags',
        u'meßbeträge'                  : u'messbeträge',
        u'meßbeträgen'                 : u'messbeträgen',
        u'meßbuch'                     : u'messbuch',
        u'meßbuche'                    : u'messbuche',
        u'meßbuches'                   : u'messbuches',
        u'meßbuchs'                    : u'messbuchs',
        u'meßbücher'                   : u'messbücher',
        u'meßbüchern'                  : u'messbüchern',
        u'meßdiener'                   : u'messdiener',
        u'meßdienerin'                 : u'messdienerin',
        u'meßdienerinnen'              : u'messdienerinnen',
        u'meßdienern'                  : u'messdienern',
        u'meßdieners'                  : u'messdieners',
        u'meßergebnis'                 : u'messergebnis',
        u'meßergebnisse'               : u'messergebnisse',
        u'meßergebnissen'              : u'messergebnissen',
        u'meßergebnisses'              : u'messergebnisses',
        u'meßfehler'                   : u'messfehler',
        u'meßfehlern'                  : u'messfehlern',
        u'meßfehlers'                  : u'messfehlers',
        u'meßfeier'                    : u'messfeier',
        u'meßfeiern'                   : u'messfeiern',
        u'meßgenauheit'                : u'messgenauheit',
        u'meßgenauigkeit'              : u'messgenauigkeit',
        u'meßgerät'                    : u'messgerät',
        u'meßgeräte'                   : u'messgeräte',
        u'meßgeräten'                  : u'messgeräten',
        u'meßgerätes'                  : u'messgerätes',
        u'meßgeräts'                   : u'messgeräts',
        u'meßgewand'                   : u'messgewand',
        u'meßgewandes'                 : u'messgewandes',
        u'meßgewands'                  : u'messgewands',
        u'meßinstrument'               : u'messinstrument',
        u'meßinstrumenten'             : u'messinstrumenten',
        u'meßlatte'                    : u'messlatte',
        u'meßlatten'                   : u'messlatten',
        u'meßmittel'                   : u'messmittel',
        u'meßmitteln'                  : u'messmitteln',
        u'meßmittels'                  : u'messmittels',
        u'meßner'                      : u'messner',
        u'meßnern'                     : u'messnern',
        u'meßners'                     : u'messners',
        u'meßschieber'                 : u'messschieber',
        u'meßschiebern'                : u'messschiebern',
        u'meßschiebers'                : u'messschiebers',
        u'meßstation'                  : u'messstation',
        u'meßstationen'                : u'messstationen',
        u'meßstelle'                   : u'messstelle',
        u'meßstellen'                  : u'messstellen',
        u'meßstipendien'               : u'messstipendien',
        u'meßstipendium'               : u'messstipendium',
        u'meßstipendiums'              : u'messstipendiums',
        u'meßt'                        : u'messt',
        u'meßtechnik'                  : u'messtechnik',
        u'meßverfahren'                : u'messverfahren',
        u'meßverfahrens'               : u'messverfahrens',
        u'meßwagen'                    : u'messwagen',
        u'meßwagens'                   : u'messwagens',
        u'meßwein'                     : u'messwein',
        u'meßweine'                    : u'messweine',
        u'meßweinen'                   : u'messweinen',
        u'meßweines'                   : u'messweines',
        u'meßweins'                    : u'messweins',
        u'meßwerk'                     : u'messwerk',
        u'meßwerke'                    : u'messwerke',
        u'meßwerken'                   : u'messwerken',
        u'meßwerkes'                   : u'messwerkes',
        u'meßwerks'                    : u'messwerks',
        u'meßwert'                     : u'messwert',
        u'meßwerte'                    : u'messwerte',
        u'meßwerten'                   : u'messwerten',
        u'meßwertes'                   : u'messwertes',
        u'meßwerts'                    : u'messwerts',
        u'midlife-crisis'              : u'midlife-crisis',
        u'mikrophon'                   : u'mikrofon',
        u'mikrophone'                  : u'mikrofone',
        u'milchgebiß'                  : u'milchgebiss',
        u'millenium'                   : u'millennium',
        u'milzriß'                     : u'milzriss',
        u'miteßt'                      : u'mitesst',
        u'mitißt'                      : u'mitisst',
        u'mitriß'                      : u'mitriss',
        u'mitrißt'                     : u'mitrisst',
        u'miß'                         : u'miss',
        u'miß-'                        : u'miss-',
        u'mißachte'                    : u'missachte',
        u'mißachten'                   : u'missachten',
        u'mißachtend'                  : u'missachtend',
        u'mißachtest'                  : u'missachtest',
        u'mißachtet'                   : u'missachtet',
        u'mißachtete'                  : u'missachtete',
        u'mißachteten'                 : u'missachteten',
        u'mißachtetest'                : u'missachtetest',
        u'mißachtetet'                 : u'missachtetet',
        u'mißachtung'                  : u'missachtung',
        u'mißachtungen'                : u'missachtungen',
        u'mißbefinden'                 : u'missbefinden',
        u'mißbefindlichkeit'           : u'missbefindlichkeit',
        u'mißbehagen'                  : u'missbehagen',
        u'mißbehagens'                 : u'missbehagens',
        u'mißbehaglich'                : u'missbehaglich',
        u'mißbeschaffen'               : u'missbeschaffen',
        u'mißbeschaffenheit'           : u'missbeschaffenheit',
        u'mißbildung'                  : u'missbildung',
        u'mißbildungen'                : u'missbildungen',
        u'mißbillig'                   : u'missbillig',
        u'mißbillige'                  : u'missbillige',
        u'mißbilligen'                 : u'missbilligen',
        u'mißbilligend'                : u'missbilligend',
        u'mißbilligende'               : u'missbilligende',
        u'mißbilligest'                : u'missbilligest',
        u'mißbilliget'                 : u'missbilliget',
        u'mißbilligst'                 : u'missbilligst',
        u'mißbilligt'                  : u'missbilligt',
        u'mißbilligte'                 : u'missbilligte',
        u'mißbilligten'                : u'missbilligten',
        u'mißbilligtest'               : u'missbilligtest',
        u'mißbilligtet'                : u'missbilligtet',
        u'mißbilligung'                : u'missbilligung',
        u'mißbilligungen'              : u'missbilligungen',
        u'mißbilligungsantrag'         : u'missbilligungsantrag',
        u'mißbilligungsantrage'        : u'missbilligungsantrage',
        u'mißbilligungsantrages'       : u'missbilligungsantrages',
        u'mißbilligungsantrags'        : u'missbilligungsantrags',
        u'mißbilligungsanträge'        : u'missbilligungsanträge',
        u'mißbilligungsanträgen'       : u'missbilligungsanträgen',
        u'mißbrauch'                   : u'missbrauch',
        u'mißbrauche'                  : u'missbrauche',
        u'mißbrauchen'                 : u'missbrauchen',
        u'mißbrauchend'                : u'missbrauchend',
        u'mißbrauches'                 : u'missbrauches',
        u'mißbrauchest'                : u'missbrauchest',
        u'mißbrauchet'                 : u'missbrauchet',
        u'mißbrauchs'                  : u'missbrauchs',
        u'mißbrauchskonzept'           : u'missbrauchskonzept',
        u'mißbrauchst'                 : u'missbrauchst',
        u'mißbraucht'                  : u'missbraucht',
        u'mißbrauchte'                 : u'missbrauchte',
        u'mißbrauchten'                : u'missbrauchten',
        u'mißbrauchtest'               : u'missbrauchtest',
        u'mißbrauchtet'                : u'missbrauchtet',
        u'mißbräuche'                  : u'missbräuche',
        u'mißbräuchen'                 : u'missbräuchen',
        u'mißbräuchlich'               : u'missbräuchlich',
        u'mißbräuchliche'              : u'missbräuchliche',
        u'mißbräuchlichem'             : u'missbräuchlichem',
        u'mißbräuchlichen'             : u'missbräuchlichen',
        u'mißbräuchlicher'             : u'missbräuchlicher',
        u'mißbräuchlicherweise'        : u'missbräuchlicherweise',
        u'mißbräuchliches'             : u'missbräuchliches',
        u'mißdeutbar'                  : u'missdeutbar',
        u'mißdeuten'                   : u'missdeuten',
        u'mißdeutende'                 : u'missdeutende',
        u'mißdeutet'                   : u'missdeutet',
        u'mißdeutung'                  : u'missdeutung',
        u'mißdeutungen'                : u'missdeutungen',
        u'mißempfindung'               : u'missempfindung',
        u'mißerfolg'                   : u'misserfolg',
        u'mißerfolge'                  : u'misserfolge',
        u'mißerfolgen'                 : u'misserfolgen',
        u'mißerfolges'                 : u'misserfolges',
        u'mißerfolgs'                  : u'misserfolgs',
        u'mißernte'                    : u'missernte',
        u'mißernten'                   : u'missernten',
        u'mißfall'                     : u'missfall',
        u'mißfalle'                    : u'missfalle',
        u'mißfallen'                   : u'missfallen',
        u'mißfallend'                  : u'missfallend',
        u'mißfallenskundgebung'        : u'missfallenskundgebung',
        u'mißfallensäußerung'          : u'missfallensäußerung',
        u'mißfallest'                  : u'missfallest',
        u'mißfallet'                   : u'missfallet',
        u'mißfallt'                    : u'missfallt',
        u'mißfarbe'                    : u'missfarbe',
        u'mißfarben'                   : u'missfarben',
        u'mißfarbig'                   : u'missfarbig',
        u'mißfarbige'                  : u'missfarbige',
        u'mißfiel'                     : u'missfiel',
        u'mißfiele'                    : u'missfiele',
        u'mißfielen'                   : u'missfielen',
        u'mißfielest'                  : u'missfielest',
        u'mißfielet'                   : u'missfielet',
        u'mißfielst'                   : u'missfielst',
        u'mißfielt'                    : u'missfielt',
        u'mißfällig'                   : u'missfällig',
        u'mißfällst'                   : u'missfällst',
        u'mißfällt'                    : u'missfällt',
        u'mißgebildet'                 : u'missgebildet',
        u'mißgeburt'                   : u'missgeburt',
        u'mißgeburten'                 : u'missgeburten',
        u'mißgelaunt'                  : u'missgelaunt',
        u'mißgelaunte'                 : u'missgelaunte',
        u'mißgelauntem'                : u'missgelauntem',
        u'mißgelaunten'                : u'missgelaunten',
        u'mißgelaunter'                : u'missgelaunter',
        u'mißgelauntere'               : u'missgelauntere',
        u'mißgelaunterem'              : u'missgelaunterem',
        u'mißgelaunteren'              : u'missgelaunteren',
        u'mißgelaunterer'              : u'missgelaunterer',
        u'mißgelaunteres'              : u'missgelaunteres',
        u'mißgelauntes'                : u'missgelauntes',
        u'mißgelaunteste'              : u'missgelaunteste',
        u'mißgelauntestem'             : u'missgelauntestem',
        u'mißgelauntesten'             : u'missgelauntesten',
        u'mißgelauntester'             : u'missgelauntester',
        u'mißgelauntestes'             : u'missgelauntestes',
        u'mißgelauntheit'              : u'missgelauntheit',
        u'mißgeschaffenen'             : u'missgeschaffenen',
        u'mißgeschick'                 : u'missgeschick',
        u'mißgeschicke'                : u'missgeschicke',
        u'mißgeschicken'               : u'missgeschicken',
        u'mißgeschickes'               : u'missgeschickes',
        u'mißgeschicks'                : u'missgeschicks',
        u'mißgestalt'                  : u'missgestalt',
        u'mißgestaltet'                : u'missgestaltet',
        u'mißgestimmt'                 : u'missgestimmt',
        u'mißgewachsen'                : u'missgewachsen',
        u'mißglück'                    : u'missglück',
        u'mißglücke'                   : u'missglücke',
        u'mißglücken'                  : u'missglücken',
        u'mißglückend'                 : u'missglückend',
        u'mißglückest'                 : u'missglückest',
        u'mißglücket'                  : u'missglücket',
        u'mißglückst'                  : u'missglückst',
        u'mißglückt'                   : u'missglückt',
        u'mißglückte'                  : u'missglückte',
        u'mißglücktem'                 : u'missglücktem',
        u'mißglückten'                 : u'missglückten',
        u'mißglückter'                 : u'missglückter',
        u'mißglücktere'                : u'missglücktere',
        u'mißglückterem'               : u'missglückterem',
        u'mißglückteren'               : u'missglückteren',
        u'mißglückterer'               : u'missglückterer',
        u'mißglückteres'               : u'missglückteres',
        u'mißglücktes'                 : u'missglücktes',
        u'mißglücktest'                : u'missglücktest',
        u'mißglückteste'               : u'missglückteste',
        u'mißglücktestem'              : u'missglücktestem',
        u'mißglücktesten'              : u'missglücktesten',
        u'mißglücktester'              : u'missglücktester',
        u'mißglücktestes'              : u'missglücktestes',
        u'mißglücktet'                 : u'missglücktet',
        u'mißgriff'                    : u'missgriff',
        u'mißgriffe'                   : u'missgriffe',
        u'mißgriffen'                  : u'missgriffen',
        u'mißgriffs'                   : u'missgriffs',
        u'mißgunst'                    : u'missgunst',
        u'mißgönne'                    : u'missgönne',
        u'mißgönnen'                   : u'missgönnen',
        u'mißgönnt'                    : u'missgönnt',
        u'mißgünstig'                  : u'missgünstig',
        u'mißgünstige'                 : u'missgünstige',
        u'mißgünstigem'                : u'missgünstigem',
        u'mißgünstigen'                : u'missgünstigen',
        u'mißgünstiger'                : u'missgünstiger',
        u'mißgünstigere'               : u'missgünstigere',
        u'mißgünstigerem'              : u'missgünstigerem',
        u'mißgünstigeren'              : u'missgünstigeren',
        u'mißgünstigerer'              : u'missgünstigerer',
        u'mißgünstigeres'              : u'missgünstigeres',
        u'mißgünstiges'                : u'missgünstiges',
        u'mißgünstigste'               : u'missgünstigste',
        u'mißgünstigstem'              : u'missgünstigstem',
        u'mißgünstigsten'              : u'missgünstigsten',
        u'mißgünstigster'              : u'missgünstigster',
        u'mißgünstigstes'              : u'missgünstigstes',
        u'mißhandel'                   : u'misshandel',
        u'mißhandele'                  : u'misshandele',
        u'mißhandelest'                : u'misshandelest',
        u'mißhandelet'                 : u'misshandelet',
        u'mißhandeln'                  : u'misshandeln',
        u'mißhandelnd'                 : u'misshandelnd',
        u'mißhandelst'                 : u'misshandelst',
        u'mißhandelt'                  : u'misshandelt',
        u'mißhandelte'                 : u'misshandelte',
        u'mißhandelten'                : u'misshandelten',
        u'mißhandeltes'                : u'misshandeltes',
        u'mißhandeltest'               : u'misshandeltest',
        u'mißhandeltet'                : u'misshandeltet',
        u'mißhandle'                   : u'misshandle',
        u'mißhandlest'                 : u'misshandlest',
        u'mißhandlet'                  : u'misshandlet',
        u'mißhandlung'                 : u'misshandlung',
        u'mißhandlungen'               : u'misshandlungen',
        u'mißheirat'                   : u'missheirat',
        u'mißhellig'                   : u'misshellig',
        u'mißhelligkeit'               : u'misshelligkeit',
        u'mißkennung'                  : u'misskennung',
        u'mißkennungen'                : u'misskennungen',
        u'mißklang'                    : u'missklang',
        u'mißklangs'                   : u'missklangs',
        u'mißklänge'                   : u'missklänge',
        u'mißklängen'                  : u'missklängen',
        u'mißkommunikation'            : u'misskommunikation',
        u'mißkommunikationen'          : u'misskommunikationen',
        u'mißkredit'                   : u'misskredit',
        u'mißlang'                     : u'misslang',
        u'mißlangen'                   : u'misslangen',
        u'mißlangst'                   : u'misslangst',
        u'mißlangt'                    : u'misslangt',
        u'mißlaunig'                   : u'misslaunig',
        u'mißlaut'                     : u'misslaut',
        u'mißleiten'                   : u'missleiten',
        u'mißleitung'                  : u'missleitung',
        u'mißlich'                     : u'misslich',
        u'mißliche'                    : u'missliche',
        u'mißlichem'                   : u'misslichem',
        u'mißlichen'                   : u'misslichen',
        u'mißlicher'                   : u'misslicher',
        u'mißlichere'                  : u'misslichere',
        u'mißlicherem'                 : u'misslicherem',
        u'mißlicheren'                 : u'misslicheren',
        u'mißlicherer'                 : u'misslicherer',
        u'mißlicheres'                 : u'misslicheres',
        u'mißliches'                   : u'missliches',
        u'mißlichkeit'                 : u'misslichkeit',
        u'mißlichste'                  : u'misslichste',
        u'mißlichstem'                 : u'misslichstem',
        u'mißlichsten'                 : u'misslichsten',
        u'mißlichster'                 : u'misslichster',
        u'mißlichstes'                 : u'misslichstes',
        u'mißliebig'                   : u'missliebig',
        u'mißliebige'                  : u'missliebige',
        u'mißliebigem'                 : u'missliebigem',
        u'mißliebigen'                 : u'missliebigen',
        u'mißliebiger'                 : u'missliebiger',
        u'mißliebigere'                : u'missliebigere',
        u'mißliebigerem'               : u'missliebigerem',
        u'mißliebigeren'               : u'missliebigeren',
        u'mißliebigerer'               : u'missliebigerer',
        u'mißliebigeres'               : u'missliebigeres',
        u'mißliebiges'                 : u'missliebiges',
        u'mißliebigkeit'               : u'missliebigkeit',
        u'mißliebigkeiten'             : u'missliebigkeiten',
        u'mißliebigste'                : u'missliebigste',
        u'mißliebigstem'               : u'missliebigstem',
        u'mißliebigsten'               : u'missliebigsten',
        u'mißliebigster'               : u'missliebigster',
        u'mißliebigstes'               : u'missliebigstes',
        u'mißling'                     : u'missling',
        u'mißlinge'                    : u'misslinge',
        u'mißlingen'                   : u'misslingen',
        u'mißlingend'                  : u'misslingend',
        u'mißlingest'                  : u'misslingest',
        u'mißlinget'                   : u'misslinget',
        u'mißlingst'                   : u'misslingst',
        u'mißlingt'                    : u'misslingt',
        u'mißlungen'                   : u'misslungen',
        u'mißlungenen'                 : u'misslungenen',
        u'mißlänge'                    : u'misslänge',
        u'mißlängen'                   : u'misslängen',
        u'mißlängest'                  : u'misslängest',
        u'mißlänget'                   : u'misslänget',
        u'mißlängst'                   : u'misslängst',
        u'mißlängt'                    : u'misslängt',
        u'mißmanagement'               : u'missmanagement',
        u'mißmanagements'              : u'missmanagements',
        u'mißmut'                      : u'missmut',
        u'mißmute'                     : u'missmute',
        u'mißmutes'                    : u'missmutes',
        u'mißmutig'                    : u'missmutig',
        u'mißmutige'                   : u'missmutige',
        u'mißmutigem'                  : u'missmutigem',
        u'mißmutigen'                  : u'missmutigen',
        u'mißmutiger'                  : u'missmutiger',
        u'mißmutigere'                 : u'missmutigere',
        u'mißmutigerem'                : u'missmutigerem',
        u'mißmutigeren'                : u'missmutigeren',
        u'mißmutigerer'                : u'missmutigerer',
        u'mißmutigeres'                : u'missmutigeres',
        u'mißmutiges'                  : u'missmutiges',
        u'mißmutigste'                 : u'missmutigste',
        u'mißmutigstem'                : u'missmutigstem',
        u'mißmutigsten'                : u'missmutigsten',
        u'mißmutigster'                : u'missmutigster',
        u'mißmutigstes'                : u'missmutigstes',
        u'mißmuts'                     : u'missmuts',
        u'mißrat'                      : u'missrat',
        u'mißrate'                     : u'missrate',
        u'mißraten'                    : u'missraten',
        u'mißratend'                   : u'missratend',
        u'mißratest'                   : u'missratest',
        u'mißratet'                    : u'missratet',
        u'mißriet'                     : u'missriet',
        u'mißriete'                    : u'missriete',
        u'mißrieten'                   : u'missrieten',
        u'mißrietest'                  : u'missrietest',
        u'mißrietet'                   : u'missrietet',
        u'mißrietst'                   : u'missrietst',
        u'mißrät'                      : u'missrät',
        u'mißrätst'                    : u'missrätst',
        u'mißstand'                    : u'missstand',
        u'mißstande'                   : u'missstande',
        u'mißstandes'                  : u'missstandes',
        u'mißstands'                   : u'missstands',
        u'mißstimmung'                 : u'missstimmung',
        u'mißstände'                   : u'missstände',
        u'mißständen'                  : u'missständen',
        u'mißt'                        : u'misst',
        u'mißte'                       : u'misste',
        u'mißten'                      : u'missten',
        u'mißtest'                     : u'misstest',
        u'mißtet'                      : u'misstet',
        u'mißton'                      : u'misston',
        u'mißtrau'                     : u'misstrau',
        u'mißtraue'                    : u'misstraue',
        u'mißtrauen'                   : u'misstrauen',
        u'mißtrauend'                  : u'misstrauend',
        u'mißtrauens'                  : u'misstrauens',
        u'mißtrauensantrag'            : u'misstrauensantrag',
        u'mißtrauensantrage'           : u'misstrauensantrage',
        u'mißtrauensantrages'          : u'misstrauensantrages',
        u'mißtrauensantrags'           : u'misstrauensantrags',
        u'mißtrauensanträge'           : u'misstrauensanträge',
        u'mißtrauensanträgen'          : u'misstrauensanträgen',
        u'mißtrauensvota'              : u'misstrauensvota',
        u'mißtrauensvoten'             : u'misstrauensvoten',
        u'mißtrauensvotum'             : u'misstrauensvotum',
        u'mißtrauensvotums'            : u'misstrauensvotums',
        u'mißtrauest'                  : u'misstrauest',
        u'mißtrauet'                   : u'misstrauet',
        u'mißtrauisch'                 : u'misstrauisch',
        u'mißtrauische'                : u'misstrauische',
        u'mißtrauischem'               : u'misstrauischem',
        u'mißtrauischen'               : u'misstrauischen',
        u'mißtrauischer'               : u'misstrauischer',
        u'mißtrauischere'              : u'misstrauischere',
        u'mißtrauischerem'             : u'misstrauischerem',
        u'mißtrauischeren'             : u'misstrauischeren',
        u'mißtrauischerer'             : u'misstrauischerer',
        u'mißtrauischeres'             : u'misstrauischeres',
        u'mißtrauisches'               : u'misstrauisches',
        u'mißtrauischste'              : u'misstrauischste',
        u'mißtrauischstem'             : u'misstrauischstem',
        u'mißtrauischsten'             : u'misstrauischsten',
        u'mißtrauischster'             : u'misstrauischster',
        u'mißtrauischstes'             : u'misstrauischstes',
        u'mißtraust'                   : u'misstraust',
        u'mißtraut'                    : u'misstraut',
        u'mißtraute'                   : u'misstraute',
        u'mißtrauten'                  : u'misstrauten',
        u'mißtrautest'                 : u'misstrautest',
        u'mißtrautet'                  : u'misstrautet',
        u'mißtritt'                    : u'misstritt',
        u'mißtönend'                   : u'misstönend',
        u'mißtönig'                    : u'misstönig',
        u'mißvergnügen'                : u'missvergnügen',
        u'mißvergnügens'               : u'missvergnügens',
        u'mißvergnügt'                 : u'missvergnügt',
        u'mißvergnügte'                : u'missvergnügte',
        u'mißvergnügtem'               : u'missvergnügtem',
        u'mißvergnügten'               : u'missvergnügten',
        u'mißvergnügter'               : u'missvergnügter',
        u'mißvergnügtere'              : u'missvergnügtere',
        u'mißvergnügterem'             : u'missvergnügterem',
        u'mißvergnügteren'             : u'missvergnügteren',
        u'mißvergnügterer'             : u'missvergnügterer',
        u'mißvergnügteres'             : u'missvergnügteres',
        u'mißvergnügtes'               : u'missvergnügtes',
        u'mißvergnügteste'             : u'missvergnügteste',
        u'mißvergnügtestem'            : u'missvergnügtestem',
        u'mißvergnügtesten'            : u'missvergnügtesten',
        u'mißvergnügtester'            : u'missvergnügtester',
        u'mißvergnügtestes'            : u'missvergnügtestes',
        u'mißverhältnis'               : u'missverhältnis',
        u'mißverstand'                 : u'missverstand',
        u'mißverstanden'               : u'missverstanden',
        u'mißverstandenen'             : u'missverstandenen',
        u'mißverstandener'             : u'missverstandener',
        u'mißverstehe'                 : u'missverstehe',
        u'mißverstehen'                : u'missverstehen',
        u'mißverständlich'             : u'missverständlich',
        u'mißverständliche'            : u'missverständliche',
        u'mißverständlichem'           : u'missverständlichem',
        u'mißverständlichen'           : u'missverständlichen',
        u'mißverständlicher'           : u'missverständlicher',
        u'mißverständlichere'          : u'missverständlichere',
        u'mißverständlicherem'         : u'missverständlicherem',
        u'mißverständlicheren'         : u'missverständlicheren',
        u'mißverständlicherer'         : u'missverständlicherer',
        u'mißverständlicheres'         : u'missverständlicheres',
        u'mißverständlicherweise'      : u'missverständlicherweise',
        u'mißverständliches'           : u'missverständliches',
        u'mißverständlichste'          : u'missverständlichste',
        u'mißverständlichstem'         : u'missverständlichstem',
        u'mißverständlichsten'         : u'missverständlichsten',
        u'mißverständlichster'         : u'missverständlichster',
        u'mißverständlichstes'         : u'missverständlichstes',
        u'mißverständnis'              : u'missverständnis',
        u'mißverständnisse'            : u'missverständnisse',
        u'mißverständnissen'           : u'missverständnissen',
        u'mißverständnisses'           : u'missverständnisses',
        u'mißverständniß'              : u'missverständnis',
        u'mißwachs'                    : u'misswachs',
        u'mißwachsen'                  : u'misswachsen',
        u'mißwahl'                     : u'misswahl',
        u'mißweisung'                  : u'missweisung',
        u'mißwirtschaft'               : u'misswirtschaft',
        u'mißwuchs'                    : u'misswuchs',
        u'mißzufrieden'                : u'misszufrieden',
        u'mobiltelephon'               : u'mobiltelefon',
        u'mobiltelephone'              : u'mobiltelefone',
        u'mobiltelephonen'             : u'mobiltelefonen',
        u'mobiltelephons'              : u'mobiltelefons',
        u'modebewußt'                  : u'modebewusst',
        u'modebewußte'                 : u'modebewusste',
        u'modebewußtem'                : u'modebewusstem',
        u'modebewußten'                : u'modebewussten',
        u'modebewußter'                : u'modebewusster',
        u'modebewußtere'               : u'modebewusstere',
        u'modebewußterem'              : u'modebewussterem',
        u'modebewußteren'              : u'modebewussteren',
        u'modebewußterer'              : u'modebewussterer',
        u'modebewußteres'              : u'modebewussteres',
        u'modebewußtes'                : u'modebewusstes',
        u'modebewußteste'              : u'modebewussteste',
        u'modebewußtestem'             : u'modebewusstestem',
        u'modebewußtesten'             : u'modebewusstesten',
        u'modebewußtester'             : u'modebewusstester',
        u'modebewußtestes'             : u'modebewusstestes',
        u'mohr'                        : u'moor',
        u'mohär'                       : u'mohair',
        u'mohäre'                      : u'mohaire',
        u'mohären'                     : u'mohairen',
        u'mohärs'                      : u'mohairs',
        u'momentphotographie'          : u'momentfotografie',
        u'momentphotographien'         : u'momentfotografien',
        u'monsterprozeß'               : u'monsterprozess',
        u'mop'                         : u'mopp',
        u'mops'                        : u'mopps',
        u'mordprozeß'                  : u'mordprozess',
        u'mulläppchen'                 : u'mullläppchen',
        u'muskatnuß'                   : u'muskatnuss',
        u'muskelfaserriß'              : u'muskelfaserriss',
        u'muskelriß'                   : u'muskelriss',
        u'musterprozeß'                : u'musterprozess',
        u'muth'                        : u'mut',
        u'muthe'                       : u'mute',
        u'muthes'                      : u'mutes',
        u'muths'                       : u'muts',
        u'muß'                         : u'muss',
        u'mußt'                        : u'musst',
        u'mußte'                       : u'musste',
        u'mußten'                      : u'mussten',
        u'mußtest'                     : u'musstest',
        u'mußtet'                      : u'musstet',
        u'myrre'                       : u'myrrhe',
        u'männerüberschuß'             : u'männerüberschuss',
        u'möve'                        : u'möwe',
        u'möven'                       : u'möwen',
        u'mülladung'                   : u'müllladung',
        u'müllager'                    : u'mülllager',
        u'müllaster'                   : u'mülllaster',
        u'mülleerung'                  : u'müllleerung',
        u'mülleute'                    : u'müllleute',
        u'mülleuten'                   : u'müllleuten',
        u'münztelephon'                : u'münztelefon',
        u'münztelephone'               : u'münztelefone',
        u'münztelephonen'              : u'münztelefonen',
        u'münztelephons'               : u'münztelefons',
        u'müßt'                        : u'müsst',
        u'müßte'                       : u'müsste',
        u'müßten'                      : u'müssten',
        u'müßtest'                     : u'müsstest',
        u'müßtet'                      : u'müsstet',
        u'nachkriegsphoto'             : u'nachkriegsfoto',
        u'nachkriegsphotos'            : u'nachkriegsfotos',
        u'nachlaß'                     : u'nachlass',
        u'nachlaßgericht'              : u'nachlassgericht',
        u'nachlaßordner'               : u'nachlassordner',
        u'nachlaßpfleger'              : u'nachlasspfleger',
        u'nachlaßt'                    : u'nachlasst',
        u'nachlaßverwalter'            : u'nachlassverwalter',
        u'nachlaßverwalterin'          : u'nachlassverwalterin',
        u'nachlaßverwalterinnen'       : u'nachlassverwalterinnen',
        u'nachlaßverwaltern'           : u'nachlassverwaltern',
        u'nachlaßverwalters'           : u'nachlassverwalters',
        u'nachlaßverwaltung'           : u'nachlassverwaltung',
        u'nachläßt'                    : u'nachlässt',
        u'nachschuß'                   : u'nachschuss',
        u'nachtblüthen'                : u'nachtblüten',
        u'nachtphantasien'             : u'nachtfantasien',
        u'nansenpaß'                   : u'nansenpass',
        u'naphta'                      : u'naphtha',
        u'narziß'                      : u'narziss',
        u'narzißmen'                   : u'narzissmen',
        u'narzißmus'                   : u'narzissmus',
        u'narzißt'                     : u'narzisst',
        u'narzißten'                   : u'narzissten',
        u'narzißtisch'                 : u'narzisstisch',
        u'narzißtische'                : u'narzisstische',
        u'narzißtischem'               : u'narzisstischem',
        u'narzißtischen'               : u'narzisstischen',
        u'narzißtischer'               : u'narzisstischer',
        u'narzißtischere'              : u'narzisstischere',
        u'narzißtischerem'             : u'narzisstischerem',
        u'narzißtischeren'             : u'narzisstischeren',
        u'narzißtischerer'             : u'narzisstischerer',
        u'narzißtischeres'             : u'narzisstischeres',
        u'narzißtisches'               : u'narzisstisches',
        u'narzißtischste'              : u'narzisstischste',
        u'narzißtischstem'             : u'narzisstischstem',
        u'narzißtischsten'             : u'narzisstischsten',
        u'narzißtischster'             : u'narzisstischster',
        u'narzißtischstes'             : u'narzisstischstes',
        u'nationalbewußt'              : u'nationalbewusst',
        u'nationalbewußte'             : u'nationalbewusste',
        u'nationalbewußtem'            : u'nationalbewusstem',
        u'nationalbewußten'            : u'nationalbewussten',
        u'nationalbewußter'            : u'nationalbewusster',
        u'nationalbewußtere'           : u'nationalbewusstere',
        u'nationalbewußterem'          : u'nationalbewussterem',
        u'nationalbewußteren'          : u'nationalbewussteren',
        u'nationalbewußterer'          : u'nationalbewussterer',
        u'nationalbewußteres'          : u'nationalbewussteres',
        u'nationalbewußtes'            : u'nationalbewusstes',
        u'nationalbewußteste'          : u'nationalbewussteste',
        u'nationalbewußtestem'         : u'nationalbewusstestem',
        u'nationalbewußtesten'         : u'nationalbewusstesten',
        u'nationalbewußtester'         : u'nationalbewusstester',
        u'nationalbewußtestes'         : u'nationalbewusstestes',
        u'nationalbewußtsein'          : u'nationalbewusstsein',
        u'naß'                         : u'nass',
        u'naßfest'                     : u'nassfest',
        u'naßforsch'                   : u'nassforsch',
        u'naßforsche'                  : u'nassforsche',
        u'naßforschem'                 : u'nassforschem',
        u'naßforschen'                 : u'nassforschen',
        u'naßforscher'                 : u'nassforscher',
        u'naßforschere'                : u'nassforschere',
        u'naßforscherem'               : u'nassforscherem',
        u'naßforscheren'               : u'nassforscheren',
        u'naßforscherer'               : u'nassforscherer',
        u'naßforscheres'               : u'nassforscheres',
        u'naßforsches'                 : u'nassforsches',
        u'naßforscheste'               : u'nassforscheste',
        u'naßforschestem'              : u'nassforschestem',
        u'naßforschesten'              : u'nassforschesten',
        u'naßforschester'              : u'nassforschester',
        u'naßforschestes'              : u'nassforschestes',
        u'naßforschste'                : u'nassforschste',
        u'naßforschstem'               : u'nassforschstem',
        u'naßforschsten'               : u'nassforschsten',
        u'naßforschster'               : u'nassforschster',
        u'naßforschstes'               : u'nassforschstes',
        u'naßfutter'                   : u'nassfutter',
        u'naßfuttern'                  : u'nassfuttern',
        u'naßfutters'                  : u'nassfutters',
        u'naßgeschwitzt'               : u'nassgeschwitzt',
        u'naßgeschwitzte'              : u'nassgeschwitzte',
        u'naßgeschwitztem'             : u'nassgeschwitztem',
        u'naßgeschwitzten'             : u'nassgeschwitzten',
        u'naßgeschwitzter'             : u'nassgeschwitzter',
        u'naßgeschwitztes'             : u'nassgeschwitztes',
        u'naßkalt'                     : u'nasskalt',
        u'naßkalte'                    : u'nasskalte',
        u'naßkaltem'                   : u'nasskaltem',
        u'naßkalten'                   : u'nasskalten',
        u'naßkalter'                   : u'nasskalter',
        u'naßkaltes'                   : u'nasskaltes',
        u'naßklebend'                  : u'nassklebend',
        u'naßklebende'                 : u'nassklebende',
        u'naßklebendem'                : u'nassklebendem',
        u'naßklebenden'                : u'nassklebenden',
        u'naßklebender'                : u'nassklebender',
        u'naßklebendes'                : u'nassklebendes',
        u'naßkot'                      : u'nasskot',
        u'naßkotes'                    : u'nasskotes',
        u'naßkots'                     : u'nasskots',
        u'naßkälter'                   : u'nasskälter',
        u'naßkältere'                  : u'nasskältere',
        u'naßkälterem'                 : u'nasskälterem',
        u'naßkälteren'                 : u'nasskälteren',
        u'naßkälterer'                 : u'nasskälterer',
        u'naßkälteres'                 : u'nasskälteres',
        u'naßkälteste'                 : u'nasskälteste',
        u'naßkältestem'                : u'nasskältestem',
        u'naßkältesten'                : u'nasskältesten',
        u'naßkältester'                : u'nasskältester',
        u'naßkältestes'                : u'nasskältestes',
        u'naßrasierer'                 : u'nassrasierer',
        u'naßrasur'                    : u'nassrasur',
        u'naßrasuren'                  : u'nassrasuren',
        u'naßschnee'                   : u'nassschnee',
        u'naßschnees'                  : u'nassschnees',
        u'naßwäsche'                   : u'nasswäsche',
        u'naßzelle'                    : u'nasszelle',
        u'nebelschlußleuchte'          : u'nebelschlussleuchte',
        u'nebelschlußleuchten'         : u'nebelschlussleuchten',
        u'nebenfluß'                   : u'nebenfluss',
        u'nebengelaß'                  : u'nebengelass',
        u'negerkuß'                    : u'negerkuss',
        u'negir'                       : u'negier',
        u'negire'                      : u'negiere',
        u'negiren'                     : u'negieren',
        u'negirend'                    : u'negierend',
        u'negirest'                    : u'negierest',
        u'negiret'                     : u'negieret',
        u'negirst'                     : u'negierst',
        u'negirt'                      : u'negiert',
        u'negirte'                     : u'negierte',
        u'negirten'                    : u'negierten',
        u'negirtest'                   : u'negiertest',
        u'negirtet'                    : u'negiertet',
        u'negligee'                    : u'negligé',
        u'negligees'                   : u'negligés',
        u'nessessär'                   : u'necessaire',
        u'netzanschluß'                : u'netzanschluss',
        u'neuwienerisch'               : u'neu-wienerisch',
        u'neuß'                        : u'neuss',
        u'niederlaßt'                  : u'niederlasst',
        u'niederläßt'                  : u'niederlässt',
        u'niederschoß'                 : u'niederschoss',
        u'niederschoßt'                : u'niederschosst',
        u'nivellir'                    : u'nivellier',
        u'nivellire'                   : u'nivelliere',
        u'nivelliren'                  : u'nivellieren',
        u'nivellirend'                 : u'nivellierend',
        u'nivellirest'                 : u'nivellierest',
        u'nivelliret'                  : u'nivellieret',
        u'nivellirst'                  : u'nivellierst',
        u'nivellirt'                   : u'nivelliert',
        u'nivellirte'                  : u'nivellierte',
        u'nivellirten'                 : u'nivellierten',
        u'nivellirtest'                : u'nivelliertest',
        u'nivellirtet'                 : u'nivelliertet',
        u'nordwestrußland'             : u'nordwestrussland',
        u'nordwestrußlands'            : u'nordwestrusslands',
        u'normalgeschoß'               : u'normalgeschoss',
        u'normenausschuß'              : u'normenausschuss',
        u'noth'                        : u'not',
        u'notwaßre'                    : u'notwassre',
        u'notwaßrest'                  : u'notwassrest',
        u'notwaßret'                   : u'notwassret',
        u'nullage'                     : u'nulllage',
        u'nullagen'                    : u'nulllagen',
        u'nulleiter'                   : u'nullleiter',
        u'nulleitern'                  : u'nullleitern',
        u'nulleiters'                  : u'nullleiters',
        u'nullinie'                    : u'nulllinie',
        u'nullösung'                   : u'nulllösung',
        u'nullösungen'                 : u'nulllösungen',
        u'numerier'                    : u'nummerier',
        u'numeriere'                   : u'nummeriere',
        u'numerieren'                  : u'nummerieren',
        u'numerierend'                 : u'nummerierend',
        u'numerierenden'               : u'nummerierenden',
        u'numerierest'                 : u'nummerierest',
        u'numerieret'                  : u'nummerieret',
        u'numerierst'                  : u'nummerierst',
        u'numeriert'                   : u'nummeriert',
        u'numerierte'                  : u'nummerierte',
        u'numerierten'                 : u'nummerierten',
        u'numeriertest'                : u'nummeriertest',
        u'numeriertet'                 : u'nummeriertet',
        u'numerierung'                 : u'nummerierung',
        u'numerierungen'               : u'nummerierungen',
        u'nuß'                         : u'nuss',
        u'nußbaum'                     : u'nussbaum',
        u'nußbaum-kleiderschränke'     : u'nussbaum-kleiderschränke',
        u'nußbaumberges'               : u'nussbaumberges',
        u'nußbaume'                    : u'nussbaume',
        u'nußbaumes'                   : u'nussbaumes',
        u'nußbaumholz'                 : u'nussbaumholz',
        u'nußbaumholze'                : u'nussbaumholze',
        u'nußbaumholzes'               : u'nussbaumholzes',
        u'nußbaumhölzer'               : u'nussbaumhölzer',
        u'nußbaumhölzern'              : u'nussbaumhölzern',
        u'nußbaumlade'                 : u'nussbaumlade',
        u'nußbaums'                    : u'nussbaums',
        u'nußbaumschrank'              : u'nussbaumschrank',
        u'nußblätter'                  : u'nussblätter',
        u'nußblättern'                 : u'nussblättern',
        u'nußbraun'                    : u'nussbraun',
        u'nußbäumchen'                 : u'nussbäumchen',
        u'nußbäume'                    : u'nussbäume',
        u'nußbäumen'                   : u'nussbäumen',
        u'nußecke'                     : u'nussecke',
        u'nußförmig'                   : u'nussförmig',
        u'nußförmige'                  : u'nussförmige',
        u'nußförmigem'                 : u'nussförmigem',
        u'nußförmigen'                 : u'nussförmigen',
        u'nußförmiger'                 : u'nussförmiger',
        u'nußförmiges'                 : u'nussförmiges',
        u'nußfüllung'                  : u'nussfüllung',
        u'nußfüllungen'                : u'nussfüllungen',
        u'nußholztür'                  : u'nussholztür',
        u'nußkerne'                    : u'nusskerne',
        u'nußknacker'                  : u'nussknacker',
        u'nußknackern'                 : u'nussknackern',
        u'nußknackers'                 : u'nussknackers',
        u'nußkohle'                    : u'nusskohle',
        u'nußkuchen'                   : u'nusskuchen',
        u'nußpfirsich'                 : u'nusspfirsich',
        u'nußpfirsiche'                : u'nusspfirsiche',
        u'nußpfirsichen'               : u'nusspfirsichen',
        u'nußpfirsiches'               : u'nusspfirsiches',
        u'nußpfirsichs'                : u'nusspfirsichs',
        u'nußpilz'                     : u'nusspilz',
        u'nußpilze'                    : u'nusspilze',
        u'nußpilzen'                   : u'nusspilzen',
        u'nußpilzes'                   : u'nusspilzes',
        u'nußschale'                   : u'nussschale',
        u'nußschalen'                  : u'nussschalen',
        u'nußschinken'                 : u'nussschinken',
        u'nußschinkens'                : u'nussschinkens',
        u'nußschokolade'               : u'nussschokolade',
        u'nußschokoladen'              : u'nussschokoladen',
        u'nußschälchen'                : u'nussschälchen',
        u'nußstrudel'                  : u'nussstrudel',
        u'nußstrudeln'                 : u'nussstrudeln',
        u'nußstrudels'                 : u'nussstrudels',
        u'nußtorte'                    : u'nusstorte',
        u'nußtorten'                   : u'nusstorten',
        u'nußzweig'                    : u'nusszweig',
        u'nußzweige'                   : u'nusszweige',
        u'nußöl'                       : u'nussöl',
        u'näß'                         : u'näss',
        u'näßlich'                     : u'nässlich',
        u'näßt'                        : u'nässt',
        u'näßte'                       : u'nässte',
        u'näßten'                      : u'nässten',
        u'näßtest'                     : u'nässtest',
        u'näßtet'                      : u'nässtet',
        u'nöthe'                       : u'nöte',
        u'nöthen'                      : u'nöten',
        u'nöthigen'                    : u'nötigen',
        u'nüßchen'                     : u'nüsschen',
        u'nüßchens'                    : u'nüsschens',
        u'obergränze'                  : u'obergrenze',
        u'obergränzen'                 : u'obergrenzen',
        u'objektivanschluß'            : u'objektivanschluss',
        u'october'                     : u'oktober',
        u'octobern'                    : u'oktobern',
        u'octobers'                    : u'oktobers',
        u'oel'                         : u'öl',
        u'oels'                        : u'öls',
        u'ok'                          : u'okay',
        u'ordre'                       : u'order',
        u'orientexpreß'                : u'orientexpress',
        u'orthophoto'                  : u'orthofoto',
        u'orthophotos'                 : u'orthofotos',
        u'oscillirend'                 : u'oszillierend',
        u'ostgothe'                    : u'ostgote',
        u'ostgothen'                   : u'ostgoten',
        u'ostgothenkönig'              : u'ostgotenkönig',
        u'ostgothenkönige'             : u'ostgotenkönige',
        u'ostgothenkönigen'            : u'ostgotenkönigen',
        u'ostgothenköniges'            : u'ostgotenköniges',
        u'ostgothenkönigs'             : u'ostgotenkönigs',
        u'oute'                        : u'out',
        u'owne'                        : u'own',
        u'paranuß'                     : u'paranuss',
        u'paranußbaum'                 : u'paranussbaum',
        u'parlamentsausschuß'          : u'parlamentsausschuss',
        u'parlamentsbeschluß'          : u'parlamentsbeschluss',
        u'parnaß'                      : u'parnass',
        u'parteiausschluß'             : u'parteiausschluss',
        u'parteiausschlußverfahren'    : u'parteiausschlussverfahren',
        u'parteiausschlußverfahrens'   : u'parteiausschlussverfahrens',
        u'parteibeschluß'              : u'parteibeschluss',
        u'parteikongreß'               : u'parteikongress',
        u'parteitagsbeschluß'          : u'parteitagsbeschluss',
        u'parties'                     : u'partys',
        u'passee'                      : u'passé',
        u'passivprozeß'                : u'passivprozess',
        u'patentverschluß'             : u'patentverschluss',
        u'patschenaß'                  : u'patschenass',
        u'patschnaß'                   : u'patschnass',
        u'paß'                         : u'pass',
        u'paß-ersatzpapieren'          : u'pass-ersatzpapieren',
        u'paßamt'                      : u'passamt',
        u'paßantrag'                   : u'passantrag',
        u'paßantrage'                  : u'passantrage',
        u'paßantrages'                 : u'passantrages',
        u'paßantrags'                  : u'passantrags',
        u'paßanträge'                  : u'passanträge',
        u'paßanträgen'                 : u'passanträgen',
        u'paßbehörde'                  : u'passbehörde',
        u'paßbehörden'                 : u'passbehörden',
        u'paßbild'                     : u'passbild',
        u'paßbilde'                    : u'passbilde',
        u'paßbilder'                   : u'passbilder',
        u'paßbildern'                  : u'passbildern',
        u'paßbildes'                   : u'passbildes',
        u'paßbilds'                    : u'passbilds',
        u'paßempfänger'                : u'passempfänger',
        u'paßempfängern'               : u'passempfängern',
        u'paßempfängers'               : u'passempfängers',
        u'paßform'                     : u'passform',
        u'paßfoto'                     : u'passfoto',
        u'paßfotos'                    : u'passfotos',
        u'paßgang'                     : u'passgang',
        u'paßganges'                   : u'passganges',
        u'paßgangs'                    : u'passgangs',
        u'paßgeber'                    : u'passgeber',
        u'paßgebern'                   : u'passgebern',
        u'paßgebers'                   : u'passgebers',
        u'paßgenau'                    : u'passgenau',
        u'paßgerecht'                  : u'passgerecht',
        u'paßgläser'                   : u'passgläser',
        u'paßgänger'                   : u'passgänger',
        u'paßgängern'                  : u'passgängern',
        u'paßgängers'                  : u'passgängers',
        u'paßhöhe'                     : u'passhöhe',
        u'paßinhaber'                  : u'passinhaber',
        u'paßinhabern'                 : u'passinhabern',
        u'paßinhabers'                 : u'passinhabers',
        u'paßkontrolle'                : u'passkontrolle',
        u'paßkontrollen'               : u'passkontrollen',
        u'paßpflicht'                  : u'passpflicht',
        u'paßphoto'                    : u'passfoto',
        u'paßphotos'                   : u'passfotos',
        u'paßspiel'                    : u'passspiel',
        u'paßspiele'                   : u'passspiele',
        u'paßspielen'                  : u'passspielen',
        u'paßspieles'                  : u'passspieles',
        u'paßspiels'                   : u'passspiels',
        u'paßstelle'                   : u'passstelle',
        u'paßstraße'                   : u'passstraße',
        u'paßstraßen'                  : u'passstraßen',
        u'paßt'                        : u'passt',
        u'paßte'                       : u'passte',
        u'paßten'                      : u'passten',
        u'paßtest'                     : u'passtest',
        u'paßtet'                      : u'passtet',
        u'paßträger'                   : u'passträger',
        u'paßträgern'                  : u'passträgern',
        u'paßträgers'                  : u'passträgers',
        u'paßwesen'                    : u'passwesen',
        u'paßwort'                     : u'passwort',
        u'paßworte'                    : u'passworte',
        u'paßwortes'                   : u'passwortes',
        u'paßwortgeschützt'            : u'passwortgeschützt',
        u'paßwortmanager'              : u'passwortmanager',
        u'paßwortmanagern'             : u'passwortmanagern',
        u'paßwortmanagers'             : u'passwortmanagers',
        u'paßworts'                    : u'passworts',
        u'paßwärts'                    : u'passwärts',
        u'paßwörter'                   : u'passwörter',
        u'paßwörtern'                  : u'passwörtern',
        u'paßzwang'                    : u'passzwang',
        u'pekannuß'                    : u'pekannuss',
        u'perfect'                     : u'perfekt',
        u'perfecte'                    : u'perfekte',
        u'perfecten'                   : u'perfekten',
        u'perfects'                    : u'perfekts',
        u'perlmutterfäßchens'          : u'perlmutterfässchens',
        u'personenschiffahrt'          : u'personenschifffahrt',
        u'perspektivfäßchen'           : u'perspektivfässchen',
        u'petitionsausschuß'           : u'petitionsausschuss',
        u'pfandschloß'                 : u'pfandschloss',
        u'pfeffernuß'                  : u'pfeffernuss',
        u'pfeill'                      : u'pfeil',
        u'pfeilschuß'                  : u'pfeilschuss',
        u'pferdegebiß'                 : u'pferdegebiss',
        u'pferdekuß'                   : u'pferdekuss',
        u'pflichtbewußt'               : u'pflichtbewusst',
        u'pflichtbewußte'              : u'pflichtbewusste',
        u'pflichtbewußtem'             : u'pflichtbewusstem',
        u'pflichtbewußten'             : u'pflichtbewussten',
        u'pflichtbewußter'             : u'pflichtbewusster',
        u'pflichtbewußtere'            : u'pflichtbewusstere',
        u'pflichtbewußterem'           : u'pflichtbewussterem',
        u'pflichtbewußteren'           : u'pflichtbewussteren',
        u'pflichtbewußterer'           : u'pflichtbewussterer',
        u'pflichtbewußteres'           : u'pflichtbewussteres',
        u'pflichtbewußtes'             : u'pflichtbewusstes',
        u'pflichtbewußteste'           : u'pflichtbewussteste',
        u'pflichtbewußtestem'          : u'pflichtbewusstestem',
        u'pflichtbewußtesten'          : u'pflichtbewusstesten',
        u'pflichtbewußtester'          : u'pflichtbewusstester',
        u'pflichtbewußtestes'          : u'pflichtbewusstestes',
        u'pflichtbewußtsein'           : u'pflichtbewusstsein',
        u'pflichtbewußtseins'          : u'pflichtbewusstseins',
        u'pflichtvergeßner'            : u'pflichtvergessner',
        u'pflichtvergeßnere'           : u'pflichtvergessnere',
        u'pflichtvergeßnerem'          : u'pflichtvergessnerem',
        u'pflichtvergeßneren'          : u'pflichtvergessneren',
        u'pflichtvergeßnerer'          : u'pflichtvergessnerer',
        u'pflichtvergeßneres'          : u'pflichtvergessneres',
        u'pfostenschuß'                : u'pfostenschuss',
        u'phantasie'                   : u'fantasie',
        u'phantasiebild'               : u'fantasiebild',
        u'phantasiegebilde'            : u'fantasiegebilde',
        u'phantasiegebilden'           : u'fantasiegebilden',
        u'phantasiegebildes'           : u'fantasiegebildes',
        u'phantasiegestalten'          : u'fantasiegestalten',
        u'phantasiegestaltung'         : u'fantasiegestaltung',
        u'phantasiekostüm'             : u'fantasiekostüm',
        u'phantasielos'                : u'fantasielos',
        u'phantasielosen'              : u'fantasielosen',
        u'phantasieloser'              : u'fantasieloser',
        u'phantasien'                  : u'fantasien',
        u'phantasiere'                 : u'fantasiere',
        u'phantasiereich'              : u'fantasiereich',
        u'phantasiereichen'            : u'fantasiereichen',
        u'phantasieren'                : u'fantasieren',
        u'phantasiert'                 : u'fantasiert',
        u'phantasierte'                : u'fantasierte',
        u'phantasierten'               : u'fantasierten',
        u'phantasiesessels'            : u'fantasiesessels',
        u'phantasiespiels'             : u'fantasiespiels',
        u'phantasievoll'               : u'fantasievoll',
        u'phantastereien'              : u'fantastereien',
        u'phantastisch'                : u'fantastisch',
        u'phantastische'               : u'fantastische',
        u'phantastischem'              : u'fantastischem',
        u'phantastischen'              : u'fantastischen',
        u'phantastischer'              : u'fantastischer',
        u'phantastisches'              : u'fantastisches',
        u'phantastischste'             : u'fantastischste',
        u'phantastischsten'            : u'fantastischsten',
        u'philipp'                     : u'philip',
        u'photo'                       : u'foto',
        u'photochemische'              : u'fotochemische',
        u'photograf'                   : u'fotograf',
        u'photografen'                 : u'fotografen',
        u'photografier'                : u'fotografier',
        u'photografiere'               : u'fotografiere',
        u'photografieren'              : u'fotografieren',
        u'photografierend'             : u'fotografierend',
        u'photografierest'             : u'fotografierest',
        u'photografieret'              : u'fotografieret',
        u'photografierst'              : u'fotografierst',
        u'photografiert'               : u'fotografiert',
        u'photografierte'              : u'fotografierte',
        u'photografierten'             : u'fotografierten',
        u'photografiertest'            : u'fotografiertest',
        u'photografiertet'             : u'fotografiertet',
        u'photograph'                  : u'fotograf',
        u'photographie'                : u'fotografie',
        u'photographiealbums'          : u'fotografiealbums',
        u'photographien'               : u'fotografien',
        u'photographier'               : u'fotografier',
        u'photographiere'              : u'fotografiere',
        u'photographieren'             : u'fotografieren',
        u'photographierend'            : u'fotografierend',
        u'photographierest'            : u'fotografierest',
        u'photographieret'             : u'fotografieret',
        u'photographierst'             : u'fotografierst',
        u'photographiert'              : u'fotografiert',
        u'photographierte'             : u'fotografierte',
        u'photographierten'            : u'fotografierten',
        u'photographiertest'           : u'fotografiertest',
        u'photographiertet'            : u'fotografiertet',
        u'photographische'             : u'fotografische',
        u'photographischen'            : u'fotografischen',
        u'photographischer'            : u'fotografischer',
        u'photos'                      : u'fotos',
        u'phototelegraphen'            : u'fototelegraphen',
        u'photovoltaik'                : u'fotovoltaik',
        u'photovoltaikanlagen'         : u'fotovoltaikanlagen',
        u'pikas'                       : u'pikass',
        u'pimpernuß'                   : u'pimpernuss',
        u'pinnadel'                    : u'pinnnadel',
        u'pistaziennuß'                : u'pistaziennuss',
        u'pistolenschuß'               : u'pistolenschuss',
        u'pitschenaß'                  : u'pitschenass',
        u'pitschepatschenaß'           : u'pitschepatschenass',
        u'pitschnaß'                   : u'pitschnass',
        u'piß'                         : u'piss',
        u'pißnelke'                    : u'pissnelke',
        u'pißnelken'                   : u'pissnelken',
        u'pißt'                        : u'pisst',
        u'pißte'                       : u'pisste',
        u'pißten'                      : u'pissten',
        u'pißtest'                     : u'pisstest',
        u'pißtet'                      : u'pisstet',
        u'placier'                     : u'platzier',
        u'placieren'                   : u'platzieren',
        u'placierend'                  : u'platzierend',
        u'placierest'                  : u'platzierest',
        u'placieret'                   : u'platzieret',
        u'placierst'                   : u'platzierst',
        u'placiert'                    : u'platziert',
        u'placierte'                   : u'platzierte',
        u'placierten'                  : u'platzierten',
        u'placiertest'                 : u'platziertest',
        u'placiertet'                  : u'platziertet',
        u'plagiatsprozeß'              : u'plagiatsprozess',
        u'platitüde'                   : u'plattitüde',
        u'platitüden'                  : u'plattitüden',
        u'platschnaß'                  : u'platschnass',
        u'plattenboß'                  : u'plattenboss',
        u'plazier'                     : u'platzier',
        u'plaziere'                    : u'platziere',
        u'plazieren'                   : u'platzieren',
        u'plazierend'                  : u'platzierend',
        u'plazierest'                  : u'platzierest',
        u'plazieret'                   : u'platzieret',
        u'plazierst'                   : u'platzierst',
        u'plaziert'                    : u'platziert',
        u'plazierte'                   : u'platzierte',
        u'plazierten'                  : u'platzierten',
        u'plaziertest'                 : u'platziertest',
        u'plaziertet'                  : u'platziertet',
        u'plusquamperfect'             : u'plusquamperfekt',
        u'plusquamperfecte'            : u'plusquamperfekte',
        u'plusquamperfecten'           : u'plusquamperfekten',
        u'plusquamperfects'            : u'plusquamperfekts',
        u'ponies'                      : u'ponys',
        u'pornographische'             : u'pornografische',
        u'portemonnaie'                : u'portmonee',
        u'portrait'                    : u'porträt',
        u'portraitier'                 : u'porträtier',
        u'portraitiere'                : u'porträtiere',
        u'portraitieren'               : u'porträtieren',
        u'portraitierend'              : u'porträtierend',
        u'portraitierest'              : u'porträtierest',
        u'portraitieret'               : u'porträtieret',
        u'portraitierst'               : u'porträtierst',
        u'portraitiert'                : u'porträtiert',
        u'portraitierte'               : u'porträtierte',
        u'portraitierten'              : u'porträtierten',
        u'portraitiertest'             : u'porträtiertest',
        u'portraitiertet'              : u'porträtiertet',
        u'portraits'                   : u'porträts',
        u'positivprozeß'               : u'positivprozess',
        u'potential'                   : u'potenzial',
        u'potentiale'                  : u'potenziale',
        u'potentielle'                 : u'potenzielle',
        u'potentiellen'                : u'potenziellen',
        u'potthäßlich'                 : u'potthässlich',
        u'potthäßliche'                : u'potthässliche',
        u'potthäßlichem'               : u'potthässlichem',
        u'potthäßlichen'               : u'potthässlichen',
        u'potthäßlicher'               : u'potthässlicher',
        u'potthäßliches'               : u'potthässliches',
        u'poussierstengel'             : u'poussierstängel',
        u'poussierstengeln'            : u'poussierstängeln',
        u'poussierstengels'            : u'poussierstängels',
        u'practicier'                  : u'praktizier',
        u'practiciere'                 : u'praktiziere',
        u'practicieren'                : u'praktizieren',
        u'practicierend'               : u'praktizierend',
        u'practicierest'               : u'praktizierest',
        u'practicieret'                : u'praktizieret',
        u'practicierst'                : u'praktizierst',
        u'practiciert'                 : u'praktiziert',
        u'practicierte'                : u'praktizierte',
        u'practicierten'               : u'praktizierten',
        u'practiciertest'              : u'praktiziertest',
        u'practiciertet'               : u'praktiziertet',
        u'practicir'                   : u'praktizier',
        u'practicire'                  : u'praktiziere',
        u'practiciren'                 : u'praktizieren',
        u'practicirend'                : u'praktizierend',
        u'practicirest'                : u'praktizierest',
        u'practiciret'                 : u'praktizieret',
        u'practicirst'                 : u'praktizierst',
        u'practicirt'                  : u'praktiziert',
        u'practicirte'                 : u'praktizierte',
        u'practicirten'                : u'praktizierten',
        u'practicirtest'               : u'praktiziertest',
        u'practicirtet'                : u'praktiziertet',
        u'praktizir'                   : u'praktizier',
        u'praktizire'                  : u'praktiziere',
        u'praktiziren'                 : u'praktizieren',
        u'praktizirend'                : u'praktizierend',
        u'praktizirest'                : u'praktizierest',
        u'praktiziret'                 : u'praktizieret',
        u'praktizirst'                 : u'praktizierst',
        u'praktizirt'                  : u'praktiziert',
        u'praktizirte'                 : u'praktizierte',
        u'praktizirten'                : u'praktizierten',
        u'praktizirtest'               : u'praktiziertest',
        u'praktizirtet'                : u'praktiziertet',
        u'preisbewußt'                 : u'preisbewusst',
        u'preisbewußte'                : u'preisbewusste',
        u'preisbewußtem'               : u'preisbewusstem',
        u'preisbewußten'               : u'preisbewussten',
        u'preisbewußter'               : u'preisbewusster',
        u'preisbewußtere'              : u'preisbewusstere',
        u'preisbewußterem'             : u'preisbewussterem',
        u'preisbewußteren'             : u'preisbewussteren',
        u'preisbewußterer'             : u'preisbewussterer',
        u'preisbewußteres'             : u'preisbewussteres',
        u'preisbewußtes'               : u'preisbewusstes',
        u'preisbewußteste'             : u'preisbewussteste',
        u'preisbewußtestem'            : u'preisbewusstestem',
        u'preisbewußtesten'            : u'preisbewusstesten',
        u'preisbewußtester'            : u'preisbewusstester',
        u'preisbewußtestes'            : u'preisbewusstestes',
        u'preisnachlaß'                : u'preisnachlass',
        u'prellschuß'                  : u'prellschuss',
        u'preß'                        : u'press',
        u'preßball'                    : u'pressball',
        u'preßbleiglas'                : u'pressbleiglas',
        u'preßbleiglases'              : u'pressbleiglases',
        u'preßburg'                    : u'pressburg',
        u'preßburgs'                   : u'pressburgs',
        u'preßform'                    : u'pressform',
        u'preßfrisch'                  : u'pressfrisch',
        u'preßglas'                    : u'pressglas',
        u'preßhefe'                    : u'presshefe',
        u'preßholz'                    : u'pressholz',
        u'preßkohle'                   : u'presskohle',
        u'preßkopf'                    : u'presskopf',
        u'preßkopfe'                   : u'presskopfe',
        u'preßkopfes'                  : u'presskopfes',
        u'preßkopfs'                   : u'presskopfs',
        u'preßköpfe'                   : u'pressköpfe',
        u'preßköpfen'                  : u'pressköpfen',
        u'preßling'                    : u'pressling',
        u'preßluft'                    : u'pressluft',
        u'preßluftbohrer'              : u'pressluftbohrer',
        u'preßluftbohrern'             : u'pressluftbohrern',
        u'preßluftbohrers'             : u'pressluftbohrers',
        u'preßluftflasche'             : u'pressluftflasche',
        u'preßlufthammer'              : u'presslufthammer',
        u'preßlufthammers'             : u'presslufthammers',
        u'preßlufthämmer'              : u'presslufthämmer',
        u'preßlufthämmern'             : u'presslufthämmern',
        u'preßsack'                    : u'presssack',
        u'preßsacke'                   : u'presssacke',
        u'preßsackes'                  : u'presssackes',
        u'preßsacks'                   : u'presssacks',
        u'preßschlag'                  : u'pressschlag',
        u'preßspan'                    : u'pressspan',
        u'preßspanplatte'              : u'pressspanplatte',
        u'preßstempelkanne'            : u'pressstempelkanne',
        u'preßstempelkannen'           : u'pressstempelkannen',
        u'preßstoff'                   : u'pressstoff',
        u'preßstroh'                   : u'pressstroh',
        u'preßsäcke'                   : u'presssäcke',
        u'preßsäcken'                  : u'presssäcken',
        u'preßt'                       : u'presst',
        u'preßte'                      : u'presste',
        u'preßten'                     : u'pressten',
        u'preßtest'                    : u'presstest',
        u'preßtet'                     : u'presstet',
        u'preßwehe'                    : u'presswehe',
        u'preßwerk'                    : u'presswerk',
        u'preßwerke'                   : u'presswerke',
        u'preßwerken'                  : u'presswerken',
        u'preßwerker'                  : u'presswerker',
        u'preßwerkern'                 : u'presswerkern',
        u'preßwerkers'                 : u'presswerkers',
        u'preßwerkes'                  : u'presswerkes',
        u'preßwerks'                   : u'presswerks',
        u'preßwurst'                   : u'presswurst',
        u'princip'                     : u'prinzip',
        u'principe'                    : u'prinzipe',
        u'principen'                   : u'prinzipen',
        u'principien'                  : u'prinzipien',
        u'princips'                    : u'prinzips',
        u'prinzeß'                     : u'prinzess',
        u'prinzeßbohne'                : u'prinzessbohne',
        u'prinzeßbohnen'               : u'prinzessbohnen',
        u'problemlösungsprozeß'        : u'problemlösungsprozess',
        u'proceß'                      : u'prozess',
        u'producte'                    : u'produkte',
        u'produktionsprozeß'           : u'produktionsprozess',
        u'profeß'                      : u'profess',
        u'programmusik'                : u'programmmusik',
        u'programmässig'               : u'programmmäßig',
        u'programmässige'              : u'programmmäßige',
        u'programmässigem'             : u'programmmäßigem',
        u'programmässigen'             : u'programmmäßigen',
        u'programmässiger'             : u'programmmäßiger',
        u'programmässiges'             : u'programmmäßiges',
        u'progreß'                     : u'progress',
        u'projektabschluß'             : u'projektabschluss',
        u'promotionsausschuß'          : u'promotionsausschuss',
        u'prozeß'                      : u'prozess',
        u'prozeßabwicklung'            : u'prozessabwicklung',
        u'prozeßausgang'               : u'prozessausgang',
        u'prozeßausgange'              : u'prozessausgange',
        u'prozeßausganges'             : u'prozessausganges',
        u'prozeßausgangs'              : u'prozessausgangs',
        u'prozeßausgänge'              : u'prozessausgänge',
        u'prozeßausgängen'             : u'prozessausgängen',
        u'prozeßbevollmächtigt'        : u'prozessbevollmächtigt',
        u'prozeßbevollmächtigter'      : u'prozessbevollmächtigter',
        u'prozeßfähig'                 : u'prozessfähig',
        u'prozeßfähigkeit'             : u'prozessfähigkeit',
        u'prozeßgegner'                : u'prozessgegner',
        u'prozeßkosten'                : u'prozesskosten',
        u'prozeßkostenhilfe'           : u'prozesskostenhilfe',
        u'prozeßleitelektroniker'      : u'prozessleitelektroniker',
        u'prozeßleitelektronikern'     : u'prozessleitelektronikern',
        u'prozeßleitelektronikers'     : u'prozessleitelektronikers',
        u'prozeßlüftung'               : u'prozesslüftung',
        u'prozeßlüftungen'             : u'prozesslüftungen',
        u'prozeßordnung'               : u'prozessordnung',
        u'prozeßrecht'                 : u'prozessrecht',
        u'prozeßunfähig'               : u'prozessunfähig',
        u'prozeßunfähigkeit'           : u'prozessunfähigkeit',
        u'prozeßverlauf'               : u'prozessverlauf',
        u'prozeßverlaufe'              : u'prozessverlaufe',
        u'prozeßverlaufes'             : u'prozessverlaufes',
        u'prozeßverlaufs'              : u'prozessverlaufs',
        u'prozeßverläufe'              : u'prozessverläufe',
        u'prozeßverläufen'             : u'prozessverläufen',
        u'präzisionsmeßgerät'          : u'präzisionsmessgerät',
        u'prüfungsausschuß'            : u'prüfungsausschuss',
        u'pudelnaß'                    : u'pudelnass',
        u'pulverfaß'                   : u'pulverfass',
        u'pußle'                       : u'pussle',
        u'pußlest'                     : u'pusslest',
        u'pußlet'                      : u'pusslet',
        u'pußlig'                      : u'pusslig',
        u'pökelfaß'                    : u'pökelfass',
        u'qualitätsbewußt'             : u'qualitätsbewusst',
        u'qualitätsbewußte'            : u'qualitätsbewusste',
        u'qualitätsbewußtem'           : u'qualitätsbewusstem',
        u'qualitätsbewußten'           : u'qualitätsbewussten',
        u'qualitätsbewußter'           : u'qualitätsbewusster',
        u'qualitätsbewußtes'           : u'qualitätsbewusstes',
        u'qualitätsbewußtsein'         : u'qualitätsbewusstsein',
        u'quartalabschluß'             : u'quartalabschluss',
        u'quartalsabschluß'            : u'quartalsabschluss',
        u'quatschnaß'                  : u'quatschnass',
        u'quaßle'                      : u'quassle',
        u'quaßlest'                    : u'quasslest',
        u'quaßlet'                     : u'quasslet',
        u'quellfluß'                   : u'quellfluss',
        u'quelluft'                    : u'quellluft',
        u'quellüftung'                 : u'quelllüftung',
        u'quellüftungen'               : u'quelllüftungen',
        u'quentchen'                   : u'quäntchen',
        u'quentchens'                  : u'quäntchens',
        u'querpaß'                     : u'querpass',
        u'querschuß'                   : u'querschuss',
        u'quicke'                      : u'quick',
        u'quickstep'                   : u'quickstepp',
        u'quicksteps'                  : u'quickstepps',
        u'race'                        : u'rasse',
        u'racen'                       : u'rassen',
        u'radical'                     : u'radikal',
        u'radicale'                    : u'radikale',
        u'radicalem'                   : u'radikalem',
        u'radicalen'                   : u'radikalen',
        u'radicaler'                   : u'radikaler',
        u'radicalere'                  : u'radikalere',
        u'radicalerem'                 : u'radikalerem',
        u'radicaleren'                 : u'radikaleren',
        u'radicalerer'                 : u'radikalerer',
        u'radicaleres'                 : u'radikaleres',
        u'radicales'                   : u'radikales',
        u'radicalste'                  : u'radikalste',
        u'radicalstem'                 : u'radikalstem',
        u'radicalsten'                 : u'radikalsten',
        u'radicalster'                 : u'radikalster',
        u'radicalstes'                 : u'radikalstes',
        u'radikalenerlaß'              : u'radikalenerlass',
        u'raketenabschußrampe'         : u'raketenabschussrampe',
        u'raketenabschußrampen'        : u'raketenabschussrampen',
        u'rammaschine'                 : u'rammmaschine',
        u'ranlaßt'                     : u'ranlasst',
        u'ranläßt'                     : u'ranlässt',
        u'rassenhaß'                   : u'rassenhass',
        u'rath'                        : u'rat',
        u'rathe'                       : u'rate',
        u'rathes'                      : u'rates',
        u'raths'                       : u'rats',
        u'ratsbeschluß'                : u'ratsbeschluss',
        u'ratschluß'                   : u'ratschluss',
        u'rauchfaß'                    : u'rauchfass',
        u'rauchschutzabschluß'         : u'rauchschutzabschluss',
        u'rauh'                        : u'rau',
        u'rauhbank'                    : u'raubank',
        u'rauhbauz'                    : u'raubauz',
        u'rauhbauzig'                  : u'raubauzig',
        u'rauhbauzige'                 : u'raubauzige',
        u'rauhbauzigem'                : u'raubauzigem',
        u'rauhbauzigen'                : u'raubauzigen',
        u'rauhbauziger'                : u'raubauziger',
        u'rauhbauzigere'               : u'raubauzigere',
        u'rauhbauzigerem'              : u'raubauzigerem',
        u'rauhbauzigeren'              : u'raubauzigeren',
        u'rauhbauzigerer'              : u'raubauzigerer',
        u'rauhbauzigeres'              : u'raubauzigeres',
        u'rauhbauziges'                : u'raubauziges',
        u'rauhbauzigste'               : u'raubauzigste',
        u'rauhbauzigstem'              : u'raubauzigstem',
        u'rauhbauzigsten'              : u'raubauzigsten',
        u'rauhbauzigster'              : u'raubauzigster',
        u'rauhbauzigstes'              : u'raubauzigstes',
        u'rauhbein'                    : u'raubein',
        u'rauhbeine'                   : u'raubeine',
        u'rauhbeinen'                  : u'raubeinen',
        u'rauhbeines'                  : u'raubeines',
        u'rauhbeinig'                  : u'raubeinig',
        u'rauhbeinige'                 : u'raubeinige',
        u'rauhbeinigem'                : u'raubeinigem',
        u'rauhbeinigen'                : u'raubeinigen',
        u'rauhbeiniger'                : u'raubeiniger',
        u'rauhbeinigere'               : u'raubeinigere',
        u'rauhbeinigerem'              : u'raubeinigerem',
        u'rauhbeinigeren'              : u'raubeinigeren',
        u'rauhbeinigerer'              : u'raubeinigerer',
        u'rauhbeinigeres'              : u'raubeinigeres',
        u'rauhbeiniges'                : u'raubeiniges',
        u'rauhbeinigste'               : u'raubeinigste',
        u'rauhbeinigstem'              : u'raubeinigstem',
        u'rauhbeinigsten'              : u'raubeinigsten',
        u'rauhbeinigster'              : u'raubeinigster',
        u'rauhbeinigstes'              : u'raubeinigstes',
        u'rauhbeins'                   : u'raubeins',
        u'rauhbewurf'                  : u'raubewurf',
        u'rauhbewurfe'                 : u'raubewurfe',
        u'rauhbewurfes'                : u'raubewurfes',
        u'rauhbewurfs'                 : u'raubewurfs',
        u'rauhbewürfe'                 : u'raubewürfe',
        u'rauhbewürfen'                : u'raubewürfen',
        u'rauhborstig'                 : u'rauborstig',
        u'rauhbänke'                   : u'raubänke',
        u'rauhbänken'                  : u'raubänken',
        u'rauhe'                       : u'raue',
        u'rauhem'                      : u'rauem',
        u'rauhen'                      : u'rauen',
        u'rauhend'                     : u'rauend',
        u'rauher'                      : u'rauer',
        u'rauhes'                      : u'raues',
        u'rauhest'                     : u'rauest',
        u'rauhet'                      : u'rauet',
        u'rauhfasertapete'             : u'raufasertapete',
        u'rauhfasertapeten'            : u'raufasertapeten',
        u'rauhfrost'                   : u'raufrost',
        u'rauhfroste'                  : u'raufroste',
        u'rauhfrostes'                 : u'raufrostes',
        u'rauhfrosts'                  : u'raufrosts',
        u'rauhfröste'                  : u'raufröste',
        u'rauhfrösten'                 : u'raufrösten',
        u'rauhfusskauz'                : u'raufußkauz',
        u'rauhfusskauze'               : u'raufußkauze',
        u'rauhfusskauzes'              : u'raufußkauzes',
        u'rauhfusskäuze'               : u'raufußkäuze',
        u'rauhfusskäuzen'              : u'raufußkäuzen',
        u'rauhfußkauz'                 : u'raufußkauz',
        u'rauhfußkauze'                : u'raufußkauze',
        u'rauhfußkauzes'               : u'raufußkauzes',
        u'rauhfußkäuze'                : u'raufußkäuze',
        u'rauhfußkäuzen'               : u'raufußkäuzen',
        u'rauhhaardackel'              : u'rauhaardackel',
        u'rauhhaardackeln'             : u'rauhaardackeln',
        u'rauhhaardackels'             : u'rauhaardackels',
        u'rauhheit'                    : u'rauheit',
        u'rauhleder'                   : u'rauleder',
        u'rauhledern'                  : u'rauledern',
        u'rauhleders'                  : u'rauleders',
        u'rauhputz'                    : u'rauputz',
        u'rauhputze'                   : u'rauputze',
        u'rauhputzen'                  : u'rauputzen',
        u'rauhputzes'                  : u'rauputzes',
        u'rauhreif'                    : u'raureif',
        u'rauhreife'                   : u'raureife',
        u'rauhreifes'                  : u'raureifes',
        u'rauhreifgeäste'              : u'raureifgeäste',
        u'rauhreifs'                   : u'raureifs',
        u'rauhst'                      : u'raust',
        u'rauht'                       : u'raut',
        u'rauhte'                      : u'raute',
        u'rauhten'                     : u'rauten',
        u'rauhtest'                    : u'rautest',
        u'rauhtet'                     : u'rautet',
        u'rauhware'                    : u'rauware',
        u'rauhwaren'                   : u'rauwaren',
        u'raumschiffahrt'              : u'raumschifffahrt',
        u'rausschmiß'                  : u'rausschmiss',
        u'reaction'                    : u'reaktion',
        u'reactionen'                  : u'reaktionen',
        u'realschulabschluß'           : u'realschulabschluss',
        u'rebschoß'                    : u'rebschoss',
        u'recept'                      : u'rezept',
        u'recepte'                     : u'rezepte',
        u'recepten'                    : u'rezepten',
        u'receptes'                    : u'rezeptes',
        u'recepts'                     : u'rezepts',
        u'rechtsausschuß'              : u'rechtsausschuss',
        u'rechtsbewußtsein'            : u'rechtsbewusstsein',
        u'rechtsbewußtseins'           : u'rechtsbewusstseins',
        u'rechtshandelung'             : u'rechtshandlung',
        u'rechtsschuß'                 : u'rechtsschuss',
        u'redaktionsschluß'            : u'redaktionsschluss',
        u'redefluß'                    : u'redefluss',
        u'reduction'                   : u'reduktion',
        u'reductionen'                 : u'reduktionen',
        u'reflectirtes'                : u'reflektiertes',
        u'regenereigniß'               : u'regenereignis',
        u'regenfaß'                    : u'regenfass',
        u'regenguß'                    : u'regenguss',
        u'regennaß'                    : u'regennass',
        u'regierungsbeschluß'          : u'regierungsbeschluss',
        u'regionalexpreß'              : u'regionalexpress',
        u'regreß'                      : u'regress',
        u'regreßanspruch'              : u'regressanspruch',
        u'regreßpflicht'               : u'regresspflicht',
        u'regreßpflichtig'             : u'regresspflichtig',
        u'reifeprozeß'                 : u'reifeprozess',
        u'reifungsprozeß'              : u'reifungsprozess',
        u'reisekostenzuschuß'          : u'reisekostenzuschuss',
        u'reisepaß'                    : u'reisepass',
        u'reißverschluß'               : u'reißverschluss',
        u'reißverschlußsystem'         : u'reißverschlusssystem',
        u'reißverschlußsysteme'        : u'reißverschlusssysteme',
        u'reißverschlußsystemen'       : u'reißverschlusssystemen',
        u'reißverschlußsystems'        : u'reißverschlusssystems',
        u'renaissanceschloß'           : u'renaissanceschloss',
        u'renkverschluß'               : u'renkverschluss',
        u'reproduktionsprozeß'         : u'reproduktionsprozess',
        u'repräsentationsschluß'       : u'repräsentationsschluss',
        u'reschenpaß'                  : u'reschenpass',
        u'respizir'                    : u'respizier',
        u'respizire'                   : u'respiziere',
        u'respiziren'                  : u'respizieren',
        u'respizirend'                 : u'respizierend',
        u'respizirest'                 : u'respizierest',
        u'respiziret'                  : u'respizieret',
        u'respizirst'                  : u'respizierst',
        u'respizirt'                   : u'respiziert',
        u'respizirte'                  : u'respizierte',
        u'respizirten'                 : u'respizierten',
        u'respizirtest'                : u'respiziertest',
        u'respizirtet'                 : u'respiziertet',
        u'rettungsschuß'               : u'rettungsschuss',
        u'rez-de-chaussée'             : u'rez-de-chaussée',
        u'rezeptionsprozeß'            : u'rezeptionsprozess',
        u'rezeß'                       : u'rezess',
        u'rheinschiffahrt'             : u'rheinschifffahrt',
        u'rheinschiffahrten'           : u'rheinschifffahrten',
        u'richterwahlausschuß'         : u'richterwahlausschuss',
        u'riesenkänguruh'              : u'riesenkänguru',
        u'riesenroß'                   : u'riesenross',
        u'riß'                         : u'riss',
        u'rißfest'                     : u'rissfest',
        u'rißfeste'                    : u'rissfeste',
        u'rißfestem'                   : u'rissfestem',
        u'rißfesten'                   : u'rissfesten',
        u'rißfester'                   : u'rissfester',
        u'rißfestere'                  : u'rissfestere',
        u'rißfesterem'                 : u'rissfesterem',
        u'rißfesteren'                 : u'rissfesteren',
        u'rißfesterer'                 : u'rissfesterer',
        u'rißfesteres'                 : u'rissfesteres',
        u'rißfestes'                   : u'rissfestes',
        u'rißfesteste'                 : u'rissfesteste',
        u'rißfestestem'                : u'rissfestestem',
        u'rißfestesten'                : u'rissfestesten',
        u'rißfestester'                : u'rissfestester',
        u'rißfestestes'                : u'rissfestestes',
        u'rißt'                        : u'risst',
        u'rißwunde'                    : u'risswunde',
        u'rißwunden'                   : u'risswunden',
        u'roheit'                      : u'rohheit',
        u'roheiten'                    : u'rohheiten',
        u'rohstofförderung'            : u'rohstoffförderung',
        u'rohstofförderungen'          : u'rohstoffförderungen',
        u'rolladen'                    : u'rollladen',
        u'rolladens'                   : u'rollladens',
        u'rolläden'                    : u'rollläden',
        u'roteturmpaß'                 : u'roteturmpass',
        u'rotguß'                      : u'rotguss',
        u'roth'                        : u'rot',
        u'rothe'                       : u'rote',
        u'rothem'                      : u'rotem',
        u'rothen'                      : u'roten',
        u'rother'                      : u'roter',
        u'rothere'                     : u'rotere',
        u'rotherem'                    : u'roterem',
        u'rotheren'                    : u'roteren',
        u'rotherer'                    : u'roterer',
        u'rotheres'                    : u'roteres',
        u'rothes'                      : u'rotes',
        u'rotheste'                    : u'roteste',
        u'rothestem'                   : u'rotestem',
        u'rothesten'                   : u'rotesten',
        u'rothester'                   : u'rotester',
        u'rothestes'                   : u'rotestes',
        u'rotirenden'                  : u'rotierenden',
        u'rotsiegelbeschluß'           : u'rotsiegelbeschluss',
        u'roulett'                     : u'roulette',
        u'roulett-tisch'               : u'roulettetisch',
        u'roulett-tische'              : u'roulettetische',
        u'roulett-tischen'             : u'roulettetischen',
        u'roulett-tisches'             : u'roulettetisches',
        u'roulett-tischs'              : u'roulettetischs',
        u'roulettisch'                 : u'roulettetisch',
        u'roulettische'                : u'roulettetische',
        u'roulettischen'               : u'roulettetischen',
        u'roulettisches'               : u'roulettetisches',
        u'roulettischs'                : u'roulettetischs',
        u'roulettkugel'                : u'roulettekugel',
        u'roulettkugeln'               : u'roulettekugeln',
        u'rouletts'                    : u'roulettes',
        u'rouletttisch'                : u'roulettetisch',
        u'rouletttische'               : u'roulettetische',
        u'rouletttischen'              : u'roulettetischen',
        u'rouletttisches'              : u'roulettetisches',
        u'rouletttischs'               : u'roulettetischs',
        u'rowdies'                     : u'rowdys',
        u'roß'                         : u'ross',
        u'roßapfel'                    : u'rossapfel',
        u'roßapfels'                   : u'rossapfels',
        u'roßarzt'                     : u'rossarzt',
        u'roßbreiten'                  : u'rossbreiten',
        u'roßhaar'                     : u'rosshaar',
        u'roßhaare'                    : u'rosshaare',
        u'roßhaaren'                   : u'rosshaaren',
        u'roßhaares'                   : u'rosshaares',
        u'roßhaarmatratze'             : u'rosshaarmatratze',
        u'roßhaarmatratzen'            : u'rosshaarmatratzen',
        u'roßhaars'                    : u'rosshaars',
        u'roßkamm'                     : u'rosskamm',
        u'roßkastanie'                 : u'rosskastanie',
        u'roßkastanien'                : u'rosskastanien',
        u'roßkur'                      : u'rosskur',
        u'roßkuren'                    : u'rosskuren',
        u'roßschlachter'               : u'rossschlachter',
        u'roßschlächter'               : u'rossschlächter',
        u'roßtrappe'                   : u'rosstrappe',
        u'roßtäuscher'                 : u'rosstäuscher',
        u'roßtäuscherei'               : u'rosstäuscherei',
        u'roßtäuschereien'             : u'rosstäuschereien',
        u'roßtäuschern'                : u'rosstäuschern',
        u'roßtäuschers'                : u'rosstäuschers',
        u'roßtäuschertrick'            : u'rosstäuschertrick',
        u'roßvolk'                     : u'rossvolk',
        u'roßvolke'                    : u'rossvolke',
        u'roßvolkes'                   : u'rossvolkes',
        u'roßvolks'                    : u'rossvolks',
        u'roßvölker'                   : u'rossvölker',
        u'roßvölkern'                  : u'rossvölkern',
        u'roßäpfel'                    : u'rossäpfel',
        u'roßäpfeln'                   : u'rossäpfeln',
        u'rumfaß'                      : u'rumfass',
        u'runderlaß'                   : u'runderlass',
        u'runterlaßt'                  : u'runterlasst',
        u'runterläßt'                  : u'runterlässt',
        u'russifizierungsprozeß'       : u'russifizierungsprozess',
        u'ruthe'                       : u'rute',
        u'ruthen'                      : u'ruten',
        u'ruthengänger'                : u'rutengänger',
        u'rußland'                     : u'russland',
        u'rußland-versteher'           : u'russland-versteher',
        u'rußland-verstehern'          : u'russland-verstehern',
        u'rußland-verstehers'          : u'russland-verstehers',
        u'rußlanddeutsch'              : u'russlanddeutsch',
        u'rußlanddeutsche'             : u'russlanddeutsche',
        u'rußlanddeutschem'            : u'russlanddeutschem',
        u'rußlanddeutschen'            : u'russlanddeutschen',
        u'rußlanddeutscher'            : u'russlanddeutscher',
        u'rußlanddeutsches'            : u'russlanddeutsches',
        u'rußlandgeschäft'             : u'russlandgeschäft',
        u'rußlandreise'                : u'russlandreise',
        u'rußlandreisen'               : u'russlandreisen',
        u'rußlands'                    : u'russlands',
        u'rußlandversteher'            : u'russlandversteher',
        u'rußlandverstehern'           : u'russlandverstehern',
        u'rußlandverstehers'           : u'russlandverstehers',
        u'rußländer'                   : u'russländer',
        u'rußländern'                  : u'russländern',
        u'rußländers'                  : u'russländers',
        u'rußländisch'                 : u'russländisch',
        u'rußländische'                : u'russländische',
        u'rußländischem'               : u'russländischem',
        u'rußländischen'               : u'russländischen',
        u'rußländischer'               : u'russländischer',
        u'rußländischere'              : u'russländischere',
        u'rußländischerem'             : u'russländischerem',
        u'rußländischeren'             : u'russländischeren',
        u'rußländischerer'             : u'russländischerer',
        u'rußländischeres'             : u'russländischeres',
        u'rußländisches'               : u'russländisches',
        u'rußländischste'              : u'russländischste',
        u'rußländischstem'             : u'russländischstem',
        u'rußländischsten'             : u'russländischsten',
        u'rußländischster'             : u'russländischster',
        u'rußländischstes'             : u'russländischstes',
        u'räterußland'                 : u'räterussland',
        u'räthe'                       : u'räte',
        u'räthen'                      : u'räten',
        u'räthselhafte'                : u'rätselhafte',
        u'räthselhaften'               : u'rätselhaften',
        u'räucherfäßchen'              : u'räucherfässchen',
        u'römisch-catholisch'          : u'römisch-katholisch',
        u'römisch-catholische'         : u'römisch-katholische',
        u'römisch-catholischem'        : u'römisch-katholischem',
        u'römisch-catholischen'        : u'römisch-katholischen',
        u'römisch-catholischer'        : u'römisch-katholischer',
        u'römisch-catholisches'        : u'römisch-katholisches',
        u'röntgenpaß'                  : u'röntgenpass',
        u'röther'                      : u'röter',
        u'röthere'                     : u'rötere',
        u'rötherem'                    : u'röterem',
        u'rötheren'                    : u'röteren',
        u'rötherer'                    : u'röterer',
        u'rötheres'                    : u'röteres',
        u'rötheste'                    : u'röteste',
        u'röthestem'                   : u'rötestem',
        u'röthesten'                   : u'rötesten',
        u'röthester'                   : u'rötester',
        u'röthestes'                   : u'rötestes',
        u'rößchen'                     : u'rösschen',
        u'rößlein'                     : u'rösslein',
        u'rückfluß'                    : u'rückfluss',
        u'rückpaß'                     : u'rückpass',
        u'rückpaßregel'                : u'rückpassregel',
        u'rückschluß'                  : u'rückschluss',
        u'sahneyoghurt'                : u'sahnejoghurt',
        u'sahneyoghurts'               : u'sahnejoghurts',
        u'saisonschluß'                : u'saisonschluss',
        u'salutschuß'                  : u'salutschuss',
        u'salzfaß'                     : u'salzfass',
        u'salzfäßchen'                 : u'salzfässchen',
        u'salzfäßchens'                : u'salzfässchens',
        u'samenerguß'                  : u'samenerguss',
        u'sammelanschluß'              : u'sammelanschluss',
        u'sammelnußfrucht'             : u'sammelnussfrucht',
        u'sammelnußfrüchte'            : u'sammelnussfrüchte',
        u'sammelnußfrüchten'           : u'sammelnussfrüchten',
        u'sarraß'                      : u'sarrass',
        u'satellitenphoto'             : u'satellitenfoto',
        u'satellitenphotos'            : u'satellitenfotos',
        u'satellitentelephon'          : u'satellitentelefon',
        u'satellitentelephone'         : u'satellitentelefone',
        u'satellitentelephonen'        : u'satellitentelefonen',
        u'satellitentelephons'         : u'satellitentelefons',
        u'satzschlußzeichen'           : u'satzschlusszeichen',
        u'satzschlußzeichens'          : u'satzschlusszeichens',
        u'saunaaufguß'                 : u'saunaaufguss',
        u'saßnitz'                     : u'sassnitz',
        u'schaffensprozeß'             : u'schaffensprozess',
        u'schalenguß'                  : u'schalenguss',
        u'schallehre'                  : u'schalllehre',
        u'schalleiter'                 : u'schallleiter',
        u'schalloch'                   : u'schallloch',
        u'schalterschluß'              : u'schalterschluss',
        u'scharm'                      : u'charme',
        u'scharmant'                   : u'charmant',
        u'scharmante'                  : u'charmante',
        u'scharmantem'                 : u'charmantem',
        u'scharmanten'                 : u'charmanten',
        u'scharmanter'                 : u'charmanter',
        u'scharmantere'                : u'charmantere',
        u'scharmanterem'               : u'charmanterem',
        u'scharmanteren'               : u'charmanteren',
        u'scharmanterer'               : u'charmanterer',
        u'scharmanteres'               : u'charmanteres',
        u'scharmantes'                 : u'charmantes',
        u'scharmanteste'               : u'charmanteste',
        u'scharmantestem'              : u'charmantestem',
        u'scharmantesten'              : u'charmantesten',
        u'scharmantester'              : u'charmantester',
        u'scharmantestes'              : u'charmantestes',
        u'scharms'                     : u'charmes',
        u'schattenriß'                 : u'schattenriss',
        u'schaumkuß'                   : u'schaumkuss',
        u'schauprozeß'                 : u'schauprozess',
        u'schaß'                       : u'schass',
        u'schaßt'                      : u'schasst',
        u'schaßte'                     : u'schasste',
        u'schaßten'                    : u'schassten',
        u'schaßtest'                   : u'schasstest',
        u'schaßtet'                    : u'schasstet',
        u'scheidungsprozeß'            : u'scheidungsprozess',
        u'schellenaß'                  : u'schellenass',
        u'schiffahrt'                  : u'schifffahrt',
        u'schiffahrten'                : u'schifffahrten',
        u'schiffahrtsabgabe'           : u'schifffahrtsabgabe',
        u'schiffahrtsabgaben'          : u'schifffahrtsabgaben',
        u'schiffahrtsbehörde'          : u'schifffahrtsbehörde',
        u'schiffahrtsbehörden'         : u'schifffahrtsbehörden',
        u'schiffahrtsgericht'          : u'schifffahrtsgericht',
        u'schiffahrtsgerichte'         : u'schifffahrtsgerichte',
        u'schiffahrtsgerichten'        : u'schifffahrtsgerichten',
        u'schiffahrtsgerichtes'        : u'schifffahrtsgerichtes',
        u'schiffahrtsgerichts'         : u'schifffahrtsgerichts',
        u'schiffahrtsgesellschaft'     : u'schifffahrtsgesellschaft',
        u'schiffahrtsgesellschaften'   : u'schifffahrtsgesellschaften',
        u'schiffahrtskunde'            : u'schifffahrtskunde',
        u'schiffahrtslinie'            : u'schifffahrtslinie',
        u'schiffahrtslinien'           : u'schifffahrtslinien',
        u'schiffahrtspolizei'          : u'schifffahrtspolizei',
        u'schiffahrtspolizeien'        : u'schifffahrtspolizeien',
        u'schiffahrtsrecht'            : u'schifffahrtsrecht',
        u'schiffahrtsrechte'           : u'schifffahrtsrechte',
        u'schiffahrtsrechten'          : u'schifffahrtsrechten',
        u'schiffahrtsrechtes'          : u'schifffahrtsrechtes',
        u'schiffahrtsrechts'           : u'schifffahrtsrechts',
        u'schiffahrtsroute'            : u'schifffahrtsroute',
        u'schiffahrtsrouten'           : u'schifffahrtsrouten',
        u'schiffahrtsstrasse'          : u'schifffahrtsstraße',
        u'schiffahrtsstrassen'         : u'schifffahrtsstraßen',
        u'schiffahrtsstraße'           : u'schifffahrtsstraße',
        u'schiffahrtsstraßen'          : u'schifffahrtsstraßen',
        u'schiffahrtsweg'              : u'schifffahrtsweg',
        u'schiffahrtswege'             : u'schifffahrtswege',
        u'schiffahrtswegen'            : u'schifffahrtswegen',
        u'schiffahrtsweges'            : u'schifffahrtsweges',
        u'schiffahrtszeichen'          : u'schifffahrtszeichen',
        u'schiffahrtszeichens'         : u'schifffahrtszeichens',
        u'schiffscanal'                : u'schiffskanal',
        u'schikoree'                   : u'chicorée',
        u'schinesen'                   : u'chinesen',
        u'schiß'                       : u'schiss',
        u'schißlaweng'                 : u'schisslaweng',
        u'schißt'                      : u'schisst',
        u'schlachtroß'                 : u'schlachtross',
        u'schlagfluß'                  : u'schlagfluss',
        u'schlammasse'                 : u'schlammmasse',
        u'schlangenbiß'                : u'schlangenbiss',
        u'schleppinsel'                : u'schlepppinsel',
        u'schleppinseln'               : u'schlepppinseln',
        u'schleppinsels'               : u'schlepppinsels',
        u'schleppschiffahrt'           : u'schleppschifffahrt',
        u'schlichtungsausschuß'        : u'schlichtungsausschuss',
        u'schliesse'                   : u'schließe',
        u'schliessen'                  : u'schließen',
        u'schlitzverschluß'            : u'schlitzverschluss',
        u'schliß'                      : u'schliss',
        u'schlißt'                     : u'schlisst',
        u'schloß'                      : u'schloss',
        u'schloßartige'                : u'schlossartige',
        u'schloßbaumeister'            : u'schlossbaumeister',
        u'schloßbaumeistern'           : u'schlossbaumeistern',
        u'schloßbaumeisters'           : u'schlossbaumeisters',
        u'schloßberg'                  : u'schlossberg',
        u'schloßberge'                 : u'schlossberge',
        u'schloßbergen'                : u'schlossbergen',
        u'schloßberges'                : u'schlossberges',
        u'schloßbergs'                 : u'schlossbergs',
        u'schloßbrauerei'              : u'schlossbrauerei',
        u'schloßbrauereien'            : u'schlossbrauereien',
        u'schloßbrunnen'               : u'schlossbrunnen',
        u'schloßgarten'                : u'schlossgarten',
        u'schloßgartens'               : u'schlossgartens',
        u'schloßgemächer'              : u'schlossgemächer',
        u'schloßgespenst'              : u'schlossgespenst',
        u'schloßgespenste'             : u'schlossgespenste',
        u'schloßgespenster'            : u'schlossgespenster',
        u'schloßgespenstern'           : u'schlossgespenstern',
        u'schloßgespenstes'            : u'schlossgespenstes',
        u'schloßgespensts'             : u'schlossgespensts',
        u'schloßgärten'                : u'schlossgärten',
        u'schloßherr'                  : u'schlossherr',
        u'schloßherren'                : u'schlossherren',
        u'schloßherrin'                : u'schlossherrin',
        u'schloßherrinnen'             : u'schlossherrinnen',
        u'schloßherrn'                 : u'schlossherrn',
        u'schloßherrschaft'            : u'schlossherrschaft',
        u'schloßhof'                   : u'schlosshof',
        u'schloßhofes'                 : u'schlosshofes',
        u'schloßhund'                  : u'schlosshund',
        u'schloßhunde'                 : u'schlosshunde',
        u'schloßhunden'                : u'schlosshunden',
        u'schloßhundes'                : u'schlosshundes',
        u'schloßhunds'                 : u'schlosshunds',
        u'schloßkapelle'               : u'schlosskapelle',
        u'schloßkapellen'              : u'schlosskapellen',
        u'schloßkirche'                : u'schlosskirche',
        u'schloßkirchen'               : u'schlosskirchen',
        u'schloßkorb'                  : u'schlosskorb',
        u'schloßkutscher'              : u'schlosskutscher',
        u'schloßkutschern'             : u'schlosskutschern',
        u'schloßkutschers'             : u'schlosskutschers',
        u'schloßpark'                  : u'schlosspark',
        u'schloßparke'                 : u'schlossparke',
        u'schloßparken'                : u'schlossparken',
        u'schloßparks'                 : u'schlossparks',
        u'schloßpärke'                 : u'schlosspärke',
        u'schloßpärken'                : u'schlosspärken',
        u'schloßruine'                 : u'schlossruine',
        u'schloßruinen'                : u'schlossruinen',
        u'schloßt'                     : u'schlosst',
        u'schloßtheater'               : u'schlosstheater',
        u'schloßtheaters'              : u'schlosstheaters',
        u'schloßtor'                   : u'schlosstor',
        u'schloßturm'                  : u'schlossturm',
        u'schloßturme'                 : u'schlossturme',
        u'schloßturmes'                : u'schlossturmes',
        u'schloßturms'                 : u'schlossturms',
        u'schloßtürme'                 : u'schlosstürme',
        u'schloßtürmen'                : u'schlosstürmen',
        u'schloßuhr'                   : u'schlossuhr',
        u'schloßwiese'                 : u'schlosswiese',
        u'schluß'                      : u'schluss',
        u'schluß-sitzung'              : u'schluss-sitzung',
        u'schlußabrechnung'            : u'schlussabrechnung',
        u'schlußabrechnungen'          : u'schlussabrechnungen',
        u'schlußabstimmung'            : u'schlussabstimmung',
        u'schlußakkord'                : u'schlussakkord',
        u'schlußakkorde'               : u'schlussakkorde',
        u'schlußakkorden'              : u'schlussakkorden',
        u'schlußakkordes'              : u'schlussakkordes',
        u'schlußakkords'               : u'schlussakkords',
        u'schlußakt'                   : u'schlussakt',
        u'schlußakte'                  : u'schlussakte',
        u'schlußakten'                 : u'schlussakten',
        u'schlußaktes'                 : u'schlussaktes',
        u'schlußakts'                  : u'schlussakts',
        u'schlußapotheose'             : u'schlussapotheose',
        u'schlußapothese'              : u'schlussapothese',
        u'schlußapplaus'               : u'schlussapplaus',
        u'schlußapplause'              : u'schlussapplause',
        u'schlußapplausen'             : u'schlussapplausen',
        u'schlußapplauses'             : u'schlussapplauses',
        u'schlußball'                  : u'schlussball',
        u'schlußbearbeitung'           : u'schlussbearbeitung',
        u'schlußbearbeitungen'         : u'schlussbearbeitungen',
        u'schlußbemerkung'             : u'schlussbemerkung',
        u'schlußbemerkungen'           : u'schlussbemerkungen',
        u'schlußbericht'               : u'schlussbericht',
        u'schlußberichte'              : u'schlussberichte',
        u'schlußberichten'             : u'schlussberichten',
        u'schlußberichtes'             : u'schlussberichtes',
        u'schlußberichts'              : u'schlussberichts',
        u'schlußbesprechung'           : u'schlussbesprechung',
        u'schlußbesprechungen'         : u'schlussbesprechungen',
        u'schlußbestimmungen'          : u'schlussbestimmungen',
        u'schlußbetrachtung'           : u'schlussbetrachtung',
        u'schlußbilanz'                : u'schlussbilanz',
        u'schlußbild'                  : u'schlussbild',
        u'schlußbilde'                 : u'schlussbilde',
        u'schlußbilder'                : u'schlussbilder',
        u'schlußbildern'               : u'schlussbildern',
        u'schlußbildes'                : u'schlussbildes',
        u'schlußbilds'                 : u'schlussbilds',
        u'schlußbrief'                 : u'schlussbrief',
        u'schlußchor'                  : u'schlusschor',
        u'schlußchore'                 : u'schlusschore',
        u'schlußchores'                : u'schlusschores',
        u'schlußchors'                 : u'schlusschors',
        u'schlußchöre'                 : u'schlusschöre',
        u'schlußchören'                : u'schlusschören',
        u'schlußdokument'              : u'schlussdokument',
        u'schlußdokumente'             : u'schlussdokumente',
        u'schlußdokumenten'            : u'schlussdokumenten',
        u'schlußdokumentes'            : u'schlussdokumentes',
        u'schlußdokuments'             : u'schlussdokuments',
        u'schlußdrittel'               : u'schlussdrittel',
        u'schlußdritteln'              : u'schlussdritteln',
        u'schlußdrittels'              : u'schlussdrittels',
        u'schlußeffekt'                : u'schlusseffekt',
        u'schlußeffekte'               : u'schlusseffekte',
        u'schlußeffekten'              : u'schlusseffekten',
        u'schlußeffektes'              : u'schlusseffektes',
        u'schlußeffekts'               : u'schlusseffekts',
        u'schlußendlich'               : u'schlussendlich',
        u'schlußendliche'              : u'schlussendliche',
        u'schlußendlichem'             : u'schlussendlichem',
        u'schlußendlichen'             : u'schlussendlichen',
        u'schlußendlicher'             : u'schlussendlicher',
        u'schlußendliches'             : u'schlussendliches',
        u'schlußerklärung'             : u'schlusserklärung',
        u'schlußerklärungen'           : u'schlusserklärungen',
        u'schlußetappe'                : u'schlussetappe',
        u'schlußetappen'               : u'schlussetappen',
        u'schlußfeier'                 : u'schlussfeier',
        u'schlußfeiern'                : u'schlussfeiern',
        u'schlußfeste'                 : u'schlussfeste',
        u'schlußfolge'                 : u'schlussfolge',
        u'schlußfolger'                : u'schlussfolger',
        u'schlußfolgere'               : u'schlussfolgere',
        u'schlußfolgern'               : u'schlussfolgern',
        u'schlußfolgernd'              : u'schlussfolgernd',
        u'schlußfolgernden'            : u'schlussfolgernden',
        u'schlußfolgerst'              : u'schlussfolgerst',
        u'schlußfolgert'               : u'schlussfolgert',
        u'schlußfolgerte'              : u'schlussfolgerte',
        u'schlußfolgerten'             : u'schlussfolgerten',
        u'schlußfolgertest'            : u'schlussfolgertest',
        u'schlußfolgertet'             : u'schlussfolgertet',
        u'schlußfolgerung'             : u'schlussfolgerung',
        u'schlußfolgerungen'           : u'schlussfolgerungen',
        u'schlußfolgre'                : u'schlussfolgre',
        u'schlußformel'                : u'schlussformel',
        u'schlußformeln'               : u'schlussformeln',
        u'schlußhandlung'              : u'schlusshandlung',
        u'schlußkapitel'               : u'schlusskapitel',
        u'schlußkapiteln'              : u'schlusskapiteln',
        u'schlußkapitels'              : u'schlusskapitels',
        u'schlußkatastrophe'           : u'schlusskatastrophe',
        u'schlußkette'                 : u'schlusskette',
        u'schlußketten'                : u'schlussketten',
        u'schlußkundgebung'            : u'schlusskundgebung',
        u'schlußkundgebungen'          : u'schlusskundgebungen',
        u'schlußkurs'                  : u'schlusskurs',
        u'schlußleuchte'               : u'schlussleuchte',
        u'schlußlicht'                 : u'schlusslicht',
        u'schlußlichte'                : u'schlusslichte',
        u'schlußlichter'               : u'schlusslichter',
        u'schlußlichtern'              : u'schlusslichtern',
        u'schlußlichtes'               : u'schlusslichtes',
        u'schlußlichts'                : u'schlusslichts',
        u'schlußläufer'                : u'schlussläufer',
        u'schlußläuferin'              : u'schlussläuferin',
        u'schlußläuferinnen'           : u'schlussläuferinnen',
        u'schlußläufern'               : u'schlussläufern',
        u'schlußläufers'               : u'schlussläufers',
        u'schlußmann'                  : u'schlussmann',
        u'schlußnote'                  : u'schlussnote',
        u'schlußnotierung'             : u'schlussnotierung',
        u'schlußpfiff'                 : u'schlusspfiff',
        u'schlußpfiffe'                : u'schlusspfiffe',
        u'schlußpfiffen'               : u'schlusspfiffen',
        u'schlußpfiffes'               : u'schlusspfiffes',
        u'schlußpfiffs'                : u'schlusspfiffs',
        u'schlußphase'                 : u'schlussphase',
        u'schlußphasen'                : u'schlussphasen',
        u'schlußpunkt'                 : u'schlusspunkt',
        u'schlußpunkte'                : u'schlusspunkte',
        u'schlußpunkten'               : u'schlusspunkten',
        u'schlußpunktes'               : u'schlusspunktes',
        u'schlußpunkts'                : u'schlusspunkts',
        u'schlußredaktion'             : u'schlussredaktion',
        u'schlußrunde'                 : u'schlussrunde',
        u'schlußrunden'                : u'schlussrunden',
        u'schlußsatz'                  : u'schlusssatz',
        u'schlußstein'                 : u'schlussstein',
        u'schlußsteine'                : u'schlusssteine',
        u'schlußsteinen'               : u'schlusssteinen',
        u'schlußsteines'               : u'schlusssteines',
        u'schlußsteins'                : u'schlusssteins',
        u'schlußstrich'                : u'schlussstrich',
        u'schlußstriche'               : u'schlussstriche',
        u'schlußstrichen'              : u'schlussstrichen',
        u'schlußstriches'              : u'schlussstriches',
        u'schlußstrichs'               : u'schlussstrichs',
        u'schlußtableau'               : u'schlusstableau',
        u'schlußveranstaltung'         : u'schlussveranstaltung',
        u'schlußveranstaltungen'       : u'schlussveranstaltungen',
        u'schlußverhandlung'           : u'schlussverhandlung',
        u'schlußverhandlungen'         : u'schlussverhandlungen',
        u'schlußverhör'                : u'schlussverhör',
        u'schlußverkauf'               : u'schlussverkauf',
        u'schlußverkaufe'              : u'schlussverkaufe',
        u'schlußverkaufes'             : u'schlussverkaufes',
        u'schlußverkaufs'              : u'schlussverkaufs',
        u'schlußverkäufe'              : u'schlussverkäufe',
        u'schlußverkäufen'             : u'schlussverkäufen',
        u'schlußvers'                  : u'schlussvers',
        u'schlußwort'                  : u'schlusswort',
        u'schlußworte'                 : u'schlussworte',
        u'schlußworten'                : u'schlussworten',
        u'schlußwortes'                : u'schlusswortes',
        u'schlußworts'                 : u'schlussworts',
        u'schlußziehung'               : u'schlussziehung',
        u'schlößchen'                  : u'schlösschen',
        u'schlößchens'                 : u'schlösschens',
        u'schmiß'                      : u'schmiss',
        u'schmißt'                     : u'schmisst',
        u'schmuckblattelegramm'        : u'schmuckblatttelegramm',
        u'schmuckblattelegramme'       : u'schmuckblatttelegramme',
        u'schmuckblattelegrammen'      : u'schmuckblatttelegrammen',
        u'schmuckblattelegramms'       : u'schmuckblatttelegramms',
        u'schnappschuß'                : u'schnappschuss',
        u'schnellebig'                 : u'schnelllebig',
        u'schnellebige'                : u'schnelllebige',
        u'schnellebigem'               : u'schnelllebigem',
        u'schnellebigen'               : u'schnelllebigen',
        u'schnellebiger'               : u'schnelllebiger',
        u'schnellebiges'               : u'schnelllebiges',
        u'schnellimbiß'                : u'schnellimbiss',
        u'schnellphotographen'         : u'schnellfotografen',
        u'schnelläufer'                : u'schnellläufer',
        u'schnellöschantrag'           : u'schnelllöschantrag',
        u'schnepper'                   : u'schnäpper',
        u'schneppers'                  : u'schnäppers',
        u'schneuz'                     : u'schnäuz',
        u'schneuze'                    : u'schnäuze',
        u'schneuzen'                   : u'schnäuzen',
        u'schneuzend'                  : u'schnäuzend',
        u'schneuzest'                  : u'schnäuzest',
        u'schneuzet'                   : u'schnäuzet',
        u'schneuzt'                    : u'schnäuzt',
        u'schneuzte'                   : u'schnäuzte',
        u'schneuzten'                  : u'schnäuzten',
        u'schneuztest'                 : u'schnäuztest',
        u'schneuztet'                  : u'schnäuztet',
        u'schneuztuch'                 : u'schnäuztuch',
        u'schneuztuche'                : u'schnäuztuche',
        u'schneuztuches'               : u'schnäuztuches',
        u'schneuztuchs'                : u'schnäuztuchs',
        u'schneuztücher'               : u'schnäuztücher',
        u'schneuztüchern'              : u'schnäuztüchern',
        u'schnurlostelephon'           : u'schnurlostelefon',
        u'schnurlostelephone'          : u'schnurlostelefone',
        u'schnurlostelephonen'         : u'schnurlostelefonen',
        u'schnurlostelephons'          : u'schnurlostelefons',
        u'schockphoto'                 : u'schockfoto',
        u'schockphotos'                : u'schockfotos',
        u'schokokuß'                   : u'schokokuss',
        u'schokoladengenuß'            : u'schokoladengenuss',
        u'schooß'                      : u'schoß',
        u'schose'                      : u'chose',
        u'schosen'                     : u'chosen',
        u'schoß'                       : u'schoss',
        u'schoßfest'                   : u'schossfest',
        u'schoßfeste'                  : u'schossfeste',
        u'schoßfestem'                 : u'schossfestem',
        u'schoßfesten'                 : u'schossfesten',
        u'schoßfester'                 : u'schossfester',
        u'schoßfestere'                : u'schossfestere',
        u'schoßfesterem'               : u'schossfesterem',
        u'schoßfesteren'               : u'schossfesteren',
        u'schoßfesterer'               : u'schossfesterer',
        u'schoßfesteres'               : u'schossfesteres',
        u'schoßfestes'                 : u'schossfestes',
        u'schoßfesteste'               : u'schossfesteste',
        u'schoßfestestem'              : u'schossfestestem',
        u'schoßfestesten'              : u'schossfestesten',
        u'schoßfestester'              : u'schossfestester',
        u'schoßfestestes'              : u'schossfestestes',
        u'schoßt'                      : u'schosst',
        u'schrankschloß'               : u'schrankschloss',
        u'schraubverschluß'            : u'schraubverschluss',
        u'schreckensblaß'              : u'schreckensblass',
        u'schriee'                     : u'schrie',
        u'schrittempo'                 : u'schritttempo',
        u'schrittempos'                : u'schritttempos',
        u'schrotschuß'                 : u'schrotschuss',
        u'schrubbesen'                 : u'schrubbbesen',
        u'schrubbesens'                : u'schrubbbesens',
        u'schrumpfungsprozeß'          : u'schrumpfungsprozess',
        u'schuhbendel'                 : u'schuhbändel',
        u'schuhbendeln'                : u'schuhbändeln',
        u'schuhbendels'                : u'schuhbändels',
        u'schulabschluß'               : u'schulabschluss',
        u'schulabschlußfeier'          : u'schulabschlussfeier',
        u'schulabschlußfeiern'         : u'schulabschlussfeiern',
        u'schuldbewußt'                : u'schuldbewusst',
        u'schuldbewußte'               : u'schuldbewusste',
        u'schuldbewußtem'              : u'schuldbewusstem',
        u'schuldbewußten'              : u'schuldbewussten',
        u'schuldbewußter'              : u'schuldbewusster',
        u'schuldbewußtere'             : u'schuldbewusstere',
        u'schuldbewußterem'            : u'schuldbewussterem',
        u'schuldbewußteren'            : u'schuldbewussteren',
        u'schuldbewußterer'            : u'schuldbewussterer',
        u'schuldbewußteres'            : u'schuldbewussteres',
        u'schuldbewußtes'              : u'schuldbewusstes',
        u'schuldbewußteste'            : u'schuldbewussteste',
        u'schuldbewußtestem'           : u'schuldbewusstestem',
        u'schuldbewußtesten'           : u'schuldbewusstesten',
        u'schuldbewußtester'           : u'schuldbewusstester',
        u'schuldbewußtestes'           : u'schuldbewusstestes',
        u'schuldenerlaß'               : u'schuldenerlass',
        u'schulschluß'                 : u'schulschluss',
        u'schuß'                       : u'schuss',
        u'schußbereit'                 : u'schussbereit',
        u'schußbereite'                : u'schussbereite',
        u'schußbereitem'               : u'schussbereitem',
        u'schußbereiten'               : u'schussbereiten',
        u'schußbereiter'               : u'schussbereiter',
        u'schußbereites'               : u'schussbereites',
        u'schußfahrt'                  : u'schussfahrt',
        u'schußfahrten'                : u'schussfahrten',
        u'schußfeld'                   : u'schussfeld',
        u'schußfertig'                 : u'schussfertig',
        u'schußfertige'                : u'schussfertige',
        u'schußfertigem'               : u'schussfertigem',
        u'schußfertigen'               : u'schussfertigen',
        u'schußfertiger'               : u'schussfertiger',
        u'schußfertiges'               : u'schussfertiges',
        u'schußfest'                   : u'schussfest',
        u'schußfeste'                  : u'schussfeste',
        u'schußfestem'                 : u'schussfestem',
        u'schußfesten'                 : u'schussfesten',
        u'schußfester'                 : u'schussfester',
        u'schußfestes'                 : u'schussfestes',
        u'schußkreis'                  : u'schusskreis',
        u'schußkreise'                 : u'schusskreise',
        u'schußkreisen'                : u'schusskreisen',
        u'schußkreises'                : u'schusskreises',
        u'schußlig'                    : u'schusslig',
        u'schußlige'                   : u'schusslige',
        u'schußligem'                  : u'schussligem',
        u'schußligen'                  : u'schussligen',
        u'schußliger'                  : u'schussliger',
        u'schußligere'                 : u'schussligere',
        u'schußligerem'                : u'schussligerem',
        u'schußligeren'                : u'schussligeren',
        u'schußligerer'                : u'schussligerer',
        u'schußligeres'                : u'schussligeres',
        u'schußliges'                  : u'schussliges',
        u'schußligste'                 : u'schussligste',
        u'schußligstem'                : u'schussligstem',
        u'schußligsten'                : u'schussligsten',
        u'schußligster'                : u'schussligster',
        u'schußligstes'                : u'schussligstes',
        u'schußlinie'                  : u'schusslinie',
        u'schußnähe'                   : u'schussnähe',
        u'schußrichtung'               : u'schussrichtung',
        u'schußsicher'                 : u'schusssicher',
        u'schußsichere'                : u'schusssichere',
        u'schußsicherem'               : u'schusssicherem',
        u'schußsicheren'               : u'schusssicheren',
        u'schußsicherer'               : u'schusssicherer',
        u'schußsicheres'               : u'schusssicheres',
        u'schußverletzung'             : u'schussverletzung',
        u'schußverletzungen'           : u'schussverletzungen',
        u'schußwaffe'                  : u'schusswaffe',
        u'schußwaffen'                 : u'schusswaffen',
        u'schußwechsel'                : u'schusswechsel',
        u'schußwechseln'               : u'schusswechseln',
        u'schußwechsels'               : u'schusswechsels',
        u'schußweite'                  : u'schussweite',
        u'schußwunde'                  : u'schusswunde',
        u'schußwunden'                 : u'schusswunden',
        u'schwimmeister'               : u'schwimmmeister',
        u'schwimmeisterin'             : u'schwimmmeisterin',
        u'schwimmeisterinnen'          : u'schwimmmeisterinnen',
        u'schwurgerichtsprozeß'        : u'schwurgerichtsprozess',
        u'schößchen'                   : u'schösschen',
        u'schößling'                   : u'schössling',
        u'schößlinge'                  : u'schösslinge',
        u'schößlingen'                 : u'schösslingen',
        u'schößlings'                  : u'schösslings',
        u'schüßchen'                   : u'schüsschen',
        u'schüßchens'                  : u'schüsschens',
        u'science-fiction'             : u'science-fiction',
        u'seeschiffahrtsstrasse'       : u'seeschifffahrtsstraße',
        u'seeschiffahrtsstrassen'      : u'seeschifffahrtsstraßen',
        u'seeschiffahrtstrasse'        : u'seeschifffahrtstraße',
        u'seeschiffahrtstrassen'       : u'seeschifffahrtstraßen',
        u'sehnenriß'                   : u'sehnenriss',
        u'seigerriß'                   : u'seigerriss',
        u'seitenriß'                   : u'seitenriss',
        u'selbstbewußt'                : u'selbstbewusst',
        u'selbstbewußte'               : u'selbstbewusste',
        u'selbstbewußtem'              : u'selbstbewusstem',
        u'selbstbewußten'              : u'selbstbewussten',
        u'selbstbewußter'              : u'selbstbewusster',
        u'selbstbewußtere'             : u'selbstbewusstere',
        u'selbstbewußterem'            : u'selbstbewussterem',
        u'selbstbewußteren'            : u'selbstbewussteren',
        u'selbstbewußterer'            : u'selbstbewussterer',
        u'selbstbewußteres'            : u'selbstbewussteres',
        u'selbstbewußtes'              : u'selbstbewusstes',
        u'selbstbewußteste'            : u'selbstbewussteste',
        u'selbstbewußtestem'           : u'selbstbewusstestem',
        u'selbstbewußtesten'           : u'selbstbewusstesten',
        u'selbstbewußtester'           : u'selbstbewusstester',
        u'selbstbewußtestes'           : u'selbstbewusstestes',
        u'selbstbewußtsein'            : u'selbstbewusstsein',
        u'selbstportrait'              : u'selbstporträt',
        u'selbstportraits'             : u'selbstporträts',
        u'senatsbeschluß'              : u'senatsbeschluss',
        u'sendeschluß'                 : u'sendeschluss',
        u'sendungsbewußtsein'          : u'sendungsbewusstsein',
        u'sendungsbewußtseins'         : u'sendungsbewusstseins',
        u'sensationsprozeß'            : u'sensationsprozess',
        u'sexbeseßner'                 : u'sexbesessner',
        u'sexbeseßnere'                : u'sexbesessnere',
        u'sexbeseßnerem'               : u'sexbesessnerem',
        u'sexbeseßneren'               : u'sexbesessneren',
        u'sexbeseßnerer'               : u'sexbesessnerer',
        u'sexbeseßneres'               : u'sexbesessneres',
        u'sey'                         : u'sei',
        u'seyd'                        : u'seid',
        u'seyen'                       : u'seien',
        u'seyend'                      : u'seiend',
        u'seyest'                      : u'seiest',
        u'seyet'                       : u'seiet',
        u'seyn'                        : u'sein',
        u'seyst'                       : u'seist',
        u'seßhaft'                     : u'sesshaft',
        u'seßhafte'                    : u'sesshafte',
        u'seßhaftem'                   : u'sesshaftem',
        u'seßhaften'                   : u'sesshaften',
        u'seßhafter'                   : u'sesshafter',
        u'seßhaftere'                  : u'sesshaftere',
        u'seßhafterem'                 : u'sesshafterem',
        u'seßhafteren'                 : u'sesshafteren',
        u'seßhafterer'                 : u'sesshafterer',
        u'seßhafteres'                 : u'sesshafteres',
        u'seßhaftes'                   : u'sesshaftes',
        u'seßhafteste'                 : u'sesshafteste',
        u'seßhaftestem'                : u'sesshaftestem',
        u'seßhaftesten'                : u'sesshaftesten',
        u'seßhaftester'                : u'sesshaftester',
        u'seßhaftestes'                : u'sesshaftestes',
        u'sicilianisch'                : u'sizilianisch',
        u'sicilianische'               : u'sizilianische',
        u'sicilianischem'              : u'sizilianischem',
        u'sicilianischen'              : u'sizilianischen',
        u'sicilianischer'              : u'sizilianischer',
        u'sicilianisches'              : u'sizilianisches',
        u'sicilien'                    : u'sizilien',
        u'siciliens'                   : u'siziliens',
        u'siechthum'                   : u'siechtum',
        u'siechthums'                  : u'siechtums',
        u'siegesbewußt'                : u'siegesbewusst',
        u'siegesbewußte'               : u'siegesbewusste',
        u'siegesbewußtem'              : u'siegesbewusstem',
        u'siegesbewußten'              : u'siegesbewussten',
        u'siegesbewußter'              : u'siegesbewusster',
        u'siegesbewußtere'             : u'siegesbewusstere',
        u'siegesbewußterem'            : u'siegesbewussterem',
        u'siegesbewußteren'            : u'siegesbewussteren',
        u'siegesbewußterer'            : u'siegesbewussterer',
        u'siegesbewußteres'            : u'siegesbewussteres',
        u'siegesbewußtes'              : u'siegesbewusstes',
        u'siegesbewußteste'            : u'siegesbewussteste',
        u'siegesbewußtestem'           : u'siegesbewusstestem',
        u'siegesbewußtesten'           : u'siegesbewusstesten',
        u'siegesbewußtester'           : u'siegesbewusstester',
        u'siegesbewußtestes'           : u'siegesbewusstestes',
        u'siegesgewiß'                 : u'siegesgewiss',
        u'siegesgewißheit'             : u'siegesgewissheit',
        u'siemens-martin-prozeß'       : u'siemens-martin-prozess',
        u'sierraleonisch'              : u'sierra-leonisch',
        u'sierraleonische'             : u'sierra-leonische',
        u'sierraleonischem'            : u'sierra-leonischem',
        u'sierraleonischen'            : u'sierra-leonischen',
        u'sierraleonischer'            : u'sierra-leonischer',
        u'sierraleonisches'            : u'sierra-leonisches',
        u'silvesterkuß'                : u'silvesterkuss',
        u'sketsch'                     : u'sketch',
        u'sketsche'                    : u'sketche',
        u'sketschen'                   : u'sketchen',
        u'sketsches'                   : u'sketches',
        u'sketschs'                    : u'sketchs',
        u'social'                      : u'sozial',
        u'sociale'                     : u'soziale',
        u'socialem'                    : u'sozialem',
        u'socialen'                    : u'sozialen',
        u'socialer'                    : u'sozialer',
        u'socialere'                   : u'sozialere',
        u'socialerem'                  : u'sozialerem',
        u'socialeren'                  : u'sozialeren',
        u'socialerer'                  : u'sozialerer',
        u'socialeres'                  : u'sozialeres',
        u'sociales'                    : u'soziales',
        u'socialismen'                 : u'sozialismen',
        u'socialismus'                 : u'sozialismus',
        u'socialste'                   : u'sozialste',
        u'socialstem'                  : u'sozialstem',
        u'socialsten'                  : u'sozialsten',
        u'socialster'                  : u'sozialster',
        u'socialstes'                  : u'sozialstes',
        u'sockelgeschoß'               : u'sockelgeschoss',
        u'sodass'                      : u'so dass',
        u'sodaß'                       : u'sodass',
        u'solenitäten'                 : u'solennitäten',
        u'sommerschlußverkauf'         : u'sommerschlussverkauf',
        u'sommerschlußverkaufe'        : u'sommerschlussverkaufe',
        u'sommerschlußverkaufes'       : u'sommerschlussverkaufes',
        u'sommerschlußverkaufs'        : u'sommerschlussverkaufs',
        u'sommerschlußverkäufe'        : u'sommerschlussverkäufe',
        u'sommerschlußverkäufen'       : u'sommerschlussverkäufen',
        u'sondirung'                   : u'sondierung',
        u'sondirungen'                 : u'sondierungen',
        u'sozialisationsprozeß'        : u'sozialisationsprozess',
        u'sozialmißbrauch'             : u'sozialmissbrauch',
        u'sozialmißbrauches'           : u'sozialmissbrauches',
        u'sozialmißbrauchs'            : u'sozialmissbrauchs',
        u'spannbettuch'                : u'spannbetttuch',
        u'spannbettuche'               : u'spannbetttuche',
        u'spannbettuches'              : u'spannbetttuches',
        u'spannbettuchs'               : u'spannbetttuchs',
        u'spannbettücher'              : u'spannbetttücher',
        u'spannbettüchern'             : u'spannbetttüchern',
        u'spass'                       : u'spaß',
        u'specifische'                 : u'spezifische',
        u'sperrad'                     : u'sperrrad',
        u'sperrade'                    : u'sperrrade',
        u'sperrades'                   : u'sperrrades',
        u'sperrads'                    : u'sperrrads',
        u'sperriegel'                  : u'sperrriegel',
        u'sperriegeln'                 : u'sperrriegeln',
        u'sperriegels'                 : u'sperrriegels',
        u'sperräder'                   : u'sperrräder',
        u'sperrädern'                  : u'sperrrädern',
        u'spieen'                      : u'spien',
        u'spionageprozeß'              : u'spionageprozess',
        u'spliß'                       : u'spliss',
        u'splißt'                      : u'splisst',
        u'sportdreß'                   : u'sportdress',
        u'sprachbewußtsein'            : u'sprachbewusstsein',
        u'sprachbewußtseins'           : u'sprachbewusstseins',
        u'sprenggeschoß'               : u'sprenggeschoss',
        u'spritzgußteile'              : u'spritzgussteile',
        u'spritzgußteilen'             : u'spritzgussteilen',
        u'spritzgußteiles'             : u'spritzgussteiles',
        u'spritzgußteils'              : u'spritzgussteils',
        u'sproß'                       : u'spross',
        u'sproßt'                      : u'sprosst',
        u'sproßvokal'                  : u'sprossvokal',
        u'sproßvokale'                 : u'sprossvokale',
        u'sproßvokalen'                : u'sprossvokalen',
        u'sproßvokals'                 : u'sprossvokals',
        u'sprößling'                   : u'sprössling',
        u'sprößlinge'                  : u'sprösslinge',
        u'sprößlingen'                 : u'sprösslingen',
        u'sprößlings'                  : u'sprösslings',
        u'spätgothik'                  : u'spätgotik',
        u'srilankisch'                 : u'sri-lankisch',
        u'srilankische'                : u'sri-lankische',
        u'srilankischem'               : u'sri-lankischem',
        u'srilankischen'               : u'sri-lankischen',
        u'srilankischer'               : u'sri-lankischer',
        u'srilankisches'               : u'sri-lankisches',
        u'stadtrath'                   : u'stadtrat',
        u'stadtrathe'                  : u'stadtrate',
        u'stadtrathes'                 : u'stadtrates',
        u'stadtraths'                  : u'stadtrats',
        u'stadträthe'                  : u'stadträte',
        u'stadträthen'                 : u'stadträten',
        u'stadtschloß'                 : u'stadtschloss',
        u'stadtthor'                   : u'stadttor',
        u'stadtthore'                  : u'stadttore',
        u'stadtthoren'                 : u'stadttoren',
        u'stadtthores'                 : u'stadttores',
        u'stadtthors'                  : u'stadttors',
        u'stahlroß'                    : u'stahlross',
        u'stallaterne'                 : u'stalllaterne',
        u'stammorphem'                 : u'stammmorphem',
        u'stammutter'                  : u'stammmutter',
        u'standesbewußt'               : u'standesbewusst',
        u'standesbewußte'              : u'standesbewusste',
        u'standesbewußtem'             : u'standesbewusstem',
        u'standesbewußten'             : u'standesbewussten',
        u'standesbewußter'             : u'standesbewusster',
        u'standesbewußtes'             : u'standesbewusstes',
        u'stanglpaß'                   : u'stanglpass',
        u'starkregenereigniß'          : u'starkregenereignis',
        u'startschuß'                  : u'startschuss',
        u'steckschuß'                  : u'steckschuss',
        u'steilpaß'                    : u'steilpass',
        u'steinschloß'                 : u'steinschloss',
        u'stellfaß'                    : u'stellfass',
        u'stemmeissel'                 : u'stemmmeißel',
        u'stemmeisseln'                : u'stemmmeißeln',
        u'stemmeissels'                : u'stemmmeißels',
        u'stemmeißel'                  : u'stemmmeißel',
        u'stengel'                     : u'stängel',
        u'stengeln'                    : u'stängeln',
        u'stengels'                    : u'stängels',
        u'stenographen'                : u'stenografen',
        u'stenographisch'              : u'stenografisch',
        u'stenographische'             : u'stenografische',
        u'stenographischen'            : u'stenografischen',
        u'steptanz'                    : u'stepptanz',
        u'steuermeßbetrag'             : u'steuermessbetrag',
        u'steuermeßbetrage'            : u'steuermessbetrage',
        u'steuermeßbetrages'           : u'steuermessbetrages',
        u'steuermeßbetrags'            : u'steuermessbetrags',
        u'steuermeßbeträge'            : u'steuermessbeträge',
        u'steuermeßbeträgen'           : u'steuermessbeträgen',
        u'stewardeß'                   : u'stewardess',
        u'stilleben'                   : u'stillleben',
        u'stillebens'                  : u'stilllebens',
        u'stillege'                    : u'stilllege',
        u'stillegen'                   : u'stilllegen',
        u'stillegend'                  : u'stilllegend',
        u'stillegest'                  : u'stilllegest',
        u'stilleget'                   : u'stillleget',
        u'stillegst'                   : u'stilllegst',
        u'stillegt'                    : u'stilllegt',
        u'stillegte'                   : u'stilllegte',
        u'stillegten'                  : u'stilllegten',
        u'stillegtest'                 : u'stilllegtest',
        u'stillegtet'                  : u'stilllegtet',
        u'stillegung'                  : u'stilllegung',
        u'stillegungen'                : u'stilllegungen',
        u'stoffaden'                   : u'stofffaden',
        u'stoffarbe'                   : u'stofffarbe',
        u'stoffaser'                   : u'stofffaser',
        u'stoffetzen'                  : u'stofffetzen',
        u'stoffetzens'                 : u'stofffetzens',
        u'stok'                        : u'stokes',
        u'stop'                        : u'stopp',
        u'stops'                       : u'stopps',
        u'storchenbiß'                 : u'storchenbiss',
        u'stories'                     : u'storys',
        u'straferlaß'                  : u'straferlass',
        u'strafprozeß'                 : u'strafprozess',
        u'strafprozeßordnung'          : u'strafprozessordnung',
        u'strafprozeßrecht'            : u'strafprozessrecht',
        u'straß'                       : u'strass',
        u'streifschuß'                 : u'streifschuss',
        u'streß'                       : u'stress',
        u'streßfrei'                   : u'stressfrei',
        u'streßfreie'                  : u'stressfreie',
        u'streßfreiem'                 : u'stressfreiem',
        u'streßfreien'                 : u'stressfreien',
        u'streßfreier'                 : u'stressfreier',
        u'streßfreies'                 : u'stressfreies',
        u'streßhormon'                 : u'stresshormon',
        u'streßhormone'                : u'stresshormone',
        u'streßhormonen'               : u'stresshormonen',
        u'streßhormons'                : u'stresshormons',
        u'streßresistent'              : u'stressresistent',
        u'streßresistente'             : u'stressresistente',
        u'streßresistentem'            : u'stressresistentem',
        u'streßresistenten'            : u'stressresistenten',
        u'streßresistenter'            : u'stressresistenter',
        u'streßresistentere'           : u'stressresistentere',
        u'streßresistenterem'          : u'stressresistenterem',
        u'streßresistenteren'          : u'stressresistenteren',
        u'streßresistenterer'          : u'stressresistenterer',
        u'streßresistenteres'          : u'stressresistenteres',
        u'streßresistentes'            : u'stressresistentes',
        u'streßresistenteste'          : u'stressresistenteste',
        u'streßresistentestem'         : u'stressresistentestem',
        u'streßresistentesten'         : u'stressresistentesten',
        u'streßresistentester'         : u'stressresistentester',
        u'streßresistentestes'         : u'stressresistentestes',
        u'streßsituation'              : u'stresssituation',
        u'streßt'                      : u'stresst',
        u'streßte'                     : u'stresste',
        u'streßten'                    : u'stressten',
        u'streßtest'                   : u'stresstest',
        u'streßtests'                  : u'stresstests',
        u'streßtet'                    : u'stresstet',
        u'stromanschluß'               : u'stromanschluss',
        u'studir'                      : u'studier',
        u'studire'                     : u'studiere',
        u'studiren'                    : u'studieren',
        u'studirend'                   : u'studierend',
        u'studirest'                   : u'studierest',
        u'studiret'                    : u'studieret',
        u'studirst'                    : u'studierst',
        u'studirt'                     : u'studiert',
        u'studirte'                    : u'studierte',
        u'studirten'                   : u'studierten',
        u'studirtest'                  : u'studiertest',
        u'studirtet'                   : u'studiertet',
        u'stukkateur'                  : u'stuckateur',
        u'stukkateure'                 : u'stuckateure',
        u'stukkateuren'                : u'stuckateuren',
        u'stukkateurs'                 : u'stuckateurs',
        u'stusz'                       : u'stuss',
        u'stuß'                        : u'stuss',
        u'subcutan'                    : u'subkutan',
        u'subcutane'                   : u'subkutane',
        u'subcutanem'                  : u'subkutanem',
        u'subcutanen'                  : u'subkutanen',
        u'subcutaner'                  : u'subkutaner',
        u'subcutanes'                  : u'subkutanes',
        u'submiß'                      : u'submiss',
        u'substantielle'               : u'substanzielle',
        u'sukzeß'                      : u'sukzess',
        u'sutane'                      : u'soutane',
        u'sutanen'                     : u'soutanen',
        u'swimming-pool'               : u'swimmingpool',
        u'swimming-pools'              : u'swimmingpools',
        u'sympathieen'                 : u'sympathien',
        u'säbelraßler'                 : u'säbelrassler',
        u'sängerbaß'                   : u'sängerbass',
        u'säuberungsprozeß'            : u'säuberungsprozess',
        u'säugethier'                  : u'säugetier',
        u'säugethiere'                 : u'säugetiere',
        u'säugethieren'                : u'säugetieren',
        u'säugethieres'                : u'säugetieres',
        u'säugethiers'                 : u'säugetiers',
        u'säulenabschluß'              : u'säulenabschluss',
        u'säureüberschuß'              : u'säureüberschuss',
        u'südrußland'                  : u'südrussland',
        u'südrußlands'                 : u'südrusslands',
        u'tablettenmißbrauch'          : u'tablettenmissbrauch',
        u'tablettenmißbrauche'         : u'tablettenmissbrauche',
        u'tablettenmißbrauches'        : u'tablettenmissbrauches',
        u'tablettenmißbrauchs'         : u'tablettenmissbrauchs',
        u'taferlklaßler'               : u'taferlklassler',
        u'taferlklaßlern'              : u'taferlklasslern',
        u'taferlklaßlers'              : u'taferlklasslers',
        u'talke'                       : u'talk',
        u'talpaß'                      : u'talpass',
        u'tankschloß'                  : u'tankschloss',
        u'tankverschluß'               : u'tankverschluss',
        u'tarifabschluß'               : u'tarifabschluss',
        u'tatortphoto'                 : u'tatortfoto',
        u'tatortphotos'                : u'tatortfotos',
        u'tauernexpreß'                : u'tauernexpress',
        u'taunaß'                      : u'taunass',
        u'teerfaß'                     : u'teerfass',
        u'telefonanschluß'             : u'telefonanschluss',
        u'telegraphieren'              : u'telegrafieren',
        u'telegraphiert'               : u'telegrafiert',
        u'telegraphierte'              : u'telegrafierte',
        u'telegraphierten'             : u'telegrafierten',
        u'telephon'                    : u'telefon',
        u'telephonanruf'               : u'telefonanruf',
        u'telephonanrufe'              : u'telefonanrufe',
        u'telephonanrufen'             : u'telefonanrufen',
        u'telephonanrufes'             : u'telefonanrufes',
        u'telephonanrufs'              : u'telefonanrufs',
        u'telephonapparat'             : u'telefonapparat',
        u'telephonat'                  : u'telefonat',
        u'telephonate'                 : u'telefonate',
        u'telephonaten'                : u'telefonaten',
        u'telephonates'                : u'telefonates',
        u'telephonats'                 : u'telefonats',
        u'telephonauskunft'            : u'telefonauskunft',
        u'telephonbuch'                : u'telefonbuch',
        u'telephonbuche'               : u'telefonbuche',
        u'telephonbuches'              : u'telefonbuches',
        u'telephonbuchs'               : u'telefonbuchs',
        u'telephonbücher'              : u'telefonbücher',
        u'telephonbüchern'             : u'telefonbüchern',
        u'telephone'                   : u'telefone',
        u'telephonen'                  : u'telefonen',
        u'telephongespräch'            : u'telefongespräch',
        u'telephongespräche'           : u'telefongespräche',
        u'telephongesprächen'          : u'telefongesprächen',
        u'telephongespräches'          : u'telefongespräches',
        u'telephongesprächs'           : u'telefongesprächs',
        u'telephonhäuschen'            : u'telefonhäuschen',
        u'telephonhäuschens'           : u'telefonhäuschens',
        u'telephonhörer'               : u'telefonhörer',
        u'telephonhörern'              : u'telefonhörern',
        u'telephonhörers'              : u'telefonhörers',
        u'telephonie'                  : u'telefonie',
        u'telephonier'                 : u'telefonier',
        u'telephoniere'                : u'telefoniere',
        u'telephonieren'               : u'telefonieren',
        u'telephonierend'              : u'telefonierend',
        u'telephonierende'             : u'telefonierende',
        u'telephonierendem'            : u'telefonierendem',
        u'telephonierenden'            : u'telefonierenden',
        u'telephonierender'            : u'telefonierender',
        u'telephonierendes'            : u'telefonierendes',
        u'telephonierest'              : u'telefonierest',
        u'telephonieret'               : u'telefonieret',
        u'telephonierst'               : u'telefonierst',
        u'telephoniert'                : u'telefoniert',
        u'telephonierte'               : u'telefonierte',
        u'telephonierten'              : u'telefonierten',
        u'telephoniertest'             : u'telefoniertest',
        u'telephoniertet'              : u'telefoniertet',
        u'telephonisch'                : u'telefonisch',
        u'telephonische'               : u'telefonische',
        u'telephonischem'              : u'telefonischem',
        u'telephonischen'              : u'telefonischen',
        u'telephonischer'              : u'telefonischer',
        u'telephonisches'              : u'telefonisches',
        u'telephonist'                 : u'telefonist',
        u'telephonisten'               : u'telefonisten',
        u'telephonistin'               : u'telefonistin',
        u'telephonistinnen'            : u'telefonistinnen',
        u'telephonjoker'               : u'telefonjoker',
        u'telephonjokern'              : u'telefonjokern',
        u'telephonjokers'              : u'telefonjokers',
        u'telephonkonferenz'           : u'telefonkonferenz',
        u'telephonkonferenzen'         : u'telefonkonferenzen',
        u'telephonkontakt'             : u'telefonkontakt',
        u'telephonkontakte'            : u'telefonkontakte',
        u'telephonkontakten'           : u'telefonkontakten',
        u'telephonkontaktes'           : u'telefonkontaktes',
        u'telephonkontakts'            : u'telefonkontakts',
        u'telephonleitung'             : u'telefonleitung',
        u'telephonleitungen'           : u'telefonleitungen',
        u'telephonmast'                : u'telefonmast',
        u'telephonmaste'               : u'telefonmaste',
        u'telephonmasten'              : u'telefonmasten',
        u'telephonmastes'              : u'telefonmastes',
        u'telephonmasts'               : u'telefonmasts',
        u'telephonnetz'                : u'telefonnetz',
        u'telephonnetze'               : u'telefonnetze',
        u'telephonnetzen'              : u'telefonnetzen',
        u'telephonnetzes'              : u'telefonnetzes',
        u'telephonnummer'              : u'telefonnummer',
        u'telephonnummern'             : u'telefonnummern',
        u'telephonrechnung'            : u'telefonrechnung',
        u'telephonrechnungen'          : u'telefonrechnungen',
        u'telephons'                   : u'telefons',
        u'telephonseelsorge'           : u'telefonseelsorge',
        u'telephonsex'                 : u'telefonsex',
        u'telephonsexes'               : u'telefonsexes',
        u'telephonterror'              : u'telefonterror',
        u'telephonterrors'             : u'telefonterrors',
        u'telephontisch'               : u'telefontisch',
        u'telephontischchen'           : u'telefontischchen',
        u'telephontischchens'          : u'telefontischchens',
        u'telephontische'              : u'telefontische',
        u'telephontischen'             : u'telefontischen',
        u'telephontisches'             : u'telefontisches',
        u'telephontischs'              : u'telefontischs',
        u'telephonvorwahl'             : u'telefonvorwahl',
        u'telephonvorwahlen'           : u'telefonvorwahlen',
        u'telephonzelle'               : u'telefonzelle',
        u'telephonzellen'              : u'telefonzellen',
        u'telephonzentrale'            : u'telefonzentrale',
        u'telephonzentralen'           : u'telefonzentralen',
        u'terminstreß'                 : u'terminstress',
        u'terrorprozeß'                : u'terrorprozess',
        u'teufelsabbiß'                : u'teufelsabbiss',
        u'thal'                        : u'tal',
        u'thale'                       : u'tale',
        u'thales'                      : u'tales',
        u'thals'                       : u'tals',
        u'that'                        : u'tat',
        u'thaten'                      : u'taten',
        u'thatest'                     : u'tatest',
        u'thatet'                      : u'tatet',
        u'thatst'                      : u'tatst',
        u'thau'                        : u'tau',
        u'thaue'                       : u'taue',
        u'thauen'                      : u'tauen',
        u'thaues'                      : u'taues',
        u'thaus'                       : u'taus',
        u'thee'                        : u'tee',
        u'thees'                       : u'tees',
        u'theil'                       : u'teil',
        u'theile'                      : u'teile',
        u'theilen'                     : u'teilen',
        u'theiles'                     : u'teiles',
        u'theilnehmer'                 : u'teilnehmer',
        u'theilnehmerin'               : u'teilnehmerin',
        u'theilnehmerinnen'            : u'teilnehmerinnen',
        u'theilnehmern'                : u'teilnehmern',
        u'theilnehmers'                : u'teilnehmers',
        u'theils'                      : u'teils',
        u'theorieen'                   : u'theorien',
        u'thier'                       : u'tier',
        u'thiere'                      : u'tiere',
        u'thieren'                     : u'tieren',
        u'thieres'                     : u'tieres',
        u'thierisch'                   : u'tierisch',
        u'thierische'                  : u'tierische',
        u'thierischem'                 : u'tierischem',
        u'thierischen'                 : u'tierischen',
        u'thierischer'                 : u'tierischer',
        u'thierischere'                : u'tierischere',
        u'thierischerem'               : u'tierischerem',
        u'thierischeren'               : u'tierischeren',
        u'thierischerer'               : u'tierischerer',
        u'thierischeres'               : u'tierischeres',
        u'thierisches'                 : u'tierisches',
        u'thierischste'                : u'tierischste',
        u'thierischstem'               : u'tierischstem',
        u'thierischsten'               : u'tierischsten',
        u'thierischster'               : u'tierischster',
        u'thierischstes'               : u'tierischstes',
        u'thierkreislicht'             : u'tierkreislicht',
        u'thiers'                      : u'tiers',
        u'thor'                        : u'tor',
        u'thore'                       : u'tore',
        u'thoren'                      : u'toren',
        u'thores'                      : u'tores',
        u'thors'                       : u'tors',
        u'threu'                       : u'treu',
        u'threue'                      : u'treue',
        u'threuem'                     : u'treuem',
        u'threuen'                     : u'treuen',
        u'threuer'                     : u'treuer',
        u'threuere'                    : u'treuere',
        u'threuerem'                   : u'treuerem',
        u'threueren'                   : u'treueren',
        u'threuerer'                   : u'treuerer',
        u'threueres'                   : u'treueres',
        u'threues'                     : u'treues',
        u'threueste'                   : u'treueste',
        u'threuestem'                  : u'treuestem',
        u'threuesten'                  : u'treuesten',
        u'threuester'                  : u'treuester',
        u'threuestes'                  : u'treuestes',
        u'threuste'                    : u'treuste',
        u'threustem'                   : u'treustem',
        u'threusten'                   : u'treusten',
        u'threuster'                   : u'treuster',
        u'threustes'                   : u'treustes',
        u'thräne'                      : u'träne',
        u'thränen'                     : u'tränen',
        u'thu'                         : u'tu',
        u'thue'                        : u'tue',
        u'thuen'                       : u'tuen',
        u'thuend'                      : u'tuend',
        u'thuest'                      : u'tuest',
        u'thuet'                       : u'tuet',
        u'thun'                        : u'tun',
        u'thuns'                       : u'tuns',
        u'thurm'                       : u'turm',
        u'thurme'                      : u'turme',
        u'thurmes'                     : u'turmes',
        u'thurms'                      : u'turms',
        u'thust'                       : u'tust',
        u'thut'                        : u'tut',
        u'thäler'                      : u'täler',
        u'thälern'                     : u'tälern',
        u'thäte'                       : u'täte',
        u'thäten'                      : u'täten',
        u'thätest'                     : u'tätest',
        u'thätet'                      : u'tätet',
        u'thür'                        : u'tür',
        u'thüre'                       : u'türe',
        u'thüren'                      : u'türen',
        u'thürm'                       : u'türm',
        u'thürme'                      : u'türme',
        u'thürmen'                     : u'türmen',
        u'thürmend'                    : u'türmend',
        u'thürmest'                    : u'türmest',
        u'thürmet'                     : u'türmet',
        u'thürmst'                     : u'türmst',
        u'thürmt'                      : u'türmt',
        u'thürmte'                     : u'türmte',
        u'thürmten'                    : u'türmten',
        u'thürmtest'                   : u'türmtest',
        u'thürmtet'                    : u'türmtet',
        u'tiefgeschoß'                 : u'tiefgeschoss',
        u'tintenfaß'                   : u'tintenfass',
        u'tip'                         : u'tipp',
        u'tips'                        : u'tipps',
        u'todblaß'                     : u'todblass',
        u'todesschuß'                  : u'todesschuss',
        u'todte'                       : u'tote',
        u'todtengräber'                : u'totengräber',
        u'tolpatsch'                   : u'tollpatsch',
        u'tolpatsche'                  : u'tollpatsche',
        u'tolpatschen'                 : u'tollpatschen',
        u'tolpatsches'                 : u'tollpatsches',
        u'tolpatschig'                 : u'tollpatschig',
        u'tolpatschige'                : u'tollpatschige',
        u'tolpatschigem'               : u'tollpatschigem',
        u'tolpatschigen'               : u'tollpatschigen',
        u'tolpatschiger'               : u'tollpatschiger',
        u'tolpatschigere'              : u'tollpatschigere',
        u'tolpatschigerem'             : u'tollpatschigerem',
        u'tolpatschigeren'             : u'tollpatschigeren',
        u'tolpatschigerer'             : u'tollpatschigerer',
        u'tolpatschigeres'             : u'tollpatschigeres',
        u'tolpatschiges'               : u'tollpatschiges',
        u'tolpatschigste'              : u'tollpatschigste',
        u'tolpatschigstem'             : u'tollpatschigstem',
        u'tolpatschigsten'             : u'tollpatschigsten',
        u'tolpatschigster'             : u'tollpatschigster',
        u'tolpatschigstes'             : u'tollpatschigstes',
        u'tolpatschs'                  : u'tollpatschs',
        u'toresschluß'                 : u'toresschluss',
        u'torschluß'                   : u'torschluss',
        u'torschlußpanik'              : u'torschlusspanik',
        u'torschlußpaniken'            : u'torschlusspaniken',
        u'torschuß'                    : u'torschuss',
        u'tortenguß'                   : u'tortenguss',
        u'totbiß'                      : u'totbiss',
        u'totbißt'                     : u'totbisst',
        u'totenblaß'                   : u'totenblass',
        u'traditionsbewußt'            : u'traditionsbewusst',
        u'traditionsbewußte'           : u'traditionsbewusste',
        u'traditionsbewußtem'          : u'traditionsbewusstem',
        u'traditionsbewußten'          : u'traditionsbewussten',
        u'traditionsbewußter'          : u'traditionsbewusster',
        u'traditionsbewußtere'         : u'traditionsbewusstere',
        u'traditionsbewußterem'        : u'traditionsbewussterem',
        u'traditionsbewußteren'        : u'traditionsbewussteren',
        u'traditionsbewußterer'        : u'traditionsbewussterer',
        u'traditionsbewußteres'        : u'traditionsbewussteres',
        u'traditionsbewußtes'          : u'traditionsbewusstes',
        u'traditionsbewußteste'        : u'traditionsbewussteste',
        u'traditionsbewußtestem'       : u'traditionsbewusstestem',
        u'traditionsbewußtesten'       : u'traditionsbewusstesten',
        u'traditionsbewußtester'       : u'traditionsbewusstester',
        u'traditionsbewußtestes'       : u'traditionsbewusstestes',
        u'traditionsbewußtsein'        : u'traditionsbewusstsein',
        u'trampschiffahrt'             : u'trampschifffahrt',
        u'traumschloß'                 : u'traumschloss',
        u'traß'                        : u'trass',
        u'treibprozeß'                 : u'treibprozess',
        u'trendbewußt'                 : u'trendbewusst',
        u'trennetz'                    : u'trennnetz',
        u'triefnaß'                    : u'triefnass',
        u'tropfnaß'                    : u'tropfnass',
        u'troß'                        : u'tross',
        u'troßkarre'                   : u'trosskarre',
        u'troßschiff'                  : u'trossschiff',
        u'troßschiffe'                 : u'trossschiffe',
        u'troßschiffen'                : u'trossschiffen',
        u'troßschiffes'                : u'trossschiffes',
        u'troßschiffs'                 : u'trossschiffs',
        u'troßweib'                    : u'trossweib',
        u'troßweiber'                  : u'trossweiber',
        u'troßweibern'                 : u'trossweibern',
        u'troßweibes'                  : u'trossweibes',
        u'troßweibs'                   : u'trossweibs',
        u'truchseß'                    : u'truchsess',
        u'trugschluß'                  : u'trugschluss',
        u'tränenfluß'                  : u'tränenfluss',
        u'tränennaß'                   : u'tränennass',
        u'tschüß'                      : u'tschüss',
        u'tuffels'                     : u'tufffels',
        u'tuffelsen'                   : u'tufffelsen',
        u'täßchen'                     : u'tässchen',
        u'täßchens'                    : u'tässchens',
        u'türschloß'                   : u'türschloss',
        u'u-ausschuß'                  : u'u-ausschuss',
        u'ueberbleibsel'               : u'überbleibsel',
        u'umdenkprozeß'                : u'umdenkprozess',
        u'umdenkungsprozeß'            : u'umdenkungsprozess',
        u'umfaß'                       : u'umfass',
        u'umfaßt'                      : u'umfasst',
        u'umfaßte'                     : u'umfasste',
        u'umfaßten'                    : u'umfassten',
        u'umfaßtest'                   : u'umfasstest',
        u'umfaßtet'                    : u'umfasstet',
        u'umgefaßt'                    : u'umgefasst',
        u'umkehrschluß'                : u'umkehrschluss',
        u'umriß'                       : u'umriss',
        u'umrißlosen'                  : u'umrisslosen',
        u'umrißt'                      : u'umrisst',
        u'umrißzeichnung'              : u'umrisszeichnung',
        u'umschichtungsprozeß'         : u'umschichtungsprozess',
        u'umschloß'                    : u'umschloss',
        u'umschloßt'                   : u'umschlosst',
        u'umschluß'                    : u'umschluss',
        u'umstellungsprozeß'           : u'umstellungsprozess',
        u'umwandlungsprozeß'           : u'umwandlungsprozess',
        u'umweltbewußt'                : u'umweltbewusst',
        u'umweltbewußte'               : u'umweltbewusste',
        u'umweltbewußtem'              : u'umweltbewusstem',
        u'umweltbewußten'              : u'umweltbewussten',
        u'umweltbewußter'              : u'umweltbewusster',
        u'umweltbewußtere'             : u'umweltbewusstere',
        u'umweltbewußterem'            : u'umweltbewussterem',
        u'umweltbewußteren'            : u'umweltbewussteren',
        u'umweltbewußterer'            : u'umweltbewussterer',
        u'umweltbewußteres'            : u'umweltbewussteres',
        u'umweltbewußtes'              : u'umweltbewusstes',
        u'umweltbewußteste'            : u'umweltbewussteste',
        u'umweltbewußtestem'           : u'umweltbewusstestem',
        u'umweltbewußtesten'           : u'umweltbewusstesten',
        u'umweltbewußtester'           : u'umweltbewusstester',
        u'umweltbewußtestes'           : u'umweltbewusstestes',
        u'umweltbewußtsein'            : u'umweltbewusstsein',
        u'umwelteinfluß'               : u'umwelteinfluss',
        u'unangepaßt'                  : u'unangepasst',
        u'unangepaßte'                 : u'unangepasste',
        u'unangepaßtem'                : u'unangepasstem',
        u'unangepaßten'                : u'unangepassten',
        u'unangepaßter'                : u'unangepasster',
        u'unangepaßtere'               : u'unangepasstere',
        u'unangepaßterem'              : u'unangepassterem',
        u'unangepaßteren'              : u'unangepassteren',
        u'unangepaßterer'              : u'unangepassterer',
        u'unangepaßteres'              : u'unangepassteres',
        u'unangepaßtes'                : u'unangepasstes',
        u'unangepaßteste'              : u'unangepassteste',
        u'unangepaßtestem'             : u'unangepasstestem',
        u'unangepaßtesten'             : u'unangepasstesten',
        u'unangepaßtester'             : u'unangepasstester',
        u'unangepaßtestes'             : u'unangepasstestes',
        u'unangepaßtheit'              : u'unangepasstheit',
        u'unbeeinflußbar'              : u'unbeeinflussbar',
        u'unbeeinflußbare'             : u'unbeeinflussbare',
        u'unbeeinflußbarem'            : u'unbeeinflussbarem',
        u'unbeeinflußbaren'            : u'unbeeinflussbaren',
        u'unbeeinflußbarer'            : u'unbeeinflussbarer',
        u'unbeeinflußbares'            : u'unbeeinflussbares',
        u'unbeeinflußbarkeit'          : u'unbeeinflussbarkeit',
        u'unbeeinflußt'                : u'unbeeinflusst',
        u'unbeeinflußte'               : u'unbeeinflusste',
        u'unbeeinflußtem'              : u'unbeeinflusstem',
        u'unbeeinflußten'              : u'unbeeinflussten',
        u'unbeeinflußter'              : u'unbeeinflusster',
        u'unbeeinflußtes'              : u'unbeeinflusstes',
        u'unbewußt'                    : u'unbewusst',
        u'unbewußte'                   : u'unbewusste',
        u'unbewußtem'                  : u'unbewusstem',
        u'unbewußten'                  : u'unbewussten',
        u'unbewußter'                  : u'unbewusster',
        u'unbewußtere'                 : u'unbewusstere',
        u'unbewußterem'                : u'unbewussterem',
        u'unbewußteren'                : u'unbewussteren',
        u'unbewußterer'                : u'unbewussterer',
        u'unbewußteres'                : u'unbewussteres',
        u'unbewußtes'                  : u'unbewusstes',
        u'unbewußteste'                : u'unbewussteste',
        u'unbewußtestem'               : u'unbewusstestem',
        u'unbewußtesten'               : u'unbewusstesten',
        u'unbewußtester'               : u'unbewusstester',
        u'unbewußtestes'               : u'unbewusstestes',
        u'unbewußtheit'                : u'unbewusstheit',
        u'unerläßlich'                 : u'unerlässlich',
        u'unerläßliche'                : u'unerlässliche',
        u'unerläßlichem'               : u'unerlässlichem',
        u'unerläßlichen'               : u'unerlässlichen',
        u'unerläßlicher'               : u'unerlässlicher',
        u'unerläßlichere'              : u'unerlässlichere',
        u'unerläßlicherem'             : u'unerlässlicherem',
        u'unerläßlicheren'             : u'unerlässlicheren',
        u'unerläßlicherer'             : u'unerlässlicherer',
        u'unerläßlicheres'             : u'unerlässlicheres',
        u'unerläßliches'               : u'unerlässliches',
        u'unerläßlichste'              : u'unerlässlichste',
        u'unerläßlichstem'             : u'unerlässlichstem',
        u'unerläßlichsten'             : u'unerlässlichsten',
        u'unerläßlichster'             : u'unerlässlichster',
        u'unerläßlichstes'             : u'unerlässlichstes',
        u'unermeßlich'                 : u'unermesslich',
        u'unermeßliche'                : u'unermessliche',
        u'unermeßlichem'               : u'unermesslichem',
        u'unermeßlichen'               : u'unermesslichen',
        u'unermeßlicher'               : u'unermesslicher',
        u'unermeßlichere'              : u'unermesslichere',
        u'unermeßlicherem'             : u'unermesslicherem',
        u'unermeßlicheren'             : u'unermesslicheren',
        u'unermeßlicherer'             : u'unermesslicherer',
        u'unermeßlicheres'             : u'unermesslicheres',
        u'unermeßliches'               : u'unermessliches',
        u'unermeßlichkeit'             : u'unermesslichkeit',
        u'unermeßlichste'              : u'unermesslichste',
        u'unermeßlichstem'             : u'unermesslichstem',
        u'unermeßlichsten'             : u'unermesslichsten',
        u'unermeßlichster'             : u'unermesslichster',
        u'unermeßlichstes'             : u'unermesslichstes',
        u'unfaßbar'                    : u'unfassbar',
        u'unfaßbare'                   : u'unfassbare',
        u'unfaßbarem'                  : u'unfassbarem',
        u'unfaßbaren'                  : u'unfassbaren',
        u'unfaßbarer'                  : u'unfassbarer',
        u'unfaßbarere'                 : u'unfassbarere',
        u'unfaßbarerem'                : u'unfassbarerem',
        u'unfaßbareren'                : u'unfassbareren',
        u'unfaßbarerer'                : u'unfassbarerer',
        u'unfaßbareres'                : u'unfassbareres',
        u'unfaßbares'                  : u'unfassbares',
        u'unfaßbarste'                 : u'unfassbarste',
        u'unfaßbarstem'                : u'unfassbarstem',
        u'unfaßbarsten'                : u'unfassbarsten',
        u'unfaßbarster'                : u'unfassbarster',
        u'unfaßbarstes'                : u'unfassbarstes',
        u'unfaßlich'                   : u'unfasslich',
        u'unfaßliche'                  : u'unfassliche',
        u'unfaßlichem'                 : u'unfasslichem',
        u'unfaßlichen'                 : u'unfasslichen',
        u'unfaßlicher'                 : u'unfasslicher',
        u'unfaßlichere'                : u'unfasslichere',
        u'unfaßlicherem'               : u'unfasslicherem',
        u'unfaßlicheren'               : u'unfasslicheren',
        u'unfaßlicherer'               : u'unfasslicherer',
        u'unfaßlicheres'               : u'unfasslicheres',
        u'unfaßliches'                 : u'unfassliches',
        u'unfaßlichste'                : u'unfasslichste',
        u'unfaßlichstem'               : u'unfasslichstem',
        u'unfaßlichsten'               : u'unfasslichsten',
        u'unfaßlichster'               : u'unfasslichster',
        u'unfaßlichstes'               : u'unfasslichstes',
        u'ungewiß'                     : u'ungewiss',
        u'ungewißheit'                 : u'ungewissheit',
        u'universitätsabschluß'        : u'universitätsabschluss',
        u'unmißverständlich'           : u'unmissverständlich',
        u'unmißverständliche'          : u'unmissverständliche',
        u'unmißverständlichem'         : u'unmissverständlichem',
        u'unmißverständlichen'         : u'unmissverständlichen',
        u'unmißverständlicher'         : u'unmissverständlicher',
        u'unmißverständlichere'        : u'unmissverständlichere',
        u'unmißverständlicherem'       : u'unmissverständlicherem',
        u'unmißverständlicheren'       : u'unmissverständlicheren',
        u'unmißverständlicherer'       : u'unmissverständlicherer',
        u'unmißverständlicheres'       : u'unmissverständlicheres',
        u'unmißverständliches'         : u'unmissverständliches',
        u'unmißverständlichste'        : u'unmissverständlichste',
        u'unmißverständlichstem'       : u'unmissverständlichstem',
        u'unmißverständlichsten'       : u'unmissverständlichsten',
        u'unmißverständlichster'       : u'unmissverständlichster',
        u'unmißverständlichstes'       : u'unmissverständlichstes',
        u'unpaß'                       : u'unpass',
        u'unpäßlich'                   : u'unpässlich',
        u'unpäßliche'                  : u'unpässliche',
        u'unpäßlichem'                 : u'unpässlichem',
        u'unpäßlichen'                 : u'unpässlichen',
        u'unpäßlicher'                 : u'unpässlicher',
        u'unpäßlichere'                : u'unpässlichere',
        u'unpäßlicherem'               : u'unpässlicherem',
        u'unpäßlicheren'               : u'unpässlicheren',
        u'unpäßlicherer'               : u'unpässlicherer',
        u'unpäßlicheres'               : u'unpässlicheres',
        u'unpäßliches'                 : u'unpässliches',
        u'unpäßlichkeit'               : u'unpässlichkeit',
        u'unpäßlichkeiten'             : u'unpässlichkeiten',
        u'unpäßlichste'                : u'unpässlichste',
        u'unpäßlichstem'               : u'unpässlichstem',
        u'unpäßlichsten'               : u'unpässlichsten',
        u'unpäßlichster'               : u'unpässlichster',
        u'unpäßlichstes'               : u'unpässlichstes',
        u'unrechtsbewußtsein'          : u'unrechtsbewusstsein',
        u'unselbständig'               : u'unselbstständig',
        u'unterausschuß'               : u'unterausschuss',
        u'unterbewußt'                 : u'unterbewusst',
        u'unterbewußte'                : u'unterbewusste',
        u'unterbewußtem'               : u'unterbewusstem',
        u'unterbewußten'               : u'unterbewussten',
        u'unterbewußter'               : u'unterbewusster',
        u'unterbewußtes'               : u'unterbewusstes',
        u'unterbewußtsein'             : u'unterbewusstsein',
        u'unterbewußtseins'            : u'unterbewusstseins',
        u'unterbiß'                    : u'unterbiss',
        u'untergränze'                 : u'untergrenze',
        u'untergränzen'                : u'untergrenzen',
        u'unterlaß'                    : u'unterlass',
        u'unterlaßt'                   : u'unterlasst',
        u'unterläßt'                   : u'unterlässt',
        u'unterschuß'                  : u'unterschuss',
        u'untersuchungsausschuß'       : u'untersuchungsausschuss',
        u'unvergeßlich'                : u'unvergesslich',
        u'unvergeßliche'               : u'unvergessliche',
        u'unvergeßlichem'              : u'unvergesslichem',
        u'unvergeßlichen'              : u'unvergesslichen',
        u'unvergeßlicher'              : u'unvergesslicher',
        u'unvergeßlichere'             : u'unvergesslichere',
        u'unvergeßlicherem'            : u'unvergesslicherem',
        u'unvergeßlicheren'            : u'unvergesslicheren',
        u'unvergeßlicherer'            : u'unvergesslicherer',
        u'unvergeßlicheres'            : u'unvergesslicheres',
        u'unvergeßliches'              : u'unvergessliches',
        u'unvergeßlichste'             : u'unvergesslichste',
        u'unvergeßlichstem'            : u'unvergesslichstem',
        u'unvergeßlichsten'            : u'unvergesslichsten',
        u'unvergeßlichster'            : u'unvergesslichster',
        u'unvergeßlichstes'            : u'unvergesslichstes',
        u'unverläßlich'                : u'unverlässlich',
        u'vakuummeßröhre'              : u'vakuummessröhre',
        u'vakuummeßröhren'             : u'vakuummessröhren',
        u'varietee'                    : u'varieté',
        u'varietees'                   : u'varietés',
        u'vaterschaftsprozeß'          : u'vaterschaftsprozess',
        u'veranlaß'                    : u'veranlass',
        u'veranlaßt'                   : u'veranlasst',
        u'veranlaßte'                  : u'veranlasste',
        u'veranlaßten'                 : u'veranlassten',
        u'veranlaßtest'                : u'veranlasstest',
        u'veranlaßtet'                 : u'veranlasstet',
        u'verantwortungsbewußt'        : u'verantwortungsbewusst',
        u'verantwortungsbewußte'       : u'verantwortungsbewusste',
        u'verantwortungsbewußtem'      : u'verantwortungsbewusstem',
        u'verantwortungsbewußten'      : u'verantwortungsbewussten',
        u'verantwortungsbewußter'      : u'verantwortungsbewusster',
        u'verantwortungsbewußtere'     : u'verantwortungsbewusstere',
        u'verantwortungsbewußterem'    : u'verantwortungsbewussterem',
        u'verantwortungsbewußteren'    : u'verantwortungsbewussteren',
        u'verantwortungsbewußterer'    : u'verantwortungsbewussterer',
        u'verantwortungsbewußteres'    : u'verantwortungsbewussteres',
        u'verantwortungsbewußtes'      : u'verantwortungsbewusstes',
        u'verantwortungsbewußteste'    : u'verantwortungsbewussteste',
        u'verantwortungsbewußtestem'   : u'verantwortungsbewusstestem',
        u'verantwortungsbewußtesten'   : u'verantwortungsbewusstesten',
        u'verantwortungsbewußtester'   : u'verantwortungsbewusstester',
        u'verantwortungsbewußtestes'   : u'verantwortungsbewusstestes',
        u'verantwortungsbewußtsein'    : u'verantwortungsbewusstsein',
        u'verantwortungsbewußtseins'   : u'verantwortungsbewusstseins',
        u'verbeßre'                    : u'verbessre',
        u'verbiß'                      : u'verbiss',
        u'verbißt'                     : u'verbisst',
        u'verblaßt'                    : u'verblasst',
        u'verblaßte'                   : u'verblasste',
        u'verblaßtem'                  : u'verblasstem',
        u'verblaßten'                  : u'verblassten',
        u'verblaßter'                  : u'verblasster',
        u'verblaßtere'                 : u'verblasstere',
        u'verblaßterem'                : u'verblassterem',
        u'verblaßteren'                : u'verblassteren',
        u'verblaßterer'                : u'verblassterer',
        u'verblaßteres'                : u'verblassteres',
        u'verblaßtes'                  : u'verblasstes',
        u'verblaßteste'                : u'verblassteste',
        u'verblaßtestem'               : u'verblasstestem',
        u'verblaßtesten'               : u'verblasstesten',
        u'verblaßtester'               : u'verblasstester',
        u'verblaßtestes'               : u'verblasstestes',
        u'verbleu'                     : u'verbläu',
        u'verbleue'                    : u'verbläue',
        u'verbleuen'                   : u'verbläuen',
        u'verbleuend'                  : u'verbläuend',
        u'verbleuest'                  : u'verbläuest',
        u'verbleuet'                   : u'verbläuet',
        u'verbleust'                   : u'verbläust',
        u'verbleut'                    : u'verbläut',
        u'verbleute'                   : u'verbläute',
        u'verbleuten'                  : u'verbläuten',
        u'verbleutest'                 : u'verbläutest',
        u'verbleutet'                  : u'verbläutet',
        u'verbrennungsprozeß'          : u'verbrennungsprozess',
        u'verbundschipaß'              : u'verbundschipass',
        u'verbundskipaß'               : u'verbundskipass',
        u'verdauungsprozeß'            : u'verdauungsprozess',
        u'verdroß'                     : u'verdross',
        u'verdroßt'                    : u'verdrosst',
        u'verdruß'                     : u'verdruss',
        u'verdrängungsprozeß'          : u'verdrängungsprozess',
        u'verfaß'                      : u'verfass',
        u'verfaßt'                     : u'verfasst',
        u'verfaßte'                    : u'verfasste',
        u'verfaßten'                   : u'verfassten',
        u'verfaßtes'                   : u'verfasstes',
        u'verfaßtest'                  : u'verfasstest',
        u'verfaßtet'                   : u'verfasstet',
        u'verfloß'                     : u'verfloss',
        u'verfreßt'                    : u'verfresst',
        u'verfriß'                     : u'verfriss',
        u'verfrißt'                    : u'verfrisst',
        u'vergeßlich'                  : u'vergesslich',
        u'vergeßliche'                 : u'vergessliche',
        u'vergeßlichem'                : u'vergesslichem',
        u'vergeßlichen'                : u'vergesslichen',
        u'vergeßlicher'                : u'vergesslicher',
        u'vergeßlichere'               : u'vergesslichere',
        u'vergeßlicherem'              : u'vergesslicherem',
        u'vergeßlicheren'              : u'vergesslicheren',
        u'vergeßlicherer'              : u'vergesslicherer',
        u'vergeßlicheres'              : u'vergesslicheres',
        u'vergeßliches'                : u'vergessliches',
        u'vergeßlichkeit'              : u'vergesslichkeit',
        u'vergeßlichste'               : u'vergesslichste',
        u'vergeßlichstem'              : u'vergesslichstem',
        u'vergeßlichsten'              : u'vergesslichsten',
        u'vergeßlichster'              : u'vergesslichster',
        u'vergeßlichstes'              : u'vergesslichstes',
        u'vergeßt'                     : u'vergesst',
        u'vergiß'                      : u'vergiss',
        u'vergißmeinnicht'             : u'vergissmeinnicht',
        u'vergißmeinnichte'            : u'vergissmeinnichte',
        u'vergißmeinnichten'           : u'vergissmeinnichten',
        u'vergißmeinnichtes'           : u'vergissmeinnichtes',
        u'vergißmeinnichts'            : u'vergissmeinnichts',
        u'vergißt'                     : u'vergisst',
        u'vergoß'                      : u'vergoss',
        u'vergoßt'                     : u'vergosst',
        u'verhaßt'                     : u'verhasst',
        u'verhaßte'                    : u'verhasste',
        u'verhaßtem'                   : u'verhasstem',
        u'verhaßten'                   : u'verhassten',
        u'verhaßter'                   : u'verhasster',
        u'verhaßtere'                  : u'verhasstere',
        u'verhaßterem'                 : u'verhassterem',
        u'verhaßteren'                 : u'verhassteren',
        u'verhaßterer'                 : u'verhassterer',
        u'verhaßteres'                 : u'verhassteres',
        u'verhaßtes'                   : u'verhasstes',
        u'verhaßteste'                 : u'verhassteste',
        u'verhaßtestem'                : u'verhasstestem',
        u'verhaßtesten'                : u'verhasstesten',
        u'verhaßtester'                : u'verhasstester',
        u'verhaßtestes'                : u'verhasstestes',
        u'verhältniß'                  : u'verhältnis',
        u'verkehrsfluß'                : u'verkehrsfluss',
        u'verlaß'                      : u'verlass',
        u'verlaßt'                     : u'verlasst',
        u'verleumdungsprozeß'          : u'verleumdungsprozess',
        u'verläßlich'                  : u'verlässlich',
        u'verläßliche'                 : u'verlässliche',
        u'verläßlichem'                : u'verlässlichem',
        u'verläßlichen'                : u'verlässlichen',
        u'verläßlicher'                : u'verlässlicher',
        u'verläßlichere'               : u'verlässlichere',
        u'verläßlicherem'              : u'verlässlicherem',
        u'verläßlicheren'              : u'verlässlicheren',
        u'verläßlicherer'              : u'verlässlicherer',
        u'verläßlicheres'              : u'verlässlicheres',
        u'verläßliches'                : u'verlässliches',
        u'verläßlichkeit'              : u'verlässlichkeit',
        u'verläßlichste'               : u'verlässlichste',
        u'verläßlichstem'              : u'verlässlichstem',
        u'verläßlichsten'              : u'verlässlichsten',
        u'verläßlichster'              : u'verlässlichster',
        u'verläßlichstes'              : u'verlässlichstes',
        u'verläßt'                     : u'verlässt',
        u'vermaßle'                    : u'vermassle',
        u'vermaßlest'                  : u'vermasslest',
        u'vermaßlet'                   : u'vermasslet',
        u'vermeßt'                     : u'vermesst',
        u'vermittlungsausschuß'        : u'vermittlungsausschuss',
        u'vermiß'                      : u'vermiss',
        u'vermißt'                     : u'vermisst',
        u'vermißte'                    : u'vermisste',
        u'vermißtem'                   : u'vermisstem',
        u'vermißten'                   : u'vermissten',
        u'vermißtenanzeige'            : u'vermisstenanzeige',
        u'vermißtenanzeigen'           : u'vermisstenanzeigen',
        u'vermißtenfall'               : u'vermisstenfall',
        u'vermißtenfalle'              : u'vermisstenfalle',
        u'vermißtenfalles'             : u'vermisstenfalles',
        u'vermißtenfalls'              : u'vermisstenfalls',
        u'vermißtenfälle'              : u'vermisstenfälle',
        u'vermißtenfällen'             : u'vermisstenfällen',
        u'vermißter'                   : u'vermisster',
        u'vermißtes'                   : u'vermisstes',
        u'vermißtest'                  : u'vermisstest',
        u'vermißtet'                   : u'vermisstet',
        u'vermuth'                     : u'vermut',
        u'vermuthe'                    : u'vermute',
        u'vermuthen'                   : u'vermuten',
        u'vermuthend'                  : u'vermutend',
        u'vermuthest'                  : u'vermutest',
        u'vermuthet'                   : u'vermutet',
        u'vermuthete'                  : u'vermutete',
        u'vermutheten'                 : u'vermuteten',
        u'vermuthetest'                : u'vermutetest',
        u'vermuthetet'                 : u'vermutetet',
        u'verpaß'                      : u'verpass',
        u'verpaßt'                     : u'verpasst',
        u'verpaßte'                    : u'verpasste',
        u'verpaßten'                   : u'verpassten',
        u'verpaßtest'                  : u'verpasstest',
        u'verpaßtet'                   : u'verpasstet',
        u'verpiß'                      : u'verpiss',
        u'verpißt'                     : u'verpisst',
        u'verpißte'                    : u'verpisste',
        u'verpißten'                   : u'verpissten',
        u'verpißtest'                  : u'verpisstest',
        u'verpißtet'                   : u'verpisstet',
        u'verpraßt'                    : u'verprasst',
        u'verpraßte'                   : u'verprasste',
        u'verpraßtet'                  : u'verprasstet',
        u'verriß'                      : u'verriss',
        u'verrißt'                     : u'verrisst',
        u'verschiß'                    : u'verschiss',
        u'verschißt'                   : u'verschisst',
        u'verschleißprozeß'            : u'verschleißprozess',
        u'verschlimmbeßre'             : u'verschlimmbessre',
        u'verschliß'                   : u'verschliss',
        u'verschlißt'                  : u'verschlisst',
        u'verschloß'                   : u'verschloss',
        u'verschloßt'                  : u'verschlosst',
        u'verschluß'                   : u'verschluss',
        u'verschlußdeckel'             : u'verschlussdeckel',
        u'verschlußkappe'              : u'verschlusskappe',
        u'verschlußlaut'               : u'verschlusslaut',
        u'verschlußlaute'              : u'verschlusslaute',
        u'verschlußlauten'             : u'verschlusslauten',
        u'verschlußlautes'             : u'verschlusslautes',
        u'verschlußlauts'              : u'verschlusslauts',
        u'verschlußsache'              : u'verschlusssache',
        u'verschlußsachen'             : u'verschlusssachen',
        u'verschlußschraube'           : u'verschlussschraube',
        u'verschlußstreifen'           : u'verschlussstreifen',
        u'verschlußzeit'               : u'verschlusszeit',
        u'verschlüßle'                 : u'verschlüssle',
        u'verschlüßlest'               : u'verschlüsslest',
        u'verschlüßlet'                : u'verschlüsslet',
        u'verschoß'                    : u'verschoss',
        u'verschoßt'                   : u'verschosst',
        u'versicherungsmißbrauch'      : u'versicherungsmissbrauch',
        u'versorgungsengpaß'           : u'versorgungsengpass',
        u'versorgungsgenuß'            : u'versorgungsgenuss',
        u'verspliß'                    : u'verspliss',
        u'versplißt'                   : u'versplisst',
        u'vertelephonier'              : u'vertelefonier',
        u'vertelephoniere'             : u'vertelefoniere',
        u'vertelephonieren'            : u'vertelefonieren',
        u'vertelephonierend'           : u'vertelefonierend',
        u'vertelephonierende'          : u'vertelefonierende',
        u'vertelephonierendem'         : u'vertelefonierendem',
        u'vertelephonierenden'         : u'vertelefonierenden',
        u'vertelephonierender'         : u'vertelefonierender',
        u'vertelephonierendes'         : u'vertelefonierendes',
        u'vertelephonierest'           : u'vertelefonierest',
        u'vertelephonieret'            : u'vertelefonieret',
        u'vertelephonierst'            : u'vertelefonierst',
        u'vertelephoniert'             : u'vertelefoniert',
        u'vertelephonierte'            : u'vertelefonierte',
        u'vertelephoniertem'           : u'vertelefoniertem',
        u'vertelephonierten'           : u'vertelefonierten',
        u'vertelephonierter'           : u'vertelefonierter',
        u'vertelephoniertes'           : u'vertelefoniertes',
        u'vertelephoniertest'          : u'vertelefoniertest',
        u'vertelephoniertet'           : u'vertelefoniertet',
        u'vertragsabschluß'            : u'vertragsabschluss',
        u'vertragsschluß'              : u'vertragsschluss',
        u'vertrauensvorschuß'          : u'vertrauensvorschuss',
        u'verwäßre'                    : u'verwässre',
        u'veränderungsprozeß'          : u'veränderungsprozess',
        u'vierpaß'                     : u'vierpass',
        u'viertkläßler'                : u'viertklässler',
        u'viertkläßlern'               : u'viertklässlern',
        u'viertkläßlers'               : u'viertklässlers',
        u'violetton'                   : u'violettton',
        u'violettone'                  : u'violetttone',
        u'violettones'                 : u'violetttones',
        u'violettons'                  : u'violetttons',
        u'violettöne'                  : u'violetttöne',
        u'violettönen'                 : u'violetttönen',
        u'vocal'                       : u'vokal',
        u'vocale'                      : u'vokale',
        u'vocalen'                     : u'vokalen',
        u'vocals'                      : u'vokals',
        u'vocativ'                     : u'vokativ',
        u'vocative'                    : u'vokative',
        u'vocativen'                   : u'vokativen',
        u'vocativs'                    : u'vokativs',
        u'vogelschiß'                  : u'vogelschiss',
        u'volkskongreß'                : u'volkskongress',
        u'volksmässig'                 : u'volksmäßig',
        u'volladen'                    : u'vollladen',
        u'vollast'                     : u'volllast',
        u'vollasten'                   : u'volllasten',
        u'vollaufen'                   : u'volllaufen',
        u'vollaut'                     : u'volllaut',
        u'vollaute'                    : u'volllaute',
        u'vollauten'                   : u'volllauten',
        u'vollautes'                   : u'volllautes',
        u'vollauts'                    : u'volllauts',
        u'volleibig'                   : u'vollleibig',
        u'vollgeschoß'                 : u'vollgeschoss',
        u'vollkomne'                   : u'vollkommne',
        u'vollschiß'                   : u'vollschiss',
        u'vollschißt'                  : u'vollschisst',
        u'vonnöthen'                   : u'vonnöten',
        u'vorgefaßt'                   : u'vorgefasst',
        u'vorgefaßte'                  : u'vorgefasste',
        u'vorgefaßtem'                 : u'vorgefasstem',
        u'vorgefaßten'                 : u'vorgefassten',
        u'vorgefaßter'                 : u'vorgefasster',
        u'vorgefaßtere'                : u'vorgefasstere',
        u'vorgefaßterem'               : u'vorgefassterem',
        u'vorgefaßteren'               : u'vorgefassteren',
        u'vorgefaßterer'               : u'vorgefassterer',
        u'vorgefaßteres'               : u'vorgefassteres',
        u'vorgefaßtes'                 : u'vorgefasstes',
        u'vorgefaßteste'               : u'vorgefassteste',
        u'vorgefaßtestem'              : u'vorgefasstestem',
        u'vorgefaßtesten'              : u'vorgefasstesten',
        u'vorgefaßtester'              : u'vorgefasstester',
        u'vorgefaßtestes'              : u'vorgefasstestes',
        u'vorhangschloß'               : u'vorhangschloss',
        u'vorhängeschloß'              : u'vorhängeschloss',
        u'vorlagenmißbrauch'           : u'vorlagenmissbrauch',
        u'vorlaßt'                     : u'vorlasst',
        u'vorlegeschloß'               : u'vorlegeschloss',
        u'vorläßt'                     : u'vorlässt',
        u'vorschlußrunde'              : u'vorschlussrunde',
        u'vorschlußrunden'             : u'vorschlussrunden',
        u'vorschoß'                    : u'vorschoss',
        u'vorschoßt'                   : u'vorschosst',
        u'vorschuß'                    : u'vorschuss',
        u'vorschußlorbeeren'           : u'vorschusslorbeeren',
        u'vorschußweise'               : u'vorschussweise',
        u'vorschußzahlung'             : u'vorschusszahlung',
        u'vorschußzins'                : u'vorschusszins',
        u'vorschußzinse'               : u'vorschusszinse',
        u'vorschußzinsen'              : u'vorschusszinsen',
        u'vorschußzinses'              : u'vorschusszinses',
        u'wachgeküßt'                  : u'wachgeküsst',
        u'wachküßt'                    : u'wachküsst',
        u'wachküßte'                   : u'wachküsste',
        u'wachküßten'                  : u'wachküssten',
        u'wachküßtest'                 : u'wachküsstest',
        u'wachküßtet'                  : u'wachküsstet',
        u'wachsabguß'                  : u'wachsabguss',
        u'wachsthum'                   : u'wachstum',
        u'wachsthums'                  : u'wachstums',
        u'wachstumsprozeß'             : u'wachstumsprozess',
        u'waffenpaß'                   : u'waffenpass',
        u'wage'                        : u'waage',
        u'wagen'                       : u'waagen',
        u'wahlausschuß'                : u'wahlausschuss',
        u'waldschlußzeit'              : u'waldschlusszeit',
        u'waldschlußzeiten'            : u'waldschlusszeiten',
        u'walke'                       : u'walk',
        u'walkfaß'                     : u'walkfass',
        u'wallfisch'                   : u'walfisch',
        u'walnuß'                      : u'walnuss',
        u'walnußbaum'                  : u'walnussbaum',
        u'walnußbaume'                 : u'walnussbaume',
        u'walnußbaumes'                : u'walnussbaumes',
        u'walnußbaums'                 : u'walnussbaums',
        u'walnußbäume'                 : u'walnussbäume',
        u'walnußbäumen'                : u'walnussbäumen',
        u'walnußschale'                : u'walnussschale',
        u'walnußschalen'               : u'walnussschalen',
        u'walroß'                      : u'walross',
        u'walther'                     : u'walter',
        u'wandale'                     : u'vandale',
        u'wandalen'                    : u'vandalen',
        u'wandalier'                   : u'vandalier',
        u'wandaliere'                  : u'vandaliere',
        u'wandalieren'                 : u'vandalieren',
        u'wandalierend'                : u'vandalierend',
        u'wandalierest'                : u'vandalierest',
        u'wandalieret'                 : u'vandalieret',
        u'wandalierst'                 : u'vandalierst',
        u'wandaliert'                  : u'vandaliert',
        u'wandalierte'                 : u'vandalierte',
        u'wandalierten'                : u'vandalierten',
        u'wandaliertest'               : u'vandaliertest',
        u'wandaliertet'                : u'vandaliertet',
        u'wandalismus'                 : u'vandalismus',
        u'wandalismussicherung'        : u'vandalismussicherung',
        u'wandalismussicherungen'      : u'vandalismussicherungen',
        u'wandlungsprozeß'             : u'wandlungsprozess',
        u'warnschuß'                   : u'warnschuss',
        u'wasseranschluß'              : u'wasseranschluss',
        u'wasserfaß'                   : u'wasserfass',
        u'wasserschloß'                : u'wasserschloss',
        u'waßre'                       : u'wassre',
        u'waßrest'                     : u'wassrest',
        u'waßret'                      : u'wassret',
        u'wechselprozeß'               : u'wechselprozess',
        u'wechselregreß'               : u'wechselregress',
        u'weggemußt'                   : u'weggemusst',
        u'weglaßprobe'                 : u'weglassprobe',
        u'weglaßproben'                : u'weglassproben',
        u'weglaßt'                     : u'weglasst',
        u'wegläßt'                     : u'weglässt',
        u'wegschmiß'                   : u'wegschmiss',
        u'wegschmißt'                  : u'wegschmisst',
        u'wehrpaß'                     : u'wehrpass',
        u'weidenruthe'                 : u'weidenrute',
        u'weinfaß'                     : u'weinfass',
        u'weisst'                      : u'weißt',
        u'weitereßt'                   : u'weiteresst',
        u'weitergemußt'                : u'weitergemusst',
        u'weitergewußt'                : u'weitergewusst',
        u'weiterißt'                   : u'weiterisst',
        u'weitermuß'                   : u'weitermuss',
        u'weitermußt'                  : u'weitermusst',
        u'weitermußte'                 : u'weitermusste',
        u'weitermußten'                : u'weitermussten',
        u'weitermußtest'               : u'weitermusstest',
        u'weitermußtet'                : u'weitermusstet',
        u'weitermüßt'                  : u'weitermüsst',
        u'weitermüßte'                 : u'weitermüsste',
        u'weitermüßten'                : u'weitermüssten',
        u'weitermüßtest'               : u'weitermüsstest',
        u'weitermüßtet'                : u'weitermüsstet',
        u'weiterwißt'                  : u'weiterwisst',
        u'weiterwußte'                 : u'weiterwusste',
        u'weiterwußten'                : u'weiterwussten',
        u'weiterwußtest'               : u'weiterwusstest',
        u'weiterwußtet'                : u'weiterwusstet',
        u'weiterwüßte'                 : u'weiterwüsste',
        u'weiterwüßten'                : u'weiterwüssten',
        u'weiterwüßtest'               : u'weiterwüsstest',
        u'weiterwüßtet'                : u'weiterwüsstet',
        u'weitschuß'                   : u'weitschuss',
        u'weißfluß'                    : u'weißfluss',
        u'weißrußland'                 : u'weißrussland',
        u'weißrußlands'                : u'weißrusslands',
        u'weltprozeß'                  : u'weltprozess',
        u'werkstattermin'              : u'werkstatttermin',
        u'werkstattermine'             : u'werkstatttermine',
        u'werkstatterminen'            : u'werkstattterminen',
        u'werkstattermines'            : u'werkstatttermines',
        u'werkstattermins'             : u'werkstatttermins',
        u'werkstofforschung'           : u'werkstoffforschung',
        u'werth'                       : u'wert',
        u'werthhaltig'                 : u'werthaltig',
        u'westgothe'                   : u'westgote',
        u'westgothen'                  : u'westgoten',
        u'wettauchen'                  : u'wetttauchen',
        u'wetteufel'                   : u'wettteufel',
        u'wetturnen'                   : u'wettturnen',
        u'weßhalb'                     : u'weshalb',
        u'wiktionary:teestube/archiv/2016/10' : u'teestube/archiv/2016/10',
        u'willensbildungsprozeß'       : u'willensbildungsprozess',
        u'winterschlußverkauf'         : u'winterschlussverkauf',
        u'winterschlußverkaufe'        : u'winterschlussverkaufe',
        u'winterschlußverkaufes'       : u'winterschlussverkaufes',
        u'winterschlußverkaufs'        : u'winterschlussverkaufs',
        u'winterschlußverkäufe'        : u'winterschlussverkäufe',
        u'winterschlußverkäufen'       : u'winterschlussverkäufen',
        u'wirth'                       : u'wirt',
        u'wirthe'                      : u'wirte',
        u'wirthen'                     : u'wirten',
        u'wirthes'                     : u'wirtes',
        u'wirths'                      : u'wirts',
        u'wirtschaftsausschuß'         : u'wirtschaftsausschuss',
        u'wischmop'                    : u'wischmopp',
        u'wischmops'                   : u'wischmopps',
        u'witterungseinfluß'           : u'witterungseinfluss',
        u'wißbegier'                   : u'wissbegier',
        u'wißbegierde'                 : u'wissbegierde',
        u'wißbegierig'                 : u'wissbegierig',
        u'wißbegierige'                : u'wissbegierige',
        u'wißbegierigem'               : u'wissbegierigem',
        u'wißbegierigen'               : u'wissbegierigen',
        u'wißbegieriger'               : u'wissbegieriger',
        u'wißbegierigere'              : u'wissbegierigere',
        u'wißbegierigerem'             : u'wissbegierigerem',
        u'wißbegierigeren'             : u'wissbegierigeren',
        u'wißbegierigerer'             : u'wissbegierigerer',
        u'wißbegierigeres'             : u'wissbegierigeres',
        u'wißbegieriges'               : u'wissbegieriges',
        u'wißbegierigste'              : u'wissbegierigste',
        u'wißbegierigstem'             : u'wissbegierigstem',
        u'wißbegierigsten'             : u'wissbegierigsten',
        u'wißbegierigster'             : u'wissbegierigster',
        u'wißbegierigstes'             : u'wissbegierigstes',
        u'wißt'                        : u'wisst',
        u'wochenfluß'                  : u'wochenfluss',
        u'wohlfahrtsausschuß'          : u'wohlfahrtsausschuss',
        u'wollappen'                   : u'wolllappen',
        u'wollaus'                     : u'wolllaus',
        u'wolläuse'                    : u'wollläuse',
        u'wolläusen'                   : u'wollläusen',
        u'wußte'                       : u'wusste',
        u'wußten'                      : u'wussten',
        u'wußtest'                     : u'wusstest',
        u'wußtet'                      : u'wusstet',
        u'wächte'                      : u'wechte',
        u'wächten'                     : u'wechten',
        u'wählscheibentelephon'        : u'wählscheibentelefon',
        u'wählscheibentelephone'       : u'wählscheibentelefone',
        u'wählscheibentelephonen'      : u'wählscheibentelefonen',
        u'wählscheibentelephons'       : u'wählscheibentelefons',
        u'wäßre'                       : u'wässre',
        u'wäßrig'                      : u'wässrig',
        u'wäßrige'                     : u'wässrige',
        u'wäßrigem'                    : u'wässrigem',
        u'wäßrigen'                    : u'wässrigen',
        u'wäßriger'                    : u'wässriger',
        u'wäßrigere'                   : u'wässrigere',
        u'wäßrigerem'                  : u'wässrigerem',
        u'wäßrigeren'                  : u'wässrigeren',
        u'wäßrigerer'                  : u'wässrigerer',
        u'wäßrigeres'                  : u'wässrigeres',
        u'wäßriges'                    : u'wässriges',
        u'wäßrigkeit'                  : u'wässrigkeit',
        u'wäßrigste'                   : u'wässrigste',
        u'wäßrigstem'                  : u'wässrigstem',
        u'wäßrigsten'                  : u'wässrigsten',
        u'wäßrigster'                  : u'wässrigster',
        u'wäßrigstes'                  : u'wässrigstes',
        u'wünschelruthe'               : u'wünschelrute',
        u'wünschelruthen'              : u'wünschelruten',
        u'wüßte'                       : u'wüsste',
        u'wüßten'                      : u'wüssten',
        u'wüßtest'                     : u'wüsstest',
        u'wüßtet'                      : u'wüsstet',
        u'yaleschloß'                  : u'yaleschloss',
        u'yoghurt'                     : u'joghurt',
        u'yoghurtbecher'               : u'joghurtbecher',
        u'yoghurtbechern'              : u'joghurtbechern',
        u'yoghurtbechers'              : u'joghurtbechers',
        u'yoghurts'                    : u'joghurts',
        u'zahlenschloß'                : u'zahlenschloss',
        u'zahlungsbilanzüberschuß'     : u'zahlungsbilanzüberschuss',
        u'zahlungsfluß'                : u'zahlungsfluss',
        u'zaubernuß'                   : u'zaubernuss',
        u'zechenstillegung'            : u'zechenstilllegung',
        u'zeckenbiß'                   : u'zeckenbiss',
        u'zeilengußmaschine'           : u'zeilengussmaschine',
        u'zellehre'                    : u'zelllehre',
        u'zellstoffabrik'              : u'zellstofffabrik',
        u'zentralausschuß'             : u'zentralausschuss',
        u'zentralrußland'              : u'zentralrussland',
        u'zentralrußlands'             : u'zentralrusslands',
        u'zerbiß'                      : u'zerbiss',
        u'zerbißt'                     : u'zerbisst',
        u'zerfallsprozeß'              : u'zerfallsprozess',
        u'zerfloß'                     : u'zerfloss',
        u'zerfloßt'                    : u'zerflosst',
        u'zerfreßt'                    : u'zerfresst',
        u'zerfriß'                     : u'zerfriss',
        u'zerfrißt'                    : u'zerfrisst',
        u'zerlaß'                      : u'zerlass',
        u'zerlaßt'                     : u'zerlasst',
        u'zerläßt'                     : u'zerlässt',
        u'zerriß'                      : u'zerriss',
        u'zerrißt'                     : u'zerrisst',
        u'zerschliß'                   : u'zerschliss',
        u'zerschlißt'                  : u'zerschlisst',
        u'zerschmiß'                   : u'zerschmiss',
        u'zerschmißt'                  : u'zerschmisst',
        u'zerschoß'                    : u'zerschoss',
        u'zerschoßt'                   : u'zerschosst',
        u'zersetzungsprozeß'           : u'zersetzungsprozess',
        u'zielbewußt'                  : u'zielbewusst',
        u'zielbewußte'                 : u'zielbewusste',
        u'zielbewußtem'                : u'zielbewusstem',
        u'zielbewußten'                : u'zielbewussten',
        u'zielbewußter'                : u'zielbewusster',
        u'zielbewußtere'               : u'zielbewusstere',
        u'zielbewußterem'              : u'zielbewussterem',
        u'zielbewußteren'              : u'zielbewussteren',
        u'zielbewußterer'              : u'zielbewussterer',
        u'zielbewußteres'              : u'zielbewussteres',
        u'zielbewußtes'                : u'zielbewusstes',
        u'zielbewußteste'              : u'zielbewussteste',
        u'zielbewußtestem'             : u'zielbewusstestem',
        u'zielbewußtesten'             : u'zielbewusstesten',
        u'zielbewußtester'             : u'zielbewusstester',
        u'zielbewußtestes'             : u'zielbewusstestes',
        u'zielbewußtheit'              : u'zielbewusstheit',
        u'zielphoto'                   : u'zielfoto',
        u'zielphotos'                  : u'zielfotos',
        u'zierat'                      : u'zierrat',
        u'zieraten'                    : u'zierraten',
        u'zippverschluß'               : u'zippverschluss',
        u'zirbelnuß'                   : u'zirbelnuss',
        u'zirkelschluß'                : u'zirkelschluss',
        u'zivilprozeß'                 : u'zivilprozess',
        u'zivilprozeßordnung'          : u'zivilprozessordnung',
        u'zivilprozeßrecht'            : u'zivilprozessrecht',
        u'zollang'                     : u'zolllang',
        u'zollinie'                    : u'zolllinie',
        u'zubiß'                       : u'zubiss',
        u'zubißt'                      : u'zubisst',
        u'zuckerguß'                   : u'zuckerguss',
        u'zuckerpaß'                   : u'zuckerpass',
        u'zufloß'                      : u'zufloss',
        u'zufloßt'                     : u'zuflosst',
        u'zufluß'                      : u'zufluss',
        u'zuflußgebiet'                : u'zuflussgebiet',
        u'zugepaßt'                    : u'zugepasst',
        u'zugepreßt'                   : u'zugepresst',
        u'zugschlußsignal'             : u'zugschlusssignal',
        u'zugschlußsignale'            : u'zugschlusssignale',
        u'zugschlußsignalen'           : u'zugschlusssignalen',
        u'zugschlußsignals'            : u'zugschlusssignals',
        u'zulaßt'                      : u'zulasst',
        u'zuläßt'                      : u'zulässt',
        u'zungenkuß'                   : u'zungenkuss',
        u'zupaßgekommen'               : u'zupassgekommen',
        u'zupaßkam'                    : u'zupasskam',
        u'zupaßkamen'                  : u'zupasskamen',
        u'zupaßkamst'                  : u'zupasskamst',
        u'zupaßkamt'                   : u'zupasskamt',
        u'zupaßkomme'                  : u'zupasskomme',
        u'zupaßkommend'                : u'zupasskommend',
        u'zupaßkommest'                : u'zupasskommest',
        u'zupaßkommet'                 : u'zupasskommet',
        u'zupaßkommst'                 : u'zupasskommst',
        u'zupaßkommt'                  : u'zupasskommt',
        u'zupaßkäme'                   : u'zupasskäme',
        u'zupaßkämen'                  : u'zupasskämen',
        u'zupaßkämest'                 : u'zupasskämest',
        u'zupaßkämet'                  : u'zupasskämet',
        u'zupaßkämst'                  : u'zupasskämst',
        u'zupaßkämt'                   : u'zupasskämt',
        u'zupaßzukommen'               : u'zupasszukommen',
        u'zurückgemußt'                : u'zurückgemusst',
        u'zusammenfaßt'                : u'zusammenfasst',
        u'zusammenfaßte'               : u'zusammenfasste',
        u'zusammenfaßten'              : u'zusammenfassten',
        u'zusammenfaßtest'             : u'zusammenfasstest',
        u'zusammenfaßtet'              : u'zusammenfasstet',
        u'zusammenfloß'                : u'zusammenfloss',
        u'zusammenfloßt'               : u'zusammenflosst',
        u'zusammenfluß'                : u'zusammenfluss',
        u'zusammengefaßt'              : u'zusammengefasst',
        u'zusammengefaßte'             : u'zusammengefasste',
        u'zusammengefaßtem'            : u'zusammengefasstem',
        u'zusammengefaßten'            : u'zusammengefassten',
        u'zusammengefaßter'            : u'zusammengefasster',
        u'zusammengefaßtes'            : u'zusammengefasstes',
        u'zusammengepaßt'              : u'zusammengepasst',
        u'zusammengepreßt'             : u'zusammengepresst',
        u'zusammenpreßt'               : u'zusammenpresst',
        u'zusammenpreßte'              : u'zusammenpresste',
        u'zusammenpreßten'             : u'zusammenpressten',
        u'zusammenpreßtest'            : u'zusammenpresstest',
        u'zusammenpreßtet'             : u'zusammenpresstet',
        u'zusammenriß'                 : u'zusammenriss',
        u'zusammenrißt'                : u'zusammenrisst',
        u'zusammenschloß'              : u'zusammenschloss',
        u'zusammenschloßt'             : u'zusammenschlosst',
        u'zusammenschluß'              : u'zusammenschluss',
        u'zuschloß'                    : u'zuschloss',
        u'zuschloßt'                   : u'zuschlosst',
        u'zuschoß'                     : u'zuschoss',
        u'zuschoßt'                    : u'zuschosst',
        u'zuschuß'                     : u'zuschuss',
        u'zuschußantrag'               : u'zuschussantrag',
        u'zuschußantrage'              : u'zuschussantrage',
        u'zuschußantrages'             : u'zuschussantrages',
        u'zuschußantrags'              : u'zuschussantrags',
        u'zuschußanträge'              : u'zuschussanträge',
        u'zuschußanträgen'             : u'zuschussanträgen',
        u'zuschußbetrieb'              : u'zuschussbetrieb',
        u'zuschußbogen'                : u'zuschussbogen',
        u'zuschußwirtschaft'           : u'zuschusswirtschaft',
        u'zuthun'                      : u'zutun',
        u'zuthuns'                     : u'zutuns',
        u'zweitkläßler'                : u'zweitklässler',
        u'zweitkläßlerin'              : u'zweitklässlerin',
        u'zweitkläßlerinnen'           : u'zweitklässlerinnen',
        u'zweitkläßlern'               : u'zweitklässlern',
        u'zweitkläßlers'               : u'zweitklässlers',
        u'zweitplaciert'               : u'zweitplatziert',
        u'zweitplaziert'               : u'zweitplatziert',
        u'zweitplazierte'              : u'zweitplatzierte',
        u'zweitplaziertem'             : u'zweitplatziertem',
        u'zweitplazierten'             : u'zweitplatzierten',
        u'zweitplazierter'             : u'zweitplatzierter',
        u'zwey'                        : u'zwei',
        u'zweye'                       : u'zweie',
        u'zweyen'                      : u'zweien',
        u'zweyer'                      : u'zweier',
        u'zwillingsfluß'               : u'zwillingsfluss',
        u'zwischengeschoß'             : u'zwischengeschoss',
        u'zylinderschloß'              : u'zylinderschloss',
        u'zäheit'                      : u'zähheit',
        u'zärter'                      : u'zarter',
        u'zärtesten'                   : u'zartesten',
        u'zündschloß'                  : u'zündschloss',
        u'öldruckkontrollampe'         : u'öldruckkontrolllampe',
        u'öldruckkontrollampen'        : u'öldruckkontrolllampen',
        u'ölfaß'                       : u'ölfass',
        u'ölmeßstab'                   : u'ölmessstab',
        u'ölmeßstabe'                  : u'ölmessstabe',
        u'ölmeßstabes'                 : u'ölmessstabes',
        u'ölmeßstabs'                  : u'ölmessstabs',
        u'ölmeßstäbe'                  : u'ölmessstäbe',
        u'ölmeßstäben'                 : u'ölmessstäben',
        u'ösfaß'                       : u'ösfass',
        u'überbiß'                     : u'überbiss',
        u'überdruß'                    : u'überdruss',
        u'überfluß'                    : u'überfluss',
        u'überflußgesellschaft'        : u'überflussgesellschaft',
        u'übergoß'                     : u'übergoss',
        u'übergoßt'                    : u'übergosst',
        u'überguß'                     : u'überguss',
        u'überlaß'                     : u'überlass',
        u'überlaßt'                    : u'überlasst',
        u'überläßt'                    : u'überlässt',
        u'überriß'                     : u'überriss',
        u'überrißt'                    : u'überrisst',
        u'überschuß'                   : u'überschuss',
        u'überschußproduktion'         : u'überschussproduktion',
        u'überschwenglich'             : u'überschwänglich',
        u'überschwengliche'            : u'überschwängliche',
        u'überschwenglichem'           : u'überschwänglichem',
        u'überschwenglichen'           : u'überschwänglichen',
        u'überschwenglicher'           : u'überschwänglicher',
        u'überschwenglichere'          : u'überschwänglichere',
        u'überschwenglicherem'         : u'überschwänglicherem',
        u'überschwenglicheren'         : u'überschwänglicheren',
        u'überschwenglicherer'         : u'überschwänglicherer',
        u'überschwenglicheres'         : u'überschwänglicheres',
        u'überschwengliches'           : u'überschwängliches',
        u'überschwenglichkeit'         : u'überschwänglichkeit',
        u'überschwenglichkeiten'       : u'überschwänglichkeiten',
        u'überschwenglichste'          : u'überschwänglichste',
        u'überschwenglichstem'         : u'überschwänglichstem',
        u'überschwenglichsten'         : u'überschwänglichsten',
        u'überschwenglichster'         : u'überschwänglichster',
        u'überschwenglichstes'         : u'überschwänglichstes',

        # manual entries


        u'absentirt'           : u'absentiert',
        u'absnitt'             : u'abschnitt',
        u'achtzeen'            : u'achtzehn',
        u'actenstöße'          : u'aktenstöße',
        u'advocat'             : u'advokat',
        u'aehnlichkeit'        : u'ähnlichkeit',
        u'aendern'             : u'ändern',
        u'aepfel'              : u'äpfel',
        u'aequator'            : u'äquator',
        u'aerztin'             : u'ärztin',
        u'aethertraum'         : u'äthertraum',
        u'aether'              : u'äther',
        u'aeußeren'            : u'äußeren',
        u'aeußeres'            : u'äußeres',
        u'allernothwendigsten' : u'allernotwendigsten',
        u'alterthümliche'      : u'altertümliche',
        u'altonaer'            : u'altonär',
        u'amüsirt'             : u'amüsiert',
        u'angethan'            : u'angetan',
        u'anknüpfungspunct'    : u'anknüpfungspunkt',
        u'athemzug'            : u'atemzug',
        u'aufgethan'           : u'aufgetan',
        u'aufraeumt'           : u'aufräumt',
        u'aufzaehlen'          : u'aufzählen',
        u'begräbniß'           : u'begräbnis',
        u'beschaeftigt'        : u'beschäftigt',
        u'betruege'            : u'betrüge',
        u'bischen'             : u'bischen',
        u'bloedes'             : u'blödes',
        u'bloede'              : u'blöde',
        u'bloedsinn'           : u'blödsinn',
        u'bloed'               : u'blöd',
        u'blutrother'          : u'blutroter',
        u'boese'               : u'böse',
        u'buecher'             : u'bücher',
        u'bueck'               : u'bück',
        u'bündniß'             : u'bündnis',
        u'cameraden'           : u'kameraden',
        u'capuze'              : u'kapuze',
        u'claviergeklimper'    : u'klaviergeklimper',
        u'communications'      : u'kommunikations',
        u'conferenzen'         : u'konferenzen',
        u'console'             : u'konsole',
        u'controliren'         : u'kontrollieren',
        u'convenienzen'        : u'konvenienzen',
        u'curen'               : u'kuren',
        u'daemliche'           : u'dämliche',
        u'daemlich'            : u'dämlich',
        u'darnach'             : u'danach',
        u'darueber'            : u'darüber',
        u'desshalb'            : u'deshalb',
        u'discretion'          : u'diskretion',
        u'documente'           : u'dokumente',
        u'doener'              : u'döner',
        u'duesseldorf'         : u'düsseldorf',
        u'dunkelroth'          : u'dunkelrot',
        u'edinburg'            : u'edinburgh',
        u'ehrenwerth'          : u'ehrenwert',
        u'eigenthümliches'     : u'eigentümliches',
        u'eigenthümliche'      : u'eigentümliche',
        u'eingefaßt'           : u'eingefasst',
        u'einsylbig'           : u'einsilbig',
        u'einundzwanzigte'     : u'einundzwanzigste',
        u'erklaeren'           : u'erklären',
        u'erklaerung'          : u'erklärung',
        u'ermuthigend'         : u'ermutigend',
        u'erröthend'           : u'errötend',
        u'erröthen'            : u'erröten',
        u'erzaehlen'           : u'erzählen',
        u'erzaehle'            : u'erzähle',
        u'erzaehlt'            : u'erzählt',
        u'erzaehl'             : u'erzähl',
        u'eßwaren'             : u'esswaren',
        u'existirte'           : u'existierte',
        u'faellt'              : u'fällt',
        u'faengst'             : u'fängst',
        u'fallthür'            : u'falltür',
        u'farrnkrautsamen'     : u'farnkrautsamen',
        u'farrnkraut'          : u'farnkraut',
        u'fixirend'            : u'fixierend',
        u'fluth'               : u'flut',
        u'fortläßt'            : u'fortlässt',
        u'fortpreßte'          : u'fortpresste',
        u'franzoesisch'        : u'französisch',
        u'freudenthränen'      : u'freudentränen',
        u'froehlich'           : u'fröhlich',
        u'fuehlen'             : u'fühlen',
        u'fuehle'              : u'fühle',
        u'fuehlst'             : u'fühlst',
        u'fuehlt'              : u'fühlt',
        u'fuenf'               : u'fünf',
        u'fuer'                : u'für',
        u'fuessen'             : u'füssen',
        u'gartenthür'          : u'gartentür',
        u'gefaellt'            : u'gefällt',
        u'gefuehle'            : u'gefühle',
        u'gefuehl'             : u'gefühl',
        u'gegenueber'          : u'gegenüber',
        u'geheimnißvolle'      : u'geheimnisvolle',
        u'gehoert'             : u'gehört',
        u'gemüthliches'        : u'gemütliches',
        u'gemüthlichste'       : u'gemütlichste',
        u'geradeso'            : u'geradeso',
        u'gerathend'           : u'geratend',
        u'geräth'              : u'gerät',
        u'geruecht'            : u'gerücht',
        u'gespraechig'         : u'gesprächig',
        u'gespraech'           : u'gespräch',
        u'gewirthschaftet'     : u'gewirtschaftet',
        u'gitterthür'          : u'gittertür',
        u'glasthür'            : u'glastür',
        u'gluecklich'          : u'glücklich',
        u'glutherfüllte'       : u'gluterfüllte',
        u'gluthgefüllten'      : u'glutgefüllten',
        u'gluth'               : u'glut',
        u'gruenen'             : u'grünen',
        u'gruener'             : u'grüner',
        u'gruen'               : u'grün',
        u'gutmüthiges'         : u'gutmütiges',
        u'haeltst'             : u'hältst',
        u'haende'              : u'hände',
        u'haesslicher'         : u'hässlicher',
        u'haesslich'           : u'hässlich',
        u'haettest'            : u'hättest',
        u'haette'              : u'hätte',
        u'hausthür'            : u'haustür',
        u'heiligthum'          : u'heiligtum',
        u'hochrothe'           : u'hochrote',
        u'hoechste'            : u'höchste',
        u'hoeflicher'          : u'höflicher',
        u'hoeflich'            : u'höflich',
        u'hoelle'              : u'hölle',
        u'hoeren'              : u'hören',
        u'hoere'               : u'höre',
        u'hoerst'              : u'hörst',
        u'hoer'                : u'hör',
        u'huebsch'             : u'hübsch',
        u'indeß'               : u'indess',
        u'interessiren'        : u'interessieren',
        u'justizrath'          : u'justizrat',
        u'kluegste'            : u'klügste',
        u'koennen'             : u'können',
        u'koenntest'           : u'könntest',
        u'koennte'             : u'könnte',
        u'koerperlich'         : u'körperlich',
        u'koerper'             : u'körper',
        u'krystallenen'        : u'kristallenen',
        u'kuenstliche'         : u'künstliche',
        u'kuenstlich'          : u'künstlich',
        u'lackirte'            : u'lackierte',
        u'ladenthür'           : u'ladentür',
        u'laecherlich'         : u'lächerlich',
        u'laender'             : u'länder',
        u'laeufst'             : u'läufst',
        u'laeuft'              : u'läuft',
        u'läugnen'             : u'leugnen',
        u'legitimiren'         : u'legitimieren',
        u'legitimirt'          : u'legitimiert',
        u'lernfaehig'          : u'lernfähig',
        u'liebesfaehig'        : u'liebesfähig',
        u'liebwerthestes'      : u'liebwertestes',
        u'loewe'               : u'löwe',
        u'luegen'              : u'lügen',
        u'luegner'             : u'lügner',
        u'luegst'              : u'lügst',
        u'maedchen'            : u'mädchen',
        u'maenner'             : u'männer',
        u'maennlich'           : u'männlich',
        u'maerchen'            : u'märchen',
        u'manuscript'          : u'manuskript',
        u'mitgetheilt'         : u'mitgeteilt',
        u'mittelthür'          : u'mitteltür',
        u'mittheilen'          : u'mitteilen',
        u'mitzutheilen'        : u'mitzuteilen',
        u'moechtest'           : u'möchtest',
        u'moechte'             : u'möchte',
        u'moeglichkeit'        : u'möglichkeit',
        u'moeglich'            : u'möglich',
        u'monroe'              : u'monrö',
        u'muede'               : u'müde',
        u'muelheim'            : u'mülheim',
        u'muelltonne'          : u'mülltonne',
        u'muesstest'           : u'müsstest',
        u'musicirte'           : u'musizierte',
        u'muthigsten'          : u'mutigsten',
        u'muthwillig'          : u'mutwillig',
        u'naechsten'           : u'nächsten',
        u'naechstes'           : u'nächstes',
        u'naechste'            : u'nächste',
        u'narrenthurm'         : u'narrenturm',
        u'natuerlich'          : u'natürlich',
        u'niedergepreßte'      : u'niedergepresste',
        u'noetig'              : u'nötig',
        u'nothwendige'         : u'notwendige',
        u'nothwendigsten'      : u'notwendigsten',
        u'nothwendig'          : u'notwendig',
        u'oeffnung'            : u'öffnung',
        u'oefters'             : u'öfters',
        u'oefter'              : u'öfter',
        u'oelen'               : u'ölen',
        u'officier'            : u'offizier',
        u'pikirt'              : u'pikiert',
        u'polirten'            : u'polierten',
        u'präparirt'           : u'präpariert',
        u'principals'          : u'prinzipals',
        u'processes'           : u'prozesses',
        u'processiren'         : u'prozessieren',
        u'proceßkosten'        : u'prozesskosten',
        u'producirt'           : u'produziert',
        u'protestirt'          : u'protestiert',
        u'protocoll'           : u'protokoll',
        u'publicum'            : u'publikum',
        u'rathhaus'            : u'rathaus',
        u'räthin'              : u'rätin',
        u'rathlos'             : u'ratlos',
        u'regierungsräthin'    : u'regierungsrätin',
        u'relativitaetstheorie': u'relativitätstheorie',
        u'rothgestreiften'     : u'rotgestreiften',
        u'röthlichen'          : u'rötlichen',
        u'saarbruecken'        : u'saarbrücken',
        u'saetze'              : u'sätze',
        u'sammt'               : u'samt',
        u'scalpells'           : u'skalpells',
        u'scenenwechsel'       : u'szenenwechsel',
        u'scepter'             : u'szepter',
        u'schaetze'            : u'schätze',
        u'schlaefst'           : u'schläfst',
        u'schloßplatz'         : u'schlossplatz',
        u'schluesselwoerter'   : u'schluesselwörter',
        u'schluesselwoerter'   : u'schlüsselwörter',
        u'schoenen'            : u'schönen',
        u'schoene'             : u'schöne',
        u'schoen'              : u'schön',
        u'schoepfer'           : u'schöpfer',
        u'schroeder'           : u'schröder',
        u'schuechtern'         : u'schüchtern',
        u'schueler'            : u'schüler',
        u'schuetze'            : u'schütze',
        u'schuldbewußtsein'    : u'schuldbewusstsein',
        u'sebstbezueglich'     : u'sebstbezüglich',
        u'secretair'           : u'sekretär',
        u'secunden'            : u'sekunden',
        u'spaet'               : u'spät',
        u'spruenge'            : u'sprünge',
        u'standpunct'          : u'standpunkt',
        u'stoert'              : u'stört',
        u'stubenthüren'        : u'stubentüren',
        u'studirzimmers'       : u'studierzimmers',
        u'subject'             : u'subjekt',
        u'suess'               : u'süß',
        u'tact'                : u'takt',
        u'taeglich'            : u'täglich',
        u'tapetenthür'         : u'tapetentür',
        u'telegraphisch'       : u'telegrafisch',
        u'thaler'              : u'taler',
        u'theegeschirr'        : u'teegeschirr',
        u'theekanne'           : u'teekanne',
        u'theemaschine'        : u'teemaschine',
        u'theestunde'          : u'teestunde',
        u'thörichte'           : u'törichte',
        u'todtenähnlich'       : u'totenähnlich',
        u'todtenstille'        : u'totenstille',
        u'todten'              : u'toten',
        u'todtgeschlagen'      : u'totgeschlagen',
        u'tödtliche'           : u'tötliche',
        u'todt'                : u'tot',
        u'traegst'             : u'trägst',
        u'traeumen'            : u'träumen',
        u'traeume'             : u'träume',
        u'tschuess'            : u'tschüss',
        u'tuerkisch'           : u'türkisch',
        u'ubersetzt'           : u'übersetzt',
        u'ueberall'            : u'überall',
        u'uebereinkommen'      : u'übereinkommen',
        u'ueberfluß'           : u'überfluss',
        u'uebergang'           : u'übergang',
        u'ueberhaupt'          : u'überhaupt',
        u'ueberlegen'          : u'überlegen',
        u'ueberlieferung'      : u'überlieferung',
        u'uebernatürliches'    : u'übernatürliches',
        u'uebernatürliche'     : u'übernatürliche',
        u'ueberraschte'        : u'überraschte',
        u'ueberraschung'       : u'überraschung',
        u'uebersetzer'         : u'übersetzer',
        u'uebersetzt'          : u'übersetzt',
        u'uebersiedelung'      : u'übersiedelung',
        u'ueberstürzen'        : u'überstürzen',
        u'uebertragung'        : u'übertragung',
        u'uebertreibung'       : u'übertreibung',
        u'ueber'               : u'über',
        u'uebrige'             : u'übrige',
        u'ungefaehr'           : u'ungefähr',
        u'unmoeglich'          : u'unmöglich',
        u'unnöthige'           : u'unnötige',
        u'verhältnißmäßig'     : u'verhältnismäßig',
        u'verhängnißvollen'    : u'verhängnisvollen',
        u'verhoer'             : u'verhör',
        u'vermuthung'          : u'vermutung',
        u'verrieth'            : u'verriet',
        u'verrueckt'           : u'verrückt',
        u'vertheidigte'        : u'verteidigte',
        u'viertheile'          : u'vierteile',
        u'voellig'             : u'völlig',
        u'vorsaalthür'         : u'vorsaaltür',
        u'waehle'              : u'wähle',
        u'waehl'               : u'wähl',
        u'wehmüthigen'         : u'wehmütigen',
        u'wehmuth'             : u'wehmut',
        u'wehmuthvollen'       : u'wehmutvollen',
        u'weitererzaehlen'     : u'weitererzählen',
        u'wirthschaft'         : u'wirtschaft',
        u'woerter'             : u'wörter',
        u'wofuer'              : u'wofür',
        u'wohlthätiges'        : u'wohltätiges',
        u'worueber'            : u'worüber',
        u'wuerdest'            : u'würdest',
        u'wuerde'              : u'würde',
        u'wuesste'             : u'wüsste',
        u'wuetend'             : u'wütend',
        u'zuerich'             : u'zürich',
        u'zuhoeren'            : u'zuhören',
        u'zurueck'             : u'zurück',
        u'zusammengepreßten'   : u'zusammengepressten',

        u'swr3'             : u'swr drei',
        u'zx81'             : u'zx einundachtzig',

        u'§'                : u'paragraph'}


symb_abbrev_norm = [
        (u'Abk.'    , u'abkürzung '),
        (u'Abk '    , u'abkürzung '),
        (u'Prof.'   , u'professor '),
        (u'Dipl.'   , u'diplom '),
        (u'Ing.'    , u'ingenieur '),
        (u'Inf.'    , u'informatiker '),
        (u'Inform.' , u'informatiker '),
        (u'Tel.'    , u'telefon '),
        (u'bzw.'    , u'beziehungsweise '),
        (u'bzw '    , u'beziehungsweise '),
        (u'bspw.'   , u'beispielsweise '),
        (u'bspw '   , u'beispielsweise '),
        (u'bzgl.'   , u'bezüglich '),
        (u'ca.'     , u'circa '),
        (u'd.h.'    , u'das heißt '),
        (u'd. h.'   , u'das heißt '),
        (u'Dr. '    , u'doktor '),
        (u'evtl.'   , u'eventuell '),
        (u'evtl '   , u'eventuell '),
        (u'geb.'    , u'geborene '),
        (u'ggf.'    , u'gegebenenfalls '),
        (u'ggf '    , u'gegebenenfalls '),
        (u'kath.'   , u'katholische '),
        (u'Hrsg.'   , u'herausgeber '),
        (u'Mr.'     , u'mister '),
        (u'Mrd.'    , u'milliarden '),
        (u'Mrs.'    , u'misses '),
        (u'Nr.'     , u'nummer '),
        (u'Nrn.'    , u'nummern '),
        (u'o.k.'    , u'okay '),
        (u'o. k.'   , u'okay '),
        (u's.a.'    , u'siehe auch '),
        (u's. a.'   , u'siehe auch '),
        (u's.o.'    , u'siehe oben '),
        (u's. o.'   , u'siehe oben '),
        (u's.u.'    , u'siehe unten '),
        (u's. u.'   , u'siehe unten '),
        (u'jr.'     , u'junior '),
        (u'Str.'    , u'strasse '),
        (u'u.a.'    , u'unter anderem '),
        (u'u. a.'   , u'unter anderem '),
        (u'u.U.'    , u'unter umständen '),
        (u'u. U.'   , u'unter umständen '),
        (u'usw.'    , u'und so weiter '),
        (u'u.s.w.'  , u'und so weiter '),
        (u'u. s. w.', u'und so weiter '),
        (u'v.a.'    , u'vor allem '),
        (u'vgl.'    , u'vergleiche '),
        (u'vgl '    , u'vergleiche '),
        (u'Wdh.'    , u'wiederholung '),
        (u'Ziff.'   , u'ziffer '),
        (u'z.B.'    , u'zum beispiel '),
        (u'z. B.'   , u'zum beispiel '),
        (u'z.T.'    , u'zum teil '),
        (u'z. T.'   , u'zum teil '),
        (u'z.Zt.'   , u'zur zeit '),
        (u'z. Zt.'  , u'zur zeit '),
        (u'GHz'     , u'gigahertz '),
        (u'\ufeff'  , u' '),
        (u'\u2019'  , u' '),
        (u'\xa0'    , u' '),
        (u'\u203a'  , u' '),
        (u'\u2039'  , u' '),
        (u'&'       , u'und'),
        (u'\xa020'  , u' '),
        (u'„'       , u' '),
        (u'“'       , u' '),
        (u'$'       , u'dollar ')
    ]


# based on code from: http://www.python-forum.de/viewtopic.php?f=11&t=22543

w1 = u"null ein zwei drei vier fünf sechs sieben acht neun zehn elf zwölf dreizehn vierzehn fünfzehn sechzehn siebzehn achtzehn neunzehn".split()
w2 = u"zwanzig dreißig vierzig fünfzig sechzig siebzig achtzig neunzig".split()
 
def zahl_in_worten(n, s=True, z=False, e=False):
    if n < 0: raise ValueError
    if n == 0 and z: return ""
    if n == 1 and s: return "eins"
    if n == 1 and e: return "eine"
    if n < 20: return w1[n]
    if n < 100:
        w = w2[(n - 20) // 10]
        if n % 10:
            w = w1[n % 10] + "und" + w
        return w
    if n < 1000:
        if n // 100 == 1:
            if z: return "einhundert" + zahl_in_worten(n % 100, z=True)
            return "hundert" + zahl_in_worten(n % 100, z=True)
        return w1[n // 100] + "hundert" + zahl_in_worten(n % 100, z=True)
    if n < 2000:
        if n < 1100:
            return  "tausend" + zahl_in_worten(n % 1000, z=True)
        return w1[n // 100] + "hundert" + zahl_in_worten(n % 100, z=True)
    if n < 1000000:
        return zahl_in_worten(n // 1000, s=False) + "tausend" + zahl_in_worten(n % 1000, z=True)
    if n < 1000000000:
        m = n // 1000000
        suff = 'millionen' if m>1 else 'million'
        return zahl_in_worten(m, s=False, e=True) + suff + zahl_in_worten(n % 1000000, z=True)
    # raise ValueError
    logging.warn('zahl_in_worten: cannot handle %s' % n)
    return str(n)

def spellout_number (m):

    numstr = m.group(0)

    # print "spellout number:", numstr

    if numstr[0].isspace():
        numstr = numstr[1:]
        res = ' '
    else:
        res = ''

    if numstr[0] == '-':
        res += 'minus '
        numstr = numstr[1:].strip()

    if not '.' in numstr:
        numstr = numstr.replace(',','.')

    parts = numstr.split ('.')

    # print repr(parts)

    res += zahl_in_worten(int(parts[0]))

    if len(parts)>1 and len(parts[1])>0:

        # spell out fractional part in digits

        res += ' komma'

        for c in parts[1]:
            res += ' ' + zahl_in_worten(int(c))
    
    return res

def tokenize (s, lang='de', keep_punctuation=False, keep_macros=False, keep_underscores=True):

    global wrt

    if lang == 'en':
        return tokenize_en(s, keep_punctuation, keep_macros, keep_underscores)

    if lang == 'fr':
        return tokenize_fr(s, keep_punctuation, keep_macros, keep_underscores)

    if lang != 'de':
        # FIXME
        raise Exception ("FIXME: implement tokenizer support for language: " + lang)

    # print '#1', s

    for san in symb_abbrev_norm:
        srch = san[0]
        repl = san[1]

        s = s.replace (srch, repl)

    # print '#2', s

    # deal with numbers
    s = NUMBER_PATTERN.sub(spellout_number, s)

    # deal with punctuation
    if keep_punctuation:
        for p in PUNCTUATION:
            s = s.replace (p, u' ' + p + u' ')
        if not keep_macros:
            for p in MACRO_PUNCTUATION:
                s = s.replace (p, u' ' + p + u' ')
    else:
        for p in PUNCTUATION:
            s = s.replace(p,' ')
        if not keep_macros:
            for p in MACRO_PUNCTUATION:
                s = s.replace(p,' ')
        if not keep_underscores:
            s = s.replace('_',' ')

    # print '#3', s

    res = []

    words = re.split ('\s+', s)

    # print repr(words)

    for word in words:

        w = word.rstrip().replace(u'–',u'').lower()
        if len(w) > 0:

            if w in wrt:
                w = wrt[w]
        
                words2 = re.split('\s+', w)
                for w2 in words2:
                    res.append(w2)

            else:
                res.append (w)

    return res

def kill_umlauts(s):
    return s.replace(u'ß',u'ss') \
            .replace(u'Ä',u'Ae') \
            .replace(u'Ö',u'Oe') \
            .replace(u'Ü',u'Ue') \
            .replace(u'ä',u'ae') \
            .replace(u'ö',u'oe') \
            .replace(u'ü',u'ue') 

