# frozen_string_literal: true

require_relative 'runner/strategies/captured'
require_relative 'runner/strategies/passthrough'

module Reviewer
  # Wrapper for executng a command and printing the results
  class Runner
    extend Forwardable

    attr_accessor :strategy

    attr_reader :command, :shell, :output

    def_delegators :@command, :tool
    def_delegators :@shell, :result, :timer
    def_delegators :result, :exit_status, :stdout, :stderr, :rerunnable?

    # Creates a wrapper for running commansd through Reviewer in order to provide a more accessible
    #   API for recording execution time and interpreting the results of a command in a more
    #   generous way so that non-zero exit statuses can still potentiall be passing.
    # @param tool [Symbol] the key for the desired tool to run
    # @param command_type [Symbol] the key for the type of command to run
    # @param strategy = Strategies::Captured [Runner::Strategies] how to execute and handle the
    #   results of the command
    # @param output: Reviewer.output [Review::Output] the output formatter for the results
    #
    # @return [self]
    def initialize(tool, command_type, strategy = Strategies::Captured, output: Reviewer.output)
      @command = Command.new(tool, command_type)
      @strategy = strategy
      @shell = Shell.new
      @output = output
    end

    def run
      # Show which tool is running
      identify_tool

      # Use the provided strategy to run the command
      execute_strategy

      # If it failed, display guidance to help them get back on track
      guidance.show unless success?

      # Return the exit status generated by the tool as interpreted by the Result
      exit_status
    end

    def success?
      # Some review tools return a range of non-zero exit statuses and almost never return 0.
      # (`yarn audit` is a good example.) Those tools can be configured to accept a non-zero exit
      # status so they aren't constantly considered to be failing over minor issues.
      #
      # But when other command types (prepare, install, format) are run, they either succeed or they
      # fail. With no shades of gray in those cases, anything other than a 0 is a failure.
      if command.type == :review
        exit_status <= tool.max_exit_status
      else
        exit_status.zero?
      end
    end

    # Prints the tool name and description to the console as a frame of reference
    #
    # @return [void]
    def identify_tool
      # If there's an existing result, the runner is being re-run, and identifying the tool would
      # be redundant.
      return if result.exists?

      output.tool_summary(tool)
    end

    # Runs the relevant strategy to either capture or pass through command output.
    #
    # @return [void]
    def execute_strategy
      # Run the provided strategy
      strategy.new(self).tap do |run_strategy|
        run_strategy.prepare if run_prepare_step?
        run_strategy.run
      end
    end

    # Determines whether a preparation step should be run before the primary command. If/when the
    #   primary command is a `:prepare` command, then it shouldn't run twice. So it skips what would
    #   be a superfluous run of the preparation.
    #
    # @return [Boolean] true the primary command is not prepare and the tool needs to be prepare
    def run_prepare_step?
      command.type != :prepare && tool.prepare?
    end

    # Creates_an instance of the prepare command for a tool
    #
    # @return [Comman] the current tool's prepare command
    def prepare_command
      @prepare_command ||= Command.new(tool, :prepare)
    end

    # Updates the 'last prepared at' timestamp that Reviewer uses to know if a tool's preparation
    #   step is stale and needs to be run again.
    #
    # @return [Time] the timestamp `last_prepared_at` is updated to
    def update_last_prepared_at
      # Touch the `last_prepared_at` timestamp for the tool so it waits before running again.
      tool.last_prepared_at = Time.now
    end

    # Saves the last 5 elapsed times for the commands used this run by using the raw command as a
    #   unique key. This enables the ability to compare times across runs while taking into
    #   consideration that different iterations of the command may be running on fewer files. So
    #   comparing a full run to the average time for a partial run wouldn't be helpful. By using the
    #   raw command string, it will always be apples to apples.
    #
    # @return [void]
    def record_timing
      tool.record_timing(prepare_command, timer.prep)
      tool.record_timing(command, timer.main)
    end

    # Uses the result of the runner to determine what, if any, guidance to display to help the user
    #   get back on track in the event of an unsuccessful run.
    #
    # @return [Guidance] the relevant guidance based on the result of the runner
    def guidance
      @guidance ||= Reviewer::Guidance.new(command: command, result: result, output: output)
    end
  end
end
