import { Application } from '../modules/application/index'
import { Organization } from '../modules/organization/index';
import { User } from '../modules/user/index';

import {
  config_T,
  config_F,
  createApplicationPayloadForUser_T,
  createApplicationPayloadForUser_F,
  createApplicationPayloadForOrg_T,
  createApplicationPayloadForOrg_F,
  getApplicationPayload_F,
  getApplicationPayload_T,
  updateApplicationPayload_T,
  createAPIKeyPayloadForApplication,
  createAPIKeyPayloadForUser,
  createAPIKeyPayloadForOrg,
  getAPIKeyInfoPayloadForApplication_T,
  getAPIKeyInfoPayloadForApplication_F,
  getAPIKeyInfoPayloadForUser_T,
  getAPIKeyInfoPayloadForUser_F,
  getAPIKeyInfoPayloadForOrganization_T,
  getAPIKeyInfoPayloadForOrganization_F,
  updateAPIKeyUserPayloadForApplication_T,
  updateAPIKeyUserPayloadForApplication_F,
  updateAPIKeyUserPayloadForUser_T,
  updateAPIKeyUserPayloadForUser_F,
  updateAPIKeyUserPayloadForOrg_F,
  updateAPIKeyUserPayloadForOrg_T,
  getCollaboratorInfoPayloadForApplication_User_T,
  getCollaboratorInfoPayloadForApplication_User_F,
  getCollaboratorInfoPayloadForApplication_Org_T,
  getCollaboratorInfoPayloadForApplication_Org_F,
  getCollaboratorInfoPayloadForOrg_T,
  getCollaboratorInfoPayloadForOrg_F,
  setCollaboratorPayloadForApplication_User_T,
  setCollaboratorPayloadForApplication_User_F,
  setCollaboratorPayloadForApplication_Org_T,
  setCollaboratorPayloadForApplication_Org_F,
  setCollaboratorPayloadForOrg_T,
  setCollaboratorPayloadForOrg_F,
  getCollaboratorListUserPayload_T,
  getCollaboratorListUserPayload_F,
  getApplicationListPayload_T,
  getApplicationListPayload_F,
  getAPIKeyList_T,
  getAPIKeyList_F
} from './data/appAPI-data';

const app = new Application('meet69', config_T);
const user = new User('meetsavaj', config_T);
const org = new Organization('organ1', config_T);

describe('No Config Errors', () => {
  test('Correct User ID', async () => {
    const appFailed = new User('meetsavaj', config_T);
    expect(appFailed).toHaveProperty('USER_ID', 'meetsavaj');
  });

  test('API Key Given', async () => {
    const appFailed = new User('meetsavaj', config_T);
    expect(appFailed).toHaveProperty(
      'API_KEY',
      process.env.API_KEY
    );
  });
});

describe('Config Errors', () => {
  test('User ID', async () => {
    const appFailed = new User('meetsavj', config_F);
    expect(appFailed).not.toHaveProperty('USER_ID', 'meetsavaj');
  });

  test('API Key Missing', async () => {
    const appFailed = new User('meetsavaj', config_F);
    expect(appFailed).not.toHaveProperty(
      'API_KEY',
      process.env.API_KEY
    );
  });
});

describe('Application Creation (User)', () => {
  test('Application Created Successfully', async () => {
    const response = await user.createApplication(createApplicationPayloadForUser_T);
    expect(response.ids.application_id).toBe(createApplicationPayloadForUser_T.application_id);
    expect(response.name).toBe(createApplicationPayloadForUser_T.name);
    expect(response.administrative_contact.user_ids.user_id).toBe('meetsavaj');
    expect(response.technical_contact.user_ids.user_id).toBe('meetsavaj');
  });
});

describe('Application (User) Creation Errors', () => {
  test('Error, Already Created', async () => {
    try {
      await user.createApplication(createApplicationPayloadForUser_F);
    } catch (error: any) {
      expect(error).toHaveProperty('code', 6);
    }
  });
});

describe('Application Creation (Org)', () => {
  test('Application (Org) Created Successfully', async () => {
    const response = await org.createApplication(createApplicationPayloadForOrg_T);
    expect(response.ids.application_id).toBe(createApplicationPayloadForOrg_T.application_id);
    expect(response.name).toBe(createApplicationPayloadForOrg_T.name);
    expect(response.administrative_contact.organization_ids.organization_id).toBe('organ1');
    expect(response.technical_contact.organization_ids.organization_id).toBe('organ1');
  });
});

describe('Application (Org) Creation Errors', () => {
  test('Error, Already Created', async () => {
    try {
      await org.createApplication(createApplicationPayloadForOrg_F);
    } catch (error: any) {
      expect(error).toHaveProperty('code', 6);
    }
  });
});

describe('Get an Application by ID', () => {
  test('Get An Application Successfully', async () => {
    const response = await app.getApplicationInfo();
    expect(response.ids?.application_id).toBe(getApplicationPayload_T.application_id);
  });
});

describe('Get Application Errors', () => {
  test('Application ID Not Matching', async () => {
    const response = await app.getApplicationInfo();
    expect(response.ids?.application_id).not.toBe(getApplicationPayload_F.application_id);
  });
});

describe('List Applications', () => {
  test('Getting Application List Successfully', async () => {
    const response = await app.getApplicationList(getApplicationListPayload_T);
    expect(response).toBeDefined();
  });
});

describe('List Applications Errors', () => {
  test('Invalid Authorization', async () => {
    try {
      const app = new Application('meet165656', config_F);
      const response = await app.getApplicationList(getApplicationListPayload_T);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 16);
    }
  });

  test('Invalid field values', async () => {
    try {
      const response = await app.getApplicationList(getApplicationListPayload_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 3);
    }
  });
});

describe('List Applications (User)', () => {
  test('Getting Application (User) List Successfully', async () => {
    const response = await user.getApplicationList(getApplicationListPayload_T);
    expect(response).toBeDefined();
  });
});

describe('List Applications (User) Errors', () => {
  test('No Rights', async () => {
    try {
      const user = new User('meetsa', config_T);
      const response = await user.getApplicationList(getApplicationListPayload_T);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });

  test('Invalid field values', async () => {
    try {
      const response = await user.getApplicationList(getApplicationListPayload_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 3);
    }
  });
});

describe('List Applications (Org)', () => {
  test('Getting Application (Org) List Successfully', async () => {
    const response = await org.getApplicationList(getApplicationListPayload_T);
    expect(response).toBeDefined();
  });
});

describe('List Applications (Org) Errors', () => {
  test('No Rights', async () => {
    try {
      const org = new Organization('orga1', config_T);
      const response = await org.getApplicationList(getApplicationListPayload_T);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });

  test('Invalid field values', async () => {
    try {
      const response = await org.getApplicationList(getApplicationListPayload_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 3);
    }
  });
});

describe('Update Application', () => {
  test('Application Updated Successfully', async () => {
    const response = await app.updateApplication(updateApplicationPayload_T);
    expect(response.ids?.application_id).toBe('meet69');
    expect(response.name).toBe(updateApplicationPayload_T.name);
    expect(response.attributes).toEqual(updateApplicationPayload_T.attributes);
  });
});

describe('Delete Application', () => {
  test('Application Deleted Successfully', async () => {
    const app = new Application('meet69', config_T);
    const response = await app.deleteApplication();
    expect(response).not.toBeNull;
  });
});

describe('Delete Application Errors', () => {
  test('No Rights', async () => {
    try {
      const app = new Application('meet165656', config_T);
      const response = await app.deleteApplication();
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });

  test('Application Has Devices', async () => {
    try {
      const app = new Application('meet68', config_T);
      const response = await app.deleteApplication();
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 9);
    }
  });
});

describe('Restore Application', () => {
  test('Application Restored Successfully', async () => {
    const app = new Application('meet69', config_T);
    const response = await app.restoreApplication();
    expect(response).not.toBeNull;
  });
});

describe('Restore Application Errors', () => {
  test('Application ID Already Present', async () => {
    try {
      const app = new Application('meet69', config_T);
      await app.restoreApplication();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('Application ID Not Found', async () => {
    try {
      const app = new Application('meet16565df6', config_T);
      await app.restoreApplication();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Purge Application', () => {
  test('Purge Application Successfully', async () => {
    const app = new Application('meet69', config_T);
    const response = await app.purgeApplication();
    expect(response).not.toBeNull;
  });
});

describe('Purge Application Errors', () => {
  test('Applications May Only Be Purged By Admins', async () => {
    try {
      const app = new Application('meet69', config_T);
      await app.purgeApplication();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });

  test('No Rights', async () => {
    try {
      const app = new Application('meet165656', config_T);
      const response = await app.purgeApplication();
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Search Application', () => {
  test('Searching Applications Successfully', async () => {
    const response = await app.searchApplication();
    expect(response).toBeDefined();
  });
});

describe('List Application Rights', () => {
  test('Listing Applications Rights Successfully', async () => {
    const response = await app.getApplicationRight();
    expect(response).not.toEqual({});
  });
});

describe('List Application Rights Errors', () => {
  test('ID Not Matching', async () => {
    const app = new Application('meet165656', config_T);
    const response = await app.getApplicationRight();
    expect(response).toEqual({});
  });
});

describe('API Key Creation', () => {
  test('API Key Created Successfully', async () => {
    const response = await app.createAPIKey(createAPIKeyPayloadForApplication);
    expect(response.name).toBe(createAPIKeyPayloadForApplication.name);
    expect(response.rights).toEqual(createAPIKeyPayloadForApplication.rights);
    expect(response).not.toHaveProperty('code', 7);
  });
});

describe('API Key Creation', () => {
  test('API Key Created Successfully', async () => {
    const response = await user.createAPIKey(createAPIKeyPayloadForUser);
    expect(response.name).toBe(createAPIKeyPayloadForUser.name);
    expect(response.rights).toEqual(createAPIKeyPayloadForUser.rights);
    expect(response).not.toHaveProperty('code', 7);
  });
});

describe('API Key Creation', () => {
  test('API Key Created Successfully', async () => {
    const response = await org.createAPIKey(createAPIKeyPayloadForOrg);
    expect(response.name).toBe(createAPIKeyPayloadForOrg.name);
    expect(response.rights).toEqual(createAPIKeyPayloadForOrg.rights);
    expect(response).not.toHaveProperty('code', 7);
  });
});

describe('List API Keys (Application)', () => {
  test('Listing API Keys Successfully', async () => {
    const response = await app.getAPIKeyList(getAPIKeyList_T);
    expect(response).toBeDefined();
  });
});

describe('List API Keys Errors (Application)', () => {
  test('No Rights', async () => {
    try {
      const app = new Application('meet165656', config_T);
      const response = await app.getAPIKeyList(getAPIKeyList_T);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });

  test('Invalid field values', async () => {
    try {
      const response = await app.getAPIKeyList(getAPIKeyList_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 3);
    }
  });
});

describe('List API Keys (User)', () => {
  test('Listing API Keys Successfully', async () => {
    const response = await user.getAPIKeyList(getAPIKeyList_T);
    expect(response).toBeDefined();
  });
});

describe('List API Keys Errors (User)', () => {
  test('No Rights', async () => {
    try {
      const user = new User('meetsava', config_T);
      const response = await user.getAPIKeyList(getAPIKeyList_T);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });

  test('Invalid field values', async () => {
    try {
      const response = await user.getAPIKeyList(getAPIKeyList_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 3);
    }
  });
});

describe('List API Keys (Org)', () => {
  test('Listing API Keys Successfully', async () => {
    const response = await org.getAPIKeyList(getAPIKeyList_T);
    expect(response).toBeDefined();
  });
});

describe('List API Keys Errors (Org)', () => {
  test('No Rights', async () => {
    try {
      const org = new Organization('orga1', config_T);
      const response = await org.getAPIKeyList(getAPIKeyList_T);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });

  test('Invalid field values', async () => {
    try {
      const response = await org.getAPIKeyList(getAPIKeyList_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 3);
    }
  });
});

describe('Get API Key (Application)', () => {
  test('Getting API Key Successfully', async () => {
    const response = await app.getAPIKeyInfo(getAPIKeyInfoPayloadForApplication_T);
    expect(response.id).toBe(getAPIKeyInfoPayloadForApplication_T.key_id);
  });
});

describe('Get API Key Errors (Application)', () => {
  test('Api Key With ID Not Found', async () => {
    try {
      const response = await app.getAPIKeyInfo(getAPIKeyInfoPayloadForApplication_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const app = new Application('jgyua', config_T);
      const response = await app.getAPIKeyInfo(getAPIKeyInfoPayloadForApplication_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Get API Key (User)', () => {
  test('Getting API Key Successfully', async () => {
    const response = await user.getAPIKeyInfo(getAPIKeyInfoPayloadForUser_T);
    expect(response.id).toBe(getAPIKeyInfoPayloadForUser_T.key_id);
  });
});

describe('Get API Key Errors (User)', () => {
  test('Api Key With ID Not Found', async () => {
    try {
      const response = await user.getAPIKeyInfo(getAPIKeyInfoPayloadForUser_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const user = new Application('jgyua', config_T);
      const response = await user.getAPIKeyInfo(getAPIKeyInfoPayloadForUser_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Get API Key (Org)', () => {
  test('Getting API Key Successfully', async () => {
    const response = await org.getAPIKeyInfo(getAPIKeyInfoPayloadForOrganization_T);
    expect(response.id).toBe(getAPIKeyInfoPayloadForOrganization_T.key_id);
  });
});

describe('Get API Key Errors (Org)', () => {
  test('Api Key With ID Not Found', async () => {
    try {
      const response = await org.getAPIKeyInfo(getAPIKeyInfoPayloadForOrganization_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const org = new Organization('jgyua', config_T);
      const response = await org.getAPIKeyInfo(getAPIKeyInfoPayloadForOrganization_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('API Key Update (Application)', () => {
  test('API Key Updated Successfully', async () => {
    const response = await app.updateAPIKey(updateAPIKeyUserPayloadForApplication_T);
    expect(response.id).toBe(updateAPIKeyUserPayloadForApplication_T.api_key_id);
    expect(response.name).toBe(updateAPIKeyUserPayloadForApplication_T.api_key_name);
    expect(response.rights).toEqual(updateAPIKeyUserPayloadForApplication_T.api_key_rights);
  });
});

describe('API Key Update Errors (Application)', () => {
  test('Api Key With ID Not Found', async () => {
    try {
      const response = await app.updateAPIKey(updateAPIKeyUserPayloadForApplication_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const app = new Application('jgyua', config_T);
      const response = await app.updateAPIKey(updateAPIKeyUserPayloadForApplication_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('API Key Update (User)', () => {
  test('API Key Updated Successfully', async () => {
    const response = await user.updateAPIKey(updateAPIKeyUserPayloadForUser_T);
    expect(response.id).toBe(updateAPIKeyUserPayloadForUser_T.api_key_id);
    expect(response.name).toBe(updateAPIKeyUserPayloadForUser_T.api_key_name);
    expect(response.rights).toEqual(updateAPIKeyUserPayloadForUser_T.api_key_rights);
  });
});

describe('API Key Update Errors (User)', () => {
  test('Api Key With ID Not Found', async () => {
    try {
      const response = await user.updateAPIKey(updateAPIKeyUserPayloadForUser_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const user = new User('jgyua', config_T);
      const response = await user.updateAPIKey(updateAPIKeyUserPayloadForUser_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('API Key Update (Org)', () => {
  test('API Key Updated Successfully', async () => {
    const response = await org.updateAPIKey(updateAPIKeyUserPayloadForOrg_T);
    expect(response.id).toBe(updateAPIKeyUserPayloadForOrg_T.api_key_id);
    expect(response.name).toBe(updateAPIKeyUserPayloadForOrg_T.api_key_name);
    expect(response.rights).toEqual(updateAPIKeyUserPayloadForOrg_T.api_key_rights);
  });
});

describe('API Key Update Errors (Org)', () => {
  test('Api Key With ID Not Found', async () => {
    try {
      const response = await org.updateAPIKey(updateAPIKeyUserPayloadForOrg_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const org = new Organization('jgyua', config_T);
      const response = await org.updateAPIKey(updateAPIKeyUserPayloadForOrg_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Get Collaborator (App-User)', () => {
  test('Getting Collaborator Successfully', async () => {
    const response = await app.getCollaboratorInfoOfUser(
      getCollaboratorInfoPayloadForApplication_User_T
    );
    expect(response.ids.user_ids.user_id).toBe(
      getCollaboratorInfoPayloadForApplication_User_T.user_id
    );
  });
});

describe('Get Collaborator (App-User) Errors', () => {
  test('User Account Not Found', async () => {
    try {
      const response = await app.getCollaboratorInfoOfUser(
        getCollaboratorInfoPayloadForApplication_User_F
      );
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const app = new Application('jgyua', config_T);
      const response = await app.getCollaboratorInfoOfUser(
        getCollaboratorInfoPayloadForApplication_User_F
      );
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Get Collaborator (App-Org)', () => {
  test('Getting Collaborator Successfully', async () => {
    const response = await app.getCollaboratorInfoOfOrg(
      getCollaboratorInfoPayloadForApplication_Org_T
    );
    expect(response.ids.organization_ids.organization_id).toBe(
      getCollaboratorInfoPayloadForApplication_Org_T.organization_id
    );
  });
});

describe('Get Collaborator (App-Org) Errors', () => {
  test('User Account Not Found', async () => {
    try {
      const response = await app.getCollaboratorInfoOfOrg(
        getCollaboratorInfoPayloadForApplication_Org_F
      );
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const app = new Application('jgyua', config_T);
      const response = await app.getCollaboratorInfoOfOrg(
        getCollaboratorInfoPayloadForApplication_Org_F
      );
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Get Collaborator (Org)', () => {
  test('Getting Collaborator Successfully', async () => {
    const response = await org.getCollaboratorInfo(getCollaboratorInfoPayloadForOrg_T);
    expect(response.ids.user_ids.user_id).toBe(getCollaboratorInfoPayloadForOrg_T.user_id);
  });
});

describe('Get Collaborator (Org) Errors', () => {
  test('User Account Not Found', async () => {
    try {
      const response = await org.getCollaboratorInfo(getCollaboratorInfoPayloadForOrg_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights', async () => {
    try {
      const org = new Organization('jgyua', config_T);
      const response = await org.getCollaboratorInfo(getCollaboratorInfoPayloadForOrg_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Set Collaborator (App-User)', () => {
  test('Set Collaborator Successfully', async () => {
    const app = new Application('meet69', config_T);
    const response = await app.setCollaboratorOfUser(setCollaboratorPayloadForApplication_User_T);
    expect(response).toBeDefined();
  });
});

describe('Set Collaborator Errors (App-User)', () => {
  test('User Account Not Found', async () => {
    try {
      const response = await app.setCollaboratorOfUser(setCollaboratorPayloadForApplication_User_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights or Insufficient Rights', async () => {
    try {
      const app = new Application('jgyua', config_T);
      const response = await app.setCollaboratorOfUser(setCollaboratorPayloadForApplication_User_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Set Collaborator (App-Org)', () => {
  test('Set Collaborator Successfully', async () => {
    const app = new Application('meet69', config_T);
    const response = await app.setCollaboratorOfOrg(setCollaboratorPayloadForApplication_Org_T);
    expect(response).toBeDefined();
  });
});

describe('Set Collaborator Errors (App-Org)', () => {
  test('User Account Not Found', async () => {
    try {
      const response = await app.setCollaboratorOfOrg(setCollaboratorPayloadForApplication_Org_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights or Insufficient Rights', async () => {
    try {
      const app = new Application('jgyua', config_T);
      const response = await app.setCollaboratorOfOrg(setCollaboratorPayloadForApplication_Org_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('Set Collaborator (Org)', () => {
  test('Set Collaborator Successfully', async () => {
    const org = new Organization('organ2', config_T);
    const response = await org.setCollaborator(setCollaboratorPayloadForOrg_T);
    expect(response).toBeDefined();
  });
});

describe('Set Collaborator Errors (Org)', () => {
  test('Org Account Not Found', async () => {
    try {
      const response = await org.setCollaborator(setCollaboratorPayloadForOrg_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('No Rights or Insufficient Rights', async () => {
    try {
      const org = new Organization('jgyua', config_T);
      const response = await org.setCollaborator(setCollaboratorPayloadForOrg_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 7);
    }
  });
});

describe('List Collaborators', () => {
  test('Listing Collaborators Successfully', async () => {
    const response = await app.getCollaboratorList(getCollaboratorListUserPayload_T);
    expect(response).toBeDefined();
  });
});

describe('List Collaborators Errors', () => {
  test('Application Entity Not Found', async () => {
    try {
      const app = new Application('bdvkjh', config_T);
      const response = await app.getCollaboratorList(getCollaboratorListUserPayload_T);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 5);
    }
  });

  test('Invalid field values', async () => {
    try {
      const response = await app.getCollaboratorList(getCollaboratorListUserPayload_F);
      expect(response).toBeUndefined();
    } catch (error: any) {
      expect(error).toHaveProperty('code', 3);
    }
  });
});
