<?php

declare(strict_types=1);

namespace Fp\Streams;

use Fp\Collections\Map;
use Fp\Functional\Option\Option;
use Fp\Operations\FoldOperation;

/**
 * @template-covariant TV
 *
 * @psalm-suppress InvalidTemplateParam
 */
interface StreamTerminalOps
{
    /**
     * Returns true if every stream element satisfy the condition
     * and false otherwise
     *
     * ```php
     * >>> Stream::emits([1, 2])->every(fn($elem) => $elem > 0);
     * => true
     *
     * >>> Stream::emits([1, 2])->every(fn($elem) => $elem > 1);
     * => false
     * ```
     *
     * @param callable(TV): bool $predicate
     */
    public function every(callable $predicate): bool;

    /**
     * Same as {@see StreamTerminalOps::every()}, but deconstruct input tuple and pass it to the $predicate function.
     *
     * @param callable(mixed...): bool $predicate
     */
    public function everyN(callable $predicate): bool;

    /**
     * Find if there is element which satisfies the condition
     *
     * ```php
     * >>> Stream::emits([1, 2])->exists(fn($elem) => 2 === $elem);
     * => true
     *
     * >>> Stream::emits([1, 2])->exists(fn($elem) => 3 === $elem);
     * => false
     * ```
     *
     * @param callable(TV): bool $predicate
     */
    public function exists(callable $predicate): bool;

    /**
     * Same as {@see StreamTerminalOps::exists()}, but deconstruct input tuple and pass it to the $predicate function.
     *
     * @param callable(mixed...): bool $predicate
     */
    public function existsN(callable $predicate): bool;

    /**
     * Produces a new Map of elements by assigning the values to keys generated by a transformation function (callback).
     *
     * ```php
     * >>> Stream::emits([1, 2, 3])->reindex(fn($v) => "key-{$v}");
     * => HashMap('key-1' -> 1, 'key-2' -> 2, 'key-3' -> 3)
     * ```
     *
     * @template TKO
     *
     * @param callable(TV): TKO $callback
     * @return Map<TKO, TV>
     */
    public function reindex(callable $callback): Map;

    /**
     * Same as {@see StreamTerminalOps::reindex()}, but deconstruct input tuple and pass it to the $callback function.
     *
     * @template TKO
     *
     * @param callable(mixed...): TKO $callback
     * @return Map<TKO, TV>
     */
    public function reindexN(callable $callback): Map;

    /**
     * Find first element which satisfies the condition
     *
     * ```php
     * >>> Stream::emits([1, 2, 3])->first(fn($elem) => $elem > 1)->get();
     * => 2
     * ```
     *
     * @param callable(TV): bool $predicate
     * @return Option<TV>
     */
    public function first(callable $predicate): Option;

    /**
     * Same as {@see StreamTerminalOps::first()}, but deconstruct input tuple and pass it to the $predicate function.
     *
     * @param callable(mixed...): bool $predicate
     * @return Option<TV>
     */
    public function firstN(callable $predicate): Option;

    /**
     * A combined {@see Stream::first} and {@see Stream::map}.
     *
     * Filtering is handled via Option instead of Boolean.
     * So the output type TVO can be different from the input type TV.
     *
     * ```php
     * >>> Stream::emits(['zero', '1', '2'])
     * >>>     ->firstMap(fn($elem) => Option::when(is_numeric($elem), fn() => (int) $elem));
     * => Some(1)
     * ```
     *
     * @template TVO
     *
     * @param callable(TV): Option<TVO> $callback
     * @return Option<TVO>
     */
    public function firstMap(callable $callback): Option;

    /**
     * Fold many elements into one
     *
     * ```php
     * >>> Stream::emits(['1', '2'])->fold('0')(fn($acc, $cur) => $acc . $cur);
     * => '012'
     * ```
     *
     * @template TVO
     *
     * @param TVO $init
     * @return FoldOperation<TV, TVO>
     *
     * @see FoldMethodReturnTypeProvider
     */
    public function fold(mixed $init): FoldOperation;

    /**
     * Return first stream element
     *
     * ```php
     * >>> Stream::emits([1, 2])->head()->get();
     * => 1
     * ```
     *
     * @return Option<TV>
     */
    public function head(): Option;

    /**
     * Returns last stream element which satisfies the condition
     *
     * ```php
     * >>> Stream::emits([1, 0, 2])->last(fn($elem) => $elem > 0)->get();
     * => 2
     * ```
     *
     * @param callable(TV): bool $predicate
     * @return Option<TV>
     */
    public function last(callable $predicate): Option;

    /**
     * A combined {@see Stream::last} and {@see Stream::map}.
     *
     * Filtering is handled via Option instead of Boolean.
     * So the output type TVO can be different from the input type TV.
     *
     * ```php
     * >>> Stream::emits(['zero', '1', '2'])
     * >>>     ->lastMap(fn($elem) => Option::when(is_numeric($elem), fn() => (int) $elem));
     * => Some(2)
     * ```
     *
     * @template TVO
     *
     * @param callable(TV): Option<TVO> $callback
     * @return Option<TVO>
     */
    public function lastMap(callable $callback): Option;

    /**
     * Same as {@see StreamTerminalOps::last()}, but deconstruct input tuple and pass it to the $predicate function.
     *
     * @param callable(mixed...): bool $predicate
     * @return Option<TV>
     */
    public function lastN(callable $predicate): Option;

    /**
     * Returns first stream element
     * Alias for {@see SeqOps::head}
     *
     * ```php
     * >>> Stream::emits([1, 2])->firstElement()->get();
     * => 1
     * ```
     *
     * @return Option<TV>
     */
    public function firstElement(): Option;

    /**
     * Returns last stream element
     *
     * ```php
     * >>> Stream::emits([1, 2])->lastElement()->get();
     * => 2
     * ```
     *
     * @return Option<TV>
     */
    public function lastElement(): Option;

    /**
     * Run the stream.
     *
     * This is useful if you care only for side effects.
     *
     * ```php
     * >>> Stream::drain([1, 2])->lines()->drain();
     * 1
     * 2
     * ```
     */
    public function drain(): void;

    /**
     * Displays all elements of this collection in a string
     * using start, end, and separator strings.
     *
     * ```php
     * >>> Stream::emits([1, 2, 3])->mkString("(", ",", ")")
     * => '(1,2,3)'
     *
     * >>> Stream::emits([])->mkString("(", ",", ")")
     * => '()'
     * ```
     */
    public function mkString(string $start = '', string $sep = ',', string $end = ''): string;

    /**
     * Returns the maximum value from collection
     *
     * ```php
     * >>> Stream::emits([1, 4, 2])->max()->get();
     * => 4
     * ```
     *
     * @return Option<TV>
     */
    public function max(): Option;

    /**
     * Returns the maximum value from collection by iterating each element using the callback
     *
     * ```php
     * >>> Stream::emits([new Foo(1), new Bar(6), new Foo(2)])->maxBy(fn(Foo $foo) => $foo->a)->get();
     * => Bar(6)
     * ```
     *
     * @param callable(TV): mixed $callback
     * @return Option<TV>
     */
    public function maxBy(callable $callback): Option;

    /**
     * Returns the minimum value from collection
     *
     * ```php
     * >>> Stream::emits([1, 4, 2])->min()->get();
     * => 1
     * ```
     *
     * @return Option<TV>
     */
    public function min(): Option;

    /**
     * Returns the minimum value from collection by iterating each element using the callback
     *
     * ```php
     * >>> Stream::emits([new Foo(1), new Bar(6), new Foo(2)])->minBy(fn(Foo $foo) => $foo->a)->get();
     * => Foo(1)
     * ```
     *
     * @param callable(TV): mixed $callback
     * @return Option<TV>
     */
    public function minBy(callable $callback): Option;
}
