<?php

declare(strict_types=1);

namespace Fp\Collection;

use Fp\Operations\ReindexOperation;

use function Fp\Callable\dropFirstArg;
use function Fp\Cast\asArray;

/**
 * Produces a new array of elements by assigning the values to keys generated by a transformation function (callback).
 *
 * ```php
 * >>> reindex([1, 'a' => 2], fn (int $value) => $value);
 * => [1 => 1, 2 => 2]
 * ```
 *
 * @template TV
 * @template TKO of array-key
 *
 * @param iterable<TV> $collection
 * @param callable(TV): TKO $callback
 * @return array<TKO, TV>
 *
 * @psalm-return ($collection is non-empty-array
 *     ? non-empty-array<TKO, TV>
 *     : array<TKO, TV>)
 */
function reindex(iterable $collection, callable $callback): array
{
    return reindexKV($collection, dropFirstArg($callback));
}

/**
 * Same as {@see reindex()}, but passing also the key to the $callback function.
 *
 * ```php
 * >>> reindexKV(['a' => 1, 'b' => 2], fn (string $key, int $value) => "{$key}-{$value}");
 * => ['a-1' => 1, 'b-2' => 2]
 * ```
 *
 * @template TK
 * @template TV
 * @template TKO of array-key
 *
 * @param iterable<TK, TV> $collection
 * @param callable(TK, TV): TKO $callback
 * @return array<TKO, TV>
 *
 * @psalm-return ($collection is non-empty-array
 *     ? non-empty-array<TKO, TV>
 *     : array<TKO, TV>)
 */
function reindexKV(iterable $collection, callable $callback): array
{
    return asArray(ReindexOperation::of($collection)($callback));
}
