<?php

declare(strict_types=1);

namespace Fp\Collections;

use Fp\Psalm\Hook\MethodReturnTypeProvider\MapTapNMethodReturnTypeProvider;

/**
 * @template-covariant TK
 * @template-covariant TV
 *
 * @psalm-suppress InvalidTemplateParam
 */
interface NonEmptyMapChainableOps
{
    /**
     * Produces new collection with given element
     *
     * ```php
     * >>> NonEmptyHashMap::collectPairsNonEmpty([['a', 1], ['b', 2]])->updated('b', 3)->toList();
     * => [['a', 1], ['b', 3]]
     * ```
     *
     * @template TKI
     * @template TVI
     *
     * @param TKI $key
     * @param TVI $value
     * @return NonEmptyMap<TK|TKI, TV|TVI>
     */
    public function appended(mixed $key, mixed $value): NonEmptyMap;

    /**
     * Produces a new collection of elements by mapping each element in collection
     * through a transformation function (callback)
     *
     * ```php
     * >>> $collection = NonEmptyHashMap::collectPairsNonEmpty([['1', 1], ['2', 2]]);
     * => NonEmptyHashMap('1' -> 1, '2' -> 2)
     *
     * >>> $collection->map(fn($elem) => $elem + 1);
     * => NonEmptyHashMap('1' -> 2, '2' -> 3)
     * ```
     *
     * @template TVO
     *
     * @param callable(TV): TVO $callback
     * @return NonEmptyMap<TK, TVO>
     */
    public function map(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::map()}, but passing also the key to the $callback function.
     *
     * ```php
     * >>> $collection = NonEmptyHashMap::collectPairsNonEmpty([['1', 1], ['2', 2]]);
     * => NonEmptyHashMap('1' -> 1, '2' -> 2)
     *
     * >>> $collection->mapKV(fn($key, $elem) => "{$key}-{$elem}");
     * => NonEmptyHashMap('1' -> '1-1', '2' -> '2-2')
     * ```
     *
     * @template TVO
     *
     * @param callable(TK, TV): TVO $callback
     * @return NonEmptyMap<TK, TVO>
     */
    public function mapKV(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::map()}, but deconstruct input tuple and pass it to the $callback function.
     *
     * @template TVO
     *
     * @param callable(mixed...): TVO $callback
     * @return NonEmptyMap<TK, TVO>
     *
     * @see MapTapNMethodReturnTypeProvider
     */
    public function mapN(callable $callback): NonEmptyMap;

    /**
     * Map collection and flatten the result
     *
     * ```php
     * >>> $collection = NonEmptyHashMap::collectPairsNonEmpty([['2', 2], ['5', 5]]);
     * => NonEmptyHashMap('2' -> 2, '5' -> 5)
     *
     * >>> $collection
     * >>>     ->flatMap(fn(int $val) => [
     * >>>         ($val - 1) => $val - 1,
     * >>>         ($val) => $val,
     * >>>         ($val + 1) => $val + 1,
     * >>>     ])
     * >>>     ->toList();
     * => [[1, 1], [2, 2], [3, 3], [4, 4], [5, 5], [6, 6]]
     * ```
     *
     * @template TKO
     * @template TVO
     *
     * @param callable(TV): (non-empty-array<TKO, TVO>|NonEmptyCollection<TKO, TVO>) $callback
     * @return NonEmptyMap<TKO, TVO>
     */
    public function flatMap(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::flatMap()}, but deconstruct input tuple and pass it to the $callback function.
     *
     * @template TKO
     * @template TVO
     *
     * @param callable(mixed...): (non-empty-array<TKO, TVO>|NonEmptyCollection<TKO, TVO>) $callback
     * @return NonEmptyMap<TKO, TVO>
     */
    public function flatMapN(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::flatMap()}, but passing also the key to the $callback function.
     *
     * @template TKO
     * @template TVO
     *
     * @param callable(TK, TV): (non-empty-array<TKO, TVO>|NonEmptyCollection<TKO, TVO>) $callback
     * @return NonEmptyMap<TKO, TVO>
     */
    public function flatMapKV(callable $callback): NonEmptyMap;

    /**
     * Call a function for every collection element
     *
     * ```php
     * >>> NonEmptyHashMap::collectParisNonEmpty([['fst', new Foo(1)], ['snd', new Foo(2)]])
     * >>>     ->tap(fn(Foo $foo) => $foo->a = $foo->a + 1)
     * >>>     ->map(fn(Foo $foo) => $foo->a)
     * >>>     ->toList();
     * => [2, 3]
     * ```
     *
     * @param callable(TV): void $callback
     * @return NonEmptyMap<TK, TV>
     */
    public function tap(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::tap()}, but deconstruct input tuple and pass it to the $callback function.
     *
     * @param callable(mixed...): void $callback
     * @return NonEmptyMap<TK, TV>
     */
    public function tapN(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::tap()}, but passing also the key to the $callback function.
     *
     * @param callable(TK, TV): void $callback
     * @return NonEmptyMap<TK, TV>
     */
    public function tapKV(callable $callback): NonEmptyMap;

    /**
     * Produces a new collection of elements by assigning the values to keys generated by a transformation function (callback).
     *
     * ```php
     * >>> $collection = NonEmptyHashMap::collectPairsNonEmpty([['1', 1], ['2', 2]]);
     * => NonEmptyHashMap('1' -> 1, '2' -> 2)
     *
     * >>> $collection->reindex(fn($v) => $v + 1);
     * => NonEmptyHashMap(2 -> 1, 3 -> 2)
     * ```
     *
     * @template TKO
     *
     * @param callable(TV): TKO $callback
     * @return NonEmptyMap<TKO, TV>
     */
    public function reindex(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::reindex()}, but deconstruct input tuple and pass it to the $callback function.
     *
     * @template TKO
     *
     * @param callable(mixed...): TKO $callback
     * @return NonEmptyMap<TKO, TV>
     */
    public function reindexN(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::reindex()}, but passing also the key to the $callback function.
     *
     * ```php
     * >>> $collection = NonEmptyHashMap::collectPairs([['1', 1], ['2', 2]]);
     * => HashMap('1' -> 1, '2' -> 2)
     *
     * >>> $collection->reindexKV(fn($k, $v) => "{$k}-{$v}");
     * => NonEmptyHashMap('1-1' -> 1, '2-2' -> 2)
     * ```
     *
     * @template TKO
     *
     * @param callable(TK, TV): TKO $callback
     * @return NonEmptyMap<TKO, TV>
     */
    public function reindexKV(callable $callback): NonEmptyMap;

    /**
     * Group elements
     *
     * ```php
     * >>> NonEmptyHashMap::collectNonEmpty(['fst' => 1, 'snd' => 2, 'thr' => 3])
     * >>>     ->groupBy(fn($i) => 0 === $i % 2 ? 'even' : 'odd')
     * => NonEmptyHashMap('odd' => NonEmptyHashMap('fst' => 1, 'trd' => 3), 'even' => NonEmptyHashMap('snd' => 2))
     * ```
     *
     * @template TKO
     *
     * @param callable(TV): TKO $callback
     * @return NonEmptyMap<TKO, NonEmptyMap<TK, TV>>
     */
    public function groupBy(callable $callback): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::groupBy()}, but passing also the key to the $callback function.
     *
     * @template TKO
     *
     * @param callable(TK, TV): TKO $callback
     * @return NonEmptyMap<TKO, NonEmptyMap<TK, TV>>
     */
    public function groupByKV(callable $callback): NonEmptyMap;

    /**
     * ```php
     * >>> NonEmptyHashMap::collectNonEmpty([
     * >>>     '10-1' => ['id' => 10, 'sum' => 10],
     * >>>     '10-2' => ['id' => 10, 'sum' => 15],
     * >>>     '10-3' => ['id' => 10, 'sum' => 20],
     * >>>     '20-1' => ['id' => 20, 'sum' => 10],
     * >>>     '20-2' => ['id' => 20, 'sum' => 15],
     * >>>     '30-1' => ['id' => 30, 'sum' => 20],
     * >>> ])->groupMap(
     * >>>     fn(array $a) => $a['id'],
     * >>>     fn(array $a) => $a['sum'] + 1,
     * >>> );
     * => NonEmptyHashMap(
     * =>   10 -> NonEmptyHashMap('10-3' => 21, '10-2' => 16, '10-1' => 11),
     * =>   20 -> NonEmptyHashMap('20-2' => 16, '20-1' => 11),
     * =>   30 -> NonEmptyHashMap('30-1' => 21),
     * => )
     * ```
     *
     * @template TKO
     * @template TVO
     *
     * @param callable(TV): TKO $group
     * @param callable(TV): TVO $map
     * @return NonEmptyMap<TKO, NonEmptyMap<TK, TVO>>
     */
    public function groupMap(callable $group, callable $map): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::groupMap()}, but passing also the key to the $group and $map function.
     *
     * @template TKO
     * @template TVO
     *
     * @param callable(TK, TV): TKO $group
     * @param callable(TK, TV): TVO $map
     * @return NonEmptyMap<TKO, NonEmptyMap<TK, TVO>>
     */
    public function groupMapKV(callable $group, callable $map): NonEmptyMap;

    /**
     * Partitions this NonEmptyMap<TK, TV> into a NonEmptyMap<TKO, TVO> according to a discriminator function $group.
     * All the values that have the same discriminator are then transformed by the $map and
     * then reduced into a single value with the $reduce.
     *
     * ```php
     * >>> NonEmptyHashMap::collectNonEmpty([
     * >>>     '10-1' => ['id' => 10, 'sum' => 10],
     * >>>     '10-2' => ['id' => 10, 'sum' => 15],
     * >>>     '10-3' => ['id' => 10, 'sum' => 20],
     * >>>     '20-1' => ['id' => 20, 'sum' => 10],
     * >>>     '20-2' => ['id' => 20, 'sum' => 15],
     * >>>     '30-1' => ['id' => 30, 'sum' => 20],
     * >>> ])->groupMapReduce(
     * >>>     fn(array $a) => $a['id'],
     * >>>     fn(array $a) => $a['sum'],
     * >>>     fn(int $old, int $new) => $old + $new,
     * >>> );
     * => NonEmptyHashMap([10 => 45, 20 => 25, 30 => 20])
     * ```
     *
     * @template TKO
     * @template TVO
     *
     * @param callable(TV): TKO $group
     * @param callable(TV): TVO $map
     * @param callable(TVO, TVO): TVO $reduce
     *
     * @return NonEmptyMap<TKO, TVO>
     */
    public function groupMapReduce(callable $group, callable $map, callable $reduce): NonEmptyMap;

    /**
     * Same as {@see NonEmptyMapChainableOps::groupMapReduce()}, but passing also the key to the $group and $map function.
     *
     * @template TKO
     * @template TVO
     *
     * @param callable(TK, TV): TKO $group
     * @param callable(TK, TV): TVO $map
     * @param callable(TVO, TVO): TVO $reduce
     *
     * @return NonEmptyMap<TKO, TVO>
     */
    public function groupMapReduceKV(callable $group, callable $map, callable $reduce): NonEmptyMap;
}
