import { Index } from '../indexes';
import { KeyCreation, Config, IndexOptions, IndexObject, Key, Health, Stats, Version, TokenSearchRules, TokenOptions, TasksQuery, WaitOptions, KeyUpdate, IndexesQuery, IndexesResults, KeysQuery, KeysResults, TasksResults, SwapIndexesParams, CancelTasksQuery, DeleteTasksQuery, MultiSearchParams, MultiSearchResponse } from '../types';
import { HttpRequests } from '../http-requests';
import { TaskClient, Task } from '../task';
import { EnqueuedTask } from '../enqueued-task';
declare class Client {
    config: Config;
    httpRequest: HttpRequests;
    tasks: TaskClient;
    /**
     * Creates new MeiliSearch instance
     *
     * @param config - Configuration object
     */
    constructor(config: Config);
    /**
     * Return an Index instance
     *
     * @param indexUid - The index UID
     * @returns Instance of Index
     */
    index<T extends Record<string, any> = Record<string, any>>(indexUid: string): Index<T>;
    /**
     * Gather information about an index by calling MeiliSearch and return an
     * Index instance with the gathered information
     *
     * @param indexUid - The index UID
     * @returns Promise returning Index instance
     */
    getIndex<T extends Record<string, any> = Record<string, any>>(indexUid: string): Promise<Index<T>>;
    /**
     * Gather information about an index by calling MeiliSearch and return the raw
     * JSON response
     *
     * @param indexUid - The index UID
     * @returns Promise returning index information
     */
    getRawIndex(indexUid: string): Promise<IndexObject>;
    /**
     * Get all the indexes as Index instances.
     *
     * @param parameters - Parameters to browse the indexes
     * @returns Promise returning array of raw index information
     */
    getIndexes(parameters?: IndexesQuery): Promise<IndexesResults<Index[]>>;
    /**
     * Get all the indexes in their raw value (no Index instances).
     *
     * @param parameters - Parameters to browse the indexes
     * @returns Promise returning array of raw index information
     */
    getRawIndexes(parameters?: IndexesQuery): Promise<IndexesResults<IndexObject[]>>;
    /**
     * Create a new index
     *
     * @param uid - The index UID
     * @param options - Index options
     * @returns Promise returning Index instance
     */
    createIndex(uid: string, options?: IndexOptions): Promise<EnqueuedTask>;
    /**
     * Update an index
     *
     * @param uid - The index UID
     * @param options - Index options to update
     * @returns Promise returning Index instance after updating
     */
    updateIndex(uid: string, options?: IndexOptions): Promise<EnqueuedTask>;
    /**
     * Delete an index
     *
     * @param uid - The index UID
     * @returns Promise which resolves when index is deleted successfully
     */
    deleteIndex(uid: string): Promise<EnqueuedTask>;
    /**
     * Deletes an index if it already exists.
     *
     * @param uid - The index UID
     * @returns Promise which resolves to true when index exists and is deleted
     *   successfully, otherwise false if it does not exist
     */
    deleteIndexIfExists(uid: string): Promise<boolean>;
    /**
     * Swaps a list of index tuples.
     *
     * @param params - List of indexes tuples to swap.
     * @returns Promise returning object of the enqueued task
     */
    swapIndexes(params: SwapIndexesParams): Promise<EnqueuedTask>;
    /**
     * Perform multiple search queries.
     *
     * It is possible to make multiple search queries on the same index or on
     * different ones
     *
     * @example
     *
     * ```ts
     * client.multiSearch({
     *   queries: [
     *     { indexUid: 'movies', q: 'wonder' },
     *     { indexUid: 'books', q: 'flower' },
     *   ],
     * })
     * ```
     *
     * @param queries - Search queries
     * @param config - Additional request configuration options
     * @returns Promise containing the search responses
     */
    multiSearch<T extends Record<string, any> = Record<string, any>>(queries?: MultiSearchParams, config?: Partial<Request>): Promise<MultiSearchResponse<T>>;
    /**
     * Get the list of all client tasks
     *
     * @param parameters - Parameters to browse the tasks
     * @returns Promise returning all tasks
     */
    getTasks(parameters?: TasksQuery): Promise<TasksResults>;
    /**
     * Get one task on the client scope
     *
     * @param taskUid - Task identifier
     * @returns Promise returning a task
     */
    getTask(taskUid: number): Promise<Task>;
    /**
     * Wait for multiple tasks to be finished.
     *
     * @param taskUids - Tasks identifier
     * @param waitOptions - Options on timeout and interval
     * @returns Promise returning an array of tasks
     */
    waitForTasks(taskUids: number[], { timeOutMs, intervalMs }?: WaitOptions): Promise<Task[]>;
    /**
     * Wait for a task to be finished.
     *
     * @param taskUid - Task identifier
     * @param waitOptions - Options on timeout and interval
     * @returns Promise returning an array of tasks
     */
    waitForTask(taskUid: number, { timeOutMs, intervalMs }?: WaitOptions): Promise<Task>;
    /**
     * Cancel a list of enqueued or processing tasks.
     *
     * @param parameters - Parameters to filter the tasks.
     * @returns Promise containing an EnqueuedTask
     */
    cancelTasks(parameters: CancelTasksQuery): Promise<EnqueuedTask>;
    /**
     * Delete a list of tasks.
     *
     * @param parameters - Parameters to filter the tasks.
     * @returns Promise containing an EnqueuedTask
     */
    deleteTasks(parameters?: DeleteTasksQuery): Promise<EnqueuedTask>;
    /**
     * Get all API keys
     *
     * @param parameters - Parameters to browse the indexes
     * @returns Promise returning an object with keys
     */
    getKeys(parameters?: KeysQuery): Promise<KeysResults>;
    /**
     * Get one API key
     *
     * @param keyOrUid - Key or uid of the API key
     * @returns Promise returning a key
     */
    getKey(keyOrUid: string): Promise<Key>;
    /**
     * Create one API key
     *
     * @param options - Key options
     * @returns Promise returning a key
     */
    createKey(options: KeyCreation): Promise<Key>;
    /**
     * Update one API key
     *
     * @param keyOrUid - Key
     * @param options - Key options
     * @returns Promise returning a key
     */
    updateKey(keyOrUid: string, options: KeyUpdate): Promise<Key>;
    /**
     * Delete one API key
     *
     * @param keyOrUid - Key
     * @returns
     */
    deleteKey(keyOrUid: string): Promise<void>;
    /**
     * Checks if the server is healthy, otherwise an error will be thrown.
     *
     * @returns Promise returning an object with health details
     */
    health(): Promise<Health>;
    /**
     * Checks if the server is healthy, return true or false.
     *
     * @returns Promise returning a boolean
     */
    isHealthy(): Promise<boolean>;
    /**
     * Get the stats of all the database
     *
     * @returns Promise returning object of all the stats
     */
    getStats(): Promise<Stats>;
    /**
     * Get the version of MeiliSearch
     *
     * @returns Promise returning object with version details
     */
    getVersion(): Promise<Version>;
    /**
     * Creates a dump
     *
     * @returns Promise returning object of the enqueued task
     */
    createDump(): Promise<EnqueuedTask>;
    /**
     * Generate a tenant token
     *
     * @param apiKeyUid - The uid of the api key used as issuer of the token.
     * @param searchRules - Search rules that are applied to every search.
     * @param options - Token options to customize some aspect of the token.
     * @returns The token in JWT format.
     */
    generateTenantToken(_apiKeyUid: string, _searchRules: TokenSearchRules, _options?: TokenOptions): string;
}
export { Client };
//# sourceMappingURL=client.d.ts.map