"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isEndpointFiltered = exports.isCodebaseFiltered = exports.groupEndpointsByCodebase = exports.targetCodebases = exports.getFunctionLabel = exports.getEndpointFilters = exports.parseFunctionSelector = exports.endpointMatchesFilter = exports.endpointMatchesAnyFilter = void 0;
const backend = require("./backend");
const projectConfig_1 = require("../../functions/projectConfig");
function endpointMatchesAnyFilter(endpoint, filters) {
    if (!filters) {
        return true;
    }
    return filters.some((filter) => endpointMatchesFilter(endpoint, filter));
}
exports.endpointMatchesAnyFilter = endpointMatchesAnyFilter;
function endpointMatchesFilter(endpoint, filter) {
    if (endpoint.codebase && filter.codebase) {
        if (endpoint.codebase !== filter.codebase) {
            return false;
        }
    }
    if (!filter.idChunks) {
        return true;
    }
    const idChunks = endpoint.id.split("-");
    if (idChunks.length < filter.idChunks.length) {
        return false;
    }
    for (let i = 0; i < filter.idChunks.length; i += 1) {
        if (idChunks[i] !== filter.idChunks[i]) {
            return false;
        }
    }
    return true;
}
exports.endpointMatchesFilter = endpointMatchesFilter;
function parseFunctionSelector(selector) {
    const fragments = selector.split(":");
    if (fragments.length < 2) {
        return [
            { codebase: fragments[0] },
            { codebase: projectConfig_1.DEFAULT_CODEBASE, idChunks: fragments[0].split(/[-.]/) },
        ];
    }
    return [
        {
            codebase: fragments[0],
            idChunks: fragments[1].split(/[-.]/),
        },
    ];
}
exports.parseFunctionSelector = parseFunctionSelector;
function getEndpointFilters(options) {
    if (!options.only) {
        return undefined;
    }
    const selectors = options.only.split(",");
    const filters = [];
    for (let selector of selectors) {
        if (selector.startsWith("functions:")) {
            selector = selector.replace("functions:", "");
            if (selector.length > 0) {
                filters.push(...parseFunctionSelector(selector));
            }
        }
    }
    if (filters.length === 0) {
        return undefined;
    }
    return filters;
}
exports.getEndpointFilters = getEndpointFilters;
function getFunctionLabel(fn) {
    let id = `${fn.id}(${fn.region})`;
    if (fn.codebase && fn.codebase !== projectConfig_1.DEFAULT_CODEBASE) {
        id = `${fn.codebase}:${id}`;
    }
    return id;
}
exports.getFunctionLabel = getFunctionLabel;
function targetCodebases(config, filters) {
    const codebasesFromConfig = [...new Set(Object.values(config).map((c) => c.codebase))];
    if (!filters) {
        return [...codebasesFromConfig];
    }
    const codebasesFromFilters = [
        ...new Set(filters.map((f) => f.codebase).filter((c) => c !== undefined)),
    ];
    if (codebasesFromFilters.length === 0) {
        return [...codebasesFromConfig];
    }
    const intersections = [];
    for (const codebase of codebasesFromConfig) {
        if (codebasesFromFilters.includes(codebase)) {
            intersections.push(codebase);
        }
    }
    return intersections;
}
exports.targetCodebases = targetCodebases;
function groupEndpointsByCodebase(wantBackends, haveEndpoints) {
    const grouped = {};
    let endpointsToAssign = haveEndpoints;
    for (const codebase of Object.keys(wantBackends)) {
        const names = backend.allEndpoints(wantBackends[codebase]).map((e) => backend.functionName(e));
        grouped[codebase] = backend.of(...endpointsToAssign.filter((e) => names.includes(backend.functionName(e))));
        endpointsToAssign = endpointsToAssign.filter((e) => !names.includes(backend.functionName(e)));
    }
    for (const codebase of Object.keys(wantBackends)) {
        const matchedEndpoints = endpointsToAssign.filter((e) => e.codebase === codebase);
        grouped[codebase] = backend.merge(grouped[codebase], backend.of(...matchedEndpoints));
        const matchedNames = matchedEndpoints.map((e) => backend.functionName(e));
        endpointsToAssign = endpointsToAssign.filter((e) => {
            return !matchedNames.includes(backend.functionName(e));
        });
    }
    return grouped;
}
exports.groupEndpointsByCodebase = groupEndpointsByCodebase;
function isCodebaseFiltered(codebase, filters) {
    return filters.some((filter) => {
        const noIdChunks = (filter.idChunks || []).length === 0;
        return noIdChunks && filter.codebase === codebase;
    });
}
exports.isCodebaseFiltered = isCodebaseFiltered;
function isEndpointFiltered(endpoint, filters) {
    return filters.some((filter) => endpointMatchesFilter(endpoint, filter));
}
exports.isEndpointFiltered = isEndpointFiltered;
