"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.runBuildtoolsCommand = exports.fetchBuildtoolsJar = void 0;
const fs = require("fs-extra");
const os = require("os");
const path = require("path");
const spawn = require("cross-spawn");
const downloadUtils = require("../downloadUtils");
const error_1 = require("../error");
const logger_1 = require("../logger");
const rimraf = require("rimraf");
const utils = require("../utils");
const JAR_CACHE_DIR = process.env.FIREBASE_CRASHLYTICS_BUILDTOOLS_PATH ||
    path.join(os.homedir(), ".cache", "firebase", "crashlytics", "buildtools");
const JAR_VERSION = "2.9.2";
const JAR_URL = `https://dl.google.com/android/maven2/com/google/firebase/firebase-crashlytics-buildtools/${JAR_VERSION}/firebase-crashlytics-buildtools-${JAR_VERSION}.jar`;
async function fetchBuildtoolsJar() {
    if (process.env.CRASHLYTICS_LOCAL_JAR) {
        return process.env.CRASHLYTICS_LOCAL_JAR;
    }
    const jarPath = path.join(JAR_CACHE_DIR, `crashlytics-buildtools-${JAR_VERSION}.jar`);
    if (fs.existsSync(jarPath)) {
        logger_1.logger.debug(`Buildtools Jar already downloaded at ${jarPath}`);
        return jarPath;
    }
    if (fs.existsSync(JAR_CACHE_DIR)) {
        logger_1.logger.debug(`Deleting Jar cache at ${JAR_CACHE_DIR} because the CLI was run with a newer Jar version`);
        rimraf.sync(JAR_CACHE_DIR);
    }
    utils.logBullet("Downloading crashlytics-buildtools.jar to " + jarPath);
    utils.logBullet("For open source licenses used by this command, look in the META-INF directory in the buildtools.jar file");
    const tmpfile = await downloadUtils.downloadToTmp(JAR_URL);
    fs.mkdirSync(JAR_CACHE_DIR, { recursive: true });
    fs.copySync(tmpfile, jarPath);
    return jarPath;
}
exports.fetchBuildtoolsJar = fetchBuildtoolsJar;
function runBuildtoolsCommand(jarFile, args, debug) {
    var _a;
    const fullArgs = ["-jar", jarFile, ...args, "-clientName", "firebase-cli;crashlytics-buildtools"];
    const outputs = spawn.sync("java", fullArgs, {
        stdio: debug ? "inherit" : "pipe",
    });
    if (outputs.status !== 0) {
        if (!debug) {
            utils.logWarning(((_a = outputs.stdout) === null || _a === void 0 ? void 0 : _a.toString()) || "An unknown error occurred");
        }
        throw new error_1.FirebaseError(`java command failed with args: ${fullArgs}`);
    }
}
exports.runBuildtoolsCommand = runBuildtoolsCommand;
