"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.command = void 0;
const command_1 = require("../command");
const error_1 = require("../error");
const projectUtils_1 = require("../projectUtils");
const requirePermissions_1 = require("../requirePermissions");
const backend = require("../deploy/functions/backend");
const logger_1 = require("../logger");
const Table = require("cli-table");
exports.command = new command_1.Command("functions:list")
    .description("list all deployed functions in your Firebase project")
    .before(requirePermissions_1.requirePermissions, ["cloudfunctions.functions.list"])
    .action(async (options) => {
    try {
        const context = {
            projectId: (0, projectUtils_1.needProjectId)(options),
        };
        const existing = await backend.existingBackend(context);
        const endpointsList = backend.allEndpoints(existing).sort(backend.compareFunctions);
        const table = new Table({
            head: ["Function", "Version", "Trigger", "Location", "Memory", "Runtime"],
            style: { head: ["yellow"] },
        });
        for (const endpoint of endpointsList) {
            const trigger = backend.endpointTriggerType(endpoint);
            const availableMemoryMb = endpoint.availableMemoryMb || "---";
            const entry = [
                endpoint.id,
                endpoint.platform === "gcfv2" ? "v2" : "v1",
                trigger,
                endpoint.region,
                availableMemoryMb,
                endpoint.runtime,
            ];
            table.push(entry);
        }
        logger_1.logger.info(table.toString());
        return endpointsList;
    }
    catch (err) {
        throw new error_1.FirebaseError("Failed to list functions", {
            exit: 1,
            original: err,
        });
    }
});
