// ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
// ┃ ██████ ██████ ██████       █      █      █      █      █ █▄  ▀███ █       ┃
// ┃ ▄▄▄▄▄█ █▄▄▄▄▄ ▄▄▄▄▄█  ▀▀▀▀▀█▀▀▀▀▀ █ ▀▀▀▀▀█ ████████▌▐███ ███▄  ▀█ █ ▀▀▀▀▀ ┃
// ┃ █▀▀▀▀▀ █▀▀▀▀▀ █▀██▀▀ ▄▄▄▄▄ █ ▄▄▄▄▄█ ▄▄▄▄▄█ ████████▌▐███ █████▄   █ ▄▄▄▄▄ ┃
// ┃ █      ██████ █  ▀█▄       █ ██████      █      ███▌▐███ ███████▄ █       ┃
// ┣━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┫
// ┃ Copyright (c) 2017, the Perspective Authors.                              ┃
// ┃ ╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌ ┃
// ┃ This file is part of the Perspective library, distributed under the terms ┃
// ┃ of the [Apache License 2.0](https://www.apache.org/licenses/LICENSE-2.0). ┃
// ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛

const fs = require("fs");
const path = require("path");

/**
 * Returns an `ArrayBuffer` containing the contents of a `.arrow` file
 * located at `arrow_path`.
 *
 * Because `fs.readFileSync` shares its underlying buffer
 * between calls to `readFileSync`, we need to get a slice
 * of the `ArrayBuffer` specifically at its byte offset.
 *
 * See https://github.com/nodejs/node/issues/11132 for more details.
 *
 * @param arrow_path {String} a path to an arrow file.
 * @returns {ArrayBuffer} an ArrayBuffer containing the arrow-serialized data.
 */
function load_arrow(arrow_path) {
    const data = fs.readFileSync(arrow_path);
    return data.buffer.slice(
        data.byteOffset,
        data.byteOffset + data.byteLength
    );
}

const int_float_str_arrow = load_arrow(
    path.join(__dirname, "..", "arrow", "int_float_str.arrow")
);

module.exports = {
    int_float_str_arrow,
};
