// ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
// ┃ ██████ ██████ ██████       █      █      █      █      █ █▄  ▀███ █       ┃
// ┃ ▄▄▄▄▄█ █▄▄▄▄▄ ▄▄▄▄▄█  ▀▀▀▀▀█▀▀▀▀▀ █ ▀▀▀▀▀█ ████████▌▐███ ███▄  ▀█ █ ▀▀▀▀▀ ┃
// ┃ █▀▀▀▀▀ █▀▀▀▀▀ █▀██▀▀ ▄▄▄▄▄ █ ▄▄▄▄▄█ ▄▄▄▄▄█ ████████▌▐███ █████▄   █ ▄▄▄▄▄ ┃
// ┃ █      ██████ █  ▀█▄       █ ██████      █      ███▌▐███ ███████▄ █       ┃
// ┣━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┫
// ┃ Copyright (c) 2017, the Perspective Authors.                              ┃
// ┃ ╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌╌ ┃
// ┃ This file is part of the Perspective library, distributed under the terms ┃
// ┃ of the [Apache License 2.0](https://www.apache.org/licenses/LICENSE-2.0). ┃
// ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛

#pragma once

#ifdef PSP_PARALLEL_FOR
#include "base.h"
#include <arrow/util/parallel.h>
#include <arrow/status.h>
#include <mutex>
#else
#include "raw_types.h"
#endif

namespace perspective {

template <class FUNCTION>
void
parallel_for(int num_tasks, FUNCTION&& func) {
#ifdef PSP_PARALLEL_FOR
    std::exception_ptr e;
    std::mutex e_mtx;
    const auto rethrow_wrapper = [&](int64_t task) {
        try {
            return func(task);
        } catch (...) {
            std::lock_guard<std::mutex> lg(e_mtx);
            e = std::current_exception();
        }
    };
    auto status = arrow::internal::ParallelFor(num_tasks, rethrow_wrapper);
    if (!status.ok()) {
        PSP_COMPLAIN_AND_ABORT("ParallelFor failed");
    }
    if (e) {
        std::rethrow_exception(e);
    }
#else
    for (t_uindex aggnum = 0; aggnum < num_tasks; ++aggnum) {
        func(aggnum);
    }
#endif
}

} // namespace perspective
