/**
 * @vitest-environment happy-dom
 */
import type { EndpointMeta } from '@fiction/core/utils'
import type { TestUtils } from '../../test-utils/init'

import fs from 'node:fs'
import { FormData } from 'formdata-node'
import { fileFromPath } from 'formdata-node/file-from-path'
import { describe, expect, it } from 'vitest'
import { FictionMedia } from '..'
import { FictionAws } from '../../plugin-aws'
import { testEnvFile, testImgPath } from '../../test-utils'
import { createTestUtils } from '../../test-utils/init'
import { getEnvVars } from '../../utils'

describe('media upload/download tests', async () => {
  let url: string | undefined

  if (!fs.existsSync(testEnvFile))
    console.warn(`missing test env file ${testEnvFile}`)

  const testUtils = createTestUtils({ envFiles: [testEnvFile] }) as (TestUtils & { fictionMedia?: FictionMedia })

  const v = getEnvVars(testUtils.fictionEnv, ['AWS_ACCESS_KEY', 'AWS_ACCESS_KEY_SECRET', 'UNSPLASH_ACCESS_KEY', 'AWS_BUCKET_MEDIA'] as const)

  const { awsAccessKey, awsAccessKeySecret, unsplashAccessKey, awsBucketMedia } = v

  const fictionAws = new FictionAws({ ...testUtils, awsAccessKey, awsAccessKeySecret })
  testUtils.fictionMedia = new FictionMedia({ ...testUtils, fictionAws, awsBucketMedia, unsplashAccessKey })
  testUtils.initialized = await testUtils.init()

  it('creates media from an external URL', async () => {
    const sourceImageUrl = 'https://images.unsplash.com/photo-1708724195876-1156245fce21'

    // Call the function under test
    const result = await testUtils.fictionMedia?.queries.ManageMedia.serve({
      _action: 'createFromUrl',
      orgId: testUtils.initialized?.orgId || '',
      userId: testUtils.initialized?.user?.userId || '',
      fields: { sourceImageUrl },
    }, { bearer: testUtils.initialized?.user } as EndpointMeta) // Mock the meta as well if needed

    // Assertions to check if the media was created correctly
    expect(result?.status).toBe('success')
    const media = result?.data?.[0]
    expect(media?.url).toContain('fiction-media')
    expect(media?.mime).toBe('image/jpeg') // or the correct mime type

    expect(Object.keys(media || {}).sort()).toMatchInlineSnapshot(`
      [
        "alt",
        "blurhash",
        "bucket",
        "caption",
        "contentEncoding",
        "createdAt",
        "duration",
        "etag",
        "filePath",
        "hash",
        "height",
        "mediaId",
        "mime",
        "orgId",
        "orientation",
        "originUrl",
        "preview",
        "prompt",
        "rasterUrl",
        "size",
        "sourceImageUrl",
        "tags",
        "thumbFilePath",
        "thumbOriginUrl",
        "thumbUrl",
        "updatedAt",
        "url",
        "userId",
        "width",
      ]
    `)
  })

  it('uploads a file', async () => {
    const formData = new FormData()
    const f = await fileFromPath(testImgPath, { type: 'image/jpeg' })

    formData.append('field1', 'val1')
    formData.append('field2', 'val2')
    formData.append('imageFile', f)

    const r = await testUtils?.fictionMedia?.uploadFile({ formData, caller: 'testUploadsAFile' })

    expect(r?.data?.mediaId).toBeDefined()

    url = r?.data?.url

    if (!url)
      throw new Error('no url')
    expect(url).toContain('fiction-media')
    expect(url).toContain('test.jpg')
    expect(r?.data?.url).toContain('.jpg')
    expect(r?.message).toMatchInlineSnapshot('"uploaded successfully"')
    expect(r?.data?.mime).toBe('image/jpeg')
    expect(r?.data?.size).toMatchInlineSnapshot(`123812`)
    expect(r?.data?.userId).toBe(testUtils?.initialized?.user?.userId)

    const img = await fetch(url)
    expect(img.status).toBe(200)
  })

  it('gets index of files uploaded', async () => {
    const r = await testUtils?.fictionMedia?.requests.MediaIndex.projectRequest({ _action: 'list' })

    expect(r?.data?.length).toBeGreaterThan(0)
    expect(r?.message).toBeFalsy()
    expect(r?.data?.[0].url).toContain('.jpg')
  })

  it('deletes a file', async () => {
    if (!url)
      throw new Error('no url')

    const img1 = await fetch(url)

    expect(img1.status).toBe(200)

    const r = await testUtils?.fictionMedia?.requests.ManageMedia.projectRequest({
      _action: 'delete',
      where: [{ url }],
    })

    const media = r?.data?.[0]
    expect(media?.mediaId).toBeTruthy()
    expect(r?.message).toMatchInlineSnapshot(`"1 Media items deleted"`)
    expect(r?.status).toBe('success')

    const img2 = await fetch(url)
    expect(img2.status).toBe(404)
  })
})
