// @unocss-include

import { toLabel } from '@factor/api'

interface MediaItemParams<T extends string> {
  key: T
  name?: string
  icon: string
  path?: string
  class?: string
  target?: '_blank'
}

export class MediaItem<T extends string> {
  key: T
  name: string
  icon: string
  path?: string
  target?: '_blank'
  class?: string
  constructor(params: MediaItemParams<T>) {
    this.key = params.key
    this.name = params.name || toLabel(params.key)
    this.icon = params.icon
    this.path = params.path
    this.target = params.target
  }
}

export function about() {
  return {
    missionStatement: [
      'At PageLines, we\'re working on the future of the internet using AI.',
      'We\'re passionate about assisting businesses, large and small, in enhancing their competitive edge. With a focus on introducing the era of conversational web, we employ AI to foster more adaptive marketing and create helpful websites.',
    ],
    subHeadline: {
      title: 'AI for Websites',
      desc: 'Improving web-driven businesses with AI.',
    },
    values: [
      {
        title: 'Create Awesome Experiences',
        desc: 'We want to make using our products a breeze. We\'re always looking for ways to make things easier for our customers and we really listen when you tell us how we can do better.',
        icon: 'i-heroicons-outline-light-bulb',
      },
      {
        title: 'Be an Artist',
        desc: 'We don\'t just make things work; we make things beautiful. We\'re all about stretching our creativity and shooting for excellence.',
        icon: 'i-heroicons-outline-palette',
      },
      {
        title: 'Karma Comes First',
        desc: 'We believe in doing good things and treating people right. We know that when we help our community grow, we grow too.',
        icon: 'i-heroicons-outline-heart',
      },
      {
        title: 'Have Fun, Be Quirky',
        desc: 'We\'re not just another tech company. We like to mix things up, keep things interesting, and add a little sparkle wherever we can.',
        icon: 'i-heroicons-outline-happy',
      },
      {
        title: 'Focus and Get Stuff Done',
        desc: 'At the end of the day, we\'re here to get the job done and we\'re not afraid of hard work. We know that building a great product and a great community takes time and effort, but we\'re up for the challenge.',
        icon: 'i-heroicons-outline-check-circle',
      },
    ],
    metrics: [
      {
        title: 'Countries Served',
        icon: 'i-heroicons-outline-globe-alt',
        value: 22,
      },
      {
        title: 'Messages Sent',
        icon: 'i-heroicons-outline-mail-open',
        value: 387_123,
      },
      {
        title: 'Businesses Assisted',
        icon: 'i-heroicons-outline-briefcase',
        value: 763,
      },
      {
        title: 'Product Iterations',
        icon: 'i-heroicons-outline-refresh',
        value: 34,
      },
    ],
  }
}

export const socialList = [
  new MediaItem({
    key: 'linkedin',
    path: 'https://www.linkedin.com/company/pagelines',
    target: '_blank',
    name: 'LinkedIn',
    icon: `<svg class="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M0 7.557h5.349V24H0zM20 7.75c-.057-.018-.111-.037-.17-.054s-.144-.03-.217-.042a4.793 4.793 0 00-.96-.1 7.431 7.431 0 00-5.748 3.144V7.557H7.557V24h5.349v-8.969s4.042-5.63 5.748-1.495V24h5.347V12.9A5.333 5.333 0 0020 7.75z" fill="currentColor"/> <circle cx="3" cy="3" r="3" fill="currentColor"/></svg>`,
  }),
  new MediaItem({
    key: 'facebook',
    path: 'https://www.facebook.com/pagelines',
    target: '_blank',
    name: 'Facebook',
    icon: `<svg class="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 12 24"><path d="M7.5 8.25v-3A1.5 1.5 0 019 3.75h1.5V0h-3A4.5 4.5 0 003 4.5v3.75H0V12h3v12h4.5V12h3L12 8.25z" fill="currentColor"/></svg>`,
  }),
  new MediaItem({
    key: 'github',
    path: 'https://github.com/pagelines',
    target: '_blank',
    name: 'Github',
    icon: `<svg class="h-5 w-5" fill="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M12 .297c-6.63 0-12 5.373-12 12 0 5.303 3.438 9.8 8.205 11.385.6.113.82-.258.82-.577 0-.285-.01-1.04-.015-2.04-3.338.724-4.042-1.61-4.042-1.61C4.422 18.07 3.633 17.7 3.633 17.7c-1.087-.744.084-.729.084-.729 1.205.084 1.838 1.236 1.838 1.236 1.07 1.835 2.809 1.305 3.495.998.108-.776.417-1.305.76-1.605-2.665-.3-5.466-1.332-5.466-5.93 0-1.31.465-2.38 1.235-3.22-.135-.303-.54-1.523.105-3.176 0 0 1.005-.322 3.3 1.23.96-.267 1.98-.399 3-.405 1.02.006 2.04.138 3 .405 2.28-1.552 3.285-1.23 3.285-1.23.645 1.653.24 2.873.12 3.176.765.84 1.23 1.91 1.23 3.22 0 4.61-2.805 5.625-5.475 5.92.42.36.81 1.096.81 2.22 0 1.606-.015 2.896-.015 3.286 0 .315.21.69.825.57C20.565 22.092 24 17.592 24 12.297c0-6.627-5.373-12-12-12"/></svg>`,
  }),
  new MediaItem({
    key: 'twitter',
    path: 'https://www.twitter.com/pagelines',
    target: '_blank',
    name: 'Twitter',
    icon: `<svg class="h-5 w-5" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path fill="currentColor" d="M24 4.557c-.883.392-1.832.656-2.828.775 1.017-.609 1.798-1.574 2.165-2.724-.951.564-2.005.974-3.127 1.195-.897-.957-2.178-1.555-3.594-1.555-3.179 0-5.515 2.966-4.797 6.045-4.091-.205-7.719-2.165-10.148-5.144-1.29 2.213-.669 5.108 1.523 6.574-.806-.026-1.566-.247-2.229-.616-.054 2.281 1.581 4.415 3.949 4.89-.693.188-1.452.232-2.224.084.626 1.956 2.444 3.379 4.6 3.419-2.07 1.623-4.678 2.348-7.29 2.04 2.179 1.397 4.768 2.212 7.548 2.212 9.142 0 14.307-7.721 13.995-14.646.962-.695 1.797-1.562 2.457-2.549z"/></svg>`,
  }),
]

export const integrationsList = [
  new MediaItem({
    key: 'bigCommerce',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 132 30"> <path fill="#34313F" fill-rule="evenodd" d="M38.619 17.68c-.517-.702-1.396-1.275-2.46-1.275-1.972 0-3.393 1.504-3.393 3.542 0 2.039 1.421 3.543 3.393 3.543.92 0 1.782-.392 2.232-.784a.136.136 0 00.046-.104v-1.105a.126.126 0 00-.126-.126h-2.535a.183.183 0 01-.183-.183v-1.64c0-.101.082-.183.183-.183h4.546V16.93l-1.468.804a.187.187 0 01-.235-.055zM17.278 18.83h2.708c.775 0 1.26-.437 1.26-1.132 0-.664-.485-1.133-1.26-1.133h-2.708a.168.168 0 00-.168.168v1.93c0 .092.075.167.168.167zm11.01 6.353a.15.15 0 01-.15.15h-1.994a.15.15 0 01-.15-.15v-10.49a.15.15 0 01.15-.15h1.993a.15.15 0 01.15.15v10.49zM39.748.07L19.977 14.544h.51c2.053 0 3.103 1.31 3.103 2.75 0 1.12-.585 1.948-1.37 2.313a.163.163 0 00.008.3c.915.375 1.572 1.355 1.572 2.5 0 1.633-1.067 2.927-3.134 2.927h-5.682a.168.168 0 01-.168-.168v-6.844L.148 29.06a.36.36 0 00.213.652h39.6a.36.36 0 00.36-.36V23.92c-1.059.99-2.452 1.609-4.162 1.609-3.167 0-5.752-2.184-5.752-5.582 0-3.413 2.585-5.58 5.752-5.58 1.99 0 3.325.902 4.163 2.006V.36a.36.36 0 00-.573-.29zm-22.47 23.242h2.805c.856 0 1.373-.453 1.373-1.23 0-.678-.484-1.229-1.373-1.229h-2.805a.168.168 0 00-.168.168v2.123c0 .093.075.168.168.168zm34.536-.327l-1.236-.642a.177.177 0 00-.228.065c-.523.82-1.487 1.43-2.535 1.43-2.057 0-3.584-1.65-3.584-3.9 0-2.28 1.527-3.898 3.584-3.898 1.047 0 2.011.594 2.534 1.426.049.077.15.107.231.063l1.227-.669a.181.181 0 00.069-.257c-.776-1.223-2.061-2.246-4.06-2.246-3.071 0-5.456 2.25-5.456 5.581 0 3.317 2.385 5.582 5.455 5.582 1.998 0 3.268-1.05 4.07-2.273a.18.18 0 00-.07-.262zm6.442.853c-2.106 0-3.448-1.683-3.448-3.9 0-2.232 1.342-3.898 3.448-3.898 2.09 0 3.448 1.666 3.448 3.899 0 2.216-1.357 3.899-3.448 3.899zm-1.473-9.292c-2.317.62-3.847 2.708-3.847 5.393 0 3.22 2.2 5.581 5.32 5.581 2.657 0 4.663-1.73 5.186-4.247.852-4.1-2.743-7.774-6.659-6.727zm19.735-.01h-2.365c-.105 0-.2.064-.238.162l-2.86 7.404-2.877-7.405a.256.256 0 00-.238-.161h-2.366a.127.127 0 00-.126.127v10.536c0 .07.056.127.126.127h1.637c.07 0 .127-.057.127-.127v-8.204l3.29 8.274a.09.09 0 00.084.057h.685a.09.09 0 00.084-.057l3.274-8.274v8.204c0 .07.056.127.126.127h1.637c.07 0 .127-.057.127-.127V14.663a.127.127 0 00-.127-.127zm24.177 0H93.7a.195.195 0 00-.195.195v10.4c0 .108.087.195.195.195h6.995a.195.195 0 00.194-.195v-1.276a.195.195 0 00-.194-.195H95.59a.195.195 0 01-.195-.195v-2.62c0-.107.087-.195.195-.195h4.99a.195.195 0 00.196-.195V19.18a.195.195 0 00-.195-.195H95.59a.195.195 0 01-.195-.195v-2.392c0-.108.087-.195.195-.195h5.104a.195.195 0 00.194-.195V14.73a.195.195 0 00-.194-.195zm7.373 5.095h-2.408a.193.193 0 01-.193-.193v-3.043c0-.107.086-.193.193-.193h2.408c1.066 0 1.842.663 1.842 1.699 0 1.05-.776 1.73-1.842 1.73zm1.484 1.4c1.206-.304 2.297-1.338 2.297-3.114 0-2.006-1.406-3.381-3.539-3.381h-4.54a.193.193 0 00-.194.193v10.403c0 .107.087.194.193.194h1.504a.194.194 0 00.194-.194v-3.641c0-.107.086-.193.193-.193h1.587c.068 0 .131.036.166.094l2.263 3.839a.193.193 0 00.167.095h1.738a.193.193 0 00.164-.295l-2.32-3.71c-.07-.114-.004-.257.127-.29zm17.145-4.83h5.113a.19.19 0 00.19-.19v-1.285a.19.19 0 00-.19-.19h-7.004a.19.19 0 00-.19.19v10.41c0 .105.085.19.19.19h7.004a.19.19 0 00.19-.19V23.85a.19.19 0 00-.19-.19h-5.113a.19.19 0 01-.19-.19v-2.63a.19.19 0 01.19-.19h5a.19.19 0 00.19-.19v-1.285a.19.19 0 00-.19-.19h-5a.19.19 0 01-.19-.191v-2.402a.19.19 0 01.19-.19zm-4.01 6.739l-1.236-.642a.177.177 0 00-.228.065c-.523.82-1.487 1.43-2.535 1.43-2.057 0-3.585-1.65-3.585-3.9 0-2.28 1.528-3.898 3.585-3.898 1.047 0 2.01.594 2.534 1.426.048.077.15.107.23.063l1.227-.669a.181.181 0 00.069-.257c-.776-1.223-2.06-2.245-4.06-2.245-3.07 0-5.455 2.249-5.455 5.58 0 3.317 2.384 5.582 5.455 5.582 1.997 0 3.267-1.05 4.07-2.273a.18.18 0 00-.071-.262zm-32.21-8.404H88.11c-.105 0-.2.064-.238.162l-2.86 7.404-2.877-7.405a.256.256 0 00-.238-.161h-2.365a.127.127 0 00-.127.127v10.536c0 .07.057.127.127.127h1.637c.07 0 .127-.057.127-.127v-8.204l3.29 8.274a.09.09 0 00.084.057h.684a.09.09 0 00.084-.057l3.274-8.274v8.204c0 .07.056.127.127.127h1.636c.07 0 .127-.057.127-.127V14.663a.127.127 0 00-.127-.127z"/> </svg>`,
    path: '/docs/bigcommerce-installation',
  }),
  new MediaItem({
    key: 'elementor',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 163 27"><g id="e-text"><path id="Path" d="M102.054 12.939 C102.054 12.939 100.65 13.273 99.465 13.557 L97.66 13.968 C97.655 13.968 97.647 13.968 97.645 13.968 97.645 13.482 97.68 12.969 97.796 12.497 97.946 11.891 98.274 11.183 98.842 10.872 99.461 10.534 100.216 10.494 100.872 10.756 101.551 11.025 101.861 11.68 101.995 12.359 102.032 12.548 102.058 12.738 102.075 12.931 L102.054 12.939 Z M106.756 13.966 C106.756 9.259 103.792 7.235 100.006 7.235 95.724 7.235 93.043 10.201 93.043 13.989 93.043 18.108 95.325 20.79 100.241 20.79 102.899 20.79 104.405 20.32 106.192 19.425 L105.51 16.341 C104.145 16.953 102.876 17.329 101.181 17.329 99.324 17.329 98.264 16.623 97.865 15.305 L106.638 15.305 C106.709 14.953 106.756 14.554 106.756 13.966 Z" fill="#B74573" stroke="none"></path><path id="Path-1" d="M62.925 12.939 C62.925 12.939 61.521 13.273 60.336 13.557 L58.532 13.968 C58.526 13.968 58.518 13.968 58.516 13.968 58.516 13.482 58.551 12.969 58.667 12.497 58.817 11.891 59.145 11.183 59.713 10.872 60.332 10.534 61.087 10.494 61.743 10.756 62.422 11.025 62.732 11.68 62.866 12.359 62.903 12.548 62.929 12.738 62.947 12.931 L62.925 12.939 Z M67.627 13.966 C67.627 9.259 64.663 7.235 60.877 7.235 56.595 7.235 53.914 10.201 53.914 13.989 53.914 18.108 56.196 20.79 61.112 20.79 63.77 20.79 65.276 20.32 67.063 19.425 L66.381 16.341 C65.017 16.953 63.747 17.329 62.052 17.329 60.195 17.329 59.135 16.623 58.736 15.305 L67.509 15.305 C67.58 14.953 67.627 14.554 67.627 13.966 Z" fill="#B74573" stroke="none"></path><path id="Path-2" d="M52.311 3.776 L47.967 3.776 47.967 20.32 52.311 20.32 52.311 3.776 Z" fill="#B74573" stroke="none"></path><path id="Path-3" d="M107.674 7.658 L112.238 7.658 113.198 10.583 C113.799 9.139 115.151 7.281 117.55 7.281 120.842 7.281 122.631 8.951 122.631 13.258 L122.631 20.319 118.069 20.319 C118.069 18.847 118.071 17.376 118.072 15.905 118.072 15.231 118.061 14.556 118.071 13.881 118.078 13.258 118.122 12.615 117.791 12.056 117.567 11.679 117.2 11.401 116.803 11.212 115.997 10.827 115.124 10.839 114.334 11.25 114.139 11.35 113.199 11.859 113.199 12.095 L113.199 20.319 108.637 20.319 108.637 11.075 107.674 7.658 Z" fill="#B74573" stroke="none"></path><path id="Path-4" d="M126.324 10.976 L124.23 10.976 124.23 7.658 126.324 7.658 126.324 5.583 130.886 4.509 130.886 7.658 135.472 7.658 135.472 10.976 130.886 10.976 130.886 14.695 C130.886 16.154 131.592 16.837 132.65 16.837 133.731 16.837 134.344 16.695 135.26 16.39 L135.801 19.826 C134.554 20.367 133.002 20.626 131.425 20.626 128.109 20.626 126.32 19.049 126.32 15.991 L126.32 10.976 126.324 10.976 Z" fill="#B74573" stroke="none"></path><path id="Path-5" d="M144.252 17.026 C145.921 17.026 146.91 15.826 146.91 13.895 146.91 11.964 145.968 10.858 144.323 10.858 142.652 10.858 141.689 11.964 141.689 13.966 141.689 15.848 142.631 17.026 144.252 17.026 Z M144.299 7.165 C148.581 7.165 151.708 9.847 151.708 14.013 151.708 18.202 148.581 20.72 144.252 20.72 139.947 20.72 136.889 18.131 136.889 14.013 136.891 9.847 139.926 7.165 144.299 7.165 Z" fill="#B74573" stroke="none"></path><path id="Path-6" d="M88.954 7.619 C88.146 7.287 87.232 7.163 86.361 7.288 85.917 7.353 85.483 7.481 85.08 7.68 83.975 8.227 83.112 9.472 82.648 10.583 82.345 9.304 81.459 8.152 80.165 7.619 79.357 7.287 78.443 7.163 77.573 7.288 77.128 7.353 76.694 7.481 76.291 7.68 75.188 8.225 74.327 9.466 73.863 10.573 L73.863 10.492 72.934 7.658 68.371 7.658 69.332 11.075 69.332 20.32 73.865 20.32 73.865 12.06 C73.881 11.999 74.084 11.885 74.119 11.857 74.65 11.48 75.275 11.09 75.939 11.041 76.617 10.99 77.286 11.336 77.691 11.875 77.734 11.934 77.775 11.993 77.813 12.056 78.145 12.615 78.1 13.258 78.092 13.881 78.084 14.556 78.095 15.23 78.094 15.905 78.092 17.376 78.09 18.847 78.09 20.318 L82.654 20.318 82.654 13.258 C82.654 13.222 82.654 13.187 82.654 13.151 L82.654 12.062 C82.664 12.003 82.872 11.883 82.91 11.857 83.44 11.48 84.065 11.09 84.73 11.041 85.408 10.99 86.076 11.336 86.481 11.875 86.525 11.934 86.566 11.993 86.603 12.056 86.935 12.615 86.89 13.258 86.884 13.881 86.876 14.556 86.888 15.23 86.886 15.905 86.884 17.376 86.882 18.847 86.882 20.318 L91.445 20.318 91.445 13.258 C91.443 11.189 91.146 8.522 88.954 7.619 Z" fill="#B74573" stroke="none"></path><path id="Path-7" d="M162.503 7.283 C160.105 7.283 158.752 9.141 158.151 10.585 L157.189 7.66 152.625 7.66 153.586 11.077 153.586 20.322 158.151 20.322 158.151 11.779 C158.801 11.665 162.33 12.314 163 12.56 L163 7.298 C162.837 7.289 162.672 7.283 162.503 7.283 Z" fill="#B74573" stroke="none"></path><path id="Path-8" d="M41.663 12.516 C41.663 12.516 40.26 12.851 39.074 13.134 L37.27 13.545 C37.264 13.545 37.256 13.545 37.254 13.545 37.254 13.059 37.29 12.546 37.405 12.074 37.555 11.468 37.883 10.76 38.451 10.449 39.07 10.111 39.825 10.071 40.482 10.333 41.16 10.603 41.471 11.257 41.604 11.936 41.642 12.125 41.667 12.316 41.685 12.508 L41.663 12.516 Z M46.367 13.543 C46.367 8.836 43.403 6.813 39.617 6.813 35.335 6.813 32.654 9.778 32.654 13.566 32.654 17.685 34.937 20.368 39.853 20.368 42.51 20.368 44.016 19.897 45.803 19.003 L45.121 15.919 C43.757 16.53 42.487 16.906 40.792 16.906 38.935 16.906 37.875 16.2 37.476 14.882 L46.249 14.882 C46.32 14.532 46.367 14.131 46.367 13.543 Z" fill="#B74573" stroke="none"></path></g><path fill-rule="evenodd" clip-rule="evenodd" d="M0 13.2084C0 20.4107 5.83624 26.2501 13.0347 26.2501C20.2332 26.2501 26.0695 20.4107 26.0695 13.2084C26.0695 6.00609 20.2332 0.166748 13.0347 0.166748C5.83624 0.166748 0 6.00609 0 13.2084ZM9.77554 7.77424H7.60342V18.6426H9.77554V7.77424ZM11.9477 7.77424H18.4641V9.94753H11.9477V7.77424ZM18.4641 12.1208H11.9477V14.2941H18.4641V12.1208ZM11.9477 16.4693H18.4641V18.6426H11.9477V16.4693Z" fill="#B74573"></path></svg>`,
  }),
  new MediaItem({
    key: 'magento',
    icon: `<svg class="h-12" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 179.07 60.13"> <path d="M21.43 0L0 12.37v24.72l6.12 3.53-.04-24.71 15.31-8.85 15.31 8.85v24.7l6.12-3.52V12.35L21.43 0z" fill="#f26322" /> <path d="M24.47 40.62l-3.06 1.77-3.07-1.76V15.91l-6.12 3.53.01 24.71 9.18 5.3 9.18-5.3V19.44l-6.12-3.53v24.71z" fill="#f26322" /> <path fill="#4d4d4d" d="M56.84 12.52l8.41 21.26h.07l8.21-21.26h3.2v24.36h-2.21V15.66h-.07c-.11.38-.24.77-.37 1.15l-.4 1.06-.39 1.13-7.05 17.88h-2L57.11 19c-.14-.32-.27-.66-.41-1s-.26-.73-.37-1.07l-.41-1.19h-.07v21.14h-2.11V12.52zM83.17 37a5.12 5.12 0 01-1.82-.92 4.33 4.33 0 01-1.21-1.53 4.83 4.83 0 01-.45-2.15 5.06 5.06 0 01.58-2.55 4.51 4.51 0 011.57-1.59 8.06 8.06 0 012.3-.9 26.88 26.88 0 012.86-.5c.86-.09 1.6-.18 2.2-.29a6.44 6.44 0 001.46-.39 1.75 1.75 0 00.82-.63 2 2 0 00.26-1.07v-.34a3.24 3.24 0 00-1.09-2.68 5 5 0 00-3.14-.86q-4.69 0-4.94 3.91h-2.13a5.8 5.8 0 011.9-4.09 7.48 7.48 0 015.12-1.53A7.38 7.38 0 0192 20.17a4.92 4.92 0 011.67 4.17v9.58a1.77 1.77 0 00.33 1.14 1.11 1.11 0 00.86.39 1.47 1.47 0 00.36 0 2.89 2.89 0 00.43-.1h.1v1.54a4 4 0 01-.58.18 3.4 3.4 0 01-.85.09 2.68 2.68 0 01-1.84-.6 2.62 2.62 0 01-.75-1.89v-.37h-.1a10.13 10.13 0 01-.9 1.05 5.88 5.88 0 01-1.28 1 7.28 7.28 0 01-1.75.65 8.57 8.57 0 01-2.25.27 7.91 7.91 0 01-2.28-.27m5.19-1.86A5.79 5.79 0 0090.22 34a4.8 4.8 0 001.13-1.62 4.7 4.7 0 00.37-1.82v-3.22a7.2 7.2 0 01-2.06.73c-.81.17-1.64.32-2.5.43s-1.45.23-2.08.36a5.1 5.1 0 00-1.65.59 3 3 0 00-1.11 1.11 3.61 3.61 0 00-.41 1.82 3.35 3.35 0 00.32 1.54 2.89 2.89 0 00.85 1 3.35 3.35 0 001.25.53 7.2 7.2 0 001.5.15 6.45 6.45 0 002.53-.46m10.99 6.59a4.69 4.69 0 01-2-3.28h2a2.88 2.88 0 001.65 2.19 7.38 7.38 0 003.28.67A5.69 5.69 0 00108.5 40a5 5 0 001.34-3.69v-2.8h-.1a7.28 7.28 0 01-2.28 2.28 6.24 6.24 0 01-3.31.79 7.39 7.39 0 01-3.13-.65 6.83 6.83 0 01-2.38-1.8 8.12 8.12 0 01-1.52-2.78 11.51 11.51 0 01-.53-3.56A11.09 11.09 0 0197.2 24a8.53 8.53 0 011.64-2.78 6.87 6.87 0 012.4-1.7 7.56 7.56 0 012.91-.58 6.23 6.23 0 013.38.84 6.93 6.93 0 012.21 2.22h.1v-2.73h2.08V36.2a6.77 6.77 0 01-1.63 4.73 7.77 7.77 0 01-5.9 2.12 8.31 8.31 0 01-5-1.32m9.23-8.9a8.58 8.58 0 001.4-5.11 11.19 11.19 0 00-.34-2.87 6.32 6.32 0 00-1.06-2.23 4.89 4.89 0 00-1.79-1.45 5.75 5.75 0 00-2.5-.51 4.77 4.77 0 00-4.07 1.95 8.41 8.41 0 00-1.45 5.17 11.08 11.08 0 00.34 2.78 6.72 6.72 0 001 2.21 4.83 4.83 0 001.71 1.47 5.21 5.21 0 002.42.53 5 5 0 004.32-1.94m10.64 3.79a7.09 7.09 0 01-2.57-1.94 8.61 8.61 0 01-1.58-2.93 11.61 11.61 0 01-.55-3.66 11.32 11.32 0 01.58-3.66 9.23 9.23 0 011.62-2.93 7.17 7.17 0 012.52-1.94 7.65 7.65 0 013.32-.7 7.18 7.18 0 013.56.8 6.57 6.57 0 012.35 2.15 8.76 8.76 0 011.28 3.08 16.92 16.92 0 01.38 3.58h-13.43a10.57 10.57 0 00.43 2.77 7.08 7.08 0 001.12 2.23 5.06 5.06 0 001.86 1.5 5.84 5.84 0 002.62.55 5 5 0 003.43-1.08 5.88 5.88 0 001.72-2.87h2a7.6 7.6 0 01-2.38 4.19 7.12 7.12 0 01-4.81 1.56 8.37 8.37 0 01-3.51-.7m8.32-12.26a6.13 6.13 0 00-1-2 4.6 4.6 0 00-1.64-1.31 5.33 5.33 0 00-2.35-.47 5.64 5.64 0 00-2.37.47 5.2 5.2 0 00-1.75 1.31 6.43 6.43 0 00-1.14 2 9.84 9.84 0 00-.56 2.46h11.17a9.51 9.51 0 00-.37-2.44m7.37-5.11V22h.12a7.23 7.23 0 012.33-2.2 6.81 6.81 0 013.56-.87 5.81 5.81 0 014 1.42 5.19 5.19 0 011.58 4.07v12.5h-2.11V24.58a3.75 3.75 0 00-1.07-2.95 4.37 4.37 0 00-2.95-.93 5.89 5.89 0 00-2.11.37 5.63 5.63 0 00-1.74 1 4.79 4.79 0 00-1.6 3.6v11.21h-2.11V19.27zm17.03 17.01a2.93 2.93 0 01-.92-2.43V21h-2.66v-1.73H151v-5.42h2.11v5.42h3.27V21h-3.28v12.51a1.67 1.67 0 00.43 1.31 1.94 1.94 0 001.28.36 3.35 3.35 0 00.76-.09 3.22 3.22 0 00.63-.19h.11v1.81a6 6 0 01-1.95.31 3.81 3.81 0 01-2.45-.74m10.72.34a7.3 7.3 0 01-2.58-1.94 8.67 8.67 0 01-1.61-2.93 12 12 0 010-7.32 8.67 8.67 0 011.61-2.93 7.3 7.3 0 012.58-1.94 8.68 8.68 0 016.83 0A7.28 7.28 0 01172 21.5a8.32 8.32 0 011.6 2.93 12.45 12.45 0 010 7.32 8.32 8.32 0 01-1.6 2.93 7.28 7.28 0 01-2.55 1.94 8.68 8.68 0 01-6.83 0m6-1.65a5.24 5.24 0 001.87-1.6 7.28 7.28 0 001.15-2.37 11.55 11.55 0 000-5.8 7.28 7.28 0 00-1.13-2.38 5.24 5.24 0 00-1.87-1.6 6.12 6.12 0 00-5.2 0 5.42 5.42 0 00-1.87 1.6 7.1 7.1 0 00-1.14 2.38 11.26 11.26 0 000 5.8 7.1 7.1 0 001.14 2.38 5.42 5.42 0 001.87 1.6 6.12 6.12 0 005.2 0m8.22-12.79a2.13 2.13 0 01-2.22-2.26 2.16 2.16 0 012.22-2.28 2.13 2.13 0 012.21 2.28 2.14 2.14 0 01-2.21 2.26m0-4.28a1.86 1.86 0 00-1.9 2 1.9 1.9 0 103.8 0 1.86 1.86 0 00-1.9-2m.65 3.23l-.75-1.08h-.24v1h-.43v-2.47h.76c.52 0 .88.26.88.73a.68.68 0 01-.51.69l.72 1zm-.66-2.14h-.33v.74h.31c.28 0 .46-.12.46-.37s-.16-.37-.44-.37M56 53.8l-1.07 3a.11.11 0 01-.11.09h-1q-.12 0-.09-.12l3.22-9c0-.06 0-.12.06-.2a2.22 2.22 0 000-.4.07.07 0 01.08-.08h1.44a.09.09 0 01.09.08l3.47 9.58c0 .07 0 .1-.06.1H61a.13.13 0 01-.12-.07l-1.07-3zm3.41-1.14c-.08-.23-.19-.56-.34-1s-.29-.84-.45-1.28-.3-.87-.44-1.29-.24-.75-.31-1c-.06.23-.15.53-.26.88s-.24.72-.39 1.13l-.44 1.27c-.15.44-.3.86-.43 1.26zm3.99-1.3v-1.57c0-.05 0-.09.08-.09h.92a.14.14 0 01.13.09 1.29 1.29 0 010 .29v.44a3.09 3.09 0 012.26-1 3 3 0 01.82.12 1.91 1.91 0 01.77.43A2.38 2.38 0 0169 51a3.85 3.85 0 01.23 1.43v4.37c0 .07 0 .1-.11.1h-1c-.07 0-.1 0-.1-.12v-4.26a2.33 2.33 0 00-.37-1.36 1.31 1.31 0 00-1.15-.52 2.2 2.2 0 00-1 .24 2.71 2.71 0 00-.82.66v5.23c0 .07 0 .1-.09.1h-1.11c-.06 0-.08 0-.08-.1zm12.66 2.44l-1.06 3a.12.12 0 01-.12.09h-1c-.08 0-.1 0-.08-.12l3.21-9c0-.06 0-.12.06-.2a2 2 0 000-.4s0-.08.07-.08h1.45s.07 0 .08.08l3.47 9.58c0 .07 0 .1-.06.1H81a.13.13 0 01-.12-.07l-1.07-3zm3.41-1.14c-.08-.23-.19-.56-.34-1l-.44-1.28c-.16-.45-.31-.87-.45-1.29s-.24-.75-.31-1c-.06.23-.15.53-.26.88s-.24.72-.38 1.13-.3.83-.45 1.27-.29.86-.43 1.26zM86.52 57a4.42 4.42 0 01-1.43-.23 3.08 3.08 0 01-1.93-1.85 4.12 4.12 0 01-.28-1.6 3.89 3.89 0 01.28-1.5 3.75 3.75 0 01.84-1.19 3.84 3.84 0 011.27-.79 4.56 4.56 0 011.64-.29 3.74 3.74 0 01.52 0l.41.06v-3.23a.08.08 0 01.09-.09H89c.06 0 .09 0 .09.07V56a2.4 2.4 0 000 .37.1.1 0 01-.07.1 5.33 5.33 0 01-1.26.42 5.74 5.74 0 01-1.24.11zm1.3-6.26a1.58 1.58 0 00-.43-.11 2.93 2.93 0 00-.53 0 3.08 3.08 0 00-1.13.2 2.42 2.42 0 00-.86.56 2.54 2.54 0 00-.55.85 3.07 3.07 0 00-.19 1.06 3.19 3.19 0 00.21 1.23 2.15 2.15 0 001.36 1.28 3 3 0 00.94.15 4.26 4.26 0 00.61 0 1.71 1.71 0 00.57-.15zM94 49.55a3.35 3.35 0 011.41.28 3.13 3.13 0 011 .78 3.52 3.52 0 01.67 1.18 4.46 4.46 0 01.24 1.48 4.39 4.39 0 01-.25 1.53 3.36 3.36 0 01-.7 1.18 2.91 2.91 0 01-1.06.76A3.18 3.18 0 0194 57a3.28 3.28 0 01-1.52-.33 3.07 3.07 0 01-1.06-.85 3.73 3.73 0 01-.61-1.21 4.74 4.74 0 01-.2-1.35 4.38 4.38 0 01.24-1.46 3.42 3.42 0 01.68-1.19 3 3 0 011.07-.79 3.3 3.3 0 011.4-.27zm0 1.06a2 2 0 00-.86.19 1.92 1.92 0 00-.66.54 2.55 2.55 0 00-.42.84 3.29 3.29 0 00-.16 1.07 3.62 3.62 0 00.1 1.06 2.76 2.76 0 00.42.86 1.9 1.9 0 00.68.58 2.16 2.16 0 001.7 0 1.84 1.84 0 00.67-.49 2.6 2.6 0 00.47-.82 3.51 3.51 0 00.18-1.19 4.09 4.09 0 00-.13-1 2.63 2.63 0 00-.38-.85 1.94 1.94 0 00-1.61-.79zm6-4.32h.08v3.54a4.38 4.38 0 011.6-.29 3.66 3.66 0 011.39.26 3.11 3.11 0 011.06.72 3.4 3.4 0 01.68 1.09 4.48 4.48 0 01-.11 3.13 3.8 3.8 0 01-.92 1.26 3.73 3.73 0 01-1.31.76A4.84 4.84 0 01101 57a7 7 0 01-1 0 4.72 4.72 0 01-1-.23c-.06 0-.09 0-.09-.13V46.38c0-.06 0-.09.11-.09zm3.81 6.82a2.58 2.58 0 00-.62-1.81 2.17 2.17 0 00-1.68-.67 4.5 4.5 0 00-.8.06 3.63 3.63 0 00-.61.19v5a1.63 1.63 0 00.45.09 4.27 4.27 0 00.53 0 3 3 0 001-.17 2.58 2.58 0 00.87-.53 2.52 2.52 0 00.61-.9 3.19 3.19 0 00.29-1.26zm3.68.32a3.12 3.12 0 00.15 1 2.2 2.2 0 00.46.8 2.25 2.25 0 00.77.53 2.83 2.83 0 001.09.19 4.32 4.32 0 001-.1 3.46 3.46 0 00.89-.3s.08 0 .08.06v.84a.16.16 0 01-.08.15 3.94 3.94 0 01-.88.31 5.11 5.11 0 01-1.2.12 3.74 3.74 0 01-1.62-.32 2.91 2.91 0 01-1.07-.84 3.36 3.36 0 01-.59-1.18 5 5 0 01-.19-1.34 4.7 4.7 0 01.23-1.46 3.6 3.6 0 01.64-1.21 3.13 3.13 0 011-.82 3.08 3.08 0 011.38-.3 2.88 2.88 0 011.34.29 2.62 2.62 0 01.88.72 2.76 2.76 0 01.5 1 3.62 3.62 0 01.15 1v.47a1.29 1.29 0 010 .29c0 .06 0 .09-.1.09a4.62 4.62 0 01-.6.05h-4.14zm2.76-.91h.97a.19.19 0 000-.08v-.09a2.11 2.11 0 00-.07-.49 1.7 1.7 0 00-.27-.58 1.79 1.79 0 00-.54-.48 1.61 1.61 0 00-.86-.2 1.84 1.84 0 00-.8.16 1.72 1.72 0 00-.59.44 2.1 2.1 0 00-.39.62 3.12 3.12 0 00-.18.73zm13.65 4.07c0 .09 0 .14-.07.15a3 3 0 01-.83.26 9.47 9.47 0 01-1.12.06 6.23 6.23 0 01-1.82-.28 4.42 4.42 0 01-1.6-.89 4.36 4.36 0 01-1.13-1.56 5.58 5.58 0 01-.44-2.32 5.35 5.35 0 01.37-2 4.47 4.47 0 012.67-2.6A5.83 5.83 0 01122 47a7.72 7.72 0 011.2.07 2.78 2.78 0 01.69.19.11.11 0 01.07.12v1c0 .07 0 .09-.1.07a4.19 4.19 0 00-1.89-.34 4.28 4.28 0 00-1.59.28 3.71 3.71 0 00-1.23.77 3.39 3.39 0 00-.79 1.22 4.3 4.3 0 00-.29 1.6 4.21 4.21 0 00.39 1.92 3.43 3.43 0 002.23 1.85 4.94 4.94 0 001.25.19 7.36 7.36 0 001-.06 3.21 3.21 0 00.85-.21c.06 0 .08 0 .08.06zm4.34-7.04a3.35 3.35 0 011.41.28 3.13 3.13 0 011 .78 3.52 3.52 0 01.67 1.18 4.46 4.46 0 01.24 1.48 4.39 4.39 0 01-.25 1.53 3.36 3.36 0 01-.7 1.18 2.91 2.91 0 01-1.06.76 3.18 3.18 0 01-1.36.28 3.25 3.25 0 01-1.52-.33 3.07 3.07 0 01-1.06-.85 3.73 3.73 0 01-.61-1.21 4.74 4.74 0 01-.2-1.35 4.38 4.38 0 01.24-1.46 3.57 3.57 0 01.67-1.19 3.2 3.2 0 011.07-.79 3.38 3.38 0 011.46-.29zm-.06 1.06a2 2 0 00-.85.19 1.92 1.92 0 00-.66.54 2.77 2.77 0 00-.43.84 3.86 3.86 0 000 2.13 2.76 2.76 0 00.42.86 1.9 1.9 0 00.68.58 2.16 2.16 0 001.7 0 1.84 1.84 0 00.67-.49 2.6 2.6 0 00.47-.82 3.52 3.52 0 00.17-1.19 4.09 4.09 0 00-.12-1 2.63 2.63 0 00-.38-.85 1.94 1.94 0 00-1.67-.84zm5.05 6.26c-.06 0-.09 0-.09-.1V49.79c0-.13 0-.09.08-.09h.93a.13.13 0 01.12.07 1.94 1.94 0 01.05.33v.38a3.16 3.16 0 012.21-.93 2.09 2.09 0 012 1.08 4.5 4.5 0 011.06-.77 2.92 2.92 0 011.4-.31 3 3 0 01.84.12 2 2 0 01.76.45 2.39 2.39 0 01.57.86 3.72 3.72 0 01.22 1.39v4.41c0 .06 0 .09-.1.09h-1c-.07 0-.1 0-.1-.1v-4.31a2.27 2.27 0 00-.36-1.29 1.31 1.31 0 00-1.15-.53 2.24 2.24 0 00-1 .23 2.54 2.54 0 00-.8.63v5.27c0 .07 0 .1-.1.1h-1a.09.09 0 01-.1-.1v-4.36a2 2 0 00-.38-1.3 1.32 1.32 0 00-1.09-.47 2.2 2.2 0 00-.94.2 2.91 2.91 0 00-.85.63v5.31a.08.08 0 01-.09.09zm14.94-7.32a3.71 3.71 0 011.45.27 3.08 3.08 0 011.09.73 3.24 3.24 0 01.68 1.1 4.16 4.16 0 01.23 1.38 4 4 0 01-.36 1.76 3.64 3.64 0 01-1 1.25 4 4 0 01-1.36.73 4.78 4.78 0 01-1.54.25L147 57h-.33v2.95a.09.09 0 01-.08.1h-1.07c-.05 0-.08 0-.08-.09V50.2a.06.06 0 010-.06c.26-.1.62-.22 1.1-.37a5.45 5.45 0 011.63-.22zm-.17 1.08a4 4 0 00-.72.06 3.45 3.45 0 00-.65.17v5a1.76 1.76 0 00.41.07h.43a3.48 3.48 0 001-.16 2.88 2.88 0 00.92-.52 2.62 2.62 0 00.66-.89 3.09 3.09 0 00.25-1.27 2.62 2.62 0 00-.61-1.81 2.23 2.23 0 00-1.69-.65zm10.23 4.99V56a2 2 0 000 .34.06.06 0 010 .06 5.63 5.63 0 01-1.45.51 6.47 6.47 0 01-1.31.13 3.83 3.83 0 01-1.09-.15 2.55 2.55 0 01-.87-.44 1.92 1.92 0 01-.57-.72 2.31 2.31 0 01-.21-1 1.83 1.83 0 01.26-1 2.13 2.13 0 01.7-.7 3.15 3.15 0 011-.42 6.56 6.56 0 011.29-.13H157v-.35a1.91 1.91 0 00-.06-.43 1.21 1.21 0 00-.24-.48 1.38 1.38 0 00-.5-.4 2 2 0 00-.87-.17 4.17 4.17 0 00-1 .12 4 4 0 00-.86.32c-.07 0-.11 0-.11-.07v-.9a.14.14 0 01.08-.12 4.16 4.16 0 011-.32 5.22 5.22 0 011.13-.12 3 3 0 011.22.22 2.25 2.25 0 01.83.6 2.4 2.4 0 01.47.88 3.69 3.69 0 01.15 1.07zM157 53.45a3.34 3.34 0 00-.4-.05h-.58a3 3 0 00-1.6.34 1.11 1.11 0 00-.52 1 1.37 1.37 0 00.09.49 1 1 0 00.3.42 1.5 1.5 0 00.55.29 2.43 2.43 0 00.82.11 3.44 3.44 0 001.34-.25zm3.3-2.09v-1.29a1.6 1.6 0 000-.28.08.08 0 01.09-.09h.91a.15.15 0 01.14.09 1.43 1.43 0 010 .29c0 .13 0 .28.05.44a3.09 3.09 0 012.25-1 3 3 0 01.82.12 2 2 0 01.78.43 2.37 2.37 0 01.58.87 3.85 3.85 0 01.23 1.43v4.37c0 .07 0 .1-.1.1h-1c-.07 0-.1 0-.1-.12v-4.2a2.26 2.26 0 00-.37-1.36 1.31 1.31 0 00-1.15-.52 2.23 2.23 0 00-1 .24 2.71 2.71 0 00-.82.66v5.23c0 .07 0 .1-.08.1h-1.05c-.06 0-.09 0-.09-.1zm8.06-1.66a.1.1 0 01.1.07c.31.82.57 1.54.81 2.16s.43 1.15.59 1.6.31.83.41 1.13l.23.7c.23-.67.51-1.46.83-2.35s.71-2 1.16-3.24a.13.13 0 01.12-.07h1c.07 0 .09 0 .06.1q-.42 1.2-.9 2.49c-.32.86-.65 1.72-1 2.6s-.7 1.73-1.06 2.59-.71 1.67-1.07 2.46a.18.18 0 01-.16.1l-1.11.09c-.06 0-.07 0 0-.12l.71-1.52q.38-.85.75-1.74a.21.21 0 000-.19l-.27-.65-.4-1-.5-1.22c-.17-.43-.35-.88-.54-1.33s-.36-.9-.53-1.34l-.49-1.24c0-.06 0-.09.07-.09z"/> </svg>`,
  }),
  new MediaItem({
    key: 'hubspot',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 114 32" > <path d="M12.311 5.107v9.09H3.963v-9.09H0v21.699h3.963V18.01h8.348v8.795h3.964V5.107H12.31zm16.648 14.685a3.279 3.279 0 01-6.558 0v-9.294h-3.76v9.294a7.032 7.032 0 0014.064 0v-9.294h-3.746v9.294zm27.854-8.337c0-1.905 1.26-2.509 2.641-2.509 1.113 0 2.583.847 3.544 1.874l2.462-2.9c-1.229-1.662-3.722-2.812-5.763-2.812-4.078 0-7.028 2.389-7.028 6.348 0 7.346 8.98 5.016 8.98 9.127 0 1.268-1.232 2.388-2.642 2.388-2.223 0-2.944-1.087-3.965-2.237l-2.732 2.841c1.748 2.146 3.903 3.235 6.486 3.235 3.874 0 6.99-2.417 6.99-6.196 0-8.156-8.979-5.622-8.979-9.158h.006zm55.549 11.884c-2.221 0-2.852-.96-2.852-2.433v-6.515h3.452v-3.3h-3.452V6.733l-3.812 1.711v13.271c0 3.394 2.342 5.105 5.553 5.105a7.68 7.68 0 001.503-.12l.931-3.424c-.42.03-.902.058-1.323.058v.004zM43.269 10.607c-1.863 0-3.162.54-4.418 1.773V5.237h-3.765v13.238c0 4.956 3.582 8.35 7.608 8.35 4.466 0 8.395-3.455 8.395-8.107 0-4.595-3.616-8.11-7.82-8.11v-.002zm-.024 12.43a4.272 4.272 0 110-8.544 4.272 4.272 0 010 8.543zm40.908-4.517c0-4.66-3.921-8.108-8.395-8.108-4.025 0-7.608 3.395-7.608 8.35v13.246h3.765v-7.152c1.254 1.231 2.555 1.773 4.416 1.773 4.204 0 7.82-3.513 7.82-8.11h.002zm-3.566-.047a4.272 4.272 0 11-8.545 0 4.272 4.272 0 018.545 0z" fill="#425B76" /> <path d="M96.28 10.352V6.608a2.882 2.882 0 001.662-2.598v-.087a2.888 2.888 0 00-2.88-2.88h-.087a2.887 2.887 0 00-2.88 2.88v.087a2.882 2.882 0 001.662 2.598v3.744a8.167 8.167 0 00-3.884 1.709l-10.28-7.999c.073-.263.112-.536.117-.81a3.249 3.249 0 10-3.253 3.245 3.213 3.213 0 001.606-.436l10.107 7.868a8.19 8.19 0 00.127 9.236L85.22 26.24a2.64 2.64 0 00-.764-.124 2.67 2.67 0 102.67 2.67 2.63 2.63 0 00-.124-.763l3.043-3.043a8.204 8.204 0 106.234-14.628zm-1.26 12.315a4.21 4.21 0 110-8.419 4.21 4.21 0 010 8.42" fill="#FF7A59" /> </svg>`,
    path: '/docs/hubspot-installation',
  }),
  new MediaItem({
    key: 'wordpress',
    name: 'WordPress',
    icon: `<svg class="h-10" viewBox="0 0 142 32" fill="none" xmlns="http://www.w3.org/2000/svg" > <path d="M76.454 12.595V13.184C78.194 13.184 78.474 13.556 78.474 15.758V19.728C78.474 21.93 78.195 22.333 76.454 22.333C75.117 22.148 74.214 21.434 72.971 20.069L71.541 18.519C73.468 18.177 74.494 16.968 74.494 15.603C74.494 13.898 73.033 12.595 70.296 12.595H64.822V13.184C66.564 13.184 66.844 13.556 66.844 15.758V19.728C66.844 21.93 66.564 22.333 64.822 22.333V22.923H71.012V22.333C69.27 22.333 68.99 21.93 68.99 19.728V18.612H69.518L72.971 22.922H82.021C86.468 22.922 88.396 20.566 88.396 17.743C88.396 14.921 86.468 12.595 82.021 12.595H76.454ZM68.99 17.65V13.525H70.265C71.665 13.525 72.286 14.487 72.286 15.603C72.286 16.689 71.664 17.65 70.265 17.65H68.99ZM82.114 21.961H81.896C80.776 21.961 80.621 21.682 80.621 20.256V13.526H82.114C85.348 13.526 85.939 15.882 85.939 17.712C85.939 19.604 85.349 21.961 82.114 21.961ZM47.5 18.798L49.647 12.47C50.269 10.64 49.989 10.113 47.999 10.113V9.49302H53.845V10.113C51.885 10.113 51.42 10.579 50.642 12.843L47.096 23.45H46.692L43.52 13.742L40.286 23.45H39.881L36.43 12.843C35.684 10.579 35.186 10.113 33.383 10.113V9.49302H40.287V10.113C38.452 10.113 37.954 10.547 38.607 12.47L40.691 18.798L43.831 9.49302H44.423L47.502 18.798H47.5ZM58.074 23.326C54.654 23.326 51.854 20.814 51.854 17.743C51.854 14.703 54.654 12.191 58.074 12.191C61.495 12.191 64.294 14.703 64.294 17.743C64.294 20.813 61.495 23.326 58.074 23.326ZM58.074 13.153C55.214 13.153 54.218 15.727 54.218 17.743C54.218 19.79 55.213 22.333 58.074 22.333C60.967 22.333 61.962 19.79 61.962 17.743C61.962 15.727 60.967 13.153 58.074 13.153ZM96.077 22.302V22.922H88.955V22.302C91.039 22.302 91.412 21.775 91.412 18.673V13.711C91.412 10.609 91.039 10.113 88.955 10.113V9.49302H95.393C98.596 9.49302 100.369 11.136 100.369 13.339C100.369 15.479 98.596 17.154 95.393 17.154H93.62V18.673C93.62 21.775 93.993 22.302 96.077 22.302ZM95.393 10.609H93.62V16.006H95.393C97.135 16.006 97.943 14.796 97.943 13.339C97.943 11.85 97.135 10.609 95.393 10.609ZM121.734 20.007L121.578 20.565C121.299 21.589 120.956 21.961 118.748 21.961H118.313C116.696 21.961 116.416 21.589 116.416 19.387V17.961C118.842 17.961 119.028 18.177 119.028 19.791H119.619V15.169H119.028C119.028 16.782 118.842 16.999 116.416 16.999V13.525H118.126C120.335 13.525 120.676 13.898 120.956 14.921L121.112 15.51H121.609L121.392 12.595H112.248V13.184C113.99 13.184 114.27 13.556 114.27 15.758V19.728C114.27 21.743 114.032 22.25 112.659 22.323C111.354 22.123 110.458 21.413 109.232 20.069L107.802 18.519C109.73 18.177 110.756 16.968 110.756 15.603C110.756 13.898 109.295 12.595 106.558 12.595H101.084V13.184C102.826 13.184 103.106 13.556 103.106 15.758V19.728C103.106 21.93 102.826 22.333 101.084 22.333V22.923H107.273V22.333C105.532 22.333 105.252 21.93 105.252 19.728V18.612H105.781L109.233 22.922H122.014L122.201 20.008L121.734 20.007ZM105.252 17.65V13.525H106.526C107.926 13.525 108.548 14.487 108.548 15.603C108.548 16.689 107.926 17.65 106.526 17.65H105.252ZM128.047 23.326C126.803 23.326 125.714 22.674 125.248 22.271C125.093 22.426 124.813 22.891 124.751 23.326H124.16V19.046H124.782C125.031 21.093 126.461 22.302 128.296 22.302C129.291 22.302 130.1 21.744 130.1 20.814C130.1 20.008 129.385 19.387 128.109 18.798L126.337 17.961C125.093 17.371 124.16 16.347 124.16 14.983C124.16 13.494 125.559 12.223 127.487 12.223C128.514 12.223 129.384 12.595 129.913 13.029C130.069 12.905 130.224 12.564 130.318 12.191H130.908V15.851H130.255C130.037 14.393 129.229 13.184 127.581 13.184C126.71 13.184 125.901 13.68 125.901 14.456C125.901 15.262 126.554 15.696 128.047 16.379L129.757 17.216C131.251 17.929 131.841 19.077 131.841 20.008C131.841 21.961 130.131 23.326 128.047 23.326ZM137.626 23.326C136.381 23.326 135.293 22.674 134.827 22.271C134.671 22.426 134.391 22.891 134.329 23.326H133.738V19.046H134.36C134.609 21.093 136.039 22.302 137.874 22.302C138.87 22.302 139.678 21.744 139.678 20.814C139.678 20.008 138.963 19.387 137.688 18.798L135.915 17.961C134.671 17.371 133.738 16.347 133.738 14.983C133.738 13.494 135.137 12.223 137.066 12.223C138.092 12.223 138.963 12.595 139.492 13.029C139.647 12.905 139.802 12.564 139.895 12.191H140.486V15.851H139.834C139.616 14.393 138.807 13.184 137.159 13.184C136.288 13.184 135.479 13.68 135.479 14.456C135.479 15.262 136.133 15.696 137.626 16.379L139.336 17.216C140.829 17.929 141.419 19.077 141.419 20.008C141.419 21.961 139.709 23.326 137.626 23.326ZM2.28 15.999C2.28 21.432 5.446 26.127 10.037 28.352L3.472 10.412C2.68502 12.1697 2.27912 14.0742 2.281 16L2.28 15.999ZM25.335 15.306C25.335 13.61 24.725 12.435 24.2 11.521C23.502 10.39 22.848 9.43302 22.848 8.30102C22.848 7.04102 23.808 5.86602 25.16 5.86602C25.22 5.86602 25.279 5.87302 25.338 5.87602C22.8006 3.5534 19.4839 2.26763 16.044 2.27302C11.235 2.27302 7.005 4.73302 4.544 8.46002C4.867 8.47002 5.172 8.47702 5.43 8.47702C6.87 8.47702 9.098 8.30202 9.098 8.30202C9.84 8.25902 9.928 9.34602 9.186 9.43302C9.186 9.43302 8.441 9.52102 7.611 9.56302L12.623 24.431L15.635 15.423L13.491 9.56302C12.749 9.52002 12.047 9.43302 12.047 9.43302C11.306 9.38902 11.393 8.25802 12.134 8.30202C12.134 8.30202 14.407 8.47602 15.76 8.47602C17.199 8.47602 19.428 8.30202 19.428 8.30202C20.17 8.25802 20.258 9.34502 19.516 9.43202C19.516 9.43202 18.769 9.52002 17.941 9.56302L22.914 24.318L24.287 19.743C24.882 17.845 25.335 16.481 25.335 15.306Z" fill="#32373B" /> <path d="M16.285 17.2L12.155 29.166C14.9283 29.9811 17.8876 29.9045 20.615 28.947C20.5765 28.8871 20.5433 28.8238 20.516 28.758L16.285 17.2ZM28.12 9.41302C28.18 9.85102 28.213 10.32 28.213 10.825C28.213 12.218 27.953 13.783 27.167 15.741L22.962 27.863C27.054 25.483 29.806 21.063 29.806 15.999C29.806 13.613 29.196 11.369 28.12 9.41302Z" fill="#32373B" /> <path d="M16.044 -0.000976562C7.197 -0.000976562 0 7.17602 0 15.999C0 24.822 7.197 31.999 16.044 31.999C24.89 31.999 32.088 24.822 32.088 15.999C32.088 7.17602 24.89 -0.000976562 16.044 -0.000976562ZM16.044 31.266C7.604 31.266 0.736 24.416 0.736 15.999C0.736 7.58002 7.603 0.733023 16.044 0.733023C24.484 0.733023 31.351 7.58102 31.351 15.999C31.351 24.417 24.484 31.266 16.044 31.266Z" fill="#32373B" /> </svg>`,
    path: '/docs/wordpress-installation',
  }),
  new MediaItem({
    key: 'instapage',
    icon: `<svg class="h-9" viewBox="0 0 606.6 122.09" xmlns="http://www.w3.org/2000/svg"> <path d="M0,18.18A1.78,1.78,0,0,0,1.17,18c7.93-.47,8.4-.24,8.4,7.46v67.9c0,8.64-.94,9.57-9.57,8.17ZM35.23,60.41v-56c0-4,1.17-4.9,5.14-4.2,23.56,4,47.36,7.7,71.16,11.44,4,.7,5.14,2.1,5.14,6.06q-.36,42.71,0,85.4c0,4-1.4,5.37-5.14,6.07-23.56,4-47.36,7.93-70.93,12.13-4.9.94-5.13.7-5.13-4.43C35.23,98,35.23,79.31,35.23,60.41Zm12.84.94v41.76c0,3.74,1.16,4.67,4.9,4.2,16.56-2.56,33.36-4.66,49.93-7,2.8-.46,4-1.63,4-4.43V25.65c0-2.57-.7-4-3.5-4.2-16.8-1.87-33.37-4-50.17-6.07-4-.47-5.13.7-4.9,4.43C48.3,33.58,48.07,47.35,48.07,61.35Zm482.3,31.26c-15.4,9.57-29.17,8.87-38.5-2.33-8.87-10.73-8.87-29.4.46-39.9,9.8-11.2,23.1-11.67,38.5-1.63.7-1.64,1.17-4.9,1.87-4.9a37,37,0,0,1,8.63,1.4,3.76,3.76,0,0,1,1.17,2.56c0,16.1.7,32.2-.47,48.3s-12.36,25.44-28.46,25c-6.07-.23-12.14-2.33-18-3.5-2.57-.47-3.5-2.1-1.87-4.43,1.4-2.1.94-6.07,5.6-4.67,5.14,1.4,10.27,2.8,15.64,2.57,10.5,0,17.73-8.87,15.4-18.44Zm1.16-22.4c0-10.73-7.46-18.66-17.5-18.43-9.8.23-16.8,8.17-16.8,19.13,0,10.5,7.47,18.67,17.27,18.67C524.53,89.58,531.53,81.41,531.53,70.21ZM375,92.15v21.93c0,7.23-2.57,9.33-9.57,7.23-1.17-.23-2.1-3-2.1-4.66-.23-22.64,0-45.5,0-68.14V46.88c0-2.33.93-3.73,3.5-2.8,3,.93,7.7-.47,7.93,5.83,7-7,14.94-9.33,24.27-7.46a26.51,26.51,0,0,1,15.4,9.33c8.87,11.2,8.17,29.87-1.63,40.13C403.43,101.71,389.67,102.18,375,92.15Zm33.6-21.24c.23-10.5-6.3-18.43-15.87-19.13s-17,6.07-18.2,16.33c-1.17,11.2,4.67,19.84,14.7,21.24C400.17,90.75,408.33,83.05,408.57,70.91Zm155.16,2.34c-.46,10.73,9.57,18.43,20.77,16.1a26.59,26.59,0,0,0,10.27-5.14c4.2-3.26,5.6.7,7.93,2.1,2.8,1.87.23,3.5-.93,4.9-9.34,10-27.3,11.44-39.2,3-11.44-8.17-14.7-25-7.7-38.5,6.3-12.14,21-17.27,34.53-12.6,10.73,3.73,18.43,16.33,17,27.3-.46,3-2.33,2.8-4.43,2.8H563.73Zm.47-7.94h29.87c.23-7.93-5.6-13.76-13.77-14C571.2,50.85,564.67,56.68,564.2,65.31Zm-224,29.4c-10.73,7-23.33,6.3-31-1.4A17.34,17.34,0,0,1,311.73,67c7.47-4.9,15.87-5.37,27.54-.94,2.56-7.93-1.64-14.46-9.8-14.46a45.39,45.39,0,0,0-14.7,2.8c-4.44,1.63-4.67-1.87-6.07-4.2-1.63-2.57.23-3.5,2.1-4.44,8.4-4,17-5.36,25.9-3.5,8.4,1.87,14,7.47,14.47,16.1.46,12.14.46,24.5.46,36.64,0,1.16-1.4,3-1.86,3-2.8-.23-6.3,1.87-8.64-1.63C340.67,96.11,340.43,95.41,340.2,94.71Zm-12.37-3.5c2.8-.93,5.37-1.16,7.47-2.56,6.07-3.27,6.07-11.44.47-15.17-4.2-2.8-11.9-2.57-16.1,0-5.37,3.73-5.6,11.2,0,14.7C322.23,89.58,325.27,90.28,327.83,91.21Zm136,3.5c-7.94,6.3-16.34,6.3-24.74,2.8-7-3-11.2-8.4-11.2-16.33,0-7.7,4-13.07,11-15.87a25.93,25.93,0,0,1,21.23,0c.94.47,1.87.7,2.8,1.17,1.87-10-2.56-15.4-12.36-14.7a45.59,45.59,0,0,0-12.14,2.8c-4.43,1.87-4.66-1.63-6.06-4-1.87-2.8.23-3.73,2.33-4.66,7.7-3.5,15.4-5.14,23.8-3.74,10.5,1.64,16.33,7.7,16.57,18.2.23,11.44.23,22.87,0,34.54,0,1.16-1.17,3.26-1.87,3.26a52.68,52.68,0,0,1-8.4-1.16,5.08,5.08,0,0,0-.93-2.34ZM452.2,71.15c-2.33.46-4.2.7-5.83,1.16-4.44,1.64-6.77,4.9-6.54,9.1.24,4.44,2.8,7.47,6.77,8.17a30.53,30.53,0,0,0,12.13-.7c3.5-.93,5.14-4.43,5.14-8.17a8.62,8.62,0,0,0-5.14-7.93,64.88,64.88,0,0,0-6.53-1.63ZM177.57,49c5.6-6.3,12.13-8.63,19.83-7.23,9.33,1.63,15.17,8.63,15.63,20.06.47,11,.24,22.17.24,33.14,0,2.33-1.17,3.26-3.5,3.26-8.4.47-8.4.47-8.4-7.93V64.38c0-7.93-4.2-12.6-11-12.83-7.47-.24-12.6,4.66-12.83,12.6q-.36,14.7,0,29.4c0,3.5-1.17,4.66-4.67,4.66-7-.23-7,0-7-7V49.45c0-2.34-2.1-6.07,3-6.54C175.93,42.45,176.4,42.68,177.57,49Zm83.06,2.8c-1.86,1.4-3.5,3-5.6,3.5-1.16.47-3.26-.23-4.2-1.17-3.5-2.8-7.23-4.43-11.66-2.8-1.87.7-4.2,2.8-4.44,4.44-.23,1.86,1.17,4.66,2.8,6.06,1.4,1.64,4,2.1,6.07,3a79.88,79.88,0,0,1,10.27,4.43c11,6.07,11.2,20.53.46,27.07-9.1,5.6-22.63,4.2-29.86-3.5-1.4-1.4-4-3-1.4-5.6,3.5-3.5,5.13-4,8.16-1.17a14.3,14.3,0,0,0,14,3.73c2.1-.7,4.9-3,5.37-5.13s-1.4-5.37-3.27-6.53c-4.43-2.57-9.33-4.2-14-6.3-5.36-2.57-9.56-6.3-9.56-12.6a20.28,20.28,0,0,1,2.33-9.8c6.77-11.67,27.77-10.27,34.53,2.33ZM17.5,60.41V13.51c0-2.1-.7-4.66,2.8-4.66,5.13,0,6.77.93,6.77,4.66v93.1c0,5.14-3.5,4.2-6.54,4.67-3.26.7-3-2.1-3-4.43V60.41ZM284.9,43.15c3.27,0,5.6.23,8.17,0,3.5-.24,3.5,1.86,3.5,4.43s.23,5.13-3.5,4.9h-7.7c0,10.5-.24,20.53.23,30.57.23,4.2,3.73,6.06,7.7,5.83,4.67-.23,3.27,3.27,3.5,5.6.23,2.8-.47,4.43-4,4.67-11.9.7-18.66-5.14-19.36-17.5-.47-7.94-.24-15.87-.24-23.8V53c-6.53-1.17-7.23-1.64-6.3-7a4.93,4.93,0,0,1,3-3c2.8-.23,3.26-1.16,3.5-3.73.23-8.17.23-8.17,7.93-6.53,2.8.46,4,1.86,3.73,4.66A45.74,45.74,0,0,0,284.9,43.15ZM139.77,59.48V26.81c0-6.3,0-6.3,6.3-5.13.93.23,1.86,0,2.56.7,1.17,1.17,3,2.33,3,3.73.23,23.1.23,46,.23,69.07,0,2.33-.93,3.27-3.27,3.5-8.63.7-8.63.7-8.63-7.93C139.77,80,139.77,69.75,139.77,59.48Z" fill="#2D67F7" /> </svg>`,
    path: '/docs/instapage-installation',
  }),
  new MediaItem({
    key: 'squarespace',
    icon: `<svg class="h-8" viewBox="0 0 100 13.91" xmlns="http://www.w3.org/2000/svg">    <path      d="M35.93 6.69a3.62 3.62 0 10-7.24 0 3.56 3.56 0 003.61 3.65 3.52 3.52 0 002.24-.76l.93.85.55-.6-1-.78a3.66 3.66 0 00.91-2.36zm-1.45 1.86l-1.31-1.19-.54.6L34 9.08a2.64 2.64 0 01-1.63.54 2.83 2.83 0 01-2.8-2.93 2.79 2.79 0 012.73-2.93 2.83 2.83 0 012.8 2.93 3 3 0 01-.62 1.86zm8.14-1.28c0 1.55-.83 2.34-2.14 2.34s-2.17-.86-2.17-2.39V3.14h-.8v4.13a3 3 0 105.91-.06V3.14h-.8zm48 2.35a2.8 2.8 0 01-2.75-2.93 2.79 2.79 0 012.73-2.93 3.12 3.12 0 012.23 1l.55-.58A3.69 3.69 0 0090.61 3 3.57 3.57 0 0087 6.69a3.54 3.54 0 003.56 3.65 3.7 3.7 0 002.84-1.25l-.53-.51a3.08 3.08 0 01-2.27 1.04zm5-.11V7h3.87v-.7h-3.87V3.87H100v-.73h-5.18v7.1H100v-.73zM76.34 3.14h-2.66v7.1h.8v-2.5h1.73c1.51 0 2.78-.8 2.78-2.33C79 4 77.94 3.14 76.34 3.14zm1.84 2.3c0 .94-.77 1.57-1.94 1.57h-1.76V3.88h1.8c1.14 0 1.9.53 1.9 1.54zm-53.12.87C23.5 6 23.13 5.6 23.13 4.93c0-.64.59-1.15 1.51-1.15a3.11 3.11 0 012.05.77l.47-.62a3.75 3.75 0 00-2.5-.93 2.09 2.09 0 00-2.33 2c0 1.16.75 1.69 2.38 2.05s1.85.69 1.85 1.34c0 .7-.63 1.21-1.58 1.21a3.38 3.38 0 01-2.44-1l-.49.59A4.14 4.14 0 0025 10.34c1.41 0 2.42-.79 2.42-2-.05-1.11-.78-1.69-2.36-2.03zM61.2 7h3.87v-.7H61.2V3.87h4.33v-.73H60.4v7.1h5.18v-.73H61.2zm8.54-.72c-1.56-.28-1.93-.68-1.93-1.35 0-.64.59-1.15 1.51-1.15a3.11 3.11 0 012.05.77l.47-.62a3.75 3.75 0 00-2.5-.93A2.09 2.09 0 0067 5c0 1.16.75 1.69 2.38 2.05s1.85.69 1.85 1.34c0 .7-.63 1.21-1.58 1.21a3.38 3.38 0 01-2.44-1l-.49.59a4.14 4.14 0 002.9 1.13c1.41 0 2.42-.79 2.42-2 .01-1.09-.72-1.67-2.3-2.01zm-22-3.22l-3.24 7.15h.82l.85-1.88H50l.83 1.88h.86l-3.25-7.12zm-1.31 4.58L48.05 4l1.61 3.62zm35.94-4.55l-3.24 7.15H80l.84-1.88h3.86l.83 1.88h.86l-3.27-7.15zm-1.26 4.55L82.73 4l1.62 3.62zM58.69 5.27a2 2 0 00-.55-1.4 2.84 2.84 0 00-2-.71h-3v7.1h.8V7.48h2L58 10.24h1l-2.18-2.9a2.07 2.07 0 001.87-2.07zm-4.82 1.49V3.88h2.19c1.15 0 1.82.53 1.82 1.4 0 .91-.76 1.46-1.83 1.46zm-51.09.79a.84.84 0 001.22 0l5.34-5.37a1.69 1.69 0 012.39 0 .85.85 0 001.2 0 .84.84 0 000-1.19A3.38 3.38 0 008.15 1L2.78 6.36a.84.84 0 000 1.19zM11.14 4a.85.85 0 00-1.2-1.19L4.57 8.15a1.69 1.69 0 01-2.39-2.39L7.45.5a1.69 1.69 0 00-2.39 0L1 4.57a3.38 3.38 0 004.76 4.78zm.59.6L6.36 9.94a.85.85 0 000 1.2.84.84 0 001.19 0l5.38-5.38a1.69 1.69 0 112.38 2.39L10 13.42a1.69 1.69 0 002.39 0l4.08-4.07a3.38 3.38 0 00-4.78-4.78zm3 1.79a.85.85 0 00-1.2 0l-5.38 5.34a1.69 1.69 0 01-2.39 0 .84.84 0 00-1.19 0 .85.85 0 000 1.2 3.39 3.39 0 004.77 0l5.38-5.38a.84.84 0 000-1.19z"      fill="#000000"    />  </svg>`,
    path: '/docs/squarespace-installation',
  }),
  new MediaItem({
    key: 'weebly',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 580 201.77" > <path d="M333.24,23.87C334.84,12.51,343.9,2.45,355.35.54,366.6-1.78,379,3.65,384.5,13.8c4.7,7.71,4.15,17.07,4.06,25.76,14.77-2,30.21,1.8,41.85,11.26.12-8.51-.45-17.06.32-25.56C431.83,13.38,441.16,2.72,453,.61c11.67-2.55,24.68,3.07,30.3,13.72,4.62,7.86,3.82,17.24,3.79,26,8.17-.83,16.89.78,23.26,6.27,6.43,5.08,9.12,13.1,11.76,20.53,2.52-6.33,4.57-13.19,9.41-18.21,8.12-8.81,22.19-11.38,32.87-5.92C573,47,579.61,55.61,579.93,65.28c.61,7.73-2.88,14.85-5.71,21.81Q555,134.55,535.84,182c-2.53,6.3-6.7,12.13-12.54,15.75-9.34,6-22,4.91-31.1-1a25.08,25.08,0,0,1-11.3-23.05c.89-8.29,2.48-16.5,4.08-24.67-4.37,9.45-13.65,16.78-24.28,17.09-11.16,1.11-21.85-5.4-27.16-15-15,14.61-38.78,19.55-57.91,10.5-7.67,3.61-16.72,5-24.77,1.8-5.11-1.75-9.08-5.52-12.69-9.38-18.73,11.69-42.79,15.89-63.88,8.49a60.74,60.74,0,0,1-25.22-17c-13.07,14.66-33.51,20.69-52.64,20.67-21.23,0-42-11.6-52.53-30.16-3.11,9.88-7.44,20.4-16.8,25.88-11.26,6.8-27.16,5.6-36.73-3.66-3-2.81-5.08-6.41-7.15-9.92-2.82,5.56-6.69,10.8-12.27,13.84-11.42,6.46-27.22,5-36.65-4.35-4.82-4.61-7.76-10.75-9.93-17-7.06-20-14.25-40-21.3-60-3-8.55-4.86-18.5-.42-26.9,7.19-14.3,27.42-19.06,40.67-10.46,6.15,3.7,10.08,10.08,12.28,16.78,2.21-5.9,5.29-11.78,10.53-15.54,10.86-8.13,27.71-7.27,37.25,2.55,3.66,3.59,5.81,8.32,7.61,13,2.44-8,7.87-15.36,15.72-18.69,11.9-5.42,26.86-1.61,34.9,8.66A59.4,59.4,0,0,1,207,40.19c15.57,2.76,29.28,12.46,38.41,25.19,8.8-12.43,21.93-22.06,37-25a59.46,59.46,0,0,1,50.31,12.8c.19-9.78-.58-19.6.48-29.33Zm15.23,5.72c-.07,34.7,0,69.39,0,104.07,0,3.86.27,8,2.74,11.19,3.68,5.19,11.56,6.07,16.78,2.79,2.94-1.74,4-5.15,5.43-8,11.05,15.05,34.65,13.57,47.54,1.73,19.34-17,20.8-48.13,7.45-69.18-6.34-10.32-17.92-17.3-30.1-17.46-9.2-.52-18.11,3-25.43,8.38-.1-12,.36-24-.19-35.93-.21-7.15-7.33-13-14.39-11.25-6.42,1-10.13,7.61-9.79,13.7Zm105.1-12.84c-5.61,2.33-7.84,8.72-7.5,14.39.06,35.17-.07,70.34.06,105.49-.4,6.69,4.45,13.69,11.52,13.93,7.22,1,13.72-5.44,13.61-12.55.33-34.6,0-69.23.18-103.83,0-4.7.22-9.95-2.92-13.82-3.34-4.53-9.93-5.91-14.95-3.61ZM16,62.66c-1.12,6.21,1.73,12.2,3.63,18,6.92,19.27,13.66,38.61,20.63,57.87,2.09,6.06,7.06,11.83,13.94,11.95,6.87.92,13.44-3.77,15.34-10.3,4.75-14.07,9-28.29,13.75-42.37,4.35,13.22,8.56,26.48,12.82,39.74,1.44,4.42,3.51,9.17,7.87,11.4,5.35,2.64,12.49,2.27,17-1.89,3.25-3,4.93-7.18,6.41-11.22,7.25-20.59,14.69-41.12,21.94-61.71,1.7-4.56,2.52-10.1-.47-14.32-5.17-7.63-19-6.49-21.84,2.65-5.52,17-10.35,34.21-15.72,51.24-5.53-16.88-10.44-34-16.14-50.8C92.77,54.86,82,52.35,75.62,57c-3,2.21-4.27,5.87-5.41,9.25-4.95,15.8-9.83,31.61-14.89,47.38-5.15-16.51-10-33.11-15.12-49.62C38,52.39,19.47,51.62,16,62.66ZM165.1,68c-15.91,16.31-18.22,43.68-5.87,62.7a43.4,43.4,0,0,0,26.3,18.38c13.6,3.3,28.21.9,40.61-5.32,4.66-2.5,9.48-5.57,11.9-10.45,3.6-7-4.08-16.4-11.63-14-5.88,2-10.85,6-16.73,8-8.77,3.17-19.5,2-26.69-4.14-3.6-3-5.12-7.53-6.39-11.85,18.49-.23,37,.24,55.49-.21a8.92,8.92,0,0,0,8.82-8.74c1.21-22.64-17-44.89-39.75-47.31A42.87,42.87,0,0,0,165.1,68Zm102.78-4c-15.56,12.28-21.51,34.33-16.21,53.17,3.17,12.17,11.57,23.05,23,28.49,15.52,7.61,34.33,5.73,49.52-1.75,4.95-2.63,10.19-5.88,12.53-11.22,3-7-4.64-15.89-12-13.45-5.23,1.85-9.74,5.25-14.86,7.32-8.89,3.91-20.09,3.15-27.81-2.94-3.94-3-5.66-7.76-7-12.32,18.4-.31,36.81.18,55.2-.22a9,9,0,0,0,9.14-8.78c1.14-21.06-14.5-41.74-35-46.52a43.14,43.14,0,0,0-36.41,8.22ZM480.81,63c-1.54,3.9-.13,8.11,1.31,11.81,9,22.13,17.86,44.3,26.8,66.45-3.45,8.63-7,17.22-10.48,25.84-1.38,3.71-2.87,7.91-1.35,11.81,2.27,5.84,9.55,8.27,15.29,6.84,5.23-1.55,7.88-6.8,9.73-11.48,13.59-33.87,27.37-67.67,41-101.54,2.45-5.25,1.18-12.24-4.24-15.08-6.32-4.26-15.85-1.19-18.42,6C534,79.44,527.93,95.4,521.61,111.24q-8.26-22.41-16.45-44.84c-1.43-4-3.69-8.16-7.89-9.69-6-2.7-13.86.22-16.46,6.29ZM184,80.89c-3.53,3.5-4.94,8.39-6.14,13.06q20.12.09,40.25,0c-1.27-4.64-2.61-9.53-6.1-13-7.11-7.87-20.89-7.86-28,0Zm97.71-.14c-3.59,3.53-5,8.47-6.27,13.19q20.12.14,40.24,0c-1.24-4.54-2.53-9.33-5.89-12.83-7-8-20.87-8.19-28.08-.38Zm106.8-2.83A17.65,17.65,0,0,0,377.39,86a33.2,33.2,0,0,0,1.27,34.54c5.5,8.33,18.6,9.76,25.69,2.67,9.83-9.92,10.25-27,2.49-38.3a17.3,17.3,0,0,0-18.38-7Z" fill="#1f8eed" /> </svg>`,
    path: '/docs/weebly-installation',
  }),
  new MediaItem({
    key: 'joomla',
    icon: `<svg class="h-10" viewBox="0 0 256 258" xmlns="http://www.w3.org/2000/svg" > <path d="M255.6673,35.5610895 C255.6673,16.9537743 240.583222,1.8696965 221.975907,1.8696965 C204.986272,1.8696965 190.941136,14.4475642 188.627175,30.7986926 C188.411019,30.6841401 188.28551,30.619393 188.28551,30.619393 C160.733136,18.1919377 137.523798,40.2029572 137.523798,40.2029572 L76.132607,101.894973 L100.389852,125.254724 L149.802833,76.7093541 C172.8767,53.6344903 185.440623,69.2524825 185.440623,69.2524825 C202.81077,83.9261634 186.039284,101.296311 186.039284,101.296311 L210.896187,125.254724 C231.193899,103.332358 232.3713,84.1990973 226.201401,68.9785525 C242.812514,66.898677 255.6673,52.736 255.6673,35.5610895" fill="#F9AE41" /> <path d="M226.514179,190.506833 C226.628732,190.291673 226.693479,190.166163 226.693479,190.166163 C239.12193,162.612794 217.109914,139.404451 217.109914,139.404451 L155.416903,78.0112685 L132.058148,102.26951 L180.603518,151.68249 C203.678381,174.757354 188.060389,187.321276 188.060389,187.321276 C173.386708,204.691424 156.01656,187.919938 156.01656,187.919938 L132.058148,212.775844 C153.979518,233.073556 173.113774,234.250957 188.333323,228.081058 C190.413198,244.693167 204.576872,257.547953 221.751782,257.547953 C240.359097,257.547953 255.442179,242.463875 255.442179,223.85656 C255.442179,206.866926 242.865307,192.82179 226.514179,190.506833" fill="#EE4035" /> <path d="M155.941852,133.040311 L106.528872,181.584685 C83.4540078,204.660545 70.8900856,189.043549 70.8900856,189.043549 C53.5199377,174.368872 70.2914241,156.99972 70.2914241,156.99972 L45.4355175,133.041307 C25.1378054,154.962677 23.9604047,174.095938 30.1293074,189.316482 C13.5171984,191.396358 0.66340856,205.558039 0.66340856,222.733946 C0.66340856,241.341261 15.7474864,256.424342 34.3548016,256.424342 C51.3444358,256.425339 65.389572,243.848467 67.7035331,227.497339 C67.9196887,227.611891 68.0451984,227.675642 68.0451984,227.675642 C95.5985681,240.105089 118.806911,218.092078 118.806911,218.092078 L180.199097,156.399066 L155.941852,133.040311" fill="#4F91CD" /> <path d="M75.7271907,106.612545 C52.6523268,83.5376809 68.2683268,70.9737588 68.2683268,70.9737588 C82.944,53.6036109 100.313152,70.3750973 100.313152,70.3750973 L124.271564,45.5201868 C102.350195,25.2224747 83.2169339,24.0450739 67.9973852,30.2139767 C65.9165136,13.6008716 51.7548327,0.747081712 34.57793,0.747081712 C15.9716109,0.747081712 0.888529183,15.8311595 0.887533074,34.4384747 C0.887533074,51.4291051 13.4644047,65.4742412 29.8155331,67.7882023 C29.7009805,68.004358 29.6372296,68.1298677 29.6372296,68.1298677 C17.206786,95.6822412 39.2197977,118.889588 39.2197977,118.889588 L100.913805,180.28277 L124.27256,156.025525 L75.7271907,106.612545" fill="#7AC043" /> </svg>`,
    path: '/docs/joomla-installation',
  }),
  new MediaItem({
    key: 'shopify',
    icon: `<svg class="h-10" viewBox="0 0 500 143" fill="none" xmlns="http://www.w3.org/2000/svg" > <path d="M107.4 27.0998C107.3 26.3998 106.7 25.9998 106.2 25.9998C105.7 25.9998 95.8 25.7998 95.8 25.7998C95.8 25.7998 87.5 17.7998 86.7 16.8998C85.9 16.0998 84.3 16.2998 83.7 16.4998C83.7 16.4998 82.1 16.9998 79.5 17.7998C79.1 16.3998 78.4 14.6998 77.5 12.8998C74.6 7.29982 70.2 4.29982 65 4.29982C64.6 4.29982 64.3 4.29982 63.9 4.39982C63.7 4.19982 63.6 3.99982 63.4 3.89982C61.1 1.49982 58.2 0.299818 54.7 0.399818C48 0.599818 41.3 5.49982 35.8 14.0998C32 20.1998 29.1 27.7998 28.2 33.6998C20.5 36.0998 15.1 37.7998 14.9 37.7998C11 38.9998 10.9 39.0998 10.4 42.7998C10.2 45.5998 0 124.5 0 124.5L85.6 139.3L122.7 130.1C122.6 130.1 107.5 27.7998 107.4 27.0998ZM75.2 19.1998C73.2 19.7998 71 20.4998 68.6 21.2998C68.6 17.8998 68.1 13.0998 66.6 9.09982C71.7 9.89982 74.2 15.6998 75.2 19.1998ZM64.1 22.5998C59.6 23.9998 54.7 25.4998 49.8 26.9998C51.2 21.6998 53.8 16.4998 57 12.9998C58.2 11.6998 59.9 10.2998 61.8 9.49982C63.8 13.3998 64.2 18.8998 64.1 22.5998ZM54.9 4.89982C56.5 4.89982 57.8 5.19982 58.9 5.99982C57.1 6.89982 55.3 8.29982 53.7 10.0998C49.4 14.6998 46.1 21.7998 44.8 28.6998C40.7 29.9998 36.7 31.1998 33.1 32.2998C35.5 21.3998 44.6 5.19982 54.9 4.89982Z" fill="#95BF47" /> <path d="M106.2 25.9999C105.7 25.9999 95.8001 25.7999 95.8001 25.7999C95.8001 25.7999 87.5001 17.7999 86.7001 16.8999C86.4001 16.5999 86.0001 16.3999 85.6001 16.3999V139.3L122.7 130.1C122.7 130.1 107.6 27.7999 107.5 27.0999C107.3 26.3999 106.7 25.9999 106.2 25.9999Z" fill="#5E8E3E" /> <path d="M64.9998 45.1L60.6998 61.2C60.6998 61.2 55.8998 59 50.1998 59.4C41.7998 59.9 41.7998 65.2 41.7998 66.5C42.2998 73.7 61.1998 75.3 62.2998 92.2C63.0998 105.5 55.2998 114.6 43.8998 115.3C30.2998 116 22.7998 108 22.7998 108L25.6998 95.7C25.6998 95.7 33.2998 101.4 39.2998 101C43.1998 100.8 44.6998 97.5 44.4998 95.3C43.8998 85.9 28.4998 86.5 27.4998 71C26.6998 58 35.1998 44.9 53.9998 43.7C61.2998 43.2 64.9998 45.1 64.9998 45.1Z" fill="#ffffff" /> <path d="M172.9 79.3998C168.6 77.0998 166.4 75.0998 166.4 72.3998C166.4 68.9998 169.5 66.7998 174.3 66.7998C179.9 66.7998 184.9 69.0998 184.9 69.0998L188.8 57.0998C188.8 57.0998 185.2 54.2998 174.6 54.2998C159.8 54.2998 149.5 62.7998 149.5 74.6998C149.5 81.4998 154.3 86.5998 160.7 90.2998C165.9 93.1998 167.7 95.2998 167.7 98.3998C167.7 101.6 165.1 104.2 160.3 104.2C153.2 104.2 146.4 100.5 146.4 100.5L142.2 112.5C142.2 112.5 148.4 116.7 158.9 116.7C174.1 116.7 185.1 109.2 185.1 95.6998C185 88.3998 179.5 83.1998 172.9 79.3998Z" fill="#000000" /> <path d="M233.5 54.1C226 54.1 220.1 57.7 215.6 63.1L215.4 63L221.9 29H205L188.5 115.6H205.4L211 86C213.2 74.8 219 67.9 224.4 67.9C228.2 67.9 229.7 70.5 229.7 74.2C229.7 76.5 229.5 79.4 229 81.7L222.6 115.6H239.5L246.1 80.6C246.8 76.9 247.3 72.5 247.3 69.5C247.4 59.9 242.4 54.1 233.5 54.1Z" fill="#000000" /> <path d="M285.7 54.1001C265.3 54.1001 251.8 72.5001 251.8 93.0001C251.8 106.1 259.9 116.7 275.1 116.7C295.1 116.7 308.6 98.8001 308.6 77.8001C308.7 65.7001 301.6 54.1001 285.7 54.1001ZM277.4 103.8C271.6 103.8 269.2 98.9001 269.2 92.7001C269.2 83.0001 274.2 67.2001 283.4 67.2001C289.4 67.2001 291.4 72.4001 291.4 77.4001C291.4 87.8001 286.3 103.8 277.4 103.8Z" fill="#000000" /> <path d="M352 54.1001C340.6 54.1001 334.1 64.2001 334.1 64.2001H333.9L334.9 55.1001H319.9C319.2 61.2001 317.8 70.6001 316.5 77.6001L304.7 139.6H321.6L326.3 114.5H326.7C326.7 114.5 330.2 116.7 336.6 116.7C356.5 116.7 369.5 96.3001 369.5 75.7001C369.5 64.3001 364.4 54.1001 352 54.1001ZM335.8 104C331.4 104 328.8 101.5 328.8 101.5L331.6 85.7001C333.6 75.1001 339.1 68.1001 345 68.1001C350.2 68.1001 351.8 72.9001 351.8 77.4001C351.8 88.4001 345.3 104 335.8 104Z" fill="#000000" /> <path d="M393.7 29.7998C388.3 29.7998 384 34.0998 384 39.5998C384 44.5998 387.2 48.0998 392 48.0998H392.2C397.5 48.0998 402 44.4998 402.1 38.2998C402.1 33.3998 398.8 29.7998 393.7 29.7998Z" fill="#000000" /> <path d="M370 115.5H386.9L398.4 55.5H381.4L370 115.5Z" fill="#000000" /> <path d="M441.5 55.3998H429.7L430.3 52.5998C431.3 46.7998 434.7 41.6998 440.4 41.6998C443.4 41.6998 445.8 42.5998 445.8 42.5998L449.1 29.2998C449.1 29.2998 446.2 27.7998 439.9 27.7998C433.9 27.7998 427.9 29.4998 423.3 33.3998C417.5 38.2998 414.8 45.3998 413.5 52.5998L413 55.3998H405.1L402.6 68.1998H410.5L401.5 115.6H418.4L427.4 68.1998H439.1L441.5 55.3998Z" fill="#000000" /> <path d="M482.3 55.5C482.3 55.5 471.7 82.2 467 96.8H466.8C466.5 92.1 462.6 55.5 462.6 55.5H444.8L455 110.6C455.2 111.8 455.1 112.6 454.6 113.4C452.6 117.2 449.3 120.9 445.4 123.6C442.2 125.9 438.6 127.4 435.8 128.4L440.5 142.8C443.9 142.1 451.1 139.2 457.1 133.6C464.8 126.4 472 115.2 479.3 100L500 55.5H482.3Z" fill="#000000" /> </svg>`,
    path: '/docs/shopify-installation',
  }),
  new MediaItem({
    key: 'wix',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 311 121" > <path d="M178 2.3c-6 3-8.6 8.6-8.6 23.8 0 0 3-3 7.8-4.8 3.5-1.3 6-3 7.8-4.3 5.2-3.9 6-8.6 6-16.8C190.9 0.2 182.7-0.3 178 2.3z" fill="#FBBD71" /> <path d="M141.3 5.8c-5.2 4.3-6.5 11.7-6.5 11.7l-16.8 64.4 -13.8-52.7c-1.3-5.6-3.9-12.5-7.8-17.3C91.6 5.8 81.6 5.4 80.4 5.4c-0.9 0-10.8 0.4-16 6.5 -3.9 4.8-6.5 11.7-7.8 17.3l-13 52.7L26.8 17.5c0 0-1.3-6.9-6.5-11.7C12.1-1.6 0 0.2 0 0.2l32 120.5c0 0 10.4 0.9 15.6-1.7 6.9-3.5 10.4-6 14.3-22.5 3.9-14.7 14.3-57.5 15.1-60.5 0.4-1.3 1.3-5.2 3.9-5.2 2.6 0 3.5 3.5 3.9 5.2 0.9 3 11.2 45.8 15.1 60.5 4.3 16.4 7.3 19 14.3 22.5 5.2 2.6 15.6 1.7 15.6 1.7L161.6 0.2C161.6 0.2 149.5-1.5 141.3 5.8z" fill="#000000" /> <path d="M190.9 19.6c0 0-2.2 3-6.5 5.6 -3 1.7-5.6 2.6-8.6 4.3 -5.2 2.6-6.5 5.2-6.5 9.1v1.3 6.5l0 0v1.3 72.6c0 0 8.2 0.9 13.4-1.7 6.9-3.5 8.2-6.9 8.2-21.6V24.4l0 0L190.9 19.6 190.9 19.6z" fill="#000000" /> <path d="M270.4 60.7L311 0.6c0 0-16.8-3-25.5 4.8 -5.6 4.8-11.2 13.8-11.2 13.8l-14.7 21.6c-0.9 1.3-1.7 2.2-3 2.2s-2.6-1.3-3-2.2l-14.7-21.6c0 0-6-8.6-11.2-13.8 -8.6-7.8-25.5-4.8-25.5-4.8l39.3 60 -40.2 60c0 0 17.7 2.2 26.4-5.6 5.6-4.8 10.8-13 10.8-13l14.7-21.6c0.9-1.3 1.7-2.2 3-2.2s2.6 1.3 3 2.2l14.7 21.6c0 0 5.6 8.2 10.8 13 8.6 7.8 25.9 5.6 25.9 5.6L270.4 60.7z" fill="#000000" /> </svg>`,
    path: '/docs/wix-installation',
  }),
  new MediaItem({
    key: 'wwbflow',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 808.3 203.3" > <path d="M261,79.5c0-14.5-14.1-29.9-38.5-29.9c-27.4,0-57.5,20.1-62.1,60.2c-4.7,40.5,20.5,58.5,45.9,58.5s38.7-9.9,52.3-23.1 c-11.7-14.7-26.8-7.9-29.7-6.4c-3.2,1.6-7.3,3.8-15.7,3.8c-9.8,0-19.8-4.4-19.8-22.7C256.2,113.7,261,94,261,79.5z M229.9,81.9 c-0.4,4.5-2.2,12.2-34,16.5c6.7-23.8,19.5-25.6,25.4-25.6C226.8,72.8,230.3,77.1,229.9,81.9z M123.5,85.8c0,0-13.5,42.4-14.6,45.9 C108.5,128.1,98.6,52,98.6,52c-23,0-35.3,16.4-41.8,33.7c0,0-16.4,42.5-17.8,46.1c-0.1-3.3-2.5-45.6-2.5-45.6 C35.1,65,15.7,52.1,0,52.1l19,115.3c24.1-0.1,37.1-16.4,43.9-33.7c0,0,14.5-37.5,15.1-39.2c0.1,1.6,10.4,72.9,10.4,72.9 c24.2,0,37.2-15.3,44.2-32l33.9-83.3C142.6,52.1,130,68.4,123.5,85.8z M349.8,49.4c-14.9,0-26.3,8.1-35.9,20v-0.1L322.5,0 c-19.8,0-35.9,17.2-39,42.8l-15,123.7c11.4,0,23.5-3.3,30-11.7c5.8,7.5,14.5,13.5,27.4,13.5c33.4,0,56.3-38.8,56.3-75.2 C382,59.9,365.9,49.4,349.8,49.4z M346.7,108.9c-3.5,20.3-14.8,34.1-25.7,34.1c-11,0-15.7-4.9-15.7-4.9c2.1-17.8,3.4-28.7,7.4-38.1 c4-9.4,13.5-24.4,23.4-24.4C345.8,75.6,350.2,88.5,346.7,108.9z M465.6,52.1h-23.3l0.1-1.2c1.6-15.2,5.2-23.2,17.1-24.5 c8.1-0.8,11.7-5,12.6-9.6c0.7-3.6,2.9-16.1,2.9-16.1c-46.8-0.3-61.5,19.9-65.3,50.7l-0.1,0.7h-0.5c-7.6,0-16,8.6-17.4,19.5l-0.5,4 h15.6l-13.1,108.2l-4,19.4c0.5,0,1.1,0.1,1.6,0.1c21.9-0.8,35.9-18.1,39-43.2l10.2-84.5h7.3c7.2,0,15.6-7.2,17.1-19.2L465.6,52.1z M586.8,50.1c-26.9,0-52.3,19.8-60.1,50.9s4,67.9,43.6,67.9c39.5,0,62.3-38.3,62.3-70.2C632.6,67,611.1,50.1,586.8,50.1z M597.7,107.9c-1.4,14.1-7.6,35.5-24.6,35.5c-17.1,0-14.7-25-13-36.9c1.8-12.7,8.9-30.8,24.2-30.8C598,75.7,599.3,92,597.7,107.9z M765.5,85.8c0,0-13.5,42.4-14.6,45.9c-0.4-3.6-10.3-79.7-10.3-79.7c-23,0-35.3,16.4-41.8,33.7c0,0-16.4,42.5-17.8,46.1 c-0.1-3.3-2.5-45.6-2.5-45.6C677.1,65,657.7,52.1,642,52.1l18.9,115.3c24.1-0.1,37.1-16.4,43.9-33.7c0,0,14.5-37.5,15-39.2 c0.2,1.6,10.4,72.9,10.4,72.9c24.2,0,37.2-15.3,44.2-32l33.9-83.3C784.5,52.1,771.9,68.4,765.5,85.8z M487.6,0.7L468,158.8l0,0 l-4,19.5c0.5,0,1.1,0.1,1.6,0.1l0,0c21.1-0.3,36.1-18.8,38.9-42.1l11.3-90.7C519.2,17.9,502.7,0.7,487.6,0.7z" fill="#3545ee" /> </svg>`,
    path: '/docs/webflow-installation',
  }),
  new MediaItem({
    key: 'segment',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 740.94 159.36"> <rect width="88.52" height="15.36" x="55.6" y="52.81" fill="#52bd94"  rx="2.86"/> <rect width="88.52" height="15.36" x=".83" y="91.19" fill="#52bd94"  rx="2.86"/> <path d="M11.45 63.32a2.87 2.87 0 003.48-2 60.42 60.42 0 0170.59-40.61 2.84 2.84 0 003.32-2.06l2.5-9.33a2.87 2.87 0 00-2.17-3.53 75.76 75.76 0 00-89 51.48 2.86 2.86 0 002 3.56zM133.52 96a2.88 2.88 0 00-3.52 2 60.42 60.42 0 01-70.59 40.66 2.83 2.83 0 00-3.32 2.06L53.64 150a2.87 2.87 0 002.17 3.54 75.8 75.8 0 0089-51.49 2.84 2.84 0 00-2-3.55z" fill="#52bd94" fill-rule="evenodd" /> <circle cx="116.24" cy="27.53" r="8.42" fill="#52bd94" /> <circle cx="28.73" cy="131.82" r="8.42" fill="#52bd94" /> <path d="M350.33 64.41a32.21 32.21 0 0124.27-10.52c8.86 0 15.76 2.74 20.12 8v-4.36a2.44 2.44 0 012.3-2.32h11.68a2.45 2.45 0 012.3 2.32v64.85a39.51 39.51 0 01-8 25.44c-6.3 7.66-15.86 11.54-28.42 11.54-17.46 0-31.24-11.05-33.51-26.83a2.45 2.45 0 011.76-2.59l11.41-2.68a1.13 1.13 0 01.18 0 2.28 2.28 0 012.6 1.93 17.26 17.26 0 0017.58 15.21c13.64 0 20-7 20-21.63v-5.68a25.66 25.66 0 01-20 8.56c-19.41 0-33.49-15.07-33.49-35.81a37.76 37.76 0 019.22-25.43zM576.82 53h.61c20.87.28 33.8 14.81 33.8 38.06a35.09 35.09 0 01-.2 3.69 2.27 2.27 0 01-2.24 2h-49.61a19.08 19.08 0 0019.35 18.51 17.39 17.39 0 0017.75-12.39 2.29 2.29 0 012.85-1.37L609 105a2.13 2.13 0 011.29 1.17 2.22 2.22 0 01.12 1.73 32.66 32.66 0 01-31.92 22.2 35.26 35.26 0 01-25.17-10.22 39.36 39.36 0 01-10.81-28.56c0-24 17.44-38.37 34.29-38.37zm-280.68 0c21.24 0 34.44 14.59 34.44 38.07a35.09 35.09 0 01-.2 3.69 2.24 2.24 0 01-2.24 2h-49.61A19.08 19.08 0 00298 115.31a17.37 17.37 0 0017.75-12.39 2.26 2.26 0 012.85-1.37l9.91 3.49a2.11 2.11 0 011.29 1.17 2.26 2.26 0 010 1.73 32.66 32.66 0 01-31.9 22.2 35.29 35.29 0 01-25.22-10.22 39.5 39.5 0 01-10.81-28.56c0-24 17.44-38.37 34.3-38.37zM221 53c9.53 0 17.24 2.31 22.95 6.88a25 25 0 018.22 11.92 1.55 1.55 0 01-1.17 1.87l-11.83 3.91a2.29 2.29 0 01-1.63 0 1.72 1.72 0 01-.93-1.08c-1-3.83-4.44-10.26-15.63-10.26-7.81 0-12.49 4.48-12.49 8.82 0 2.88 1.49 6.59 8.53 7.93l13 2.44c14.53 2.93 22.85 10.88 22.85 21.95v-.19c0 11-10.58 22.75-30.24 22.75-22.8 0-31.76-12.9-33.41-20.51a1.64 1.64 0 011.21-1.81l12.2-3.8a1.83 1.83 0 01.66 0 2.05 2.05 0 011 .24 1.68 1.68 0 01.9 1.1c1.54 7.32 8.15 11.68 17.66 11.68 8.85 0 12.88-4.63 12.88-8.83s-3.59-7.29-9.76-8.53l-13.68-2.66c-12.76-2.41-20.71-10.34-20.71-20.66C191.58 63.6 205.07 53 221 53zm502.55-20.35a2.44 2.44 0 012.29 2.44v20h12.83a2.45 2.45 0 012.27 2.44v10.66a2.3 2.3 0 01-2.27 2.31h-12.83v35.59c0 5.9 2.34 8.34 8.36 8.34a31.2 31.2 0 004.1-.34 2.28 2.28 0 011.86.51 2.44 2.44 0 01.78 1.76v10a2.3 2.3 0 01-1.64 2.22 31.61 31.61 0 01-9 1.17c-12.86 0-20.83-8.27-20.83-21.61V70.65h-11.1a2.44 2.44 0 01-2.29-2.32V57.65a2.43 2.43 0 012.29-2.44h2.88c3.49 0 9.37-1.22 9.37-9.54V35.09a2.43 2.43 0 012.29-2.44zm-240.8 32.27A26.4 26.4 0 01506 53c12.68 0 25.49 8.93 25.49 28.44v45.16a2.45 2.45 0 01-2.3 2.32H517.5a2.45 2.45 0 01-2.29-2.32V82.89c0-5.53-1.73-14.85-13.44-14.85-8.56 0-14.64 6.73-14.64 16.49v42.07a2.44 2.44 0 01-2.29 2.32H473a2.45 2.45 0 01-2.3-2.32V82.89C470.72 77.36 469 68 457.28 68c-8.9 0-14.9 6.68-14.9 16.63v41.97a2.45 2.45 0 01-2.29 2.32h-11.68a2.44 2.44 0 01-2.27-2.32V57.38a2.45 2.45 0 012.27-2.32h11.24a2.45 2.45 0 012.29 2.32v4.17A26.29 26.29 0 01461.7 53a23.35 23.35 0 0121.05 11.92zM660.86 53a24.69 24.69 0 0119.81 8.61 32.2 32.2 0 016.53 20.85v44.14a2.43 2.43 0 01-2.26 2.32h-11.83a2.46 2.46 0 01-2.3-2.32V84.82c0-10.93-5.14-16.49-14.9-16.49a13.55 13.55 0 00-11.09 5.17 21.38 21.38 0 00-3.81 13v40.1a2.45 2.45 0 01-2.29 2.32h-11.81a2.45 2.45 0 01-2.29-2.32V57.33a2.43 2.43 0 012.29-2.33h11.54a2.43 2.43 0 012.27 2.31v4.88A25.41 25.41 0 01660.86 53zM376.63 68.48c-11.52 0-18.83 8.19-18.83 21.36s7.22 21.35 18.83 21.35c11.07 0 18.53-8.59 18.53-21.35s-7.46-21.36-18.53-21.36zm-80.37-.66A17.07 17.07 0 00279 83h34.7c-.83-9.37-7.49-15.18-17.44-15.18zm280.75 0A17.08 17.08 0 00559.77 83h34.71c-.86-9.37-7.48-15.18-17.48-15.18z" fill="#52bd94" fill-rule="evenodd" /> <path d="M208.79 6.19v23.26h-6.59V6.19h-9.11V.55H218v5.64zM259.13 29.55h-6.85L246 8.25l-6.39 21.3h-6.71l-8.5-29h7l5.39 20.89L242.87.55h7l6 20.89L261.09.55h6.44zM276.6 29.45V.55h6.49v28.9zM294.77 29.45V.55h6.59v23.21h11.28v5.69zM322.46 29.45V.55H329v28.9zM353.47 30c-8 0-14.4-5.74-14.4-14.95s6.49-15 14.45-15 14.39 5.74 14.39 15S361.47 30 353.47 30zm0-24.66c-5 0-7.91 4.12-7.91 9.66s3 9.71 7.91 9.71 7.8-4.22 7.8-9.66-2.92-9.71-7.75-9.71z" fill="#52bd94" /> </svg>`,
  }),
  new MediaItem({
    key: 'volusion',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 380 80"> <path d="M40.2 50.4L26.9 27.3H13.5c-5.5 0-8.9 6-6.2 10.7l20 34.7c2.7 4.7 9.6 4.8 12.4 0l12.9-22.3c-2.7 4.8-9.6 4.7-12.4 0z" fill="#8759f2"/> <path d="M53.6 27.3H26.9l13.3 23.1c2.7 4.7 9.6 4.8 12.4 0L59.8 38c2.7-4.8-.8-10.7-6.2-10.7z" fill="#ff3e80"/> <path d="M66.5 5H26.4c-5.5 0-8.9 5.9-6.2 10.7l6.7 11.6h26.7c5.5 0 8.9 6 6.2 10.7l12.9-22.3C75.4 11 72 5 66.5 5z" fill="#ffc300"/> <path d="M20.2 15.7a6.95 6.95 0 010-7.1L7.3 30.9c1.3-2.2 3.6-3.6 6.2-3.6h13.4l-6.7-11.6z" fill="#00d4e6"/> <path d="M313.9 26.5c-10.8 0-19.3 8.3-19.3 18.6s8.3 18.6 19.3 18.6 19.3-8.3 19.3-18.6-8.5-18.6-19.3-18.6zm0 29.7c-6.1 0-10.6-4.9-10.6-11s4.7-11 10.6-11c5.9 0 10.6 4.9 10.6 11s-4.6 11-10.6 11zm-126.2-44h-3.1c-2 0-3 1-3 3v44.7c0 2 1 3 3 3h3.1c2 0 3-1 3-3V15.2c0-2-1-3-3-3zm96.6 15.2h-3.1c-2 0-3 1-3 3V60c0 2 1 3 3 3h3.1c2 0 3-1 3-3V30.4c0-2-1.1-3-3-3zm-154.6-.1h-2.9c-2 0-2.9.7-3.6 2.6L114.7 53l-8.5-23.1c-.7-1.9-1.6-2.6-3.6-2.6h-3c-1.9 0-3.6.8-2.6 3.4l11.7 29.1c1 2.6 2 3.1 4.1 3.1h3.7c2.1 0 3.1-.6 4.1-3.1l11.7-29.1c1-2.6-.7-3.4-2.6-3.4zm101.2.1h-3.1c-2 0-3 1-3 3v14.9c-.3 7.1-4.4 10.9-9.4 10.9-4 0-6.2-3.5-6.2-8.8v-17c0-2-1-3-3-3h-3.1c-2 0-3 1-3 3v17.7c0 6.3 2.8 15.7 13.2 15.7 8.3 0 10.9-5.4 11.5-7.2V60c0 2 1 3 3 3h3.1c2 0 3-1 3-3V30.4c0-2-1-3-3-3zM343.3 63h3.1c2 0 3-1 3-3V45c.3-7.1 4.4-10.9 9.4-10.9 4 0 6.2 3.5 6.2 8.8v17c0 2 1 3 3 3h3.1c2 0 3-1 3-3V42.2c0-6.3-2.8-15.7-13.2-15.7-8.3 0-10.9 5.4-11.5 7.2v-3.3c0-2-1-3-3-3h-3.1c-2 0-3 1-3 3V60c0 2 1.1 3 3 3zm-54.9-46.1c0 2.9-2.1 5.3-5.7 5.3s-5.7-2.4-5.7-5.3 2.2-5.3 5.7-5.3 5.7 2.4 5.7 5.3zm-20.1 15.2c.2-.7.2-1.4-.1-2.1-.3-.6-.9-1.1-1.5-1.3-2.8-1-6.6-2.2-11-2.2-7.8 0-13.2 4.4-13.5 10.1-.2 4.6 2.4 7.6 5.7 9.4 3.9 2.1 8.4 2.8 10.4 3.5 2.1.8 3.2 1.7 3.3 3.5 0 1.8-1.7 3.4-5.2 3.4-4.4 0-8.1-1.5-10.5-2.5-1.3-.6-2.8-.1-3.5 1.2l-.7 1.9c-.6 1.3-.1 2.8 1.1 3.5 2.6 1.4 7.5 3.3 13.2 3.3 8.8 0 14.1-4.7 14.1-10.9 0-2.8-.9-6.2-5.6-8.7-2.5-1.4-9.4-2.7-11.7-4-1.6-.8-2.3-2.1-2.1-3.5.2-2 2.1-2.9 4.5-3.1 3.9-.2 6.7 1.1 9.5 1.8 1.3.3 2.6-.5 3-1.7l.6-1.6zm-113.6-5.6c-10.8 0-19.3 8.3-19.3 18.6s8.3 18.6 19.3 18.6S174 55.4 174 45.1s-8.5-18.6-19.3-18.6zm0 29.7c-6.1 0-10.6-4.9-10.6-11s4.7-11 10.6-11c5.9 0 10.6 4.9 10.6 11s-4.5 11-10.6 11z" fill="#221b43"/> </svg>`,
  }),
  new MediaItem({
    key: 'bigcartel',
    name: 'Big Cartel',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 129 29"> <path class="a" d="M46.18 4A1.91 1.91 0 1150 4a1.91 1.91 0 11-3.81 0zm78.76 14.35a.81.81 0 01-.25-.69V2.23h-4.52v2.68h1.58V18c0 2.14 1.11 3.28 3.2 3.28a6.56 6.56 0 002-.42v-2.78c-1.28.51-1.8.38-2.01.23zm-75.27.05h1.43V21h-5.79v-2.64h1.43v-8h-1.61V7.7h4.54zm-11.09-11a5.81 5.81 0 00-4.36 1.87V2.23H29.7v2.68h1.62v12.71a.81.81 0 01-.25.69c-.43.33-1.5 0-2.06-.23v2.76a6.59 6.59 0 002.05.42 2.92 2.92 0 002.74-1.32 6.86 6.86 0 004.27 1.34 6.66 6.66 0 006.86-7c-.01-4.06-2.68-6.97-6.35-6.97zM38 18.57c-2.15 0-3.72-1.75-3.72-4.17s1.55-4.32 3.78-4.32a3.75 3.75 0 013.75 4.1c.02 2.55-1.58 4.39-3.81 4.39zm68.86-.49v2.76a7.52 7.52 0 01-2 .42c-2.09 0-3.2-1.14-3.2-3.28V5.36l2.93-1.92V7.7h3.81l-1.91 2.68h-1.9v7.24a.77.77 0 00.26.69c.53.4 2.03-.23 2.03-.23zM58 7.31c-3.88 0-6.69 3-6.69 7.11s2.66 6.87 6.33 6.87A6.14 6.14 0 0062 19.45v1.42c0 2.72-2.4 3.45-3.83 3.45a5.22 5.22 0 01-3.12-1.06l-2.36 1.83A7.78 7.78 0 0058 27c3.21 0 6.95-1.63 6.95-6.21v-6.38A6.84 6.84 0 0058 7.31zm.12 11.29c-2.19 0-3.72-1.74-3.72-4.23s1.6-4.21 3.8-4.21a3.94 3.94 0 013.8 4.21 4 4 0 01-3.85 4.23zM120.6 15c0-5.05-3.45-7.69-6.85-7.69a6.76 6.76 0 00-6.58 7.06 7.13 7.13 0 0012.69 4.39l-2.22-1.71a4.24 4.24 0 01-3.44 1.69 3.86 3.86 0 01-3.94-3.11h10.28a3.09 3.09 0 00.06-.63zm-10.35-2.08a3.65 3.65 0 013.56-3.11 3.77 3.77 0 013.77 3.11zM94.92 10V7.7h-4v2.68h1.62v8h-2.35a.83.83 0 01-.61-.19.91.91 0 01-.19-.64v-5.77c0-1.31-.65-4.47-5.52-4.47a7.85 7.85 0 00-4.14 1.18l1.17 2.23a5 5 0 013-.78 3.26 3.26 0 011.76.49 2.08 2.08 0 01.81 1.78v1.25a6.58 6.58 0 00-3.21-.92c-2.58 0-3.91 1.62-5.19 3.2s-2.51 3-5 2.92c-2.27-.11-3.63-1.86-3.63-4.43 0-2.39 1.45-4 3.54-4a3.75 3.75 0 013.7 3.18l2.45-1.9a6.3 6.3 0 00-6.3-4.18 6.72 6.72 0 00-6.48 7c0 3.43 2.1 7.16 6.62 7.16a7.45 7.45 0 005.87-2.73 4.38 4.38 0 004.08 2.57 5.91 5.91 0 003.93-1.66 2.87 2.87 0 002.8 1.33h7.42v-2.64h-1.61v-3.21c0-3.3 1.09-4.77 3.56-4.77a3.68 3.68 0 011.45.31V8a4.81 4.81 0 00-1.74-.25A4.21 4.21 0 0094.92 10zm-8.47 7.48a4.93 4.93 0 01-3.16 1.41 1.9 1.9 0 01-2.05-1.91 2 2 0 012.18-2 5 5 0 013 1.18zM11 14.52v-.89a1.78 1.78 0 01.83-1.51l7.35-4.7A1.79 1.79 0 0020 5.91V2.05L11 7.8 2 2.05V14.2a8.69 8.69 0 003.88 7.58L11 25.05l5.12-3.27A8.69 8.69 0 0020 14.2V8.77z"/> </svg>`,
  }),

  new MediaItem({
    key: 'blog',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" version="1.0" viewBox="0 0 432.513 125.044"> <path fill="#036" d="M161.8 81.98c1.2.26 2.6.26 4.7.26 5.4 0 10.2-2.34 10.2-8.49 0-5.9-4.8-8.24-10.8-8.24h-4.1v16.47zm0-25.78h3.7c5.9 0 9-2.97 9-7.47 0-4.59-2.9-7.02-7.9-7.02-2.5 0-3.9.18-4.8.45V56.2zm-12.4-24.47c3.4 0 9.9-.01 16.2-.01 7.7 0 12.1-.16 16 2.67 3.8 2.78 5.8 6.99 5.8 12.57 0 5.31-1.2 11.19-8.2 13.71v.18c6.3.76 11.1 6.83 11.1 14.66 0 5.58-1.9 10.53-5.4 13.5-4 3.5-8.2 4.45-21.9 4.45-6.2 0-10.8.01-13.6.01V31.73m94.2 39.15c0-7.49-.9-12.51-6.3-12.51-5.5 0-6.6 4.9-6.6 12.59 0 7.95 2 12.51 6.5 12.51s6.4-3.97 6.4-12.59zm-6.4 22.62c-13.7 0-18.1-8.35-18.1-22.39 0-14.03 5.1-23 18.7-23 13.7 0 17.9 9.14 17.9 22.29 0 15.81-4.8 23.1-18.5 23.1m47.6-26.19c0-3.69-.8-8.58-5.4-8.58-4.5 0-6.3 4.52-6.3 12.56 0 6.59 1.3 11.75 6.3 11.75 4.9 0 5.4-6.85 5.4-9.22v-6.51zm12.3 19.79c0 8.49.2 15.6-4 20.1-4 4.1-10.6 4.8-16 4.8-4.4 0-10.1-.9-13.3-2.7-1-.6-1-1.8-.6-3.1.4-1.2 1.5-5.1 1.8-5.8.4-1.23 1.8-1.29 2.2-1.07 2.4 1.07 5.7 1.57 9.3 1.57 5.4 0 8.2-.5 8.2-8.77v-2.8h-.2c-2.3 3-4.4 4.17-8.6 4.17-11.5 0-15.3-9.2-15.3-21.85 0-14.27 3.9-23.31 16.5-23.31 5 0 7.7 2.54 9.5 6.15l.9-4.11s.2-1.04 1.8-1.04h5.6c1.7 0 2.4.4 2.5 1.22 0 .83-.3 6.27-.3 11.96V87.1m30.5-19.79c0-3.69-.8-8.58-5.4-8.58-4.5 0-6.3 4.52-6.3 12.56 0 6.59 1.3 11.75 6.3 11.75s5.4-6.85 5.4-9.22v-6.51zm12.3 19.79c0 8.49.2 15.6-4 20.1-3.9 4.1-10.6 4.8-15.9 4.8-4.5 0-10.2-.9-13.4-2.7-.9-.6-.9-1.8-.5-3.1.3-1.2 1.5-5.1 1.7-5.8.4-1.23 1.8-1.29 2.3-1.07 2.3 1.07 5.7 1.57 9.2 1.57 5.4 0 8.2-.5 8.2-8.77v-2.8h-.2c-2.2 3-4.4 4.17-8.5 4.17-11.6 0-15.4-9.2-15.4-21.85 0-14.27 3.9-23.31 16.5-23.31 5 0 7.7 2.54 9.6 6.15l.8-4.11s.2-1.04 1.8-1.04h5.7c1.6 0 2.3.4 2.4 1.22 0 .83-.3 6.27-.3 11.96V87.1m29.5-21.12c0-3.03-.2-8.33-5.1-8.33-4.7 0-5.5 4.96-5.7 8.33h10.8zM359 74.32c.4 5.66 2.9 9.15 8.3 9.15 3.5 0 3.7-.19 8.6-1.91.8-.28 2.3-.48 2.7.61.5 1.27 1.7 5.02 2.1 6 .6 1.68-1.6 2.55-1.6 2.55-3.8 1.86-8.4 2.74-13.5 2.74-12.7 0-19.9-8.48-19.9-22.09 0-10.96 4.2-23.07 18.9-23.07 14.1 0 16.6 10.87 16.6 21.56 0 2.3-.3 4.55-2.4 4.55l-19.8-.09m-148 7.62c4.8 0 5 0 6.1 7.9.3 2.11-1.5 3.65-3.9 3.65h-9.8c-3.3 0-6.4-3.57-6.5-6.86V32.94c0-.69.5-1.25 1.2-1.25h9.2c.7 0 1.3.56 1.3 1.25v46.89c0 1.73.2 2.11 2.4 2.11m199.5-33.82c-1.8-.02-6.5-.67-11 5.1h-.3l-.5-3.91-10.6.11v42.82c0 .69.5 1.25 1.2 1.25h10.2c.7 0 1.3-.56 1.3-1.25l-.1-21.76c0-1.15.1-2.73.2-3.63.4-5.4 1.8-7.4 6.4-7.4 1.5 0 4.2.77 5.4 1.16 1.3.39 2-1.04 2-1.04s1-5.82 1.4-8.04c.3-2.22-3.8-3.39-5.6-3.41m10.9 3.16v-4.86H420v-1.16h4.1v1.16h-1.5v4.86h-1.2m11.1 0H431v-3.99l-1 2.57h-1l-1.1-2.57v3.99h-1.3v-6.02h1.6l1.3 2.97 1.3-2.97h1.7v6.02"/> <path fill="#F60" d="M125 106.3c0 10.4-8.4 18.7-18.7 18.7H18.75C8.39 125 0 116.7 0 106.3V18.75C0 8.4 8.395 0 18.75 0h87.55C116.6 0 125 8.396 125 18.75v87.55"/> <path fill="#FFF" d="M49.12 23.58c-13.8 0-24.99 11.19-25 25v27.43c0 13.81 11.2 24.99 25 24.99h26.91C89.84 101 101 89.82 101 76.01V58.29c0-2.76-2.21-5-4.97-5h-4.31c-2.64 0-4.79-2.06-4.97-4.65l-.03-.06c0-13.81-11.19-25-25-25h-12.6zm-.62 20.28h14.16c2.65 0 4.81 2.15 4.81 4.81 0 2.65-2.16 4.81-4.81 4.81H48.5c-2.65 0-4.81-2.16-4.81-4.81 0-2.66 2.16-4.81 4.81-4.81zm-.03 28.15h28.41c2.51 0 4.56 2.02 4.56 4.53 0 2.51-2.05 4.57-4.56 4.57H48.47c-2.51 0-4.53-2.06-4.53-4.57a4.52 4.52 0 014.53-4.53z"/> </svg>`,
  }),
  new MediaItem({
    key: 'openCart',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill-rule="evenodd" stroke-linejoin="round" stroke-miterlimit="1.414" clip-rule="evenodd" viewBox="0 0 400 78"> <path fill="url(#_Linear1)" fill-rule="nonzero" d="M248.385 255.483c0 37.322-11.097 68.759-33.37 94.208-23.243 26.664-53.445 40.05-90.416 40.05-37.069 0-67.173-13.386-90.417-40.05C11.9 324.242.72 292.805.72 255.483c0-37.425 11.18-68.852 33.462-94.209 23.502-26.415 53.705-39.616 90.417-39.616 36.971 0 67.173 13.201 90.416 39.616 22.273 25.357 33.37 56.784 33.37 94.209zm-73.61 0c0-16.905-4.046-31.174-12.146-42.71-9.063-13.113-21.746-19.627-38.03-19.627-16.29 0-28.968 6.514-38.04 19.627-8.184 11.536-12.234 25.805-12.234 42.71 0 16.811 4.05 31.251 12.234 43.134 9.072 13.113 21.75 19.637 38.04 19.637 16.284 0 28.967-6.524 38.03-19.637 8.1-11.883 12.146-26.323 12.146-43.134zm339.76 0c0 34.862-10.038 65.323-30.202 91.383-21.57 28.523-49.391 42.875-83.199 42.875-23.863 0-44.106-6.954-60.659-20.686 0 3.162.352 10.824 1.059 23.073.703 12.312 1.05 22.35 1.05 30.275 0 43.769-13.03 65.597-39.084 65.597-10.658 0-19.369-3.787-26.328-11.273-6.866-7.564-10.302-16.65-10.302-27.377V164.71c0-10.39 3.524-19.374 10.566-26.938 7.041-7.486 15.762-11.263 26.151-11.263 15.665 0 27.114 7.818 34.336 23.414 15.406-18.841 36.449-28.265 63.123-28.265 33.896 0 61.629 14.079 83.287 42.349 20.076 26.152 30.202 56.613 30.202 91.476zm-73.61 0c0-16.905-4.128-31.174-12.229-42.71-9.077-13.113-21.75-19.627-38.04-19.627-16.284 0-28.962 6.514-38.034 19.627-8.101 11.536-12.147 25.805-12.147 42.71 0 16.811 4.046 31.251 12.147 43.134 9.072 13.113 21.75 19.637 38.034 19.637 16.29 0 28.963-6.524 38.04-19.637 8.101-11.883 12.229-26.323 12.229-43.134zm341.429-7.916c0 12.313-2.908 20.765-8.72 25.528-5.807 4.66-14.874 7.037-27.113 7.037H613.663c0 14.088 6.607 24.912 19.812 32.476 10.653 5.91 23.683 8.911 39.089 8.911 11.981 0 26.328-3.977 43.144-11.8 16.817-7.93 27.641-11.81 32.316-11.81 8.535 0 15.752 3.085 21.74 9.424 5.895 6.256 8.896 13.825 8.896 22.545 0 20.15-15.054 35.839-45.174 47.102-22.975 8.447-46.306 12.761-70.16 12.761-38.386 0-69.379-12.761-93.061-38.2-23.688-25.45-35.566-57.409-35.566-96.058 0-36.81 12.322-68.237 36.971-94.463 24.561-26.249 55.12-39.362 91.564-39.362 33.896 0 62.244 12.415 84.965 37.24 22.707 24.834 34.155 54.412 34.155 88.669zm-73.6-18.5c-.356-11.273-5.548-21.482-15.586-30.734-10.039-9.238-20.687-13.81-31.96-13.81-11.619 0-22.452 4.572-32.486 13.81-10.033 9.252-15.059 19.461-15.059 30.734h95.091zm320.906 117.174c0 10.751-3.43 19.813-10.29 27.386-6.96 7.491-15.68 11.278-26.331 11.278-25.708 0-38.567-20.428-38.567-61.19 0-7.837.532-20.159 1.586-37.152 1.147-16.909 1.679-29.319 1.679-37.156 0-43.237-14.001-64.884-41.91-64.884-27.913 0-41.821 21.647-41.821 64.884 0 7.837.439 20.247 1.23 37.156.883 16.993 1.322 29.315 1.322 37.152 0 40.762-12.678 61.19-38.03 61.19-10.658 0-19.369-3.787-26.24-11.278-6.954-7.573-10.389-16.635-10.389-27.386V164.632c0-10.673 3.435-19.725 10.389-27.036 6.871-7.398 15.675-11.087 26.328-11.087 17.607 0 29.841 7.818 36.707 23.414 11.888-18.841 32.926-28.265 63.04-28.265 27.909 0 50.098 7.389 66.387 22.092 16.64 15.772 24.91 37.42 24.91 64.982v137.509zm230.93-7.823c0 16.636-10.81 29.837-32.48 39.533a130.815 130.815 0 01-54.67 11.79c-37.08 0-67.18-13.386-90.42-40.05-22.28-25.449-33.46-56.886-33.46-94.208 0-37.425 11.36-69.023 34.16-94.911 22.8-25.967 52.64-38.914 89.72-38.914 21.39 0 39.79 3.168 55.12 9.424 21.38 8.452 32.03 21.652 32.03 39.616 0 9.062-2.81 17.075-8.45 24.034-5.63 6.866-12.86 10.389-21.65 10.389-6 0-14.63-2.025-25.89-5.978-11.27-4.06-20.24-5.997-26.84-5.997-17.53 0-31.18 6.251-40.95 18.748-9.07 11.273-13.65 25.713-13.65 43.589 0 16.811 4.05 31.08 12.25 42.612 9.05 13.473 21.83 20.159 38.11 20.159 7.23 0 17.08-2.113 29.67-6.437 12.6-4.211 21.67-6.412 27.3-6.412 8.19 0 15.22 3.514 21.22 10.385 5.9 6.954 8.88 14.43 8.88 22.628zm267.56 7.823c0 10.751-3.43 19.813-10.3 27.386-6.95 7.491-15.75 11.278-26.42 11.278-16.02 0-27.46-7.842-34.33-23.424-15.4 18.847-36.45 28.26-63.12 28.26-34.15 0-62.15-14.352-83.82-42.875-19.72-26.06-29.65-56.521-29.65-91.383 0-34.863 9.93-65.324 29.65-91.476 22.02-28.27 49.92-42.349 83.82-42.349 26.67 0 47.72 9.424 63.12 28.265 6.87-15.596 18.31-23.414 34.33-23.414 10.4 0 19.12 3.777 26.17 11.263 7.03 7.564 10.55 16.46 10.55 26.86v181.609zm-73.6-90.758c0-16.905-4.04-31.174-12.15-42.71-9.42-13.113-22.1-19.627-38.03-19.627-16.29 0-28.96 6.514-38.04 19.627-8.18 11.536-12.22 25.805-12.22 42.71 0 16.811 3.94 31.251 11.77 43.134 9.35 13.113 22.2 19.637 38.49 19.637 16.28 0 28.96-6.524 38.03-19.637 8.11-11.883 12.15-26.323 12.15-43.134zm261.4-95.707c0 15.758-7.66 26.586-22.98 32.481a2620.842 2620.842 0 00-42.35 14.616c-15.42 8.813-23.07 25.186-23.07 48.961v90.407c0 10.751-3.53 19.813-10.38 27.386-6.97 7.491-15.77 11.278-26.42 11.278-10.73 0-19.55-3.787-26.4-11.278-6.97-7.573-10.4-16.635-10.4-27.386V164.632c0-10.673 3.43-19.725 10.3-27.036 6.95-7.398 15.77-11.087 26.42-11.087 17.6 0 29.83 7.818 36.71 23.414 15.76-18.841 32.67-28.265 50.89-28.265 10.38 0 19.28 3.778 26.67 11.263 7.3 7.574 11.01 16.553 11.01 26.855zm192.02 3.348c0 16.372-6.43 26.674-19.18 31.085-6.27 2.196-19.4 3.338-39.38 3.338v148.694c0 10.751-3.52 19.813-10.37 27.386-6.97 7.491-15.77 11.278-26.42 11.278-10.38 0-19.2-3.787-26.25-11.278-7.03-7.573-10.55-16.635-10.55-27.386V197.547c-15.15 0-26.15-1.937-33.1-5.714-9.5-5.285-14.2-14.884-14.2-28.709 0-13.816 4.85-23.239 14.62-28.265 6.61-3.699 17.53-5.627 32.68-5.627v-31.52c0-37.063 12.22-55.549 36.8-55.549 24.47 0 36.79 18.486 36.79 55.549v31.52c19.98 0 33.11 1.142 39.38 3.338 12.75 4.402 19.18 14.625 19.18 30.554zm591.31-44.706c-.2-22.599-41.96-32.857-150.06-32.857-253.64 0-317.99-10.834-386.85-83.297 28.97 43.51 44.82 125.119 312.91 121.058 279.71-4.309 109.07 72.122 59.42 158.474 89.54-79.915 164.31-132.59 164.58-163.051v-.327zm-356.42 232.059c-.01 21.006-17.29 38.288-38.3 38.293-21.22 0-38.38-17.158-38.38-38.293v-.03c0-21.041 17.31-38.356 38.36-38.356h.02c21.15 0 38.3 17.163 38.3 38.386zm174.67 0c-.01 21.006-17.29 38.288-38.3 38.293-21.2 0-38.37-17.158-38.37-38.293v-.03c0-21.041 17.31-38.356 38.35-38.356h.02c21.15 0 38.3 17.163 38.3 38.386z" transform="scale(.16 .15984)"/> <defs> <linearGradient id="_Linear1" x1="0" x2="1" y1="0" y2="0" gradientTransform="rotate(-90 818.25 -431.75) scale(384.236)" gradientUnits="userSpaceOnUse"> <stop offset="0" stop-color="#04b6f0"/> <stop offset="1" stop-color="#3ecaf0"/> </linearGradient> </defs> </svg>`,
  }),
  new MediaItem({
    key: 'wooCommerce',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 113 23"> <path fill="#7f54b3" fill-rule="evenodd" d="M34.71 0H3.551A3.553 3.553 0 000 3.568V15.46a3.566 3.566 0 003.568 3.568h14.757l6.745 3.756-1.534-3.756h11.173a3.566 3.566 0 003.568-3.568V3.568A3.566 3.566 0 0034.709 0zM2.894 2.754c-.438.031-.767.188-.986.485-.219.282-.297.642-.25 1.049.923 5.868 1.784 9.827 2.582 11.877.313.751.673 1.111 1.095 1.08.658-.047 1.44-.955 2.363-2.723a293.572 293.572 0 012.254-4.507c.845 2.958 2.003 5.18 3.458 6.667.407.422.83.61 1.236.579.36-.032.642-.22.83-.564.156-.297.219-.641.188-1.033-.094-1.424.047-3.411.438-5.962.407-2.629.907-4.522 1.518-5.649.125-.235.172-.47.156-.751a1.226 1.226 0 00-.485-.892 1.36 1.36 0 00-.986-.313c-.454.031-.798.25-1.033.688-.97 1.769-1.658 4.633-2.065 8.607-.595-1.502-1.096-3.27-1.487-5.352-.172-.923-.595-1.361-1.283-1.314-.47.031-.86.344-1.174.939L5.837 12.19c-.563-2.269-1.095-5.039-1.58-8.31-.11-.813-.564-1.188-1.362-1.126zM33.05 3.881c1.111.235 1.94.83 2.504 1.815.501.845.751 1.862.751 3.083 0 1.612-.407 3.083-1.22 4.429-.94 1.565-2.16 2.347-3.678 2.347-.266 0-.547-.031-.845-.094-1.11-.235-1.94-.83-2.504-1.815-.5-.86-.75-1.894-.75-3.099 0-1.612.406-3.082 1.22-4.413.955-1.565 2.175-2.347 3.677-2.347.266 0 .548.031.845.094zm-.657 8.466c.58-.516.97-1.283 1.19-2.316.062-.36.109-.751.109-1.158 0-.454-.094-.939-.282-1.424-.234-.61-.547-.94-.923-1.017-.563-.11-1.111.203-1.627.97a5.06 5.06 0 00-.83 1.862c-.078.36-.11.751-.11 1.143 0 .453.095.938.282 1.424.235.61.548.938.924 1.017.39.078.813-.094 1.267-.501zm-6.65-6.65c-.564-.987-1.409-1.581-2.504-1.816a4.109 4.109 0 00-.846-.094c-1.502 0-2.722.782-3.677 2.347-.814 1.33-1.22 2.801-1.22 4.413 0 1.205.25 2.238.75 3.099.564.986 1.393 1.58 2.504 1.815.298.063.58.094.846.094 1.517 0 2.738-.782 3.677-2.347.814-1.346 1.22-2.817 1.22-4.429 0-1.22-.25-2.238-.75-3.083zM23.77 10.03c-.22 1.033-.61 1.8-1.19 2.316-.453.407-.876.579-1.267.5-.376-.078-.689-.406-.924-1.016a3.971 3.971 0 01-.281-1.425c0-.39.031-.782.11-1.142a5.06 5.06 0 01.829-1.862c.516-.767 1.064-1.08 1.627-.97.376.078.689.407.924 1.017.187.485.281.97.281 1.424 0 .407-.031.798-.11 1.158z" clip-rule="evenodd"/> <path fill="#000" d="M42.236 5.555c-1.048 1.033-1.565 2.348-1.565 3.944 0 1.705.517 3.098 1.55 4.147 1.032 1.048 2.378 1.58 4.053 1.58.485 0 1.032-.078 1.627-.25V12.44c-.548.156-1.017.234-1.424.234-.83 0-1.487-.281-1.987-.829-.501-.563-.752-1.314-.752-2.269 0-.892.25-1.627.736-2.19.5-.58 1.111-.861 1.862-.861.485 0 1.002.078 1.565.234V4.225a6.537 6.537 0 00-1.706-.203c-1.596-.016-2.91.5-3.959 1.533zm10.86-1.549c-1.44 0-2.566.485-3.38 1.44-.813.954-1.205 2.3-1.205 4.021 0 1.863.407 3.287 1.205 4.273.798.986 1.972 1.486 3.506 1.486 1.486 0 2.629-.5 3.427-1.486s1.205-2.379 1.205-4.163c0-1.784-.407-3.161-1.22-4.131-.83-.955-2.004-1.44-3.538-1.44zm1.237 8.137c-.282.439-.704.658-1.236.658-.501 0-.877-.22-1.143-.658-.266-.438-.391-1.314-.391-2.644 0-2.05.516-3.067 1.565-3.067 1.095 0 1.658 1.032 1.658 3.114-.015 1.283-.172 2.16-.453 2.597zm11.251-7.84l-.563 2.395c-.141.61-.282 1.236-.407 1.877l-.313 1.66a142.34 142.34 0 00-1.22-5.932H59.45L58.088 14.96h2.723l.736-7.34 1.862 7.34h1.94l1.784-7.323.767 7.323h2.848l-1.44-10.657h-3.724zm13.036 0l-.564 2.395c-.14.61-.281 1.236-.407 1.877l-.313 1.66a142.34 142.34 0 00-1.22-5.932h-3.63L71.123 14.96h2.723l.735-7.34 1.862 7.34h1.94l1.77-7.323.766 7.323h2.848l-1.44-10.657H78.62zm8.857 6.448h2.55V8.544h-2.55V6.588h2.942V4.32h-5.822v10.657h5.838v-2.27h-2.958v-1.955zM98.54 9.06c.298-.486.454-.986.454-1.503 0-1.001-.39-1.8-1.173-2.378-.783-.58-1.862-.877-3.208-.877h-3.35V14.96h2.88v-4.85h.047l2.332 4.85h3.036l-2.3-4.804c.547-.25.985-.61 1.282-1.095zm-4.412-.173V6.353c.688.016 1.173.126 1.47.345.298.219.439.563.439 1.064 0 .735-.642 1.11-1.91 1.126zm6.556-3.333C99.636 6.588 99.12 7.903 99.12 9.5c0 1.705.516 3.098 1.549 4.147 1.033 1.048 2.378 1.58 4.053 1.58.485 0 1.033-.078 1.627-.25V12.44c-.547.156-1.017.234-1.424.234-.829 0-1.486-.281-1.987-.829-.501-.563-.751-1.314-.751-2.269 0-.892.25-1.627.735-2.19.501-.58 1.111-.861 1.862-.861.485 0 1.002.078 1.565.234V4.225a6.53 6.53 0 00-1.705-.203c-1.581-.016-2.911.5-3.96 1.533zm9.343 7.12v-1.94h2.55V8.529h-2.55V6.572h2.957V4.303h-5.821V14.96H113v-2.269h-2.973v-.016z"/> </svg>`,
  }),
  new MediaItem({
    key: 'drupal',
    icon: `<svg class="h-10 xmlns="http://www.w3.org/2000/svg"  viewBox="0 0 570.10212 140.79732"><path d="M347.4029,53.48845H329.79877l.02871,32.93891c0,9.62552-4.12693,17.41265-13.75245,17.41265-9.63031,0-13.79556-7.78713-13.79556-17.41265V53.51957H284.72082L284.716,86.42736c0,19.02129,12.22761,34.43746,31.25127,34.43746,19.01884,0,31.43077-15.41617,31.43077-34.43746Z" fill="#0678be" /><rect x="506.14274" y="33.41672" width="17.51792" height="85.74888" fill="#0678be" /><polygon points="536.595 36.58 536.595 33.499 550.247 33.499 550.247 36.58 545.253 36.58 545.253 50.158 541.591 50.158 541.591 36.58 536.595 36.58" fill="#0678be"  /><polygon points="557.314 33.5 561.211 44.954 561.257 44.954 564.941 33.5 570.102 33.5 570.102 50.158 566.672 50.158 566.672 38.352 566.624 38.352 562.542 50.158 559.718 50.158 555.636 38.469 555.588 38.469 555.588 50.158 552.16 50.158 552.16 33.5 557.314 33.5" fill="#0678be" /><path d="M270.40107,52.77245c-15.1002-3.48538-24.62518,11.51665-25.02255,12.321-.1939.395-.20108.62483-.869.608-.553-.01191-.61522-.608-.61522-.608L242.02,53.61753H226.32615v65.49233h17.59214v-35.45c0-2.8965,7.79906-16.78542,22.91359-13.2115,7.64346,1.80976,10.88708,5.0534,10.88708,5.0534V55.54928a28.10653,28.10653,0,0,0-7.31789-2.77683" fill="#0678be" /><path d="M388.32465,69.36661a17.55266,17.55266,0,1,1-17.549,17.549,17.55407,17.55407,0,0,1,17.549-17.549m-17.05111,71.3141V117.116l.00479.00482.00478-8.826s.02634-.70613.66307-.71337c.56734-.00716.69423.3687.83307.71337,1.33333,3.32021,8.668,15.962,24.94592,11.99778a34.67561,34.67561,0,1,0-44.08928-33.37694v53.76506Z" fill="#0678be" /><path d="M481.89532,86.91587a17.55266,17.55266,0,1,1-17.55145-17.549,17.55509,17.55509,0,0,1,17.55145,17.549M481.395,119.139h17.63761V86.91587a34.67441,34.67441,0,1,0-44.08928,33.37461c16.27795,3.9665,23.61259-8.67758,24.94593-11.99545.13883-.34467.26332-.72053.83306-.71571.63673.00957.66307.71571.66307.71571" fill="#0678be"  /><path d="M175.99834,48.224H168.927v55.92189l7.26284.18672c14.89671,0,24.49113-1.35485,24.49113-28.14164,0-25.68566-8.46932-27.967-24.68264-27.967m-4.77565,70.77789h-20.0099V33.38232h21.47012c25.99683,0,45.7147,4.7709,45.7147,42.80865,0,37.6738-20.886,42.81092-47.17492,42.81092" fill="#0678be" /><path d="M76.05087,29.98905C66.1436,20.08646,56.69111,10.64652,53.87792,0,51.0646,10.64652,41.61059,20.08646,31.705,29.98905,16.84646,44.83818,0,61.66742,0,86.90838a53.8795,53.8795,0,1,0,107.759,0c0-25.23931-16.84507-42.0702-31.70813-56.91933M22.90047,99.46319C19.59681,99.351,7.40431,78.33555,30.02336,55.95932L44.99129,72.30924a1.27945,1.27945,0,0,1-.09991,1.90927c-3.57172,3.66322-18.79529,18.92938-20.68747,24.20805-.39056,1.08955-.961,1.04834-1.30344,1.03663M53.87957,127.161A18.53041,18.53041,0,0,1,35.349,108.63041,19.30921,19.30921,0,0,1,39.96771,96.3905C43.309,92.30474,53.87709,80.81316,53.87709,80.81316S64.28339,92.47342,67.76153,96.3515a18.12182,18.12182,0,0,1,4.64858,12.27891A18.53058,18.53058,0,0,1,53.87957,127.161M89.347,97.111c-.39939.87346-1.30538,2.33166-2.5282,2.37618-2.17965.07938-2.41256-1.03746-4.0236-3.42177-3.537-5.23416-34.404-37.49382-40.17746-43.73278-5.07829-5.48746-.71512-9.35616,1.30881-11.38353,2.53922-2.54377,9.9511-9.95111,9.9511-9.95111S75.97811,51.96686,85.18433,66.29437,91.2178,93.01972,89.347,97.111" fill="#0678be" /></svg>`,
  }),
  new MediaItem({
    key: 'siteBuilder',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 236 64"> <mask id="a" width="14" height="22" x="0" y="21" maskUnits="userSpaceOnUse"> <path fill="#fff" d="M0 42.311V21.459h13.161v20.852H0z"/> </mask> <g mask="url(#a)"> <path fill="#1D79BC" fill-rule="evenodd" d="M5.681 30.014c-1.56-.164-3.238-1.121-3.238-2.836 0-1.877 1.543-3.295 3.591-3.303 1.412.08 2.434.492 3.413 1.375a.696.696 0 001.064-.163l.555-.944a.695.695 0 00-.121-.856c-1.19-1.127-3.076-1.828-4.92-1.828-3.378 0-6.025 2.512-6.025 5.72 0 2.828 1.906 4.687 5.367 5.236.16.035.367.061.626.094 2.44.308 4.697.956 4.697 3.582 0 2.24-1.565 3.804-3.804 3.804-.034 0-3.443-.026-4.887-2.264a.695.695 0 00-1.134-.046l-.638.833a.695.695 0 00-.024.811c2.041 3.017 6.49 3.08 6.701 3.082 3.567-.094 6.257-2.708 6.257-6.08 0-2.489-1.298-5.586-7.48-6.217" clip-rule="evenodd"/> </g> <mask id="b" width="3" height="22" x="16" y="21" maskUnits="userSpaceOnUse"> <path fill="#fff" d="M16.283 21.348h2.443v20.907h-2.443V21.348z"/> </mask> <g mask="url(#b)"> <path fill="#1D79BC" fill-rule="evenodd" d="M18.033 21.348h-1.055a.694.694 0 00-.693.694v19.52c0 .383.31.693.694.693h1.055c.383 0 .694-.31.694-.694V22.04a.696.696 0 00-.695-.692" clip-rule="evenodd"/> </g> <path fill="#1D79BC" fill-rule="evenodd" d="M37.955 21.486h-15.52a.694.694 0 00-.695.694v1.11c0 .384.31.694.694.694h6.524v17.742c0 .384.311.695.695.695h1.082a.694.694 0 00.694-.695V23.984h6.526c.383 0 .694-.31.694-.694v-1.11a.694.694 0 00-.694-.694zm15.451.083h-11.05a.694.694 0 00-.695.694v19.38c0 .384.31.695.694.695h10.8a.693.693 0 00.695-.694v-1.11a.694.694 0 00-.694-.695H44.16v-6.414h6.886c.383 0 .694-.31.694-.694v-1.083a.694.694 0 00-.694-.694H44.16v-6.942h9.246c.383 0 .694-.31.694-.694v-1.055a.694.694 0 00-.694-.694" clip-rule="evenodd"/> <path fill="#2C92D0" fill-rule="evenodd" d="M63.998 30.815h-4.443v-6.747h4.443a3.35 3.35 0 013.387 3.387c0 1.884-1.488 3.36-3.387 3.36zm1.416 8.996h-5.86v-6.47h5.776c1.85 0 3.193 1.343 3.193 3.193 0 1.899-1.307 3.277-3.11 3.277zm2.936-8.15l-.135-.08.114-.118.026-.03c1-1.16 1.529-2.536 1.529-3.978 0-3.214-2.613-5.855-5.83-5.886H57.75a.695.695 0 00-.695.694v19.325c0 .383.31.694.695.694h7.643c3.134-.032 5.685-2.61 5.685-5.748a5.664 5.664 0 00-2.728-4.872v-.001zM87.959 21.43h-1.166a.694.694 0 00-.694.694V34.84c0 2.694-2.205 4.887-4.915 4.887a4.92 4.92 0 01-4.914-4.915v-12.69a.694.694 0 00-.694-.693h-1.194a.694.694 0 00-.694.694v12.63c-.083 1.88.624 3.7 1.992 5.13 1.457 1.524 3.463 2.398 5.504 2.398 4.098 0 7.378-3.31 7.47-7.552V22.124a.694.694 0 00-.695-.694" clip-rule="evenodd"/> <mask id="c" width="3" height="22" x="92" y="21" maskUnits="userSpaceOnUse"> <path fill="#fff" d="M92.123 21.348h2.443v20.907h-2.443V21.348z"/> </mask> <g mask="url(#c)"> <path fill="#2C92D0" fill-rule="evenodd" d="M93.872 21.348h-1.055a.693.693 0 00-.694.694v19.52c0 .383.31.693.694.693h1.055c.383 0 .694-.31.694-.694V22.04a.694.694 0 00-.694-.692" clip-rule="evenodd"/> </g> <path fill="#2C92D0" fill-rule="evenodd" d="M108.809 39.839h-8.247v-17.66a.693.693 0 00-.694-.693h-1.22a.694.694 0 00-.695.694v19.463c0 .384.31.695.694.695h10.162a.695.695 0 00.694-.695v-1.11a.694.694 0 00-.694-.694zm6.204-.03l-.025-15.797h2.83c4.23 0 7.802 3.586 7.802 7.83 0 4.32-3.572 7.969-7.802 7.969h-2.805v-.001zm2.945-18.351h-4.804a.696.696 0 00-.694.694v19.436c0 .383.31.694.694.694h4.811c5.66-.06 10.266-4.743 10.266-10.44 0-5.636-4.604-10.295-10.273-10.384zm25.057.11h-11.05a.699.699 0 00-.642.429.693.693 0 00-.053.266v19.38c0 .384.311.695.694.695h10.801a.694.694 0 00.694-.694v-1.111a.694.694 0 00-.694-.694h-8.996v-6.414h6.886c.384 0 .694-.31.694-.694v-1.083a.694.694 0 00-.694-.694h-6.886v-6.942h9.246c.383 0 .694-.31.694-.694v-1.055a.694.694 0 00-.694-.694zm6.122 2.278h3.249a4.448 4.448 0 014.442 4.442 4.45 4.45 0 01-4.442 4.443h-3.249v-8.885zm10.184 17.54l-3.279-7.281.11-.07c1.936-1.26 3.092-3.41 3.092-5.747 0-3.72-3.022-6.796-6.747-6.858h-5.109a.696.696 0 00-.694.694V41.67c0 .384.31.694.694.694h1.055c.383 0 .694-.31.694-.694V35.17l3.362-.023c.23 0 .692 0 1.267-.107l3.124 6.918c.112.248.36.408.632.408h1.167a.696.696 0 00.632-.98z" clip-rule="evenodd"/> <path fill="#1D79BC" fill-rule="evenodd" d="M188.661 23.568l-.583 1.083c-1.305-.833-2.832-1.22-4.415-1.22-4.637 0-8.551 3.886-8.551 8.522 0 4.637 3.914 8.525 8.551 8.525a8.45 8.45 0 004.47-1.278l.556 1.11a9.714 9.714 0 01-5.026 1.39c-5.359 0-9.745-4.36-9.745-9.747 0-5.386 4.386-9.773 9.745-9.773 1.777 0 3.499.5 4.998 1.388zm19.658 8.274c0-4.636-3.776-8.552-8.274-8.552-4.47 0-8.274 3.916-8.274 8.552 0 4.665 3.803 8.58 8.274 8.58 4.498 0 8.274-3.915 8.274-8.58zm1.027 0c0 5.36-4.165 9.746-9.301 9.746-5.137 0-9.302-4.387-9.302-9.746 0-5.358 4.165-9.717 9.302-9.717 5.136 0 9.301 4.36 9.301 9.717zm14.605 8.22l-7.247-16.633-3.554 18.16h-1.111l3.748-19.603 1.527.027 6.637 15.216 6.58-15.216 1.527-.027 3.776 19.602h-1.111l-3.526-18.159-7.246 16.632z" clip-rule="evenodd"/> <mask id="d" width="10" height="16" x="165" y="37" maskUnits="userSpaceOnUse"> <path fill="#fff" d="M165.771 52.404h9.191v-15.4h-9.191v15.4z"/> </mask> <g mask="url(#d)"> <path fill="#F4A81C" fill-rule="evenodd" d="M174.962 46.322l-9.191-9.318.869 13.059 2.901-2.537 2.193 4.878 1.695-.762-2.19-4.872 3.723-.448z" clip-rule="evenodd"/> </g> </svg>`,
  }),
  new MediaItem({
    key: 'duda',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 600 254"> <defs/> <g clip-path="url(#clip0)"> <path fill="#F66135" d="M600 226.3c0 15.4-12.6 28-28 28H28c-15.4 0-28-12.6-28-28V28C0 12.6 12.6 0 28 0h544.1c15.4 0 28 12.6 28 28v198.3h-.1z"/> <path fill="#fff" d="M128.499 167.5v-19.2c-1.6-13.3-9.9-21.1-24.9-23.4-19 1.5-29.7 11.4-32.1 29.6.8 22.2 11.5 33.6 32.1 34.4 15-2.4 23.3-9.5 24.9-21.4zm33.3-129.3v178h-33.2v-9.5c-8.9 6.3-17.6 9.5-26.1 9.5-18.2 0-32.6-5.9-43.3-17.8-10.7-11.9-16-25.7-16-41.6 0-15.9 5.3-29.7 16-41.6 10.7-11.9 25.1-17.8 43.3-17.8 11.9.8 20.6 4 26.1 9.6V38l33.2.2zm97.4 135.8V97.3h33.2v118.8h-33.2v-10.8c-8 7.1-17.5 10.7-28.5 10.7-15 0-26.5-4.6-34.4-13.7-7.9-9.1-11.9-19.6-11.9-31.5V97.2h33.2v65.3c0 7.9 2 14.1 5.9 18.4 4 4.3 9.9 6.6 17.9 6.9 7-.4 12.9-5 17.8-13.8zm138-6.5v-19.2c-1.6-13.3-9.9-21.1-24.9-23.4-19 1.5-29.7 11.4-32.1 29.6.8 22.2 11.5 33.6 32.1 34.4 15-2.4 23.3-9.5 24.9-21.4zm33.2-129.3v178h-33.2v-9.5c-8.9 6.3-17.6 9.5-26.1 9.5-18.2 0-32.6-5.9-43.3-17.8-10.7-11.9-16-25.7-16-41.6 0-15.9 5.3-29.7 16-41.6 10.7-11.9 25.1-17.8 43.3-17.8 11.9.8 20.6 4 26.1 9.6V38l33.2.2zm93.4 142.4v-13.4c-3.2-6.1-10.7-9.1-22.6-9.1-14.2 1.6-21.8 7.1-22.6 16.6.8 9.4 8.3 14.5 22.6 15.3 11.9-.1 19.4-3.2 22.6-9.4zm-65.3-68.9c15-9.5 32.5-14.2 52.2-14.2 14.2 0 25.5 4 33.8 11.9 8.3 7.9 12.5 19.8 12.5 35.6v71.2h-33.2v-9.5c-9.7 6.3-18.4 9.5-26.1 9.5-15.8 0-27.7-3.4-35.6-10.1-7.9-6.7-11.9-16-11.9-27.9 0-11.1 3.7-21 11.2-29.7 7.5-8.7 19.6-13.1 36.3-13.1 7.7 0 16.4 2.4 26.1 7.1V139c-.2-8.7-6.5-13.5-19-14.3-14.2 0-25.3 2.4-33.2 7.1l-13.1-20.1z"/> </g> <defs> <clipPath id="clip0"> <path fill="#fff" d="M0 0h600v254H0z"/> </clipPath> </defs> </svg>`,
  }),
  new MediaItem({
    key: 'jimdo',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 100 37"> <path fill="#010334" d="M82.37 9.905a8.58 8.58 0 00-8.579 8.58 8.58 8.58 0 008.58 8.58 8.58 8.58 0 008.58-8.58 8.58 8.58 0 00-8.58-8.58zm0 13.393a4.813 4.813 0 110-9.625 4.813 4.813 0 010 9.625zM35.137 10.23h3.608l5.469 10.267 5.444-10.267h3.633v16.513h-4.057V17.92l-4.618 8.823h-.803l-4.618-8.8v8.8h-4.057V10.23h-.001zm-12.252.057h-4.072v9.86a2.847 2.847 0 01-5.693 0H9.048a6.919 6.919 0 1013.837 0v-9.86zm40.54-.057h-6.1v16.513h6.1a8.256 8.256 0 100-16.513zm-.495 13.068h-1.546v-9.625h1.546a4.812 4.812 0 110 9.625zM31.04 10.287h-4.086v16.4h4.086v-16.4z"/> <path fill="#010334" d="M97.537 2.463V34.51H2.463V2.463h95.074zM100 0H0v36.973h100V0z"/> </svg>`,
  }),
  new MediaItem({
    key: 'zyro',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 60 29"> <path fill-rule="evenodd" d="M54.684 11.343a4.073 4.073 0 01-2.303 1.962c-1.012.355-1.974.31-2.886-.132-.916-.46-1.514-1.182-1.795-2.17-.285-.995-.173-1.977.335-2.947.515-.953 1.275-1.615 2.283-1.984 1.012-.37 1.975-.336 2.89.102.934.456 1.545 1.19 1.834 2.2.286 1.001.166 1.992-.358 2.97zM53.21 1.063c-2.314-.425-4.453.061-6.417 1.45-1.953 1.38-3.163 3.243-3.64 5.587-.468 2.338-.078 4.481 1.166 6.442 1.24 1.954 2.981 3.168 5.235 3.646 2.264.472 4.392.056 6.388-1.26 2.004-1.322 3.281-3.168 3.82-5.542.536-2.396.165-4.61-1.116-6.628-1.286-2.026-3.102-3.259-5.436-3.695zM.153 12.57V8.25l11.776-1.545v3.652l-6.184 8.666 6.667-.733v4.435L0 23.978v-3.564l6.037-8.543-5.884.7zm24.795-7.983v8.237c0 1.017-.334 1.923-.998 2.718-.654.792-1.445 1.238-2.371 1.34-.923.1-1.71-.17-2.362-.812-.65-.64-.975-1.46-.975-2.462V5.479l-4.159.554v8.061c0 2.21.72 4.031 2.169 5.469 1.456 1.427 3.23 2.04 5.327 1.83 1.542-.154 2.667-.69 3.369-1.605v.57a4.064 4.064 0 01-.998 2.701c-.654.783-1.445 1.216-2.371 1.304 0 0-2.454.308-4.095-1.444l-2.582 2.214-.832.713s2.606 3.056 6.714 3.068l.002-.005c.26.006.524.003.793-.02 2.097-.178 3.905-1.14 5.418-2.891 1.521-1.744 2.29-3.764 2.3-6.054V4.008l-4.35.58zm16.64-2.133v3.91c-4.375.391-4.985 3.58-5.068 4.627l-.134 8.948-4.053.35V3.668l4.305-.565-.01.75a3.81 3.81 0 01.608-.423c1.494-.946 4.352-.976 4.352-.976z" clip-rule="evenodd"/> </svg>`,
  }),
  new MediaItem({
    key: 'pinegrow',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill-rule="evenodd" stroke-linejoin="round" stroke-miterlimit="1.414" clip-rule="evenodd" viewBox="0 0 188 37"> <path fill="none" d="M0 0h187.982v36.105H0z"/> <path fill="#0198cd" fill-rule="nonzero" d="M59.542 20.277c0 .451.036.901.14 1.318.693 2.564 2.911 4.228 5.477 4.228 3.812 0 5.996-3.086 5.996-7.523 0-3.743-2.08-7.105-5.858-7.105-2.426 0-4.784 1.664-5.511 4.402-.104.451-.244.971-.244 1.421v3.259zm-1.663-4.784c0-2.08-.035-3.71-.139-5.303h1.594l.104 3.154h.07c1.215-2.184 3.259-3.536 6.17-3.536 4.264 0 7.141 3.536 7.141 8.424 0 5.927-3.535 8.979-7.557 8.979-2.391 0-4.506-1.075-5.65-3.085h-.07v9.566h-1.663V15.493zM77.05 10.19h1.664v16.639H77.05V10.19zm.763-3.328c-.763 0-1.283-.658-1.283-1.422 0-.761.589-1.456 1.352-1.456.762 0 1.318.659 1.318 1.456 0 .764-.521 1.422-1.353 1.422h-.034zM84.295 14.176c0-1.525-.035-2.669-.139-3.986h1.56l.103 2.982h.07c.902-1.873 3.017-3.364 5.72-3.364 1.595 0 5.824.797 5.824 7.037v9.984h-1.664v-9.844c0-3.052-1.179-5.789-4.611-5.789-2.322 0-4.332 1.663-4.991 3.846a5.897 5.897 0 00-.208 1.457v10.33h-1.664V14.176zM113.727 17.019c.035-2.322-.937-5.823-4.958-5.823-3.674 0-5.233 3.259-5.545 5.823h10.503zm-10.607 1.318c0 5.303 2.912 7.418 6.343 7.418 2.426 0 3.674-.486 4.611-.937l.416 1.284c-.624.346-2.324 1.073-5.235 1.073-4.784 0-7.764-3.466-7.764-8.353 0-5.546 3.188-9.013 7.487-9.013 5.442 0 6.482 5.13 6.482 7.522 0 .451 0 .693-.069 1.006H103.12zM131.303 16.603c0-.624-.035-1.144-.209-1.629-.692-1.975-2.426-3.778-5.165-3.778-3.535 0-5.962 2.876-5.962 7.278 0 3.744 1.976 6.968 5.894 6.968 2.218 0 4.296-1.386 5.165-3.743a6.122 6.122 0 00.277-1.769v-3.327zm1.664 7.8c0 4.263-.901 6.377-2.289 7.696-1.525 1.455-3.777 1.975-5.65 1.975-1.907 0-4.021-.486-5.304-1.456l.59-1.317c1.178.761 2.807 1.386 4.957 1.386 3.501 0 6.032-1.803 6.032-6.725v-2.357h-.07c-.867 1.7-2.808 3.224-5.789 3.224-4.194 0-7.14-3.641-7.14-8.181 0-5.789 3.812-8.839 7.487-8.839 3.293 0 4.922 1.802 5.616 3.19h.069l.104-2.809h1.526c-.105 1.248-.139 2.565-.139 4.298v9.915zM138.235 15.182c0-1.664-.034-3.362-.138-4.992h1.526l.069 3.363h.104c.728-2.115 2.461-3.745 4.714-3.745.243 0 .485.035.694.069v1.561c-.243-.035-.486-.035-.798-.035-2.287 0-3.917 1.976-4.367 4.645-.105.486-.139 1.04-.139 1.594v9.187h-1.665V15.182zM154.46 25.823c3.467 0 6.205-3.119 6.205-7.384 0-3.05-1.698-7.243-6.101-7.243-4.263 0-6.17 3.777-6.17 7.348 0 4.16 2.635 7.279 6.031 7.279h.035zm-.104 1.388c-4.298 0-7.625-3.364-7.625-8.563 0-5.685 3.743-8.84 7.868-8.84 4.61 0 7.73 3.467 7.73 8.562 0 6.17-4.264 8.841-7.938 8.841h-.035zM166.074 10.19l2.808 9.462c.555 1.943 1.04 3.606 1.421 5.202h.07c.416-1.526 1.006-3.295 1.664-5.202l3.328-9.462h1.56l3.188 9.358c.694 2.047 1.213 3.744 1.629 5.306h.07c.347-1.562.866-3.225 1.526-5.235l2.946-9.429h1.698l-5.546 16.639h-1.422l-3.153-9.29c-.694-2.011-1.283-3.779-1.734-5.651h-.069c-.486 1.942-1.11 3.778-1.802 5.72l-3.294 9.221h-1.456l-5.13-16.639h1.698z"/> <g opacity=".9"> <clipPath id="a"> <path d="M0-.003h35.022v36.11H0z"/> </clipPath> <g clip-path="url(#a)"> <path fill="#0198cd" fill-rule="nonzero" d="M32.959 23.102l-2.029 2.444c-.318.389-1.073.543-1.521.313l-6.088-3.13a.293.293 0 00-.041-.023.162.162 0 01.041-.022l5.168-2.774c.538-.287 1.623-.287 2.163-.003l.564.3c.707.375 1.641 1.124 2.002 1.605.153.205.129.822-.259 1.29m-8.532 10.292c-.315.384-1.043.521-1.479.284l-.264-.145-.992-.551a.697.697 0 01-.15-.101.805.805 0 01.142-.111l6.691-4.128-3.948 4.752zm-11.74.993c-.669 0-1.673-.474-2.103-.993l-1.825-2.199c-.116-.146-.167-.3-.139-.439.026-.139.131-.264.295-.349l1.214-.653c.264-.139.647-.223 1.058-.223.414 0 .822.09 1.088.236l8.388 4.62h-7.976zM3.784 25.19l5.53 2.968.07.039a.406.406 0 01-.07.039l-1.212.651a1.143 1.143 0 01-.529.121c-.379 0-.76-.16-.969-.411l-2.82-3.407zm-1.767-3.674l3.651-4.868c.291-.39.995-.555 1.431-.327l3.047 1.567c.018.009.033.016.045.024a.16.16 0 01-.045.025l-7.019 3.786c-.405.218-.918.357-1.311.357-.031 0-.056 0-.082-.005.031-.129.111-.332.283-.559m5.674-7.569l2.788-3.713c.195-.258.558-.426.927-.426.177 0 .345.037.485.112l2.513 1.339h.001l4.04 2.151c.005 0 .008.003.012.006.022.009.041.024.055.033-.016.01-.037.024-.061.035l-5.325 2.854c-.276.145-.694.233-1.12.233-.417 0-.806-.078-1.072-.213l-3.047-1.57a.57.57 0 01-.319-.37c-.034-.151.01-.316.123-.471m4.834-6.371c1.877-2.371 4.196-5.078 5.032-5.765.504.355 1.576 1.45 2.496 2.573.119.144.168.3.138.441-.026.139-.133.259-.295.351l-5.88 3.125c-.533.158-1.239.141-1.551-.023a.223.223 0 01-.124-.162c-.026-.115.011-.32.184-.54m11.587 2.1c.116.155.16.326.126.482a.594.594 0 01-.317.389l-2.5 1.339c-.535.29-1.618.29-2.158.006l-.002-.003-4.049-2.154a.353.353 0 01-.064-.035l.064-.037 5.498-2.962c.425-.23 1.114-.075 1.404.311l1.998 2.664zm5.993 7.987l-1.613-.86c-.005 0-.005 0-.006-.004l-6.245-3.321c-.001-.002-.004-.002-.009-.002-.022-.014-.042-.023-.057-.036a.675.675 0 01.062-.037l2.497-1.34c.426-.226 1.117-.071 1.407.31l3.964 5.29zM21.388 21.4c-.413 0-.803-.078-1.07-.218l-6.379-3.28c-.015-.008-.029-.014-.043-.024.014-.004.028-.014.043-.023l5.324-2.853c.538-.286 1.622-.29 2.163-.006h.004l6.26 3.335c.02.006.034.015.05.025-.02.013-.042.024-.064.037l-5.169 2.772c-.275.147-.694.235-1.119.235m6.265 5.491c.05.03.093.056.122.077a.764.764 0 01-.118.088l-6.876 4.248c-.498.309-1.535.333-2.049.048l-5.627-3.096a.523.523 0 01-.081-.049l.081-.049 7.239-3.879c.541-.29 1.644-.297 2.189-.019l2.262 1.163 2.858 1.468zM3.938 23.315a.243.243 0 01-.066-.035l.07-.043 7.021-3.782c.538-.291 1.644-.301 2.186-.02l6.38 3.281a.178.178 0 01.043.023c-.014.003-.027.012-.043.023l-7.236 3.878c-.54.292-1.625.292-2.164 0l-6.191-3.325zm30.659-2.538c-.291-.386-.787-.848-1.326-1.269 0-.495-.333-.94-.48-1.139l-5.273-7.028a2.8 2.8 0 00-1.576-1.018 2.331 2.331 0 00-.452-1.682l-2-2.665a2.8 2.8 0 00-1.593-1.027 2.262 2.262 0 00-.514-1.658C18.684 0 17.796 0 17.507 0c-.416 0-1.186 0-6.332 6.507a2.455 2.455 0 00-.544 1.682A2.818 2.818 0 009.1 9.202l-2.785 3.712a2.322 2.322 0 00-.451 1.673 2.82 2.82 0 00-1.575 1.029L.638 20.483c-.52.693-.737 1.462-.596 2.101.032.143.085.27.148.389-.034.528.352 1.004.538 1.227l4.551 5.494c.404.485.986.824 1.63.966a2.262 2.262 0 00.528 1.633l1.823 2.2c.762.921 2.235 1.612 3.427 1.612h9.635c.884 0 1.292-.294 1.48-.616.768-.079 1.48-.429 1.952-.992l4.361-5.255c.326-.393.815-1.088.883-1.747.492-.176.938-.459 1.26-.848l2.024-2.447c.85-1.023.986-2.526.315-3.423"/> </g> </g> </svg>`,
  }),
  new MediaItem({
    key: 'unbounce',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" version="1.1" viewBox="0 0 553.7 135" xml:space="preserve"> <path id="e_3_" d="M523.7 91.3c7.7 0 15.5-2.6 21.2-7.1.1-.1.1-.2 0-.2l-5.6-6.8c-.1-.1-.2-.1-.2 0-4.7 3.3-9.8 5-15.2 5-8.1 0-13-3.8-14.2-10.9l-.1-.8h36.5V68c0-13.4-10.8-24.4-24-24.5h-.3c-7.2 0-13.6 2.4-18.1 6.9-4.3 4.3-6.7 10.4-6.7 17 .1 14.5 10.5 23.9 26.7 23.9zM510.6 61c1.7-5.3 5.7-8.2 11.2-8.2s9.4 2.9 11.2 8.2c.2.5.1 1-.2 1.4s-.8.7-1.3.7h-19.3c-.5 0-1-.3-1.3-.7-.4-.4-.5-.9-.3-1.4z" class="st0"/> <path id="c_3_" d="M450.1 67.6c0 13.6 11 23.9 25.7 23.9 7.2 0 13.8-2.5 18.6-7.1v-.3l-6.8-7.4c-2.6 3.3-6.7 5.3-11 5.3-2.2 0-4.3-.5-6.3-1.5-.1 0-.2-.1-.3-.2l-.1-.1-.2-.1c-2.8-1.4-5.1-3.9-6.3-6.9-.3-.7-.5-1.4-.7-2.2-.1-.4-.2-.7-.2-1l-.1-.5c0-.2-.1-.5-.1-.7v-.9-.3-1.1-.2c.2-1.6.6-3.3 1.3-4.8.1-.2.2-.5.3-.7 1.3-2.6 3.3-4.5 5.7-5.8 2.1-1.2 4.5-1.8 7-1.8 4.3 0 8.3 1.9 11.1 5.3l6.8-7.6c-4.8-4.6-11.5-7.2-18.8-7.2-14.6 0-25.6 10.3-25.6 23.9z" class="st0"/> <path id="n_7_" d="M443.3 89.3V66c-.4-10.6-3.3-20.4-21-20.4-3.7 0-7 .5-9.8 1.6l-1 .4v-1.8c0-.1-.1-.1-.1-.1h-9.9c-.1 0-.1.1-.1.1v43.5c0 .1.1.2.2.2H413c.1 0 .2-.1.2-.2V64.8c0-1.4.2-2.7.5-3.9 1.3-3.4 4.8-5.8 8.7-5.9 3.8 0 7.4 2.4 8.7 5.8.4 1.2.5 2.5.5 3.9v24.5c0 .1.1.2.2.2l11.5-.1c-.1.2 0 .1 0 0z" class="st0"/> <path id="u_7_" d="M349.6 45.8v23.4c.4 10.6 3.3 20.4 21 20.4 18 0 20.6-12 21-20.9V45.8c0-.1-.1-.2-.2-.2H380c-.1 0-.2.1-.2.2v24.5a16 16 0 01-.5 3.9c-1.3 3.4-4.8 5.8-8.7 5.9-3.8 0-7.4-2.4-8.7-5.8-.4-1.2-.5-2.5-.5-3.9V45.8c0-.1-.1-.2-.2-.2h-11.4c-.1 0-.2.1-.2.2z" class="st0"/> <path id="o_3_" d="M316.5 43.5c-14.6 0-25.7 10.3-25.7 23.9s11 23.9 25.7 23.9 25.8-10.3 25.8-23.9c0-13.7-11.1-23.9-25.8-23.9zm0 37.9c-7.6 0-13.4-6-13.4-14 0-7.9 5.9-14 13.4-14 7.6 0 13.5 6.2 13.5 14s-6 14-13.5 14z" class="st0"/> <path id="b_3_" d="M264 90.5c12.1 0 21.3-10 21.3-23.2 0-13.1-9.3-23.4-21.3-23.4-6.4 0-12.2 2.6-16.3 7.4l-1.3 1.4V23.5c0-.1-.1-.2-.2-.2h-11.4c-.1 0-.2.1-.2.2v65.8c0 .1.1.2.2.2h10.7c.1 0 .2-.1.2-.2v-8.2l1.3 1.5c4.1 5.1 10.1 7.9 17 7.9zm-2.5-36.6c7.6 0 13.2 5.6 13.2 13.3 0 7.6-5.7 13.3-13.2 13.3s-13.4-5.8-13.4-13.2c0-7.6 5.8-13.4 13.4-13.4z" class="st0"/> <path id="n_6_" d="M225.4 89.5c.1 0 .2-.1.2-.2V66c-.4-10.6-3.3-20.4-21-20.4-3.7 0-7 .5-9.8 1.6l-1 .4v-1.8c0-.1-.1-.1-.2-.1h-9.9c-.1 0-.1.1-.1.1v43.5c0 .1.1.2.2.2h11.4c.1 0 .2-.1.2-.2V64.8c0-1.4.2-2.7.5-3.9 1.3-3.4 4.8-5.8 8.7-5.9 3.8 0 7.4 2.4 8.7 5.8.4 1.2.5 2.5.5 3.9v24.5c0 .1.1.2.2.2h11.4v.1z" class="st0"/> <path id="u_6_" d="M132.6 45.8v23.4c.4 10.6 3.3 20.4 21 20.4 18 0 20.6-12 21-20.9V45.8c0-.1-.1-.2-.2-.2H163c-.1 0-.2.1-.2.2v24.5a16 16 0 01-.5 3.9c-1.3 3.4-4.8 5.8-8.7 5.9-3.8 0-7.4-2.4-8.7-5.8-.4-1.2-.5-2.5-.5-3.9V45.8c0-.1-.1-.2-.2-.2h-11.4c-.2 0-.2.1-.2.2z" class="st0"/> <path d="M60.8 14.3c-29.4 0-53.2 23.8-53.2 53.2s23.8 53.2 53.2 53.2S114 96.9 114 67.5 90.2 14.3 60.8 14.3zm-1.9 74.1L55 99.9c-.1.4-.3.7-.5 1-.5.8-1.1 1.4-1.9 1.8-.7.4-1.5.6-2.3.6-.5 0-1.1-.1-1.6-.3-2.2-.8-3.6-3-3.3-5.2-11-5.6-18.5-17.1-18.5-30.3 0-18.8 15.2-34.1 34.1-34.1 2.2 0 4.3.2 6.4.6 1.2-1.9 3.6-2.8 5.8-2.1 1.2.4 2.1 1.2 2.7 2.2.5.9.8 1.9.7 3 0 .4-.1.8-.2 1.1l-3.9 11.5-13.6 38.7zm4.3 13.1l4.7-14.2c8.1-2.9 13.9-10.6 13.9-19.8 0-4.3-1.3-8.4-3.6-11.7l4.7-14.2c7.3 6.2 12 15.5 12 25.9-.1 18-14 32.7-31.7 34z" class="st0"/> <path d="M39.8 67.5c0 7.4 3.9 13.9 9.7 17.7l13.2-38.6c-.6-.1-1.3-.1-1.9-.1-11.6 0-21 9.4-21 21z" /> </svg>`,
  }),
  new MediaItem({
    key: 'goDaddy',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg"  viewBox="0 0 166 34"> <path d="M30.824 1.453c-3.713-2.316-8.603-1.77-12.963.972-4.347-2.74-9.239-3.288-12.948-.972C-.955 5.12-1.667 14.564 3.32 22.546c3.677 5.887 9.428 9.335 14.548 9.271 5.119.064 10.87-3.384 14.547-9.27C37.4 14.563 36.692 5.12 30.824 1.452zM6.02 20.863a19.112 19.112 0 01-2.303-5.273 15.067 15.067 0 01-.5-4.97C3.44 7.673 4.64 5.375 6.598 4.15c1.957-1.224 4.544-1.295 7.299-.203.419.169.829.358 1.228.568a22.496 22.496 0 00-3.956 4.754C8.14 14.116 7.215 19.511 8.272 23.81a19.723 19.723 0 01-2.253-2.947zm26.003-5.273a19.158 19.158 0 01-2.302 5.272 19.733 19.733 0 01-2.254 2.954c.946-3.855.303-8.576-2.014-13.005a.583.583 0 00-.837-.228l-7.225 4.515a.59.59 0 00-.188.813l1.06 1.695a.59.59 0 00.814.188l4.683-2.925c.151.454.303.901.416 1.363.442 1.617.611 3.297.5 4.97-.223 2.95-1.424 5.247-3.38 6.471a6.607 6.607 0 01-3.35.97h-.15a6.599 6.599 0 01-3.35-.97c-1.958-1.224-3.159-3.521-3.381-6.472a15.09 15.09 0 01.5-4.969 19.671 19.671 0 016.03-9.65 15.108 15.108 0 014.241-2.629c2.748-1.092 5.34-1.02 7.298.203 1.958 1.224 3.157 3.522 3.38 6.47a15.105 15.105 0 01-.49 4.964z" /> <path d="M74.065 12.892c-4.06 0-7.34 3.169-7.34 7.2 0 4.004 3.28 7.121 7.34 7.121 4.086 0 7.366-3.112 7.366-7.12 0-4.03-3.275-7.2-7.366-7.2zm0 10.557c-1.871 0-3.295-1.513-3.295-3.384s1.424-3.407 3.295-3.407c1.898 0 3.322 1.54 3.322 3.412 0 1.87-1.424 3.385-3.322 3.385v-.006zM90.583 7.362h-7.468a.6.6 0 00-.61.612v18.208a.605.605 0 00.61.648h7.468c5.977 0 10.13-3.975 10.13-9.758 0-5.818-4.153-9.71-10.13-9.71zm.177 15.622h-4.087V11.198h4.087c3.308 0 5.588 2.474 5.588 5.866 0 3.336-2.28 5.92-5.588 5.92zm24.82-9.7h-2.809a.633.633 0 00-.582.612v.833c-.64-1.057-2.085-1.835-3.884-1.835-3.503 0-6.783 2.751-6.783 7.145 0 4.37 3.251 7.171 6.755 7.171 1.806 0 3.28-.777 3.92-1.834v.861a.587.587 0 00.582.585h2.808a.571.571 0 00.584-.585V13.896a.594.594 0 00-.592-.612zm-6.533 10.196c-1.86 0-3.256-1.43-3.256-3.412s1.397-3.41 3.256-3.41c1.86 0 3.257 1.426 3.257 3.408 0 1.983-1.395 3.412-3.257 3.412v.002zm22.294-16.118h-2.808a.592.592 0 00-.61.584v6.728c-.648-1.002-2.114-1.78-3.948-1.78-3.476 0-6.7 2.751-6.7 7.145 0 4.37 3.252 7.171 6.755 7.171 1.806 0 3.17-.777 3.92-1.834v.861c0 .322.261.583.583.585h2.808a.57.57 0 00.584-.585V7.95a.57.57 0 00-.584-.588zm-6.532 16.152c-1.852 0-3.237-1.444-3.237-3.448 0-2.003 1.39-3.447 3.237-3.447 1.846 0 3.237 1.444 3.237 3.447 0 2.004-1.384 3.448-3.237 3.448zm22.29-16.152h-2.803a.594.594 0 00-.612.584v6.728c-.64-1.002-2.114-1.78-3.947-1.78-3.477 0-6.7 2.751-6.7 7.145 0 4.37 3.253 7.171 6.755 7.171 1.807 0 3.168-.777 3.92-1.834v.861a.587.587 0 00.584.585h2.803a.568.568 0 00.582-.585V7.95a.568.568 0 00-.582-.588zm-6.532 16.152c-1.853 0-3.237-1.444-3.237-3.448 0-2.003 1.39-3.447 3.237-3.447 1.846 0 3.237 1.444 3.237 3.447 0 2.004-1.38 3.448-3.232 3.448h-.005zm21.906-9.283l-4.19 14.37c-.809 2.556-2.613 4.086-5.421 4.086-1.277 0-2.44-.261-3.35-.782-.531-.303-.971-.58-.971-1.023 0-.275.089-.417.25-.675l.832-1.246c.235-.348.408-.461.66-.461a.96.96 0 01.554.192c.523.339 1.008.63 1.748.63.864 0 1.524-.277 1.88-1.306l.36-1.193h-1.696c-.418 0-.648-.249-.751-.584l-3.75-12.008c-.14-.473-.011-.946.683-.946h2.954c.36 0 .613.123.771.64l2.77 9.67 2.589-9.67c.082-.334.306-.64.75-.64h2.802c.552-.001.719.387.526.946zm-96.806 4.274v7.676a.625.625 0 01-.635.634h-2.317a.623.623 0 01-.634-.634v-2.015c-1.472 1.858-4.03 3.028-6.924 3.028-5.434 0-9.681-4.088-9.681-9.908 0-6.048 4.585-10.217 10.377-10.217 4.276 0 7.694 1.839 9.212 5.537a.843.843 0 01.07.309c0 .175-.116.307-.486.435l-2.706 1.042a.694.694 0 01-.511.009.783.783 0 01-.324-.371c-.971-1.847-2.7-3.1-5.36-3.1-3.45 0-5.922 2.694-5.922 6.188 0 3.387 2.104 6.172 6.02 6.172 2.06 0 3.703-.97 4.469-2.037H57.87a.625.625 0 01-.634-.635v-2.086a.625.625 0 01.634-.634h7.161a.604.604 0 01.635.607z"/> <path d="M162.55 10.442v-.186c0-.013.004-.026.012-.036a.052.052 0 01.034-.013h1.183c.013 0 .025.005.034.013.009.01.013.023.013.036v.186a.052.052 0 01-.013.034.047.047 0 01-.034.013h-.432v1.167a.05.05 0 01-.048.048h-.224a.044.044 0 01-.032-.014.047.047 0 01-.015-.034v-1.167h-.432a.044.044 0 01-.047-.047zm2.068-.193l.323.75.324-.75a.066.066 0 01.026-.032.087.087 0 01.044-.01h.375a.044.044 0 01.047.046v1.404a.044.044 0 01-.045.047h-.21a.047.047 0 01-.035-.013.042.042 0 01-.013-.034v-1.06l-.348.77a.084.084 0 01-.026.038.08.08 0 01-.043.01h-.172a.078.078 0 01-.042-.01.073.073 0 01-.026-.037l-.346-.77v1.06a.052.052 0 01-.013.033.052.052 0 01-.032.013h-.216a.042.042 0 01-.03-.013.042.042 0 01-.013-.034v-1.404a.044.044 0 01.047-.046h.364a.09.09 0 01.042.01c.009.009.015.02.018.032z"/> <path d="M27.889 30.386v-.21c0-.015.005-.03.015-.04a.048.048 0 01.038-.015h1.342a.046.046 0 01.038.015c.01.01.016.025.015.04v.21a.051.051 0 01-.015.04.048.048 0 01-.038.014h-.49v1.322a.05.05 0 01-.017.04.051.051 0 01-.038.017h-.253a.055.055 0 01-.038-.017.06.06 0 01-.015-.04V30.44h-.49a.048.048 0 01-.039-.015.051.051 0 01-.015-.04zm2.345-.217l.374.848.375-.848a.073.073 0 01.03-.036.08.08 0 01.05-.012h.4a.051.051 0 01.053.053v1.592a.056.056 0 01-.013.037.05.05 0 01-.037.016h-.24a.055.055 0 01-.038-.016.056.056 0 01-.015-.037v-1.203l-.393.874a.09.09 0 01-.03.04.086.086 0 01-.05.013h-.194a.085.085 0 01-.048-.012.09.09 0 01-.03-.041l-.393-.874v1.203a.05.05 0 01-.015.037.053.053 0 01-.036.016h-.243a.05.05 0 01-.036-.016.056.056 0 01-.015-.037v-1.592a.05.05 0 01.017-.038.047.047 0 01.036-.015h.413a.083.083 0 01.049.012.071.071 0 01.029.036z" /> </svg>`,
  }),
  new MediaItem({
    key: 'googleTagManager',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 123 38"> <g clip-path="url(#clip0)"> <path fill="#767677" d="M59.57 8.3h-7v2.1h5c-.2 2.9-2.7 4.2-5 4.2a5.459 5.459 0 01-5.5-5.5V9a5.132 5.132 0 011.6-3.9 5.7 5.7 0 013.9-1.6 4.76 4.76 0 013.7 1.5l1.5-1.5a7.4 7.4 0 00-5.3-2.1 7.67 7.67 0 00-7.79 7.5 7.431 7.431 0 001.09 4.1 7.631 7.631 0 006.8 3.7 6.75 6.75 0 007.1-7 7.076 7.076 0 00-.1-1.4zm5.81-1.5a4.9 4.9 0 00-5 4.8v.1a5 5 0 009.9.1 4.86 4.86 0 00-4.9-5zm0 2a2.84 2.84 0 012.8 3 2.807 2.807 0 01-2.805 2.977A2.81 2.81 0 0162.57 11.8a2.802 2.802 0 012.81-3zm10.69-2a4.83 4.83 0 00-4.9 4.8v.1a5 5 0 009.9.1 4.927 4.927 0 00-5-5zm0 2a2.85 2.85 0 012.81 3 2.807 2.807 0 01-2.805 2.977A2.81 2.81 0 0173.27 11.8a2.79 2.79 0 012.8-3zm10.5-2a4.9 4.9 0 000 9.8 3.261 3.261 0 002.7-1.2v1a2.45 2.45 0 01-2.6 2.7 2.67 2.67 0 01-2.5-1.8l-1.9.8a4.76 4.76 0 004.4 2.9c2.6 0 4.6-1.7 4.6-5.1V7.1h-2.1v.8a3.31 3.31 0 00-2.6-1.1zm.21 1.9a2.76 2.76 0 012.6 3 2.708 2.708 0 01-2.7 2.942 2.71 2.71 0 01-2.7-2.942 3 3 0 012.8-3zm13.69-1.9a4.66 4.66 0 00-4.59 4.9 4.78 4.78 0 004.59 4.9h.2a4.783 4.783 0 004.1-2.2l-1.7-1.1a2.71 2.71 0 01-4.8-.1l6.6-2.7-.3-.8a4.32 4.32 0 00-4.1-2.9zm.1 1.9a1.903 1.903 0 011.81 1.1L98 11.7a2.81 2.81 0 012.6-3h-.03zm-7.79 7.7H95V1.9h-2.22v14.5z"/> <path fill="#757575" d="M48.77 25.152v8.85h-1.29v-8.8h-2.8v-1.28h6.89v1.23h-2.8zM54.24 26.861a3.21 3.21 0 012.26.77 2.7 2.7 0 01.82 2.09v4.28h-1.23v-1H56a2.45 2.45 0 01-2.12 1.17 2.76 2.76 0 01-1.88-.67 2.15 2.15 0 01-.77-1.68 2 2 0 01.81-1.69 3.39 3.39 0 012.15-.63 3.78 3.78 0 011.89.42v-.3a1.471 1.471 0 00-.53-1.14 1.85 1.85 0 00-1.24-.48 2.001 2.001 0 00-1.71.91l-1.13-.71a3.15 3.15 0 012.77-1.34zm-1.66 5a1 1 0 00.42.84c.29.208.643.314 1 .3a2 2 0 001.45-.61 1.86 1.86 0 00.64-1.39 2.62 2.62 0 00-1.68-.48 2.21 2.21 0 00-1.31.38 1.12 1.12 0 00-.52.95v.01zM61.75 37.2a3.805 3.805 0 01-1.08-.15 3.186 3.186 0 01-.92-.41 2.999 2.999 0 01-.7-.63 2.48 2.48 0 01-.45-.81l1.22-.51c.126.405.383.756.73 1 .351.245.772.371 1.2.36a2.13 2.13 0 001.64-.63 2.45 2.45 0 00.61-1.76V33h-.06a2.52 2.52 0 01-1 .86c-.418.21-.881.316-1.35.31a3 3 0 01-2.31-1.05 4 4 0 010-5.19 3 3 0 012.31-1.07c.47-.004.932.106 1.35.32.404.192.75.49 1 .86H64v-.95h1.23v6.57a3.49 3.49 0 01-.94 2.59 3.41 3.41 0 01-2.54.95zm0-4.2a1.999 1.999 0 001.56-.69 2.54 2.54 0 00.69-1.79 2.57 2.57 0 00-.63-1.78 2.082 2.082 0 00-3.11 0 2.57 2.57 0 00-.63 1.78 2.53 2.53 0 00.63 1.78 2 2 0 001.53.7h-.04zM71.37 34h-1.29V24h1.29l3.49 6h.06l3.48-6.11h1.29V34H78.4v-6l.06-1.68h-.06l-3.13 5.5h-.76l-3.14-5.5h-.05l.05 1.68v6zM83.88 26.86a3.21 3.21 0 012.26.77 2.702 2.702 0 01.86 2.09V34h-1.27v-1h-.06a2.46 2.46 0 01-2.13 1.17 2.75 2.75 0 01-1.89-.67 2.15 2.15 0 01-.77-1.68 2 2 0 01.81-1.69 3.39 3.39 0 012.15-.63 3.832 3.832 0 011.89.42v-.3a1.471 1.471 0 00-.53-1.14A1.85 1.85 0 0084 28a2 2 0 00-1.71.91l-1.13-.71a3.15 3.15 0 012.72-1.34zm-1.67 5a1 1 0 00.43.84 1.61 1.61 0 001 .34 2.002 2.002 0 001.45-.61 1.86 1.86 0 00.64-1.43 2.62 2.62 0 00-1.68-.48 2.21 2.21 0 00-1.31.38 1.13 1.13 0 00-.53.95v.01zM88.42 27.092h1.23v.91h.06c.212-.362.525-.654.9-.84a2.66 2.66 0 011.27-.34 2.542 2.542 0 011.94.72 2.92 2.92 0 01.68 2.06v4.4h-1.29v-4.28a1.52 1.52 0 00-1.7-1.72 1.6 1.6 0 00-1.28.62 2.26 2.26 0 00-.52 1.49v3.89h-1.29v-6.91zM98.55 26.861a3.18 3.18 0 012.25.77 2.69 2.69 0 01.83 2.09v4.28h-1.24v-1a2.457 2.457 0 01-2.13 1.17 2.76 2.76 0 01-1.9-.67 2.139 2.139 0 01-.76-1.68 2 2 0 01.8-1.69 3.41 3.41 0 012.15-.63 3.78 3.78 0 011.89.42v-.3a1.432 1.432 0 00-.53-1.14 1.83 1.83 0 00-1.29-.48 2 2 0 00-1.71.91l-1.14-.71a3.17 3.17 0 012.78-1.34zm-1.67 5a1 1 0 00.42.84 1.62 1.62 0 001 .34 2 2 0 001.45-.61 1.896 1.896 0 00.63-1.41 2.62 2.62 0 00-1.68-.48 2.21 2.21 0 00-1.31.38 1.12 1.12 0 00-.51.93v.01zM106.05 37.2a3.814 3.814 0 01-1.08-.15 3.142 3.142 0 01-.91-.41 3.084 3.084 0 01-.71-.63 2.496 2.496 0 01-.45-.81l1.22-.51c.126.405.383.756.73 1a2 2 0 001.2.36 2.13 2.13 0 001.64-.63 2.447 2.447 0 00.59-1.73V33h-.06c-.252.369-.597.665-1 .86a2.998 2.998 0 01-1.35.31 3.06 3.06 0 01-2.32-1.05 4 4 0 010-5.19 3 3 0 012.32-1.07 3 3 0 011.35.32c.403.194.748.49 1 .86h.06v-.95h1.23v6.57a3.488 3.488 0 01-.94 2.59 3.408 3.408 0 01-2.52.95zm0-4.2a1.996 1.996 0 001.56-.69 2.538 2.538 0 00.63-1.79 2.569 2.569 0 00-.63-1.78 1.994 1.994 0 00-1.56-.7 2 2 0 00-1.55.7 2.573 2.573 0 00-.63 1.78c-.027.652.198 1.29.63 1.78a2.009 2.009 0 001.59.7h-.04zM114.09 34.169a3.268 3.268 0 01-2.49-1 3.683 3.683 0 01-1-2.62 3.73 3.73 0 01.95-2.61 3.168 3.168 0 012.45-1.08 3.122 3.122 0 012.43 1c.638.777.959 1.766.9 2.77v.14h-5.39a2.194 2.194 0 002.561 2.204 2.173 2.173 0 001.579-1.194l1.15.56a3.487 3.487 0 01-1.23 1.35c-.58.333-1.241.499-1.91.48zm-2.09-4.48h4a1.718 1.718 0 00-.58-1.18 2.05 2.05 0 00-1.42-.51 1.86 1.86 0 00-1.25.45 2.2 2.2 0 00-.75 1.24zM119.81 33.999h-1.29v-6.91h1.23v1.12h.06a1.82 1.82 0 01.8-.93 2.24 2.24 0 011.2-.39 2.44 2.44 0 011 .17l-.39 1.25a1.998 1.998 0 00-.76-.1 1.644 1.644 0 00-1.26.59 1.997 1.997 0 00-.54 1.37l-.05 3.83z"/> <path fill="#8AB4F8" d="M22.07 36.169l-6.53-6.53 14-14.17 6.65 6.65-14.12 14.05z"/> <path fill="#4285F4" d="M22.12 7.999l-6.65-6.62-14.09 14.09a4.69 4.69 0 000 6.64l14.09 14.1 6.53-6.55-10.66-10.86 10.78-10.8z"/> <path fill="#8AB4F8" d="M36.21 15.47L22.12 1.38A4.7 4.7 0 0015.47 8l14.1 14.1a4.702 4.702 0 106.65-6.65l-.01.02z"/> <path fill="#246FDB" d="M18.74 37.601a4.66 4.66 0 100-9.32 4.66 4.66 0 000 9.32z"/> </g> <defs> <clipPath id="clip0"> <path fill="#fff" d="M0 0h122.76v37.6H0z"/> </clipPath> </defs> </svg>`,
    path: '/docs/google-tag-manager-installation',
  }),
  new MediaItem({
    key: 'clickFunnels',
    icon: `<svg class="h-10" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 368.51 71.32"> <path d="M80.63 0H4.89A4.89 4.89 0 000 4.89v61.55a4.89 4.89 0 004.89 4.89h75.74a4.89 4.89 0 004.89-4.89V4.89A4.89 4.89 0 0080.63 0zM76 4.7a3.12 3.12 0 11-3.12 3.12A3.12 3.12 0 0176 4.7zm-11.13 0a3.12 3.12 0 11-3.12 3.12 3.12 3.12 0 013.12-3.12zm-11.13 0a3.12 3.12 0 11-3.12 3.12 3.12 3.12 0 013.12-3.12zM79.24 65h-73V15.15h73v49.82z" class="a"/> <path d="M29 59.7a4.45 4.45 0 015-2 3.39 3.39 0 004.35-2.2 3.6 3.6 0 00-2-4.6 4.2 4.2 0 01-2.4-4.56 4.13 4.13 0 013.99-3.5 3.63 3.63 0 00-.12-7.26 4.3 4.3 0 01-4.1-3.46 4.73 4.73 0 010-1.75 4.07 4.07 0 012.2-3 3.63 3.63 0 001.55-4.79 3.56 3.56 0 00-4.53-2.21 4.1 4.1 0 01-2.83.08 5.09 5.09 0 01-2-1.61 4.65 4.65 0 01-.84-3.7H10.61c7.73 4.15 15.09 12.49 15.27 23.56.2 12-6.15 21.69-15.49 26.29h18.82A4.25 4.25 0 0129 59.7z" class="b"/> <path fill="#e74f38" d="M52.53 19.95a4.24 4.24 0 011.06 5.25 4.45 4.45 0 01-4.65 2.73 3.39 3.39 0 00-4 2.84 3.6 3.6 0 002.63 4.25 4.19 4.19 0 013.07 4.14 4.13 4.13 0 01-3.32 4 3.632 3.632 0 001.23 7.16 4.3 4.3 0 014.58 2.79 4.75 4.75 0 01.26 1.73 4.08 4.08 0 01-1.71 3.33 3.63 3.63 0 00-.8 5 5.09 5.09 0 002 1.8h22.26c-7.08-3.51-13.66-10.11-15.27-19.5-2.28-13.31 4-24.91 14.55-30.33H51.73a3.54 3.54 0 00.8 4.81z"/> <path d="M58.94 65.7h-.65a1.15 1.15 0 00.65 0z" class="b"/> <path d="M102.21 35.13a13.26 13.26 0 01.6-4 8.9 8.9 0 011.88-3.31 8.77 8.77 0 013.25-2.26 12.81 12.81 0 014.81-.81 12.24 12.24 0 014.19.64 7.71 7.71 0 012.86 1.74 2.28 2.28 0 01.67 1.55 2.74 2.74 0 01-.4 1.64 2.4 2.4 0 01-1.38 1 3.09 3.09 0 01-2.26-.38 12.08 12.08 0 00-1.57-.69 6.2 6.2 0 00-2.1-.31 4.58 4.58 0 00-3.72 1.48 6 6 0 00-1.24 4v3.43a6 6 0 001.24 4 4.57 4.57 0 003.72 1.48 8.82 8.82 0 002.24-.26 6.37 6.37 0 001.57-.6 2.86 2.86 0 011.88-.36 2.41 2.41 0 011.41.76 2.55 2.55 0 01.62 1.43 2.4 2.4 0 01-.54 1.7 5.56 5.56 0 01-2.88 1.88 13.88 13.88 0 01-4.31.64 12.82 12.82 0 01-4.81-.82 8.76 8.76 0 01-3.29-2.24 8.89 8.89 0 01-1.88-3.31 13.25 13.25 0 01-.6-4v-4zM125.8 18.89a2 2 0 01.79-1.55 3 3 0 012.07-.69 2.78 2.78 0 012 .69 2.08 2.08 0 01.74 1.55v28.4a2.08 2.08 0 01-.74 1.55 2.78 2.78 0 01-2 .69 3 3 0 01-2.07-.69 2 2 0 01-.79-1.55v-28.4zM138.99 18.44a2 2 0 01.79-1.54 3 3 0 012.07-.69 2.79 2.79 0 012 .69 2.06 2.06 0 01.74 1.54v1.13a2.06 2.06 0 01-.74 1.54 2.79 2.79 0 01-2 .68 3 3 0 01-2.07-.68 2 2 0 01-.79-1.54v-1.13zm0 8.55a2 2 0 01.79-1.55 3 3 0 012.07-.69 2.78 2.78 0 012 .69 2.08 2.08 0 01.74 1.55v20.3a2.08 2.08 0 01-.74 1.55 2.78 2.78 0 01-2 .69 3 3 0 01-2.07-.69 2 2 0 01-.79-1.55v-20.3zM151 35.13a13.26 13.26 0 01.6-4 8.9 8.9 0 011.88-3.31 8.76 8.76 0 013.29-2.24 12.81 12.81 0 014.81-.81 12.24 12.24 0 014.19.64 7.71 7.71 0 012.86 1.74 2.28 2.28 0 01.67 1.55 2.74 2.74 0 01-.4 1.64 2.4 2.4 0 01-1.38 1 3.09 3.09 0 01-2.26-.38 12.11 12.11 0 00-1.57-.69 6.21 6.21 0 00-2.1-.31 4.58 4.58 0 00-3.72 1.48 6 6 0 00-1.24 4v3.43a6 6 0 001.24 4 4.57 4.57 0 003.72 1.48 8.82 8.82 0 002.24-.26 6.39 6.39 0 001.57-.6 2.86 2.86 0 011.88-.36 2.41 2.41 0 011.41.76 2.55 2.55 0 01.62 1.43 2.4 2.4 0 01-.52 1.69 5.56 5.56 0 01-2.85 1.87 13.88 13.88 0 01-4.31.64 12.82 12.82 0 01-4.81-.81 8.75 8.75 0 01-3.29-2.24 8.89 8.89 0 01-1.88-3.31 13.26 13.26 0 01-.6-4v-4zM174.94 18.89a2 2 0 01.79-1.55 3 3 0 012.07-.69 2.78 2.78 0 012 .69 2.08 2.08 0 01.74 1.55V33.8h2.19l6.91-8.39a2.65 2.65 0 011.81-.64 2.81 2.81 0 012 .83 2.36 2.36 0 01.64 2 2.63 2.63 0 01-.83 1.67l-6.15 6.77 7.58 9.48a1.89 1.89 0 01.38 1.67 2.83 2.83 0 01-1.29 1.76 2.71 2.71 0 01-2 .52 2.27 2.27 0 01-1.55-.81l-7.59-9.96h-2.1v8.58a2.08 2.08 0 01-.74 1.55 2.78 2.78 0 01-2 .69 3 3 0 01-2.07-.69 2 2 0 01-.79-1.55V18.89zM213.51 25.22v-1.43q0-4.48 2.17-6.29t6.36-1.8a10.24 10.24 0 012 .19 5.94 5.94 0 011.69.6 3.6 3.6 0 011.17 1 2.09 2.09 0 01.43 1.29 2.42 2.42 0 01-.55 1.67 2.08 2.08 0 01-1.64.62 5.12 5.12 0 01-1.45-.19 5.93 5.93 0 00-1.64-.19 3.25 3.25 0 00-2.33.67 3.17 3.17 0 00-.67 2.29v1.62h3.67a2 2 0 011.43.6 2.05 2.05 0 01.62 1.55 2.18 2.18 0 01-.62 1.64 2 2 0 01-1.43.6h-3.68v17.62a2.08 2.08 0 01-.74 1.55 2.71 2.71 0 01-1.93.69 3 3 0 01-2.1-.69 2.05 2.05 0 01-.76-1.55V29.61h-1.67a2 2 0 01-1.43-.6 2.18 2.18 0 01-.62-1.64 2.05 2.05 0 01.62-1.55 2 2 0 011.43-.6h1.67zM240.05 49.52a12.38 12.38 0 01-4.6-.79 8.89 8.89 0 01-5.31-5.53 12.66 12.66 0 01-.67-4.17v-12a2 2 0 01.79-1.55 3 3 0 012.07-.69 2.78 2.78 0 012 .69 2.08 2.08 0 01.74 1.55v12.2a4.84 4.84 0 001.43 3.86 5.26 5.26 0 003.57 1.24 5.17 5.17 0 003.55-1.24 4.89 4.89 0 001.41-3.86v-12.2a2.06 2.06 0 01.76-1.55 2.87 2.87 0 012-.69 3 3 0 012 .69 2 2 0 01.81 1.55v12a12.69 12.69 0 01-.67 4.17 8.89 8.89 0 01-5.28 5.5 12.39 12.39 0 01-4.6.82zM267.88 24.7a13.51 13.51 0 014.6.71 9 9 0 013.31 2 8.32 8.32 0 012 3.17 12.06 12.06 0 01.67 4.1v12.6a2 2 0 01-.81 1.55 3 3 0 01-2 .69 2.87 2.87 0 01-2-.69 2.06 2.06 0 01-.76-1.55V35.04a4.89 4.89 0 00-1.41-3.86 5.17 5.17 0 00-3.55-1.24 5.26 5.26 0 00-3.57 1.24 4.84 4.84 0 00-1.43 3.86v12.24a2.08 2.08 0 01-.74 1.55 2.78 2.78 0 01-2 .69 3 3 0 01-2.07-.69 2 2 0 01-.79-1.55V35.23a12.66 12.66 0 01.67-4.17 8.9 8.9 0 015.31-5.53 12.39 12.39 0 014.57-.83zM295.7 24.7a13.5 13.5 0 014.6.71 9 9 0 013.31 2 8.3 8.3 0 012 3.17 12 12 0 01.67 4.1v12.6a2 2 0 01-.81 1.55 3.05 3.05 0 01-2 .69 2.87 2.87 0 01-2-.69 2.05 2.05 0 01-.76-1.55V35.04a4.89 4.89 0 00-1.41-3.86 5.17 5.17 0 00-3.55-1.24 5.26 5.26 0 00-3.57 1.24 4.84 4.84 0 00-1.43 3.86v12.24a2.08 2.08 0 01-.74 1.55 2.78 2.78 0 01-2 .69 3 3 0 01-2.07-.69 2 2 0 01-.79-1.55V35.23a12.66 12.66 0 01.67-4.17 8.9 8.9 0 015.31-5.53 12.39 12.39 0 014.57-.83zM332.68 34.42a4.54 4.54 0 01-.86 3.15 3.81 3.81 0 01-2.86.9h-11.29v1.23a4.92 4.92 0 001.41 3.67 5.43 5.43 0 004 1.38 15.32 15.32 0 002.36-.17 6.54 6.54 0 001.69-.45l1.29-.67a2.44 2.44 0 011.14-.33 1.9 1.9 0 011.48.57 2.29 2.29 0 01.57 1.48 2.33 2.33 0 01-.67 1.52 6.46 6.46 0 01-1.81 1.38 12 12 0 01-2.69 1 13.07 13.07 0 01-3.31.4 12.37 12.37 0 01-4.43-.76 11.11 11.11 0 01-3.45-2 9.15 9.15 0 01-2.31-3.02 8.62 8.62 0 01-.81-3.69v-5.07a10.14 10.14 0 013-7.19 10.29 10.29 0 013.29-2.19 10.61 10.61 0 014.19-.81 11 11 0 014.62.88 9 9 0 014.91 5.41 10.66 10.66 0 01.54 3.38zm-10.15-5.05a4.43 4.43 0 00-3.41 1.33 5.83 5.83 0 00-1.45 3.6h9.72v-.69a3.83 3.83 0 00-1.36-3.1 5.25 5.25 0 00-3.5-1.14zM339.01 18.89a2 2 0 01.79-1.55 3 3 0 012.07-.69 2.78 2.78 0 012 .69 2.08 2.08 0 01.74 1.55v28.4a2.08 2.08 0 01-.74 1.55 2.78 2.78 0 01-2 .69 3 3 0 01-2.07-.69 2 2 0 01-.79-1.55v-28.4zM351.02 46.09a2.13 2.13 0 01-.19-.71 2.24 2.24 0 01.07-.86 2.67 2.67 0 01.38-.81 1.57 1.57 0 01.74-.57 3 3 0 011.17-.29 2.19 2.19 0 01.88.17 3.32 3.32 0 01.71.43l.67.5a7.19 7.19 0 004.34 1 4.11 4.11 0 002.45-.67 2.16 2.16 0 00.07-3.5 7.89 7.89 0 00-2.17-1.08q-1.31-.45-2.81-.91a12.35 12.35 0 01-2.81-1.24 7 7 0 01-2.17-2 5.77 5.77 0 01-.86-3.34 6.89 6.89 0 012.33-5.41 9.35 9.35 0 016.44-2.1 14 14 0 013.15.31 10.64 10.64 0 012.22.74 5.1 5.1 0 011.38.93 3.26 3.26 0 01.69.93 2.89 2.89 0 01.24 1.59 1.75 1.75 0 01-.74 1.36 2.18 2.18 0 01-1.76.4 6.07 6.07 0 01-1.48-.6q-.38-.19-1.21-.55a6.66 6.66 0 00-2.5-.36 3.26 3.26 0 00-2.31.74 2.42 2.42 0 00-.79 1.83 1.56 1.56 0 00.83 1.41 10 10 0 002.1.91l2.74.86a10.62 10.62 0 012.74 1.29 6.43 6.43 0 012.93 5.79 6.54 6.54 0 01-2.33 5.29 9.64 9.64 0 01-6.38 2 15.77 15.77 0 01-2.88-.26 13.19 13.19 0 01-2.57-.74 9.69 9.69 0 01-2-1.1 4.61 4.61 0 01-1.31-1.38z" class="a"/> </svg>`,
    path: '/docs/clickfunnels-installation',
  }),
]

export type KeysUtility<T extends MediaItem<string>[]> = {
  [K in keyof T]: T[K] extends MediaItem<infer T> ? T : never
}[number]

export type integrationKeys = KeysUtility<typeof integrationsList>
export type socialKeys = KeysUtility<typeof socialList>
