import Cookies, { CookieAttributes } from "js-cookie"

/**
 * Gets the domain name without any sub domains.
 * Needed for cookies, this is not made available any easier way.
 * It works by setting cookies until they can't be set
 * https://developers.livechat.com/updates/setting-cookies-to-subdomains-in-javascript
 * http://jsfiddle.net/zEwsP/4/
 */
export const getCookieDomain = (): string | undefined => {
  if (typeof window == "undefined") return undefined

  let i = 0
  let domain = window.location.hostname

  //if (domain.includes("localhost")) return undefined

  const p = domain.split(".")
  const s = `_gd${Date.now()}`
  while (i < p.length - 1 && !document.cookie.includes(`${s}=${s}`)) {
    domain = p.slice(-1 - ++i).join(".")
    // eslint-disable-next-line unicorn/no-document-cookie
    document.cookie = `${s}=${s};domain=${domain};`
  }
  // remove cookie
  // eslint-disable-next-line unicorn/no-document-cookie
  document.cookie = `${s}=;expires=Thu, 01 Jan 1970 00:00:01 GMT;domain=${domain};`

  return domain
}
/**
 * Simple set browser cookie
 * https://stackoverflow.com/a/23086139/1858322
 */
export const setCookie = (
  name: string,
  value: string,
  attributes: CookieAttributes,
): string | undefined => {
  return Cookies.set(name, value, attributes)
}
export const setCookieWithDomain = (
  name: string,
  value: string,
  attributes: CookieAttributes = {},
): string | undefined => {
  const domain = getCookieDomain()

  attributes.domain = domain

  return setCookie(name, value, attributes)
}
/**
 * Get a browser cookie by name
 */
export const getCookie = (name: string): string | undefined => {
  return Cookies.get(name)
}
export const removeCookie = (
  name: string,
  attributes: CookieAttributes,
): void => {
  return Cookies.remove(name, attributes)
}

export const removeCookieWithDomain = (
  name: string,
  attributes: CookieAttributes = {},
): void => {
  const domain = getCookieDomain()
  return Cookies.remove(name, { domain, ...attributes })
}
