/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.core.data.callableindex;

import static eu.fasten.core.data.callableindex.GraphMetadata.ReceiverRecord.CallType.DYNAMIC;
import static eu.fasten.core.data.callableindex.GraphMetadata.ReceiverRecord.CallType.STATIC;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.HashSet;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.json.JSONObject;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.rocksdb.RocksDBException;

import eu.fasten.core.data.callableindex.GraphMetadata.ReceiverRecord;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongList;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;

public class RocksDaoTest {

    private File rocksDaoDir;
    private RocksDao rocksDao;

    @BeforeEach
    public void setUp() throws RocksDBException, IOException {
        rocksDaoDir = Files.createTempDirectory(RocksDaoTest.class.getSimpleName()).toFile();
        rocksDao = new RocksDao(rocksDaoDir.toString(), false);
    }

    @AfterEach
    public void tearDown() throws IOException {
        rocksDao.close();
        FileUtils.deleteDirectory(rocksDaoDir);
    }

    @Test
    public void extendedGidGraphTest() throws IOException, RocksDBException {
        final var json = new JSONObject("{\n" +
                "        \"index\": 1,\n" +
                "        \"product\": \"test\",\n" +
                "           \"version\": \"0.0.1\",\n" +
                "           \"nodes\":[0, 1, 2],\n" +
                "           \"numInternalNodes\": 2,\n" +
                "           \"edges\": [\n" +
                "              [0, 1],\n" +
                "              [0, 2],\n" +
                "              [1, 2]\n" +
                "           ],\n" +
                "           \"callsites_info\": {\n" +
                "                \"[0, 1]\": {\n" +
                "                    \"line\": 5,\n" +
                "                    \"call_type\": \"static\",\n" +
                "                    \"receiver_type_ids\": [1,2]\n" +
                "                 },\n" +
                "              \"[0, 2]\": {\n" +
                "                    \"line\": 12,\n" +
                "                    \"call_type\": \"static\",\n" +
                "                    \"receiver_type_ids\": [1,2]\n" +
                "                },\n" +
                "              \"[1, 2]\": {\n" +
                "                    \"line\": 25,\n" +
                "                    \"call_type\": \"dynamic\",\n" +
                "                    \"receiver_type_ids\": [3]\n" +
                "                } \n" +
                "           },\n" +
                "           \"gid_to_uri\": {\n" +
                "                   \"0\": \"/java.lang/String.get()long\",\n" +
                "                   \"1\": \"/java.lang/Object.hashCode()int\",\n" +
                "                   \"2\": \"/my.package/Klass.method(int)int\"\n" +
                "           },\n" +
                "           \"types_map\": {\n" +
                "                \"1\": \"/java.lang/String\",\n" +
                "                \"2\": \"/product/Interface\",\n" +
                "                \"3\": \"/java.lang/Object\"\n" +
                "           }\n" +
                "}");
        final var graph = ExtendedGidGraph.getGraph(json);
        rocksDao.saveToRocksDb(graph);
        final var graphData = rocksDao.getGraphData(graph.getIndex());
        assertEquals(graph.getNumInternalNodes(), graphData.nodes().size() - graphData.externalNodes().size());
        assertEquals(graph.getNodes().size(), graphData.nodes().size());
        assertEquals(new LongOpenHashSet(graph.getNodes()), graphData.nodes());
        assertEquals(LongList.of(1L, 2L), graphData.successors(0L));
        assertEquals(new LongArrayList(List.of(2L)), graphData.successors(1L));
        assertEquals(new LongArrayList(List.of(0L)), graphData.predecessors(1L));
        assertEquals(LongList.of(0L, 1L), graphData.predecessors(2L));
        assertEquals(graph.getEdges().size(), graphData.numArcs());
        assertEquals(new LongOpenHashSet(List.of(2L)), graphData.externalNodes());
        assertEquals(new HashSet<>(graph.getNodes()), graph.getGidToUriMap().keySet());

        final GraphMetadata graphMetadata = rocksDao.getGraphMetadata(graph.getIndex(), graphData);
		assertEquals(
                new GraphMetadata.NodeMetadata("/java.lang/String", "get()/java.lang/long",
						List.of(new ReceiverRecord(5, STATIC, "put()/java.lang/long", List.of("/java.lang/String", "/product/Interface")), new ReceiverRecord(12, STATIC, "put()/java.lang/long", List.of("/java.lang/String", "/product/Interface")))),
                graphMetadata.gid2NodeMetadata.get(0));
        assertEquals(
                new GraphMetadata.NodeMetadata("/java.lang/Object", "hashCode()/java.lang/int",
						List.of(new ReceiverRecord(25, DYNAMIC, "equals()/java.lang/long", List.of("/java.lang/Object")))),
                graphMetadata.gid2NodeMetadata.get(1));
        assertEquals(
                new GraphMetadata.NodeMetadata("/my.package/Klass", "method(/my.package/int)/my.package/int",
                        List.of()),
				graphMetadata.gid2NodeMetadata.get(2));
    }

    @Test
    public void databaseTest1() throws IOException, RocksDBException {
        final var json = new JSONObject("{" +
                "\"index\": 1," +
                "\"product\": \"test\"," +
                "\"version\": \"0.0.1\"," +
                "\"nodes\": [0, 1, 2]," +
                "\"numInternalNodes\": 2," +
                "\"edges\": [[0, 1], [1, 2]]" +
                "}");
        final var graph = GidGraph.getGraph(json);
        rocksDao.saveToRocksDb(graph.getIndex(), graph.getNodes(), graph.getNumInternalNodes(), graph.getEdges());
        final var graphData = rocksDao.getGraphData(graph.getIndex());
        assertEquals(graph.getNumInternalNodes(), graphData.nodes().size() - graphData.externalNodes().size());
        assertEquals(graph.getNodes().size(), graphData.nodes().size());
        assertEquals(new LongOpenHashSet(graph.getNodes()), graphData.nodes());
        assertEquals(new LongArrayList(List.of(1L)), graphData.successors(0L));
        assertEquals(new LongArrayList(List.of(2L)), graphData.successors(1L));
        assertEquals(new LongArrayList(List.of(0L)), graphData.predecessors(1L));
        assertEquals(new LongArrayList(List.of(1L)), graphData.predecessors(2L));
        assertEquals(graph.getEdges().size(), graphData.numArcs());
        assertEquals(new LongOpenHashSet(List.of(2L)), graphData.externalNodes());
    }

    @Test
    public void databaseTest2() throws IOException, RocksDBException {
        final var json = new JSONObject("{" +
                "\"index\": 2," +
                "\"product\": \"test\"," +
                "\"version\": \"0.0.1\"," +
                "\"nodes\": [255, 256, 257, 258]," +
                "\"numInternalNodes\": 3," +
                "\"edges\": [[255, 256], [255, 258], [256, 257], [257, 258]]" +
                "}");
        final var graph = GidGraph.getGraph(json);
        rocksDao.saveToRocksDb(graph.getIndex(), graph.getNodes(), graph.getNumInternalNodes(), graph.getEdges());
        final var graphData = rocksDao.getGraphData(graph.getIndex());
        assertEquals(graph.getNumInternalNodes(), graphData.nodes().size() - graphData.externalNodes().size());
        assertEquals(graph.getNodes().size(), graphData.nodes().size());
        assertEquals(new LongOpenHashSet(graph.getNodes()), graphData.nodes());
        assertEquals(new LongOpenHashSet(List.of(256L, 258L)), new LongOpenHashSet(graphData.successors(255L)));
        assertEquals(new LongArrayList(List.of(257L)), graphData.successors(256L));
        assertEquals(new LongArrayList(List.of(258L)), graphData.successors(257L));
        assertEquals(new LongArrayList(), graphData.predecessors(255L));
        assertEquals(new LongArrayList(List.of(255L)), graphData.predecessors(256L));
        assertEquals(new LongArrayList(List.of(256L)), graphData.predecessors(257L));
        assertEquals(new LongOpenHashSet(List.of(255L, 257L)), new LongOpenHashSet(graphData.predecessors(258L)));
        assertEquals(graph.getEdges().size(), graphData.numArcs());
        assertEquals(new LongOpenHashSet(List.of(258L)), graphData.externalNodes());
    }

    @Test
    public void databaseTest3() throws IOException, RocksDBException {
        final var json = new JSONObject("{" +
                "\"index\": 3," +
                "\"product\": \"test\"," +
                "\"version\": \"0.0.1\"," +
                "\"nodes\": [255, 256, 257, 258]," +
                "\"numInternalNodes\": 4," +
                "\"edges\": [[255, 256], [255, 258], [256, 257], [257, 258]]" +
                "}");
        final var graph = GidGraph.getGraph(json);
        rocksDao.saveToRocksDb(graph.getIndex(), graph.getNodes(), graph.getNumInternalNodes(), graph.getEdges());
        final var graphData = rocksDao.getGraphData(graph.getIndex());
        assertEquals(graph.getNumInternalNodes(), graphData.nodes().size() - graphData.externalNodes().size());
        assertEquals(graph.getNodes().size(), graphData.nodes().size());
        assertEquals(new LongOpenHashSet((graph.getNodes())), new LongOpenHashSet(graphData.nodes()));
        assertEquals(new LongOpenHashSet(List.of(256L, 258L)), new LongOpenHashSet(graphData.successors(255L)));
        assertEquals(new LongArrayList(List.of(257L)), graphData.successors(256L));
        assertEquals(new LongArrayList(List.of(258L)), graphData.successors(257L));
        assertEquals(new LongArrayList(List.of()), graphData.predecessors(255L));
        assertEquals(new LongArrayList(List.of(255L)), graphData.predecessors(256L));
        assertEquals(new LongArrayList(List.of(256L)), graphData.predecessors(257L));
        assertEquals(new LongOpenHashSet(List.of(255L, 257L)), new LongOpenHashSet(graphData.predecessors(258L)));
        assertEquals(graph.getEdges().size(), graphData.numArcs());
        assertEquals(new LongOpenHashSet(), graphData.externalNodes());
    }

    @Test
    public void databaseTest4() throws IOException, RocksDBException {
        final var json = new JSONObject("{" +
                "\"index\": 4," +
                "\"product\": \"test\"," +
                "\"version\": \"0.0.1\"," +
                "\"nodes\": [255, 256, 257, 258]," +
                "\"numInternalNodes\": 0," +
                "\"edges\": [[255, 256], [255, 258], [256, 257], [257, 258]]" +
                "}");
        final var graph = GidGraph.getGraph(json);
        rocksDao.saveToRocksDb(graph.getIndex(), graph.getNodes(), graph.getNumInternalNodes(), graph.getEdges());
        final var graphData = rocksDao.getGraphData(graph.getIndex());
        assertEquals(graph.getNumInternalNodes(), graphData.nodes().size() - graphData.externalNodes().size());
        assertEquals(graph.getNodes().size(), graphData.nodes().size());
        assertEquals(new LongOpenHashSet((graph.getNodes())), new LongOpenHashSet(graphData.nodes()));
        assertEquals(new LongOpenHashSet(List.of(256L, 258L)), new LongOpenHashSet(graphData.successors(255L)));
        assertEquals(new LongArrayList(List.of(257L)), graphData.successors(256L));
        assertEquals(new LongArrayList(List.of(258L)), graphData.successors(257L));
        assertEquals(new LongArrayList(List.of()), graphData.predecessors(255L));
        assertEquals(new LongArrayList(List.of(255L)), graphData.predecessors(256L));
        assertEquals(new LongArrayList(List.of(256L)), graphData.predecessors(257L));
        assertEquals(new LongOpenHashSet(List.of(255L, 257L)), new LongOpenHashSet(graphData.predecessors(258L)));
        assertEquals(graph.getEdges().size(), graphData.numArcs());
        assertEquals(new LongOpenHashSet(graph.getNodes()), graphData.externalNodes());
    }

    @Test
    public void databaseTest5() throws IOException, RocksDBException {
        final var json = new JSONObject("{" +
                "\"index\": 2," +
                "\"product\": \"test\"," +
                "\"version\": \"0.0.1\"," +
                "\"nodes\": [9223372036854775804, 9223372036854775805, 9223372036854775806, 9223372036854775807]," +
                "\"numInternalNodes\": 3," +
                "\"edges\": [[9223372036854775804, 9223372036854775805], [9223372036854775804, 9223372036854775807], [9223372036854775805, 9223372036854775806], [9223372036854775806, 9223372036854775807]]" +
                "}");
        final var graph = GidGraph.getGraph(json);
        rocksDao.saveToRocksDb(graph.getIndex(), graph.getNodes(), graph.getNumInternalNodes(), graph.getEdges());
        final var graphData = rocksDao.getGraphData(graph.getIndex());
        assertEquals(graph.getNumInternalNodes(), graphData.nodes().size() - graphData.externalNodes().size());
        assertEquals(graph.getNodes().size(), graphData.nodes().size());
        assertEquals(new LongOpenHashSet(graph.getNodes()), graphData.nodes());
        assertEquals(new LongOpenHashSet(List.of(9223372036854775807L, 9223372036854775805L)), new LongOpenHashSet(graphData.successors(9223372036854775804L)));
        assertEquals(new LongArrayList(List.of(9223372036854775806L)), graphData.successors(9223372036854775805L));
        assertEquals(new LongArrayList(List.of(9223372036854775807L)), graphData.successors(9223372036854775806L));
        assertEquals(new LongArrayList(), graphData.predecessors(9223372036854775804L));
        assertEquals(new LongArrayList(List.of(9223372036854775804L)), graphData.predecessors(9223372036854775805L));
        assertEquals(new LongArrayList(List.of(9223372036854775805L)), graphData.predecessors(9223372036854775806L));
        assertEquals(new LongOpenHashSet(List.of(9223372036854775804L, 9223372036854775806L)), new LongOpenHashSet(graphData.predecessors(9223372036854775807L)));
        assertEquals(graph.getEdges().size(), graphData.numArcs());
        assertEquals(new LongOpenHashSet(List.of(9223372036854775807L)), graphData.externalNodes());
    }

    @Test
    public void databaseBigTest() throws IOException, RocksDBException {
        final var json = new JSONObject("{\"product\":\"ai.h2o.h2o-bindings\",\"nodes\":[8613,8612,8611,8610,8609,8608,8607,8606,8605,8604,8603,8602,8601,8600,8599,8598,8597,8596,8595,8594,8593,8592,8591,8590,8589,8588,8587,8586,8585,8584,8583,8582,8581,8580,8579,8578,8577,8576,8575,8574,8573,8572,8571,8570,8569,8568,8567,8566,8565,8564,8563,8562,8561,8560,8559,8558,8557,8556,8555,8554,8553,8552,8551,8550,8549,8548,8547,8546,8545,8544,8543,8542,8541,8540,8539,8538,8537,8536,8535,8534,8533,8532,8531,8530,8529,8528,8527,8526,8525,8524,8523,8522,8521,8520,8519,8518,8517,8516,8515,8514,8513,8512,8511,8510,8509,8508,8507,8506,8505,8504,8503,8502,8501,8500,8499,8498,8497,8496,8495,8494,8493,8492,8491,8490,8489,8488,8487,8486,8485,8484,8483,8482,8481,8480,8479,8478,8477,8476,8475,8474,8473,8472,8471,8470,8469,8468,8467,8466,8465,8464,8463,8462,8461,8460,8459,8458,8457,8456,8455,8454,8453,8452,8451,8450,8449,8448,8447,8446,8445,8444,8443,8442,8441,8440,8439,8438,8437,8436,8435,8434,8433,8432,8431,8430,8429,8428,8427,8426,8425,8424,8423,8422,8421,8420,8419,8418,8417,8416,8415,8414,8413,8412,8411,8410,8409,8408,8407,8406,8405,8404,8403,8402,8401,8400,8399,8398,8397,8396,8395,8394,8393,8392,8391,8390,8389,8388,8387,8386,8385,8384,8383,8382,8381,8380,8379,8378,8377,8376,8375,8374,8373,8372,8371,8370,8369,8368,8367,8366,8365,8364,8363,8362,8361,8360,8359,8358,8357,8356,8355,8354,8353,8352,8351,8350,8349,8348,8347,8346,8345,8344,8343,8342,8341,8340,8339,8338,8337,8336,8335,8334,8333,8332,8331,8330,8329,8328,8327,8326,8325,8324,8323,8322,8321,8320,8319,8318,8317,8316,8315,8314,8313,8312,8311,8310,8309,8308,8307,8306,8305,8304,8303,8302,8301,8300,8299,8298,8297,8296,8295,8294,8293,8292,8291,8290,8289,8288,8287,8286,8285,8284,8283,8282,8281,8280,8279,8278,8277,8276,8275,8274,8273,8272,8271,8270,8269,8268,8267,8266,8265,8264,8263,8262,8261,8260,8259,8258,8257,8256,8255,8254,8253,8252,8251,8250,8249,8248,8247,8246,8245,8244,8243,8242,8241,8240,8239,8238,8237,8236,8235,8234,8233,8232,8231,8230,8229,8228,8227,8226,8225,8224,8223,8222,8221,8220,8219,8218,8217,8216,8215,8214,8213,8212,8211,8210,8209,8208,8207,8206,8205,8204,8203,8202,8201,8200,8199,8198,8197,8196,8195,8194,8193,8192,8191,8190,8189,8188,8187,8186,8185,8184,8183,8182,8181,8180,8179,8178,8177,8176,8175,8174,8173,8172,8171,8170,8169,8168,8167,8166,8165,8164,8163,8162,8161,8160,8159,8158,8157,8156,8155,8154,8153,8152,8151,8150,8149,8148,8147,8146,8145,8144,8143,8142,8141,8140,8139,8138,8137,8136,8135,8134,8133,8132,8131,8130,8129,8128,8127,8126,8125,8124,8123,8122,8121,8120,8119,8118,8117,8116,8115,8114,8113,8112,8111,8110,8109,8108,8107,8106,8105,8104,8103,8102,8101,8100,8099,8098,8097,8096,8095,8094,8093,8092,8091,8090,8089,8088,8087,8086,8085,8084,8083,8082,8081,8080,8079,8078,8077,8076,8075,8074,8073,8072,8071,8070,8069,8068,8067,8066,8065,8064,8063,8062,8061,8060,8059,8058,8057,8056,8055,8054,8053,8052,8051,8050,8049,8048,8047,8046,8045,8044,8043,8042,8041,8040,8039,8038,8037,8036,8035,8034,8033,8032,8031,8030,8029,8028,8027,8026,8025,8024,8023,8022,8021,8020,8019,8018,8017,8016,8015,8014,8013,8012,8011,8010,8009,8008,8007,8006,8005,8004,8003,8002,8001,8000,7999,7998,7997,7996,7995,7994,7993,7992,7991,7990,7989,7988,7987,7986,7985,7984,7983,7982,7981,7980,7979,7978,7977,7976,7975,7974,7973,7972,7971,7970,7969,7968,7967,7966,7965,7964,7963,7962,7961,7960,7959,7958,7957,7956,7955,7954,7953,7952,7951,7950,7949,7948,7947,7946,7945,7944,7943,7942,7941,7940,7939,7938,7937,7936,7935,7934,7933,7932,7931,7930,7929,7928,7927,7926,7925,7924,7923,7922,7921,7920,7919,7918,7917,7916,7915,7914,7913,7912,7911,7910,7909,7908,7907,7906,7905,7904,7903,7902,7901,7900,7899,7898,7897,7896,7895,7894,7893,7892,7891,7890,7889,7888,7887,7886,7885,7884,7883,7882,7881,7880,7879,7878,7877,7876,7875,7874,7873,7872,7871,7870,7869,7868,7867,7866,7865,7864,7863,7862,7861,7860,7859,7858,7857,7856,7855,7854,7853,7852,7851,7850,7849,7848,7847,7846,7845,7844,7843,7842,7841,7840,7839,7838,7837,7836,7835,7834,7833,7832,7831,7830,7829,7828,7827,7826,7825,7824,7823,7822,7821,7820,7819,7818,7817,7816,7815,7814,7813,7812,7811,7810,7809,7808,7807,7806,7805,7804,7803,7802,7801,7800,7799,7798,7797,7796,7795,7794,7793,7792,7791,7790,7789,7788,7787,7786,7785,7784,7783,7782,7781,7780,7779,7778,7777,7776,7775,7774,7773,7772,7771,7770,7769,7768,7767,7766,7765,7764,7763,7762,7761,7760,7759,7758,7757,7756,7755,7754,7753,7752,7751,7750,7749,7748,7747,7746,7745,7744,7743,7742,7741,7740,7739,7738,7737,7736,7735,7734,7733,7732,7731,7730,7729,7728,7727,7726,7725,7724,7723,7722,7721,7720,7719,7718,7717,7716,7715,7714,7713,7712,7711,7710,7709,7708,7707,7706,7705,7704,7703,7702,7701,7700,7699,7698,7697,7696,7695,7694,7693,7692,7691,7690,7689,7688,7687,7686,7685,7684,7683,7682,7681,7680,7679,7678,7677,7676,7675,7674,7673,7672,7671,7670,7669,7668,7667,7666,7665,7664,7663,7662,7661,7660,7659,7658,7657,7656,7655,7654,7653,7652,7651,7650,7649,7648,7647,7646,7645,7644,7643,7642,7641,7640,7639,7638,7637,7636,7635,7634,7633,7632,7631,7630,7629,7628,7627,7626,7625,7624,7623,7622,7621,7620,7619,7618,7617,7616,7615,7614,7613,7612,7611,7610,7609,7608,7607,7606,7605,7604,7603,7602,7601,7600,7599,7598,7597,7596,7595,7594,7593,7592,7591,7590,7589,7588,7587,7586,7585,7584,7583,7582,7581,7580,7579,7578,7577,7576,7575,7574,7573,7572,7571,7570,7569,7568,7567,7566,7565,7564,7563,7562,7561,7560,7559,7558,7557,7556,7555,7554,7553,7552,7551,7550,7549,7548,7547,7546,7545,7544,7543,7542,7541,7540,7539,7538,7537,7536,7535,7534,7533,7532,7531,7530,7529,6471,6472,6473,6474,6475,6476,6477,6478,6479,6480,6481,6482,6483,6484,6485,6486,6487,6488,6489,6490,6491,6492,6493,6494,6495,6496,6497,6498,6499,6500,6501,6502,6503,6504,6505,6506,6507,6508,6509,6510,6511,6512,6513,6514,6515,6516,6517,6518,6519,6520,6521,6522,6523,6524,6525,6526,6527,6528,6529,6530,6531,6532,6533,6534,6535,6536,6537,6538,6539,6540,6541,6542,6543,6544,6545,6546,6547,6548,6549,6550,6551,6552,6553,6554,6555,6556,6557,6558,6559,6560,6561,6562,6563,6564,6565,6566,6567,6568,6569,6570,6571,6572,6573,6574,6575,6576,6577,6578,6579,6580,6581,6582,6583,6584,6585,6586,6587,6588,6589,6590,6591,6592,6593,6594,6595,6596,6597,6598,6599,6600,6601,6602,6603,6604,6605,6606,6607,6608,6609,6610,6611,6612,6613,6614,6615,6616,6617,6618,6619,6620,6621,6622,6623,6624,6625,6626,6627,6628,6629,6630,6631,6632,6633,6634,6635,6636,6637,6638,6639,6640,6641,6642,6643,6644,6645,6646,6647,6648,6649,6650,6651,6652,6653,6654,6655,6656,6657,6658,6659,6660,6661,6662,6663,6664,6665,6666,6667,6668,6669,6670,6671,6672,6673,6674,6675,6676,6677,6678,6679,6680,6681,6682,6683,6684,6685,6686,6687,6688,6689,6690,6691,6692,6693,6694,6695,6696,6697,6698,6699,6700,6701,6702,6703,6704,6705,6706,6707,6708,6709,6710,6711,6712,6713,6714,6715,6716,6717,6718,6719,6720,6721,6722,6723,6724,6725,6726,6727,6728,6729,6730,6731,6732,6733,6734,6735,6736,6737,6738,6739,6740,6741,6742,6743,6744,6745,6746,6747,6748,6749,6750,6751,6752,6753,6754,6755,6756,6757,6758,6759,6760,6761,6762,6763,6764,6765,6766,6767,6768,6769,6770,6771,6772,6773,6774,6775,6776,6777,6778,6779,6780,6781,6782,6783,6784,6785,6786,6787,6788,6789,6790,6791,6792,6793,6794,6795,6796,6797,6798,6799,6800,6801,6802,6803,6804,6805,6806,6807,6808,6809,6810,6811,6812,6813,6814,6815,6816,6817,6818,6819,6820,6821,6822,6823,6824,6825,6826,6827,6828,6829,6830,6831,6832,6833,6834,6835,6836,6837,6838,6839,6840,6841,6842,6843,6844,6845,6846,6847,6848,6849,6850,6851,6852,6853,6854,6855,6856,6857,6858,6859,6860,6861,6862,6863,6864,6865,6866,6867,6868,6869,6870,6871,6872,6873,6874,6875,6876,6877,6878,6879,6880,6881,6882,6883,6884,6885,6886,6887,6888,6889,6890,6891,6892,6893,6894,6895,6896,6897,6898,6899,6900,6901,6902,6903,6904,6905,6906,6907,6908,6909,6910,6911,6912,6913,6914,6915,6916,6917,6918,6919,6920,6921,6922,6923,6924,6925,6926,6927,6928,6929,6930,6931,6932,6933,6934,6935,6936,6937,6938,6939,6940,6941,6942,6943,6944,6945,6946,6947,6948,6949,6950,6951,6952,6953,6954,6955,6956,6957,6958,6959,6960,6961,6962,6963,6964,6965,6966,6967,6968,6969,6970,6971,6972,6973,6974,6975,6976,6977,6978,6979,6980,6981,6982,6983,6984,6985,6986,6987,6988,6989,6990,6991,6992,6993,6994,6995,6996,6997,6998,6999,7000,7001,7002,7003,7004,7005,7006,7007,7008,7009,7010,7011,7012,7013,7014,7015,7016,7017,7018,7019,7020,7021,7022,7023,7024,7025,7026,7027,7028,7029,7030,7031,7032,7033,7034,7035,7036,7037,7038,7039,7040,7041,7042,7043,7044,7045,7046,7047,7048,7049,7050,7051,7052,7053,7054,7055,7056,7057,7058,7059,7060,7061,7062,7063,7064,7065,7066,7067,7068,7069,7070,7071,7072,7073,7074,7075,7076,7077,7078,7079,7080,7081,7082,7083,7084,7085,7086,7087,7088,7089,7090,7091,7092,7093,7094,7095,7096,7097,7098,7099,7100,7101,7102,7103,7104,7105,7106,7107,7108,7109,7110,7111,7112,7113,7114,7115,7116,7117,7118,7119,7120,7121,7122,7123,7124,7125,7126,7127,7128,7129,7130,7131,7132,7133,7134,7135,7136,7137,7138,7139,7140,7141,7142,7143,7144,7145,7146,7147,7148,7149,7150,7151,7152,7153,7154,7155,7156,7157,7158,7159,7160,7161,7162,7163,7164,7165,7166,7167,7168,7169,7170,7171,7172,7173,7174,7175,7176,7177,7178,7179,7180,7181,7182,7183,7184,7185,7186,7187,7188,7189,7190,7191,7192,7193,7194,7195,7196,7197,7198,7199,7200,7201,7202,7203,7204,7205,7206,7207,7208,7209,7210,7211,7212,7213,7214,7215,7216,7217,7218,7219,7220,7221,7222,7223,7224,7225,7226,7227,7228,7229,7230,7231,7232,7233,7234,7235,7236,7237,7238,7239,7240,7241,7242,7243,7244,7245,7246,7247,7248,7249,7250,7251,7252,7253,7254,7255,7256,7257,7258,7259,7260,7261,7262,7263,7264,7265,7266,7267,7268,7269,7270,7271,7272,7273,7274,7275,7276,7277,7278,7279,7280,7281,7282,7283,7284,7285,7286,7287,7288,7289,7290,7291,7292,7293,7294,7295,7296,7297,7298,7299,7300,7301,7302,7303,7304,7305,7306,7307,7308,7309,7310,7311,7312,7313,7314,7315,7316,7317,7318,7319,7320,7321,7322,7323,7324,7325,7326,7327,7328,7329,7330,7331,7332,7333,7334,7335,7336,7337,7338,7339,7340,7341,7342,7343,7344,7345,7346,7347,7348,7349,7350,7351,7352,7353,7354,7355,7356,7357,7358,7359,7360,7361,7362,7363,7364,7365,7366,7367,7368,7369,7370,7371,7372,7373,7374,7375,7376,7377,7378,7379,7380,7381,7382,7383,7384,7385,7386,7387,7388,7389,7390,7391,7392,7393,7394,7395,7396,7397,7398,7399,7400,7401,7402,7403,7404,7405,7406,7407,7408,7409,7410,7411,7412,7413,7414,7415,7416,7417,7418,7419,7420,7421,7422,7423,7424,7425,7426,7427,7428,7429,7430,7431,7432,7433,7434,7435,7436,7437,7438,7439,7440,7441,7442,7443,7444,7445,7446,7447,7448,7449,7450,7451,7452,7453,7454,7455,7456,7457,7458,7459,7460,7461,7462,7463,7464,7465,7466,7467,7468,7469,7470,7471,7472,7473,7474,7475,7476,7477,7478,7479,7480,7481,7482,7483,7484,7485,7486,7487,7488,7489,7490,7491,7492,7493,7494,7495,7496,7497,7498,7499,7500,7501,7502,7503,7504,7505,7506,7507,7508,7509,7510,7511,7512,7513,7514,7515,7516,7517,7518,7519,7520,7521,7522,7523,7524,7525,7526,7527,7528],\"edges\":[[7648,8560],[8560,7870],[7870,8059],[8007,7747],[7747,8059],[8566,7980],[7980,8059],[7874,8444],[8444,7747],[8480,7888],[7888,8059],[8507,7971],[8509,8506],[8487,7971],[8489,8486],[8420,8560],[7939,8431],[7939,8146],[7939,8328],[7939,8172],[7939,8406],[7939,8176],[7939,7654],[7939,8410],[7939,8408],[7939,8178],[7939,8113],[7939,8310],[7939,7833],[7939,8047],[7939,8388],[7939,8151],[7939,8244],[7939,7731],[7939,8279],[7939,8331],[7940,7939],[7941,7940],[8593,7971],[8595,8592],[7733,7747],[7752,7971],[7754,7946],[8005,7595],[7595,7747],[7549,7888],[8553,8059],[8504,8059],[7749,8059],[8418,8560],[7600,7971],[7602,8146],[7755,7980],[8033,7747],[7803,7749],[7739,8059],[8095,8560],[8570,8059],[8104,8059],[7759,7749],[7908,8059],[7967,7747],[7671,7595],[8440,8078],[8078,8059],[7656,8093],[8093,7980],[7958,7595],[7638,7980],[7797,8059],[8117,7749],[8478,8476],[8478,8028],[8071,7749],[8479,8030],[7789,7749],[8468,8032],[8556,8039],[8469,8014],[8470,8016],[8107,7749],[8471,8018],[7960,7749],[8472,8020],[8534,7541],[8473,8022],[7833,7541],[8474,8024],[7661,7749],[8475,8026],[7902,7747],[7884,7747],[8428,7971],[8430,8431],[8453,8059],[8518,7749],[8011,7971],[8009,8148],[7788,7595],[7735,8059],[8037,8059],[7581,7971],[7583,8149],[8101,7747],[7950,7747],[8559,7747],[7771,7747],[8442,8560],[8596,8111],[8111,8059],[8602,7747],[7541,7749],[7926,7980],[8484,8113],[8113,7747],[8425,7971],[8427,8424],[7898,8059],[7646,8560],[8584,7607],[7607,7888],[7882,7747],[7952,7747],[7632,8059],[8089,7595],[7928,8059],[7773,8560],[8043,7971],[8045,8151],[8121,7747],[7892,7595],[8434,8059],[7566,7564],[7566,7685],[7567,7564],[7567,7687],[7568,7689],[7569,7691],[7570,7693],[7571,7695],[7572,7697],[7573,7699],[7574,7701],[7575,7703],[7576,7705],[7577,7707],[7578,7709],[7557,7711],[7558,7713],[7559,7715],[7560,7717],[7561,7719],[7562,7721],[7563,7723],[7634,7888],[7533,7980],[7605,7747],[8462,7747],[8578,7747],[7837,8059],[7769,8059],[8600,8560],[7529,7980],[7742,7980],[7904,7747],[7628,7749],[8456,8113],[8039,7749],[7866,8059],[8134,7971],[8136,8152],[8140,7971],[8143,8153],[7763,7747],[8502,7595],[8448,7962],[7962,8059],[7896,7788],[7974,7548],[7974,7762],[7974,8105],[7974,8541],[7974,7833],[7974,7593],[7974,8047],[7974,7560],[7974,7978],[7974,7861],[7974,7917],[7975,7548],[7975,7828],[7975,7863],[7975,8446],[7975,7978],[7975,7826],[7976,7974],[7976,7975],[7876,7747],[8146,8431],[7767,8111],[8151,8407],[8151,8152],[8152,8154],[8152,7556],[8153,8155],[8153,8052],[8154,8156],[8154,8439],[8155,8157],[8155,8529],[8156,8158],[8156,8528],[8157,8159],[8157,8528],[8158,8160],[8158,8531],[8159,8161],[8159,8530],[8160,8162],[8160,8447],[8538,7747],[8161,8163],[8161,8407],[8161,8446],[8162,8164],[8162,7925],[7969,8059],[8163,8165],[8163,8407],[8163,8405],[8163,8588],[8164,8166],[8164,7758],[8165,8167],[8165,8407],[8165,7757],[7593,8111],[8166,8168],[8166,8407],[8166,7621],[8167,8169],[8167,8407],[8167,7620],[8168,8170],[8168,8407],[8168,8432],[8169,8171],[8169,8407],[8169,8433],[8047,8059],[8170,8172],[8170,8407],[8170,8173],[8170,8174],[8170,7811],[7872,8059],[8171,8175],[8171,8407],[8171,8176],[8171,8177],[8171,8178],[8171,8547],[8172,8179],[8172,7761],[8173,8180],[8173,7762],[8174,8181],[8174,7586],[8175,8182],[8175,7585],[8176,8183],[8176,8405],[8176,8106],[7630,7747],[8177,8184],[8177,8405],[8177,8105],[7654,7747],[8178,8185],[8178,8407],[8178,8405],[8178,8541],[8179,8186],[8179,8405],[8179,8098],[8180,8187],[8180,8407],[8180,8405],[8180,8097],[8181,8188],[8181,8407],[8181,8405],[8181,8097],[8182,8189],[8182,7666],[8183,8190],[8183,7665],[8184,8191],[8184,7665],[8185,8192],[8185,7668],[8186,8193],[8186,7667],[8187,8194],[8187,7667],[8188,8195],[8188,7554],[8189,8196],[8189,8407],[8189,7553],[8190,8197],[8190,8407],[8190,7553],[8191,8198],[8191,8577],[8192,8199],[8192,8576],[8193,8200],[8193,8451],[8194,8201],[8194,8450],[8195,8202],[8195,8573],[8196,8203],[8196,8572],[8197,8204],[8197,7623],[8198,8205],[8198,7622],[8199,8206],[8199,8483],[8200,8207],[8200,8482],[8201,8208],[8201,8100],[8202,8209],[8202,8099],[8203,8210],[8203,8490],[8204,8211],[8204,8120],[8205,8212],[8205,8414],[8206,8213],[8206,8079],[8207,8214],[8207,8082],[8208,8215],[8208,8415],[8209,8216],[8209,8562],[8210,8217],[8210,8083],[8211,8218],[8211,8460],[8212,8219],[8212,8085],[8213,8220],[8213,8088],[8214,8221],[8214,8087],[8215,8222],[8215,7730],[8216,8223],[8216,7729],[8217,8224],[8217,7729],[8218,8225],[8218,8407],[8218,7556],[8219,8226],[8219,8407],[8219,7555],[8220,8227],[8220,8407],[8220,7552],[8221,8228],[8221,8407],[8221,7551],[8222,8229],[8222,8517],[7900,7747],[8223,8230],[8223,8516],[8224,8231],[8224,8407],[8224,7814],[7920,7747],[8225,8232],[8225,8407],[8225,7813],[8226,8233],[8226,8407],[8226,7816],[8227,8234],[8227,8407],[8227,7815],[8228,8235],[8228,8407],[8228,7818],[8229,8236],[8229,8407],[8229,7817],[8230,8237],[8230,8407],[8230,7820],[8231,8238],[8231,8407],[8231,7819],[8232,8239],[8232,8407],[8232,7822],[8233,8240],[8233,8407],[8233,7821],[8234,8241],[8234,8407],[8234,7824],[8235,8242],[8235,8407],[8235,7823],[8236,8243],[8236,8407],[8236,7826],[8237,8244],[8237,8407],[8237,7825],[8238,8245],[8238,7828],[8239,8246],[8239,8407],[8239,7827],[8240,8247],[8240,8407],[8240,7830],[8241,8248],[8241,8407],[8241,7829],[8242,8249],[8242,7832],[8243,8250],[8243,8407],[8243,7831],[7912,8111],[8244,8251],[8244,8407],[8244,7861],[7972,7747],[8245,8252],[8245,8407],[8245,7860],[8246,8253],[8246,7863],[8247,8254],[8247,8407],[8247,7862],[8248,8255],[8248,8407],[8248,7865],[8249,8256],[8249,8407],[8249,7864],[8250,8257],[8250,7849],[8251,8258],[8251,8407],[8251,7848],[8252,8259],[8252,8407],[8252,7851],[8253,8260],[8253,8407],[8253,7850],[8254,8261],[8254,8407],[8254,7853],[8255,8262],[8255,8407],[8255,7852],[8256,8263],[8256,8407],[8256,7855],[8257,8264],[8257,8407],[8257,7854],[8258,8265],[8258,8407],[8258,7857],[8259,8266],[8259,8407],[8259,7856],[8260,8267],[8260,8407],[8260,7859],[8548,7747],[8261,8268],[8261,8407],[8261,7858],[8262,8269],[8262,8407],[8262,8126],[8263,8270],[8263,8407],[8263,8405],[8263,8125],[8264,8271],[8264,8127],[8265,8272],[8265,7680],[8266,8273],[8266,7679],[8267,8274],[8267,7682],[8268,8275],[8268,7681],[8269,8276],[8269,7684],[8270,8277],[8270,7683],[8271,8278],[8271,7683],[8272,8279],[8272,8407],[8272,8608],[7731,7747],[8273,8280],[8273,8407],[8273,8607],[8274,8281],[8274,8407],[8274,8610],[8275,8282],[8275,8407],[8275,8609],[8276,8283],[8276,8407],[8276,8612],[8277,8284],[8277,8407],[8277,8611],[8278,8285],[8278,8407],[8278,7918],[8279,8286],[8279,8407],[8279,7917],[8280,8287],[8280,8407],[8280,7916],[8281,8288],[8281,8407],[8281,7919],[8282,8289],[8282,8606],[8283,8290],[8283,8613],[8284,8291],[8284,7625],[8285,8292],[8285,7624],[8286,8293],[8286,8131],[8287,8294],[8287,8130],[8288,8295],[8288,8133],[8289,8296],[8289,8132],[8290,8297],[8290,8132],[8291,8298],[8291,7988],[8580,7747],[8292,8299],[8292,7987],[8293,8300],[8293,7990],[8294,8301],[8294,7989],[8295,8302],[8295,7992],[8296,8303],[8296,7991],[8297,8304],[8297,7994],[8298,8305],[8298,7993],[8299,8306],[8299,7996],[8300,8307],[8300,7995],[8301,8308],[8301,7998],[8302,8309],[8302,7997],[8303,8310],[8303,8000],[8304,8311],[8304,7999],[8305,8312],[8305,8533],[8306,8313],[8306,8532],[8307,8314],[8307,7780],[8308,8315],[8308,7779],[8309,8316],[8309,7965],[8310,8317],[8310,7964],[8311,8318],[8311,7984],[8312,8319],[8312,8407],[8312,7983],[8313,8320],[8313,7986],[8314,8321],[8314,8407],[8314,7985],[8315,8322],[8315,8407],[8315,8512],[8316,8323],[8316,8407],[8316,8515],[8317,8324],[8317,8407],[8317,8515],[8318,8325],[8318,8407],[8318,8514],[8319,8326],[8319,8407],[8319,8513],[8320,8327],[8320,8407],[8320,8513],[8321,8328],[8321,8407],[8321,7792],[8322,8329],[8322,8407],[8322,7791],[8323,8330],[8323,7794],[8324,8331],[8324,7793],[8325,8332],[8325,7949],[8326,8333],[8326,7948],[8327,8334],[8327,7948],[8328,8335],[8328,8002],[8329,8336],[8329,8001],[8330,8337],[8330,8001],[8331,8338],[8331,8004],[8332,8339],[8332,8003],[8333,8340],[8333,8003],[8334,8341],[8334,7619],[8335,8342],[8335,7678],[8336,8343],[8336,7677],[8337,8344],[8337,7677],[8338,8345],[8338,7966],[8339,8346],[8339,7686],[8340,8347],[8340,8405],[8340,8407],[8340,8348],[8340,8349],[8340,8350],[8340,8351],[8340,7685],[8341,8352],[8341,7688],[8342,8353],[8342,8405],[8342,8407],[8342,8354],[8342,8355],[8342,8356],[8342,8357],[8342,7687],[8343,8358],[8343,8029],[8344,8359],[8344,8405],[8344,8407],[8344,8360],[8344,8361],[8344,8362],[8344,8363],[8344,8028],[8345,8364],[8345,7690],[8346,8365],[8346,8407],[8346,8366],[8346,8367],[8346,8368],[8346,8369],[8346,7689],[8347,8370],[8347,7692],[8348,8371],[8348,8407],[8348,8372],[8348,8373],[8348,8374],[8348,8375],[8348,7691],[8349,8376],[8349,8031],[8350,8377],[8350,8407],[8350,8378],[8350,8379],[8350,8380],[8350,8381],[8350,8030],[8351,8382],[8351,7694],[8352,8383],[8352,8407],[8352,8384],[8352,8385],[8352,8386],[8352,8387],[8352,7693],[8353,8388],[8353,7696],[8354,8389],[8354,8407],[8354,8390],[8354,8391],[8354,8392],[8354,8393],[8354,7695],[8355,8394],[8355,8013],[8356,8395],[8356,8407],[8356,8396],[8356,8397],[8356,8398],[8356,8399],[8356,8032],[8357,8400],[8357,7698],[8358,8401],[8358,8407],[8358,8402],[8358,8403],[8358,8404],[8358,8405],[8358,7697],[8359,8406],[8359,7700],[8360,8407],[8360,8408],[8360,8409],[8360,8410],[8360,8411],[8360,7699],[8361,8412],[8361,8015],[8362,8412],[8362,8407],[8362,8413],[8362,8014],[8363,8412],[8363,7702],[8364,8412],[8364,8407],[8364,8413],[8364,7701],[8365,8412],[8365,7704],[8366,8412],[8366,8407],[8366,8413],[8366,7703],[8367,8412],[8367,8017],[8368,8412],[8368,8407],[8368,8413],[8368,8016],[8369,8412],[8369,7706],[8370,8412],[8370,8407],[8370,8413],[8370,7705],[8371,8412],[8371,7708],[8372,8412],[8372,8407],[8372,8413],[8372,7707],[8373,8412],[8373,8019],[8374,8412],[8374,8407],[8374,8413],[8374,8018],[8375,8412],[8375,7710],[8376,8412],[8376,8407],[8376,8413],[8376,7709],[8377,8412],[8377,7712],[8378,8412],[8378,8407],[8378,8413],[8378,7711],[8379,8412],[8379,8021],[8380,8412],[8380,8407],[8380,8413],[8380,8020],[8381,8412],[8381,7714],[8382,8412],[8382,8407],[8382,8413],[8382,7713],[8383,8412],[8383,7716],[8384,8412],[8384,8407],[8384,8413],[8384,7715],[8385,8412],[8385,8023],[8386,8412],[8386,8407],[8386,8413],[8386,8022],[8387,8412],[8387,7718],[8388,8412],[8388,8407],[8388,8413],[8388,7717],[8389,8412],[8389,7720],[8390,8412],[8390,8407],[8390,8413],[8390,7719],[8391,8412],[8391,8025],[8392,8412],[8392,8407],[8392,8413],[8392,8024],[8393,8412],[8393,7722],[8394,8412],[8394,8407],[8394,8413],[8394,7721],[8395,8412],[8395,7724],[8396,8412],[8396,8407],[8396,8413],[8396,7723],[8397,8412],[8397,8027],[8398,8412],[8398,8407],[8398,8413],[8398,8026],[8399,8412],[8399,8407],[8399,7766],[8400,8412],[8400,8407],[8400,7765],[8401,8412],[8401,8407],[8401,7532],[7539,8059],[8402,8412],[8402,8407],[8402,7531],[8403,8412],[8403,8501],[8404,8412],[8404,8407],[8404,8500],[8408,7593],[8409,8047],[8035,7888],[7801,8093],[8068,7971],[8070,8067],[8061,7634],[8464,7747],[8115,7747],[7610,7971],[7612,7659],[8073,8444],[8064,7971],[8066,8063],[7743,8078],[7930,8604],[8604,7888],[8075,8113],[7914,8113],[8498,8059],[7643,7971],[7645,8055],[7587,8059],[8522,7971],[8525,8523],[7800,8059],[7650,8059],[8521,8560],[7954,7971],[7956,8057],[8144,7595],[8526,8059],[7636,7595],[7835,7626],[7846,7980],[7781,7595],[7890,8059],[7936,7595],[7844,7747],[7783,7747],[7663,8059],[8554,7888],[8128,7626],[7886,8113],[8466,7595],[8138,7626],[8436,7971],[8438,8439],[8041,8059],[8091,8113],[7579,7747],[7910,8466],[8568,7749],[8550,8059],[8416,7747],[7785,8059],[8582,8560],[8422,8418],[8109,7980],[8049,7971],[8051,8052],[8053,8560],[7932,7971],[7935,8576],[7805,7971],[7808,8577],[8544,7971],[8546,8547],[7809,7971],[7812,7811],[8454,7962],[7669,7747],[7543,8059],[7981,7980],[7745,7747],[7737,7747],[7597,8059],[8598,8560],[8123,8418],[8542,7971],[8540,8541],[7535,7971],[7537,8497],[7725,7971],[7727,8120],[7906,7595],[7652,7980],[8536,8078],[7878,7888],[8510,8059],[7894,8560],[7868,7749],[8491,7971],[8493,8490],[7777,7595],[7880,8111],[8495,7980],[7614,8431],[7614,8146],[7614,8148],[7614,8328],[7614,8172],[7614,7630],[7614,8406],[7614,8177],[7614,7654],[7614,8410],[7614,8408],[7614,8178],[7614,8113],[7614,8310],[7614,8331],[7615,7614],[7616,7615],[7603,8059],[7591,8466],[8586,7747],[8589,7971],[8591,8588],[7676,7971],[7674,7925],[8574,8059],[7795,8553],[7839,7747],[7589,7870],[7775,8059],[7548,8079],[7978,7556],[8412,8414],[8414,8415],[8415,8413],[8415,7946],[8415,8565],[8415,8057],[8415,8461],[8413,8082],[7946,8083],[8565,8562],[8057,8085],[8461,8460],[8347,6471],[8348,6472],[8345,6473],[8346,6474],[8343,6475],[8344,6476],[8341,6477],[8342,6478],[8339,6479],[8340,6480],[8337,6481],[8338,6482],[8335,6483],[8336,6484],[8333,6485],[8334,6486],[8363,6487],[8364,6488],[8361,6489],[8362,6490],[8359,6491],[8360,6492],[8357,6493],[8358,6494],[8355,6495],[8356,6496],[8353,6497],[8354,6498],[8351,6499],[8352,6500],[8349,6501],[7626,6502],[8350,6503],[8379,6504],[8380,6505],[8377,6506],[8378,6507],[8375,6508],[8415,6509],[8376,6510],[8373,6511],[8374,6512],[7975,6513],[8371,6514],[7974,6515],[8372,6516],[8369,6517],[8370,6518],[8367,6519],[7939,6520],[8368,6521],[8365,6522],[8366,6523],[8395,6524],[8396,6525],[8393,6526],[8394,6527],[8391,6528],[8392,6529],[8389,6530],[7978,6531],[8390,6532],[7975,6533],[8387,6534],[7974,6535],[8388,6536],[8385,6537],[8386,6538],[8383,6539],[8384,6540],[8381,6541],[8382,6542],[8283,6543],[8284,6544],[8281,6545],[8282,6546],[8279,6547],[8280,6548],[8277,6549],[8278,6550],[8275,6551],[8276,6552],[8273,6553],[8274,6554],[8271,6555],[8272,6556],[8269,6557],[8270,6558],[8299,6559],[8300,6560],[8297,6561],[8298,6562],[8295,6563],[8296,6564],[8293,6565],[8294,6566],[8291,6567],[8292,6568],[8289,6569],[8290,6570],[8287,6571],[8288,6572],[8285,6573],[8286,6574],[8315,6575],[8316,6576],[8313,6577],[8415,6578],[8314,6579],[8311,6580],[8312,6581],[8309,6582],[8310,6583],[8307,6584],[8308,6585],[8305,6586],[8306,6587],[8303,6588],[8304,6589],[8301,6590],[8302,6591],[8331,6592],[8332,6593],[8329,6594],[8330,6595],[8327,6596],[8328,6597],[8325,6598],[8326,6599],[8323,6600],[8324,6601],[8321,6602],[8322,6603],[8319,6604],[8320,6605],[8317,6606],[8318,6607],[8219,6608],[8220,6609],[8217,6610],[8218,6611],[8215,6612],[8216,6613],[8213,6614],[8214,6615],[8211,6616],[8212,6617],[8209,6618],[8210,6619],[8207,6620],[7977,6621],[8208,6622],[8205,6623],[8206,6624],[7938,6625],[8235,6626],[8236,6627],[8233,6628],[8234,6629],[8231,6630],[7613,6631],[8232,6632],[8229,6633],[8230,6634],[8227,6635],[8228,6636],[8225,6637],[8226,6638],[8223,6639],[8224,6640],[8221,6641],[8222,6642],[8251,6643],[8252,6644],[8249,6645],[8250,6646],[8247,6647],[8248,6648],[8245,6649],[8246,6650],[8243,6651],[8244,6652],[8241,6653],[7974,6654],[8242,6655],[8239,6656],[8240,6657],[7974,6658],[7975,6659],[8237,6660],[7974,6661],[7975,6662],[8238,6663],[8267,6664],[7978,6665],[8268,6666],[8265,6667],[8266,6668],[8263,6669],[8406,6670],[8264,6671],[8261,6672],[8262,6673],[8259,6674],[8260,6675],[8257,6676],[8258,6677],[8255,6678],[8256,6679],[8253,6680],[8254,6681],[8155,6682],[8156,6683],[8153,6684],[8154,6685],[8152,6686],[7978,6687],[7974,6688],[8171,6689],[7975,6690],[8172,6691],[8169,6692],[8170,6693],[8167,6694],[8168,6695],[8165,6696],[8166,6697],[8163,6698],[8164,6699],[8161,6700],[8162,6701],[8159,6702],[8160,6703],[8157,6704],[8158,6705],[8187,6706],[8188,6707],[8185,6708],[8186,6709],[8183,6710],[8184,6711],[8181,6712],[8182,6713],[8179,6714],[8180,6715],[8177,6716],[8059,6717],[8178,6718],[8175,6719],[8176,6720],[8173,6721],[8174,6722],[8203,6723],[8204,6724],[8201,6725],[8202,6726],[8199,6727],[8200,6728],[8197,6729],[8198,6730],[8195,6731],[8196,6732],[8193,6733],[8194,6734],[8191,6735],[8192,6736],[8189,6737],[8190,6738],[7974,6739],[8415,6740],[7978,6741],[8406,6742],[8415,6743],[7640,6744],[8152,6745],[7978,6746],[8403,6747],[8404,6748],[8401,6749],[8402,6750],[8399,6751],[8400,6752],[8397,6753],[8398,6754],[7978,6755],[8415,6756],[7975,6757],[7974,6758],[7939,6759],[7975,6760],[7974,6761],[7939,6762],[7838,6763],[8365,6764],[8366,6765],[8367,6766],[7778,6767],[8368,6768],[7647,6769],[8369,6770],[8594,6771],[8370,6772],[8371,6773],[8372,6774],[8373,6775],[7649,6776],[8374,6777],[8375,6778],[8376,6779],[8377,6780],[8378,6781],[7899,6782],[8379,6783],[8380,6784],[8381,6785],[8503,6786],[8382,6787],[8152,6788],[8383,6789],[8543,6790],[8384,6791],[8385,6792],[8386,6793],[8387,6794],[8145,6795],[8388,6796],[8389,6797],[8585,6798],[8390,6799],[8391,6800],[7770,6801],[8392,6802],[8393,6803],[8394,6804],[8072,6805],[8395,6806],[8396,6807],[8406,6808],[7915,6809],[8333,6810],[7804,6811],[8334,6812],[7810,6813],[8335,6814],[8038,6815],[8336,6816],[8337,6817],[7536,6818],[8338,6819],[7840,6820],[7975,6821],[8339,6822],[7974,6823],[8340,6824],[8520,6825],[8341,6826],[7790,6827],[8342,6828],[8437,6829],[7772,6830],[8343,6831],[8344,6832],[7927,6833],[8345,6834],[7955,6835],[8346,6836],[8347,6837],[8348,6838],[8349,6839],[8350,6840],[8351,6841],[7885,6842],[8352,6843],[8353,6844],[7611,6845],[8354,6846],[8599,6847],[8355,6848],[8558,6849],[8356,6850],[8357,6851],[8457,6852],[8358,6853],[8359,6854],[8360,6855],[8361,6856],[8362,6857],[8363,6858],[7901,6859],[8364,6860],[8406,6861],[8301,6862],[8302,6863],[7746,6864],[8303,6865],[7639,6866],[8304,6867],[8305,6868],[8306,6869],[7796,6870],[8307,6871],[8308,6872],[7799,6873],[8309,6874],[7631,6875],[8310,6876],[8567,6877],[8311,6878],[8312,6879],[8313,6880],[8314,6881],[8315,6882],[7736,6883],[8316,6884],[8317,6885],[7756,6886],[8318,6887],[8319,6888],[8320,6889],[8581,6890],[8321,6891],[7953,6892],[8322,6893],[8410,6894],[8323,6895],[8324,6896],[8325,6897],[8326,6898],[8511,6899],[8327,6900],[8328,6901],[8583,6902],[8329,6903],[8419,6904],[8330,6905],[8331,6906],[8124,6907],[8332,6908],[8269,6909],[8270,6910],[8069,6911],[8271,6912],[8272,6913],[8273,6914],[7895,6915],[8274,6916],[8519,6917],[8275,6918],[8276,6919],[8277,6920],[8545,6921],[7768,6922],[8278,6923],[7905,6924],[8279,6925],[8280,6926],[8281,6927],[8282,6928],[8283,6929],[8284,6930],[7887,6931],[8285,6932],[7921,6933],[8286,6934],[8287,6935],[7664,6936],[8288,6937],[8289,6938],[8499,6939],[8290,6940],[8291,6941],[7873,6942],[8292,6943],[8293,6944],[7544,6945],[8294,6946],[8295,6947],[7881,6948],[8296,6949],[8297,6950],[7879,6951],[8298,6952],[8299,6953],[8527,6954],[8300,6955],[8122,6956],[8237,6957],[8238,6958],[8239,6959],[7784,6960],[8240,6961],[7929,6962],[8241,6963],[8242,6964],[8426,6965],[8243,6966],[8587,6967],[8244,6968],[8245,6969],[8246,6970],[8569,6971],[8247,6972],[8248,6973],[7973,6974],[8050,6975],[8249,6976],[8250,6977],[8251,6978],[8252,6979],[8253,6980],[8116,6981],[8254,6982],[8255,6983],[8256,6984],[8257,6985],[8044,6986],[8258,6987],[8259,6988],[8260,6989],[7897,6990],[8261,6991],[8262,6992],[8263,6993],[8264,6994],[8265,6995],[8266,6996],[8267,6997],[8268,6998],[8205,6999],[7847,7000],[8206,7001],[8207,7002],[7907,7003],[8208,7004],[7662,7005],[8209,7006],[8431,7007],[8210,7008],[8211,7009],[7670,7010],[8212,7011],[8213,7012],[8214,7013],[7657,7014],[8215,7015],[8216,7016],[8217,7017],[8118,7018],[8218,7019],[8429,7020],[8219,7021],[8110,7022],[8220,7023],[8221,7024],[8222,7025],[7961,7026],[8223,7027],[8224,7028],[7869,7029],[8225,7030],[8226,7031],[8549,7032],[8141,7033],[8227,7034],[8228,7035],[8494,7036],[8571,7037],[8229,7038],[8230,7039],[7606,7040],[8231,7041],[8232,7042],[8233,7043],[7588,7044],[8234,7045],[7673,7046],[8235,7047],[8236,7048],[7893,7049],[8173,7050],[8174,7051],[7974,7052],[8175,7053],[7911,7054],[8176,7055],[8177,7056],[8539,7057],[8178,7058],[8179,7059],[7655,7060],[8180,7061],[7951,7062],[8181,7063],[8182,7064],[8183,7065],[7982,7066],[8184,7067],[8185,7068],[7732,7069],[8186,7070],[8449,7071],[8187,7072],[8188,7073],[8189,7074],[8190,7075],[8191,7076],[8192,7077],[8193,7078],[8054,7079],[8194,7080],[8195,7081],[7913,7082],[8196,7083],[7806,7084],[8197,7085],[7782,7086],[8590,7087],[8198,7088],[8199,7089],[8200,7090],[8201,7091],[8406,7092],[7734,7093],[8202,7094],[8203,7095],[8204,7096],[7937,7097],[8006,7098],[8415,7099],[7978,7100],[8575,7101],[8092,7102],[7974,7103],[8465,7104],[7975,7105],[7959,7106],[7672,7107],[8152,7108],[7740,7109],[8153,7110],[8154,7111],[7760,7112],[8155,7113],[8156,7114],[8034,7115],[8157,7116],[8158,7117],[8159,7118],[8129,7119],[8160,7120],[8161,7121],[7540,7122],[8162,7123],[7530,7124],[8163,7125],[7601,7126],[8164,7127],[8597,7128],[8165,7129],[8166,7130],[7867,7131],[8167,7132],[8168,7133],[7798,7134],[8169,7135],[8170,7136],[7974,7137],[8171,7138],[7641,7139],[7975,7140],[8172,7141],[7970,7142],[8040,7143],[8557,7144],[7877,7145],[8605,7146],[7931,7147],[7534,7148],[8488,7149],[8505,7150],[7753,7151],[8410,7152],[8452,7153],[7875,7154],[8551,7155],[7580,7156],[8062,7157],[7834,7158],[7644,7159],[7550,7160],[7651,7161],[7787,7162],[7598,7163],[8492,7164],[8139,7165],[8481,7166],[8415,7167],[7582,7168],[7909,7169],[8114,7170],[8076,7171],[8552,7172],[8008,7173],[8406,7174],[8108,7175],[7845,7176],[8537,7177],[7750,7178],[7629,7179],[8579,7180],[8077,7181],[7871,7182],[7596,7183],[8090,7184],[8441,7185],[7565,7186],[8012,7187],[7774,7188],[8048,7189],[7744,7190],[8555,7191],[7635,7192],[8417,7193],[8102,7194],[8423,7195],[7542,7196],[8535,7197],[8455,7198],[7653,7199],[7590,7200],[8463,7201],[7627,7202],[8601,7203],[8152,7204],[7836,7205],[8485,7206],[8443,7207],[8421,7208],[7637,7209],[7776,7210],[7738,7211],[7975,7212],[8103,7213],[7974,7214],[8036,7215],[7903,7216],[7764,7217],[7633,7218],[7786,7219],[7748,7220],[7978,7221],[8079,7222],[7968,7223],[8152,7224],[7963,7225],[8083,7226],[8082,7227],[7979,7228],[7741,7229],[8445,7230],[8074,7231],[8460,7232],[8085,7233],[8562,7234],[8467,7235],[7592,7236],[8412,7237],[7726,7238],[7939,7239],[8060,7240],[7975,7241],[7891,7242],[7974,7243],[7889,7244],[7608,7245],[7975,7246],[7604,7247],[8603,7248],[7883,7249],[8508,7250],[8042,7251],[8561,7252],[8096,7253],[8112,7254],[7594,7255],[8135,7256],[8415,7257],[8435,7258],[8094,7259],[7802,7260],[8152,7261],[8065,7262],[7614,7263],[7933,7264],[8477,7265],[8397,7266],[8398,7267],[8399,7268],[8400,7269],[8401,7270],[8402,7271],[8403,7272],[8404,7273],[8524,7274],[7939,7275],[7588,7276],[7606,7277],[8571,7278],[8494,7279],[8549,7280],[7869,7281],[7961,7282],[8110,7283],[8118,7284],[7657,7285],[7670,7286],[7662,7287],[7907,7288],[7847,7289],[7897,7290],[8116,7291],[7973,7292],[8569,7293],[8587,7294],[7929,7295],[7784,7296],[8122,7297],[7641,7298],[7798,7299],[7867,7300],[8597,7301],[7530,7302],[7540,7303],[8129,7304],[8034,7305],[7760,7306],[7740,7307],[7672,7308],[7959,7309],[8465,7310],[8092,7311],[8575,7312],[8006,7313],[7937,7314],[7734,7315],[7782,7316],[7913,7317],[8054,7318],[8449,7319],[7975,7320],[8152,7321],[7974,7322],[7732,7323],[7982,7324],[7951,7325],[7655,7326],[8539,7327],[7911,7328],[7893,7329],[8415,7330],[7901,7331],[7974,7332],[7975,7333],[8457,7334],[8558,7335],[8599,7336],[7885,7337],[8506,7338],[8486,7339],[7927,7340],[8592,7341],[7946,7342],[7772,7343],[8146,7344],[8431,7345],[7790,7346],[8148,7347],[8520,7348],[8149,7349],[8424,7350],[7840,7351],[8151,7352],[8152,7353],[8038,7354],[8153,7355],[7978,7356],[7804,7357],[7915,7358],[8072,7359],[7770,7360],[8585,7361],[8145,7362],[8503,7363],[7899,7364],[7649,7365],[7974,7366],[7647,7367],[7975,7368],[7778,7369],[7838,7370],[8527,7371],[7879,7372],[7881,7373],[8152,7374],[7544,7375],[7873,7376],[8499,7377],[7664,7378],[7921,7379],[7887,7380],[7905,7381],[7768,7382],[8519,7383],[8415,7384],[7895,7385],[8124,7386],[8419,7387],[8583,7388],[8511,7389],[7953,7390],[8581,7391],[7756,7392],[7736,7393],[8567,7394],[7631,7395],[7799,7396],[7796,7397],[7639,7398],[7746,7399],[7802,7400],[8094,7401],[8435,7402],[7594,7403],[8112,7404],[8096,7405],[8561,7406],[8042,7407],[7883,7408],[8603,7409],[7604,7410],[7608,7411],[7889,7412],[7891,7413],[8060,7414],[7592,7415],[8467,7416],[8074,7417],[8445,7418],[7741,7419],[7979,7420],[7963,7421],[7968,7422],[7748,7423],[7975,7424],[7974,7425],[7939,7426],[7978,7427],[8537,7428],[8067,7429],[7845,7430],[7659,7431],[8063,7432],[8108,7433],[8055,7434],[8523,7435],[8008,7436],[8057,7437],[8439,7438],[8552,7439],[8052,7440],[8576,7441],[8577,7442],[8076,7443],[8547,7444],[8114,7445],[7811,7446],[8541,7447],[7909,7448],[8497,7449],[8120,7450],[8481,7451],[8139,7452],[7598,7453],[7787,7454],[7651,7455],[7550,7456],[7834,7457],[8062,7458],[7580,7459],[8551,7460],[7875,7461],[8452,7462],[8505,7463],[7534,7464],[8490,7465],[8588,7466],[7925,7467],[7931,7468],[8605,7469],[7877,7470],[8557,7471],[8040,7472],[7970,7473],[7786,7474],[7633,7475],[7764,7476],[7903,7477],[8410,7478],[8036,7479],[8103,7480],[7738,7481],[7776,7482],[7637,7483],[8421,7484],[8443,7485],[8485,7486],[7836,7487],[7627,7488],[8601,7489],[8463,7490],[7590,7491],[7653,7492],[8535,7493],[8455,7494],[7542,7495],[8423,7496],[8102,7497],[8417,7498],[7635,7499],[8555,7500],[7744,7501],[8048,7502],[7774,7503],[8441,7504],[8090,7505],[7596,7506],[7871,7507],[8077,7508],[8579,7509],[7629,7510],[7750,7511],[8415,7512],[7975,7513],[8415,7514],[7974,7515],[7939,7516],[7978,7517],[8410,7518],[8152,7519],[8415,7520],[7978,7521],[7975,7522],[8410,7523],[8410,7524],[8410,7525],[8415,7526],[8152,7527],[8415,7528]],\"index\":17,\"version\":\"3.10.0.7\",\"numInternalNodes\":1085}");
        final var graph = GidGraph.getGraph(json);
        rocksDao.saveToRocksDb(graph.getIndex(), graph.getNodes(), graph.getNumInternalNodes(), graph.getEdges());
        final var graphData = rocksDao.getGraphData(graph.getIndex());
        assertEquals(graph.getNumInternalNodes(), graphData.nodes().size() - graphData.externalNodes().size());
        assertEquals(graph.getNodes().size(), graphData.nodes().size());
        assertEquals(new LongOpenHashSet((graph.getNodes())), new LongOpenHashSet(graphData.nodes()));
        assertEquals(graph.getEdges().size(), graphData.numArcs());
        assertEquals(new LongOpenHashSet(graph.getNodes().subList(graph.getNumInternalNodes(), graph.getNodes().size())), graphData.externalNodes());
    }

    @Test
    public void graphNotFoundTest() throws RocksDBException {
        final var graph = rocksDao.getGraphData(-1);
        assertNull(graph);
    }

    @Test
    public void multipleGraphsDatabaseTest() throws IOException, RocksDBException {
        final var json1 = new JSONObject("{" +
                "\"index\": 1," +
                "\"product\": \"test1\"," +
                "\"version\": \"0.0.1\"," +
                "\"nodes\": [1, 2, 3, 4, 5]," +
                "\"numInternalNodes\": 3," +
                "\"edges\": [[1, 3], [1, 5], [2, 4], [3, 4], [1, 2], [4, 1]]" +
                "}");
        final var graph1 = GidGraph.getGraph(json1);
        rocksDao.saveToRocksDb(graph1.getIndex(), graph1.getNodes(), graph1.getNumInternalNodes(), graph1.getEdges());
        final var graphData1 = rocksDao.getGraphData(graph1.getIndex());
        assertEquals(graph1.getNumInternalNodes(), graphData1.nodes().size() - graphData1.externalNodes().size());
        assertEquals(graph1.getNodes().size(), graphData1.nodes().size());
        assertEquals(new LongOpenHashSet(graph1.getNodes()), new LongOpenHashSet(graphData1.nodes()));
        assertEquals(new LongOpenHashSet(List.of(5L, 3L, 2L)), new LongOpenHashSet(graphData1.successors(1L)));
        assertEquals(new LongArrayList(List.of(4L)), graphData1.successors(2L));
        assertEquals(new LongArrayList(List.of(4L)), graphData1.successors(3L));
        assertEquals(new LongArrayList(List.of(1L)), graphData1.successors(4L));
        assertEquals(new LongArrayList(), graphData1.successors(5L));
        assertEquals(new LongArrayList(List.of(4L)), graphData1.predecessors(1L));
        assertEquals(new LongArrayList(List.of(1L)), graphData1.predecessors(2L));
        assertEquals(new LongArrayList(List.of(1L)), graphData1.predecessors(3L));
        assertEquals(new LongOpenHashSet(List.of(3L, 2L)), new LongOpenHashSet(graphData1.predecessors(4L)));
        assertEquals(new LongArrayList(List.of(1L)), graphData1.predecessors(5L));
        assertEquals(graph1.getEdges().size(), graphData1.numArcs());
        assertEquals(new LongOpenHashSet(List.of(4L, 5L)), graphData1.externalNodes());

        final var json2 = new JSONObject("{" +
                "\"index\": 2," +
                "\"product\": \"test2\"," +
                "\"version\": \"0.0.1\"," +
                "\"nodes\": [255, 256, 257, 258]," +
                "\"numInternalNodes\": 3," +
                "\"edges\": [[255, 256], [255, 258], [256, 257], [257, 258]]" +
                "}");
        final var graph2 = GidGraph.getGraph(json2);
        rocksDao.saveToRocksDb(graph2.getIndex(), graph2.getNodes(), graph2.getNumInternalNodes(), graph2.getEdges());
        final var graphData2 = rocksDao.getGraphData(graph2.getIndex());
        assertEquals(graph2.getNumInternalNodes(), graphData2.nodes().size() - graphData2.externalNodes().size());
        assertEquals(graph2.getNodes().size(), graphData2.nodes().size());
        assertEquals(new LongOpenHashSet(graph2.getNodes()), new LongOpenHashSet(graphData2.nodes()));
        assertEquals(new LongOpenHashSet(List.of(256L, 258L)), new LongOpenHashSet(graphData2.successors(255L)));
        assertEquals(new LongArrayList(List.of(257L)), graphData2.successors(256L));
        assertEquals(new LongArrayList(List.of(258L)), graphData2.successors(257L));
        assertEquals(new LongArrayList(), graphData2.predecessors(255L));
        assertEquals(new LongArrayList(List.of(255L)), graphData2.predecessors(256L));
        assertEquals(new LongArrayList(List.of(256L)), graphData2.predecessors(257L));
        assertEquals(new LongOpenHashSet(List.of(255L, 257L)), new LongOpenHashSet(graphData2.predecessors(258L)));
        assertEquals(graph2.getEdges().size(), graphData2.numArcs());
        assertEquals(new LongOpenHashSet(List.of(258L)), graphData2.externalNodes());
    }
}
