/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.core.data.vulnerability;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import eu.fasten.core.data.Constants;

import java.io.IOException;
import java.io.Serializable;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

public class Vulnerability implements Serializable {

    private String id;
    private LinkedHashSet<String> purls;
    @SerializedName(value = "base_cpe")
    private String baseCpe;
    @SerializedName(value = "first_patched_purls")
    private LinkedHashSet<String> firstPatchedPurls;

    private transient LinkedHashSet<Purl> validatedPurls;
    private transient LinkedHashSet<Purl> validatedFirstPatchedPurls;

    private Double scoreCVSS2;
    private Double scoreCVSS3;
    private String vectorCVSS2;
    private String vectorCVSS3;
    @SerializedName("cwe_ids")
    private List<String> cweIds;
    private Severity severity;

    @SerializedName(value = "published_date")
    private String publishedDate;
    @SerializedName(value = "last_modified_date")
    private String lastModifiedDate;
    @SerializedName(value = "patch_date")
    private String patchDate;

    private String description;
    private HashSet<String> references;
    @SerializedName(value = "patch_links")
    private HashSet<String> patchLinks;
    private HashSet<String> exploits;
    private HashSet<Patch> patches;
    @SerializedName(value = "fasten_uris")
    private HashSet<String> fastenUris;

    /**
     * Empty Constructor for Serialization
     */
    public Vulnerability() {
        this.purls = new LinkedHashSet<>();
        this.firstPatchedPurls = new LinkedHashSet<>();
        this.references = new HashSet<>();
        this.patches = new HashSet<>();
        this.patchLinks = new HashSet<>();
        this.exploits = new HashSet<>();
    }

    /**
     * Constructor for a Vulnerability Object.
     *
     * @param id - takes a String ID
     */
    public Vulnerability(String id) {
        this();
        this.id = id;
    }

    public String getId() {
        return id;
    }

    public Double getScoreCVSS2() {
        return scoreCVSS2;
    }

    public void setScoreCVSS2(Double scoreCVSS2) {
        this.scoreCVSS2 = scoreCVSS2;
    }

    public Double getScoreCVSS3() {
        return scoreCVSS3;
    }

    public void setScoreCVSS3(Double scoreCVSS3) {
        this.scoreCVSS3 = scoreCVSS3;
    }

    public String getVectorCVSS2() {
        return vectorCVSS2;
    }

    public void setVectorCVSS2(String vectorCVSS2) {
        this.vectorCVSS2 = vectorCVSS2;
    }

    public String getVectorCVSS3() {
        return vectorCVSS3;
    }

    public void setVectorCVSS3(String vectorCVSS3) {
        this.vectorCVSS3 = vectorCVSS3;
    }

    public java.util.List<String> getCweIds() {
        return cweIds;
    }

    public void setCweIds(java.util.List<String> cweIds) {
        this.cweIds = cweIds;
    }

    public String getPublishedDate() {
        return publishedDate;
    }

    public void setPublishedDate(String publishedDate) {
        this.publishedDate = publishedDate;
    }

    public String getLastModifiedDate() {
        return lastModifiedDate;
    }

    public void setLastModifiedDate(String lastModifiedDate) {
        this.lastModifiedDate = lastModifiedDate;
    }

    public String getPatchDate() { return patchDate; };

    public void setPatchDate(String patchDate) { this.patchDate = patchDate; };

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public HashSet<String> getReferences() {
        return this.references;
    }

    public void addReference(String url) {
        this.references.add(url);
    }

    public HashSet<String> getPatchLinks() {
        return patchLinks;
    }

    public void addExploit(String url) {
        this.exploits.add(url);
    }

    public HashSet<String> getExploits() {
        return exploits;
    }

    public void addPatchLink(String url) {
        this.patchLinks.add(url);
    }

    public void addPatch(Patch p) {
        this.patches.add(p);
    }

    public HashSet<Patch> getPatches() {
        return patches;
    }

    public void addPurl(String purl) {
        var newPurls = getPurls();
        newPurls.add(purl);
        setPurls(newPurls);
    }

    public LinkedHashSet<String> getPurls() {
        return new LinkedHashSet<>(purls);
    }

    public LinkedHashSet<String> getFirstPatchedPurls() {
        return new LinkedHashSet<>(firstPatchedPurls);
    }

    public void addFirstPatchedPurl(String purl) {
        var newFirstPatchedPurls = getFirstPatchedPurls();
        newFirstPatchedPurls.add(purl);
        setFirstPatchedPurls(newFirstPatchedPurls);
    }

    public Severity getSeverity() {
        return severity;
    }

    public void setSeverity(Severity severity) {
        this.severity = severity;
    }

    public String getBaseCpe() {
        return baseCpe;
    }

    public void setBaseCpe(String baseCpe) {
        this.baseCpe = baseCpe;
    }

    public void setPatchLinks(HashSet<String> patchLinks) {
        this.patchLinks = patchLinks;
    }

    public void setPatches(HashSet<Patch> patches) {
        this.patches = patches;
    }

    public void setId(String id) {
        this.id = id;
    }

    public void setPurls(LinkedHashSet<String> purls) {
        this.purls = purls;
        this.validatedPurls = null;
    }

    public void setFirstPatchedPurls(LinkedHashSet<String> firstPatchedPurls) {
        this.firstPatchedPurls = firstPatchedPurls;
        this.validatedPurls = null;
    }

    public void setReferences(HashSet<String> references) {
        this.references = references;
    }

    public void setExploits(HashSet<String> exploits) {
        this.exploits = exploits;
    }

    public HashSet<String> getFastenUris() {
        return fastenUris;
    }

    public void setFastenUris(HashSet<String> fastenUris) {
        this.fastenUris = fastenUris;
    }

    public LinkedHashSet<Purl> getValidatedPurls() {
        if(Objects.isNull(validatedPurls)) {
            validatedPurls = validateAndSortPurls(purls);
        }
        return validatedPurls;
    }

    public LinkedHashSet<Purl> getValidatedFirstPatchedPurls() {
        if(Objects.isNull(validatedFirstPatchedPurls)) {
            validatedFirstPatchedPurls = validateAndSortPurls(firstPatchedPurls);
        }
        return validatedFirstPatchedPurls;
    }

    public static LinkedHashSet<Purl> validateAndSortPurls(LinkedHashSet<String> purls) {
        var validatedPurls = new LinkedHashSet<Purl>();
        purls.forEach(p -> {
            var parsedPurl = Purl.of(p);
            if(parsedPurl != null) {
                validatedPurls.add(parsedPurl);
            }
        });
        return sortPurls(validatedPurls);
    }

    public static LinkedHashSet<Purl> sortPurls(LinkedHashSet<Purl> purls) {
        return purls.stream().sorted(Comparator.comparing(Purl::getVersion)).
            collect(Collectors.toCollection(LinkedHashSet::new));
    }

    public List<String> getEcosystems() {
        return getValidatedPurls().stream()
                .map(Purl::getEcosystem)
                .distinct()
                .collect(Collectors.toList());
    }

    public void filterUnsupportedPatches(String[] extensions) {
        var supportedPatches = new HashSet<Patch>();
        this.getPatches().forEach(p -> {
            if (Arrays.stream(extensions).anyMatch(p.getFileName()::endsWith)) {
                supportedPatches.add(p);
            }
        });
        this.setPatches(supportedPatches);
    }

    public void updatePatchDate() throws ParseException {
        final var sdf = new SimpleDateFormat("yyyy-MM-dd");
        if (this.getPatches().size() == 0) return;
        boolean set = false;
        var latestDate = sdf.parse("1950-01-01");
        for (Patch patch : this.getPatches()) {
            var patchDate = patch.getPatchDate();
            if(Objects.nonNull(patchDate)) {
                var parsedPatchDate = sdf.parse(patchDate);
                if (parsedPatchDate.after(latestDate)) {
                    latestDate = parsedPatchDate;
                    set = true;
                }
            }
        }
        if (set)
            this.setPatchDate(sdf.format(latestDate));
    }

    public static class PackageVersionMetadata {
        private String id;
        private Double scoreCVSS2;
        private Double scoreCVSS3;
        private String vectorCVSS2;
        private String vectorCVSS3;
        @SerializedName("cwe_ids")
        private List<String> cweIds;
        private Severity severity;
        @SerializedName(value = "published_date")
        private String publishedDate;
        @SerializedName(value = "last_modified_date")
        private String lastModifiedDate;
        @SerializedName(value = "patch_date")
        private String patchDate;
        private String description;
        @SerializedName(value = "fasten_uris")
        private HashSet<String> fastenUris;
    }
    public PackageVersionMetadata getPackageVersionMetadata() {
        var obj = new PackageVersionMetadata();
        obj.id = this.id;
        obj.scoreCVSS2 = this.scoreCVSS2;
        obj.scoreCVSS3 = this.scoreCVSS3;
        obj.vectorCVSS2 = this.vectorCVSS2;
        obj.vectorCVSS3 = this.vectorCVSS3;
        obj.cweIds = this.cweIds;
        obj.severity = this.severity;
        obj.publishedDate = this.publishedDate;
        obj.lastModifiedDate = this.lastModifiedDate;
        obj.patchDate = this.patchDate;
        obj.description = this.description;
        return obj;
    }

    public static class CallableMetadata {
        private String id;
        private Double scoreCVSS2;
        private Double scoreCVSS3;
        private String vectorCVSS2;
        private String vectorCVSS3;
        @SerializedName("cwe_ids")
        private List<String> cweIds;
        private Severity severity;
    }
    public CallableMetadata getCallableMetadata() {
        var obj = new CallableMetadata();
        obj.id = this.id;
        obj.scoreCVSS2 = this.scoreCVSS2;
        obj.scoreCVSS3 = this.scoreCVSS3;
        obj.vectorCVSS2 = this.vectorCVSS2;
        obj.vectorCVSS3 = this.vectorCVSS3;
        obj.cweIds = this.cweIds;
        obj.severity = this.severity;
        return obj;
    }

    @Override
    public String toString() {
        return this.toJson();
    }

    /**
     * JSON Representation of the Vulnerability using GSON.
     *
     * @return String JSON
     */
    public String toJson() {
        Gson gson = new Gson();
        return gson.toJson(this);
    }

    /**
     * Merges the vulnerability object with another one
     *
     * @param v2 - vulnerability to merge with
     */
    public void merge(Vulnerability v2) {
        assert this.getId().equals(v2.getId());
        var mergedPurls = this.getPurls();
        mergedPurls.addAll(v2.getPurls());
        this.setPurls(mergedPurls);
        var mergedFirstPatchedPurls = this.getFirstPatchedPurls();
        mergedFirstPatchedPurls.addAll(v2.getFirstPatchedPurls());
        this.setFirstPatchedPurls(mergedFirstPatchedPurls);
        this.references.addAll(v2.getReferences());
        this.patches.addAll(v2.getPatches());
        this.exploits.addAll(v2.getExploits());
        this.patchLinks.addAll(v2.getPatchLinks());
        if (v2.description != null && this.description != null) {
            if (v2.getDescription().length() > this.getDescription().length()) {
                this.setDescription(v2.getDescription());
            }
        }
        if (this.severity == null) {
            if (v2.severity != null) {
                this.setSeverity(v2.severity);
            }
        }
        if (this.scoreCVSS2 == null) {
            if (v2.scoreCVSS2 != null) {
                this.setScoreCVSS2(v2.scoreCVSS2);
            }
        }
        if (this.scoreCVSS3 == null) {
            if (v2.scoreCVSS3 != null) {
                this.setScoreCVSS3(v2.scoreCVSS3);
            }
        }
        if (this.publishedDate == null) {
            if (v2.publishedDate != null) {
                this.setPublishedDate(v2.publishedDate);
            }
        }
        if (this.lastModifiedDate == null) {
            if (v2.lastModifiedDate != null) {
                this.setLastModifiedDate(v2.lastModifiedDate);
            }
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Vulnerability that = (Vulnerability) o;
        return this.toJson().equals(that.toJson());
    }

    @Override
    public int hashCode() {
        return Objects.hash(toJson());
    }
}


