/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.core.data.vulnerability;

import eu.fasten.core.data.Constants;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.maven.artifact.versioning.ComparableVersion;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Helper class to deal with PURLs, dissecting them into Objects.
 * This makes it easier to gather the information needed
 */
public class Purl {
    private String type = "";
    private String namespace = "";
    private String name = "";
    private ComparableVersion version = new ComparableVersion("");
    private List<Pair<String, String>> qualifiers = new ArrayList<>();

    public Purl() {
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public ComparableVersion getVersion() { return version; }

    public void setVersion(ComparableVersion version) {
        this.version = version;
    }

    public List<Pair<String, String>> getQualifiers() {
        return qualifiers;
    }

    public void setQualifiers(List<Pair<String, String>> qualifiers) {
        this.qualifiers = qualifiers;
    }

    public String getBasePurl() {
        var sb = new StringBuilder();
        sb.append("pkg:");
        sb.append(type); sb.append('/');
        if(!namespace.equals("")) {
            sb.append(namespace);
            sb.append("/");
        }
        sb.append(name);
        return sb.toString();
    }

    public String getPackageName() {
        var sb = new StringBuilder();
        if(type.equals("maven")) {
            sb.append(namespace);
            sb.append(Constants.mvnCoordinateSeparator);
        }
        sb.append(name);
        return sb.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Purl that = (Purl) o;
        if (!Objects.equals(namespace, "")) {
            return Objects.equals(type, that.type) &&
                    namespace.equals(that.namespace) &&
                    Objects.equals(name, that.name) &&
                    Objects.equals(version, that.version) &&
                    qualifiers.equals(that.qualifiers);
        } else {
            return Objects.equals(type, that.type) &&
                    Objects.equals(name, that.name) &&
                    Objects.equals(version, that.version) &&
                    qualifiers.equals(that.qualifiers);
        }
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, namespace, name, version, qualifiers);
    }

    @Override
    public String toString() {
        var sb = new StringBuilder();
        sb.append("pkg:"); sb.append(type); sb.append("/");
        if(!namespace.equals("")) {
            sb.append(namespace);
            sb.append("/");
        }
        sb.append(name);
        sb.append("@"); sb.append(version);
        if(qualifiers.size() > 0) {
            sb.append("?");
            var it = qualifiers.iterator();
            while(it.hasNext()) {
                var q = it.next();
                sb.append(q.getLeft()); sb.append("="); sb.append(q.getRight());
                if(it.hasNext()) sb.append('&');
            }
        }
        return sb.toString();
    }

    /**
     * Builds a PURL Object from a string
     * @param purl - follows purl-spec guidelines
     * @return - PURLPkg Object
     */
    public static Purl of(String purl) {
        Purl pp = new Purl();

        String[] temp = purl.substring(4).split("/");
        if(temp.length == 0) {
            return null;
        }
        String type = temp[0];
        pp.setType(type);
        String extraInfo = null;
        switch (type) {
            case "deb":
            case "maven":
                pp.setNamespace(temp[1]);
                extraInfo = temp[2];
                break;
            case "pypi":
                extraInfo = temp[1];
                break;
        }

        if(extraInfo != null) {
            // Parse extra info
            String[] infoFirst = extraInfo.split("@");
            pp.setName(infoFirst[0]);
            String[] infoSecond = infoFirst[1].split("\\?");
            pp.setVersion(new ComparableVersion(infoSecond[0]));

            // Check if there is any qualifier
            if (infoSecond.length > 1) {
                // Parse qualifiers
                String[] qualifiers = infoSecond[1].split("&");
                for (String qualifier : qualifiers) {
                    String[] res = qualifier.split("=");
                    pp.getQualifiers().add(new ImmutablePair<>(res[0], res[1]));
                }
            }
        }

        return pp;
    }

    public String getEcosystem() {
        switch (type) {
            case "deb":
                return Constants.debianForge;
            case "maven":
                return Constants.mvnForge;
            case "pypi":
                return Constants.pypiForge;
            default:
                return "unsupported ecosystem";
        }
    }
}
