/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.core.data.vulnerability;

import com.google.gson.annotations.SerializedName;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class Patch implements Serializable {
    @SerializedName(value = "filename")
    private String fileName;
    @SerializedName(value = "patch_date")
    private String patchDate;
    @SerializedName(value = "patch_url")
    private String patchUrl;
    @SerializedName(value = "original_changed_line_numbers")
    private List<Integer> originalChangedLineNumbers;
    @SerializedName(value = "new_changed_line_numbers")
    private List<Integer> newChangedLineNumbers;

    /**
     * Empty Constructor for Serialization.
     */
    public Patch() {
        this.originalChangedLineNumbers = new ArrayList<>();
        this.newChangedLineNumbers      = new ArrayList<>();
    }

    /**
     * Patch class constructor.
     * @param fileName - file that was changed.
     * @param originalChangedLineNumbers - changed lines in the original file.
     * @param newChangedLineNumbers - changed lines in the new file.
     * @param patchUrl - url where the patch was retrieved.
     * @param patchDate - date of the patch.
     */
    public Patch(String fileName,
                 List<Integer> originalChangedLineNumbers,
                 List<Integer> newChangedLineNumbers,
                 String patchUrl,
                 String patchDate) {
        this.originalChangedLineNumbers = originalChangedLineNumbers;
        this.newChangedLineNumbers      = newChangedLineNumbers;
        this.fileName  = fileName;
        this.patchUrl  = patchUrl;
        this.patchDate = patchDate;
    }

    public String getFileName() {
        return fileName;
    }

    public void setFileName(String fileName) {
        this.fileName = fileName;
    }

    public String getPatchDate() {
        return patchDate;
    }

    public void setPatchDate(String patchDate) {
        this.patchDate = patchDate;
    }

    public String getPatchUrl() {
        return patchUrl;
    }

    public void setPatchUrl(String patchUrl) {
        this.patchUrl = patchUrl;
    }

    public List<Integer> getOriginalChangedLineNumbers() {
        return originalChangedLineNumbers;
    }

    public void setOriginalChangedLineNumbers(List<Integer> originalChangedLineNumbers) {
        this.originalChangedLineNumbers = originalChangedLineNumbers;
    }

    public List<Integer> getNewChangedLineNumbers() {
        return newChangedLineNumbers;
    }

    public void setNewChangedLineNumbers(List<Integer> newChangedLineNumbers) {
        this.newChangedLineNumbers = newChangedLineNumbers;
    }

    /**
     * Utility to merge patches.
     * @param p1 - first Patch to merge
     * @param p2 - second Patch to merge
     * @return first Patch enriched with info from the second.
     */
    public Patch mergePatches(Patch p1, Patch p2) {
        if (p1.getFileName() == null)  p1.setFileName(p2.getFileName());
        if (p1.getPatchDate() == null) p1.setPatchDate(p2.getPatchDate());
        if (p1.getPatchUrl() == null)  p1.setPatchUrl(p2.getPatchUrl());
        p1.getOriginalChangedLineNumbers().addAll(p2.originalChangedLineNumbers);
        p1.getNewChangedLineNumbers().addAll(p2.getNewChangedLineNumbers());
        return p1;
    }

    public void mergeWithOther(Patch p) {
        this.getOriginalChangedLineNumbers().addAll(p.getOriginalChangedLineNumbers());
        this.getNewChangedLineNumbers().addAll(p.getNewChangedLineNumbers());
        if (this.getFileName() == null)  this.setFileName(p.getFileName());
        if (this.getPatchDate() == null) this.setPatchDate(p.getPatchDate());
        if (this.getPatchUrl() == null)  this.setPatchUrl(p.getPatchUrl());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Patch patch = (Patch) o;
        return Objects.equals(fileName, patch.fileName) &&
                Objects.equals(originalChangedLineNumbers, patch.originalChangedLineNumbers) &&
                Objects.equals(newChangedLineNumbers, patch.newChangedLineNumbers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(fileName, originalChangedLineNumbers, newChangedLineNumbers);
    }

    @Override
    public String toString() {
        return "Patch{" +
                "fileName='" + fileName + '\'' +
                ", patchDate='" + patchDate + '\'' +
                ", patchUrl='" + patchUrl + '\'' +
                ", originalChangedLineNumbers=" + originalChangedLineNumbers +
                ", newChangedLineNumbers=" + newChangedLineNumbers +
                '}';
    }
}